/**
 * Titan Drilling - Dynamic Page Loader
 * Provides SPA-like functionality with AJAX page loading
 */

class DynamicLoader {
    constructor() {
        this.currentPage = 'Dashboard';
        this.currentUrl = '/';
        this.isLoading = false;
        this.pageCache = new Map();
        this.maxCacheSize = 20;

        this.init();
    }

    init() {
        this.bindEvents();
        this.initializePageComponents();
        this.setupBreadcrumb();
    }

    bindEvents() {
        // Handle browser back/forward buttons
        window.addEventListener('popstate', (event) => {
            if (event.state && event.state.url) {
                this.loadPage(event.state.url, event.state.title, false);
            }
        });

        // Handle form submissions
        document.addEventListener('submit', (e) => {
            if (e.target.hasAttribute('data-dynamic')) {
                e.preventDefault();
                this.handleDynamicForm(e.target);
            }
        });

        // Handle link clicks
        document.addEventListener('click', (e) => {
            if (e.target.matches('[data-dynamic-link]')) {
                e.preventDefault();
                const url = e.target.getAttribute('href');
                const title = e.target.getAttribute('data-title') || e.target.textContent.trim();
                this.loadPage(url, title);
            }
        });
    }

    showLoading() {
        const overlay = document.getElementById('loadingOverlay');
        if (overlay) {
            overlay.style.display = 'flex';
            this.isLoading = true;
        }
    }

    hideLoading() {
        const overlay = document.getElementById('loadingOverlay');
        if (overlay) {
            overlay.style.display = 'none';
            this.isLoading = false;
        }
    }

    async loadPage(url, pageTitle, updateHistory = true) {
        if (this.isLoading) return;

        try {
            console.log('Starting to load page:', url, 'with title:', pageTitle);
            this.showLoading();

            // Update browser history if needed
            if (updateHistory && window.history && window.history.pushState) {
                window.history.pushState({ url: url, title: pageTitle }, pageTitle, url);
            }

            // Update current page info
            this.currentPage = pageTitle;
            this.currentUrl = url;

            // Update page title and breadcrumb
            document.title = `Titan Drilling - ${pageTitle}`;
            this.updateBreadcrumb(pageTitle);

            // Check cache first
            if (this.pageCache.has(url)) {
                console.log('Using cached page for:', url);
                this.renderPage(this.pageCache.get(url), pageTitle);
                this.hideLoading();
                return;
            }

            console.log('Fetching page from server:', url);
            // Fetch page content
            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'text/html, application/xhtml+xml',
                    'Cache-Control': 'no-cache'
                },
                credentials: 'same-origin'
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const html = await response.text();
            console.log('Received HTML response, length:', html.length);

            // Cache the response
            this.cachePage(url, html);

            // Render the page
            this.renderPage(html, pageTitle);

        } catch (error) {
            console.error('Error loading page:', error);
            this.showErrorPage(url, pageTitle, error);
        } finally {
            this.hideLoading();
        }
    }

    renderPage(html, pageTitle) {
        const parser = new DOMParser();
        const doc = parser.parseFromString(html, 'text/html');

        // Extract content from the response - handle both direct content and Blade templates
        let content = null;

        // First try to find content in the response
        if (doc.querySelector('#dynamicContent')) {
            content = doc.querySelector('#dynamicContent');
        } else if (doc.querySelector('.content-box')) {
            content = doc.querySelector('.content-box');
        } else if (doc.querySelector('.content')) {
            content = doc.querySelector('.content');
        } else {
            // For Blade templates, look for the content section
            const contentSection = doc.querySelector('[data-content-section]') ||
                doc.querySelector('main') ||
                doc.querySelector('article') ||
                doc.querySelector('.container') ||
                doc.querySelector('.container-fluid') ||
                doc.querySelector('.container-bluid') ||
                doc.querySelector('body > div:not([id="sidebar"])');

            if (contentSection) {
                content = contentSection;
            } else {
                // Fallback: try to find any meaningful content
                const bodyContent = doc.querySelector('body');
                if (bodyContent) {
                    // Remove sidebar and other layout elements
                    const sidebar = bodyContent.querySelector('#sidebar');
                    const navbar = bodyContent.querySelector('nav');
                    if (sidebar) sidebar.remove();
                    if (navbar) navbar.remove();
                    content = bodyContent;
                }
            }
        }

        if (content) {
            const dynamicContent = document.getElementById('dynamicContent');
            if (dynamicContent) {
                // Remove loaded class for animation
                dynamicContent.classList.remove('loaded');

                // Update content
                dynamicContent.innerHTML = content.innerHTML;

                // Trigger page transition animation
                setTimeout(() => {
                    dynamicContent.classList.add('loaded');
                }, 50);

                // Reinitialize components
                this.initializePageComponents();

                // Update active navigation
                this.updateActiveNavigation();

                console.log('Page loaded successfully:', pageTitle);
            }
        } else {
            console.error('No content found in response');
            this.showErrorPage('', pageTitle, new Error('No content found in response'));
        }
    }

    showErrorPage(url, pageTitle, error) {
        const dynamicContent = document.getElementById('dynamicContent');
        if (dynamicContent) {
            dynamicContent.innerHTML = `
                <div class="alert alert-danger" role="alert">
                    <div class="d-flex align-items-center mb-3">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <h4 class="mb-0">Error Loading Page</h4>
                    </div>
                    <p class="mb-3">Sorry, there was an error loading "${pageTitle}". Please try again.</p>
                    <div class="d-flex gap-2">
                        <button class="btn btn-primary" onclick="dynamicLoader.loadPage('${url}', '${pageTitle}')">
                            <i class="fas fa-redo me-1"></i>Retry
                        </button>
                        <button class="btn btn-secondary" onclick="dynamicLoader.loadPage('/', 'Dashboard')">
                            <i class="fas fa-home me-1"></i>Go to Dashboard
                        </button>
                    </div>
                    <details class="mt-3">
                        <summary class="text-muted">Error Details</summary>
                        <code class="d-block mt-2 p-2 bg-light rounded">${error.message}</code>
                    </details>
                </div>
            `;
        }
    }

    cachePage(url, html) {
        // Remove oldest entries if cache is full
        if (this.pageCache.size >= this.maxCacheSize) {
            const firstKey = this.pageCache.keys().next().value;
            this.pageCache.delete(firstKey);
        }

        this.pageCache.set(url, html);
    }

    updateBreadcrumb(pageTitle) {
        const currentPageElement = document.getElementById('currentPage');
        if (currentPageElement) {
            currentPageElement.textContent = pageTitle;
        }
    }

    setupBreadcrumb() {
        const breadcrumbNav = document.querySelector('.breadcrumb-nav');
        if (breadcrumbNav) {
            breadcrumbNav.addEventListener('click', (e) => {
                if (e.target.matches('.breadcrumb-item a')) {
                    e.preventDefault();
                    const url = e.target.getAttribute('onclick')?.match(/loadPage\('([^']+)'/)?.[1];
                    if (url) {
                        this.loadPage(url, 'Dashboard');
                    }
                }
            });
        }
    }

    updateActiveNavigation() {
        // Remove active class from all navigation items
        document.querySelectorAll('.nav-link, .nav-sublink').forEach(item => {
            item.classList.remove('active');
        });

        // Add active class to current page navigation
        const currentUrl = this.currentUrl;
        const navItem = document.querySelector(`[onclick*="${currentUrl}"]`) ||
            document.querySelector(`[href="${currentUrl}"]`);

        if (navItem) {
            navItem.classList.add('active');

            // Expand parent menu if it's a sublink
            const parentMenu = navItem.closest('.collapse');
            if (parentMenu) {
                parentMenu.classList.add('show');
            }
        }
    }

    initializePageComponents() {
        // Check if this page should skip initialization
        if (document.body.hasAttribute('data-skip-dynamic-loader') ||
            document.querySelector('[data-skip-dynamic-loader]')) {
            console.log('Skipping DynamicLoader initialization for this page');
            return;
        }

        // Reinitialize Bootstrap components
        this.initializeBootstrapComponents();

        // Reinitialize charts
        this.initializeCharts();

        // Reinitialize custom components
        this.initializeCustomComponents();
    }

    initializeBootstrapComponents() {
        if (typeof bootstrap !== 'undefined') {
            // Reinitialize tooltips
            try {
                if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
                    const tooltipElements = document.querySelectorAll('[data-bs-toggle="tooltip"]');
                    if (tooltipElements && tooltipElements.length > 0) {
                        Array.from(tooltipElements).forEach(tooltipTriggerEl => {
                            try {
                                new bootstrap.Tooltip(tooltipTriggerEl);
                            } catch (e) {
                                console.error('Error creating tooltip:', e);
                            }
                        });
                    }
                }
            } catch (e) {
                console.error('Error initializing tooltips:', e);
            }

            // Reinitialize popovers
            try {
                if (typeof bootstrap !== 'undefined' && bootstrap.Popover) {
                    const popoverElements = document.querySelectorAll('[data-bs-toggle="popover"]');
                    if (popoverElements && popoverElements.length > 0) {
                        Array.from(popoverElements).forEach(popoverTriggerEl => {
                            try {
                                new bootstrap.Popover(popoverTriggerEl);
                            } catch (e) {
                                console.error('Error creating popover:', e);
                            }
                        });
                    }
                }
            } catch (e) {
                console.error('Error initializing popovers:', e);
            }

            // Reinitialize modals
            try {
                if (typeof bootstrap !== 'undefined' && bootstrap.Modal) {
                    const modalElements = document.querySelectorAll('[data-bs-toggle="modal"]');
                    if (modalElements && modalElements.length > 0) {
                        Array.from(modalElements).forEach(modalTriggerEl => {
                            modalTriggerEl.addEventListener('click', (e) => {
                                try {
                                    const targetModal = modalTriggerEl.getAttribute('data-bs-target');
                                    if (targetModal) {
                                        const modal = new bootstrap.Modal(document.querySelector(targetModal));
                                        modal.show();
                                    }
                                } catch (e) {
                                    console.error('Error showing modal:', e);
                                }
                            });
                        });
                    }
                }
            } catch (e) {
                console.error('Error initializing modals:', e);
            }
        }
    }

    initializeCharts() {
        try {
            if (typeof Chart !== 'undefined') {
                // Destroy existing charts
                if (Chart.instances && typeof Chart.instances.forEach === 'function') {
                    Chart.instances.forEach(chart => {
                        try {
                            chart.destroy();
                        } catch (e) {
                            console.error('Error destroying chart:', e);
                        }
                    });
                }

                // Initialize Fault Summary Chart
                this.initializeFaultSummaryChart();

                // Initialize Vehicle Status Chart
                this.initializeVehicleStatusChart();
            }
        } catch (e) {
            console.error('Error initializing charts:', e);
        }
    }

    initializeFaultSummaryChart() {
        try {
            const ctx = document.getElementById('myChart');
            if (ctx) {
                new Chart(ctx.getContext('2d'), {
                type: 'bar',
                data: {
                    labels: ['Faulty Vehicles', 'Repair Items', 'Replace Items'],
                    datasets: [{
                        data: [2, 1, 3],
                        backgroundColor: ['#DC0D0E', '#F5D904', '#ED4A7B'],
                        borderWidth: 0,
                        borderRadius: 4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(0,0,0,0.1)'
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
        }
        } catch (e) {
            console.error('Error initializing fault summary chart:', e);
        }
    }

    initializeVehicleStatusChart() {
        try {
            const ctx = document.getElementById('vehicle-status');
            if (ctx) {
                new Chart(ctx.getContext('2d'), {
                type: 'doughnut',
                data: {
                    labels: ['Active', 'Breakdown', 'Maintenance'],
                    datasets: [{
                        data: [10, 7, 3],
                        backgroundColor: ['#19A979', '#DC0D0E', '#F5D904'],
                        borderWidth: 0,
                        hoverOffset: 4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                padding: 20,
                                usePointStyle: true
                            }
                        }
                    }
                }
            });
        }
        } catch (e) {
            console.error('Error initializing vehicle status chart:', e);
        }
    }

    initializeCustomComponents() {
        // Add any custom component initialization here
        // For example, custom date pickers, sliders, etc.

        // Initialize any custom event listeners
        this.initializeCustomEventListeners();
    }

    initializeCustomEventListeners() {
        // Add custom event listeners for dynamic content
        document.addEventListener('click', (e) => {
            // Handle custom button clicks
            if (e.target.matches('.custom-action-btn')) {
                this.handleCustomAction(e.target);
            }

            // Handle data table actions
            if (e.target.matches('.table-action-btn')) {
                this.handleTableAction(e.target);
            }
        });
    }

    handleCustomAction(button) {
        const action = button.getAttribute('data-action');
        const itemId = button.getAttribute('data-id');

        console.log(`Custom action: ${action} for item: ${itemId}`);

        // Add your custom action handling logic here
        // For example, show modals, make AJAX calls, etc.
    }

    handleTableAction(button) {
        const action = button.getAttribute('data-action');
        const rowId = button.getAttribute('data-row-id');

        console.log(`Table action: ${action} for row: ${rowId}`);

        // Add your table action handling logic here
        // For example, edit, delete, view details, etc.
    }

    handleDynamicForm(form) {
        const formData = new FormData(form);
        const url = form.action || this.currentUrl;

        // Add loading state to form
        const submitBtn = form.querySelector('[type="submit"]');
        const originalText = submitBtn?.textContent;

        if (submitBtn) {
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Processing...';
        }

        // Submit form via AJAX
        fetch(url, {
            method: form.method || 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
            .then(response => response.json())
            .then(data => {
                // Handle successful submission
                this.showSuccessMessage(data.message || 'Form submitted successfully!');

                // Reset form if needed
                if (data.resetForm !== false) {
                    form.reset();
                }

                // Refresh page content if needed
                if (data.refreshPage) {
                    this.loadPage(this.currentUrl, this.currentPage);
                }
            })
            .catch(error => {
                // Handle error
                this.showErrorMessage('An error occurred while submitting the form. Please try again.');
                console.error('Form submission error:', error);
            })
            .finally(() => {
                // Reset button state
                if (submitBtn) {
                    submitBtn.disabled = false;
                    submitBtn.textContent = originalText;
                }
            });
    }

    showSuccessMessage(message) {
        this.showNotification(message, 'success');
    }

    showErrorMessage(message) {
        this.showNotification(message, 'error');
    }

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `alert alert-${type === 'error' ? 'danger' : type === 'success' ? 'success' : 'info'} alert-dismissible fade show position-fixed`;
        notification.style.cssText = 'top: 20px; right: 20px; z-index: 10000; min-width: 300px;';
        notification.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;

        // Add to page
        document.body.appendChild(notification);

        // Auto-remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 5000);
    }

    // Utility methods
    getCurrentPage() {
        return this.currentPage;
    }

    getCurrentUrl() {
        return this.currentUrl;
    }

    clearCache() {
        this.pageCache.clear();
    }

    preloadPage(url) {
        // Preload a page in the background
        fetch(url, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
            .then(response => response.text())
            .then(html => {
                this.cachePage(url, html);
            })
            .catch(error => {
                console.log('Preload failed for:', url, error);
            });
    }
}

// Initialize the dynamic loader when DOM is ready
let dynamicLoader;
document.addEventListener('DOMContentLoaded', () => {
    console.log('Initializing DynamicLoader...');
    dynamicLoader = new DynamicLoader();
    console.log('DynamicLoader initialized:', dynamicLoader);

    // Add loaded class to trigger initial animation
    setTimeout(() => {
        const dynamicContent = document.getElementById('dynamicContent');
        if (dynamicContent) {
            dynamicContent.classList.add('loaded');
        }
    }, 100);
});

// Global function for backward compatibility
window.loadPage = function (url, pageTitle) {
    console.log('Global loadPage called with:', url, pageTitle);
    if (dynamicLoader) {
        console.log('Using dynamicLoader instance');
        dynamicLoader.loadPage(url, pageTitle);
    } else {
        console.log('dynamicLoader not available, falling back to direct navigation');
        window.location.href = url;
    }
};
