// Custom Alert System - Global Implementation
(function() {
    'use strict';

    // Custom Alert System
    window.showCustomAlert = function(message, type = 'info', options = {}) {
        const alertId = 'custom-alert-' + Date.now();
        const alertOverlay = document.createElement('div');
        alertOverlay.id = alertId;
        alertOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.6);
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: all 0.3s ease;
            backdrop-filter: blur(4px);
        `;

        const alertBox = document.createElement('div');
        alertBox.style.cssText = `
            background: white;
            border-radius: 12px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
            max-width: 400px;
            width: 90%;
            transform: scale(0.8) translateY(20px);
            transition: all 0.3s ease;
            overflow: hidden;
        `;

        const typeColors = {
            success: { bg: '#10b981', icon: '✓', title: 'Success' },
            error: { bg: '#ef4444', icon: '✕', title: 'Error' },
            warning: { bg: '#f59e0b', icon: '⚠', title: 'Warning' },
            info: { bg: '#3b82f6', icon: 'ℹ', title: 'Information' },
            confirm: { bg: '#8b5cf6', icon: '?', title: 'Confirm' }
        };

        const colors = typeColors[type] || typeColors.info;

        alertBox.innerHTML = `
            <div style="background: ${colors.bg}; color: white; padding: 20px; text-align: center;">
                <div style="font-size: 32px; margin-bottom: 8px;">${colors.icon}</div>
                <div style="font-size: 18px; font-weight: 600;">${colors.title}</div>
            </div>
            <div style="padding: 24px; text-align: center;">
                <div style="font-size: 16px; color: #374151; line-height: 1.5; margin-bottom: 24px;">${message}</div>
                <div style="display: flex; gap: 12px; justify-content: center;">
                    ${type === 'confirm' ? 
                        `<button id="${alertId}-cancel" style="
                            background: #f3f4f6;
                            color: #374151;
                            border: none;
                            padding: 12px 24px;
                            border-radius: 8px;
                            font-weight: 500;
                            cursor: pointer;
                            transition: all 0.2s ease;
                            min-width: 80px;
                        ">Cancel</button>
                        <button id="${alertId}-confirm" style="
                            background: ${colors.bg};
                            color: white;
                            border: none;
                            padding: 12px 24px;
                            border-radius: 8px;
                            font-weight: 500;
                            cursor: pointer;
                            transition: all 0.2s ease;
                            min-width: 80px;
                        ">OK</button>` :
                        `<button id="${alertId}-ok" style="
                            background: ${colors.bg};
                            color: white;
                            border: none;
                            padding: 12px 24px;
                            border-radius: 8px;
                            font-weight: 500;
                            cursor: pointer;
                            transition: all 0.2s ease;
                            min-width: 80px;
                        ">OK</button>`
                    }
                </div>
            </div>
        `;

        alertOverlay.appendChild(alertBox);
        document.body.appendChild(alertOverlay);

        // Animate in
        setTimeout(() => {
            alertOverlay.style.opacity = '1';
            alertBox.style.transform = 'scale(1) translateY(0)';
        }, 10);

        // Add hover effects
        const buttons = alertBox.querySelectorAll('button');
        buttons.forEach(button => {
            button.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-2px)';
                this.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.2)';
            });
            button.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
                this.style.boxShadow = 'none';
            });
        });

        return new Promise((resolve) => {
            if (type === 'confirm') {
                const cancelBtn = document.getElementById(`${alertId}-cancel`);
                const confirmBtn = document.getElementById(`${alertId}-confirm`);
                
                cancelBtn.addEventListener('click', () => {
                    closeAlert(false);
                });
                
                confirmBtn.addEventListener('click', () => {
                    closeAlert(true);
                });
            } else {
                const okBtn = document.getElementById(`${alertId}-ok`);
                okBtn.addEventListener('click', () => {
                    closeAlert(true);
                });
            }

            // Close on overlay click
            alertOverlay.addEventListener('click', (e) => {
                if (e.target === alertOverlay) {
                    closeAlert(type === 'confirm' ? false : true);
                }
            });

            function closeAlert(result) {
                alertOverlay.style.opacity = '0';
                alertBox.style.transform = 'scale(0.8) translateY(20px)';
                setTimeout(() => {
                    if (alertOverlay.parentNode) {
                        document.body.removeChild(alertOverlay);
                    }
                    resolve(result);
                }, 300);
            }
        });
    };

    // Enhanced Notification System
    window.showNotification = function(message, type = 'info', duration = 3000) {
        const notificationId = 'notification-' + Date.now();
        const notification = document.createElement('div');
        notification.id = notificationId;
        notification.className = `custom-notification notification-${type}`;
        
        const typeConfig = {
            success: { 
                bg: '#10b981', 
                icon: '✓', 
                border: '#059669',
                title: 'Success'
            },
            error: { 
                bg: '#ef4444', 
                icon: '✕', 
                border: '#dc2626',
                title: 'Error'
            },
            warning: { 
                bg: '#f59e0b', 
                icon: '⚠', 
                border: '#d97706',
                title: 'Warning'
            },
            info: { 
                bg: '#3b82f6', 
                icon: 'ℹ', 
                border: '#1d4ed8',
                title: 'Information'
            }
        };
        
        const config = typeConfig[type] || typeConfig.info;
        
        notification.innerHTML = `
            <div style="
                display: flex;
                align-items: center;
                gap: 12px;
                padding: 16px 20px;
                background: white;
                border-left: 4px solid ${config.bg};
                border-radius: 8px;
                box-shadow: 0 8px 24px rgba(0, 0, 0, 0.15);
                min-width: 300px;
                max-width: 400px;
            ">
                <div style="
                    width: 40px;
                    height: 40px;
                    background: ${config.bg};
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    color: white;
                    font-size: 18px;
                    font-weight: bold;
                    flex-shrink: 0;
                ">${config.icon}</div>
                <div style="flex: 1;">
                    <div style="
                        font-weight: 600;
                        color: #1f2937;
                        font-size: 14px;
                        margin-bottom: 4px;
                    ">${config.title}</div>
                    <div style="
                        color: #6b7280;
                        font-size: 13px;
                        line-height: 1.4;
                    ">${message}</div>
                </div>
                <button onclick="closeNotification('${notificationId}')" style="
                    background: none;
                    border: none;
                    color: #9ca3af;
                    cursor: pointer;
                    padding: 4px;
                    border-radius: 4px;
                    transition: all 0.2s ease;
                    font-size: 18px;
                    line-height: 1;
                " title="Close">×</button>
            </div>
        `;
        
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 10001;
            opacity: 0;
            transform: translateX(100%) scale(0.9);
            transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        `;
        
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateX(0) scale(1)';
        }, 10);
        
        // Auto close
        if (duration > 0) {
            setTimeout(() => {
                closeNotification(notificationId);
            }, duration);
        }
        
        // Add hover effects
        const closeBtn = notification.querySelector('button');
        closeBtn.addEventListener('mouseenter', function() {
            this.style.background = '#f3f4f6';
            this.style.color = '#374151';
        });
        closeBtn.addEventListener('mouseleave', function() {
            this.style.background = 'none';
            this.style.color = '#9ca3af';
        });
    };
    
    // Close notification function
    window.closeNotification = function(notificationId) {
        const notification = document.getElementById(notificationId);
        if (notification) {
            notification.style.opacity = '0';
            notification.style.transform = 'translateX(100%) scale(0.9)';
            setTimeout(() => {
                if (notification.parentNode) {
                    document.body.removeChild(notification);
                }
            }, 300);
        }
    };

    // Global alert functions for easy use
    window.showSuccess = function(message) {
        return window.showCustomAlert(message, 'success');
    };

    window.showError = function(message) {
        return window.showCustomAlert(message, 'error');
    };

    window.showWarning = function(message) {
        return window.showCustomAlert(message, 'warning');
    };

    window.showInfo = function(message) {
        return window.showCustomAlert(message, 'info');
    };

    window.showConfirm = function(message) {
        return window.showCustomAlert(message, 'confirm');
    };

    // Replace default browser alerts globally
    window.alert = function(message) {
        return window.showCustomAlert(message, 'info');
    };

    window.confirm = function(message) {
        return window.showCustomAlert(message, 'confirm');
    };

    // Override native confirm for all elements
    function overrideConfirms() {
        const elements = document.querySelectorAll('[onclick*="confirm("]');
        elements.forEach(element => {
            if (!element.hasAttribute('data-custom-alert-override')) {
                element.setAttribute('data-custom-alert-override', 'true');
                const originalOnclick = element.getAttribute('onclick');
                if (originalOnclick && originalOnclick.includes('confirm(')) {
                    element.removeAttribute('onclick');
                    element.addEventListener('click', function(e) {
                        e.preventDefault();
                        // Extract the confirm message from the original onclick
                        const match = originalOnclick.match(/confirm\(['"`]([^'"`]+)['"`]\)/);
                        if (match) {
                            const message = match[1];
                            window.showCustomAlert(message, 'confirm').then(result => {
                                if (result) {
                                    // Execute the rest of the original onclick if user confirms
                                    const remainingCode = originalOnclick.replace(/return\s+confirm\([^)]+\)[;]?/, '').trim();
                                    if (remainingCode) {
                                        try {
                                            eval(remainingCode);
                                        } catch (error) {
                                            console.error('Error executing onclick code:', error);
                                        }
                                    }
                                }
                            });
                        }
                    });
                }
            }
        });
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', overrideConfirms);
    } else {
        overrideConfirms();
    }

    // Also run periodically to catch dynamically added elements
    setInterval(overrideConfirms, 1000);

})();
