<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\UploadedFile;

class FirebaseStorageService
{
    private $bucketName = 'titan-drilling-1f8e9.appspot.com';
    private $baseUrl = 'https://firebasestorage.googleapis.com/v0/b/';

    public function uploadImage(UploadedFile $image, string $path, string $fileName = null): ?string
    {
        try {
            $fileName = $fileName ?: time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $fullPath = $path . '/' . $fileName;
            
            // Encode the path for URL
            $encodedPath = urlencode($fullPath);
            $uploadUrl = $this->baseUrl . $this->bucketName . '/o/' . $encodedPath;
            
            $fileContent = file_get_contents($image->getRealPath());
            $mimeType = $image->getMimeType();
            
            $response = Http::withHeaders([
                'Content-Type' => $mimeType,
            ])->put($uploadUrl, $fileContent);
            
            if ($response->successful()) {
                // Return the download URL
                return $this->getDownloadUrl($fullPath);
            } else {
                Log::error('Firebase Storage upload failed: ' . $response->body());
                return null;
            }
        } catch (\Exception $e) {
            Log::error('Error uploading to Firebase Storage: ' . $e->getMessage());
            return null;
        }
    }

    public function deleteImage(string $path): bool
    {
        try {
            $encodedPath = urlencode($path);
            $deleteUrl = $this->baseUrl . $this->bucketName . '/o/' . $encodedPath;
            
            $response = Http::delete($deleteUrl);
            
            return $response->successful();
        } catch (\Exception $e) {
            Log::error('Error deleting from Firebase Storage: ' . $e->getMessage());
            return false;
        }
    }

    private function getDownloadUrl(string $path): string
    {
        $encodedPath = urlencode($path);
        return $this->baseUrl . $this->bucketName . '/o/' . $encodedPath . '?alt=media';
    }
}
