<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class MaintenanceRequestController extends Controller
{
    private $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalRequestMaintenancereport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json';
    private $lastRequestNumberUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalRequestMaintenancereport/admins/lastRequestNumber.json';

    public function create()
    {
        // Generate request number
        $requestNumber = $this->generateRequestNumber();

        return view('maintenance.Create_New_Request_Maintenance', compact('requestNumber'));
    }

    public function store(Request $request)
    {
        try {
            // Validate the request
            $validated = $request->validate([
                'workOrderTitle' => 'required|string|max:255',
                'country' => 'required|string',
                'project' => 'required|string',
                'memo' => 'nullable|string',
                'priority' => 'nullable|string',
                'selectedVehicle' => 'nullable|string',
                'assignTo' => 'nullable|string',
                'requestType' => 'nullable|string',
                'tasks' => 'nullable|string',
            ]);

            // Generate request number
            $requestNumber = $this->generateRequestNumber($validated['country'], $validated['project']);

            // Prepare work order data
            $workOrderData = [
                'id' => uniqid(),
                'requestNumber' => $requestNumber,
                'workOrderTitle' => $validated['workOrderTitle'],
                'country' => $validated['country'],
                'project' => $validated['project'],
                'memo' => $validated['memo'] ?? '',
                'priority' => $validated['priority'] ?? 'Low',
                'vehicle' => $validated['selectedVehicle'] ? json_decode($validated['selectedVehicle'], true) : null,
                'assignTo' => $validated['assignTo'] ? json_decode($validated['assignTo'], true) : [],
                'requestType' => $validated['requestType'] ?? 'General',
                'tasks' => $validated['tasks'] ? json_decode($validated['tasks'], true) : [],
                'status' => 'Initiated',
                'createdAt' => Carbon::now()->toISOString(),
                'createdBy' => auth()->user()->id ?? 'system',
            ];

            // Save to Firebase
            $this->saveToFirebase($workOrderData);

            // Generate PDF
            $pdfPath = $this->generatePDF($workOrderData);

            return response()->json([
                'success' => true,
                'message' => 'Maintenance request created successfully!',
                'requestId' => $workOrderData['id'],
                'requestNumber' => $requestNumber,
                'pdfPath' => $pdfPath
            ]);

        } catch (\Exception $e) {
            Log::error('Error creating maintenance request: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create maintenance request. Please try again.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $workOrder = $this->getWorkOrderById($id);
            
            if (!$workOrder) {
                return response()->json([
                    'success' => false,
                    'message' => 'Maintenance request not found'
                ], 404);
            }

            return view('maintenance.request-details', compact('workOrder'));
        } catch (\Exception $e) {
            Log::error('Error fetching maintenance request: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch maintenance request'
            ], 500);
        }
    }

    public function pdf($id)
    {
        try {
            $workOrder = $this->getWorkOrderById($id);
            
            if (!$workOrder) {
                abort(404, 'Maintenance request not found');
            }

            $pdfPath = $this->generatePDF($workOrder);
            
            return response()->file($pdfPath);
        } catch (\Exception $e) {
            Log::error('Error generating PDF: ' . $e->getMessage());
            abort(500, 'Failed to generate PDF');
        }
    }

    // API endpoints for modals
    public function getEquipment()
    {
        try {
            $equipment = $this->fetchEquipmentData();
            
            return response()->json([
                'success' => true,
                'equipment' => $equipment
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching equipment: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load equipment data'
            ], 500);
        }
    }

    public function getUsers()
    {
        try {
            $users = $this->fetchUsersData();
            
            return response()->json([
                'success' => true,
                'users' => $users
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching users: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load team members'
            ], 500);
        }
    }

    private function generateRequestNumber($country = null, $project = null)
    {
        $countryCode = $country === 'Zambia' ? 'ZAM' : ($country === 'DRC' ? 'DRC' : 'XXX');
        $projectCode = $project ? strtoupper(substr($project, 0, 3)) : 'XXX';
        
        // Get last request number from Firebase
        $lastRequestNumber = $this->getLastRequestNumber();
        
        // Extract the number from the last request number (e.g., "ZAM-KOB-RM-021" -> 21)
        $lastNumber = $this->extractNumberFromRequestNumber($lastRequestNumber);
        $nextNumber = $lastNumber + 1;
        
        $generatedNumber = "{$countryCode}-{$projectCode}-RM-" . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);
        
        Log::info('Generated new request number', [
            'country' => $country,
            'project' => $project,
            'country_code' => $countryCode,
            'project_code' => $projectCode,
            'last_request_number' => $lastRequestNumber,
            'last_number' => $lastNumber,
            'next_number' => $nextNumber,
            'generated_number' => $generatedNumber
        ]);
        
        return $generatedNumber;
    }

    private function getLastRequestNumber()
    {
        try {
            // Get the last request number from Firebase
            $response = Http::get($this->lastRequestNumberUrl);
            if ($response->successful()) {
                $data = $response->json();
                if ($data && is_string($data)) {
                    Log::info('Fetched last request number from Firebase: ' . $data);
                    return $data; // Return the full request number string
                }
            }
        } catch (\Exception $e) {
            Log::warning('Could not fetch last request number from Firebase: ' . $e->getMessage());
        }

        // Fallback to default
        return 'ZAM-KOB-RM-000';
    }

    private function extractNumberFromRequestNumber($requestNumber)
    {
        // Extract number from format like "ZAM-KOB-RM-021" -> 21
        preg_match('/-RM-(\d+)$/', $requestNumber, $matches);
        $extractedNumber = isset($matches[1]) ? (int)$matches[1] : 0;
        
        Log::info('Extracted number from request number', [
            'request_number' => $requestNumber,
            'extracted_number' => $extractedNumber
        ]);
        
        return $extractedNumber;
    }

    private function saveToFirebase($workOrderData)
    {
        try {
            // Save the work order data
            $response = Http::put($this->firebaseUrl, $workOrderData);
            
            if (!$response->successful()) {
                throw new \Exception('Failed to save to Firebase: ' . $response->body());
            }

            // Update the last request number counter in Firebase
            $this->updateLastRequestNumber($workOrderData['requestNumber']);

            Log::info('Maintenance request saved to Firebase successfully', ['requestId' => $workOrderData['id']]);
        } catch (\Exception $e) {
            Log::error('Error saving to Firebase: ' . $e->getMessage());
            throw $e;
        }
    }

    private function updateLastRequestNumber($requestNumber)
    {
        try {
            Log::info('Updating last request number in Firebase', [
                'url' => $this->lastRequestNumberUrl,
                'new_request_number' => $requestNumber
            ]);
            
            // Update the last request number in Firebase
            $response = Http::put($this->lastRequestNumberUrl, $requestNumber);
            
            if (!$response->successful()) {
                Log::warning('Failed to update last request number in Firebase', [
                    'status_code' => $response->status(),
                    'response_body' => $response->body(),
                    'request_number' => $requestNumber
                ]);
            } else {
                Log::info('Last request number updated successfully in Firebase', [
                    'request_number' => $requestNumber,
                    'status_code' => $response->status()
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Error updating last request number in Firebase', [
                'error' => $e->getMessage(),
                'request_number' => $requestNumber
            ]);
        }
    }

    private function generatePDF($workOrderData)
    {
        try {
            $html = $this->generatePDFHtml($workOrderData);
            
            // Use DomPDF to generate PDF
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'portrait');
            $dompdf->render();
            
            // Save PDF to storage
            $filename = 'maintenance-request-' . $workOrderData['id'] . '.pdf';
            $path = 'pdfs/maintenance/' . $filename;
            
            Storage::disk('public')->put($path, $dompdf->output());
            
            return storage_path('app/public/' . $path);
        } catch (\Exception $e) {
            Log::error('Error generating PDF: ' . $e->getMessage());
            throw $e;
        }
    }

    private function generatePDFHtml($workOrderData)
    {
        return view('pdf.maintenance-request', compact('workOrderData'))->render();
    }

    private function getWorkOrderById($id)
    {
        try {
            $response = Http::get($this->firebaseUrl);
            if ($response->successful()) {
                $data = $response->json();
                return $data['id'] === $id ? $data : null;
            }
        } catch (\Exception $e) {
            Log::error('Error fetching work order: ' . $e->getMessage());
        }
        
        return null;
    }

    private function fetchEquipmentData()
    {
        try {
            $response = Http::get('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData.json');
            
            if ($response->successful()) {
                $data = $response->json();
                return is_array($data) ? array_values($data) : [];
            }
        } catch (\Exception $e) {
            Log::error('Error fetching equipment data: ' . $e->getMessage());
        }
        
        return [];
    }

    private function fetchUsersData()
    {
        try {
            $response = Http::get('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/Globalusers/teamEmploy.json');
            
            if ($response->successful()) {
                $data = $response->json();
                return is_array($data) ? array_values($data) : [];
            }
        } catch (\Exception $e) {
            Log::error('Error fetching users data: ' . $e->getMessage());
        }
        
        return [];
    }

    // Test method to verify dynamic number generation
    public function testNumberGeneration()
    {
        try {
            // Get current database state
            $currentLastNumber = $this->getLastRequestNumber();
            $extractedNumber = $this->extractNumberFromRequestNumber($currentLastNumber);
            
            // Test with different countries and projects
            $testCases = [
                ['country' => 'Zambia', 'project' => 'KOB'],
                ['country' => 'DRC', 'project' => 'LUB'],
                ['country' => 'Other', 'project' => 'ABC'],
            ];

            $results = [];
            foreach ($testCases as $case) {
                $requestNumber = $this->generateRequestNumber($case['country'], $case['project']);
                $results[] = [
                    'country' => $case['country'],
                    'project' => $case['project'],
                    'generated_number' => $requestNumber
                ];
            }

            // Test Firebase update (without actually updating)
            $testUpdateNumber = "ZAM-KOB-RM-999";
            $updateTest = $this->testFirebaseUpdate($testUpdateNumber);

            return response()->json([
                'success' => true,
                'message' => 'Number generation test completed',
                'database_info' => [
                    'current_last_number' => $currentLastNumber,
                    'extracted_number' => $extractedNumber,
                    'next_number' => $extractedNumber + 1,
                    'firebase_url' => $this->lastRequestNumberUrl
                ],
                'firebase_update_test' => $updateTest,
                'test_results' => $results
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Test failed: ' . $e->getMessage(),
                'error_details' => $e->getTraceAsString()
            ], 500);
        }
    }

    // Test Firebase update without actually updating
    private function testFirebaseUpdate($testNumber)
    {
        try {
            Log::info('Testing Firebase update connection', [
                'url' => $this->lastRequestNumberUrl,
                'test_number' => $testNumber
            ]);
            
            // Test the connection by making a GET request first
            $getResponse = Http::get($this->lastRequestNumberUrl);
            
            return [
                'get_status' => $getResponse->status(),
                'get_successful' => $getResponse->successful(),
                'current_value' => $getResponse->json(),
                'url_accessible' => true
            ];
        } catch (\Exception $e) {
            return [
                'url_accessible' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    // Manual test method to update Firebase counter
    public function testUpdateCounter()
    {
        try {
            // Get current value
            $currentValue = $this->getLastRequestNumber();
            $extractedNumber = $this->extractNumberFromRequestNumber($currentValue);
            $nextNumber = $extractedNumber + 1;
            
            // Generate test number
            $testNumber = "ZAM-KOB-RM-" . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);
            
            // Update Firebase
            $this->updateLastRequestNumber($testNumber);
            
            // Verify the update
            $updatedValue = $this->getLastRequestNumber();
            
            return response()->json([
                'success' => true,
                'message' => 'Counter update test completed',
                'before_update' => $currentValue,
                'test_number' => $testNumber,
                'after_update' => $updatedValue,
                'update_successful' => $updatedValue === $testNumber
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Counter update test failed: ' . $e->getMessage()
            ], 500);
        }
    }
}
