<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Http;
use App\Models\User;

class AuthController extends Controller
{
    /**
     * Show the login form
     */
    public function showLoginForm()
    {
        // If user is already logged in, redirect to dashboard
        if (Auth::check()) {
            return redirect()->route('admin.dashboard');
        }
        
        return view('auth.sign-in');
    }

    /**
     * Handle login request - MongoDB Authentication
     * Supports both email and employeeNumber login
     */
    public function login(Request $request)
    {
        // Get login identifier (can be email or employeeNumber)
        $loginIdentifier = $request->input('loginIdentifier') ?: $request->input('email') ?: $request->input('employeeNumber');
        
        // Validate the request
        $validator = Validator::make($request->all(), [
            'loginIdentifier' => 'required|string',
            'password' => 'required|min:6'
        ], [
            'loginIdentifier.required' => 'Please enter your email address or employee number.',
            'loginIdentifier.string' => 'Email address or employee number must be valid.',
            'password.required' => 'Password is required.',
            'password.min' => 'Password must be at least 6 characters.'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput($request->except('password'));
        }

        // Determine if loginIdentifier is email or employeeNumber
        $email = null;
        $employeeNumber = null;
        
        if (filter_var($loginIdentifier, FILTER_VALIDATE_EMAIL)) {
            // It's an email
            $email = $loginIdentifier;
        } else {
            // It's an employee number
            $employeeNumber = $loginIdentifier;
        }
        
        $password = $request->input('password');
        $remember = $request->has('remember');

        // Get MongoDB login URL before try block so it's available in catch blocks
        $mongoLoginUrl = env('MONGODB_WEBPROADMIN_LOGIN_URL', 'https://api.titandrillingzm.com:6050/api/webproadmin/login');
        
        // Build request payload - send email or employeeNumber
        $loginPayload = [
            'password' => $password
        ];
        
        if (!empty($email)) {
            $loginPayload['email'] = $email;
        }
        
        if (!empty($employeeNumber)) {
            $loginPayload['employeeNumber'] = $employeeNumber;
        }

        try {
            // Log the request for debugging
            \Log::info('Attempting MongoDB login', [
                'url' => $mongoLoginUrl,
                'has_email' => !empty($email),
                'has_employeeNumber' => !empty($employeeNumber),
                'email' => $email ?: null,
                'employeeNumber' => $employeeNumber ?: null
            ]);
            
            // Make HTTP request with SSL verification disabled (for self-signed certificates)
            // This is needed for many live servers with self-signed SSL certificates
            $response = Http::timeout(15)
                ->retry(2, 100) // Retry 2 times with 100ms delay
                ->withoutVerifying() // Disable SSL verification for self-signed certs
                ->post($mongoLoginUrl, $loginPayload);

            if ($response->successful() && $response->json('success')) {
                $mongoUser = $response->json('user');
                
                // Find or create user in local database for session management
                $user = User::where('email', $mongoUser['email'])->first();
                
                if (!$user) {
                    // Create local user for session
                    $user = User::create([
                        'name' => $mongoUser['name'],
                        'email' => $mongoUser['email'],
                        'password' => Hash::make($password), // Store hashed password
                        'role' => $mongoUser['role'] ?? 'WebProAdmin',
                        'email_verified_at' => now()
                    ]);
                } else {
                    // Update user info from MongoDB
                    $user->update([
                        'name' => $mongoUser['name'],
                        'role' => $mongoUser['role'] ?? 'WebProAdmin'
                    ]);
                }
                
                // Log the user in
                Auth::login($user, $remember);
                $request->session()->regenerate();
                
                // Store MongoDB user data in session
                $request->session()->put('mongo_user_data', $mongoUser);
                $request->session()->put('mongo_user_uid', $mongoUser['uid']);
                
                // Log successful login
                \Log::info('User logged in successfully via MongoDB', [
                    'email' => $email ?: 'N/A',
                    'employeeNumber' => $employeeNumber ?: 'N/A',
                    'loginIdentifier' => $loginIdentifier,
                    'uid' => $mongoUser['uid'],
                    'role' => $mongoUser['role'],
                    'ip' => $request->ip(),
                    'user_agent' => $request->userAgent()
                ]);

                return redirect()->intended(route('admin.dashboard'))
                    ->with('success', 'Welcome back! You have been logged in successfully.');
            }

            // Log failed login attempt
            \Log::warning('Failed MongoDB login attempt', [
                'email' => $email,
                'employeeNumber' => $employeeNumber,
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'mongo_response' => $response->json()
            ]);

            $errorField = !empty($email) ? 'email' : 'employeeNumber';
            return redirect()->back()
                ->withErrors([$errorField => 'Invalid email/employee number or password. Please try again.'])
                ->withInput($request->except('password'));

        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            // Connection timeout or unreachable server
            \Log::error('MongoDB authentication connection error', [
                'email' => $email ?: null,
                'employeeNumber' => $employeeNumber ?: null,
                'url' => $mongoLoginUrl,
                'error' => $e->getMessage(),
                'error_type' => 'ConnectionException',
                'error_code' => method_exists($e, 'getCode') ? $e->getCode() : 'N/A',
                'suggestion' => 'Check if MongoDB authentication service is running on port 6050'
            ]);

            $errorField = !empty($email) ? 'email' : 'employeeNumber';
            $errorMessage = 'Cannot connect to authentication service. Please verify the MongoDB login service is running on port 6050.';
            
            return redirect()->back()
                ->withErrors([$errorField => $errorMessage])
                ->withInput($request->except('password'));
                
        } catch (\Illuminate\Http\Client\RequestException $e) {
            // HTTP request error (non-200 response)
            $response = $e->response;
            \Log::error('MongoDB authentication HTTP error', [
                'email' => $email ?: null,
                'employeeNumber' => $employeeNumber ?: null,
                'url' => $mongoLoginUrl,
                'http_status' => $response ? $response->status() : 'N/A',
                'error' => $e->getMessage(),
                'response_body' => $response ? $response->body() : 'N/A'
            ]);

            $errorField = !empty($email) ? 'email' : 'employeeNumber';
            $errorMessage = 'Authentication service returned an error. Please verify your credentials and try again.';
            
            return redirect()->back()
                ->withErrors([$errorField => $errorMessage])
                ->withInput($request->except('password'));
                
        } catch (\Exception $e) {
            // Log exception with more details
            \Log::error('MongoDB authentication error', [
                'email' => $email ?: null,
                'employeeNumber' => $employeeNumber ?: null,
                'url' => $mongoLoginUrl,
                'error' => $e->getMessage(),
                'error_class' => get_class($e),
                'error_file' => $e->getFile(),
                'error_line' => $e->getLine(),
                'trace' => substr($e->getTraceAsString(), 0, 1000) // Limit trace length
            ]);

            $errorField = !empty($email) ? 'email' : 'employeeNumber';
            
            // Provide more specific error message based on error type
            $errorMessage = 'Authentication service is currently unavailable. Please try again later.';
            $errorMsg = $e->getMessage();
            
            if (str_contains($errorMsg, 'Connection') || str_contains($errorMsg, 'timeout') || str_contains($errorMsg, 'Failed to connect')) {
                $errorMessage = 'Cannot connect to authentication service. Please verify the MongoDB login service is running on port 6050.';
            } elseif (str_contains($errorMsg, 'SSL') || str_contains($errorMsg, 'certificate') || str_contains($errorMsg, 'TLS')) {
                $errorMessage = 'SSL/TLS connection error. Please contact administrator.';
            } elseif (str_contains($errorMsg, 'resolve') || str_contains($errorMsg, 'DNS')) {
                $errorMessage = 'Cannot resolve authentication service hostname. Please check your network connection.';
            }
            
            return redirect()->back()
                ->withErrors([$errorField => $errorMessage])
                ->withInput($request->except('password'));
        }
    }

    /**
     * Handle logout request - MongoDB Logout
     */
    public function logout(Request $request)
    {
        $mongoUserUid = $request->session()->get('mongo_user_uid');
        
        // Log logout
        \Log::info('User logged out', [
            'user_id' => Auth::id(),
            'email' => Auth::user()->email ?? 'Unknown',
            'mongo_uid' => $mongoUserUid,
            'ip' => $request->ip()
        ]);

        // Notify MongoDB API about logout
        if ($mongoUserUid) {
            try {
                $mongoLogoutUrl = env('MONGODB_WEBPROADMIN_LOGOUT_URL', 'https://api.titandrillingzm.com:6050/api/webproadmin/logout');
                
                Http::timeout(5)->post($mongoLogoutUrl, [
                    'uid' => $mongoUserUid
                ]);
            } catch (\Exception $e) {
                \Log::warning('MongoDB logout notification failed', [
                    'uid' => $mongoUserUid,
                    'error' => $e->getMessage()
                ]);
            }
        }

        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('admin.login')
            ->with('success', 'You have been logged out successfully.');
    }

    /**
     * Create a default admin user if it doesn't exist
     */
    public function createDefaultAdmin()
    {
        $adminExists = User::where('email', 'admin@titan.com')->exists();
        
        if (!$adminExists) {
            User::create([
                'name' => 'Admin User',
                'email' => 'admin@titan.com',
                'password' => Hash::make('admin123'),
                'role' => 'admin',
                'email_verified_at' => now()
            ]);
            
            \Log::info('Default admin user created', [
                'email' => 'admin@titan.com',
                'password' => 'admin123'
            ]);
            
            return true;
        }
        
        return false;
    }

    /**
     * Handle password change request
     */
    public function changePassword(Request $request)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'new_password' => 'required|min:8'
        ], [
            'new_password.required' => 'New password is required.',
            'new_password.min' => 'New password must be at least 8 characters.'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ], 422);
        }

        $user = Auth::user();

        // Update password
        $user->password = Hash::make($request->new_password);
        $user->save();

        // Log password change
        \Log::info('User changed password', [
            'user_id' => $user->id,
            'email' => $user->email,
            'ip' => $request->ip()
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully!'
        ]);
    }
}
