# Request Maintenance MongoDB Migration Guide

## Overview
This document outlines the complete migration of the Request Maintenance functionality from Firebase to MongoDB, including data storage and email notification systems.

---

## 🔄 Migration Summary

### What Changed
- **REMOVED**: All Firebase Realtime Database connections
- **ADDED**: MongoDB API integration
- **UPDATED**: Data structure to match MongoDB schema
- **ENHANCED**: Email notification system via dedicated API

---

## 📊 MongoDB Data Storage

### Database Structure
```
Database: titan_drilling
Collection: GlobalrequestMaintenance
```

### Document Schema
```javascript
{
  _id: String,                    // MongoDB ID
  id: String,                     // Request ID (timestamp-based)
  adminUid: String,               // Creator's UID
  requestNumber: String,          // Format: ZAM-KAN-RM-001
  workOrderTitle: String,         // Request title
  requestType: String,            // General, Preventive, Breakdown, etc.
  priority: String,               // Low, Medium, High, Emergency
  description: String,            // Request description/memo
  equipment: {                    // Equipment details
    _id: String,
    id: String,
    equipmentName: String,
    equipmentNumber: String,
    model: String,
    serialNumber: String,
    mainCategory: String,
    status: String,
    country: String,
    project: String
  },
  assignTo: [                     // Array of assigned users
    {
      _id: String,
      fullName: String,
      name: String,
      email: String,
      phoneNumber: String,
      role: String
    }
  ],
  country: String,                // Zambia or DRC
  project: String,                // Project name
  tasks: [                        // Array of tasks
    {
      id: String,
      title: String,
      description: String,
      note: String,
      status: String,             // Initiated, On Hold, Completed, Cancelled
      createdAt: Date,
      updatedAt: Date,
      assignees: [                // Task-specific assignees
        {
          id: String,
          fullName: String,
          employeeNumber: String,
          role: String,
          profilePic: {
            url: String,
            path: String
          }
        }
      ]
    }
  ],
  status: String,                 // Requested, In Progress, Completed, etc.
  createdAt: Date,
  updatedAt: Date,
  userId: String,                 // Creator's user ID
  isArchived: Boolean             // Soft delete flag
}
```

---

## 🔌 MongoDB API Endpoints

### Base URL
```
http://168.231.113.219:5009
```

### Available Endpoints

#### 1. Get All Requests
```http
GET /request-maintenance
```
**Response:**
```json
{
  "success": true,
  "count": 10,
  "data": [...]
}
```

#### 2. Create New Request
```http
POST /request-maintenance
Content-Type: application/json

{
  "_id": "1234567890",
  "requestNumber": "ZAM-KAN-RM-001",
  "workOrderTitle": "Engine Repair",
  ...
}
```
**Response:**
```json
{
  "success": true,
  "data": {...}
}
```

#### 3. Get Request by ID
```http
GET /request-maintenance/:id
```

#### 4. Update Request
```http
PUT /request-maintenance/:id
Content-Type: application/json

{
  "status": "In Progress",
  ...
}
```

#### 5. Delete Request (Soft Delete)
```http
DELETE /request-maintenance/:id
```

---

## 📧 Email Notification System

### Notification API
```
Base URL: http://168.231.113.219:5022
Endpoint: POST /api/maintenance-notifications/new-request
```

### How It Works

#### 1. **Automatic Notification Trigger**
When a maintenance request is created, the system automatically sends notifications to:
- **ProAdmins**: All ProAdmin users (no filtering)
- **Supervisors**: Filtered by country AND project
- **Managers**: Filtered by country AND project
  - HrManager
  - Workshop Manager
  - Maintenance Manager
  - FinanceManager
  - Project Manager
  - Site Manager

#### 2. **Notification Types**
The system sends THREE types of notifications:

**a) MongoDB Notifications** (In-app)
- Saved to MongoDB `Notifications` collection
- Accessible via port 5017
- Includes notification ID, title, body, data

**b) Push Notifications** (Mobile)
- Sent via Firebase Cloud Messaging (FCM)
- Delivered to user's mobile devices
- Handled by port 5016

**c) Email Notifications**
- Sent to recipient emails
- Confirmation email to requester
- HTML formatted with company branding

#### 3. **Email Content**

**For Recipients:**
```
Subject: New Maintenance Request: [REQUEST_NUMBER]

Content:
- 🔧 New Maintenance Request
- Priority indicator (if High/Emergency)
- Request details (number, title, priority, equipment, etc.)
- Equipment information
- Country & Project
- Tasks list
- Requester information
```

**For Requester (Confirmation):**
```
Subject: Maintenance Request Submitted: [REQUEST_NUMBER]

Content:
- ✅ Request Submitted Successfully
- Request details
- Confirmation that request was sent to supervisors/managers
```

#### 4. **Notification Request Format**
```javascript
{
  "maintenanceData": {
    "_id": "1234567890",
    "id": "1234567890",
    "requestNumber": "ZAM-KAN-RM-001",
    "workOrderTitle": "Engine Repair",
    "equipmentName": "CAT 320D",
    "priority": "High",
    "country": "Zambia",
    "project": "Kansanshi",
    "tasksCount": 3,
    "description": "Engine requires urgent repair"
  },
  "createdByName": "John Doe",
  "requesterEmail": "john.doe@titandrillingzm.com"
}
```

#### 5. **Notification Response**
```json
{
  "success": true,
  "message": "Notifications sent for maintenance request: ZAM-KAN-RM-001",
  "results": {
    "mongodb": {
      "success": true,
      "saved": 15,
      "error": null
    },
    "push": {
      "success": true,
      "sent": 12,
      "failed": 3,
      "error": null
    },
    "emails": {
      "success": true,
      "sent": 14,
      "failed": 1,
      "total": 15,
      "error": null
    },
    "requesterEmail": {
      "success": true,
      "error": null
    }
  },
  "recipients": {
    "proAdmins": 5,
    "supervisors": 6,
    "managers": 4,
    "total": 15
  }
}
```

---

## 🔧 Implementation Details

### Frontend Changes

#### File: `Create_New_Request_Maintenance.blade.php`

**1. Request Number Generation**
```javascript
// NOW: Fetches from MongoDB
async function updateRequestNumber() {
    const response = await fetch('http://168.231.113.219:5009/request-maintenance');
    const result = await response.json();
    
    // Find highest request number and increment
    let maxNumber = 0;
    result.data.forEach(req => {
        const num = extractNumberFromRequestNumber(req.requestNumber);
        if (num > maxNumber) maxNumber = num;
    });
    
    const nextNumber = maxNumber + 1;
    requestNumber = generateRequestNumber(country, project, nextNumber);
}
```

**2. Load Equipment**
```javascript
// NOW: Loads from MongoDB Equipment API (Port 5001)
async function loadVehiclesFromFirebase() {
    const response = await fetch('http://168.231.113.219:5001/equipment');
    const result = await response.json();
    // Process equipment data...
}
```

**3. Load Users**
```javascript
// NOW: Loads from MongoDB Users API (Port 5003)
async function loadUsersFromFirebase() {
    const response = await fetch('http://168.231.113.219:5003/users/all');
    const result = await response.json();
    // Process user data...
}
```

**4. Save Request**
```javascript
// NOW: Saves to MongoDB and triggers notifications
async function saveMaintenanceRequest() {
    // 1. Save to MongoDB
    const response = await fetch('http://168.231.113.219:5009/request-maintenance', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(maintenanceRequestData)
    });
    
    // 2. Send notifications
    await fetch('http://168.231.113.219:5022/api/maintenance-notifications/new-request', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(notificationData)
    });
    
    // 3. Generate PDF
    await generatePDF();
    
    // 4. Show success modal
    showSuccessModal();
}
```

#### File: `Request_Maintenance.blade.php`

**Load Requests**
```javascript
// NOW: Loads from MongoDB
async function loadMaintenanceRequests() {
    const response = await fetch('http://168.231.113.219:5009/request-maintenance');
    const result = await response.json();
    
    if (result.success && result.data) {
        displayMaintenanceRequests(result.data);
    }
}
```

---

## 🚀 Backend Services

### 1. Request Maintenance Handler (Port 5009)
**File:** `Appmongosetup/backend/GlobalRequestMaintenanceHandler.js`

**Responsibilities:**
- CRUD operations for maintenance requests
- MongoDB connection management
- Data validation
- User permissions validation

### 2. Maintenance Notifications (Port 5022)
**File:** `Appmongosetup/backend/EmailsandPushN/RequestMaintenanceNotifications.js`

**Responsibilities:**
- Email notifications (via Nodemailer)
- Push notifications (via port 5016)
- MongoDB notifications (via port 5017)
- Recipient filtering (by role, country, project)

### 3. Equipment API (Port 5001)
**File:** `Appmongosetup/backend/GlobalEquipmentJsonData.js`

**Responsibilities:**
- Equipment CRUD operations
- Equipment data retrieval

### 4. Users API (Port 5003)
**Responsibilities:**
- User data retrieval
- User authentication
- User permissions

---

## 📝 Testing Guide

### 1. Test Data Flow

**Step 1: Create Request**
```bash
# Open browser console on Create New Request Maintenance page
# Fill in form and submit
# Check console logs:
✅ Loading vehicles from MongoDB...
✅ Loading users from MongoDB...
✅ Maintenance request saved successfully to MongoDB
✅ Notifications sent
```

**Step 2: Verify MongoDB Storage**
```bash
# Check MongoDB directly
mongo titan_drilling --eval "db.GlobalrequestMaintenance.find().pretty()"
```

**Step 3: Verify Notifications**
```bash
# Check email inbox for:
- Recipient emails (supervisors, managers, ProAdmins)
- Requester confirmation email

# Check MongoDB notifications
mongo titan_drilling --eval "db.Notifications.find({type: 'request_maintenance_created'}).pretty()"
```

**Step 4: View Requests**
```bash
# Open Request Maintenance page
# Should see all requests loaded from MongoDB
```

### 2. Test API Endpoints

**Get All Requests:**
```bash
curl http://168.231.113.219:5009/request-maintenance
```

**Create Request:**
```bash
curl -X POST http://168.231.113.219:5009/request-maintenance \
  -H "Content-Type: application/json" \
  -d '{
    "_id": "test123",
    "requestNumber": "ZAM-KAN-RM-999",
    "workOrderTitle": "Test Request",
    ...
  }'
```

**Send Notifications:**
```bash
curl -X POST http://168.231.113.219:5022/api/maintenance-notifications/new-request \
  -H "Content-Type: application/json" \
  -d '{
    "maintenanceData": {...},
    "createdByName": "Test User",
    "requesterEmail": "test@example.com"
  }'
```

---

## 🔍 Troubleshooting

### Issue: Requests not loading
**Solution:**
1. Check MongoDB service: `systemctl status mongod`
2. Check API service: `pm2 status` or `pm2 logs 5009`
3. Verify MongoDB connection in backend logs
4. Check browser console for CORS errors

### Issue: Emails not sending
**Solution:**
1. Check notification service: `pm2 logs 5022`
2. Verify email credentials in `RequestMaintenanceNotifications.js`
3. Check recipient email addresses in MongoDB
4. Verify SMTP settings

### Issue: Request number not incrementing
**Solution:**
1. Check MongoDB connection
2. Verify request data format
3. Check console logs for number extraction
4. Manually verify request numbers in MongoDB

### Issue: Equipment/Users not loading
**Solution:**
1. Check Equipment API (port 5001): `pm2 logs 5001`
2. Check Users API (port 5003): `pm2 logs 5003`
3. Verify MongoDB collections: `GlobalEquipments`, `GlobalUsers`
4. Check CORS settings

---

## 📊 Monitoring & Logs

### View Service Logs
```bash
# Request Maintenance Handler
pm2 logs 5009

# Notifications Service
pm2 logs 5022

# Equipment API
pm2 logs 5001

# Users API
pm2 logs 5003
```

### MongoDB Queries
```javascript
// Count all requests
db.GlobalrequestMaintenance.count()

// Find requests by country
db.GlobalrequestMaintenance.find({country: "Zambia"})

// Find requests by status
db.GlobalrequestMaintenance.find({status: "Requested"})

// Find recent requests
db.GlobalrequestMaintenance.find().sort({createdAt: -1}).limit(10)
```

---

## 🔐 Security Considerations

1. **API Authentication**: Consider adding JWT/API key authentication
2. **User Validation**: Always validate user permissions before CRUD operations
3. **Email Privacy**: Use BCC for bulk emails
4. **Data Validation**: Validate all input data before MongoDB insertion
5. **Rate Limiting**: Implement rate limiting for API endpoints

---

## 📚 References

- MongoDB Documentation: https://docs.mongodb.com/
- Nodemailer Documentation: https://nodemailer.com/
- Express.js Documentation: https://expressjs.com/
- Mongoose Documentation: https://mongoosejs.com/

---

## ✅ Migration Checklist

- [x] Remove Firebase dependencies
- [x] Update request number generation (MongoDB)
- [x] Update equipment loading (MongoDB API)
- [x] Update users loading (MongoDB API)
- [x] Update save functionality (MongoDB API)
- [x] Update display page (MongoDB API)
- [x] Implement email notifications
- [x] Test create flow
- [x] Test display flow
- [x] Test email notifications
- [x] Update environment variables
- [x] Document changes

---

## 🎯 Next Steps

1. **Add User Authentication**: Replace hardcoded `cahj3jJqe5Z16pQdLGSWrNgm84o1` with actual logged-in user
2. **Implement View Request**: Add detail view functionality for requests
3. **Add Edit Functionality**: Allow updating existing requests
4. **Add Status Workflow**: Implement status change notifications
5. **Add File Attachments**: Allow uploading documents with requests
6. **Add Comments**: Implement commenting system for requests
7. **Add Reports**: Create dashboard and analytics

---

## 📞 Support

For issues or questions:
- Backend: Check `pm2 logs` for service errors
- Frontend: Check browser console for JavaScript errors
- MongoDB: Check MongoDB logs: `/var/log/mongodb/mongod.log`

---

**Last Updated:** November 3, 2025  
**Version:** 1.0.0  
**Migration Status:** ✅ Complete

