# Inspection Data Mapping Guide

## Firebase Data Structure (From React Native App)

```
/GlobalInspectionReport/admins/{userId}/{reportId}/
├── id: "timestamp_string"
├── step1/
│   ├── requestNumber: "ZAM-KOB-IR-001"
│   ├── country: "Zambia"
│   ├── project: "Kansanshi"
│   ├── inspector: "John Doe"
│   ├── selectedEquipment/
│   │   ├── equipmentName: "Toyota Hilux"
│   │   ├── model: "Hilux 2020"
│   │   ├── vehicleNumber: "ZAM-001"
│   │   ├── meterReading: 45000
│   │   ├── mainCategory: "Light Vehicle"
│   │   └── status: "Available"
│   ├── coordinates: "-15.4167, 28.2833"
│   ├── gpsAddress: "Kansanshi Mine, Solwezi"
│   ├── currentMeterReading: "45500"
│   └── createdAt: "2025-01-06T10:30:00.000Z"
├── step2/
│   ├── selections/
│   │   └── {itemKey}: "Okay" | "Faulty" | "Reject" | "N/A"
│   ├── selectedFiles/
│   │   └── {itemKey}: "file:///path/to/image.jpg"
│   ├── numericInputs/
│   │   └── {itemKey}: "123"
│   ├── notes/
│   │   └── {itemKey}: "Some note text"
│   ├── sections/
│   │   └── [{sectionName, subItems: [...]}]
│   ├── headerInputValues/
│   │   ├── date: "1/6/2025"
│   │   ├── rig: "Rig 5"
│   │   ├── model: "Hilux 2020"
│   │   ├── fleet_no: "ZAM-001"
│   │   ├── meter_reading: "45500"
│   │   └── equipment_name: "Toyota Hilux"
│   ├── footerInputValues/
│   │   └── {signatureFields}
│   ├── checklistTitle: "Light Vehicle Checklist"
│   ├── faultyItems: 3
│   └── timestamp: "2025-01-06T10:45:00.000Z"
└── step3/
    ├── faultyItems: 3
    ├── equipmentStatus: "Operational" | "Faulty" | "Down"
    ├── pdfDownloadUrl: "https://firebasestorage.../report.pdf"
    ├── signatureDownloadUrl: "https://firebasestorage.../signature.png"
    ├── signatureFields: {...}
    ├── signatureValues: {...}
    ├── individualSignatures: {...}
    └── createdAt: "2025-01-06T11:00:00.000Z"
```

## Correct Data Extraction in Laravel Blade

### 1. Request Number
```javascript
// ✅ CORRECT
const requestNumber = inspection.step1?.requestNumber || 'N/A';

// ❌ WRONG
const requestNumber = inspection.requestNumber;
```

### 2. Created Date
```javascript
// ✅ CORRECT - Check multiple sources in priority order
const createdAt = inspection.step3?.createdAt || 
                  inspection.step1?.createdAt || 
                  inspection.createdAt || 
                  new Date().toISOString();
```

### 3. Equipment Name
```javascript
// ✅ CORRECT
let equipmentName = 'N/A';
if (inspection.step1?.selectedEquipment) {
    const equipment = inspection.step1.selectedEquipment;
    equipmentName = equipment.equipmentName || 
                    equipment.vehicleNumber || 
                    equipment.model || 
                    'N/A';
}
```

### 4. Equipment Status
```javascript
// ✅ CORRECT
const equipmentStatus = inspection.step3?.equipmentStatus || 
                       inspection.step1?.selectedEquipment?.status || 
                       'N/A';
```

### 5. Location (Country)
```javascript
// ✅ CORRECT
const location = inspection.step1?.country || 'N/A';
```

### 6. Project
```javascript
// ✅ CORRECT
const project = inspection.step1?.project || 'N/A';
```

### 7. Inspector
```javascript
// ✅ CORRECT
const inspector = inspection.step1?.inspector || 'N/A';
```

### 8. Faulty Items Count
```javascript
// ✅ CORRECT - Check multiple locations
let faultyItems = 0;

// Priority 1: step3 (final calculated value)
if (inspection.step3?.faultyItems !== undefined) {
    faultyItems = inspection.step3.faultyItems;
}
// Priority 2: step2 (from inspection data)
else if (inspection.step2?.faultyItems !== undefined) {
    faultyItems = inspection.step2.faultyItems;
}
// Priority 3: Calculate from selections
else if (inspection.step2?.selections) {
    const selections = inspection.step2.selections;
    faultyItems = Object.values(selections).filter(value => 
        ['Faulty', 'Reject', 'Not Ok'].includes(value)
    ).length;
}

// Ensure it's a number
faultyItems = parseInt(faultyItems) || 0;
```

### 9. PDF URL
```javascript
// ✅ CORRECT
const pdfUrl = inspection.step3?.pdfDownloadUrl || null;
```

### 10. Corrective Action Status
```javascript
// ✅ CORRECT - Check if corrective action exists
const hasCorrectiveAction = inspection.correctiveAction?.pdfDownloadUrl ? true : false;
const correctiveActionNumber = inspection.correctiveActionNumber || 'N/A';
```

### 11. Work Order (Job Card) Status
```javascript
// ✅ CORRECT
const hasWorkOrder = inspection.workOrder?.id ? true : false;
const workOrderNumber = inspection.workOrder?.requestNumber || 'N/A';
```

## Common Issues and Solutions

### Issue 1: "Invalid Date" displayed
**Cause:** Incorrect date parsing from Firebase timestamp
**Solution:**
```javascript
function safeParseDate(dateString) {
    if (!dateString) return null;
    
    try {
        let date;
        
        // Check if it's a timestamp string or number
        if (/^\d+$/.test(String(dateString))) {
            date = new Date(parseInt(dateString));
        } else {
            // It's an ISO string
            date = new Date(dateString);
        }
        
        // Validate the date
        if (isNaN(date.getTime())) {
            return null;
        }
        
        return date;
    } catch (error) {
        console.error('Error parsing date:', error);
        return null;
    }
}

// Usage
const date = safeParseDate(inspection.step3?.createdAt);
const formattedDate = date ? date.toLocaleDateString('en-US', {
    month: 'short',
    day: 'numeric',
    year: 'numeric',
    hour: '2-digit',
    minute: '2-digit'
}) : 'Invalid Date';
```

### Issue 2: Equipment name shows "N/A" or "[object Object]"
**Cause:** Equipment data is nested or in wrong format
**Solution:**
```javascript
function extractEquipmentName(inspection) {
    if (!inspection.step1?.selectedEquipment) return 'N/A';
    
    const equipment = inspection.step1.selectedEquipment;
    
    // Handle if equipment is a string
    if (typeof equipment === 'string') {
        return equipment;
    }
    
    // Handle if equipment is an object
    if (typeof equipment === 'object') {
        return equipment.equipmentName || 
               equipment.vehicleNumber || 
               equipment.name || 
               equipment.model || 
               JSON.stringify(equipment).substring(0, 30) || 
               'N/A';
    }
    
    return 'N/A';
}
```

### Issue 3: Faulty items count is always 0
**Cause:** Looking in wrong location or calculating incorrectly
**Solution:**
```javascript
function calculateFaultyItems(inspection) {
    // Method 1: From step3 (final count)
    if (inspection.step3?.faultyItems !== undefined && 
        inspection.step3.faultyItems !== null) {
        return parseInt(inspection.step3.faultyItems) || 0;
    }
    
    // Method 2: From step2 (inspection data)
    if (inspection.step2?.faultyItems !== undefined && 
        inspection.step2.faultyItems !== null) {
        return parseInt(inspection.step2.faultyItems) || 0;
    }
    
    // Method 3: Calculate from selections
    if (inspection.step2?.selections) {
        const faultySelections = ['Faulty', 'Reject', 'Not Ok', 'Not OK'];
        let count = 0;
        
        for (const [key, value] of Object.entries(inspection.step2.selections)) {
            if (faultySelections.includes(value)) {
                count++;
            }
        }
        
        return count;
    }
    
    return 0;
}
```

### Issue 4: PDF link not working
**Cause:** Incorrect path construction or missing PDF URL
**Solution:**
```javascript
function getInspectionPdfUrl(inspection) {
    // Check if PDF URL exists in step3
    if (inspection.step3?.pdfDownloadUrl && 
        inspection.step3.pdfDownloadUrl.startsWith('http')) {
        return inspection.step3.pdfDownloadUrl;
    }
    
    // Construct Firebase path if needed
    if (inspection.adminUID && inspection.timestamp) {
        const firebasePath = `GlobalInspectionReport/admins/${inspection.adminUID}/${inspection.timestamp}/step3/pdfDownloadUrl.json`;
        return `https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/${firebasePath}`;
    }
    
    return null;
}
```

## Complete Updated JavaScript for inspection.blade.php

Replace the `renderTable()` function in your blade file with this improved version:

```javascript
function renderTable() {
    if (!elements.tableBody) return;
    
    const startIndex = (state.currentPage - 1) * 10;
    const endIndex = startIndex + 10;
    const pageData = state.filteredData.slice(startIndex, endIndex);
    
    if (pageData.length === 0) {
        showNoData();
        return;
    }
    
    elements.tableBody.innerHTML = pageData.map((inspection, index) => {
        // Serial number (highest first for newest reports)
        const serialNumber = state.totalRecords - ((state.currentPage - 1) * 10 + index);
        
        // Extract data using correct paths
        const requestNumber = inspection.step1?.requestNumber || 'N/A';
        const inspector = inspection.step1?.inspector || 'N/A';
        const location = inspection.step1?.country || 'N/A';
        const project = inspection.step1?.project || 'N/A';
        
        // Equipment name
        let equipmentName = 'N/A';
        if (inspection.step1?.selectedEquipment) {
            const eq = inspection.step1.selectedEquipment;
            equipmentName = eq.equipmentName || eq.vehicleNumber || eq.model || eq.name || 'N/A';
        }
        
        // Equipment status
        const equipmentStatus = inspection.step3?.equipmentStatus || 
                               inspection.step1?.selectedEquipment?.status || 
                               'N/A';
        
        // Faulty items count
        let faultyItems = 0;
        if (inspection.step3?.faultyItems !== undefined) {
            faultyItems = parseInt(inspection.step3.faultyItems) || 0;
        } else if (inspection.step2?.faultyItems !== undefined) {
            faultyItems = parseInt(inspection.step2.faultyItems) || 0;
        } else if (inspection.step2?.selections) {
            faultyItems = Object.values(inspection.step2.selections)
                .filter(v => ['Faulty', 'Reject', 'Not Ok'].includes(v)).length;
        }
        
        // Date formatting
        let formattedDate = 'N/A';
        const dateSource = inspection.step3?.createdAt || 
                          inspection.step1?.createdAt || 
                          inspection.timestamp;
        if (dateSource) {
            const date = safeParseDate(dateSource);
            if (date) {
                formattedDate = date.toLocaleDateString('en-US', {
                    month: 'short',
                    day: 'numeric',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
            }
        }
        
        // Corrective action
        const correctiveAction = inspection.correctiveAction?.pdfDownloadUrl ? 
            'Completed' : 'Pending';
        
        // Linked work order
        const linkedWo = inspection.workOrder?.requestNumber || 'Not linked';
        
        // Check if this is user's report
        const isUserReport = inspection.adminUID === 'OKNMRqRKmrPz96Izvl9OYEzeTFn2';
        const rowClass = isUserReport ? 'table-row user-report' : 'table-row';
        
        return `
            <tr class="${rowClass}" role="row">
                <td class="serial-number-cell" role="cell">
                    <span class="serial-number">${serialNumber}</span>
                </td>
                <td class="action-cell" role="cell">
                    <div class="dropdown">
                        <button class="action-btn" type="button" data-bs-toggle="dropdown">
                            Actions
                        </button>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="#" onclick="viewPdf('${inspection.id}')">View PDF</a></li>
                            <li><a class="dropdown-item" href="#" onclick="shareInspection('${inspection.id}')">Share</a></li>
                            <li><a class="dropdown-item" href="#" onclick="createWorkorder('${inspection.id}')">Create Job Card</a></li>
                            <li><a class="dropdown-item" href="#" onclick="correctiveAction('${inspection.id}')">Corrective Action</a></li>
                            <li class="dropdown-divider"></li>
                            <li><a class="dropdown-item danger" href="#" onclick="deleteInspection('${inspection.id}')">Delete</a></li>
                        </ul>
                    </div>
                </td>
                <td class="report-number">
                    <span class="report-id">${requestNumber}</span>
                    ${isUserReport ? '<br><small style="color: #16a34a; font-weight: bold;">✓ MY REPORT</small>' : ''}
                </td>
                <td class="date-cell">${formattedDate}</td>
                <td class="vehicle-cell">
                    <div class="vehicle-info">
                        <span class="vehicle-name">${equipmentName}</span>
                        <span class="vehicle-status">${equipmentStatus}</span>
                    </div>
                </td>
                <td class="location-cell">${location}</td>
                <td class="inspector-cell">${inspector}</td>
                <td class="fault-cell">
                    ${faultyItems > 0 ? `
                        <div class="fault-info">
                            <span class="fault-badge">${faultyItems} faults</span>
                        </div>
                    ` : '<span class="no-faults">No faults</span>'}
                </td>
                <td class="action-cell">
                    <span class="corrective-action">${correctiveAction}</span>
                </td>
                <td class="linked-cell">
                    ${linkedWo !== 'Not linked' ? 
                        `<span class="linked-badge">${linkedWo}</span>` : 
                        '<span class="not-linked">Not linked</span>'}
                </td>
            </tr>
        `;
    }).join('');
    
    // Show/hide pagination
    document.getElementById('paginationSection').style.display = 
        state.totalRecords > 10 ? 'block' : 'none';
}
```

## Testing Your Implementation

1. **Check Firebase Data Structure:**
   - Open browser console
   - Look for logs showing the raw Firebase data
   - Verify the structure matches the guide above

2. **Test Each Field:**
   - Request Number: Should show like "ZAM-KOB-IR-001"
   - Date: Should show formatted date, not "Invalid Date"
   - Equipment: Should show name, not "[object Object]"
   - Faults: Should show correct count, not always 0
   - PDF Link: Should open PDF when clicked

3. **Debug Console Logs:**
```javascript
// Add these logs to your renderTable() function
console.log('Sample inspection structure:', pageData[0]);
console.log('Step1 data:', pageData[0]?.step1);
console.log('Step2 data:', pageData[0]?.step2);
console.log('Step3 data:', pageData[0]?.step3);
console.log('Faulty items:', faultyItems);
console.log('Equipment:', equipmentName);
```

## Summary of Key Paths

| Field | Correct Path | Fallback Paths |
|-------|-------------|----------------|
| Request Number | `step1.requestNumber` | - |
| Date | `step3.createdAt` | `step1.createdAt`, `timestamp` |
| Equipment Name | `step1.selectedEquipment.equipmentName` | `vehicleNumber`, `model` |
| Equipment Status | `step3.equipmentStatus` | `step1.selectedEquipment.status` |
| Location | `step1.country` | - |
| Project | `step1.project` | - |
| Inspector | `step1.inspector` | - |
| Faulty Items | `step3.faultyItems` | `step2.faultyItems`, calculate from `step2.selections` |
| PDF URL | `step3.pdfDownloadUrl` | - |

