import React, { useState, useRef, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  ScrollView,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import DateTimePicker from '@react-native-community/datetimepicker';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs'; // For deleting the temporary PDF file
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import Toast from 'react-native-toast-message';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;

const CreateWorkOrder = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const scrollViewRef = useRef(null);
  const [workOrderTitle, setWorkOrderTitle] = useState('');
  const [memo, setMemo] = useState('');
  const [priority, setPriority] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [assignTo, setAssignTo] = useState([]);
  const [assignToModalVisible, setAssignToModalVisible] = useState(false);
  const [assignToSearchQuery, setAssignToSearchQuery] = useState('');
  const [teamMembers, setTeamMembers] = useState([]);
  const [startDate, setStartDate] = useState(null);
  const [dueDate, setDueDate] = useState(null);
  const [showStartDatePicker, setShowStartDatePicker] = useState(false);
  const [showDueDatePicker, setShowDueDatePicker] = useState(false);
  const [additionalDetails, setAdditionalDetails] = useState('');
  const [isAdditionalDetailsOpen, setIsAdditionalDetailsOpen] = useState(false);
  const [workOrderType, setWorkOrderType] = useState('General');
  const [workOrderTypeModalVisible, setWorkOrderTypeModalVisible] = useState(false);
  const [estimatedCost, setEstimatedCost] = useState('');
  const [estimatedLaborTime, setEstimatedLaborTime] = useState('');
  const [linkInspections, setLinkInspections] = useState([]); // Changed to array for multiple selections
  const [linkInspectionModalVisible, setLinkInspectionModalVisible] = useState(false);
  const [linkedWorkOrder, setLinkedWorkOrder] = useState(null);
  const [linkedWorkOrderModalVisible, setLinkedWorkOrderModalVisible] = useState(false);
  const [isCostDetailsOpen, setIsCostDetailsOpen] = useState(false);
  const [partMaterialCost, setPartMaterialCost] = useState('');
  const [totalLaborCost, setTotalLaborCost] = useState('');
  const [additionalCost, setAdditionalCost] = useState('');
  const [taxAmount, setTaxAmount] = useState('');
  const [taxPercentage, setTaxPercentage] = useState('');
  const [totalWoCost, setTotalWoCost] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('Task');
  const [addItemModalVisible, setAddItemModalVisible] = useState(false);
  const [newItemText, setNewItemText] = useState('');
  const [items, setItems] = useState([]);
  const [taskType, setTaskType] = useState('Repair');
  const [note, setNote] = useState('');
  const [tempAssignTo, setTempAssignTo] = useState([]);
  const [statusModalVisible, setStatusModalVisible] = useState(false);
  const [selectedItemId, setSelectedItemId] = useState(null);
  const [editModalVisible, setEditModalVisible] = useState(false);
  const [editItem, setEditItem] = useState(null);
  const [addLaborModalVisible, setAddLaborModalVisible] = useState(false);
  const [addPartsModalVisible, setAddPartsModalVisible] = useState(false);
  const [addAdditionalCostModalVisible, setAddAdditionalCostModalVisible] = useState(false);
  const [addRemarksModalVisible, setAddRemarksModalVisible] = useState(false);
  const [laborWorker, setLaborWorker] = useState('');
  const [laborCode, setLaborCode] = useState('');
  const [laborRate, setLaborRate] = useState('');
  const [laborHours, setLaborHours] = useState('');
  const [laborMinutes, setLaborMinutes] = useState('');
  const [laborNote, setLaborNote] = useState('');
  const [partsItemNumber, setPartsItemNumber] = useState('');
  const [partsPartNumber, setPartsPartNumber] = useState('');
  const [partsItemName, setPartsItemName] = useState('');
  const [partsQuantity, setPartsQuantity] = useState('');
  const [partsUnit, setPartsUnit] = useState('Piece');
  const [partsItemCost, setPartsItemCost] = useState('');
  const [partsSite, setPartsSite] = useState('');
  const [partsNote, setPartsNote] = useState('');
  const [additionalCostDescription, setAdditionalCostDescription] = useState('');
  const [additionalCostAmount, setAdditionalCostAmount] = useState('');
  const [remarksDescription, setRemarksDescription] = useState('');
  const [faultListModalVisible, setFaultListModalVisible] = useState(false);
  const [selectedFaults, setSelectedFaults] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [workOrderId, setWorkOrderId] = useState(null);
  const [savedPdfPath, setSavedPdfPath] = useState(null);
  const loadingAnimation = useRef(new Animated.Value(0)).current;
  const [requestNumber, setRequestNumber] = useState('WO1001');
  const [inspections, setInspections] = useState([]);
  const [workOrders, setWorkOrders] = useState([]);
  const [faultListData, setFaultListData] = useState([]);
  const [errors, setErrors] = useState({ title: '', equipment: '' });

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const priorityOptions = [
    { label: 'Low', image: require('../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../Images/adminhome/inspreport/Priority4.png') },
  ];

  const workOrderTypes = [
    { id: '1', name: 'General' },
    { id: '2', name: 'Maintenance' },
    { id: '3', name: 'Repair' },
    { id: '4', name: 'Installation' },
  ];

  const categories = ['Task', 'Parts', 'Labor', 'Additional Cost', 'Remarks'];
  const taskTypes = ['Repair', 'Replace'];
  const statuses = [
    { label: 'Initiated', color: '#A9A9A9' },
    { label: 'In Progress', color: '#FFD700' },
    { label: 'On Hold', color: '#FF4500' },
    { label: 'Complete', color: '#008000' },
  ];
  const units = ['Piece', 'Unit', 'Box'];

  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  useEffect(() => {
    if (route.params?.selectedEquipment) {
      setSelectedEquipment(route.params.selectedEquipment);
      setErrors({ ...errors, equipment: '' });
    }
    // If a selected report is passed (from Inspection_Report_Home), pre-select it
    if (route.params?.selectedReport) {
      setLinkInspections([{ id: route.params.selectedReport.id, name: route.params.selectedReport.step1?.requestNumber || 'Inspection Report' }]);
    }
  }, [route.params?.selectedEquipment, route.params?.selectedReport]);

  useEffect(() => {
    fetchInitialData();

    Animated.loop(
      Animated.timing(loadingAnimation, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();
  }, []);

 // Update the fetchInitialData function to ensure inspections include more fields
const fetchInitialData = async () => {
  try {
    const adminUid = await AsyncStorage.getItem('userUid');
    if (!adminUid) {
      showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
      navigation.navigate('AdminLoginScreen');
      return;
    }

    const teamMembersRef = database().ref(`/TeamMember/employ/${adminUid}`);
    teamMembersRef.on('value', (snapshot) => {
      const data = snapshot.val();
      if (data) {
        const teamList = Object.keys(data).map((key) => ({
          id: key,
          ...data[key],
        }));
        setTeamMembers(teamList);
      } else {
        setTeamMembers([]);
      }
    });

    const inspectionsRef = database().ref(`/users/${adminUid}/inspectionReports`);
    inspectionsRef.on('value', (snapshot) => {
      const data = snapshot.val();
      if (data) {
        const inspectionList = Object.keys(data).map((key) => ({
          id: key,
          name: data[key].step1?.requestNumber || 'Inspection Report',
          equipmentName: data[key].step1?.equipment?.equipmentName || 'N/A', // Add equipment name
          createdAt: data[key].createdAt || 'N/A', // Add creation date
          ...data[key],
        }));
        setInspections(inspectionList);
      } else {
        setInspections([]);
      }
    });

    const workOrdersRef = database().ref(`/workOrders/${adminUid}`);
    workOrdersRef.on('value', (snapshot) => {
      const data = snapshot.val();
      if (data) {
        const workOrderList = Object.keys(data)
          .filter((key) => key !== 'lastRequestNumber')
          .map((key) => ({
            id: key,
            ...data[key],
          }));
        setWorkOrders(workOrderList);
      } else {
        setWorkOrders([]);
      }
    });

    const faultsRef = database().ref(`/faults/${adminUid}`);
    faultsRef.on('value', (snapshot) => {
      const data = snapshot.val();
      if (data) {
        const faultList = Object.keys(data).map((key) => ({
          id: key,
          ...data[key],
        }));
        setFaultListData(faultList);
      } else {
        setFaultListData([]);
      }
    });

    const lastRequestRef = database().ref(`/workOrders/${adminUid}/lastRequestNumber`);
    lastRequestRef.once('value', (snapshot) => {
      const lastNumber = snapshot.val();
      if (lastNumber) {
        const nextNumber = parseInt(lastNumber.replace('WO', '')) + 1;
        setRequestNumber(`WO${nextNumber.toString().padStart(4, '0')}`);
      } else {
        setRequestNumber('WO1001');
      }
    });

    return () => {
      teamMembersRef.off('value');
      inspectionsRef.off('value');
      workOrdersRef.off('value');
      faultsRef.off('value');
    };
  } catch (error) {
    console.error('Error fetching initial data:', error);
    showToast('error', 'Error', 'Failed to load data.');
  }
};










  const validateFields = () => {
    let isValid = true;
    const newErrors = { title: '', equipment: '' };

    if (!workOrderTitle.trim()) {
      newErrors.title = 'Job Card Title is required';
      isValid = false;
    }
    if (!selectedEquipment) {
      newErrors.equipment = 'Equipment is required';
      isValid = false;
    }

    setErrors(newErrors);
    return isValid;
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath) => {
    try {
      const reference = storage().ref(storagePath);
      await reference.putFile(filePath);
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading file to Firebase Storage:', error);
      throw error;
    }
  };

  const generatePDFWithHtml = async (workOrderId) => {
    try {
      let taskRows = '';
      const taskItems = items.filter((item) => item.category === 'Task');
      if (taskItems.length === 0) {
        taskRows += `
          <tr>
            <td colspan="5">No tasks available</td>
          </tr>
        `;
      } else {
        taskItems.forEach((item, index) => {
          taskRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${item.text || 'N/A'}</td>
              <td>${item.type || 'N/A'}</td>
              <td>${item.status || 'N/A'}</td>
              <td>${item.note || 'N/A'}</td>
            </tr>
          `;
        });
      }

      let laborRows = '';
      const laborItems = items.filter((item) => item.category === 'Labor');
      if (laborItems.length === 0) {
        laborRows += `
          <tr>
            <td colspan="5">No labor items available</td>
          </tr>
        `;
      } else {
        laborItems.forEach((item, index) => {
          laborRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${item.worker || 'N/A'}</td>
              <td>${item.code || 'N/A'}</td>
              <td>${item.rate || 'N/A'}</td>
              <td>${item.hours || '0'}h ${item.minutes || '0'}m</td>
            </tr>
          `;
        });
      }

      let partsRows = '';
      const partsItems = items.filter((item) => item.category === 'Parts');
      if (partsItems.length === 0) {
        partsRows += `
          <tr>
            <td colspan="6">No parts available</td>
          </tr>
        `;
      } else {
        partsItems.forEach((item, index) => {
          partsRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${item.itemNumber || 'N/A'}</td>
              <td>${item.partNumber || 'N/A'}</td>
              <td>${item.itemName || 'N/A'}</td>
              <td>${item.quantity || '0'} ${item.unit || 'N/A'}</td>
              <td>${item.itemCost || 'N/A'}</td>
            </tr>
          `;
        });
      }

      let additionalCostRows = '';
      const additionalCostItems = items.filter((item) => item.category === 'Additional Cost');
      if (additionalCostItems.length === 0) {
        additionalCostRows += `
          <tr>
            <td colspan="2">No additional costs available</td>
          </tr>
        `;
      } else {
        additionalCostItems.forEach((item, index) => {
          additionalCostRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${item.description || 'N/A'}</td>
              <td>${item.amount || 'N/A'}</td>
            </tr>
          `;
        });
      }

      let remarksRows = '';
      const remarksItems = items.filter((item) => item.category === 'Remarks');
      if (remarksItems.length === 0) {
        remarksRows += `
          <tr>
            <td colspan="1">No remarks available</td>
          </tr>
        `;
      } else {
        remarksItems.forEach((item, index) => {
          remarksRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${item.description || 'N/A'}</td>
            </tr>
          `;
        });
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1.0" />
          <title>Job Card: ${workOrderTitle || 'Untitled'}</title>
          <style>
            body { font-family: Arial, sans-serif; padding: 20px; font-size: 12px; }
            .header, .section { border: 1px solid #000; padding: 10px; margin-bottom: 10px; }
            .header-table, .checklist-table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
            .header-table td, .checklist-table th, .checklist-table td { border: 1px solid #000; padding: 5px; text-align: left; }
            .title { text-align: center; font-weight: bold; font-size: 16px; margin: 20px 0; }
            .notes { font-size: 10px; margin-bottom: 10px; }
            .checklist-table th { background-color: #ddd; font-weight: bold; }
            .summary-section { margin-top: 20px; }
            .summary-section p { margin: 5px 0; }
          </style>
        </head>
        <body>
          <div class="header">
            <table class="header-table">
              <tr>
                <td rowspan="5"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" width="100"></td>
                <td><strong>Document Name:</strong> HSE-FOR-8.1.2 Job Card</td>
              </tr>
              <tr><td><strong>Document ID:</strong> HSE-029</td></tr>
              <tr><td><strong>Version Date:</strong> 05/08/2021</td></tr>
              <tr><td><strong>Version:</strong> 1</td></tr>
              <tr><td><strong>Approved:</strong> HSM</td></tr>
            </table>
          </div>

          <div class="title">JOB CARD</div>
          <div class="section">
            <strong>NO:</strong> ${requestNumber || 'WO1001'}<br />
            <strong>DATE:</strong> ${new Date().toLocaleDateString('en-US', { year: 'numeric', month: '2-digit', day: '2-digit' })}<br />
            <strong>EQUIPMENT:</strong> ${selectedEquipment?.equipmentName || selectedEquipment?.name || 'N/A'}<br />
            <strong>CREATED AT:</strong> ${new Date().toLocaleString()}
          </div>

          <div class="summary-section">
            <h3>Job Card Details</h3>
            <p><strong>Title:</strong> ${workOrderTitle || 'N/A'}</p>
            <p><strong>Memo:</strong> ${memo || 'N/A'}</p>
            <p><strong>Priority:</strong> ${priority || 'N/A'}</p>
            <p><strong>Assigned To:</strong> ${assignTo.map((a) => a.name).join(', ') || 'None'}</p>
            <p><strong>Start Date:</strong> ${startDate ? new Date(startDate).toLocaleString() : 'N/A'}</p>
            <p><strong>Due Date:</strong> ${dueDate ? new Date(dueDate).toLocaleString() : 'N/A'}</p>
            <p><strong>Job Card Type:</strong> ${workOrderType || 'N/A'}</p>
            <p><strong>Estimated Cost:</strong> ${estimatedCost || 'N/A'}</p>
            <p><strong>Estimated Labor Time:</strong> ${estimatedLaborTime || 'N/A'}</p>
            <p><strong>Linked Inspections:</strong> ${linkInspections.length > 0 ? linkInspections.map(i => i.name).join(', ') : 'None'}</p>
            <p><strong>Linked Job Card:</strong> ${linkedWorkOrder?.title || 'None'}</p>
          </div>

          <div class="summary-section">
            <h3>Cost Details</h3>
            <p><strong>Part & Material Cost:</strong> ${partMaterialCost || '0.00'}</p>
            <p><strong>Total Labor Cost:</strong> ${totalLaborCost || '0.00'}</p>
            <p><strong>Additional Cost:</strong> ${additionalCost || '0.00'}</p>
            <p><strong>Tax Amount:</strong> ${taxAmount || '0.00'} (${taxPercentage || '0'}%)</p>
            <p><strong>Total WO Cost:</strong> ${totalWoCost || '0.00'}</p>
          </div>

          <div class="summary-section">
            <h3>Tasks</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
                <th>Type</th>
                <th>Status</th>
                <th>Note</th>
              </tr>
              ${taskRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Labor</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Worker</th>
                <th>Code</th>
                <th>Rate</th>
                <th>Time</th>
              </tr>
              ${laborRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Parts</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Item No</th>
                <th>Part No</th>
                <th>Name</th>
                <th>Quantity</th>
                <th>Cost</th>
              </tr>
              ${partsRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Additional Costs</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
                <th>Amount</th>
              </tr>
              ${additionalCostRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Remarks</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
              </tr>
              ${remarksRows}
            </table>
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `JobCard_${requestNumber || workOrderId}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const saveWorkOrder = async () => {
    if (!validateFields()) {
      showToast('error', 'Error', 'Please fill all required fields.');
      return;
    }

    setIsLoading(true);
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        setIsLoading(false);
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      const workOrderId = Date.now().toString();
      setWorkOrderId(workOrderId);

      const workOrder = {
        id: workOrderId,
        title: workOrderTitle,
        memo,
        priority,
        equipment: selectedEquipment,
        assignTo,
        startDate: startDate ? startDate.toISOString() : null,
        dueDate: dueDate ? dueDate.toISOString() : null,
        additionalDetails,
        workOrderType,
        estimatedCost,
        estimatedLaborTime,
        linkInspections, // Store array of inspections
        linkedWorkOrder,
        partMaterialCost,
        totalLaborCost,
        additionalCost,
        taxAmount,
        taxPercentage,
        totalWoCost,
        items,
        createdAt: new Date().toISOString(),
        status: 'Initiated',
        requestNumber,
      };

      const pdfPath = await generatePDFWithHtml(workOrderId);
      if (!pdfPath) {
        throw new Error('PDF generation failed: No file path returned');
      }

      const storagePath = `workOrders/${adminUid}/JobCard_${requestNumber || workOrderId}.pdf`;
      const pdfDownloadUrl = await uploadFileToFirebaseStorage(pdfPath, storagePath);

      const workOrdersRef = database().ref(`/workOrders/${adminUid}/${workOrderId}`);
      await workOrdersRef.set({
        ...workOrder,
        pdfDownloadUrl,
      });

      const nextNumber = parseInt(requestNumber.replace('WO', '')) + 1;
      await database()
        .ref(`/workOrders/${adminUid}/lastRequestNumber`)
        .set(`WO${nextNumber.toString().padStart(4, '0')}`);

      setSavedPdfPath(pdfPath);
      setShowSuccessModal(true);
      showToast('success', 'Success', 'Job Card created successfully.');

      if (route.params?.onRefresh) {
        route.params.onRefresh();
      }

      await RNFS.unlink(pdfPath);
    } catch (error) {
      console.error('Error saving work order or generating PDF:', error);
      showToast('error', 'Error', `Failed to create job card: ${error.message}`);
    } finally {
      setIsLoading(false);
    }
  };

  const handleViewPDF = async () => {
    if (!workOrderId) {
      showToast('error', 'Error', 'Job Card ID not available.');
      return;
    }

    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      const workOrderRef = database().ref(`/workOrders/${adminUid}/${workOrderId}`);
      const snapshot = await workOrderRef.once('value');
      const workOrderData = snapshot.val();

      if (!workOrderData) {
        showToast('error', 'Error', 'Job Card not found in the database.');
        return;
      }

      const pdfUrl = workOrderData.pdfDownloadUrl;
      if (!pdfUrl) {
        showToast('error', 'Error', 'PDF URL not available in the Job Card.');
        return;
      }

      setShowSuccessModal(false);
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        onGoBack: () => {
          if (route.params?.onRefresh) {
            route.params.onRefresh();
          }
          navigation.goBack();
        },
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      showToast('error', 'Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const saveItem = () => {
    if (!newItemText.trim()) {
      showToast('error', 'Error', 'Task description is required.');
      return;
    }

    const newItem = {
      id: Date.now().toString(),
      category: selectedCategory,
      text: newItemText,
      type: taskType,
      note,
      assignTo: tempAssignTo,
      status: 'Initiated',
    };
    setItems([...items, newItem]);
    resetModalFields();
  };

  const saveLabor = () => {
    if (!laborWorker || !laborCode || !laborRate || !laborHours || !laborMinutes) {
      showToast('error', 'Error', 'All labor fields are required.');
      return;
    }

    const newLabor = {
      id: Date.now().toString(),
      category: 'Labor',
      worker: laborWorker,
      code: laborCode,
      rate: laborRate,
      hours: laborHours,
      minutes: laborMinutes,
      note: laborNote,
      status: 'Initiated',
    };
    setItems([...items, newLabor]);
    resetLaborFields();
  };

  const saveParts = () => {
    if (!partsItemNumber || !partsPartNumber || !partsItemName || !partsQuantity || !partsItemCost) {
      showToast('error', 'Error', 'All parts fields are required.');
      return;
    }

    const newPart = {
      id: Date.now().toString(),
      category: 'Parts',
      itemNumber: partsItemNumber,
      partNumber: partsPartNumber,
      itemName: partsItemName,
      quantity: partsQuantity,
      unit: partsUnit,
      itemCost: partsItemCost,
      site: partsSite,
      note: partsNote,
      status: 'Initiated',
    };
    setItems([...items, newPart]);
    resetPartsFields();
  };

  const saveAdditionalCost = () => {
    if (!additionalCostDescription || !additionalCostAmount) {
      showToast('error', 'Error', 'All additional cost fields are required.');
      return;
    }

    const newAdditionalCost = {
      id: Date.now().toString(),
      category: 'Additional Cost',
      description: additionalCostDescription,
      amount: additionalCostAmount,
      status: 'Initiated',
    };
    setItems([...items, newAdditionalCost]);
    resetAdditionalCostFields();
  };

  const saveRemarks = () => {
    if (!remarksDescription) {
      showToast('error', 'Error', 'Remark description is required.');
      return;
    }

    const newRemark = {
      id: Date.now().toString(),
      category: 'Remarks',
      description: remarksDescription,
      status: 'Initiated',
    };
    setItems([...items, newRemark]);
    resetRemarksFields();
  };

  const updateItemStatus = (itemId, newStatus) => {
    setItems(
      items.map((item) =>
        item.id === itemId ? { ...item, status: newStatus } : item
      )
    );
    setStatusModalVisible(false);
  };

  const deleteItem = (itemId) => {
    setItems(items.filter((item) => item.id !== itemId));
    setEditModalVisible(false);
  };

  const editItemSave = () => {
    if (!newItemText.trim()) {
      showToast('error', 'Error', 'Task description is required.');
      return;
    }

    if (editItem) {
      setItems(
        items.map((item) =>
          item.id === editItem.id
            ? { ...editItem, text: newItemText, note, assignTo: tempAssignTo }
            : item
        )
      );
      resetModalFields();
    }
  };

  const resetModalFields = () => {
    setNewItemText('');
    setTaskType('Repair');
    setNote('');
    setTempAssignTo([]);
    setAddItemModalVisible(false);
    setEditModalVisible(false);
    setEditItem(null);
  };

  const resetLaborFields = () => {
    setLaborWorker('');
    setLaborCode('');
    setLaborRate('');
    setLaborHours('');
    setLaborMinutes('');
    setLaborNote('');
    setAddLaborModalVisible(false);
  };

  const resetPartsFields = () => {
    setPartsItemNumber('');
    setPartsPartNumber('');
    setPartsItemName('');
    setPartsQuantity('');
    setPartsUnit('Piece');
    setPartsItemCost('');
    setPartsSite('');
    setPartsNote('');
    setAddPartsModalVisible(false);
  };

  const resetAdditionalCostFields = () => {
    setAdditionalCostDescription('');
    setAdditionalCostAmount('');
    setAddAdditionalCostModalVisible(false);
  };

  const resetRemarksFields = () => {
    setRemarksDescription('');
    setAddRemarksModalVisible(false);
  };

  const filteredTeamMembers = teamMembers.filter(
    (member) =>
      member.userId.toLowerCase().includes(assignToSearchQuery.toLowerCase()) ||
      member.name.toLowerCase().includes(assignToSearchQuery.toLowerCase()) ||
      member.role.toLowerCase().includes(assignToSearchQuery.toLowerCase())
  );

  const handlePrioritySelect = (selectedPriority) => {
    setPriority(selectedPriority);
  };

  const handleEquipmentSelect = (equipmentData) => {
    setSelectedEquipment(equipmentData);
    setErrors({ ...errors, equipment: '' });
  };

  const handleAssignToSelect = (item) => {
    const isSelected = tempAssignTo.some((selected) => selected.id === item.id);
    if (isSelected) {
      setTempAssignTo(tempAssignTo.filter((selected) => selected.id !== item.id));
    } else {
      setTempAssignTo([...tempAssignTo, item]);
    }
  };

  const handleSaveAssignTo = () => {
    setAssignTo(tempAssignTo);
    setAssignToModalVisible(false);
    setAssignToSearchQuery('');
  };

  const handleRemoveAssignTo = (itemToRemove) => {
    setAssignTo(assignTo.filter((item) => item.id !== itemToRemove.id));
  };

  const handleWorkOrderTypeSelect = (type) => {
    setWorkOrderType(type.name);
    setWorkOrderTypeModalVisible(false);
  };

  const handleInspectionSelect = (inspection) => {
    const isSelected = linkInspections.some((i) => i.id === inspection.id);
    if (isSelected) {
      setLinkInspections(linkInspections.filter((i) => i.id !== inspection.id));
    } else {
      setLinkInspections([...linkInspections, { id: inspection.id, name: inspection.name }]);
    }
  };

  const handleSaveInspections = () => {
    setLinkInspectionModalVisible(false);
  };

  const handleWorkOrderSelect = (workOrder) => {
    setLinkedWorkOrder(workOrder);
    setLinkedWorkOrderModalVisible(false);
  };

  const handleCategorySelect = (category) => {
    setSelectedCategory(category);
  };

  const formatDate = (date) => {
    if (!date) return 'Select Date';
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const handleFaultSelect = (fault) => {
    const isSelected = selectedFaults.some((f) => f.id === fault.id);
    if (isSelected) {
      setSelectedFaults(selectedFaults.filter((f) => f.id !== fault.id));
    } else {
      setSelectedFaults([...selectedFaults, fault]);
    }
  };

  const handleSaveFaults = () => {
    setFaultListModalVisible(false);
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>New Job Card</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.saveButtoncreate} onPress={saveWorkOrder}>
              <Text style={styles.saveButtonText0}>Create</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
        keyboardVerticalOffset={Platform.OS === 'ios' ? TOTAL_HEADER_HEIGHT : 0}
      >
        <ScrollView
          ref={scrollViewRef}
          keyboardShouldPersistTaps="always"
          showsVerticalScrollIndicator={false}
          style={styles.mainContent}
        >
          <View style={styles.formGroup}>
            <View style={styles.requestNumberRow}>
              <Text style={styles.requestNumberLabel}>Job Card #</Text>
              <Text style={styles.requestNumberValue}>{requestNumber}</Text>
            </View>
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>
              Job Card Title
              <Text style={styles.requiredError}> *</Text>
            </Text>
            <TextInput
              style={[styles.inputmain, errors.title && styles.inputError]}
              placeholder="Job Card Title"
              placeholderTextColor="#888"
              value={workOrderTitle}
              onChangeText={(text) => {
                setWorkOrderTitle(text);
                setErrors({ ...errors, title: text.trim() ? '' : 'Job Card Title is required' });
              }}
            />
            {errors.title && <Text style={styles.errorText}>{errors.title}</Text>}
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>Write Memo</Text>
            <TextInput
              style={[styles.inputmain]}
              placeholder="Check within a day"
              placeholderTextColor="#888"
              value={memo}
              onChangeText={setMemo}
              multiline
              numberOfLines={4}
            />
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>Priority</Text>
            <View style={styles.priorityContainer}>
              {priorityOptions.map((option) => {
                const isSelected = priority === option.label;
                return (
                  <View key={option.label} style={styles.priorityItem}>
                    <TouchableOpacity
                      style={[
                        styles.priorityButton,
                        isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                      ]}
                      onPress={() => handlePrioritySelect(option.label)}
                    >
                      <Image source={option.image} style={styles.priorityImage} />
                      <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                        {option.label}
                      </Text>
                    </TouchableOpacity>
                  </View>
                );
              })}
            </View>
          </View>

          <View style={[styles.formGroup, styles.dropdownContainer]}>
            <View style={styles.rowCentered}>
              <View style={styles.equipmentContainer}>
                <Text style={styles.dropdownLabel}>
                  Equipments
                  <Text style={styles.requiredError}> *</Text>
                </Text>
                {selectedEquipment && (
                  <Text style={styles.selectedEquipmentText}>
                    {selectedEquipment.equipmentName || selectedEquipment.name}
                  </Text>
                )}
                {errors.equipment && <Text style={styles.errorText}>{errors.equipment}</Text>}
              </View>
              <TouchableOpacity
                style={styles.selectButton}
                onPress={() =>
                  navigation.navigate('Select_EquipmentWo', {
                    source: 'CreateWorkOrder',
                    onSelect: handleEquipmentSelect,
                  })
                }
              >
                <Text style={styles.selectButtonText}>Select</Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View>
          </View>

          <View style={[styles.formGroup, styles.dropdownContainer]}>
            <View style={styles.assignToContainer}>
              <View style={styles.assignToHeader}>
                <Text style={styles.dropdownLabel}>Assign To</Text>
                <TouchableOpacity
                  style={styles.selectButton}
                  onPress={() => setAssignToModalVisible(true)}
                >
                  <Text style={styles.selectButtonText}>Select</Text>
                  <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                </TouchableOpacity>
              </View>
              {assignTo.length > 0 && (
                <View style={styles.selectedItemsContainer}>
                  <ScrollView
                    horizontal={true}
                    showsHorizontalScrollIndicator={false}
                    contentContainerStyle={styles.selectedItemsRow}
                  >
                    {assignTo.map((item, index) => (
                      <View key={index} style={styles.selectedItem}>
                        <Text style={styles.selectedItemText}>{item.name}</Text>
                        <TouchableOpacity
                          style={styles.closeButton}
                          onPress={() => handleRemoveAssignTo(item)}
                        >
                          <View style={styles.closeCircle}>
                            <MaterialIcons name="close" size={width * 0.04} color="#fff" />
                          </View>
                        </TouchableOpacity>
                      </View>
                    ))}
                  </ScrollView>
                </View>
              )}
            </View>
          </View>

          <View style={styles.formsubdat}>
            <View style={styles.formGroupdates0}>
              <View style={styles.row}>
                <Text style={styles.label}>Start Date</Text>
                <TouchableOpacity
                  style={styles.dateButton}
                  onPress={() => setShowStartDatePicker(true)}
                >
                  <Text style={styles.dateText}>{formatDate(startDate)}</Text>
                  <MaterialIcons
                    name="calendar-today"
                    size={width * 0.05}
                    color="#015185"
                  />
                </TouchableOpacity>
              </View>
            </View>

            <View style={styles.mainspace0}></View>

            <View style={styles.formGroupdates1}>
              <View style={styles.row}>
                <Text style={styles.label}>Due Date</Text>
                <TouchableOpacity
                  style={styles.dateButton}
                  onPress={() => setShowDueDatePicker(true)}
                >
                  <Text style={styles.dateText}>{formatDate(dueDate)}</Text>
                  <MaterialIcons
                    name="calendar-today"
                    size={width * 0.05}
                    color="#015185"
                  />
                </TouchableOpacity>
              </View>
            </View>
          </View>

          <View style={[styles.formGroup, styles.dropdownContaineradditional]}>
            <View style={styles.rowCentered}>
              <Text style={styles.label}>Additional Details</Text>
              <TouchableOpacity
                style={styles.selectButton}
                onPress={() => setIsAdditionalDetailsOpen(!isAdditionalDetailsOpen)}
              >
                <MaterialIcons
                  name={isAdditionalDetailsOpen ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
                  size={width * 0.05}
                  color="#015185"
                />
              </TouchableOpacity>
            </View>
            {isAdditionalDetailsOpen && (
              <View style={styles.additionalDetailsContainer}>
                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Job Card Type</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={() => setWorkOrderTypeModalVisible(true)}
                    >
                      <Text style={styles.selectButtonText}>{workOrderType}</Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  <View style={styles.wotclono}></View>
                </View>

                <View style={styles.formGroup}>
                  <TextInput
                    style={styles.input}
                    placeholder="Enter estimated cost"
                    placeholderTextColor="#888"
                    value={estimatedCost}
                    onChangeText={setEstimatedCost}
                    keyboardType="numeric"
                  />
                </View>

                <View style={styles.formGroup}>
                  <TextInput
                    style={styles.input}
                    placeholder="Enter labor time in hours"
                    placeholderTextColor="#888"
                    value={estimatedLaborTime}
                    onChangeText={setEstimatedLaborTime}
                    keyboardType="numeric"
                  />
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Link Inspection</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={() => setLinkInspectionModalVisible(true)}
                    >
                      <Text style={styles.selectButtonText}>
                        {linkInspections.length > 0
                          ? `${linkInspections.length} selected`
                          : 'Select'}
                      </Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  <View style={styles.wotclono}></View>
                  {linkInspections.length > 0 && (
                    <View style={styles.selectedItemsContainer}>
                      <ScrollView
                        horizontal={true}
                        showsHorizontalScrollIndicator={false}
                        contentContainerStyle={styles.selectedItemsRow}
                      >
                        {linkInspections.map((inspection, index) => (
                          <View key={index} style={styles.selectedItem}>
                            <Text style={styles.selectedItemText}>{inspection.name}</Text>
                            <TouchableOpacity
                              style={styles.closeButton}
                              onPress={() =>
                                setLinkInspections(
                                  linkInspections.filter((i) => i.id !== inspection.id)
                                )
                              }
                            >
                              <View style={styles.closeCircle}>
                                <MaterialIcons name="close" size={width * 0.04} color="#fff" />
                              </View>
                            </TouchableOpacity>
                          </View>
                        ))}
                      </ScrollView>
                    </View>
                  )}
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Job Card</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={() => setLinkedWorkOrderModalVisible(true)}
                    >
                      <Text style={styles.selectButtonText}>
                        {linkedWorkOrder ? linkedWorkOrder.title : 'Select'}
                      </Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  <View style={styles.wotclono}></View>
                </View>
              </View>
            )}
          </View>

          <View style={[styles.formGroup, styles.dropdownContainerCostd]}>
            <View style={styles.rowCentered}>
              <Text style={styles.label}>Cost Details</Text>
              <TouchableOpacity
                style={styles.selectButton}
                onPress={() => setIsCostDetailsOpen(!isCostDetailsOpen)}
              >
                <MaterialIcons
                  name={isCostDetailsOpen ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
                  size={width * 0.05}
                  color="#015185"
                />
              </TouchableOpacity>
            </View>
            {isCostDetailsOpen && (
              <View style={styles.additionalDetailsContainer}>
                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Part & Material Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={partMaterialCost}
                      onChangeText={setPartMaterialCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Total Labor Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={totalLaborCost}
                      onChangeText={setTotalLaborCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Additional Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={additionalCost}
                      onChangeText={setAdditionalCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Tax Amount</Text>
                    <View style={styles.taxContainer}>
                      <TextInput
                        style={[styles.inputcost0tax, styles.taxInput]}
                        placeholder="0.00"
                        placeholderTextColor="#888"
                        value={taxAmount}
                        onChangeText={setTaxAmount}
                        keyboardType="numeric"
                      />
                      <Text style={styles.taxLabel}>Tax(%)</Text>
                      <TextInput
                        style={[styles.inputcost0tax, styles.taxInput]}
                        placeholder="0.00"
                        placeholderTextColor="#888"
                        value={taxPercentage}
                        onChangeText={setTaxPercentage}
                        keyboardType="numeric"
                      />
                    </View>
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Total WO Cost</Text>
                    <TextInput
                      style={[styles.input, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={totalWoCost}
                      onChangeText={setTotalWoCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>
              </View>
            )}
          </View>

          <View style={[styles.formGroup, styles.dropdownContainer]}>
            <Text style={styles.label}>Tasks & Items</Text>
            <View style={styles.tabContainer}>
              <ScrollView
                horizontal
                showsHorizontalScrollIndicator={false}
                contentContainerStyle={styles.tabRow}
              >
                {categories.map((category, index) => (
                  <TouchableOpacity
                    key={index}
                    style={[
                      styles.tabButton,
                      selectedCategory === category && styles.tabButtonActive,
                    ]}
                    onPress={() => handleCategorySelect(category)}
                  >
                    <Text
                      style={[
                        styles.tabText,
                        selectedCategory === category && styles.tabTextActive,
                      ]}
                    >
                      {category}
                    </Text>
                  </TouchableOpacity>
                ))}
              </ScrollView>
            </View>

            <View style={styles.itemsContainer}>
              <FlatList
                data={items.filter((item) => item.category === selectedCategory)}
                keyExtractor={(item) => item.id}
                renderItem={({ item }) => (
                  <View style={styles.itemRow}>
                    <View style={styles.itemContent}>
                      {item.category === 'Task' && (
                        <>
                          <Text style={styles.itemText}>{item.text}</Text>
                          <View style={styles.itemDetails}>
                            <Text style={styles.itemType}>{item.type}</Text>
                            <TouchableOpacity
                              style={[
                                styles.statusButton,
                                {
                                  backgroundColor:
                                    statuses.find((s) => s.label === item.status)?.color ||
                                    '#A9A9A9',
                                },
                              ]}
                              onPress={() => {
                                setSelectedItemId(item.id);
                                setStatusModalVisible(true);
                              }}
                            >
                              <Text style={styles.statusButtonText}>{item.status}</Text>
                            </TouchableOpacity>
                            <TouchableOpacity
                              style={styles.threeDotsButton}
                              onPress={() => {
                                setSelectedItemId(item.id);
                                setEditModalVisible(true);
                                setEditItem(item);
                                setNewItemText(item.text);
                                setTaskType(item.type);
                                setNote(item.note || '');
                                setTempAssignTo(item.assignTo || []);
                              }}
                            >
                              <MaterialIcons name="more-vert" size={width * 0.05} color="#015185" />
                            </TouchableOpacity>
                          </View>
                          <Text style={styles.itemNote}>Note: {item.note || 'N/A'}</Text>
                        </>
                      )}
                      {item.category === 'Labor' && (
                        <>
                          <Text style={styles.itemText}>Worker: {item.worker}</Text>
                          <Text style={styles.itemText}>Code: {item.code}</Text>
                          <Text style={styles.itemText}>Rate: {item.rate}</Text>
                          <Text style={styles.itemText}>
                            Hours: {item.hours} Min: {item.minutes}
                          </Text>
                          <Text style={styles.itemNote}>Note: {item.note || 'N/A'}</Text>
                        </>
                      )}
                      {item.category === 'Parts' && (
                        <>
                          <Text style={styles.itemText}>Item No: {item.itemNumber}</Text>
                          <Text style={styles.itemText}>Part No: {item.partNumber}</Text>
                          <Text style={styles.itemText}>Name: {item.itemName}</Text>
                          <Text style={styles.itemText}>
                            Qty: {item.quantity} {item.unit}
                          </Text>
                          <Text style={styles.itemText}>Cost: {item.itemCost}</Text>
                          <Text style={styles.itemText}>Site: {item.site || 'N/A'}</Text>
                          <Text style={styles.itemNote}>Note: {item.note || 'N/A'}</Text>
                        </>
                      )}
                      {item.category === 'Additional Cost' && (
                        <>
                          <Text style={styles.itemText}>Desc: {item.description}</Text>
                          <Text style={styles.itemText}>Amount: {item.amount}</Text>
                        </>
                      )}
                      {item.category === 'Remarks' && (
                        <>
                          <Text style={styles.itemText}>Desc: {item.description}</Text>
                        </>
                      )}
                    </View>
                  </View>
                )}
                ListEmptyComponent={
                  <Text style={styles.emptyItemsText}>
                    No items added for {selectedCategory || 'this category'}
                  </Text>
                }
                style={styles.itemsFlatList}
                contentContainerStyle={styles.itemsFlatListContent}
              />
            </View>

            <TouchableOpacity
              style={styles.addTaskButton}
              onPress={() => {
                if (selectedCategory === 'Task') setAddItemModalVisible(true);
                else if (selectedCategory === 'Labor') setAddLaborModalVisible(true);
                else if (selectedCategory === 'Parts') setAddPartsModalVisible(true);
                else if (selectedCategory === 'Additional Cost') setAddAdditionalCostModalVisible(true);
                else if (selectedCategory === 'Remarks') setAddRemarksModalVisible(true);
              }}
            >
              <Text style={styles.addTaskText}>+ Add {selectedCategory || 'Task'}</Text>
            </TouchableOpacity>
          </View>

          <Modal
            visible={faultListModalVisible}
            transparent
            animationType="slide"
            onRequestClose={() => setFaultListModalVisible(false)}
          >
            <View style={styles.modalOverlay}>
              <View style={[styles.modalContent, styles.faultListModal]}>
                <Text style={styles.faultListTitle}>Equipments fault list</Text>

                <FlatList
                  data={faultListData}
                  keyExtractor={(item) => item.id}
                  renderItem={({ item }) => (
                    <TouchableOpacity
                      style={styles.faultItem}
                      onPress={() => handleFaultSelect(item)}
                    >
                      <View style={styles.faultItemLeft}>
                        <Text style={styles.faultName}>{item.name}</Text>
                        <Text style={styles.faultTime}>{item.time}</Text>
                      </View>
                      <View style={styles.faultItemRight}>
                        <Text style={styles.faultCode}>{item.code}</Text>
                        {selectedFaults.some((f) => f.id === item.id) && (
                          <MaterialIcons name="check" size={24} color="#015185" />
                        )}
                      </View>
                    </TouchableOpacity>
                  )}
                  ListEmptyComponent={
                    <Text style={styles.defaultText}>No faults found</Text>
                  }
                  style={styles.faultList}
                />

                <View style={styles.faultListButtons}>
                  <TouchableOpacity
                    style={[styles.faultListActionButton, styles.closeButton]}
                    onPress={() => setFaultListModalVisible(false)}
                  >
                    <Text style={styles.faultListButtonText}>Close</Text>
                  </TouchableOpacity>
                  <TouchableOpacity
                    style={[styles.faultListActionButton, styles.saveButton]}
                    onPress={handleSaveFaults}
                  >
                    <Text style={[styles.faultListButtonText, styles.saveButtonText]}>Save</Text>
                  </TouchableOpacity>
                </View>
              </View>
            </View>
          </Modal>

          <View style={{ height: width * 0.2 }} />
        </ScrollView>
      </KeyboardAvoidingView>

      <Modal visible={isLoading} transparent animationType="fade">
        <View style={styles.loadingModalOverlayCustom}>
          <View style={styles.loadingModalContentCustom}>
            <Text style={styles.loadingModalTitleCustom}>Please Wait</Text>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[
                styles.loadingIconCustom,
                { transform: [{ rotate: rotateInterpolate }] },
              ]}
            />
            <Text style={styles.loadingModalTextCustom}>Work Order generating</Text>
          </View>
        </View>
      </Modal>

      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.successModalOverlayCustom}>
          <View style={styles.successModalContentCustom}>
            <Text style={styles.successModalTitleCustom}>Success</Text>
            <MaterialIcons name="check" size={56} color="#1A4C8B" style={styles.successModalCheckIconCustom} />
            <Text style={styles.successModalTextCustom}>Work Order Generated</Text>
            <View style={styles.successModalButtonsCustom}>
              <TouchableOpacity
                style={styles.successModalButtonCustom}
                onPress={() => {
                  setShowSuccessModal(false);
                  if (route.params?.onRefresh) route.params.onRefresh();
                  navigation.goBack();
                }}
              >
                <Text style={styles.successModalButtonTextCustom}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.successModalButtonCustom}
                onPress={handleViewPDF}
              >
                <Text style={styles.successModalButtonTextCustom}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={assignToModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAssignToModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, styles.assignToModal]}>
            <Text style={styles.modalTitle}>Assign to</Text>

            <View style={styles.searchContainer}>
              <MaterialIcons name="search" size={24} color="#666" style={styles.searchIcon} />
              <TextInput
                style={styles.assignToSearchInput}
                placeholder="Search..."
                placeholderTextColor="#888"
                value={assignToSearchQuery}
                onChangeText={setAssignToSearchQuery}
              />
            </View>

            <FlatList
              data={filteredTeamMembers}
              keyExtractor={(item) => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.assignToOption}
                  onPress={() => {
                    handleAssignToSelect(item);
                    setLaborWorker(item.name);
                  }}
                >
                  <View style={styles.assignToUserInfo}>
                    <View style={styles.userIdRow}>
                      <Text style={styles.userIdLabel}>User id</Text>
                      <Text style={styles.userIdValue}>{item.userId}</Text>
                    </View>
                    <View style={styles.nameRow}>
                      <Text style={styles.nameLabel}>Name</Text>
                      <Text style={styles.nameValue}>{item.name}</Text>
                    </View>
                  </View>
                  <View style={styles.roleContainer}>
                    <Text style={styles.roleText}>{item.role}</Text>
                  </View>
                  {tempAssignTo.some((selected) => selected.id === item.id) && (
                    <View style={styles.checkmarkContainer}>
                      <MaterialIcons name="check-circle" size={24} color="#28a745" />
                    </View>
                  )}
                </TouchableOpacity>
              )}
              ListEmptyComponent={<Text style={styles.noResultsText}>No team members found</Text>}
              style={styles.assignToList}
            />

            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setAssignToModalVisible(false)}
              >
                <Text style={styles.modalButtonText}>Close</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.modalButton, styles.saveButton]}
                onPress={handleSaveAssignTo}
              >
                <Text style={styles.saveButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={workOrderTypeModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setWorkOrderTypeModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Job Card Type</Text>
            <FlatList
              data={workOrderTypes}
              keyExtractor={(item) => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => handleWorkOrderTypeSelect(item)}
                >
                  <View style={styles.optionRow}>
                    <Text style={styles.modalOptionText}>{item.name}</Text>
                  </View>
                </TouchableOpacity>
              )}
              ListEmptyComponent={<Text style={styles.defaultText}>No types found</Text>}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setWorkOrderTypeModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

     // Update the "Link Inspection" Modal
<Modal
  visible={linkInspectionModalVisible}
  transparent
  animationType="slide"
  onRequestClose={() => setLinkInspectionModalVisible(false)}
>
  <View style={styles.modalOverlay}>
    <View style={[styles.modalContent, styles.inspectionModal]}>
      <Text style={styles.modalTitle}>Select Inspections</Text>
      <FlatList
        data={inspections}
        keyExtractor={(item) => item.id}
        renderItem={({ item }) => (
          <TouchableOpacity
            style={styles.inspectionOption}
            onPress={() => handleInspectionSelect(item)}
          >
            <View style={styles.inspectionInfo}>
              <View style={styles.inspectionRow}>
                <Text style={styles.inspectionLabel}>Request #</Text>
                <Text style={styles.inspectionValue}>{item.name}</Text>
              </View>
              <View style={styles.inspectionRow}>
                <Text style={styles.inspectionLabel}>Equipment</Text>
                <Text style={styles.inspectionValue}>{item.equipmentName}</Text>
              </View>
              <View style={styles.inspectionRow}>
                <Text style={styles.inspectionLabel}>Date</Text>
                <Text style={styles.inspectionValue}>
                  {item.createdAt !== 'N/A'
                    ? new Date(item.createdAt).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'short',
                        day: 'numeric',
                      })
                    : 'N/A'}
                </Text>
              </View>
            </View>
            {linkInspections.some((i) => i.id === item.id) && (
              <View style={styles.checkmarkContainer}>
                <MaterialIcons name="check-circle" size={24} color="#28a745" />
              </View>
            )}
          </TouchableOpacity>
        )}
        ListEmptyComponent={<Text style={styles.defaultText}>No inspections found</Text>}
        style={styles.inspectionList}
      />
      <View style={styles.modalButtonContainer}>
        <TouchableOpacity
          style={[styles.modalButton, styles.cancelButton]}
          onPress={() => setLinkInspectionModalVisible(false)}
        >
          <Text style={styles.modalButtonText}>Close</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.modalButton, styles.saveButton]}
          onPress={handleSaveInspections}
        >
          <Text style={styles.saveButtonText}>Save</Text>
        </TouchableOpacity>
      </View>
    </View>
  </View>
</Modal>

      <Modal
        visible={linkedWorkOrderModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setLinkedWorkOrderModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Job Card</Text>
            <FlatList
              data={workOrders}
              keyExtractor={(item) => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => handleWorkOrderSelect(item)}
                >
                  <View style={styles.optionRow}>
                    <Text style={styles.modalOptionText}>{item.title}</Text>
                  </View>
                </TouchableOpacity>
              )}
              ListEmptyComponent={<Text style={styles.defaultText}>No Job Cards found</Text>}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setLinkedWorkOrderModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addItemModalVisible || editModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => {
          if (addItemModalVisible) setAddItemModalVisible(false);
          if (editModalVisible) setEditModalVisible(false);
        }}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, styles.updateTaskModal]}>
            <Text style={[styles.modalTitle, styles.addTaskTitle]}>
              {editModalVisible ? 'Update Task' : 'Add Task'}
            </Text>

            <View style={styles.taskFormGroup}>
              <Text style={styles.taskLabel}>Task</Text>
              <TextInput
                style={[styles.taskTextInput, styles.bottomBorderInput]}
                placeholder="Enter task"
                placeholderTextColor="#888"
                value={newItemText}
                onChangeText={setNewItemText}
              />
            </View>

            <View style={styles.taskFormGroup}>
              <View style={styles.typeRow}>
                <Text style={styles.taskLabel}>Type</Text>
                <View style={styles.typeButtonContainer}>
                  {taskTypes.map((type) => (
                    <TouchableOpacity
                      key={type}
                      style={[
                        styles.typeButton,
                        taskType === type && styles.typeButtonSelected,
                        type === 'Repair' && taskType === type && styles.repairButtonSelected,
                      ]}
                      onPress={() => setTaskType(type)}
                    >
                      <Text
                        style={[
                          styles.typeText,
                          taskType === type && styles.typeTextSelected,
                        ]}
                      >
                        {type}
                      </Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>
            </View>

            <View style={styles.taskFormGroup}>
              <View style={styles.assignToRow}>
                <Text style={styles.taskLabel}>Assign to</Text>
                <TouchableOpacity
                  style={styles.assignToButton}
                  onPress={() => setAssignToModalVisible(true)}
                >
                  <Text style={styles.assignToButtonText}>
                    {tempAssignTo.length > 0
                      ? tempAssignTo.map((t) => t.name).join(', ')
                      : 'Select'}
                  </Text>
                  <MaterialIcons name="keyboard-arrow-down" size={24} color="#015185" />
                </TouchableOpacity>
              </View>
            </View>

            <View style={styles.taskFormGroup}>
              <Text style={styles.taskLabel}>Note</Text>
              <TextInput
                style={[styles.taskTextInput, styles.bottomBorderInput]}
                placeholder="Enter note"
                placeholderTextColor="#888"
                value={note}
                onChangeText={setNote}
                multiline
              />
            </View>

            <View style={styles.taskFormGroup}>
              <Text style={styles.taskLabel}>Fault List</Text>
              <TouchableOpacity
                style={styles.faultListButton}
                onPress={() => setFaultListModalVisible(true)}
              >
                <Text style={styles.faultListButtonText}>
                  {selectedFaults.length > 0
                    ? `${selectedFaults.length} faults selected`
                    : 'Select faults'}
                </Text>
                <MaterialIcons name="keyboard-arrow-right" size={24} color="#015185" />
              </TouchableOpacity>
            </View>

            <View style={styles.taskButtonContainer}>
              <TouchableOpacity
                style={styles.taskCancelButton}
                onPress={() => {
                  if (addItemModalVisible) setAddItemModalVisible(false);
                  if (editModalVisible) setEditModalVisible(false);
                }}
              >
                <Text style={styles.taskButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.taskSaveButton}
                onPress={editModalVisible ? editItemSave : saveItem}
              >
                <Text style={styles.taskButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={statusModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setStatusModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Update Status</Text>
            <FlatList
              data={statuses}
              keyExtractor={(item) => item.label}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.statusOption}
                  onPress={() => updateItemStatus(selectedItemId, item.label)}
                >
                  <View style={[styles.statusColor, { backgroundColor: item.color }]} />
                  <Text style={styles.statusOptionText}>{item.label}</Text>
                </TouchableOpacity>
              )}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setStatusModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addLaborModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddLaborModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, styles.laborModal]}>
            <Text style={[styles.modalTitle, styles.laborTitle]}>Add Labor Details</Text>
            <View style={styles.formGroup}>
              <View style={styles.laborRow}>
                <Text style={styles.laborLabel}>Select Worker</Text>
                <TouchableOpacity
                  style={styles.laborSelectButton}
                  onPress={() => setAssignToModalVisible(true)}
                >
                  <Text style={styles.laborSelectButtonText}>
                    {laborWorker || 'Select'}
                  </Text>
                  <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                </TouchableOpacity>
              </View>
            </View>
            <View style={styles.formGroup}>
              <TextInput
                style={styles.laborInput}
                placeholder="Labor Code"
                placeholderTextColor="#888"
                value={laborCode}
                onChangeText={setLaborCode}
              />
            </View>
            <View style={styles.formGroup}>
              <TextInput
                style={styles.laborInput}
                placeholder="Rate"
                placeholderTextColor="#888"
                value={laborRate}
                onChangeText={setLaborRate}
                keyboardType="numeric"
              />
            </View>
            <View style={styles.formGroup}>
              <Text style={styles.laborSubLabel}>Work Hours</Text>
              <View style={styles.hoursContainer}>
                <TextInput
                  style={[styles.laborInput, styles.hoursInput]}
                  placeholder="Hours"
                  placeholderTextColor="#888"
                  value={laborHours}
                  onChangeText={setLaborHours}
                  keyboardType="numeric"
                />
                <TextInput
                  style={[styles.laborInput, styles.minutesInput]}
                  placeholder="Minutes"
                  placeholderTextColor="#888"
                  value={laborMinutes}
                  onChangeText={setLaborMinutes}
                  keyboardType="numeric"
                />
              </View>
            </View>
            <View style={styles.formGroup}>
              <TextInput
                style={[styles.laborInput, styles.noteInput]}
                placeholder="Note"
                placeholderTextColor="#888"
                value={laborNote}
                onChangeText={setLaborNote}
                multiline
              />
            </View>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalTextButton}
                onPress={() => setAddLaborModalVisible(false)}
              >
                <Text style={styles.modalTextButtonText}>CLOSE</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.modalTextButton}
                onPress={saveLabor}
              >
                <Text style={styles.modalTextButtonText}>SAVE</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addPartsModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddPartsModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, styles.partsModal]}>
            <Text style={[styles.modalTitle, styles.partsTitle]}>Add Part & Material</Text>
            <View style={styles.formGroup}>
              <TextInput
                style={styles.partsInput}
                placeholder="Item Number"
                placeholderTextColor="#888"
                value={partsItemNumber}
                onChangeText={setPartsItemNumber}
              />
            </View>
            <View style={styles.formGroup}>
              <TextInput
                style={styles.partsInput}
                placeholder="Part Number"
                placeholderTextColor="#888"
                value={partsPartNumber}
                onChangeText={setPartsPartNumber}
              />
            </View>
            <View style={styles.formGroup}>
              <TextInput
                style={styles.partsInput}
                placeholder="Item Name"
                placeholderTextColor="#888"
                value={partsItemName}
                onChangeText={setPartsItemName}
              />
            </View>
            <View style={styles.formGroup}>
              <View style={styles.quantityContainer}>
                <TextInput
                  style={[styles.partsInput, styles.quantityInput]}
                  placeholder="Quantity"
                  placeholderTextColor="#888"
                  value={partsQuantity}
                  onChangeText={setPartsQuantity}
                  keyboardType="numeric"
                />
                <TouchableOpacity
                  style={styles.unitSelector}
                  onPress={() => {
                    const nextUnit = units[(units.indexOf(partsUnit) + 1) % units.length];
                    setPartsUnit(nextUnit);
                  }}
                >
                  <Text style={styles.unitText}>{partsUnit}</Text>
                  <MaterialIcons name="arrow-drop-down" size={width * 0.06} color="#015185" />
                </TouchableOpacity>
              </View>
            </View>
            <View style={styles.formGroup}>
              
              <TextInput
                style={styles.partsInput}
                placeholder="Item Cost"
                placeholderTextColor="#888"
                value={partsItemCost}
                onChangeText={setPartsItemCost}
                keyboardType="numeric"
              />
            </View>
            <View style={styles.formGroup}>
              <TextInput
                style={styles.partsInput}
                placeholder="Site (Optional)"
                placeholderTextColor="#888"
                value={partsSite}
                onChangeText={setPartsSite}
              />
            </View>
            <View style={styles.formGroup}>
              <TextInput
                style={[styles.partsInput, styles.noteInput]}
                placeholder="Note (Optional)"
                placeholderTextColor="#888"
                value={partsNote}
                onChangeText={setPartsNote}
                multiline
              />
            </View>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalTextButton}
                onPress={() => setAddPartsModalVisible(false)}
              >
                <Text style={styles.modalTextButtonText}>CLOSE</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.modalTextButton}
                onPress={saveParts}
              >
                <Text style={styles.modalTextButtonText}>SAVE</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addAdditionalCostModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddAdditionalCostModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, styles.additionalCostModal]}>
            <Text style={[styles.modalTitle, styles.additionalCostTitle]}>Add Additional Cost</Text>
            <View style={styles.formGroup}>
              <TextInput
                style={styles.additionalCostInput}
                placeholder="Description"
                placeholderTextColor="#888"
                value={additionalCostDescription}
                onChangeText={setAdditionalCostDescription}
              />
            </View>
            <View style={styles.formGroup}>
              <TextInput
                style={styles.additionalCostInput}
                placeholder="Amount"
                placeholderTextColor="#888"
                value={additionalCostAmount}
                onChangeText={setAdditionalCostAmount}
                keyboardType="numeric"
              />
            </View>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalTextButton}
                onPress={() => setAddAdditionalCostModalVisible(false)}
              >
                <Text style={styles.modalTextButtonText}>CLOSE</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.modalTextButton}
                onPress={saveAdditionalCost}
              >
                <Text style={styles.modalTextButtonText}>SAVE</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addRemarksModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setAddRemarksModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, styles.remarksModal]}>
            <Text style={[styles.modalTitle, styles.remarksTitle]}>Add Remarks</Text>
            <View style={styles.formGroup}>
              <TextInput
                style={[styles.remarksInput, styles.noteInput]}
                placeholder="Description"
                placeholderTextColor="#888"
                value={remarksDescription}
                onChangeText={setRemarksDescription}
                multiline
              />
            </View>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalTextButton}
                onPress={() => setAddRemarksModalVisible(false)}
              >
                <Text style={styles.modalTextButtonText}>CLOSE</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.modalTextButton}
                onPress={saveRemarks}
              >
                <Text style={styles.modalTextButtonText}>SAVE</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <DatePicker
        modal
        open={showStartDatePicker}
        date={startDate || new Date()}
        onConfirm={(date) => {
          setShowStartDatePicker(false);
          setStartDate(date);
        }}
        onCancel={() => setShowStartDatePicker(false)}
        mode="datetime"
      />

      <DatePicker
        modal
        open={showDueDatePicker}
        date={dueDate || new Date()}
        onConfirm={(date) => {
          setShowDueDatePicker(false);
          setDueDate(date);
        }}
        onCancel={() => setShowDueDatePicker(false)}
        mode="datetime"
      />
    </View>
  );
};
          













const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  saveButtonText: {
    color: '#FFFFFF',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  formGroup: {
    marginBottom: width * 0.02,
  },
  mainspace0: {
    width: '100%',
    height: 0.1,
    backgroundColor: '#707173',
    color: '#707173',
  },
  formsubdat: {
    marginBottom: 14,
  },
  formGroupdates0: {
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 10,
    paddingVertical: 2,
    borderBottomWidth: 1,
    borderBottomColor: '#707173',
  },
  formGroupdates1: {
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 10,
    paddingVertical: 2,
  },
  dropdownContainer: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  dropdownContaineradditional: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  dropdownContainerCostd: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  label: {
     fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  dropdownLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  input: {
    borderBottomWidth: 1,
    borderColor: '#015185',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    fontSize: width * 0.035,
    color: '#333',
    paddingHorizontal: width * 0.01,
  },
  inputmain: {
     borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputSingleLine: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
  },
  hourInput: {
    flex: 1,
    marginRight: width * 0.01,
  },
  minuteInput: {
    flex: 1,
  },
  quantityInput: {
    flex: 2,
    marginRight: width * 0.01,
  },
  unitContainer: {
    flex: 1,
  },
  unitInput: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
  },
  multilineInput: {
    height: width * 0.09,
    textAlignVertical: 'top',
  },
  costInput: {
    width: width * 0.3,
    borderWidth: 1,
    borderColor: '#3481BC',
    borderRadius: 3,
    width: '55%',
  },
  taxContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  taxInput: {
    width: width * 0.15,
    marginRight: width * 0.02,
  },
  taxLabel: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.02,
  },
  additionalDetailsContainer: {
    marginTop: width * 0.025,
    paddingHorizontal: width * 0.03,
  },
  tabContainer: {
    marginVertical: width * 0.02,
  },
  tabRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  tabButton: {
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    marginRight: width * 0.02,
    borderBottomWidth: 2,
    borderBottomColor: 'transparent',
  },
  tabButtonActive: {
    borderBottomColor: '#015185',
  },
  tabText: {
    fontSize: width * 0.035,
    color: '#707173',
    fontWeight: '500',
  },
  tabTextActive: {
    color: '#015185',
    fontWeight: '600',
  },
  itemsContainer: {
    //marginVertical: width * 0.025,
    minHeight: width * 0.1,
  },
  itemsFlatList: {
    flexGrow: 0,
  },
  itemsFlatListContent: {
    paddingBottom: width * 0.025,
  },
  itemRow: {
    marginBottom: width * 0.02,
    padding: width * 0.015,
    borderRadius: width * 0.01,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  itemContent: {
    flexDirection: 'column',
  },
  itemText: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '600',
  },
  itemDetails: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginTop: width * 0.005,
  },
  itemType: {
    fontSize: width * 0.035,
    color: '#FF0000', // Red to match screenshot
    marginRight: width * 0.02,
  },
  itemNote: {
    fontSize: width * 0.035,
    color: '#707173',
    marginTop: width * 0.005,
  },
  statusButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.02,
    borderRadius: width * 0.01,
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: '600',
  },
  threeDotsButton: {
    padding: width * 0.01,
  },
  emptyItemsText: {
    fontSize: width * 0.035,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.025,
  },
  addTaskButton: {
  paddingVertical: width * 0.015,
  paddingHorizontal: width * 0.03,
  borderRadius: width * 0.015,
  backgroundColor: '#015185',
  alignSelf: 'flex-start',
  marginBottom: width * 0.04,
},
addTaskText: {
  color: '#fff',
  fontSize: width * 0.035,
  fontWeight: '600',
},
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: 15,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  row: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  rowCentered: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  assignToContainer: {
    flexDirection: 'column',
  },
  assignToHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  equipmentContainer: {
    flexDirection: 'column',
  },
  selectedEquipmentText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginTop: width * 0.01,
  },
  selectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.01,
  },
  selectButtonText: {
    color: '#015185',
    fontSize: width * 0.035,
    marginRight: width * 0.01,
  },
  selectedItemsContainer: {
    marginTop: width * 0.01,
  },
  selectedItemsRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectedItem: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e0e0e0',
    borderRadius: width * 0.05,
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.02,
    marginRight: width * 0.02,
    marginBottom: width * 0.01,
  },
  selectedItemText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginRight: width * 0.02,
  },
  closeButton: {
    padding: width * 0.005,
  },
  closeCircle: {
    width: width * 0.04,
    height: width * 0.04,
    borderRadius: (width * 0.04) / 2,
    backgroundColor: '#d32f2f',
    justifyContent: 'center',
    alignItems: 'center',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
  },
  dateText: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.02,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
    maxHeight: '80%',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  modalOption: {
    paddingVertical: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  statusOption: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.015,
  },
  statusColor: {
    width: width * 0.03,
    height: width * 0.03,
    borderRadius: (width * 0.03) / 2,
    marginRight: width * 0.02,
  },
  statusOptionText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  optionRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-start',
  },
  optionDetails: {
    flex: 1,
  },
  modalOptionText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  modalOptionSubText: {
    fontSize: width * 0.035,
    color: '#707173',
  },
  checkmarkContainer: {
    marginRight: width * 0.03,
  },
  roleContainer: {
    backgroundColor: '#28a745',
    borderRadius: width * 0.02,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.01,
  },
  roleText: {
    fontSize: width * 0.035,
    color: '#fff',
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: width * 0.025,
  },
  modalButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.02,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: width * 0.01,
  },
  modalTextButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
  },
  modalTextButtonText: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '600',
    textTransform: 'uppercase',
  },
  cancelButton: {
   backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  deleteButton: {
    backgroundColor: '#d32f2f',
  },
  modalButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  labelwot: {
    color: 'black',
    fontWeight: '400',
    fontSize: width * 0.032,
    color: '#707173',
    paddingHorizontal: 2,
  },
  wotclono: {
    borderBottomWidth: 1,
    borderColor: '#015185',
    width: '98%',
    height: 1,
    alignSelf: 'center',
    left: -1,
  },
  inputcost0: {
    paddingHorizontal: 10,
    paddingVertical: 5,
    borderRadius: 3,
  },
  inputcost0tax: {
    borderWidth: 1,
    borderColor: '#3481BC',
    paddingVertical: 4,
    paddingHorizontal: 4,
    borderRadius: 3,
  },
  typeContainer: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginBottom: width *    0.01,
  },
  typeButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.01,
    marginRight: width * 0.02,
  },
  typeButtonSelected: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  typeText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  typeTextSelected: {
    fontSize: width * 0.035,
    color: '#fff',
    fontWeight: '600',
  },
  searchInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.025,
  },
  updateTaskModal: {
    backgroundColor: '#fff',
    width: '80%',
    borderRadius: 5,
    padding: width * 0.04,
  },
  addTaskTitle: {
    fontSize: width * 0.045,
    color: '#015185',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  taskFormGroup: {
    marginBottom: width * 0.03,
  },
  taskLabel: {
    fontSize: width * 0.035,
    color: '#000',
    marginBottom: width * 0.01,
  },
  taskTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    backgroundColor: '#fff',
  },
  typeButtonContainer: {
    flexDirection: 'row',
    gap: 10,
  },
  repairButtonSelected: {
    backgroundColor: '#FFA500', // Orange/Amber color
    borderColor: '#FFA500',
  },
  assignToButton: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  assignToButtonText: {
    fontSize: width * 0.035,
    color: '#666',
  },
  assignToArrow: {
    fontSize: width * 0.04,
    color: '#015185',
  },
  taskButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: width * 0.04,
  },
  taskCancelButton: {
    backgroundColor: '#6c757d',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: 3,
  },
  taskSaveButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: 3,
  },
  taskButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
  },
  laborModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  laborTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  laborRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  laborLabel: {
    fontSize: width * 0.035,
    color: '#333',
  },
  laborSelectButton: {
    padding: width * 0.02,
  },
  laborInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  laborSubLabel: {
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.02,
  },
  hoursContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  hoursInput: {
    flex: 1,
    marginRight: width * 0.02,
  },
  minutesInput: {
    flex: 1,
    marginLeft: width * 0.02,
  },
  partsModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  partsTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  partsRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  partsLabel: {
    fontSize: width * 0.035,
    color: '#333',
  },
  partsSelectButton: {
    padding: width * 0.02,
  },
  partsInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  quantityContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  quantityInput: {
    flex: 2,
    marginRight: width * 0.02,
  },
  unitSelector: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  unitText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  remarksModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  remarksTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  remarksInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  descriptionInput: {
    height: width * 0.2,
    textAlignVertical: 'top',
  },
  typeRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  assignToRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  bottomBorderInput: {
    borderWidth: 0,
    borderBottomWidth: 1,
    borderColor: '#ddd',
  },
  assignToModal: {
    backgroundColor: '#fff',
    width: '90%',
    maxHeight: '80%',
    borderRadius: 10,
    padding: 20,
  },
  assignToModalTitle: {
    fontSize: 24,
    color: '#015185',
    fontWeight: '600',
    textAlign: 'center',
    marginBottom: 20,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#ccc',
  },
  searchIcon: {
    marginRight: 10,
  },
  assignToSearchInput: {
    flex: 1,
    paddingVertical: 10,
    fontSize: 16,
    color: '#333',
  },
  assignToList: {
    marginBottom: 15,
  },
  assignToOption: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  assignToUserInfo: {
    flex: 1,
  },
  userIdRow: {
    flexDirection: 'row',
    marginBottom: 4,
  },
  nameRow: {
    flexDirection: 'row',
  },
  userIdLabel: {
    width: 70,
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
  },
  nameLabel: {
    width: 70,
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
  },
  userIdValue: {
    fontSize: 16,
    color: '#666',
    marginLeft: 10,
  },
  nameValue: {
    fontSize: 16,
    color: '#666',
    marginLeft: 10,
  },
  technicianBadge: {
    backgroundColor: '#015185',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 5,
    marginLeft: 10,
  },
  technicianText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '500',
  },
  noResultsText: {
    textAlign: 'center',
    color: '#666',
    fontSize: 16,
    marginTop: 20,
  },
  // selectButton: {
    // backgroundColor: '#015185',
    // paddingVertical: 12,
    // borderRadius: 5,
    // marginTop: 10,
  // },
  // selectButtonText: {
  //   color: '#fff',
  //   fontSize: 16,
  //   fontWeight: '600',
  //   textAlign: 'center',
  // },
  faultListButton: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderColor: '#ddd',
    paddingVertical: 8,
    marginTop: 5,
  },
  faultListButtonText: {
    fontSize: 14,
    color: '#666',
  },
  faultListModal: {
    backgroundColor: '#fff',
    width: '90%',
    maxHeight: '80%',
    borderRadius: 10,
    padding: 20,
  },
  faultListTitle: {
    fontSize: 20,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 20,
  },
  faultList: {
    marginBottom: 15,
  },
  faultItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  faultItemLeft: {
    flex: 1,
  },
  faultName: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    marginBottom: 4,
  },
  faultTime: {
    fontSize: 12,
    color: '#666',
  },
  faultItemRight: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 10,
  },
  faultCode: {
    fontSize: 14,
    color: '#666',
  },
  faultListButtons: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: 10,
  },
  faultListActionButton: {
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  closeButton: {
    backgroundColor: '#f0f0f0',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  saveButtonText: {
    color: '#fff',
  },
  saveButtoncreate: {
    backgroundColor: '#FFFFFF',
    paddingVertical: 8,
    paddingHorizontal: 16,
    borderRadius: 4,
    marginRight: 15,
    marginTop: 5,
  },
  saveButtonText0: {
    color: '#015185',
    fontSize: 14,
    fontWeight: '500',
  },
  loadingModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(255, 255, 255, 0.9)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    backgroundColor: '#FFFFFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
  },
  loadingModalTitle: {
    fontSize: 20,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 20,
  },
  loadingIcon: {
    width: 40,
    height: 40,
    marginVertical: 15,
    tintColor: '#015185',
  },
  loadingModalText: {
    fontSize: 16,
    color: '#333',
    marginTop: 10,
  },
  additionalCostModal: {
    backgroundColor: '#FFFFFF',
    width: '90%',
    borderRadius: 5,
    padding: 20,
  },
  additionalCostTitle: {
    color: '#015185',
    fontSize: 18,
    fontWeight: '600',
    marginBottom: 20,
    textAlign: 'center',
  },
  additionalCostInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    padding: 12,
    marginBottom: 15,
    color: '#333',
    backgroundColor: '#FFFFFF',
    fontSize: 16,
  },
  descriptionInput: {
    height: 120,
    textAlignVertical: 'top',
    paddingTop: 12,
  },
  laborCodeInput: {
    height: 50,
  },
  additionalCostButtons: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: 10,
  },
  additionalCostButton: {
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
    minWidth: 80,
  },
  cancelButton: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  cancelButtonText: {
    color: '#FFFFFF',
    fontSize: 14,
    fontWeight: '500',
    textAlign: 'center',
  },
  saveButtonText: {
    color: '#FFFFFF',
    fontSize: 14,
    fontWeight: '500',
    textAlign: 'center',
  },
  successModalContent: {
    backgroundColor: '#FFFFFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '92%',
    maxWidth: width * 0.95,
  },
  successModalTitle: {
    fontSize: 24,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 15,
  },
  successIcon: {
    marginVertical: 15,
  },
  successModalText: {
    fontSize: 18,
    color: '#333',
    marginBottom: 20,
  },
  successModalButtons: {
    flexDirection: 'row',
    justifyContent: 'center',
    gap: 10,
    width: '100%',
  },
  successModalButton: {
    paddingVertical: 5,
    paddingHorizontal: 10,
    borderRadius: 1,
    minWidth: 100,
    alignItems: 'center',
  },
  backButton: {
    backgroundColor: '#015185',
  },
  viewPDFButton: {
    backgroundColor: '#015185',
  },
  successModalButtonText: {
    color: '#FFFFFF',
    fontSize: 16,
    fontWeight: '400',
  },
    requestNumberRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.02,
  },
    requestNumberLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
    requestNumberValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
   scrollContent: {
    flexGrow: 1,
    paddingBottom: height * 0.1,
  },
   keyboardAvoidingView: {
    flex: 1,
  },
  successModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 32,
    alignItems: 'center',
    width: 340,
    maxWidth: '90%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  successModalTitleCustom: {
    fontSize: 28,
    color: '#1A4C8B',
    fontWeight: '700',
    marginBottom: 8,
  },
  successModalCheckIconCustom: {
    marginVertical: 16,
  },
  successModalTextCustom: {
    fontSize: 22,
    color: '#1A4C8B',
    marginBottom: 24,
    fontWeight: '500',
  },
  successModalButtonsCustom: {
    flexDirection: 'row',
    justifyContent: 'center',
    gap: 16,
  },
  successModalButtonCustom: {
    backgroundColor: '#1A4C8B',
    borderRadius: 4,
    paddingVertical: 10,
    paddingHorizontal: 24,
    marginHorizontal: 8,
  },
  successModalButtonTextCustom: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  loadingModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 32,
    alignItems: 'center',
    width: 380,
    maxWidth: '90%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  loadingModalTitleCustom: {
    fontSize: 28,
    color: '#1A4C8B',
    fontWeight: '700',
    marginBottom: 16,
    textAlign: 'center',
  },
  loadingIconCustom: {
    width: 56,
    height: 56,
    marginVertical: 16,
    tintColor: '#1A4C8B',
  },
  loadingModalTextCustom: {
    fontSize: 20,
    color: '#222',
    marginTop: 8,
    textAlign: 'center',
  },
});


  export default CreateWorkOrder;