import React, { useState, useRef, useEffect, useMemo } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  ScrollView,
  KeyboardAvoidingView,
  Platform,
  ActivityIndicator,
  Alert,
  InteractionManager,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';

import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import JobCardAPI from '../ConfigMongo/JobCardMongoConfig.js';
import InspectionAPI from '../ConfigMongo/InspectionMongoConfig';
import DropDownPicker from 'react-native-dropdown-picker';
import DateTimePicker from '@react-native-community/datetimepicker';
import { getFilteredTeamMembers } from '../Utils/teamMemberUtils';
import { filterEquipmentByPermission } from '../Utils/normalizeEquipmentData';
import * as ElementDropdown from 'react-native-element-dropdown';
import OfflineWorkOrderHelper from './OfflineWorkOrderHelper';

const { width, height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;

// Helper for chunked state updates
const chunkArray = (array, chunkSize) => {
  const result = [];
  for (let i = 0; i < array.length; i += chunkSize) {
    result.push(array.slice(i, i + chunkSize));
  }
  return result;
};

const CreateWorkOrder = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const scrollViewRef = useRef(null);
  const [workOrderTitle, setWorkOrderTitle] = useState('');
  const [memo, setMemo] = useState('');
  const [priority, setPriority] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [assignTo, setAssignTo] = useState([]);


  const [taskStatusMenuVisibleId, setTaskStatusMenuVisibleId] = useState(null);



  const [teamMembers, setTeamMembers] = useState([]);
  const [startDate, setStartDate] = useState(null);
  const [dueDate, setDueDate] = useState(null);
  const [showStartDatePicker, setShowStartDatePicker] = useState(false);
  const [showDueDatePicker, setShowDueDatePicker] = useState(false);
  const [additionalDetails, setAdditionalDetails] = useState('');
  const [isAdditionalDetailsOpen, setIsAdditionalDetailsOpen] = useState(false);
  const [workOrderType, setWorkOrderType] = useState('General');
  const [workOrderTypeModalVisible, setWorkOrderTypeModalVisible] = useState(false);
  const [estimatedCost, setEstimatedCost] = useState('');
  const [estimatedLaborTime, setEstimatedLaborTime] = useState('');
  const [linkInspection, setLinkInspection] = useState(null);
  const [linkInspectionModalVisible, setLinkInspectionModalVisible] = useState(false);
  const [linkedWorkOrder, setLinkedWorkOrder] = useState(null);
  const [linkedWorkOrderModalVisible, setLinkedWorkOrderModalVisible] = useState(false);
  const [isCostDetailsOpen, setIsCostDetailsOpen] = useState(false);
  const [partMaterialCost, setPartMaterialCost] = useState('');
  const [totalLaborCost, setTotalLaborCost] = useState('');
  const [additionalCost, setAdditionalCost] = useState('');
  const [taxAmount, setTaxAmount] = useState('');
  const [taxPercentage, setTaxPercentage] = useState('');
  const [totalWoCost, setTotalWoCost] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('Task');
  const [addItemModalVisible, setAddItemModalVisible] = useState(false);
  const [newItemText, setNewItemText] = useState('');
  const [items, setItems] = useState([]);
  const [taskType, setTaskType] = useState('Repair');
  const [note, setNote] = useState('');
  const [tempAssignTo, setTempAssignTo] = useState([]);
  const [statusModalVisible, setStatusModalVisible] = useState(false);
  const [selectedItemId, setSelectedItemId] = useState(null);
  const [editModalVisible, setEditModalVisible] = useState(false);
  const [editItem, setEditItem] = useState(null);
  const [editMenuVisible, setEditMenuVisible] = useState(false);
  const [addLaborModalVisible, setAddLaborModalVisible] = useState(false);
  const [addPartsModalVisible, setAddPartsModalVisible] = useState(false);
  const [addAdditionalCostModalVisible, setAddAdditionalCostModalVisible] = useState(false);
  const [addRemarksModalVisible, setAddRemarksModalVisible] = useState(false);
  const [laborWorker, setLaborWorker] = useState('');
  const [laborCode, setLaborCode] = useState('');
  const [laborRate, setLaborRate] = useState('');
  const [laborHours, setLaborHours] = useState('');
  const [laborMinutes, setLaborMinutes] = useState('');
  const [laborNote, setLaborNote] = useState('');
  const [partsItemNumber, setPartsItemNumber] = useState('');
  const [partsPartNumber, setPartsPartNumber] = useState('');
  const [partsItemName, setPartsItemName] = useState('');
  const [partsQuantity, setPartsQuantity] = useState('');
  const [partsUnit, setPartsUnit] = useState('Piece');
  const [partsItemCost, setPartsItemCost] = useState('');
  const [partsSite, setPartsSite] = useState('');
  const [partsNote, setPartsNote] = useState('');
  const [additionalCostDescription, setAdditionalCostDescription] = useState('');
  const [additionalCostAmount, setAdditionalCostAmount] = useState('');
  const [remarksDescription, setRemarksDescription] = useState('');
  const [faultListModalVisible, setFaultListModalVisible] = useState(false);
  const [selectedFaults, setSelectedFaults] = useState([]);
  const [isLoading, setIsLoading] = useState(false); // Start as false
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [workOrderId, setWorkOrderId] = useState(null);
  const [savedPdfPath, setSavedPdfPath] = useState(null);
  const [requestNumber, setRequestNumber] = useState('');
  const [inspections, setInspections] = useState([]);
  const [workOrders, setWorkOrders] = useState([]);
  const [faultListData, setFaultListData] = useState([]);
  const [errors, setErrors] = useState({ title: '', equipment: '' });
  const [checklistItems, setChecklistItems] = useState([]);
  const [selectedTaskId, setSelectedTaskId] = useState(null);
  const [assignMenuVisible, setAssignMenuVisible] = useState(false);
  const [showUserSelectionModal, setShowUserSelectionModal] = useState(false);
  const [availableUsers, setAvailableUsers] = useState([]);
  const [taskAssignments, setTaskAssignments] = useState({});
  
  // Add state for user permissions and admin profile
  const [userPermissions, setUserPermissions] = useState([]);
  const [adminProfile, setAdminProfile] = useState(null);
  
  // Country and Project state
  const [country, setCountry] = useState(null);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems] = useState([{ label: 'Zambia', value: 'Zambia' }, { label: 'DRC', value: 'DRC' }]);
  const [project, setProject] = useState(null);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState([]);

  // Add state to store corrective action statuses
  const [correctiveStatuses, setCorrectiveStatuses] = useState({});

  // 1. Add state for firebaseTasks
  const [firebaseTasks, setFirebaseTasks] = useState([]);

  const [inspectionFaultyTasks, setInspectionFaultyTasks] = useState([]);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  const extractedProjectRef = useRef(null);

  const priorityOptions = [
    { label: 'Low', image: require('../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../Images/adminhome/inspreport/Priority4.png') },
  ];

  const workOrderTypes = [
    { id: '1', name: 'General' },
    { id: '2', name: 'Maintenance' },
    { id: '3', name: 'Repair' },
    { id: '4', name: 'Installation' },
  ];

  const categories = ['Task', 'Parts', 'Labor', 'Additional Cost', 'Remarks'];
  const taskTypes = ['Repair', 'Replace'];
  const statuses = [
    { label: 'Initiated', color: '#A9A9A9' },
    { label: 'In Progress', color: '#FFD700' },
    { label: 'On Hold', color: '#FF4500' },
    { label: 'Complete', color: '#008000' },
  ];
  const units = ['Piece', 'Unit', 'Box'];

  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  const generateRequestNumber = (country, project, nextNumber) => {
    if (country && project) {
      // Get country code
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      // Get project code (first 3 letters, uppercased)
      const projectCode = project.slice(0, 3).toUpperCase();
      // Use job card count + 1 for the next number
      const nextJobCardNumber = jobCardCount + 1;
      return `${countryCode}-${projectCode}-JC-${nextJobCardNumber.toString().padStart(4, '0')}`;
    }
    // Fallback if either is missing
    const nextJobCardNumber = jobCardCount + 1;
    return `WO${nextJobCardNumber.toString().padStart(4, '0')}`;
  };

  const extractNumberFromRequestNumber = (requestNumber) => {
    if (!requestNumber) return 1000;
    
    // Try to extract number from new format: ZAM-KAN-JC-0001
    const newFormatMatch = requestNumber.match(/-JC-(\d+)$/);
    if (newFormatMatch) {
      const number = parseInt(newFormatMatch[1], 10);
      console.log('Extracted number from new format:', { requestNumber, number });
      return number;
    }
    
    // Try to extract number from old format: WO0001
    const oldFormatMatch = requestNumber.match(/WO(\d+)/);
    if (oldFormatMatch) {
      const number = parseInt(oldFormatMatch[1], 10);
      console.log('Extracted number from old format:', { requestNumber, number });
      return number;
    }
    
    // Fallback
    console.log('Using fallback number for:', requestNumber);
    return 1000;
  };

  // Handle selectedEquipment param
  useEffect(() => {
    if (
      route.params?.selectedEquipment &&
      (selectedEquipment == null || route.params.selectedEquipment.id !== selectedEquipment.id)
    ) {
      setSelectedEquipment(route.params.selectedEquipment);
      setErrors && setErrors(prev => ({ ...prev, equipment: '' }));
    }
  }, [route.params?.selectedEquipment]);

  // Handle selectedReport param
  useEffect(() => {
    if (route.params?.selectedReport) {
      const report = route.params.selectedReport;
      setLinkInspection({
        id: report.id,
        name: report.step1?.requestNumber || 'Inspection Report',
      });
      setWorkOrderTitle(`Job Card for ${report.step1?.requestNumber || 'Inspection'}`);
      setMemo(report.step2?.notes || '');
      setPriority(report.step3?.equipmentStatus === 'Critical' ? 'High' : 'Medium');

      // Fetch the latest equipment from Firebase for this inspection
      (async () => {
        try {
          // Find the admin UID for this inspection
          const adminUid = await findAdminUidForInspection(report.id);
          if (!adminUid) return;
          const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${report.id}`);
          const snapshot = await reportRef.once('value');
          const reportData = snapshot.val();
          const inspectionEquipment = reportData?.step1?.selectedEquipment || reportData?.step1?.equipment;
          if (inspectionEquipment) {
            setSelectedEquipment(inspectionEquipment);
            setErrors && setErrors(prev => ({ ...prev, equipment: '' }));
          }
        } catch (err) {
          // Silent fail
        }
      })();

      // Auto-fill country and project from requestNumber if possible
      const requestNumber = report.step1?.requestNumber || '';
      
      // First try to get from the report data directly
      let extractedCountry = report.step1?.country || null;
      let extractedProject = report.step1?.project || null;
      
      // If not available in report data, try to extract from requestNumber
      if (!extractedCountry || !extractedProject) {
        const match = requestNumber.match(/^([^-]+)-([^-]+)-IR-/);
        if (match) {
          const countryCode = match[1];
          const projectCode = match[2];
          
          // Map country code to country name
          if (!extractedCountry) {
            if (countryCode === 'ZAM') {
              extractedCountry = 'Zambia';
            } else if (countryCode === 'DRC') {
              extractedCountry = 'DRC';
            }
          }
          
          // Map project code to project name
          if (!extractedProject && extractedCountry) {
            if (extractedCountry === 'Zambia') {
              switch (projectCode) {
                case 'KAN': extractedProject = 'Kansanshi'; break;
                case 'KAL': extractedProject = 'Kalumbila'; break;
                case 'KIM': extractedProject = 'Kimteto'; break;
                case 'KOB': extractedProject = 'Kobold'; break;
                case 'FQM': extractedProject = 'FQM Exploration'; break;
                case 'MIM': extractedProject = 'Mimosa'; break;
                default: 
                  // Fallback: try to find a project that starts with the code
                  const zambiaProjects = ['Kansanshi', 'Kalumbila', 'Kimteto', 'Kobold', 'FQM Exploration', 'Mimosa'];
                  extractedProject = zambiaProjects.find(p => p.toUpperCase().startsWith(projectCode)) || null;
                  break;
              }
            } else if (extractedCountry === 'DRC') {
              switch (projectCode) {
                case 'LUB': extractedProject = 'Lubumbashi'; break;
                case 'MUS': extractedProject = 'Musompo'; break;
                case 'IME': extractedProject = 'IME'; break;
                case 'KAM': extractedProject = 'Kamoa'; break;
                default:
                  // Fallback: try to find a project that starts with the code
                  const drcProjects = ['Lubumbashi', 'Musompo', 'IME', 'Kamoa'];
                  extractedProject = drcProjects.find(p => p.toUpperCase().startsWith(projectCode)) || null;
                  break;
              }
            }
          }
        }
      }
      
      if (extractedCountry) {
        setCountry(extractedCountry);
      }
      
      if (extractedProject) {
        // Store the extracted project in a ref to set it after projectItems are populated
        if (!extractedProjectRef.current) {
          extractedProjectRef.current = extractedProject;
          console.log('Extracted project from requestNumber:', extractedProject);
        }
      }
      
      console.log('Report data:', {
        requestNumber,
        reportCountry: report.step1?.country,
        reportProject: report.step1?.project,
        extractedCountry,
        extractedProject
      });
    }
  }, [route.params?.selectedReport]);

  // Initial data fetch and loading animation
  useEffect(() => {
    const load = async () => {
      setIsLoading(true);
      try {
        await fetchInitialData();
        await fetchJobCardCount(); // Fetch job card count
      } finally {
        setIsLoading(false);
      }
    };
    load();
  }, []);

  useEffect(() => {
    let animation;
    if (isLoading) {
      rotateAnim.setValue(0); // Always reset before starting
      animation = Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 900,
          useNativeDriver: true,
          easing: undefined,
        })
      );
      animation.start();
    } else {
      rotateAnim.stopAnimation && rotateAnim.stopAnimation();
      rotateAnim.setValue(0);
      if (animation && animation.stop) animation.stop();
    }
    return () => {
      if (animation && animation.stop) animation.stop();
    };
  }, [isLoading]);

  useEffect(() => {
    if (!linkInspection) return;
    (async () => {
      try {
        // Find the admin UID for this inspection
        const adminUid = await findAdminUidForInspection(linkInspection.id);
        if (!adminUid) return;
        const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${linkInspection.id}`);
        const snapshot = await reportRef.once('value');
        const reportData = snapshot.val();
        if (!reportData?.step2?.sections || !reportData?.step2?.selections) return;
        // Find all subItems with 'Not Ok', 'Reject', or 'Faulty'
        const faultyTasks = [];
        reportData.step2.sections.forEach(section => {
          section.subItems.forEach(subItem => {
            const sel = reportData.step2.selections[subItem.key];
            if ((sel === 'Not Ok' || sel === 'Reject' || sel === 'Faulty') && subItem.itemName && subItem.itemName.trim() !== '') {
              faultyTasks.push({
                id: `task_${subItem.key}`,
                category: 'Task',
                text: subItem.itemName,
                type: 'Repair',
                note: `Auto-generated from inspection: ${linkInspection.name}`,
                assignTo: [],
                status: 'Initiated',
                section: section.name,
                inspectionId: linkInspection.id
              });
            }
          });
        });
        // Only add new tasks that are not already present
        setItems(prev => {
          const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
          const newTasks = faultyTasks.filter(t => !existingTaskIds.has(t.id));
          // Remove old auto-generated tasks for this inspection if not in checklist anymore
          const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
          return [...filteredPrev, ...newTasks];
        });
      } catch (err) {
        // Silent fail
      }
    })();
  }, [linkInspection]);

  // Add this useEffect to auto-populate Task items from the related inspection's checklist
  useEffect(() => {
    const fetchChecklistTasksFromInspection = async () => {
      if (!linkInspection) return;
      try {
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) return;
        // Fetch the inspection report from the correct global path
        const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${linkInspection.id}`);
        const snapshot = await reportRef.once('value');
        const reportData = snapshot.val();
        if (!reportData?.step2?.sections || !reportData?.step2?.selections) return;
        // Find all subItems with 'Not Ok' or 'Reject'
        const checklistTasks = [];
        reportData.step2.sections.forEach(section => {
          section.subItems.forEach(subItem => {
            const sel = reportData.step2.selections[subItem.key];
            if ((sel === 'Not Ok' || sel === 'Reject') && subItem.itemName && subItem.itemName.trim() !== '') {
              checklistTasks.push({
                id: `task_${subItem.key}`,
                category: 'Task',
                text: `Fix ${subItem.itemName}`,
                type: 'Repair',
                note: `From inspection: ${linkInspection.name}`,
                assignTo: [],
                status: 'Initiated',
                section: section.name,
                inspectionId: linkInspection.id
              });
            }
          });
        });
        // Only add new tasks that are not already present
        setItems(prev => {
          const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
          const newTasks = checklistTasks.filter(t => !existingTaskIds.has(t.id));
          // Remove old auto-generated tasks for this inspection if not in checklist anymore
          const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
          return [...filteredPrev, ...newTasks];
        });
      } catch (err) {
        // Silent fail
      }
    };
    fetchChecklistTasksFromInspection();
    // Only run when linkInspection changes
  }, [linkInspection]);

  useEffect(() => {
    if (!linkInspection) return;

    // This should be the result of your faulty items fetch
    const faultyTasks = inspectionFaultyTasks.map(fault => ({
      id: `task_${fault.id}`,
      category: 'Task',
      text: `Fix ${fault.text}`,
      type: 'Repair',
      note: `Faulty item from inspection: ${linkInspection.name}`,
      assignTo: [],
      status: 'Initiated',
      section: fault.section,
      inspectionId: linkInspection.id
    }));

    // Only add new tasks that don't already exist
    setItems(prev => {
      const existingIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
      const newOnes = faultyTasks.filter(t => !existingIds.has(t.id));
      return [...prev, ...newOnes];
    });
  }, [inspectionFaultyTasks, linkInspection]);

  // --- CHUNKED STATE FOR LARGE LISTS ---
  const [chunkedChecklistItems, setChunkedChecklistItems] = useState([]);
  const [chunkedItems, setChunkedItems] = useState([]);
  const [chunkedTeamMembers, setChunkedTeamMembers] = useState([]);
  const [chunkedWorkOrders, setChunkedWorkOrders] = useState([]);
  const [chunkedInspections, setChunkedInspections] = useState([]);
  const [chunkedFaultListData, setChunkedFaultListData] = useState([]);
  const CHUNK_SIZE = 5;

  // Helper to chunk and set state in small pieces
  const setChunkedState = (data, setState) => {
    if (!Array.isArray(data) || data.length <= CHUNK_SIZE) {
      setState(data ? [data] : []);
      return;
    }
    const chunks = chunkArray(data, CHUNK_SIZE);
    let i = 0;
    setState([]);
    const addChunk = () => {
      setState(prev => [...prev, chunks[i]]);
      i++;
      if (i < chunks.length) {
        setTimeout(addChunk, 10);
      }
    };
    addChunk();
  };

  // --- OPTIMIZED fetchInitialData ---
  const [loaderTimeout, setLoaderTimeout] = useState(null);
  const firebaseRefs = useRef([]);

  const fetchInitialData = async () => {
    try {
      setIsLoading(true);
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      // Fetch team members from MongoDB
      try {
        const teamMembers = await getFilteredTeamMembers(adminUid);
        setTeamMembers(teamMembers);
      } catch (error) {
        console.error('Error fetching team members:', error);
        setTeamMembers([]);
      }

      // Fetch inspections from MongoDB (using Inspection API)
      try {
        const inspectionResponse = await InspectionAPI.getInspections(adminUid);
        if (inspectionResponse.success && inspectionResponse.data) {
          setInspections(inspectionResponse.data);
        } else {
          setInspections([]);
        }
      } catch (error) {
        console.error('Error fetching inspections:', error);
        setInspections([]);
      }

      // Fetch work orders from MongoDB
      try {
        const workOrdersResponse = await JobCardAPI.getJobCards(adminUid);
        if (workOrdersResponse.success && workOrdersResponse.data) {
          const workOrdersArray = workOrdersResponse.data.map(wo => ({
            id: wo._id,
            ...wo
          }));
          setWorkOrders(workOrdersArray);
          
          // Update inspections with job card status
          setInspections(prev => {
            const workOrderInspectionIds = new Set(
              workOrdersArray
                .filter(wo => wo.linkInspection && wo.linkInspection.id)
                .map(wo => wo.linkInspection.id)
            );
            return prev.map(inspection => ({
              ...inspection,
              hasJobCard: workOrderInspectionIds.has(inspection.id),
            }));
          });
        } else {
          setWorkOrders([]);
        }
      } catch (error) {
        console.error('Error fetching work orders:', error);
        setWorkOrders([]);
      }

      // Fetch faults (this might need to be implemented in MongoDB)
      try {
        // For now, set empty array - this would need to be implemented
        setFaultListData([]);
      } catch (error) {
        console.error('Error fetching faults:', error);
        setFaultListData([]);
      }

      setIsLoading(false);
    } catch (error) {
      console.error('Error in fetchInitialData:', error);
      setIsLoading(false);
      showToast('error', 'Error', 'Failed to load data.');
    }
  };

  // Clean up on unmount
  useEffect(() => {
    return () => {
      if (loaderTimeout) clearTimeout(loaderTimeout);
    };
  }, []);

  // --- OPTIMIZED checklistItems and items chunking ---
  useEffect(() => {
    setChunkedState(checklistItems, setChunkedChecklistItems);
  }, [checklistItems]);
  useEffect(() => {
    setChunkedState(items, setChunkedItems);
  }, [items]);

  const validateFields = () => {
    let isValid = true;
    const newErrors = { title: '', equipment: '', country: '', project: '' };

    if (!workOrderTitle.trim()) {
      newErrors.title = 'Job Card Title is required';
      isValid = false;
    }
    if (!selectedEquipment) {
      newErrors.equipment = 'Equipment is required';
      isValid = false;
    }
    if (!country) {
      newErrors.country = 'Country is required';
      isValid = false;
    }
    if (!project) {
      newErrors.project = 'Project is required';
      isValid = false;
    }

    setErrors(newErrors);
    return isValid;
  };

  const uploadFileToMongoDB = async (filePath, adminUid) => {
    try {
      console.log('📁 Starting PDF upload to MongoDB...');
      console.log('📁 File path:', filePath);
      console.log('👤 Admin UID:', adminUid);
      
      if (!filePath || !adminUid) {
        throw new Error('Missing required parameters: filePath or adminUid');
      }
      
      // Check if file exists
      const fileExists = await RNFS.exists(filePath);
      console.log('📁 File exists:', fileExists);
      
      if (!fileExists) {
        throw new Error('PDF file does not exist at path: ' + filePath);
      }
      
      // Get file stats for validation
      const fileStats = await RNFS.stat(filePath);
      console.log('📁 File size:', fileStats.size, 'bytes');
      
      if (fileStats.size === 0) {
        throw new Error('PDF file is empty');
      }
      
      // Create FormData for file upload
      const formData = new FormData();
      formData.append('pdf', {
        uri: `file://${filePath}`, // Ensure proper file URI format
        type: 'application/pdf',
        name: `jobcard_${adminUid}_${Date.now()}.pdf`
      });
      
      console.log('📤 FormData created, calling uploadJobCardPdf...');
      
      const response = await JobCardAPI.uploadJobCardPdf(adminUid, formData);
      console.log('📊 Upload response:', JSON.stringify(response, null, 2));
      
      if (response && response.success) {
        const downloadUrl = response.downloadUrl || response.fileDownloadUrl || response.url;
        if (!downloadUrl) {
          throw new Error('Upload successful but no download URL returned');
        }
        console.log('✅ PDF uploaded successfully, URL:', downloadUrl);
        return downloadUrl;
      } else {
        const errorMsg = response?.error || response?.message || 'Failed to upload PDF';
        throw new Error(errorMsg);
      }
    } catch (error) {
      console.error('❌ Error uploading file to MongoDB:', error);
      console.error('❌ Error details:', {
        message: error.message,
        code: error.code,
        stack: error.stack
      });
      throw new Error(`PDF upload failed: ${error.message}`);
    }
  };

  const generatePDFWithHtml = async () => {
    try {
      let checklistHtml = '';
      const safeChecklistItems = Array.isArray(checklistItems) ? checklistItems : [];
      if (safeChecklistItems.length > 0) {
        checklistHtml = safeChecklistItems.map(section => `
          <div style="margin-bottom: 10px;">
            <h3 style="color: #015185; font-size: 16px; margin-bottom: 6px; border-bottom: 1.5px solid #e0e0e0; padding-bottom: 2px;">${section.sectionName}</h3>
            ${(Array.isArray(section.subItems) ? section.subItems : []).map(subItem => `
              <div style="display: flex; justify-content: space-between; padding: 5px 0; border-bottom: 1px solid #f0f0f0;">
                <span style="font-size: 14px; color: #222;">${subItem.text}</span>
                <span style="font-size: 14px; color: ${subItem.status === 'Reject' ? '#D32F2F' : '#FBC02D'}; font-weight: 600;">${subItem.status}</span>
              </div>
            `).join('')}
          </div>
        `).join('');
      } else {
        checklistHtml = '<p>No checklist items.</p>';
      }

      const safeItems = Array.isArray(items) ? items : [];
      let taskRows = safeItems.filter(item => item.category === 'Task').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.text || 'N/A'}</td>
          <td>${item.type || 'N/A'}</td>
          <td>${item.status || 'N/A'}</td>
          <td>${item.note || 'N/A'}</td>
        </tr>
      `).join('') || '<tr><td colspan="5">No tasks available</td></tr>';

      let laborRows = safeItems.filter(item => item.category === 'Labor').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.worker || 'N/A'}</td>
          <td>${item.code || 'N/A'}</td>
          <td>${item.rate || 'N/A'}</td>
          <td>${item.hours || '0'}h ${item.minutes || '0'}m</td>
        </tr>
      `).join('') || '<tr><td colspan="5">No labor items available</td></tr>';

      let partsRows = safeItems.filter(item => item.category === 'Parts').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.itemNumber || 'N/A'}</td>
          <td>${item.partNumber || 'N/A'}</td>
          <td>${item.itemName || 'N/A'}</td>
          <td>${item.quantity || '0'} ${item.unit || 'N/A'}</td>
          <td>${item.itemCost || 'N/A'}</td>
        </tr>
      `).join('') || '<tr><td colspan="6">No parts available</td></tr>';

      let additionalCostRows = safeItems.filter(item => item.category === 'Additional Cost').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.description || 'N/A'}</td>
          <td>${item.amount || 'N/A'}</td>
        </tr>
      `).join('') || '<tr><td colspan="3">No additional costs available</td></tr>';

      let remarksRows = safeItems.filter(item => item.category === 'Remarks').map((item, index) => `
        <tr>
          <td>${index + 1}</td>
          <td>${item.description || 'N/A'}</td>
        </tr>
      `).join('') || '<tr><td colspan="2">No remarks available</td></tr>';

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1.0" />
          <title>Job Card: ${workOrderTitle || 'Untitled'}</title>
          <style>
            body { font-family: 'Segoe UI', Arial, sans-serif; padding: 24px; font-size: 13px; background: #f8f9fa; color: #222; }
            .header, .section { border-radius: 8px; background: #fff; box-shadow: 0 1px 4px #e0e0e0; padding: 16px 20px; margin-bottom: 18px; }
            .header-table, .checklist-table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
            .header-table td { border: none; padding: 3px 0; }
            .checklist-table th, .checklist-table td { border: 1px solid #e0e0e0; padding: 7px 6px; text-align: left; }
            .title { text-align: center; font-weight: bold; font-size: 22px; margin: 24px 0 18px 0; color: #015185; letter-spacing: 1px; }
            .notes { font-size: 11px; margin-bottom: 10px; color: #888; }
            .checklist-table th { background-color: #015185; color: #fff; font-weight: 600; font-size: 14px; }
            .summary-section { margin-top: 24px; background: #fff; border-radius: 8px; box-shadow: 0 1px 4px #e0e0e0; padding: 18px 20px; }
            .summary-section h3 { color: #015185; font-size: 16px; margin-bottom: 10px; border-bottom: 1.5px solid #e0e0e0; padding-bottom: 2px; }
            .summary-section p { margin: 5px 0; }
            .section strong { color: #015185; }
            .section label { color: #888; font-weight: 400; }
          </style>
        </head>
        <body>
          <div class="header">
            <table class="header-table">
              <tr>
                <td rowspan="5"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" width="100"></td>
                <td><strong>Document Name:</strong> HSE-FOR-8.1.2 Job Card</td>
              </tr>
              <tr><td><strong>Document ID:</strong> HSE-029</td></tr>
              <tr><td><strong>Version Date:</strong> 05/08/2021</td></tr>
              <tr><td><strong>Version:</strong> 1</td></tr>
              <tr><td><strong>Approved:</strong> HSM</td></tr>
            </table>
          </div>

          <div class="title">JOB CARD</div>
          <div class="section">
            <div style="display: flex; flex-wrap: wrap; gap: 18px 32px;">
              <div><strong>NO:</strong> ${requestNumber}</div>
              <div><strong>DATE:</strong> ${new Date().toLocaleDateString('en-US')}</div>
              <div><strong>COUNTRY:</strong> ${country || 'N/A'}</div>
              <div><strong>PROJECT:</strong> ${project || 'N/A'}</div>
              <div><strong>EQUIPMENT:</strong> ${selectedEquipment?.equipmentName || 'N/A'}</div>
              <div><strong>CREATED AT:</strong> ${new Date().toLocaleString()}</div>
            </div>
          </div>

          <div class="summary-section">
            <h3>Checklist Items</h3>
            ${checklistHtml}
          </div>

          <div class="summary-section">
            <h3>Job Card Details</h3>
            <div style="display: flex; flex-wrap: wrap; gap: 12px 32px;">
              <div><strong>Title:</strong> ${workOrderTitle || 'N/A'}</div>
              <div><strong>Country:</strong> ${country || 'N/A'}</div>
              <div><strong>Project:</strong> ${project || 'N/A'}</div>
              <div><strong>Memo:</strong> ${memo || 'N/A'}</div>
              <div><strong>Priority:</strong> ${priority || 'N/A'}</div>
                              <div><strong>Assigned To:</strong> ${assignTo.map(a => a.fullName || a.name || 'Unknown User').join(', ') || 'None'}</div>
              <div><strong>Start Date:</strong> ${startDate ? new Date(startDate).toLocaleString() : 'N/A'}</div>
              <div><strong>Due Date:</strong> ${dueDate ? new Date(dueDate).toLocaleString() : 'N/A'}</div>
              <div><strong>Job Card Type:</strong> ${workOrderType || 'N/A'}</div>
              <div><strong>Estimated Cost:</strong> ${estimatedCost || 'N/A'}</div>
              <div><strong>Estimated Labor Time:</strong> ${estimatedLaborTime || 'N/A'}</div>
              <div><strong>Linked Inspection:</strong> ${linkInspection?.name || 'None'}</div>
              <div><strong>Linked Job Card:</strong> ${linkedWorkOrder?.title || 'None'}</div>
            </div>
          </div>

          <div class="summary-section">
            <h3>Cost Details</h3>
            <div style="display: flex; flex-wrap: wrap; gap: 12px 32px;">
              <div><strong>Part & Material Cost:</strong> ${partMaterialCost || '0.00'}</div>
              <div><strong>Total Labor Cost:</strong> ${totalLaborCost || '0.00'}</div>
              <div><strong>Additional Cost:</strong> ${additionalCost || '0.00'}</div>
              <div><strong>Tax Amount:</strong> ${taxAmount || '0.00'} (${taxPercentage}%)</div>
              <div><strong>Total WO Cost:</strong> ${totalWoCost || '0.00'}</div>
            </div>
          </div>

          <div class="summary-section">
            <h3>Tasks</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
                <th>Type</th>
                <th>Status</th>
                <th>Note</th>
              </tr>
              ${taskRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Labor</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Worker</th>
                <th>Code</th>
                <th>Rate</th>
                <th>Time</th>
              </tr>
              ${laborRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Parts</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Item No</th>
                <th>Part No</th>
                <th>Name</th>
                <th>Quantity</th>
                <th>Cost</th>
              </tr>
              ${partsRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Additional Costs</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
                <th>Amount</th>
              </tr>
              ${additionalCostRows}
            </table>
          </div>

          <div class="summary-section">
            <h3>Remarks</h3>
            <table class="checklist-table">
              <tr>
                <th>#</th>
                <th>Description</th>
              </tr>
              ${remarksRows}
            </table>
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `JobCard_${requestNumber || 'Untitled'}_${Date.now()}`,
        directory: 'Documents',
        base64: false,
        width: 595, // A4 width in points
        height: 842, // A4 height in points
        padding: 24,
      };

      console.log('📄 PDF Options:', options);
      console.log('📄 HTML length:', html.length);
      
      const file = await RNHTMLtoPDF.convert(options);
      console.log('📄 PDF generation result:', file);
      
      if (!file || !file.filePath) {
        throw new Error('PDF generation failed: No file path returned');
      }
      
      // Verify file exists before returning
      const fileExists = await RNFS.exists(file.filePath);
      if (!fileExists) {
        throw new Error('Generated PDF file does not exist');
      }
      
      console.log('✅ PDF generated successfully at:', file.filePath);
      return file.filePath;
    } catch (error) {
      console.error('❌ Error generating PDF:', error);
      console.error('❌ Error stack:', error.stack);
      throw new Error(`PDF generation failed: ${error.message}`);
    }
  };

  // 2. Add function to fetch tasks from MongoDB for the current work order
  const fetchMongoTasks = async (woId) => {
    try {
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId || !woId) {
        console.log('Skipping fetchMongoTasks: missing userId or woId');
        return;
      }
      
      console.log('Fetching tasks for workOrderId:', woId);
      const response = await JobCardAPI.getJobCard(woId);
      
      if (response.success && response.data) {
        const tasksArray = response.data.items ? response.data.items.filter(item => item.category === 'Task') : [];
        console.log('Tasks fetched successfully:', tasksArray.length);
        setFirebaseTasks(tasksArray);
      } else {
        console.log('No tasks found or job card not found');
        setFirebaseTasks([]);
      }
    } catch (error) {
      console.log('Error fetching tasks from MongoDB (non-critical):', error.message);
      setFirebaseTasks([]);
    }
  };

  // 3. Fetch MongoDB tasks when workOrderId is set (after creation)
  useEffect(() => {
    if (workOrderId) {
      fetchMongoTasks(workOrderId);
    }
  }, [workOrderId]);

  // Test API connection
  const testAPIConnection = async () => {
    try {
      console.log('🧪 Testing API connection...');
      const response = await JobCardAPI.getJobCards('test123', { limit: 1 });
      console.log('✅ API test successful:', response);
      return true;
    } catch (error) {
      console.error('❌ API test failed:', error);
      return false;
    }
  };

  // Save work order to MongoDB
  const saveWorkOrder = async () => {
    if (!validateFields()) {
      showToast('error', 'Error', 'Please fill all required fields.');
      return;
    }

    setIsLoading(true);
    setPdfModalVisible(true); // Show PDF loader modal
    
    // Check network status
    const netInfo = await NetInfo.fetch();
    const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
    
    console.log('🌐 Network status:', isOnline ? 'Online' : 'Offline');
    try {
      let adminUid = route.params?.userId;
      if (!adminUid) {
        adminUid = await AsyncStorage.getItem('userUid');
      }
      if (!adminUid) {
        throw new Error('Admin not authenticated. Please log in again.');
      }

      // Allow multiple job cards for the same inspection (validation removed)

      // Generate request number DYNAMICALLY from MongoDB (with offline suffix if offline)
      let finalRequestNumber;
      
      try {
        // Fetch LIVE job card count from MongoDB for dynamic numbering
        console.log('🔢 Fetching LIVE job card data for dynamic numbering...');
        const liveResponse = await JobCardAPI.getAllJobCards();
        
        let nextNumber = 1; // Default if no job cards exist
        
        if (liveResponse.success && liveResponse.data && liveResponse.data.length > 0) {
          // Use TOTAL COUNT for next number
          const totalCount = liveResponse.data.length;
          nextNumber = totalCount + 1;
          
          // Also log the last request number for reference
          const sortedJobCards = liveResponse.data.sort((a, b) => {
            const numA = extractNumberFromRequestNumber(a.requestNumber);
            const numB = extractNumberFromRequestNumber(b.requestNumber);
            return numB - numA; // Descending
          });
          
          const lastRequestNumber = sortedJobCards[0].requestNumber;
          
          console.log('📊 Total job cards in MongoDB:', totalCount);
          console.log('📊 Last request number (reference):', lastRequestNumber);
          console.log('✅ Next number (total + 1):', nextNumber);
        } else {
          console.log('ℹ️ No job cards in MongoDB, starting from 1');
        }
        
        // Generate with the LIVE next number
        const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
        const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
        
        finalRequestNumber = `${countryCode}-${projectCode}-JC-${String(nextNumber).padStart(4, '0')}`;
        
        if (!isOnline) {
          // Add -OFFLINE suffix for offline created work orders
          finalRequestNumber = `${finalRequestNumber}-OFFLINE`;
          console.log('📴 Generated offline request number:', finalRequestNumber);
        } else {
          console.log('🌐 Generated online request number:', finalRequestNumber);
        }
        
      } catch (numberError) {
        console.error('❌ Failed to generate dynamic request number:', numberError.message);
        // Fallback to old method
        finalRequestNumber = generateRequestNumber(country, project, jobCardCount);
        if (!isOnline) {
          finalRequestNumber = `${finalRequestNumber}-OFFLINE`;
        }
      }
      
      setRequestNumber(finalRequestNumber);
      
      // Generate work order ID (using timestamp for uniqueness)
      const workOrderIdLocal = Date.now().toString();

      // Defensive: always use array for tasks
      const tasksToSave = Array.isArray(items) ? items.filter(item => item.category === 'Task') : [];

      // Ensure all items have proper structure and IDs
      const processedItems = Array.isArray(items) ? items.map((item, index) => ({
        ...item,
        id: item.id || `item_${Date.now()}_${index}`,
        status: item.status || (item.category === 'Task' ? 'Initiated' : 'Pending'),
        category: item.category || 'Task',
        createdAt: item.createdAt || new Date().toISOString(),
        updatedAt: new Date().toISOString()
      })) : [];

      console.log('📊 Processed items for saving:', processedItems.length);
      console.log('📊 Items breakdown:', {
        tasks: processedItems.filter(i => i.category === 'Task').length,
        parts: processedItems.filter(i => i.category === 'Parts').length,
        labor: processedItems.filter(i => i.category === 'Labor').length,
        additionalCost: processedItems.filter(i => i.category === 'Additional Cost').length,
        remarks: processedItems.filter(i => i.category === 'Remarks').length
      });

      const workOrderData = {
        _id: workOrderIdLocal,
        adminUid: adminUid,
        title: `${finalRequestNumber} - ${workOrderTitle}`,
        status: 'Hold', // This matches the backend enum values: 'Hold', 'assigned', 'in_progress', 'complete', 'cancelled'
        priority: priority || 'Medium', // This matches backend enum: 'Low', 'Medium', 'High', 'Emergency'
        workOrderType: workOrderType,
        equipment: selectedEquipment ? {
          equipmentName: selectedEquipment.equipmentName || selectedEquipment.name || '',
          mainCategory: selectedEquipment.mainCategory || '',
          model: selectedEquipment.model || '',
          vehicleNumber: selectedEquipment.vehicleNumber || '',
          meterReading: selectedEquipment.meterReading || selectedEquipment.meter_reading || 0,
          status: selectedEquipment.status || ''
        } : null,
        assignTo: Array.isArray(assignTo) ? assignTo.map(user => ({
          name: user.name || user.fullName || '',
          email: user.email || '',
          uid: user.uid || user.userId || ''
        })) : [],
        startDate: startDate ? startDate.toISOString() : null,
        dueDate: dueDate ? dueDate.toISOString() : null,
        completedDate: null,
        description: memo || '',
        notes: additionalDetails || '',
        location: {
          country: country || '',
          project: project || '',
          coordinates: '',
          address: ''
        },
        serviceScheduleId: linkInspection?.id || '',
        serviceType: workOrderType || '',
        attachments: [],
        pdfDownloadUrl: '',
        signatureDownloadUrl: '',
        tracking: {
          currentStep: 'Initiated',
          progress: 0,
          lastUpdated: new Date().toISOString(),
          history: [{
            step: 'Initiated',
            timestamp: new Date().toISOString(),
            updatedBy: adminUid,
            notes: 'Job Card created'
          }]
        },
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        createdBy: adminUid,
        lastModifiedBy: adminUid,
        // Additional fields for compatibility
        requestNumber: finalRequestNumber,
        estimatedCost: estimatedCost || '',
        estimatedLaborTime: estimatedLaborTime || '',
        linkInspection: linkInspection,
        linkedWorkOrder: linkedWorkOrder,
        partMaterialCost: partMaterialCost || '',
        totalLaborCost: totalLaborCost || '',
        additionalCost: additionalCost || '',
        taxAmount: taxAmount || '',
        taxPercentage: taxPercentage || '',
        totalWoCost: totalWoCost || '',
        items: processedItems, // Use processed items with proper structure
        checklistItems: Array.isArray(checklistItems) ? checklistItems : []
      };

      // Show PDF loader modal before PDF generation
      setPdfModalVisible(true);
      console.log('📄 Generating PDF...');
      
      let pdfPath;
      try {
        pdfPath = await generatePDFWithHtml();
        console.log('✅ PDF generated successfully:', pdfPath);
        
        // Store PDF in AsyncStorage for offline viewing
        const pdfKey = `workorder_pdf_local_${finalRequestNumber}`;
        await AsyncStorage.setItem(pdfKey, pdfPath);
        console.log('✅ Stored PDF path in AsyncStorage:', pdfKey);
      } catch (pdfError) {
        console.error('❌ PDF generation failed:', pdfError);
        throw new Error('Failed to generate PDF: ' + pdfError.message);
      }
      
      if (isOnline) {
        // ONLINE MODE: Upload PDF FIRST, then save work order with PDF URL
        console.log('🌐 Online mode: Uploading PDF to MongoDB...');
        
        let pdfDownloadUrl = '';
        let pdfUploadSuccess = false;
        
        try {
          console.log('📤 Uploading PDF to MongoDB...');
          console.log('📤 PDF path:', pdfPath);
          console.log('📤 Admin UID:', adminUid);
          
          pdfDownloadUrl = await uploadFileToMongoDB(pdfPath, adminUid);
          
          if (pdfDownloadUrl && pdfDownloadUrl.startsWith('http')) {
            console.log('✅ PDF uploaded successfully!');
            console.log('✅ Download URL:', pdfDownloadUrl);
            pdfUploadSuccess = true;
            
            // CRITICAL: Update workOrderData with the PDF URL BEFORE saving
            workOrderData.pdfDownloadUrl = pdfDownloadUrl;
          } else {
            throw new Error('Invalid PDF URL returned from upload');
          }
        } catch (pdfUploadError) {
          console.error('❌ PDF upload failed:', pdfUploadError.message);
          console.error('❌ PDF upload error details:', pdfUploadError);
          
          // Continue without PDF URL - will use local PDF for viewing
          workOrderData.pdfDownloadUrl = '';
          pdfUploadSuccess = false;
        }
        
        setPdfModalVisible(false);

        // Save work order to MongoDB with retry logic
        console.log('🔧 Saving work order to MongoDB...');
        console.log('🔧 PDF URL in work order data:', workOrderData.pdfDownloadUrl);
        
        let response;
        let retryCount = 0;
        const maxRetries = 3;
        
        while (retryCount < maxRetries) {
          try {
            console.log(`🔧 Save attempt ${retryCount + 1}/${maxRetries}...`);
            response = await JobCardAPI.createJobCard(adminUid, workOrderData);
            console.log('📊 MongoDB save response:', response);
            
            if (response.success) {
              console.log('✅ Work order saved to MongoDB successfully!');
              break; // Success, exit retry loop
            } else {
              throw new Error(response.error || 'Failed to create job card');
            }
          } catch (saveError) {
            retryCount++;
            console.error(`❌ Save attempt ${retryCount} failed:`, saveError.message);
            
            if (retryCount >= maxRetries) {
              // FALLBACK: Save offline if MongoDB save fails after retries
              console.log('📴 MongoDB save failed after retries, saving offline...');
              workOrderData.pdfLocalPath = pdfPath; // Store local path for offline
              
              const saved = await OfflineWorkOrderHelper.saveOfflineWorkOrder(workOrderData);
              if (saved) {
                setWorkOrderId(workOrderIdLocal);
                setSavedPdfPath(pdfPath);
                setShowSuccessModal(true);
                showToast('warning', 'Saved Offline', 'Work order saved offline. Will sync when online.');
                return; // Exit function after offline save
              } else {
                throw saveError;
              }
            }
            
            // Wait before retrying
            await new Promise(resolve => setTimeout(resolve, 2000));
          }
        }

        // ONLINE SUCCESS - Store work order ID and PDF path
        setWorkOrderId(workOrderIdLocal);
        
        if (tasksToSave.length > 0) {
          console.log('✅ Tasks saved with work order:', tasksToSave.length);
        }

        if (linkInspection) {
          console.log('✅ Work order linked to inspection:', linkInspection.id);
        }

        // Cache the work order locally
        await AsyncStorage.setItem(`workOrder_${finalRequestNumber}`, JSON.stringify(workOrderData));

        // Update global cache
        const cachedWorkOrdersRaw = await AsyncStorage.getItem('globalWorkOrders');
        let cachedWorkOrders = cachedWorkOrdersRaw ? JSON.parse(cachedWorkOrdersRaw) : [];
        cachedWorkOrders.push(workOrderData);
        await AsyncStorage.setItem('globalWorkOrders', JSON.stringify(cachedWorkOrders));

        setSavedPdfPath(pdfPath);
        setShowSuccessModal(true);
        
        if (pdfUploadSuccess) {
          showToast('success', 'Success', 'Job Card created and PDF uploaded!');
        } else {
          showToast('success', 'Success', 'Job Card created (PDF available offline only)');
        }
        
        await fetchInitialData();

        if (route.params?.onRefresh) route.params.onRefresh();

        // KEEP the PDF file locally for offline access (don't delete)
        console.log('✅ PDF kept locally for offline access:', pdfPath);

        setTimeout(() => {
          fetchMongoTasks(workOrderIdLocal);
        }, 1000);
        
      } else {
        // OFFLINE MODE: Save work order locally
        console.log('📴 Offline mode: Saving work order locally...');
        console.log('📴 Request number:', finalRequestNumber);
        console.log('📴 PDF path:', pdfPath);
        
        setPdfModalVisible(false);
        
        // CRITICAL: Store PDF path in work order data for sync
        workOrderData.pdfLocalPath = pdfPath;
        workOrderData.pdfDownloadUrl = ''; // Will be populated during sync
        
        // Save PDF path with request number (for easy lookup during sync)
        const pdfKey = `workorder_pdf_local_${finalRequestNumber}`;
        await AsyncStorage.setItem(pdfKey, pdfPath);
        console.log('✅ Saved PDF path with key:', pdfKey);
        
        // Also save with work order ID for backward compatibility
        await OfflineWorkOrderHelper.saveOfflinePDF(workOrderIdLocal, pdfPath);
        
        // Cache user permissions for offline access
        try {
          const cachedPermissions = await AsyncStorage.getItem('workOrderModuleUserInfo');
          if (cachedPermissions) {
            const userPermissions = JSON.parse(cachedPermissions);
            workOrderData.userPermissions = userPermissions.jobcardPermissions || [];
            workOrderData.userCountries = userPermissions.countries || [];
            workOrderData.userProjects = userPermissions.projects || [];
            console.log('✅ Cached permissions for offline work order');
          }
        } catch (permError) {
          console.warn('⚠️ Could not load permissions for offline work order:', permError);
        }
        
        // Save offline work order with PDF path included
        console.log('💾 Saving offline work order to AsyncStorage...');
        console.log('   ID:', workOrderIdLocal);
        console.log('   Request#:', finalRequestNumber);
        console.log('   PDF path:', workOrderData.pdfLocalPath);
        
        const saved = await OfflineWorkOrderHelper.saveOfflineWorkOrder(workOrderData);
        
        if (saved) {
          setWorkOrderId(workOrderIdLocal);
          setSavedPdfPath(pdfPath);
          setShowSuccessModal(true);
          showToast('success', 'Saved Offline', 'Job card & PDF saved offline. Will sync when online.');
          console.log('✅ Offline work order saved successfully');
        } else {
          throw new Error('Failed to save work order offline');
        }
      }
    } catch (error) {
      console.error('❌ Error saving work order:', error);
      
      let errorMessage = 'Failed to create job card';
      
      if (error.message.includes('Network request failed') || error.message.includes('timeout')) {
        errorMessage = 'Network error - work order saved offline';
      } else if (error.message.includes('Failed to save work order offline')) {
        errorMessage = 'Failed to save work order offline';
      } else {
        errorMessage = error.message || 'Failed to create job card';
      }
      
      showToast('error', 'Error', errorMessage);
      setPdfModalVisible(false);
    } finally {
      setIsLoading(false);
    }
  };

  const handleViewPDF = async (fromSuccessModal = false) => {
    if (!workOrderId && !requestNumber) {
      showToast('error', 'Error', 'Job Card ID not available.');
      return;
    }
    
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      
      // Check network status
      const netInfo = await NetInfo.fetch();
      const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
      
      console.log('📄 Opening PDF - Network status:', isOnline ? 'Online' : 'Offline');
      
      if (isOnline) {
        // ONLINE MODE: Try to fetch from MongoDB
        console.log('🌐 Online mode: Fetching PDF from MongoDB...');
        try {
          const response = await JobCardAPI.getJobCard(workOrderId);
          if (response.success && response.data?.pdfDownloadUrl) {
            console.log('✅ PDF URL found in MongoDB:', response.data.pdfDownloadUrl);
            navigation.reset({
              index: 0,
              routes: [
                { name: 'MainApp' },
                {
                  name: 'PdfviewPage',
                  params: {
                    pdfUrl: response.data.pdfDownloadUrl,
                    userId: adminUid,
                    onGoBack: () => {
                      navigation.reset({
                        index: 1,
                        routes: [{ name: 'MainApp' }],
                      });
                    },
                  },
                },
              ],
            });
            return;
          }
        } catch (mongoError) {
          console.warn('⚠️ MongoDB PDF fetch failed, trying local...', mongoError.message);
        }
      }
      
      // OFFLINE MODE or MongoDB failed: Try to load from local storage
      console.log('📴 Loading PDF from local storage...');
      const pdfKey = `workorder_pdf_local_${requestNumber}`;
      const localPdfPath = await AsyncStorage.getItem(pdfKey);
      
      if (localPdfPath && await RNFS.exists(localPdfPath)) {
        console.log('✅ Local PDF found:', localPdfPath);
        navigation.reset({
          index: 0,
          routes: [
            { name: 'MainApp' },
            {
              name: 'PdfviewPage',
              params: {
                pdfUrl: `file://${localPdfPath}`,
                userId: adminUid,
                isLocalFile: true,
                onGoBack: () => {
                  navigation.reset({
                    index: 1,
                    routes: [{ name: 'MainApp' }],
                  });
                },
              },
            },
          ],
        });
      } else {
        throw new Error('PDF not found in local storage or MongoDB');
      }
      
    } catch (error) {
      console.error('❌ Error opening PDF:', error);
      showToast('error', 'Error', 'Could not open the PDF. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const saveItem = () => {
    if (!newItemText.trim()) {
      showToast('error', 'Error', 'Task description is required.');
      return;
    }
    const newItem = {
      id: Date.now().toString(),
      category: selectedCategory,
      text: newItemText,
      type: taskType,
      note,
      assignTo: tempAssignTo,
      status: 'Initiated',
    };
    
    // Use InteractionManager for better iOS performance
    InteractionManager.runAfterInteractions(() => {
      setItems(prev => [...prev, newItem]);
    });
    
    // Always reset modal state and close modal
    setNewItemText('');
    setTaskType('Repair');
    setNote('');
    setTempAssignTo([]);
    setAddItemModalVisible(false);
    setEditModalVisible(false);
    setEditItem(null);
    // Hard refresh after adding a task
    handleRefresh && handleRefresh();
  };

  const saveLabor = () => {
    if (!laborWorker || !laborCode || !laborRate || !laborHours || !laborMinutes) {
      showToast('error', 'Error', 'All labor fields are required.');
      return;
    }

    const newLabor = {
      id: Date.now().toString(),
      category: 'Labor',
      worker: laborWorker,
      code: laborCode,
      rate: laborRate,
      hours: laborHours,
      minutes: laborMinutes,
      note: laborNote,
      status: 'Initiated',
    };
    setItems([...items, newLabor]);
    resetLaborFields();
  };

  const saveParts = () => {
    if (!partsItemNumber || !partsPartNumber || !partsItemName || !partsQuantity || !partsItemCost) {
      showToast('error', 'Error', 'All parts fields are required.');
      return;
    }

    const newPart = {
      id: Date.now().toString(),
      category: 'Parts',
      itemNumber: partsItemNumber,
      partNumber: partsPartNumber,
      itemName: partsItemName,
      quantity: partsQuantity,
      unit: partsUnit,
      itemCost: partsItemCost,
      site: partsSite,
      note: partsNote,
      status: 'Initiated',
    };
    setItems([...items, newPart]);
    resetPartsFields();
  };

  const saveAdditionalCost = () => {
    if (!additionalCostDescription || !additionalCostAmount) {
      showToast('error', 'Error', 'All additional cost fields are required.');
      return;
    }

    const newAdditionalCost = {
      id: Date.now().toString(),
      category: 'Additional Cost',
      description: additionalCostDescription,
      amount: additionalCostAmount,
      status: 'Initiated',
    };
    setItems([...items, newAdditionalCost]);
    resetAdditionalCostFields();
  };

  const saveRemarks = () => {
    if (!remarksDescription) {
      showToast('error', 'Error', 'Remark description is required.');
      return;
    }

    const newRemark = {
      id: Date.now().toString(),
      category: 'Remarks',
      description: remarksDescription,
      status: 'Initiated',
    };
    setItems([...items, newRemark]);
    resetRemarksFields();
  };

  const updateItemStatus = (itemId, newStatus) => {
    setItems(items.map(item => item.id === itemId ? { ...item, status: newStatus } : item));
    setStatusModalVisible(false);
  };

  const deleteItem = (itemId) => {
    setItems(items.filter(item => item.id !== itemId));
    setEditModalVisible(false);
    setEditMenuVisible(false);
    setSelectedItemId(null);
  };

  const editItemSave = () => {
    if (!newItemText.trim()) {
      showToast('error', 'Error', 'Task description is required.');
      return;
    }
    if (editItem) {
      setItems(items.map(item =>
        item.id === editItem.id
          ? { ...editItem, text: newItemText, note, assignTo: tempAssignTo, type: taskType }
          : item
      ));
      resetModalFields();
    }
  };

  const resetModalFields = () => {
    setNewItemText('');
    setTaskType('Repair');
    setNote('');
    setTempAssignTo([]);
    setAddItemModalVisible(false);
    setEditModalVisible(false);
    setEditItem(null);
  };

  const resetLaborFields = () => {
    setLaborWorker('');
    setLaborCode('');
    setLaborRate('');
    setLaborHours('');
    setLaborMinutes('');
    setLaborNote('');
    setAddLaborModalVisible(false);
    setEditLaborId(null);
  };

  const resetPartsFields = () => {
    setPartsItemNumber('');
    setPartsPartNumber('');
    setPartsItemName('');
    setPartsQuantity('');
    setPartsUnit('Piece');
    setPartsItemCost('');
    setPartsSite('');
    setPartsNote('');
    setAddPartsModalVisible(false);
    setEditPartsId(null);
  };

  const resetAdditionalCostFields = () => {
    setAdditionalCostDescription('');
    setAdditionalCostAmount('');
    setAddAdditionalCostModalVisible(false);
    setEditAdditionalCostId(null);
  };

  const resetRemarksFields = () => {
    setRemarksDescription('');
    setAddRemarksModalVisible(false);
    setEditRemarksId(null);
  };

  const filteredTeamMembers = teamMembers.filter(member =>
    member.userId?.toLowerCase().includes(assignToSearchQuery.toLowerCase()) ||
    member.name?.toLowerCase().includes(assignToSearchQuery.toLowerCase()) ||
    member.role?.toLowerCase().includes(assignToSearchQuery.toLowerCase())
  );

  const handlePrioritySelect = selectedPriority => {
    setPriority(selectedPriority);
  };

  const handleEquipmentSelect = equipmentData => {
    setSelectedEquipment(equipmentData);
    setErrors({ ...errors, equipment: '' });
  };

  const handleAssignToSelect = item => {
    setTempAssignTo(prev => prev.some(selected => selected.id === item.id)
      ? prev.filter(selected => selected.id !== item.id)
      : [...prev, item]);
  };





  const handleRemoveAssignTo = itemToRemove => {
    setAssignTo(assignTo.filter(item => item.id !== itemToRemove.id));
  };

  const handleWorkOrderTypeSelect = type => {
    setWorkOrderType(type.name);
    setWorkOrderTypeModalVisible(false);
  };




  const handleWorkOrderSelect = workOrder => {
    setLinkedWorkOrder(workOrder);
    setLinkedWorkOrderModalVisible(false);
  };

  const handleCategorySelect = category => {
    setSelectedCategory(category);
  };

  const formatDate = date => date ? new Date(date).toLocaleString('en-US', {
    year: 'numeric',
    month: 'short',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
  }) : 'Select Date';

  const handleFaultSelect = fault => {
    setSelectedFaults(prev => prev.some(f => f.id === fault.id)
      ? prev.filter(f => f.id !== fault.id)
      : [...prev, fault]);
  };

  const handleSaveFaults = () => {
    setFaultListModalVisible(false);
  };

  const handleEditItem = (item) => {
    setSelectedItem(item);
    setIsEditing(true);
    setShowModal(true); // or whatever opens your modal
  };

  const handleEditLaborItem = item => {
    setLaborWorker(item.worker || '');
    setLaborCode(item.code || '');
    setLaborRate(item.rate || '');
    setLaborHours(item.hours || '');
    setLaborMinutes(item.minutes || '');
    setLaborNote(item.note || '');
    setAddLaborModalVisible(true);
    setEditMenuVisible(false);
    setSelectedItemId(null);
  };

  const handleEditPartsItem = item => {
    setPartsItemNumber(item.itemNumber || '');
    setPartsPartNumber(item.partNumber || '');
    setPartsItemName(item.itemName || '');
    setPartsQuantity(item.quantity || '');
    setPartsUnit(item.unit || 'Piece');
    setPartsItemCost(item.itemCost || '');
    setPartsSite(item.site || '');
    setPartsNote(item.note || '');
    setAddPartsModalVisible(true);
    setEditMenuVisible(false);
    setSelectedItemId(null);
    setEditPartsId(item.id);
  };

  const handleEditAdditionalCostItem = item => {
    setAdditionalCostDescription(item.description || '');
    setAdditionalCostAmount(item.amount || '');
    setAddAdditionalCostModalVisible(true);
    setEditMenuVisible(false);
    setSelectedItemId(null);
  };

  const handleEditRemarksItem = item => {
    setRemarksDescription(item.description || '');
    setAddRemarksModalVisible(true);
    setEditMenuVisible(false);
    setSelectedItemId(null);
  };

  const renderChecklistItem = ({ item: subItem }) => {
    if (!subItem.itemName || subItem.itemName.trim() === '') {
      return null; // Don't render if itemName is empty or null
    }
    const isAssigned = taskAssignments[subItem.id];
    const assignedUser = isAssigned ? availableUsers.find(u => u.id === isAssigned.assignedTo) : null;
    // Get corrective status for this subItem
    const correctiveStatus = correctiveStatuses[subItem.id.replace(/^task_/, '')];

    return (
      <View style={styles.checklistItemContainer}>
        <View style={styles.checklistItemHeader}>
          <Text style={styles.checklistItemText}>{subItem.itemName}</Text>
          <View style={styles.rightSection}>
            {isAssigned ? (
              <Text style={styles.assignedUserText}>{assignedUser ? assignedUser.name : 'N/A'}</Text>            ) : (
              <Text style={styles.notAssignedText}>Not assigned</Text>
            )}
            <TouchableOpacity 
              style={styles.threeDotsButton}
              onPress={() => {
                setSelectedTaskId(subItem.id);
                setAssignMenuVisible(true);
              }}
            >
              <MaterialIcons name="more-vert" size={24} color="#015185" />
            </TouchableOpacity>
          </View>
        </View>
        {/* Show corrective status if available */}
        {correctiveStatus && (
          <Text style={{ color: correctiveStatus === 'Fixed' ? '#28a745' : correctiveStatus === 'Not Fixed' ? '#dc3545' : '#808080', fontWeight: 'bold', marginTop: 4 }}>
            Status: {correctiveStatus}
          </Text>
        )}
      </View>
    );
  };

  const renderChecklistSection = ({ item }) => (
    <View style={styles.sectionContainer}>
      <Text style={styles.sectionName}>{item.sectionName}</Text>
      <FlatList
        data={item.subItems}
        renderItem={renderChecklistItem}
        keyExtractor={subItem => subItem.id}
      />
    </View>
  );

  const fetchAvailableUsers = async () => {
    try {
      const user = auth().currentUser;
      if (!user) return;

      const usersRef = database().ref('users');
      const snapshot = await usersRef.once('value');
      const users = snapshot.val() || {};
      
      const formattedUsers = Object.entries(users).map(([id, userData]) => ({
        id,
        name: userData.name || 'Unknown User',
        email: userData.email,
        role: userData.role || 'User'
      }));

      setAvailableUsers(formattedUsers);
    } catch (error) {
      console.error('Error fetching users:', error);
      showToast('error', 'Error', 'Failed to fetch users');
    }
  };

  const handleTaskAssignment = async (taskId, userId) => {
    try {
      const user = auth().currentUser;
      if (!user) return;

      const task = items.find(item => item.id === taskId);
      if (task?.status === 'Down Assign') {
        showToast('error', 'Cannot Assign', 'Task is in Down Assign state and cannot be reassigned');
        return;
      }

      const existingAssignment = taskAssignments[taskId];
      if (existingAssignment && existingAssignment.assignedTo === userId) {
        showToast('info', 'Already Assigned', 'This user is already assigned to this task');
        return;
      }

      const newAssignment = {
        taskId,
        assignedTo: userId,
        assignedAt: new Date().toISOString(),
        assignedBy: user.uid,
        status: 'Pending'
      };

      setTaskAssignments(prev => ({
        ...prev,
        [taskId]: newAssignment
      }));

      const assignmentRef = database().ref(`workOrders/${workOrderId}/taskAssignments/${taskId}`);
      await assignmentRef.set(newAssignment);

      showToast('success', 'Success', 'Task assigned successfully');
      setAssignMenuVisible(false);
    } catch (error) {
      console.error('Error assigning task:', error);
      showToast('error', 'Error', 'Failed to assign task');
    }
  };

  const handleRemoveAssignment = async (taskId) => {
    try {
      const user = auth().currentUser;
      if (!user) return;

      const task = items.find(item => item.id === taskId);
      if (task?.status === 'Down Assign') {
        showToast('error', 'Cannot Modify', 'Task is in Down Assign state and cannot be modified');
        return;
      }

      setTaskAssignments(prev => {
        const newAssignments = { ...prev };
        delete newAssignments[taskId];
        return newAssignments;
      });

      const assignmentRef = database().ref(`workOrders/${workOrderId}/taskAssignments/${taskId}`);
      await assignmentRef.remove();

      showToast('success', 'Success', 'Assignment removed successfully');
      setAssignMenuVisible(false);
    } catch (error) {
      console.error('Error removing assignment:', error);
      showToast('error', 'Error', 'Failed to remove assignment');
    }
  };

  const renderUserSelectionModal = () => (
    <Modal
      visible={showUserSelectionModal}
      transparent={true}
      animationType="fade"
      onRequestClose={() => setShowUserSelectionModal(false)}
    >
      <View style={styles.modalOverlay}>
        <View style={styles.userSelectionModal}>
          <View style={styles.modalHeader}>
            <Text style={styles.modalTitle}>Assign Task</Text>
            <TouchableOpacity onPress={() => setShowUserSelectionModal(false)}>
              <MaterialIcons name="close" size={24} color="#666" />
            </TouchableOpacity>
          </View>
          <FlatList
            data={availableUsers}
            keyExtractor={(item) => item.id}
            renderItem={({ item }) => (
              <TouchableOpacity
                style={styles.userItem}
                onPress={() => handleTaskAssignment(selectedTaskId, item.id)}
              >
                <View style={styles.userInfo}>
                  <Text style={styles.userName}>{item.name}</Text>
                  <Text style={styles.userRole}>{item.role}</Text>
                </View>
                <MaterialIcons name="chevron-right" size={24} color="#666" />
              </TouchableOpacity>
            )}
          />
        </View>
      </View>
    </Modal>
  );

  useEffect(() => {
    fetchAvailableUsers();
  }, []);

  // Clear extracted project ref when component unmounts or route params change
  useEffect(() => {
    return () => {
      extractedProjectRef.current = null;
    };
  }, [route.params?.selectedReport]);

  // Handle setting project when extractedProjectRef is available and projectItems are populated
  useEffect(() => {
    if (extractedProjectRef.current && projectItems.length > 0 && country) {
      const isValidProject = projectItems.some(item => item.value === extractedProjectRef.current);
      console.log('Setting project from ref:', {
        extractedProject: extractedProjectRef.current,
        projectItems: projectItems.map(item => item.value),
        isValidProject
      });
      
      if (isValidProject) {
        setProject(extractedProjectRef.current);
        console.log('Project set from ref to:', extractedProjectRef.current);
        extractedProjectRef.current = null;
      }
    }
  }, [extractedProjectRef.current, projectItems, country]);

  // Handle project items based on country selection
  const PROJECT_OPTIONS_BY_COUNTRY = useMemo(() => ({
    Zambia: [
      { label: 'Kansanshi', value: 'Kansanshi' },
      { label: 'Kalumbila', value: 'Kalumbila' },
      { label: 'Kimteto', value: 'Kimteto' },
      { label: 'Kobold', value: 'Kobold' },
      { label: 'FQM Exploration', value: 'FQM Exploration' },
      { label: 'Mimosa', value: 'Mimosa' },
    ],
    DRC: [
      { label: 'Lubumbashi', value: 'Lubumbashi' },
      { label: 'Musompo', value: 'Musompo' },
      { label: 'IME', value: 'IME' },
      { label: 'Kamoa', value: 'Kamoa' },
    ],
  }), []);

  useEffect(() => {
    const nextProjectItems = PROJECT_OPTIONS_BY_COUNTRY[country] || [];
    setProjectItems(nextProjectItems);

    if (!country || (project && !nextProjectItems.some((item) => item.value === project))) {
      setProject(null);
    }
  }, [country, project, PROJECT_OPTIONS_BY_COUNTRY]);

  // Update request number when country, project, or job card count changes
  useEffect(() => {
    const updateRequestNumber = async () => {
      try {
        // If both country and project are selected, generate the request number DYNAMICALLY
        if (country && project) {
          const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
          const projectCode = project.slice(0, 3).toUpperCase();
          const nextNumber = jobCardCount + 1; // jobCardCount now stores the LAST number
          const dynamicRequestNumber = `${countryCode}-${projectCode}-JC-${String(nextNumber).padStart(4, '0')}`;
          
          console.log('📊 Preview request number:', dynamicRequestNumber);
          console.log('   Last number in DB:', jobCardCount);
          console.log('   Next number:', nextNumber);
          
          setRequestNumber(dynamicRequestNumber);
        } else {
          setRequestNumber('Select Country & Project');
        }
      } catch (error) {
        console.error('Error updating request number:', error);
        setRequestNumber('Error generating number');
      }
    };
    updateRequestNumber();
  }, [country, project, jobCardCount]);

  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [datePickerMode, setDatePickerMode] = useState('start'); // 'start' or 'due'
  
  // iOS-specific modal state management
  const [modalKey, setModalKey] = useState(0);

  // Monitor modal state changes
  useEffect(() => {
    console.log('addItemModalVisible changed to:', addItemModalVisible);
    console.log('iosModalVisible changed to:', iosModalVisible);
    
    // Update ref when modal state changes
    const isModalOpen = addItemModalVisible || iosModalVisible;
    console.log('Modal is open:', isModalOpen);
  }, [addItemModalVisible, iosModalVisible]);

  // Add focus listener to handle modal state when returning from other screens
  useEffect(() => {
    const unsubscribe = navigation.addListener('focus', () => {
      console.log('Screen focused - checking modal states');
      console.log('wasModalOpenBeforeNavigation:', wasModalOpenBeforeNavigation);
      console.log('isNavigating:', isNavigating);
      
      // Force a re-render of modals if needed
      if (Platform.OS === 'ios') {
        forceModalUpdate();
      }
      
      // Focus listener is now disabled to prevent double reopening
      console.log('Focus listener: Disabled to prevent conflicts');
    });

    return unsubscribe;
  }, [navigation, wasModalOpenBeforeNavigation, isNavigating]);

  // Add blur listener to ensure modal is closed when leaving the screen
  useEffect(() => {
    const unsubscribe = navigation.addListener('blur', () => {
      console.log('Screen blurred - ensuring modal is closed');
      // Ensure modal is closed when leaving the screen
      setAddItemModalVisible(false);
      setIosModalVisible(false);
    });

    return unsubscribe;
  }, [navigation]);


  const [shouldReopenAddTaskModal, setShouldReopenAddTaskModal] = useState(false);
  const [wasModalOpenBeforeNavigation, setWasModalOpenBeforeNavigation] = useState(false);
  const [iosModalVisible, setIosModalVisible] = useState(false);
  const [isNavigating, setIsNavigating] = useState(false);
  const modalStateRef = useRef({ wasOpen: false, shouldReopen: false });
  
  // Force modal re-render on iOS to prevent issues
  const forceModalUpdate = () => {
    console.log('forceModalUpdate called, current modalKey:', modalKey);
    setModalKey(prev => {
      const newKey = prev + 1;
      console.log('New modalKey:', newKey);
      return newKey;
    });
  };

  // Fetch and filter team members for Assign To modal (matching TeamMemberHome)
  const fetchFilteredTeamMembers = async () => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      console.log('Admin UID:', adminUid);
      if (!adminUid) {
        console.log('No admin UID found');
        return setTeamMembers([]);
      }

      // Fetch admin's data (EXACT same as TeamMemberHome)
      const adminRef = database().ref(`/Globalusers/admins/${adminUid}`);
      const adminSnap = await adminRef.once('value');
      if (!adminSnap.exists()) {
        console.log('Admin data not found');
        return setTeamMembers([]);
      }

      const userData = adminSnap.val();
      console.log('User data:', userData);

      // Set admin profile (EXACT same as TeamMemberHome)
      const adminProfileData = {
        ...userData,
        id: adminUid,
        fullName: userData.name || 'N/A',
        employeeNumber: userData.employeeNumber || 'N/A',
        projects: userData.projects || [],
        countries: userData.countries || [],
        role: userData.role || 'Operator',
        isBlocked: userData.isBlocked || false,
        password: userData.password || 'N/A',
        profilePic: userData.profilePic || null,
      };
      setAdminProfile(adminProfileData);
      console.log('Admin profile set:', adminProfileData);

      // Get permissions (EXACT same as TeamMemberHome)
      const teamEmployModule = userData.modules?.find(m => m.module === 'teamEmploy');
      const permissions = teamEmployModule?.permissions || [];
      const userCountry = userData.countries || [];
      const userProject = userData.projects || [];
      
      console.log('Admin countries:', userCountry);
      console.log('Admin projects:', userProject);
      console.log('Admin permissions:', permissions);

      setUserPermissions(permissions);
      setHasOnlyMineView(permissions.some(p => p.toLowerCase() === 'onlymineview'));

      // Fetch all team members (EXACT same as TeamMemberHome)
      console.log('Fetching all admins from Globalusers/admins...');
      const teamMembersRef = database().ref(`/Globalusers/admins`);
      const snapshot = await teamMembersRef.once('value');
      const data = snapshot.val();
      
      let membersArray = [];
      if (data) {
        membersArray = Object.keys(data).map((key) => ({
          ...data[key],
          id: key,
          fullName: data[key].name || 'N/A',
          employeeNumber: data[key].employeeNumber || 'N/A',
          projects: data[key].projects || [],
          countries: data[key].countries || [],
          role: data[key].role || 'Operator',
          isBlocked: data[key].isBlocked || false,
          password: data[key].password || 'N/A',
          profilePic: data[key].profilePic || null,
        }));
      }

      console.log('Total members found:', membersArray.length);

      // Check for OnlyMineView permission
      const hasOnlyMineView = permissions.some(p => p.toLowerCase() === 'onlymineview');
      setHasOnlyMineView(hasOnlyMineView);
      
      console.log('Has OnlyMineView permission:', hasOnlyMineView);

      if (hasOnlyMineView) {
        // If user has OnlyMineView permission, show only their profile
        console.log('User has OnlyMineView permission, showing only admin profile');
        setTeamMembers([]); // No other team members
        return;
      }

      // Filter team members by country and project (same logic as TeamMemberHome)
      let filteredMembers = membersArray.filter(member => {
        // Skip the current admin (will be added separately)
        if (member.id === adminUid) return false;
        
        // Apply country and project filtering
        const countryMatch = userData.countries.length === 0 || 
          (member.countries && member.countries.some(c => userData.countries.includes(c)));
        const projectMatch = userData.projects.length === 0 || 
          (member.projects && member.projects.some(p => userData.projects.includes(p)));
        
        return countryMatch && projectMatch;
      });
      
      console.log('After filtering:', filteredMembers.length, 'members');
      setTeamMembers(filteredMembers);

    } catch (error) {
      console.error('Error fetching team members:', error);
      setTeamMembers([]);
    }
  };





  // Fetch corrective action statuses when linkInspection changes
  useEffect(() => {
    const fetchCorrectiveStatuses = async () => {
      if (!linkInspection) return setCorrectiveStatuses({});
      try {
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) return setCorrectiveStatuses({});
        const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${linkInspection.id}`);
        const snapshot = await reportRef.once('value');
        const reportData = snapshot.val();
        if (reportData?.correctiveAction?.subItemStatuses) {
          setCorrectiveStatuses(reportData.correctiveAction.subItemStatuses);
        } else {
          setCorrectiveStatuses({});
        }
      } catch (err) {
        setCorrectiveStatuses({});
      }
    };
    fetchCorrectiveStatuses();
  }, [linkInspection]);

  const findAdminUidForInspection = async (inspectionId) => {
    const adminsRef = database().ref('/GlobalInspectionReport/admins');
    const snapshot = await adminsRef.once('value');
    const adminsData = snapshot.val();
    if (!adminsData) return null;
    for (const [adminUid, reports] of Object.entries(adminsData)) {
      if (reports && reports[inspectionId]) {
        return adminUid;
      }
    }
    return null;
  };

  // --- Refactor handleRefresh ---
  const handleRefresh = async () => {
    setRefreshing(true);
    setIsLoading(true);
    try {
      await fetchInitialData();
    } catch (e) {
      // ignore
    } finally {
      setIsLoading(false);
      setRefreshing(false);
    }
  };

  const [refreshing, setRefreshing] = useState(false);

  // Add state for equipment selection modal and equipment list
  const [equipmentModalVisible, setEquipmentModalVisible] = useState(false);
  const [equipmentList, setEquipmentList] = useState([]);
  const [equipmentSearch, setEquipmentSearch] = useState('');
  const [equipmentLoading, setEquipmentLoading] = useState(false);
  
  // Add state for job card count
  const [jobCardCount, setJobCardCount] = useState(0);
  const [jobCardCountLoading, setJobCardCountLoading] = useState(false);
  
  // Add state for inspection selection modal
  const [inspectionModalVisible, setInspectionModalVisible] = useState(false);
  const [availableInspections, setAvailableInspections] = useState([]);
  const [inspectionLoading, setInspectionLoading] = useState(false);
  const [selectedInspection, setSelectedInspection] = useState(null);
  
  // Add state for job card selection modal
  const [jobCardModalVisible, setJobCardModalVisible] = useState(false);
  const [availableJobCards, setAvailableJobCards] = useState([]);
  const [jobCardLoading, setJobCardLoading] = useState(false);
  const [selectedJobCard, setSelectedJobCard] = useState(null);
  const [equipmentUserInfo, setEquipmentUserInfo] = useState(null);

  // Fetch job card count and last request number DYNAMICALLY
  const fetchJobCardCount = async () => {
    try {
      setJobCardCountLoading(true);
      console.log('📊 Fetching job card count and last request number...');
      
      const response = await JobCardAPI.getAllJobCards();
      
      if (response.success && response.data && response.data.length > 0) {
        const count = response.data.length;
        
        // Find the LAST request number dynamically for reference
        const sortedJobCards = response.data.sort((a, b) => {
          const numA = extractNumberFromRequestNumber(a.requestNumber);
          const numB = extractNumberFromRequestNumber(b.requestNumber);
          return numB - numA; // Descending order
        });
        
        const lastRequestNumber = sortedJobCards[0].requestNumber;
        const lastNumber = extractNumberFromRequestNumber(lastRequestNumber);
        
        console.log('📊 Total job cards in MongoDB:', count);
        console.log('📊 Last request number:', lastRequestNumber);
        console.log('📊 Extracted last number:', lastNumber);
        console.log('✅ Using TOTAL COUNT for next number:', count + 1);
        
        setJobCardCount(count); // Store the TOTAL COUNT, not extracted number
      } else {
        console.log('ℹ️ No job cards found, starting from 0');
        setJobCardCount(0);
      }
    } catch (error) {
      console.error('❌ Error fetching job card count:', error);
      setJobCardCount(0);
    } finally {
      setJobCardCountLoading(false);
    }
  };

  // Fetch inspection reports for selection

  const fetchInspectionReports = async () => {
    try {
      setInspectionLoading(true);
      console.log('📊 Fetching ALL inspection reports (no filtering)...');
      
      const userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        showToast('error', 'Error', 'No user logged in');
        return;
      }

      let allInspections = [];
      
      // Try to get all reports first
      try {
        console.log('📊 Trying to get all reports...');
        const allResponse = await InspectionAPI.getAllReports({ page: 1, limit: 1000 });
        console.log('📊 getAllReports response:', allResponse);
        if (allResponse.success && allResponse.data && allResponse.data.length > 0) {
          allInspections = allResponse.data;
          console.log('✅ All reports found:', allInspections.length);
        } else {
          console.log('❌ getAllReports returned empty, trying multiple users...');
        }
      } catch (allError) {
        console.log('❌ All reports failed, trying multiple users:', allError.message);
      }

      // If getAllReports didn't work, try multiple approaches to get all possible reports
      if (allInspections.length === 0) {
        console.log('📊 getAllReports failed, trying alternative approaches...');
        
        // First, try current user
        try {
          console.log(`📊 Trying current user: ${userId}`);
          const currentUserResponse = await InspectionAPI.getInspections(userId, { page: 1, limit: 1000 });
          console.log(`📊 Current user response:`, currentUserResponse);
          
          if (currentUserResponse.success && currentUserResponse.data && currentUserResponse.data.length > 0) {
            allInspections = [...allInspections, ...currentUserResponse.data];
            console.log(`✅ Current user added ${currentUserResponse.data.length} reports (total: ${allInspections.length})`);
          }
        } catch (currentUserError) {
          console.log(`❌ Current user failed:`, currentUserError.message);
        }

        // Try some common user patterns that might exist in your system
        const commonUserPatterns = [
          'admin',
          'administrator', 
          'test_admin',
          'proadmin',
          'user_admin',
          'admin_user',
          'test_user',
          'demo_user',
          'sample_user'
        ];

        console.log('📊 Trying common user patterns...');
        
        for (const userPattern of commonUserPatterns) {
          try {
            console.log(`📊 Trying user pattern: ${userPattern}`);
            const userResponse = await InspectionAPI.getInspections(userPattern, { page: 1, limit: 1000 });
            console.log(`📊 ${userPattern} response:`, userResponse);
            
            if (userResponse.success && userResponse.data && userResponse.data.length > 0) {
              // Add reports that don't already exist (avoid duplicates)
              const newReports = userResponse.data.filter(newReport => 
                !allInspections.some(existingReport => existingReport._id === newReport._id)
              );
              allInspections = [...allInspections, ...newReports];
              console.log(`✅ ${userPattern} added ${newReports.length} new reports (total: ${allInspections.length})`);
            } else {
              console.log(`❌ ${userPattern} returned empty`);
            }
          } catch (userError) {
            console.log(`❌ ${userPattern} failed:`, userError.message);
          }
        }

        // If still no reports, try to get a list of all users first, then fetch their reports
        if (allInspections.length === 0) {
          console.log('📊 No reports found from common patterns, trying to discover users...');
          try {
            // Try to get users from the users API
            const usersResponse = await fetch('https://api.titandrillingzm.com:6003/users/all', {
              method: 'GET',
              headers: {
                'Content-Type': 'application/json',
              },
            });
            
            if (usersResponse.ok) {
              const usersData = await usersResponse.json();
              console.log('📊 Users API response:', usersData);
              
              if (usersData.success && usersData.data && Array.isArray(usersData.data)) {
                console.log(`📊 Found ${usersData.data.length} users, trying to get their reports...`);
                
                // Try to get reports from each user (limit to first 10 to avoid too many requests)
                const usersToTry = usersData.data.slice(0, 10);
                for (const user of usersToTry) {
                  if (user._id || user.id) {
                    const userIdToTry = user._id || user.id;
                    try {
                      console.log(`📊 Trying discovered user: ${userIdToTry}`);
                      const userResponse = await InspectionAPI.getInspections(userIdToTry, { page: 1, limit: 1000 });
                      
                      if (userResponse.success && userResponse.data && userResponse.data.length > 0) {
                        const newReports = userResponse.data.filter(newReport => 
                          !allInspections.some(existingReport => existingReport._id === newReport._id)
                        );
                        allInspections = [...allInspections, ...newReports];
                        console.log(`✅ ${userIdToTry} added ${newReports.length} new reports (total: ${allInspections.length})`);
                      }
                    } catch (userError) {
                      console.log(`❌ ${userIdToTry} failed:`, userError.message);
                    }
                  }
                }
              }
            }
          } catch (usersError) {
            console.log('❌ Failed to discover users:', usersError.message);
          }
        }
      }

      console.log('📊 Final inspections count (from all users):', allInspections.length);
      setAvailableInspections(allInspections);
      
    } catch (error) {
      console.error('❌ Error fetching inspection reports:', error);
      setAvailableInspections([]);
      showToast('error', 'Error', 'Failed to load inspection reports');
    } finally {
      setInspectionLoading(false);
    }
  };

  // Open inspection selection modal
  const openInspectionModal = async () => {
    console.log('🔍 Opening inspection modal...');
    try {
      setInspectionModalVisible(true);
      console.log('🔍 Modal visibility set to true');
      await fetchInspectionReports();
      console.log('🔍 All inspection reports fetched');
    } catch (error) {
      console.error('❌ Error in openInspectionModal:', error);
    }
  };

  // Handle inspection selection
  const handleInspectionSelect = (inspection) => {
    setSelectedInspection(inspection);
    setLinkInspection(inspection);
    setInspectionModalVisible(false);
    
    // Extract faulty items from inspection and add them as tasks
    const faultyTasks = extractFaultyItemsFromInspection(inspection);
    if (faultyTasks.length > 0) {
      setItems(faultyTasks);
      console.log('✅ Added faulty items as tasks:', faultyTasks.length);
      showToast('success', 'Success', `Inspection linked and ${faultyTasks.length} faulty items added as tasks`);
    } else {
      console.log('📊 No faulty items found in inspection');
      showToast('success', 'Success', 'Inspection linked successfully');
    }
  };

  // Remove selected inspection
  const removeSelectedInspection = () => {
    setSelectedInspection(null);
    setLinkInspection(null);
    // Clear tasks that were added from the inspection
    setItems([]);
    console.log('📊 Inspection removed and tasks cleared');
    showToast('success', 'Success', 'Inspection unlinked and tasks cleared');
  };

  // Fetch job cards for selection
  const fetchJobCards = async () => {
    try {
      setJobCardLoading(true);
      console.log('📊 Fetching job cards...');
      
      const response = await JobCardAPI.getAllJobCards();
      
      console.log('📊 Full Job Card API response:', JSON.stringify(response, null, 2));
      
      if (response.success && response.data) {
        setAvailableJobCards(response.data);
        console.log('✅ Job cards fetched:', response.data.length);
        console.log('📊 First job card:', response.data[0]);
      } else {
        console.log('❌ Failed to fetch job cards:', response.message);
        console.log('📊 Response structure:', { success: response.success, hasData: !!response.data });
        setAvailableJobCards([]);
      }
    } catch (error) {
      console.error('❌ Error fetching job cards:', error);
      setAvailableJobCards([]);
      showToast('error', 'Error', 'Failed to load job cards');
    } finally {
      setJobCardLoading(false);
    }
  };

  // Open job card selection modal
  const openJobCardModal = async () => {
    setJobCardModalVisible(true);
    await fetchJobCards();
  };

  // Handle job card selection
  const handleJobCardSelect = (jobCard) => {
    setSelectedJobCard(jobCard);
    setLinkedWorkOrder(jobCard);
    setJobCardModalVisible(false);
    showToast('success', 'Success', 'Job card linked successfully');
  };

  // Remove selected job card
  const removeSelectedJobCard = () => {
    setSelectedJobCard(null);
    setLinkedWorkOrder(null);
    showToast('success', 'Success', 'Job card unlinked');
  };

  // Fetch user info and equipment list for modal
  const fetchEquipmentList = async () => {
    setEquipmentLoading(true);
    try {
      let userUid = await AsyncStorage.getItem('userUid');
      let userData = null;
      if (userUid) {
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        const snapshot = await userRef.once('value');
        userData = snapshot.val();
      }
      if (!userData) {
        const user = auth().currentUser;
        if (!user) throw new Error('User not authenticated');
        const email = user.email.replace(/\./g, '_');
        const userRef = database().ref(`/Globalusers/admins/${email}`);
        const snapshot = await userRef.once('value');
        userData = snapshot.val();
        if (userData && userData.userUid) {
          userUid = userData.userUid;
          await AsyncStorage.setItem('userUid', userUid);
        }
      }
      setEquipmentUserInfo(userData);
      // Fetch equipment from Firebase (global)
      let parsedData = [];
      const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
      const snapshot = await globalRef.once('value');
      const data = snapshot.val() || [];
      parsedData = Array.isArray(data)
        ? data.map((item, index) => ({ ...item, firebaseKey: `equip_${index}` }))
        : [];
      // Filter by country and project if set
      let filtered = filterEquipmentByPermission(parsedData, userData);
      if (country) filtered = filtered.filter(eq => eq.country === country);
      if (project) filtered = filtered.filter(eq => eq.project === project);
      setEquipmentList(filtered);
    } catch (err) {
      setEquipmentList([]);
    } finally {
      setEquipmentLoading(false);
    }
  };

  // Equipment selection modal UI
  const renderEquipmentModal = () => (
    <Modal
      visible={equipmentModalVisible}
      transparent={true}
      animationType="slide"
      onRequestClose={() => setEquipmentModalVisible(false)}
    >
      <View style={[styles.modalOverlay, { justifyContent: 'flex-end' }]}> 
        <View style={[styles.modalContent, { maxHeight: '80%' }]}> 
          <Text style={styles.modalTitle}>Select Equipment</Text>
          <TextInput
            style={[styles.inputmain, { marginBottom: 10 }]}
            placeholder="Search equipment..."
            placeholderTextColor="#888"
            value={equipmentSearch}
            onChangeText={setEquipmentSearch}
          />
          {equipmentLoading ? (
            <View style={{ alignItems: 'center', marginVertical: 20 }}>
              <Animated.View style={{ transform: [{ rotate: rotateInterpolate }] }}>
                <MaterialIcons name="autorenew" size={32} color="#015185" />
              </Animated.View>
              <Text style={{ color: '#015185', marginTop: 8 }}>Loading...</Text>
            </View>
          ) : (
            <FlatList
              data={equipmentList.filter(eq =>
                !equipmentSearch ||
                (eq.equipmentName?.toLowerCase().includes(equipmentSearch.toLowerCase()) ||
                  eq.equipmentNumber?.toLowerCase().includes(equipmentSearch.toLowerCase()) ||
                  eq.mainCategory?.toLowerCase().includes(equipmentSearch.toLowerCase()))
              )}
              keyExtractor={item => item.id}
              ListHeaderComponent={selectedEquipment ? (
                <TouchableOpacity
                  style={{ backgroundColor: '#e6f0ff', borderRadius: 6, padding: 10, marginBottom: 10, borderWidth: 1, borderColor: '#015185' }}
                  onPress={() => {
                    setSelectedEquipment(selectedEquipment);
                    setEquipmentModalVisible(false);
                  }}
                >
                  <Text style={{ color: '#015185', fontWeight: 'bold' }}>Selected: {selectedEquipment.equipmentName || selectedEquipment.name}</Text>
                  <Text style={{ color: '#888', fontSize: 12 }}>{selectedEquipment.equipmentNumber}</Text>
                  <Text style={{ color: '#888', fontSize: 12 }}>{selectedEquipment.mainCategory}</Text>
                </TouchableOpacity>
              ) : null}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={{ padding: 10, borderBottomWidth: 1, borderBottomColor: '#eee', backgroundColor: selectedEquipment?.id === item.id ? '#e6f0ff' : '#fff' }}
                  onPress={() => {
                    setSelectedEquipment(item);
                    setEquipmentModalVisible(false);
                    setErrors({ ...errors, equipment: '' });
                  }}
                >
                  <Text style={{ color: '#015185', fontWeight: selectedEquipment?.id === item.id ? 'bold' : '500' }}>{item.equipmentName || item.name}</Text>
                  <Text style={{ color: '#888', fontSize: 12 }}>{item.equipmentNumber}</Text>
                  <Text style={{ color: '#888', fontSize: 12 }}>{item.mainCategory}</Text>
                  {selectedEquipment?.id === item.id && (
                    <Text style={{ color: '#28a745', fontWeight: 'bold', fontSize: 12 }}>Selected</Text>
                  )}
                </TouchableOpacity>
              )}
              ListEmptyComponent={<Text style={{ color: '#888', textAlign: 'center', marginTop: 20 }}>No equipment found.</Text>}
              style={{ marginBottom: 10 }}
            />
          )}
          <TouchableOpacity
            style={[styles.modalCloseButton, { marginTop: 8 }]}
            onPress={() => setEquipmentModalVisible(false)}
          >
            <Text style={styles.modalCloseText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );



  // 1. Add a function to fetch and filter job cards for the popup
  const fetchFilteredWorkOrders = async () => {
    try {
      let userId = await AsyncStorage.getItem('userUid');
      if (!userId) return setWorkOrders([]);
      const workOrdersRef = database().ref(`/GlobalWorkOrders/admins/${userId}`);
      const snapshot = await workOrdersRef.once('value');
      const data = snapshot.val();
      if (!data) return setWorkOrders([]);
      // Remove lastRequestNumber if present
      const filteredData = Object.keys(data)
        .filter(key => key !== 'lastRequestNumber')
        .map(key => ({ id: key, ...data[key] }));
      // If equipment is selected, filter by equipment id
      if (selectedEquipment && selectedEquipment.id) {
        setWorkOrders(filteredData.filter(wo => wo.equipment && wo.equipment.id === selectedEquipment.id));
      } else {
        setWorkOrders(filteredData);
      }
    } catch (err) {
      setWorkOrders([]);
    }
  };

  // Add a case-insensitive check for onlyMineView permission
  const hasOnlyMineView = Array.isArray(userPermissions)
    ? userPermissions.some(p => p && typeof p === 'string' && p.toLowerCase() === 'onlymineview')
    : false;

  // --- Add state for loader when opening assign-to modal ---


  // --- Refactor open assign-to modal for Add Task ---
  const openAssignToModal = async () => {
    console.log('Navigating to UsersAssignToJC...');
    // Navigate to UsersAssignToJC with current selected users and callback
    navigation.navigate('UsersAssignToJC', {
      selectedUsers: assignTo,
      onUserSelection: (selectedUsers) => {
        console.log('Received selected users from callback:', selectedUsers);
        console.log('Selected users details:', selectedUsers.map(user => ({
          id: user.id,
          name: user.name,
          fullName: user.fullName,
          employeeNumber: user.employeeNumber,
          role: user.role
        })));
        
        // Ensure all users have consistent data structure
        const normalizedUsers = selectedUsers.map(user => ({
          ...user,
          name: user.name || user.fullName || 'Unknown User',
          fullName: user.fullName || user.name || 'Unknown User',
          employeeNumber: user.employeeNumber || 'N/A',
          role: user.role || 'Operator'
        }));
        
        console.log('Normalized users:', normalizedUsers);
        setAssignTo(normalizedUsers);
        
        // Show success message
        if (normalizedUsers.length > 0) {
          showToast('success', 'Users Selected', `${normalizedUsers.length} user(s) have been assigned to this work order.`);
        } else {
          showToast('info', 'No Users Selected', 'No users were selected for this work order.');
        }
      }
    });
  };

  // --- Open assign-to modal for task (separate from main assign-to) ---
  const openTaskAssignToModal = async () => {
    console.log('openTaskAssignToModal called - navigating to UsersAssignToJC');
    
    // Store the current modal state before navigation
    const modalWasOpen = addItemModalVisible || iosModalVisible;
    modalStateRef.current = { wasOpen: modalWasOpen, shouldReopen: modalWasOpen };
    console.log('Stored modal state before navigation:', modalWasOpen);
    
    // Close the modal before navigation
    setAddItemModalVisible(false);
    setIosModalVisible(false);
    
    // Navigate to UsersAssignToJC with current selected users and callback
    navigation.navigate('UsersAssignToJC', {
      selectedUsers: tempAssignTo,
      onUserSelection: (selectedUsers) => {
        console.log('Received selected users from callback for task:', selectedUsers);
        
        // Ensure all users have consistent data structure
        const normalizedUsers = selectedUsers.map(user => ({
          ...user,
          name: user.name || user.fullName || 'Unknown User',
          fullName: user.fullName || user.name || 'Unknown User',
          employeeNumber: user.employeeNumber || 'N/A',
          role: user.role || 'Operator'
        }));
        
        setTempAssignTo(normalizedUsers);
        
        // Show success message
        if (normalizedUsers.length > 0) {
          showToast('success', 'Users Selected', `${normalizedUsers.length} user(s) have been assigned to this task.`);
        } else {
          showToast('info', 'No Users Selected', 'No users were selected for this task.');
        }
        
        // Reopen modal if it was open before navigation
        if (modalStateRef.current.shouldReopen) {
          console.log('Reopening modal after user selection');
          setTimeout(() => {
            setAddItemModalVisible(true);
            if (Platform.OS === 'ios') {
              setIosModalVisible(true);
            }
            console.log('Modal reopened successfully');
          }, 300);
        }
      }
    });
  };

  // Loader shows for 3.5 seconds, then hides automatically
  useEffect(() => {
    let timer;
    if (isLoading) {
      timer = setTimeout(() => setIsLoading(false), 3500);
    }
    return () => clearTimeout(timer);
  }, [isLoading]);

  // iOS-specific performance optimization
  useEffect(() => {
    if (Platform.OS === 'ios') {
      // Pre-warm the component for better performance
      const preWarm = () => {
        // This helps iOS prepare the component for rendering
        requestAnimationFrame(() => {
          // Force a layout pass
          if (scrollViewRef.current) {
            scrollViewRef.current.scrollTo({ y: 0, animated: false });
          }
        });
      };
      
      // Run after component mounts
      const timer = setTimeout(preWarm, 100);
      return () => clearTimeout(timer);
    }
  }, []);

  useEffect(() => {
    if (!linkInspection) return;
    const fetchInspectionTasks = async () => {
      try {
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) return;
        // Fetch the inspection report from the correct global path
        const reportRef = database().ref(`/GlobalInspectionReport/admins/${adminUid}/${linkInspection.id}`);
        const snapshot = await reportRef.once('value');
        const reportData = snapshot.val();
        if (!reportData?.step2?.sections || !reportData?.step2?.selections) return;
        // Find all subItems with 'Not Ok', 'Reject', or 'Faulty'
        const inspectionTasks = [];
        reportData.step2.sections.forEach(section => {
          section.subItems.forEach(subItem => {
            const sel = reportData.step2.selections[subItem.key];
            if (
              (sel === 'Not Ok' || sel === 'Reject' || sel === 'Faulty') &&
              subItem.itemName && subItem.itemName.trim() !== ''
            ) {
              inspectionTasks.push({
                id: `task_${subItem.key}`,
                category: 'Task',
                text: `Fix ${subItem.itemName}`,
                type: 'Repair',
                note: `From inspection: ${linkInspection.name}`,
                assignTo: [],
                status: 'Initiated',
                section: section.name,
                inspectionId: linkInspection.id
              });
            }
          });
        });
        // Only add new tasks that are not already present
        setItems(prev => {
          const existingTaskIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
          const newTasks = inspectionTasks.filter(t => !existingTaskIds.has(t.id));
          // Remove old auto-generated tasks for this inspection if not in checklist anymore
          const filteredPrev = prev.filter(i => i.category !== 'Task' || !i.id.startsWith('task_'));
          return [...filteredPrev, ...newTasks];
        });
      } catch (err) {
        // Silent fail
      }
    };
    fetchInspectionTasks();
  }, [linkInspection]);

  const handleSaveInspections = () => {
    setLinkInspectionModalVisible(false);
  };

  // Add a useEffect to add faulty items as tasks when linkInspection changes
  useEffect(() => {
    if (!linkInspection || !linkInspection.step2 || !linkInspection.step2.sections || !linkInspection.step2.selections) return;
    // Find all faulty items
    const faultyTasks = linkInspection.step2.sections.flatMap(section =>
      (section.subItems || []).filter(subItem =>
        ['Faulty', 'Not Ok', 'Reject'].includes(linkInspection.step2.selections[subItem.key])
      ).map(subItem => ({
        id: `faulty_${linkInspection.id}_${subItem.key}`,
        category: 'Task',
        text: subItem.itemName,
        type: 'Repair',
        note: `Auto-generated from inspection: ${linkInspection.name || linkInspection.step1?.requestNumber || ''}`,
        assignTo: [],
        status: 'Initiated',
        section: section.name,
        inspectionId: linkInspection.id,
      }))
    );
    // Only add new faulty tasks that are not already present
    setChecklistItems(prev => {
      const existingIds = new Set(prev.filter(i => i.category === 'Task').map(i => i.id));
      const newOnes = faultyTasks.filter(t => !existingIds.has(t.id));
      return [...prev, ...newOnes];
    });
  }, [linkInspection]);

  // In the modal, after the FlatList, show a preview of faulty items for the selected inspection
  // ...
  {linkInspection && linkInspection.step2 && linkInspection.step2.sections && linkInspection.step2.selections && (
    <View style={{ marginTop: 16, backgroundColor: '#f9f9f9', borderRadius: 8, padding: 12 }}>
      <Text style={{ fontWeight: 'bold', fontSize: 16, marginBottom: 8 }}>Faulty Items / Tasks</Text>
      {(() => {
        const faultyItems = linkInspection.step2.sections.flatMap(section =>
          (section.subItems || []).filter(subItem =>
            ['Faulty', 'Not Ok', 'Reject'].includes(linkInspection.step2.selections[subItem.key])
          ).map(subItem => ({
            itemName: subItem.itemName,
            section: section.name,
          }))
        );
        if (faultyItems.length === 0) {
          return <Text style={styles.defaultText}>No faulty items found in this inspection.</Text>;
        }
        return faultyItems.map((item, idx) => (
          <View key={idx} style={{ flexDirection: 'row', alignItems: 'center', marginBottom: 6 }}>
            <MaterialIcons name="error-outline" size={20} color="#e74c3c" style={{ marginRight: 8 }} />
            <Text style={{ color: '#333', fontSize: 15 }}>{item.itemName} <Text style={{ color: '#888' }}>({item.section})</Text></Text>
          </View>
        ));
      })()}
    </View>
  )}
  // ... existing code ...

  {linkInspection && (
    <View style={{ marginTop: 16, backgroundColor: '#f9f9f9', borderRadius: 8, padding: 12 }}>
      <Text style={{ fontWeight: 'bold', fontSize: 16, marginBottom: 8 }}>Inspection Details</Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Request #: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.name || linkInspection.step1?.requestNumber || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Equipment: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.equipmentName || linkInspection.step1?.selectedEquipment?.equipmentName || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Date: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.createdAt ? new Date(linkInspection.createdAt).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Inspector: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.step1?.inspector || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Project: <Text style={{ color: '#333', fontWeight: 'normal' }}>{linkInspection.step1?.project || 'N/A'}</Text></Text>
      <Text style={{ color: '#015185', fontWeight: 'bold' }}>Faulty Items: <Text style={{ color: linkInspection.step3?.faultyItems > 0 ? '#e74c3c' : '#2ecc40', fontWeight: 'bold' }}>{linkInspection.step3?.faultyItems || 0}</Text></Text>
      <Text style={{ fontWeight: 'bold', fontSize: 15, marginTop: 8, marginBottom: 4 }}>Faulty Item List:</Text>
      {linkInspection.step2 && linkInspection.step2.sections && linkInspection.step2.selections ? (
        (() => {
          const faultyItems = linkInspection.step2.sections.flatMap(section =>
            (section.subItems || []).filter(subItem =>
              ['Faulty', 'Not Ok', 'Reject'].includes(linkInspection.step2.selections[subItem.key])
            ).map(subItem => ({
              itemName: subItem.itemName,
              section: section.name,
            }))
          );
          if (faultyItems.length === 0) {
            return <Text style={{ color: '#888', fontSize: 14 }}>No faulty items found in this inspection.</Text>;
          }
          return faultyItems.map((item, idx) => (
            <Text key={idx} style={{ color: '#e74c3c', fontSize: 14 }}>
              {item.itemName} <Text style={{ color: '#888' }}>({item.section})</Text>
            </Text>
          ));
        })()
      ) : (
        <Text style={{ color: '#888', fontSize: 14 }}>No faulty items found in this inspection.</Text>
      )}
    </View>
  )}
  // ... existing code ...

  const extractFaultyItemsFromInspection = (inspection) => {
    console.log('📊 Extracting faulty items from inspection:', inspection);
    let items = [];
    
    try {
      // Check different possible data structures
      console.log('📊 Inspection structure check:');
      console.log('📊 - step2 exists:', !!inspection?.step2);
      console.log('📊 - step2.sections exists:', !!inspection?.step2?.sections);
      console.log('📊 - step2.selections exists:', !!inspection?.step2?.selections);
      console.log('📊 - step3 exists:', !!inspection?.step3);
      console.log('📊 - faultyItems exists:', !!inspection?.step3?.faultyItems);
      
      // Method 1: Try step2 structure
      if (inspection?.step2?.sections && inspection?.step2?.selections) {
        console.log('📊 Using step2 structure');
        const selections = inspection.step2.selections;
        const sections = inspection.step2.sections;
        
        console.log('📊 Selections:', selections);
        console.log('📊 Sections count:', sections.length);
        
        sections.forEach((section, sectionIndex) => {
          console.log(`📊 Processing section ${sectionIndex}:`, section.name);
          if (Array.isArray(section.subItems)) {
            section.subItems.forEach((subItem, itemIndex) => {
              const sel = selections[subItem.key];
              const isFaulty = sel === 'Faulty' || sel === 'Not Ok' || sel === 'Reject';
              console.log(`📊 Item ${itemIndex}: ${subItem.itemName} - Selection: ${sel} - Faulty: ${isFaulty}`);
              
              if (isFaulty) {
                items.push({
                  id: subItem.key || `item_${sectionIndex}_${itemIndex}`,
                  text: subItem.itemName || subItem.name || 'Unknown Item',
                  fileName: subItem.fileName || 'No file',
                  category: 'Task',
                  sectionName: section.name || 'Unknown Section',
                  status: 'Pending',
                  type: 'Repair',
                });
              }
            });
          }
        });
      }
      
      // Method 2: Try step3.faultyItems structure
      if (items.length === 0 && inspection?.step3?.faultyItems) {
        console.log('📊 Using step3.faultyItems structure');
        const faultyItems = inspection.step3.faultyItems;
        
        if (Array.isArray(faultyItems)) {
          faultyItems.forEach((item, index) => {
            items.push({
              id: item.id || item.key || `faulty_${index}`,
              text: item.itemName || item.name || item.text || 'Faulty Item',
              fileName: item.fileName || 'No file',
              category: 'Task',
              sectionName: item.sectionName || item.section || 'Unknown Section',
              status: 'Pending',
              type: 'Repair',
            });
          });
        }
      }
      
      // Method 3: Try to find faulty items in any other structure
      if (items.length === 0) {
        console.log('📊 Searching for faulty items in other structures...');
        
        // Look for any items marked as faulty in the inspection data
        const searchForFaultyItems = (obj, path = '') => {
          if (typeof obj === 'object' && obj !== null) {
            Object.entries(obj).forEach(([key, value]) => {
              if (key.toLowerCase().includes('faulty') && Array.isArray(value)) {
                console.log(`📊 Found potential faulty items at ${path}.${key}:`, value);
                value.forEach((item, index) => {
                  items.push({
                    id: item.id || item.key || `found_${index}`,
                    text: item.itemName || item.name || item.text || 'Faulty Item',
                    fileName: item.fileName || 'No file',
                    category: 'Task',
                    sectionName: item.sectionName || item.section || 'Unknown Section',
                    status: 'Pending',
                    type: 'Repair',
                  });
                });
              }
              if (typeof value === 'object') {
                searchForFaultyItems(value, `${path}.${key}`);
              }
            });
          }
        };
        
        searchForFaultyItems(inspection);
      }
      
      // If still no items found, create placeholder items based on the inspection info
      if (items.length === 0) {
        console.log('📊 No faulty items found, creating placeholder tasks...');
        
        // Try to get faulty count from different possible locations
        let faultyCount = 0;
        
        // Check step3.faultyItems
        if (inspection?.step3?.faultyItems) {
          faultyCount = typeof inspection.step3.faultyItems === 'number' 
            ? inspection.step3.faultyItems 
            : (Array.isArray(inspection.step3.faultyItems) ? inspection.step3.faultyItems.length : 0);
        }
        
        // Check if there's a faulty count in the inspection metadata
        if (faultyCount === 0 && inspection?.faultyItems) {
          faultyCount = typeof inspection.faultyItems === 'number' 
            ? inspection.faultyItems 
            : (Array.isArray(inspection.faultyItems) ? inspection.faultyItems.length : 0);
        }
        
        // Check if there's a faulty count in step1
        if (faultyCount === 0 && inspection?.step1?.faultyItems) {
          faultyCount = typeof inspection.step1.faultyItems === 'number' 
            ? inspection.step1.faultyItems 
            : (Array.isArray(inspection.step1.faultyItems) ? inspection.step1.faultyItems.length : 0);
        }
        
        // If we have a faulty count, create placeholder tasks
        if (faultyCount > 0) {
          console.log(`📊 Creating ${faultyCount} placeholder tasks based on faulty count`);
          for (let i = 0; i < faultyCount; i++) {
            items.push({
              id: `placeholder_${i}`,
              text: `Faulty Item ${i + 1}`,
              fileName: 'No file',
              category: 'Task',
              sectionName: 'Unknown Section',
              status: 'Pending',
              type: 'Repair',
            });
          }
        } else {
          // If no faulty count found, check if there are any "Faulty" selections in the data
          console.log('📊 No faulty count found, searching for faulty selections...');
          
          const searchForFaultySelections = (obj, path = '') => {
            if (typeof obj === 'object' && obj !== null) {
              Object.entries(obj).forEach(([key, value]) => {
                if (typeof value === 'string' && (value === 'Faulty' || value === 'Not Ok' || value === 'Reject')) {
                  console.log(`📊 Found faulty selection at ${path}.${key}: ${value}`);
                  items.push({
                    id: `faulty_selection_${items.length}`,
                    text: `Faulty Item from ${key}`,
                    fileName: 'No file',
                    category: 'Task',
                    sectionName: 'Unknown Section',
                    status: 'Pending',
                    type: 'Repair',
                  });
                }
                if (typeof value === 'object') {
                  searchForFaultySelections(value, `${path}.${key}`);
                }
              });
            }
          };
          
          searchForFaultySelections(inspection);
        }
      }
      
    } catch (error) {
      console.error('❌ Error extracting faulty items:', error);
    }
    
    console.log('📊 Final extracted faulty items:', items.length, items);
    return items;
  };

  const handleInspectionSelected = (inspection) => {
    const faultyTasks = extractFaultyItemsFromInspection(inspection);
    setItems(faultyTasks);
    // ... any other state updates
  };

  // ... existing code ...
  // 1. Add state to track which task's menu is open:
  const [taskMenuVisibleId, setTaskMenuVisibleId] = useState(null);
  const [taskDropdownAnchor, setTaskDropdownAnchor] = useState(null);
  // ... existing code ...
  const renderTaskItem = ({ item }) => {
    const statusObj = statuses.find(opt => opt.label === item.status) || statuses[0];
    return (
      <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingVertical: 8 }}>
        <View style={{ flex: 1 }}>
          <Text style={styles.itemText}>{item.text}</Text>
          {/* Show type (Repair/Replace) in red */}
          {item.type && (
            <Text style={{ color: 'red', fontWeight: 'bold', fontSize: 14, marginTop: 2 }}>
              {item.type}
            </Text>
          )}
          {item.note && <Text style={styles.itemNote}>{item.note}</Text>}
        </View>
        {/* Status button just left of three-dot menu */}
        <TouchableOpacity
          style={{
            paddingVertical: 4,
            paddingHorizontal: 10,
            borderRadius: 12,
            backgroundColor: statusObj.color,
            marginRight: 4,
          }}
          onPress={event => {
            setTaskStatusMenuVisibleId(taskStatusMenuVisibleId === item.id ? null : item.id);
            if (event && event.nativeEvent) {
              setTaskDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
            }
          }}
        >
          <Text style={{ color: '#fff', fontWeight: 'bold', fontSize: 13 }}>{item.status}</Text>
        </TouchableOpacity>
        {/* Status dropdown as Modal */}
        {taskStatusMenuVisibleId === item.id && taskDropdownAnchor && (
          <Modal
            visible={true}
            transparent
            animationType="fade"
            onRequestClose={() => setTaskStatusMenuVisibleId(null)}
          >
            <TouchableOpacity
              style={{ flex: 1 }}
              activeOpacity={1}
              onPress={() => setTaskStatusMenuVisibleId(null)}
            >
              <View
                style={{
                  position: 'absolute',
                  left: taskDropdownAnchor.x - 120, // Adjust width as needed
                  top: taskDropdownAnchor.y - 160, // Move further up
                  backgroundColor: '#fff',
                  borderRadius: 8,
                  elevation: 5,
                  zIndex: 200,
                  shadowColor: '#000',
                  shadowOpacity: 0.1,
                  shadowRadius: 4,
                  minWidth: 120,
                }}
              >
                {statuses.map(opt => (
                  <TouchableOpacity
                    key={opt.label}
                    style={{ flexDirection: 'row', alignItems: 'center', padding: 10 }}
                    onPress={() => {
                      updateItemStatus(item.id, opt.label);
                      setTaskStatusMenuVisibleId(null);
                    }}
                  >
                    <View style={{ width: 16, height: 16, borderRadius: 8, backgroundColor: opt.color, marginRight: 8, borderWidth: item.status === opt.label ? 2 : 0, borderColor: '#333' }} />
                    <Text style={{ color: opt.color, fontWeight: item.status === opt.label ? 'bold' : 'normal' }}>{opt.label}</Text>
                  </TouchableOpacity>
                ))}
              </View>
            </TouchableOpacity>
          </Modal>
        )}
        {/* Three-dot menu button */}
        <TouchableOpacity
          style={styles.threeDotsButton}
          onPress={event => {
            setTaskMenuVisibleId(taskMenuVisibleId === item.id ? null : item.id);
            if (event && event.nativeEvent) {
              setTaskDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
            }
          }}
        >
          <MaterialIcons name="more-vert" size={24} color="#015185" />
        </TouchableOpacity>
        {taskMenuVisibleId === item.id && taskDropdownAnchor && (
          <Modal
            visible={true}
            transparent
            animationType="fade"
            onRequestClose={() => setTaskMenuVisibleId(null)}
          >
            <TouchableOpacity
              style={{ flex: 1 }}
              activeOpacity={1}
              onPress={() => setTaskMenuVisibleId(null)}
            >
              <View
                style={{
                  position: 'absolute',
                  left: taskDropdownAnchor.x - 120,
                  top: taskDropdownAnchor.y - 90,
                  backgroundColor: '#fff',
                  borderRadius: 8,
                  elevation: 5,
                  zIndex: 200,
                  shadowColor: '#000',
                  shadowOpacity: 0.1,
                  shadowRadius: 4,
                  minWidth: 120,
                }}
              >
                <TouchableOpacity
                  style={{ padding: 12 }}
                  onPress={() => {
                    setTaskMenuVisibleId(null);
                    setEditItem(item);
                    setNewItemText(item.text);
                    setNote(item.note || '');
                    setEditModalVisible(true);
                  }}
                >
                  <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={{ padding: 12 }}
                  onPress={() => {
                    setTaskMenuVisibleId(null);
                    deleteItem(item.id);
                  }}
                >
                  <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                </TouchableOpacity>
              </View>
            </TouchableOpacity>
          </Modal>
        )}
      </View>
    );
  };

  // 3. Use renderTaskItem only in the Task tab's FlatList:
  {selectedCategory === 'Task' && (
    <FlatList
      data={items.filter(i => i.category === 'Task')}
      keyExtractor={item => item.id}
      renderItem={renderTaskItem}
      ListEmptyComponent={<Text style={styles.emptyText}>No tasks added yet.</Text>}
      onScrollBeginDrag={() => { setTaskMenuVisibleId(null); setTaskStatusMenuVisibleId(null); }}
    />
  )}
  // ... existing code ...
  // 4. In the edit modal, use editItem to pre-fill and update the task on save.
  // ... existing code ...

  // Add state to track which menu is open for each category
  const [partsMenuVisibleId, setPartsMenuVisibleId] = useState(null);
  const [partsDropdownAnchor, setPartsDropdownAnchor] = useState(null);
  const [laborMenuVisibleId, setLaborMenuVisibleId] = useState(null);
  const [laborDropdownAnchor, setLaborDropdownAnchor] = useState(null);
  const [additionalCostMenuVisibleId, setAdditionalCostMenuVisibleId] = useState(null);
  const [additionalCostDropdownAnchor, setAdditionalCostDropdownAnchor] = useState(null);
  const [remarksMenuVisibleId, setRemarksMenuVisibleId] = useState(null);
  const [remarksDropdownAnchor, setRemarksDropdownAnchor] = useState(null);

  // Add editItemId state for each category
  const [editPartsId, setEditPartsId] = useState(null);
  const [editLaborId, setEditLaborId] = useState(null);
  const [editAdditionalCostId, setEditAdditionalCostId] = useState(null);
  const [editRemarksId, setEditRemarksId] = useState(null);

  const handleUpdateParts = () => {
    setItems(items.map(item =>
      item.id === editPartsId
        ? {
            ...item,
            itemNumber: partsItemNumber,
            partNumber: partsPartNumber,
            itemName: partsItemName,
            quantity: partsQuantity,
            unit: partsUnit,
            itemCost: partsItemCost,
            site: partsSite,
            note: partsNote,
          }
        : item
    ));
    resetPartsFields();
  };

  const [pdfModalVisible, setPdfModalVisible] = useState(false);

  return (
    <>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>New Job Card</Text>
          </View>
          <View style={styles.headerCenter}>
            <View style={styles.jobCardCountContainer}>
              <MaterialIcons name="work" size={16} color="#fff" />
              <Text style={styles.jobCardCountText}>
                {jobCardCountLoading ? '...' : `Next: ${String(jobCardCount + 1).padStart(4, '0')}`}
              </Text>
            </View>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.saveButtoncreate} onPress={saveWorkOrder} disabled={isLoading}>
              <Text style={styles.saveButtonText0}>Create</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
        keyboardVerticalOffset={Platform.OS === 'ios' ? TOTAL_HEADER_HEIGHT + 20 : 0}
        enabled={Platform.OS === 'ios'}
      >
        <ScrollView
          ref={scrollViewRef}
          keyboardShouldPersistTaps="always"
          showsVerticalScrollIndicator={false}
          style={styles.mainContent}
          contentContainerStyle={Platform.OS === 'ios' ? { paddingBottom: 100 } : undefined}
          bounces={Platform.OS === 'ios'}
          alwaysBounceVertical={Platform.OS === 'ios'}
        >
          <View style={styles.formGroup}>
            <View style={styles.requestNumberRow}>
              <Text style={styles.requestNumberLabel}>Job Card #</Text>
              <Text style={styles.requestNumberValue}>
                {requestNumber || (country && project ? 'Generating...' : 'Select Country & Project')}
              </Text>
            </View>
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>
              Job Card Title
              <Text style={styles.requiredError}> *</Text>
            </Text>
            <TextInput
              style={[styles.inputmain, errors.title && styles.inputError]}
              placeholder="Job Card Title"
              placeholderTextColor="#888"
              value={workOrderTitle}
              onChangeText={text => {
                setWorkOrderTitle(text);
                setErrors({ ...errors, title: text.trim() ? '' : 'Job Card Title is required' });
              }}
            />
            {errors.title && <Text style={styles.errorText}>{errors.title}</Text>}
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>
              Country
              <Text style={styles.requiredError}> *</Text>
            </Text>
            {/* Defensive: Show error if Dropdown is not available */}
            {ElementDropdown.Dropdown ? (
              <ElementDropdown.Dropdown
                style={[styles.inputmain, errors.country && styles.inputError]}
                containerStyle={[styles.dropdownContainer, { backgroundColor: '#fff', borderWidth: 1, borderColor: '#015185', borderRadius: 8 }]}
                dropdownStyle={styles.dropdownList}
                data={countryItems}
                labelField="label"
                valueField="value"
                placeholder="Select Country"
                value={country}
                onChange={item => {
                  setCountry(item.value);
                  setErrors({ ...errors, country: item.value ? '' : 'Country is required' });
                }}
                placeholderStyle={styles.placeholderStyle}
                selectedTextStyle={styles.dropdownText}
              />
            ) : (
              <Text style={{ color: 'red', fontWeight: 'bold' }}>Dropdown component is not available. Please check your react-native-element-dropdown installation.</Text>
            )}
            {errors.country && <Text style={styles.errorText}>Country is required</Text>}
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>
              Project
              <Text style={styles.requiredError}> *</Text>
            </Text>
            {ElementDropdown.Dropdown ? (
              <ElementDropdown.Dropdown
                style={[styles.inputmain, (!country && styles.disabledDropdown), errors.project && styles.inputError]}
                containerStyle={[styles.dropdownContainer, { backgroundColor: '#fff', borderWidth: 1, borderColor: '#015185', borderRadius: 8 }]}
                dropdownStyle={styles.dropdownList}
                data={projectItems}
                labelField="label"
                valueField="value"
                placeholder="Select Project"
                value={project}
                onChange={item => {
                  setProject(item.value);
                  setErrors({ ...errors, project: item.value ? '' : 'Project is required' });
                }}
                placeholderStyle={styles.placeholderStyle}
                selectedTextStyle={styles.dropdownText}
                disable={!country}
              />
            ) : (
              <Text style={{ color: 'red', fontWeight: 'bold' }}>Dropdown component is not available. Please check your react-native-element-dropdown installation.</Text>
            )}
            {errors.project && <Text style={styles.errorText}>Project is required</Text>}
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>Write Memo</Text>
            <TextInput
              style={[styles.inputmain]}
              placeholder="Check within a day"
              placeholderTextColor="#888"
              value={memo}
              onChangeText={setMemo}
              multiline
              numberOfLines={4}
            />
          </View>

          <View style={styles.formGroup}>
            <Text style={styles.label}>Priority</Text>
            <View style={styles.priorityContainer}>
              {priorityOptions.map(option => {
                const isSelected = priority === option.label;
                return (
                  <View key={option.label} style={styles.priorityItem}>
                    <TouchableOpacity
                      style={[
                        styles.priorityButton,
                        isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                      ]}
                      onPress={() => handlePrioritySelect(option.label)}
                    >
                      <Image source={option.image} style={styles.priorityImage} />
                      <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                        {option.label}
                      </Text>
                    </TouchableOpacity>
                  </View>
                );
              })}
            </View>
          </View>

          <View style={[styles.formGroup, styles.dropdownContainer]}>
            <View style={styles.rowCentered}>
              <View style={styles.equipmentContainer}>
                <Text style={styles.dropdownLabel}>
                  Equipments
                  <Text style={styles.requiredError}> *</Text>
                </Text>
                {selectedEquipment && (
                  <Text style={styles.selectedEquipmentText}>
                    {selectedEquipment.equipmentName || selectedEquipment.name}
                  </Text>
                )}
                {errors.equipment && <Text style={styles.errorText}>{errors.equipment}</Text>}
              </View>
              {/* Equipment selection UI (find the button or TouchableOpacity for equipment selection) */}
              <TouchableOpacity
                style={styles.equipmentSelectButton}
                onPress={() => {
                  navigation.navigate('Select_EquipmentWo', {
                    selectedEquipment,
                    onSelect: (equipment) => {
                      setSelectedEquipment(equipment);
                      setErrors && setErrors(prev => ({ ...prev, equipment: '' }));
                    },
                  });
                }}
              >
                <Text style={styles.equipmentSelectButtonText}>
                  {selectedEquipment ? selectedEquipment.equipmentName : 'Select Equipment'}
                </Text>
              </TouchableOpacity>
            </View>
          </View>
          {renderEquipmentModal()}

          <View style={[styles.formGroup, styles.dropdownContainer]}>
            <View style={styles.assignToContainer}>
              <View style={styles.assignToHeader}>
                <Text style={styles.dropdownLabel}>Assign To</Text>
                <TouchableOpacity
                  style={styles.selectButton}
                  onPress={openAssignToModal}
                >
                  <Text style={styles.selectButtonText}>
                    {assignTo.length > 0 ? `${assignTo.length} Selected` : 'Select'}
                  </Text>
                  <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                </TouchableOpacity>
              </View>
              {assignTo.length > 0 ? (
                <View style={styles.selectedItemsContainer}>
                  <ScrollView
                    horizontal={true}
                    showsHorizontalScrollIndicator={false}
                    contentContainerStyle={styles.selectedItemsRow}
                  >
                    {assignTo.map((item, index) => (
                      <View key={`${item.id || index}`} style={styles.selectedItem}>
                        <View style={styles.selectedItemContent}>
                          <Text style={styles.selectedItemText}>
                            {item.fullName || item.name || 'Unknown User'}
                          </Text>
                          <Text style={styles.selectedItemSubText}>
                            {item.employeeNumber || 'N/A'} • {item.role || 'Operator'}
                          </Text>
                        </View>
                        <TouchableOpacity
                          style={styles.closeButton}
                          onPress={() => handleRemoveAssignTo(item)}
                        >
                          <View style={styles.closeCircle}>
                            <MaterialIcons name="close" size={width * 0.04} color="#fff" />
                          </View>
                        </TouchableOpacity>
                      </View>
                    ))}
                  </ScrollView>
                </View>
              ) : (
                <View style={styles.noUsersSelected}>
                  <Text style={styles.noUsersSelectedText}>No users selected</Text>
                  <Text style={styles.noUsersSelectedSubText}>Tap "Select" to choose users to assign</Text>
                </View>
              )}
            </View>
          </View>

          <View style={styles.formsubdat}>
            <View style={styles.formGroupdates0}>
              <View style={styles.row}>
                <Text style={styles.label}>Start Date</Text>
                <TouchableOpacity
                  style={styles.dateInputButton}
                  onPress={() => {
                    setDatePickerMode('start');
                    setDatePickerVisible(true);
                    if (Platform.OS === 'ios') {
                      forceModalUpdate();
                    }
                  }}
                >
                  <Text style={styles.dateInputText}>{formatDate(startDate)}</Text>
                  <MaterialIcons name="calendar-today" size={width * 0.045} color="#015185" />
                </TouchableOpacity>
              </View>
            </View>

            <View style={styles.mainspace0}></View>

            <View style={styles.formGroupdates1}>
              <View style={styles.row}>
                <Text style={styles.label}>Due Date</Text>
                <TouchableOpacity
                  style={styles.dateInputButton}
                  onPress={() => {
                    setDatePickerMode('due');
                    setDatePickerVisible(true);
                    if (Platform.OS === 'ios') {
                      forceModalUpdate();
                    }
                  }}
                >
                  <Text style={styles.dateInputText}>{formatDate(dueDate)}</Text>
                  <MaterialIcons name="calendar-today" size={width * 0.045} color="#015185" />
                </TouchableOpacity>
              </View>
            </View>
          </View>

          <View style={[styles.formGroup, styles.dropdownContaineradditional]}>
            <View style={styles.rowCentered}>
              <Text style={styles.label}>Additional Details</Text>
              <TouchableOpacity
                style={styles.selectButton}
                onPress={() => setIsAdditionalDetailsOpen(!isAdditionalDetailsOpen)}
              >
                <MaterialIcons
                  name={isAdditionalDetailsOpen ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
                  size={width * 0.05}
                  color="#015185"
                />
              </TouchableOpacity>
            </View>
            {isAdditionalDetailsOpen && (
              <View style={styles.additionalDetailsContainer}>
                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Job Card Type</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={() => setWorkOrderTypeModalVisible(true)}
                    >
                      <Text style={styles.selectButtonText}>{workOrderType}</Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  <View style={styles.wotclono}></View>
                </View>

                <View style={styles.formGroup}>
                  <TextInput
                    style={styles.input}
                    placeholder="Enter estimated cost"
                    placeholderTextColor="#888"
                    value={estimatedCost}
                    onChangeText={setEstimatedCost}
                    keyboardType="numeric"
                  />
                </View>

                <View style={styles.formGroup}>
                  <TextInput
                    style={styles.input}
                    placeholder="Enter labor time in hours"
                    placeholderTextColor="#888"
                    value={estimatedLaborTime}
                    onChangeText={setEstimatedLaborTime}
                    keyboardType="numeric"
                  />
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Link Inspection</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={openInspectionModal}
                    >
                      <Text style={styles.selectButtonText}>
                        {selectedInspection ? selectedInspection.step1?.requestNumber || selectedInspection.requestNumber || 'Selected' : 'Select'}
                      </Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  {linkInspection && (
                    <View style={styles.linkedInspectionDetails}>
                      <Text style={styles.linkedInspectionText}>
                        Equipment: {linkInspection.equipment?.equipmentName || 'N/A'} | 
                        Date: {linkInspection.date ? new Date(linkInspection.date).toLocaleDateString() : 'N/A'}
                      </Text>
                    </View>
                  )}
                  <View style={styles.wotclono}></View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Job Card</Text>
                    <TouchableOpacity
                      style={styles.selectButton}
                      onPress={openJobCardModal}
                    >
                      <Text style={styles.selectButtonText}>
                        {selectedJobCard ? selectedJobCard.title : 'Select'}
                      </Text>
                      <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
                    </TouchableOpacity>
                  </View>
                  {selectedJobCard && (
                    <View style={styles.linkedJobCardDetails}>
                      <Text style={styles.linkedJobCardText}>
                        Title: {selectedJobCard.title} | Status: {selectedJobCard.status}
                      </Text>
                      <TouchableOpacity
                        style={styles.removeLinkedButton}
                        onPress={removeSelectedJobCard}
                      >
                        <MaterialIcons name="close" size={16} color="#d32f2f" />
                      </TouchableOpacity>
                    </View>
                  )}
                  <View style={styles.wotclono}></View>
                </View>
              </View>
            )}
          </View>

          <View style={[styles.formGroup, styles.dropdownContainerCostd]}>
            <View style={styles.rowCentered}>
              <Text style={styles.label}>Cost Details</Text>
              <TouchableOpacity
                style={styles.selectButton}
                onPress={() => setIsCostDetailsOpen(!isCostDetailsOpen)}
              >
                <MaterialIcons
                  name={isCostDetailsOpen ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
                  size={width * 0.05}
                  color="#015185"
                />
              </TouchableOpacity>
            </View>
            {isCostDetailsOpen && (
              <View style={styles.additionalDetailsContainer}>
                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Part & Material Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={partMaterialCost}
                      onChangeText={setPartMaterialCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Total Labor Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={totalLaborCost}
                      onChangeText={setTotalLaborCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Additional Cost</Text>
                    <TextInput
                      style={[styles.inputcost0, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={additionalCost}
                      onChangeText={setAdditionalCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Tax Amount</Text>
                    <View style={styles.taxContainer}>
                      <TextInput
                        style={[styles.inputcost0tax, styles.taxInput]}
                        placeholder="0.00"
                        placeholderTextColor="#888"
                        value={taxAmount}
                        onChangeText={setTaxAmount}
                        keyboardType="numeric"
                      />
                      <Text style={styles.taxLabel}>Tax(%)</Text>
                      <TextInput
                        style={[styles.inputcost0tax, styles.taxInput]}
                        placeholder="0.00"
                        placeholderTextColor="#888"
                        value={taxPercentage}
                        onChangeText={setTaxPercentage}
                        keyboardType="numeric"
                      />
                    </View>
                  </View>
                </View>

                <View style={styles.formGroup}>
                  <View style={styles.row}>
                    <Text style={styles.labelwot}>Total WO Cost</Text>
                    <TextInput
                      style={[styles.input, styles.costInput]}
                      placeholder="0.00"
                      placeholderTextColor="#888"
                      value={totalWoCost}
                      onChangeText={setTotalWoCost}
                      keyboardType="numeric"
                    />
                  </View>
                </View>
              </View>
            )}
          </View>

          <View style={{ marginTop: 16 }}>
            <Text style={styles.sectionHeader}>Tasks & Items</Text>
            <View style={styles.tabRow}>
              {categories.map((cat) => (
                <TouchableOpacity
                  key={cat}
                  style={[styles.tabButton, selectedCategory === cat && styles.tabButtonActive]}
                  onPress={() => setSelectedCategory(cat)}
                >
                  <Text style={[styles.tabButtonText, selectedCategory === cat && styles.tabButtonTextActive]}>{cat}</Text>
                </TouchableOpacity>
              ))}
            </View>
            {/* Render only items for the selected tab/category */}
            <View style={{ marginTop: 10 }}>
              {selectedCategory === 'Task' && (
                <FlatList
                  data={items.filter(item => item.category === 'Task')}
                  keyExtractor={item => item.id}
                  renderItem={renderTaskItem}
                  ListEmptyComponent={<Text style={styles.emptyText}>No tasks added.</Text>}
                />
              )}
              {selectedCategory === 'Parts' && (
                <FlatList
                  data={items.filter(item => item.category === 'Parts')}
                  keyExtractor={item => item.id}
                  renderItem={({ item }) => (
                    <View style={[styles.taskCard, { flexDirection: 'row', alignItems: 'center', position: 'relative' }]}> 
                      <View style={{ flex: 1 }}>
                        <Text style={styles.taskTitle}>{item.itemName}</Text>
                        <Text style={styles.taskNote}>Qty: {item.quantity} {item.unit}</Text>
                        <Text style={styles.taskNote}>Cost: {item.itemCost}</Text>
                        <Text style={styles.taskStatus}>{item.status}</Text>
                      </View>
                      <TouchableOpacity
                        style={styles.threeDotsButton}
                        onPress={event => {
                          setPartsMenuVisibleId(partsMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setPartsDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <MaterialIcons name="more-vert" size={24} color="#015185" />
                      </TouchableOpacity>
                      {partsMenuVisibleId === item.id && partsDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setPartsMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setPartsMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: partsDropdownAnchor.x - 120,
                                top: partsDropdownAnchor.y - 90,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setPartsMenuVisibleId(null);
                                  setEditPartsId(item.id);
                                  setPartsItemNumber(item.itemNumber || '');
                                  setPartsPartNumber(item.partNumber || '');
                                  setPartsItemName(item.itemName || '');
                                  setPartsQuantity(item.quantity || '');
                                  setPartsUnit(item.unit || 'Piece');
                                  setPartsItemCost(item.itemCost || '');
                                  setPartsSite(item.site || '');
                                  setPartsNote(item.note || '');
                                  setAddPartsModalVisible(true);
                                }}
                              >
                                <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                              </TouchableOpacity>
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setPartsMenuVisibleId(null);
                                  setItems(items.filter(i => i.id !== item.id));
                                }}
                              >
                                <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                              </TouchableOpacity>
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                    </View>
                  )}
                  ListEmptyComponent={<Text style={styles.emptyText}>No parts added.</Text>}
                />
              )}
              {selectedCategory === 'Labor' && (
                <FlatList
                  data={items.filter(item => item.category === 'Labor')}
                  keyExtractor={item => item.id}
                  renderItem={({ item }) => (
                    <View style={[styles.taskCard, { flexDirection: 'row', alignItems: 'center', position: 'relative' }]}> 
                      <View style={{ flex: 1 }}>
                        <Text style={styles.taskTitle}>{item.worker}</Text>
                        <Text style={styles.taskNote}>Hours: {item.hours}h {item.minutes}m</Text>
                        <Text style={styles.taskNote}>Rate: {item.rate}</Text>
                        <Text style={styles.taskStatus}>{item.status}</Text>
                      </View>
                      <TouchableOpacity
                        style={styles.threeDotsButton}
                        onPress={event => {
                          setLaborMenuVisibleId(laborMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setLaborDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <MaterialIcons name="more-vert" size={24} color="#015185" />
                      </TouchableOpacity>
                      {laborMenuVisibleId === item.id && laborDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setLaborMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setLaborMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: laborDropdownAnchor.x - 120,
                                top: laborDropdownAnchor.y - 90,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setLaborMenuVisibleId(null);
                                  setEditLaborId(item.id);
                                  setLaborWorker(item.worker || '');
                                  setLaborCode(item.code || '');
                                  setLaborRate(item.rate || '');
                                  setLaborHours(item.hours || '');
                                  setLaborMinutes(item.minutes || '');
                                  setLaborNote(item.note || '');
                                  setAddLaborModalVisible(true);
                                }}
                              >
                                <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                              </TouchableOpacity>
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setLaborMenuVisibleId(null);
                                  setItems(items.filter(i => i.id !== item.id));
                                }}
                              >
                                <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                              </TouchableOpacity>
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                    </View>
                  )}
                  ListEmptyComponent={<Text style={styles.emptyText}>No labor added.</Text>}
                />
              )}
              {selectedCategory === 'Additional Cost' && (
                <FlatList
                  data={items.filter(item => item.category === 'Additional Cost')}
                  keyExtractor={item => item.id}
                  renderItem={({ item }) => (
                    <View style={[styles.taskCard, { flexDirection: 'row', alignItems: 'center', position: 'relative' }]}> 
                      <View style={{ flex: 1 }}>
                        <Text style={styles.taskTitle}>{item.description}</Text>
                        <Text style={styles.taskNote}>Amount: {item.amount}</Text>
                        <Text style={styles.taskStatus}>{item.status}</Text>
                      </View>
                      <TouchableOpacity
                        style={styles.threeDotsButton}
                        onPress={event => {
                          setAdditionalCostMenuVisibleId(additionalCostMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setAdditionalCostDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <MaterialIcons name="more-vert" size={24} color="#015185" />
                      </TouchableOpacity>
                      {additionalCostMenuVisibleId === item.id && additionalCostDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setAdditionalCostMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setAdditionalCostMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: additionalCostDropdownAnchor.x - 120,
                                top: additionalCostDropdownAnchor.y - 90,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setAdditionalCostMenuVisibleId(null);
                                  setEditAdditionalCostId(item.id);
                                  setAdditionalCostDescription(item.description || '');
                                  setAdditionalCostAmount(item.amount || '');
                                  setAddAdditionalCostModalVisible(true);
                                }}
                              >
                                <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                              </TouchableOpacity>
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setAdditionalCostMenuVisibleId(null);
                                  setItems(items.filter(i => i.id !== item.id));
                                }}
                              >
                                <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                              </TouchableOpacity>
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                    </View>
                  )}
                  ListEmptyComponent={<Text style={styles.emptyText}>No additional cost added.</Text>}
                />
              )}
              {selectedCategory === 'Remarks' && (
                <FlatList
                  data={items.filter(item => item.category === 'Remarks')}
                  keyExtractor={item => item.id}
                  renderItem={({ item }) => (
                    <View style={[styles.taskCard, { flexDirection: 'row', alignItems: 'center', position: 'relative' }]}> 
                      <View style={{ flex: 1 }}>
                        <Text style={styles.taskTitle}>{item.description}</Text>
                        <Text style={styles.taskStatus}>{item.status}</Text>
                      </View>
                      <TouchableOpacity
                        style={styles.threeDotsButton}
                        onPress={event => {
                          setRemarksMenuVisibleId(remarksMenuVisibleId === item.id ? null : item.id);
                          if (event && event.nativeEvent) {
                            setRemarksDropdownAnchor({ x: event.nativeEvent.pageX, y: event.nativeEvent.pageY });
                          }
                        }}
                      >
                        <MaterialIcons name="more-vert" size={24} color="#015185" />
                      </TouchableOpacity>
                      {remarksMenuVisibleId === item.id && remarksDropdownAnchor && (
                        <Modal
                          visible={true}
                          transparent
                          animationType="fade"
                          onRequestClose={() => setRemarksMenuVisibleId(null)}
                        >
                          <TouchableOpacity
                            style={{ flex: 1 }}
                            activeOpacity={1}
                            onPress={() => setRemarksMenuVisibleId(null)}
                          >
                            <View
                              style={{
                                position: 'absolute',
                                left: remarksDropdownAnchor.x - 120,
                                top: remarksDropdownAnchor.y - 90,
                                backgroundColor: '#fff',
                                borderRadius: 8,
                                elevation: 5,
                                zIndex: 200,
                                shadowColor: '#000',
                                shadowOpacity: 0.1,
                                shadowRadius: 4,
                                minWidth: 120,
                              }}
                            >
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setRemarksMenuVisibleId(null);
                                  setEditRemarksId(item.id);
                                  setRemarksDescription(item.description || '');
                                  setAddRemarksModalVisible(true);
                                }}
                              >
                                <Text style={{ color: '#015185', fontWeight: 'bold' }}>Edit</Text>
                              </TouchableOpacity>
                              <TouchableOpacity
                                style={{ padding: 12 }}
                                onPress={() => {
                                  setRemarksMenuVisibleId(null);
                                  setItems(items.filter(i => i.id !== item.id));
                                }}
                              >
                                <Text style={{ color: '#dc3545', fontWeight: 'bold' }}>Delete</Text>
                              </TouchableOpacity>
                            </View>
                          </TouchableOpacity>
                        </Modal>
                      )}
                    </View>
                  )}
                  ListEmptyComponent={<Text style={styles.emptyText}>No remarks added.</Text>}
                />
              )}
            </View>
            {/* Add button for the selected tab/category if needed */}
          </View>

          <TouchableOpacity
            style={styles.addTaskButton}
            onPress={() => {
              console.log('Add button pressed for category:', selectedCategory);
              console.log('Current modal states:', {
                addItemModalVisible,
                addLaborModalVisible,
                addPartsModalVisible,
                addAdditionalCostModalVisible,
                addRemarksModalVisible
              });
              
              // Force modal update for iOS
              if (Platform.OS === 'ios') {
                forceModalUpdate();
              }
              
              // Small delay for iOS to prevent freezing
              setTimeout(() => {
                console.log('Opening modal after delay');
                if (selectedCategory === 'Task') {
                  console.log('Setting addItemModalVisible to true');
                  setAddItemModalVisible(true);
                  if (Platform.OS === 'ios') {
                    setIosModalVisible(true);
                  }
                }
                else if (selectedCategory === 'Labor') setAddLaborModalVisible(true);
                else if (selectedCategory === 'Parts') setAddPartsModalVisible(true);
                else if (selectedCategory === 'Additional Cost') setAddAdditionalCostModalVisible(true);
                else if (selectedCategory === 'Remarks') setAddRemarksModalVisible(true);
              }, Platform.OS === 'ios' ? 100 : 0);
            }}
          >
            <Text style={styles.addTaskText}>+ Add {selectedCategory}</Text>
          </TouchableOpacity>
          
  
        </ScrollView>
      </KeyboardAvoidingView>

      {/* iOS Date Picker */}
      {datePickerVisible && Platform.OS === 'ios' && (
        <Modal
          key={modalKey}
          visible={datePickerVisible}
          transparent={true}
          animationType="slide"
          onRequestClose={() => setDatePickerVisible(false)}
        >
          <View style={styles.datePickerModalOverlay}>
            <View style={styles.datePickerModalContent}>
              <View style={styles.datePickerHeader}>
                <TouchableOpacity onPress={() => setDatePickerVisible(false)}>
                  <Text style={styles.datePickerCancelText}>Cancel</Text>
                </TouchableOpacity>
                <Text style={styles.datePickerTitle}>
                  {datePickerMode === 'start' ? 'Select Start Date' : 'Select Due Date'}
                </Text>
                <TouchableOpacity onPress={() => setDatePickerVisible(false)}>
                  <Text style={styles.datePickerDoneText}>Done</Text>
                </TouchableOpacity>
              </View>
              <View style={styles.datePickerContainer}>
                <DateTimePicker
                  value={datePickerMode === 'start' ? (startDate || new Date()) : (dueDate || new Date())}
                  mode="date"
                  display="spinner"
                  onChange={(event, selectedDate) => {
                    if (event.type === 'set' && selectedDate) {
                      if (datePickerMode === 'start') setStartDate(selectedDate);
                      else setDueDate(selectedDate);
                    }
                  }}
                  style={{ width: 300, height: 200 }}
                  textColor="#015185"
                />
              </View>
            </View>
          </View>
        </Modal>
      )}

      {/* Android Date Picker */}
      {datePickerVisible && Platform.OS === 'android' && (
        <DateTimePicker
          value={datePickerMode === 'start' ? (startDate || new Date()) : (dueDate || new Date())}
          mode="date"
          display="calendar"
          onChange={(event, selectedDate) => {
            setDatePickerVisible(false);
            if (event.type === 'set' && selectedDate) {
              if (datePickerMode === 'start') setStartDate(selectedDate);
              else setDueDate(selectedDate);
            }
          }}
        />
      )}



      <Modal
        visible={workOrderTypeModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setWorkOrderTypeModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Job Card Type</Text>
            <FlatList
              data={workOrderTypes}
              keyExtractor={item => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => handleWorkOrderTypeSelect(item)}
                >
                  <Text style={styles.modalOptionText}>{item.name}</Text>
                </TouchableOpacity>
              )}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setWorkOrderTypeModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={inspectionModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => {
          console.log('🔍 Modal close requested');
          setInspectionModalVisible(false);
        }}
      >
        <View style={styles.modalOverlay}>
          {console.log('🔍 Rendering inspection modal overlay')}
          <View style={[styles.modalContent, styles.inspectionModal]}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>Select Inspection Report</Text>
              <TouchableOpacity
                onPress={() => setInspectionModalVisible(false)}
                style={styles.closeButton}
              >
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            
            {inspectionLoading ? (
              <View style={styles.loadingContainer}>
                <ActivityIndicator size="large" color="#015185" />
                <Text style={styles.loadingText}>Loading inspection reports...</Text>
              </View>
            ) : (
              <FlatList
                data={availableInspections}
                keyExtractor={item => item._id || item.id}
                renderItem={({ item }) => {
                  const isSelected = selectedInspection && selectedInspection._id === item._id;
                  return (
                    <TouchableOpacity
                      style={[
                        styles.inspectionItem,
                        isSelected && styles.selectedInspectionItem
                      ]}
                      onPress={() => handleInspectionSelect(item)}
                    >
                      <View style={styles.inspectionItemHeader}>
                        <Text style={styles.inspectionRequestNumber}>
                          {item.step1?.requestNumber || item.requestNumber || 'N/A'}
                        </Text>
                        {isSelected && (
                          <MaterialIcons name="check-circle" size={20} color="#28a745" />
                        )}
                      </View>
                      <View style={styles.inspectionItemDetails}>
                        <Text style={styles.inspectionEquipment}>
                          Equipment: {item.step1?.selectedEquipment?.equipmentName || item.equipment?.equipmentName || 'N/A'}
                        </Text>
                        <Text style={styles.inspectionDate}>
                          Date: {item.step1?.date || item.date
                            ? new Date(item.step1?.date || item.date).toLocaleDateString()
                            : 'N/A'}
                        </Text>
                        <Text style={styles.inspectionFaulty}>
                          Faulty Items: {item.step3?.faultyItems || item.faultyItems || 0}
                        </Text>
                      </View>
                    </TouchableOpacity>
                  );
                }}
                ListEmptyComponent={
                  <View style={styles.emptyContainer}>
                    <MaterialIcons name="inbox" size={48} color="#ccc" />
                    <Text style={styles.emptyText}>No inspection reports found</Text>
                    <Text style={styles.emptySubtext}>Create an inspection report first</Text>
                  </View>
                }
                style={{ maxHeight: 400 }}
              />
            )}
            
            {selectedInspection && (
              <View style={styles.selectedInspectionContainer}>
                <Text style={styles.selectedInspectionLabel}>Selected:</Text>
                <Text style={styles.selectedInspectionText}>
                  {selectedInspection.step1?.requestNumber || selectedInspection.requestNumber || 'N/A'}
                </Text>
                <TouchableOpacity
                  style={styles.removeInspectionButton}
                  onPress={removeSelectedInspection}
                >
                  <MaterialIcons name="close" size={16} color="#fff" />
                  <Text style={styles.removeInspectionText}>Remove</Text>
                </TouchableOpacity>
              </View>
            )}
            
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setInspectionModalVisible(false)}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              {selectedInspection && (
                <TouchableOpacity
                  style={[styles.modalButton, styles.saveButton]}
                  onPress={() => {
                    handleInspectionSelect(selectedInspection);
                    setInspectionModalVisible(false);
                  }}
                >
                  <Text style={styles.modalButtonText}>Select</Text>
                </TouchableOpacity>
              )}
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={jobCardModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setJobCardModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, styles.jobCardModal]}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>Select Job Card</Text>
              <TouchableOpacity
                onPress={() => setJobCardModalVisible(false)}
                style={styles.closeButton}
              >
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            
            {jobCardLoading ? (
              <View style={styles.loadingContainer}>
                <ActivityIndicator size="large" color="#015185" />
                <Text style={styles.loadingText}>Loading job cards...</Text>
              </View>
            ) : (
              <FlatList
                data={availableJobCards}
                keyExtractor={item => item._id || item.id}
                renderItem={({ item }) => {
                  const isSelected = selectedJobCard && selectedJobCard._id === item._id;
                  return (
                    <TouchableOpacity
                      style={[
                        styles.jobCardItem,
                        isSelected && styles.selectedJobCardItem
                      ]}
                      onPress={() => handleJobCardSelect(item)}
                    >
                      <View style={styles.jobCardItemHeader}>
                        <Text style={styles.jobCardTitle}>
                          {item.title || 'Untitled Job Card'}
                        </Text>
                        {isSelected && (
                          <MaterialIcons name="check-circle" size={20} color="#28a745" />
                        )}
                      </View>
                      <View style={styles.jobCardItemDetails}>
                        <Text style={styles.jobCardRequestNumber}>
                          Request #: {item.requestNumber || 'N/A'}
                        </Text>
                        <Text style={styles.jobCardStatus}>
                          Status: {item.status || 'N/A'}
                        </Text>
                        <Text style={styles.jobCardDate}>
                          Created: {item.createdAt
                            ? new Date(item.createdAt).toLocaleDateString()
                            : 'N/A'}
                        </Text>
                      </View>
                    </TouchableOpacity>
                  );
                }}
                ListEmptyComponent={
                  <View style={styles.emptyContainer}>
                    <MaterialIcons name="work" size={48} color="#ccc" />
                    <Text style={styles.emptyText}>No job cards found</Text>
                    <Text style={styles.emptySubtext}>Create a job card first</Text>
                  </View>
                }
                style={{ maxHeight: 400 }}
              />
            )}
            
            {selectedJobCard && (
              <View style={styles.selectedJobCardContainer}>
                <Text style={styles.selectedJobCardLabel}>Selected:</Text>
                <Text style={styles.selectedJobCardText}>
                  {selectedJobCard.title || 'Untitled Job Card'}
                </Text>
                <TouchableOpacity
                  style={styles.removeJobCardButton}
                  onPress={removeSelectedJobCard}
                >
                  <MaterialIcons name="close" size={16} color="#fff" />
                  <Text style={styles.removeJobCardText}>Remove</Text>
                </TouchableOpacity>
              </View>
            )}
            
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={[styles.modalButton, styles.cancelButton]}
                onPress={() => setJobCardModalVisible(false)}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              {selectedJobCard && (
                <TouchableOpacity
                  style={[styles.modalButton, styles.saveButton]}
                  onPress={() => {
                    handleJobCardSelect(selectedJobCard);
                    setJobCardModalVisible(false);
                  }}
                >
                  <Text style={styles.modalButtonText}>Select</Text>
                </TouchableOpacity>
              )}
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={linkedWorkOrderModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setLinkedWorkOrderModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Link Job Card</Text>
            <FlatList
              data={workOrders.filter(wo =>
                selectedEquipment && wo.equipment && wo.equipment.id === selectedEquipment.id
              )}
              keyExtractor={item => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => handleWorkOrderSelect(item)}
                >
                  <View>
                    <Text style={styles.modalOptionText}>{item.title} (Job Card #: {item.requestNumber || 'N/A'})</Text>
                    <Text style={{ color: '#015185', fontSize: 13 }}>Equipment: {item.equipment?.equipmentName || 'N/A'} (ID: {item.equipment?.id || 'N/A'})</Text>
                  </View>
                </TouchableOpacity>
              )}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setLinkedWorkOrderModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Add Task Modal */}
      {console.log('Rendering Add Task Modal, visible:', addItemModalVisible, 'modalKey:', modalKey, 'iosModalVisible:', iosModalVisible)}
      <Modal
        key={`addItem-${modalKey}-${addItemModalVisible}-${iosModalVisible}`}
        visible={Platform.OS === 'ios' ? iosModalVisible : addItemModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => {
          console.log('Modal onRequestClose called');
          setAddItemModalVisible(false);
          setIosModalVisible(false);
        }}
        onShow={() => {
          console.log('Add Item Modal is now visible');
        }}
        onDismiss={() => {
          console.log('Modal dismissed');
        }}
        statusBarTranslucent={true}
      >
        <View style={[styles.taskModalOverlayCustom, { backgroundColor: 'rgba(0,0,0,0.5)' }]}>
          <View style={[styles.taskModalContentCustom, { backgroundColor: '#fff' }]}>
            <Text style={styles.taskModalTitleCustom}>Add Task</Text>
            <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Task</Text>
              <TextInput
                style={styles.taskModalInputCustom}
                
                placeholderTextColor="#888"
                value={newItemText}
                onChangeText={setNewItemText}
              />
            </View>
            <View style={styles.taskModalDividerCustom} />
            <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Type</Text>
              <View style={styles.taskModalTypeButtonRowCustom}>
                {taskTypes.map(type => (
                  <TouchableOpacity
                    key={type}
                    style={[styles.taskModalTypeButtonCustom, taskType === type && styles.taskModalTypeButtonSelectedCustom]}
                    onPress={() => setTaskType(type)}
                  >
                    <Text style={[styles.taskModalTypeButtonTextCustom, taskType === type && styles.taskModalTypeButtonTextSelectedCustom]}>{type}</Text>
                  </TouchableOpacity>
                ))}
              </View>
            </View>
            <View style={styles.taskModalDividerCustom} />
            <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Assign to</Text>
              <TouchableOpacity
                style={styles.taskModalAssignButtonCustom}
                onPress={openTaskAssignToModal}
              >
                <Text style={styles.taskModalAssignButtonTextCustom}>
                  {tempAssignTo.length > 0 ? `${tempAssignTo.length} Assigned` : 'Set'}
                </Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View>
            
            {/* Selected Users Display */}
            {tempAssignTo.length > 0 && (
              <View style={styles.taskModalSelectedUsersContainer}>
                <Text style={styles.taskModalSelectedUsersTitle}>Selected Users:</Text>
                {tempAssignTo.map((user, index) => (
                  <View key={user.id} style={styles.taskModalSelectedUserItem}>
                    <View style={styles.taskModalSelectedUserInfo}>
                      <Text style={styles.taskModalSelectedUserName}>
                        {user.fullName || user.name || 'Unknown User'}
                      </Text>
                      <Text style={styles.taskModalSelectedUserDetails}>
                        Emp#: {user.employeeNumber || 'N/A'} | Role: {user.role || 'Operator'}
                      </Text>
                    </View>
                    <TouchableOpacity
                      style={styles.taskModalRemoveUserButton}
                      onPress={() => {
                        setTempAssignTo(prev => prev.filter(u => u.id !== user.id));
                      }}
                    >
                      <MaterialIcons name="close" size={16} color="#ff0000" />
                    </TouchableOpacity>
                  </View>
                ))}
              </View>
            )}
            
            <View style={styles.taskModalDividerCustom} />
            <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Note</Text>
              <TextInput
                style={styles.taskModalInputCustom}
                placeholder="Note"
                placeholderTextColor="#888"
                value={note}
                onChangeText={setNote}
              />
            </View>
            <View style={styles.taskModalDividerCustom} />
            {/* <View style={styles.taskModalRowCustom}>
              <Text style={styles.taskModalLabelCustom}>Fault List</Text>
              <TouchableOpacity
                style={styles.taskModalFaultButtonCustom}
                onPress={() => setFaultListModalVisible(true)}
              >
                <Text style={styles.taskModalFaultButtonTextCustom}>
                  {selectedFaults.length > 0 ? `${selectedFaults.length} Selected` : 'Select'}
                </Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View> */}
            <View style={styles.taskModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.taskModalButtonCustom, styles.taskModalCancelButtonCustom]}
                onPress={() => {
                  setAddItemModalVisible(false);
                  setIosModalVisible(false);
                }}
              >
                <Text style={styles.taskModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.taskModalButtonCustom, styles.taskModalSaveButtonCustom]}
                onPress={saveItem}
              >
                <Text style={styles.taskModalButtonTextCustom}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={editModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setEditModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Edit Task</Text>
            {/* Task Row */}
            <View style={[styles.rowCentered, { marginBottom: 12 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Task</Text>
              <TextInput
                style={[styles.input, { flex: 2 }]}
                placeholder="Task description"
                placeholderTextColor="#888"
                value={newItemText}
                onChangeText={setNewItemText}
              />
            </View>
            {/* Type Row */}
            <View style={[styles.rowCentered, { marginBottom: 12 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Type</Text>
              <View style={{ flex: 2, flexDirection: 'row', gap: 8 }}>
                {taskTypes.map(type => (
                  <TouchableOpacity
                    key={type}
                    style={[
                      styles.typeButton,
                      taskType === type && styles.typeButtonSelected,
                      { minWidth: 80, alignItems: 'center' },
                    ]}
                    onPress={() => setTaskType(type)}
                  >
                    <Text
                      style={[
                        styles.typeText,
                        taskType === type && styles.typeTextSelected,
                      ]}
                    >
                      {type}
                    </Text>
                  </TouchableOpacity>
                ))}
              </View>
            </View>
            {/* Assign To Row */}
            <View style={[styles.rowCentered, { marginBottom: 12 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Assign to</Text>
              <TouchableOpacity
                style={[styles.selectButton, { flex: 2, justifyContent: 'space-between' }]}
                onPress={() => setAssignToModalVisible(true)}
              >
                <Text style={styles.selectButtonText}>
                  {tempAssignTo.length > 0 ? `${tempAssignTo.length} Assigned` : 'Set'}
                </Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View>
            {/* Note Row */}
            <View style={[styles.rowCentered, { marginBottom: 12 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Note</Text>
              <TextInput
                style={[styles.input, { flex: 2 }]}
                placeholder="Note"
                placeholderTextColor="#888"
                value={note}
                onChangeText={setNote}
              />
            </View>
            {/* Fault List Row */}
            {/* <View style={[styles.rowCentered, { marginBottom: 18 }]}> 
              <Text style={[styles.label, { flex: 1 }]}>Fault List</Text>
              <TouchableOpacity
                style={[styles.selectButton, { flex: 2, justifyContent: 'space-between' }]}
                onPress={() => setFaultListModalVisible(true)}
              >
                <Text style={styles.selectButtonText}>
                  {selectedFaults.length > 0 ? `${selectedFaults.length} Selected` : 'Select'}
                </Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View> */}
            {/* Buttons */}
            <View style={[styles.rowCentered, { justifyContent: 'flex-end', marginTop: 10 }]}> 
              <TouchableOpacity
                style={[styles.saveButton, { backgroundColor: '#0078D4', marginRight: 10, minWidth: 90, borderRadius: 4, paddingVertical: 8 }]}
                onPress={() => setEditModalVisible(false)}
              >
                <Text style={[styles.saveButtonText, { color: '#fff', textAlign: 'center' }]}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.saveButton, { minWidth: 90, borderRadius: 4, paddingVertical: 8 }]}
                onPress={editItemSave}
              >
                <Text style={[styles.saveButtonText, { color: '#fff', textAlign: 'center' }]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addLaborModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setAddLaborModalVisible(false)}
      >
        <View style={styles.laborModalOverlayCustom}>
          <View style={styles.laborModalContentCustom}>
            <Text style={styles.laborModalTitleCustom}>Add Labor Details</Text>
            <TextInput
              style={styles.laborInputCustom}
              placeholder="Worker Name"
              placeholderTextColor="#888"
              value={laborWorker}
              onChangeText={setLaborWorker}
            />
            <TextInput
              style={styles.laborInputCustom}
              placeholder="Labor Code"
              placeholderTextColor="#888"
              value={laborCode}
              onChangeText={setLaborCode}
            />
            <TextInput
              style={styles.laborInputCustom}
              placeholder="Rate"
              placeholderTextColor="#888"
              value={laborRate}
              onChangeText={setLaborRate}
              keyboardType="numeric"
            />
            <View style={styles.laborHoursRowCustom}>
              <Text style={styles.laborHoursLabelCustom}>Work Hours</Text>
              <TextInput
                style={styles.laborHoursInputCustom}
                placeholder="Hours"
                placeholderTextColor="#888"
                value={laborHours}
                onChangeText={setLaborHours}
                keyboardType="numeric"
              />
              <TextInput
                style={styles.laborHoursInputCustom}
                placeholder="Minut...."
                placeholderTextColor="#888"
                value={laborMinutes}
                onChangeText={setLaborMinutes}
                keyboardType="numeric"
              />
            </View>
            <TextInput
              style={styles.laborNoteInputCustom}
              placeholder="Note"
              placeholderTextColor="#888"
              value={laborNote}
              onChangeText={setLaborNote}
              multiline
              numberOfLines={4}
            />
            <View style={styles.laborModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.laborModalButtonCustom, styles.laborModalCancelButtonCustom]}
                onPress={() => setAddLaborModalVisible(false)}
              >
                <Text style={styles.laborModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.laborModalButtonCustom, styles.laborModalSaveButtonCustom]}
                onPress={saveLabor}
              >
                <Text style={styles.laborModalButtonTextCustom}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addPartsModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setAddPartsModalVisible(false)}
      >
        <View style={styles.partsModalOverlayCustom}>
          <View style={styles.partsModalContentCustom}>
            <Text style={styles.partsModalTitleCustom}>Add Parts</Text>
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Item Number"
              placeholderTextColor="#888"
              value={partsItemNumber}
              onChangeText={setPartsItemNumber}
            />
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Part Number"
              placeholderTextColor="#888"
              value={partsPartNumber}
              onChangeText={setPartsPartNumber}
            />
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Item Name"
              placeholderTextColor="#888"
              value={partsItemName}
              onChangeText={setPartsItemName}
            />
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Quantity"
              placeholderTextColor="#888"
              value={partsQuantity}
              onChangeText={setPartsQuantity}
              keyboardType="numeric"
            />
            <View style={styles.partsUnitContainerCustom}>
              <Text style={styles.partsUnitLabelCustom}>Unit</Text>
            </View>
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Item Cost"
              placeholderTextColor="#888"
              value={partsItemCost}
              onChangeText={setPartsItemCost}
              keyboardType="numeric"
            />
            <TextInput
              style={styles.partsInputCustom}
              placeholder="Site"
              placeholderTextColor="#888"
              value={partsSite}
              onChangeText={setPartsSite}
            />
            <TextInput
              style={styles.partsNoteInputCustom}
              placeholder="Note"
              placeholderTextColor="#888"
              value={partsNote}
              onChangeText={setPartsNote}
              multiline
              numberOfLines={4}
            />
            <View style={styles.partsModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.partsModalButtonCustom, styles.partsModalCancelButtonCustom]}
                onPress={() => setAddPartsModalVisible(false)}
              >
                <Text style={styles.partsModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.partsModalButtonCustom, styles.partsModalSaveButtonCustom]}
                onPress={editPartsId ? handleUpdateParts : saveParts}
              >
                <Text style={styles.partsModalButtonTextCustom}>{editPartsId ? 'Update' : 'Add'}</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addAdditionalCostModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setAddAdditionalCostModalVisible(false)}
      >
        <View style={styles.additionalCostModalOverlayCustom}>
          <View style={styles.additionalCostModalContentCustom}>
            <Text style={styles.additionalCostModalTitleCustom}>Add Additional Cost</Text>
            <TextInput
              style={styles.additionalCostDescriptionInputCustom}
              placeholder="Description"
              placeholderTextColor="#888"
              value={additionalCostDescription}
              onChangeText={setAdditionalCostDescription}
              multiline
              numberOfLines={4}
            />
            <TextInput
              style={styles.additionalCostInputCustom}
              placeholder="Select Labor Code"
              placeholderTextColor="#888"
              value={additionalCostAmount}
              onChangeText={setAdditionalCostAmount}
            />
            <View style={styles.additionalCostModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.additionalCostModalButtonCustom, styles.additionalCostModalCancelButtonCustom]}
                onPress={() => setAddAdditionalCostModalVisible(false)}
              >
                <Text style={styles.additionalCostModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.additionalCostModalButtonCustom, styles.additionalCostModalSaveButtonCustom]}
                onPress={saveAdditionalCost}
              >
                <Text style={styles.additionalCostModalButtonTextCustom}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={addRemarksModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setAddRemarksModalVisible(false)}
      >
        <View style={styles.remarksModalOverlayCustom}>
          <View style={styles.remarksModalContentCustom}>
            <Text style={styles.remarksModalTitleCustom}>Add Remarks</Text>
            <TextInput
              style={styles.remarksDescriptionInputCustom}
              placeholder="Description"
              placeholderTextColor="#888"
              value={remarksDescription}
              onChangeText={setRemarksDescription}
              multiline
              numberOfLines={4}
            />
            <View style={styles.remarksModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.remarksModalButtonCustom, styles.remarksModalCancelButtonCustom]}
                onPress={() => setAddRemarksModalVisible(false)}
              >
                <Text style={styles.remarksModalButtonTextCustom}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.remarksModalButtonCustom, styles.remarksModalSaveButtonCustom]}
                onPress={saveRemarks}
              >
                <Text style={styles.remarksModalButtonTextCustom}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={statusModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setStatusModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Status</Text>
            <FlatList
              data={statuses}
              keyExtractor={item => item.label}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => updateItemStatus(selectedItemId, item.label)}
                >
                  <Text style={[styles.modalOptionText, { color: item.color }]}>{item.label}</Text>
                </TouchableOpacity>
              )}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setStatusModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={faultListModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setFaultListModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Faults</Text>
            <FlatList
              data={chunkedFaultListData.flat()}
              keyExtractor={item => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={styles.modalOption}
                  onPress={() => handleFaultSelect(item)}
                >
                  <Text style={styles.modalOptionText}>{item.name}</Text>
                  {selectedFaults.some(f => f.id === item.id) && (
                    <MaterialIcons name="check" size={width * 0.05} color="#015185" />
                  )}
                </TouchableOpacity>
              )}
            />
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={handleSaveFaults}
            >
              <Text style={styles.modalCloseText}>Save</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={showSuccessModal}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.successModalOverlayCustom}>
          <View style={styles.successModalContentCustom}>
            <Text style={styles.successModalTitleCustom}>Success</Text>
            <MaterialIcons name="check" size={48} color="#015185" style={{ marginVertical: 10 }} />
            <Text style={styles.successModalTextCustom}>Work Order  Generated</Text>
            <View style={styles.successModalButtonRowCustom}>
              <TouchableOpacity
                style={[styles.successModalButtonCustom]}
                onPress={async () => {
                  setShowSuccessModal(false);
                  await fetchInitialData();
                  navigation.reset({
                    index: 0,
                    routes: [
                      { name: 'MainApp' },
                      // { name: 'workOrdertab' },
                    ],
                  });
                }}
              >
                <Text style={styles.successModalButtonTextCustom}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successModalButtonCustom]}
                onPress={async () => {
                  setShowSuccessModal(false);
                  await fetchInitialData();
                  await handleViewPDF(true); // Pass a flag to indicate PDF navigation
                }}
              >
                <Text style={styles.successModalButtonTextCustom}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>






      {/* Faulty Items (Auto Tasks) preview (commented out as per user request)
      {items.filter(item => item.category === 'Task').length > 0 && (
        <View style={{ marginTop: 20 }}>
          <Text style={{ fontWeight: 'bold', fontSize: 16 }}>Faulty Items (Auto Tasks):</Text>
          {items.filter(item => item.category === 'Task').map((item, idx) => (
            <Text key={idx} style={{ color: 'red', marginVertical: 2 }}>
              {item.text}
            </Text>
          ))}
        </View>
      )} */}

      {/* Start Date Picker Modal for iOS */}
      {Platform.OS === 'ios' && (
        <Modal
          visible={showStartDatePicker}
          transparent
          animationType="slide"
          onRequestClose={() => setShowStartDatePicker(false)}
        >
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.3)' }}>
            <View style={{ backgroundColor: '#fff', borderRadius: 16, padding: 20, width: '85%', alignItems: 'center' }}>
              <DateTimePicker
                value={startDate || new Date()}
                mode="date"
                display="spinner"
                onChange={(event, date) => {
                  if (date) setStartDate(date);
                }}
                style={{ width: '100%' }}
              />
              <View style={{ flexDirection: 'row', justifyContent: 'space-between', width: '100%', marginTop: 16 }}>
                <TouchableOpacity onPress={() => setShowStartDatePicker(false)} style={{ flex: 1, alignItems: 'center' }}>
                  <Text style={{ color: '#dc3545', fontSize: 16 }}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={() => setShowStartDatePicker(false)}
                  style={{ flex: 1, alignItems: 'center' }}
                >
                  <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: 16 }}>Confirm</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
      )}

      {/* Due Date Picker Modal for iOS */}
      {Platform.OS === 'ios' && (
        <Modal
          visible={showDueDatePicker}
          transparent
          animationType="slide"
          onRequestClose={() => setShowDueDatePicker(false)}
        >
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.3)' }}>
            <View style={{ backgroundColor: '#fff', borderRadius: 16, padding: 20, width: '85%', alignItems: 'center' }}>
              <DateTimePicker
                value={dueDate || new Date()}
                mode="date"
                display="spinner"
                onChange={(event, date) => {
                  if (date) setDueDate(date);
                }}
                style={{ width: '100%' }}
              />
              <View style={{ flexDirection: 'row', justifyContent: 'space-between', width: '100%', marginTop: 16 }}>
                <TouchableOpacity onPress={() => setShowDueDatePicker(false)} style={{ flex: 1, alignItems: 'center' }}>
                  <Text style={{ color: '#dc3545', fontSize: 16 }}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={() => setShowDueDatePicker(false)}
                  style={{ flex: 1, alignItems: 'center' }}
                >
                  <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: 16 }}>Confirm</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
      )}

      

      {pdfModalVisible && (
        <Modal visible={pdfModalVisible} transparent animationType="fade">
          <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.3)', justifyContent: 'center', alignItems: 'center' }}>
            <View style={{ backgroundColor: '#fff', borderRadius: 12, padding: 32, alignItems: 'center' }}>
              <ActivityIndicator size="large" color="#015185" />
              <Text style={{ marginTop: 16, color: '#015185', fontWeight: 'bold', fontSize: 16 }}>Generating PDF, please wait...</Text>
            </View>
          </View>
        </Modal>
      )}
    </>
  );
};



const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  keyboardAvoidingView: {
    flex: 1,
  },
  // iOS Date Picker Styles
  datePickerModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  datePickerModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    alignItems: 'center',
    width: 340,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  datePickerHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    width: '100%',
    marginBottom: 10,
  },
  datePickerTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
    textAlign: 'center',
  },
  datePickerCancelText: {
    color: '#888',
    fontSize: 16,
    paddingHorizontal: 8,
  },
  datePickerDoneText: {
    color: '#015185',
    fontWeight: 'bold',
    fontSize: 16,
    paddingHorizontal: 8,
  },
  datePickerContainer: {
    marginTop: 10,
    alignItems: 'center',
  },

  // iOS Menu Styles
  menuOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  menuContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 10,
    minWidth: 150,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
    elevation: 5,
  },
  menuItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 16,
    borderRadius: 6,
  },
  menuItemText: {
    marginLeft: 12,
    fontSize: 16,
    color: '#333',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerCenter: {
    flex: 1,
    alignItems: 'center',
    justifyContent: 'center',
  },
  jobCardCountContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 15,
  },
  jobCardCountText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
    marginLeft: 6,
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  saveButtonText: {
    color: '#FFFFFF',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  formGroup: {
    marginBottom: width * 0.02,
  },
  mainspace0: {
    width: '100%',
    height: 0.1,
    backgroundColor: '#707173',
    color: '#707173',
  },
  formsubdat: {
    marginBottom: 14,
  },
  formGroupdates0: {
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 10,
    paddingVertical: 2,
    borderBottomWidth: 1,
    borderBottomColor: '#707173',
  },
  formGroupdates1: {
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 10,
    paddingVertical: 2,
  },
  dropdownContainer: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  dropdownContaineradditional: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  dropdownContainerCostd: {
    backgroundColor: '#F2F2F2',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
  },
  dropdownText: {
    fontSize: 16,
    color: '#333',
  },
  placeholderStyle: {
    fontSize: 16,
    color: '#888',
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
  label: {
     fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  dropdownLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  input: {
    borderBottomWidth: 1,
    borderColor: '#015185',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    fontSize: width * 0.035,
    color: '#333',
    paddingHorizontal: width * 0.01,
  },
  inputmain: {
     borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputSingleLine: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
  },
  hourInput: {
    flex: 1,
    marginRight: width * 0.01,
  },
  minuteInput: {
    flex: 1,
  },
  quantityInput: {
    flex: 2,
    marginRight: width * 0.01,
  },
  unitContainer: {
    flex: 1,
  },
  unitInput: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
  },
  multilineInput: {
    height: width * 0.09,
    textAlignVertical: 'top',
  },
  costInput: {
    width: width * 0.3,
    borderWidth: 1,
    borderColor: '#3481BC',
    borderRadius: 3,
    width: '55%',
  },
  taxContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  taxInput: {
    width: width * 0.15,
    marginRight: width * 0.02,
  },
  taxLabel: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.02,
  },
  additionalDetailsContainer: {
    marginTop: width * 0.025,
    paddingHorizontal: width * 0.03,
  },
  tabContainer: {
    marginVertical: width * 0.02,
  },
  tabRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  tabButton: {
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    marginRight: width * 0.02,
    borderBottomWidth: 2,
    borderBottomColor: 'transparent',
  },
  tabButtonActive: {
    borderBottomColor: '#015185',
  },
  tabText: {
    fontSize: width * 0.035,
    color: '#707173',
    fontWeight: '500',
  },
  tabTextActive: {
    color: '#015185',
    fontWeight: '600',
  },
  itemsContainer: {
    //marginVertical: width * 0.025,
    minHeight: width * 0.1,
  },
  itemsFlatList: {
    flexGrow: 0,
  },
  itemsFlatListContent: {
    paddingBottom: width * 0.025,
  },
  itemRow: {
    marginBottom: width * 0.02,
    padding: width * 0.015,
    borderRadius: width * 0.01,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  itemContent: {
    flexDirection: 'column',
  },
  itemHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  itemText: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '600',
  },
  itemDetails: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginTop: width * 0.005,
  },
  itemType: {
    fontSize: width * 0.035,
    color: '#FF0000', // Red to match screenshot
    marginRight: width * 0.02,
  },
  itemNote: {
    fontSize: width * 0.035,
    color: '#707173',
    marginTop: width * 0.005,
  },
  statusButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.02,
    borderRadius: width * 0.01,
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: '600',
  },
  threeDotsButton: {
    padding: 4,
  },
  emptyItemsText: {
    fontSize: width * 0.035,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.025,
  },
  addTaskButton: {
  paddingVertical: width * 0.015,
  paddingHorizontal: width * 0.03,
  borderRadius: width * 0.015,
  backgroundColor: '#015185',
  alignSelf: 'flex-start',
  marginBottom: width * 0.04,
},
addTaskText: {
  color: '#fff',
  fontSize: width * 0.035,
  fontWeight: '600',
},
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: 15,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  row: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  rowCentered: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  assignToContainer: {
    flexDirection: 'column',
  },
  assignToHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  equipmentContainer: {
    flexDirection: 'column',
  },
  selectedEquipmentText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginTop: width * 0.01,
  },
  selectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.01,
  },
  selectButtonText: {
    color: '#015185',
    fontSize: width * 0.035,
    marginRight: width * 0.01,
  },
  selectedItemsContainer: {
    marginTop: width * 0.01,
  },
  selectedItemsRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectedItem: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e8f5e8',
    borderRadius: width * 0.02,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.02,
    marginBottom: width * 0.01,
    borderWidth: 1,
    borderColor: '#4CAF50',
    minWidth: width * 0.4,
  },
  selectedItemContent: {
    flex: 1,
    marginRight: width * 0.015,
  },
  selectedItemText: {
    fontSize: width * 0.035,
    color: '#015185',
    fontWeight: 'bold',
  },
  selectedItemSubText: {
    fontSize: width * 0.03,
    color: '#666',
    marginTop: width * 0.005,
  },
  noUsersSelected: {
    alignItems: 'center',
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.04,
    backgroundColor: '#f8f9fa',
    borderRadius: width * 0.02,
    marginTop: width * 0.01,
    borderWidth: 1,
    borderColor: '#e9ecef',
    borderStyle: 'dashed',
  },
  noUsersSelectedText: {
    fontSize: width * 0.035,
    color: '#6c757d',
    fontWeight: '500',
  },
  noUsersSelectedSubText: {
    fontSize: width * 0.03,
    color: '#adb5bd',
    marginTop: width * 0.005,
    textAlign: 'center',
  },
  closeButton: {
    padding: width * 0.005,
  },
  closeCircle: {
    width: width * 0.04,
    height: width * 0.04,
    borderRadius: (width * 0.04) / 2,
    backgroundColor: '#d32f2f',
    justifyContent: 'center',
    alignItems: 'center',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
  },
  dateText: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.02,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
    zIndex: 9999,
    elevation: 9999,
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
    maxHeight: '80%',
    zIndex: 10000,
    elevation: 10000,
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  modalOption: {
    paddingVertical: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  statusOption: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.015,
  },
  statusColor: {
    width: width * 0.03,
    height: width * 0.03,
    borderRadius: (width * 0.03) / 2,
    marginRight: width * 0.02,
  },
  statusOptionText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  optionRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-start',
  },
  optionDetails: {
    flex: 1,
  },
  modalOptionText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  modalOptionSubtext: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: 2,
  },
  modalOptionSelected: {
    backgroundColor: '#e3f2fd',
    borderColor: '#015185',
  },
  modalDivider: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginVertical: 10,
  },
  linkedInspectionDetails: {
    backgroundColor: '#f8f9fa',
    padding: 8,
    borderRadius: 4,
    marginTop: 4,
  },
  linkedInspectionText: {
    fontSize: width * 0.03,
    color: '#666',
    fontStyle: 'italic',
  },
  inspectionTaskRow: {
    backgroundColor: '#FFF8F0',
    borderLeftWidth: 3,
    borderLeftColor: '#FF6B35',
  },
  inspectionTaskLabel: {
    fontSize: width * 0.025,
    color: '#FF6B35',
    fontStyle: 'italic',
    marginTop: 2,
    fontWeight: '500',
  },
  modalOptionSubText: {
    fontSize: width * 0.035,
    color: '#707173',
  },
  checkmarkContainer: {
    marginRight: width * 0.03,
  },
  roleContainer: {
    backgroundColor: '#28a745',
    borderRadius: width * 0.02,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.01,
  },
  roleText: {
    fontSize: width * 0.035,
    color: '#fff',
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: width * 0.025,
  },
  modalButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.02,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: width * 0.01,
  },
  modalTextButton: {
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
  },
  modalTextButtonText: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '600',
    textTransform: 'uppercase',
  },
  cancelButton: {
   backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  deleteButton: {
    backgroundColor: '#d32f2f',
  },
  modalButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  labelwot: {
    color: 'black',
    fontWeight: '400',
    fontSize: width * 0.032,
    color: '#707173',
    paddingHorizontal: 2,
  },
  wotclono: {
    borderBottomWidth: 1,
    borderColor: '#015185',
    width: '98%',
    height: 1,
    alignSelf: 'center',
    left: -1,
  },
  inputcost0: {
    paddingHorizontal: 10,
    paddingVertical: 5,
    borderRadius: 3,
  },
  inputcost0tax: {
    borderWidth: 1,
    borderColor: '#3481BC',
    paddingVertical: 4,
    paddingHorizontal: 4,
    borderRadius: 3,
  },
  typeContainer: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginBottom: width *    0.01,
  },
  typeButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.01,
    marginRight: width * 0.02,
  },
  typeButtonSelected: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  typeText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  typeTextSelected: {
    fontSize: width * 0.035,
    color: '#fff',
    fontWeight: '600',
  },
  searchInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.025,
  },
  updateTaskModal: {
    backgroundColor: '#fff',
    width: '80%',
    borderRadius: 5,
    padding: width * 0.04,
  },
  addTaskTitle: {
    fontSize: width * 0.045,
    color: '#015185',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  taskFormGroup: {
    marginBottom: width * 0.03,
  },
  taskLabel: {
    fontSize: width * 0.035,
    color: '#000',
    marginBottom: width * 0.01,
  },
  taskTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    backgroundColor: '#fff',
  },
  typeButtonContainer: {
    flexDirection: 'row',
    gap: 10,
  },
  repairButtonSelected: {
    backgroundColor: '#FFA500', // Orange/Amber color Orange
    borderColor: '#FFA500',
  },
  assignToButton: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  assignToButtonText: {
    fontSize: width * 0.035,
    color: '#666',
  },
  assignToArrow: {
    fontSize: width * 0.04,
    color: '#015185',
  },
  taskButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: width * 0.04,
  },
  taskCancelButton: {
    backgroundColor: '#6c757d',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: 3,
  },
  taskSaveButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
    borderRadius: 3,
  },
  taskButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
  },
  laborModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  laborTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  laborRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  laborLabel: {
    fontSize: width * 0.035,
    color: '#333',
  },
  laborSelectButton: {
    padding: width * 0.02,
  },
  laborInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  laborSubLabel: {
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.02,
  },
  hoursContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  hoursInput: {
    flex: 1,
    marginRight: width * 0.02,
  },
  minutesInput: {
    flex: 1,
    marginLeft: width * 0.02,
  },
  partsModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  partsTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  partsRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  partsLabel: {
    fontSize: width * 0.035,
    color: '#333',
  },
  partsSelectButton: {
    padding: width * 0.02,
  },
  partsInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  quantityContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  quantityInput: {
    flex: 2,
    marginRight: width * 0.02,
  },
  unitSelector: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingVertical: width * 0.02,
  },
  unitText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  remarksModal: {
    backgroundColor: '#fff',
    padding: width * 0.05,
  },
  remarksTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  remarksInput: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.02,
    fontSize: width * 0.035,
    color: '#333',
  },
  descriptionInput: {
    height: width * 0.2,
    textAlignVertical: 'top',
  },
  typeRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  assignToRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  bottomBorderInput: {
    borderWidth: 0,
    borderBottomWidth: 1,
    borderColor: '#ddd',
  },
  assignToModal: {
    backgroundColor: '#fff',
    width: '90%',
    maxHeight: '80%',
    borderRadius: 10,
    padding: 20,
  },
  assignToModalTitle: {
    fontSize: 24,
    color: '#015185',
    fontWeight: '600',
    textAlign: 'center',
    marginBottom: 20,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#ccc',
  },
  searchIcon: {
    marginRight: 10,
  },
  assignToSearchInput: {
    flex: 1,
    paddingVertical: 10,
    fontSize: 16,
    color: '#333',
  },
  assignToList: {
    marginBottom: 15,
  },
  assignToOption: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  assignToUserInfo: {
    flex: 1,
  },
  userIdRow: {
    flexDirection: 'row',
    marginBottom: 4,
  },
  nameRow: {
    flexDirection: 'row',
  },
  userIdLabel: {
    width: 70,
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
  },
  nameLabel: {
    width: 70,
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
  },
  userIdValue: {
    fontSize: 16,
    color: '#666',
    marginLeft: 10,
  },
  nameValue: {
    fontSize: 16,
    color: '#666',
    marginLeft: 10,
  },
  technicianBadge: {
    backgroundColor: '#015185',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 5,
    marginLeft: 10,
  },
  technicianText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '500',
  },
  noResultsText: {
    textAlign: 'center',
    color: '#666',
    fontSize: 16,
    marginTop: 20,
  },
  // selectButton: {
    // backgroundColor: '#015185',
    // paddingVertical: 12,
    // borderRadius: 5,
    // marginTop: 10,
  // },
  // selectButtonText: {
  //   color: '#fff',
  //   fontSize: 16,
  //   fontWeight: '600',
  //   textAlign: 'center',
  // },
  faultListButton: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderColor: '#ddd',
    paddingVertical: 8,
    marginTop: 5,
  },
  faultListButtonText: {
    fontSize: 14,
    color: '#666',
  },
  faultListModal: {
    backgroundColor: '#fff',
    width: '90%',
    maxHeight: '80%',
    borderRadius: 10,
    padding: 20,
  },
  faultListTitle: {
    fontSize: 20,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 20,
  },
  faultList: {
    marginBottom: 15,
  },
  faultItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  faultItemLeft: {
    flex: 1,
  },
  faultName: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    marginBottom: 4,
  },
  faultTime: {
    fontSize: 12,
    color: '#666',
  },
  faultItemRight: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 10,
  },
  faultCode: {
    fontSize: 14,
    color: '#666',
  },
  faultListButtons: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: 10,
  },
  faultListActionButton: {
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  closeButton: {
    backgroundColor: '#f0f0f0',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  saveButtonText: {
    color: '#fff',
  },
  saveButtoncreate: {
    backgroundColor: '#FFFFFF',
    paddingVertical: 8,
    paddingHorizontal: 16,
    borderRadius: 4,
    marginRight: 15,
    marginTop: 5,
  },
  saveButtonText0: {
    color: '#015185',
    fontSize: 14,
    fontWeight: '500',
  },
  loadingModalOverlay: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
    backgroundColor: 'rgba(255, 255, 255, 0.9)',
    justifyContent: 'center',
    alignItems: 'center',
    zIndex: 9999,
  },
  loadingContainer: {
    backgroundColor: '#FFFFFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 3.84,
    minWidth: 150,
    minHeight: 150,
  },
  spinner: {
    marginBottom: 15,
  },
  loadingText: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    textAlign: 'center',
  },
  additionalCostModal: {
    backgroundColor: '#FFFFFF',
    width: '90%',
    borderRadius: 5,
    padding: 20,
  },
  additionalCostTitle: {
    color: '#015185',
    fontSize: 18,
    fontWeight: '600',
    marginBottom: 20,
    textAlign: 'center',
  },
  additionalCostInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    padding: 12,
    marginBottom: 15,
    color: '#333',
    backgroundColor: '#FFFFFF',
    fontSize: 16,
  },
  descriptionInput: {
    height: 120,
    textAlignVertical: 'top',
    paddingTop: 12,
  },
  laborCodeInput: {
    height: 50,
  },
  additionalCostButtons: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    gap: 10,
    marginTop: 10,
  },
  additionalCostButton: {
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
    minWidth: 80,
  },
  cancelButton: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
  },
  saveButton: {
    backgroundColor: '#015185',
  },
  cancelButtonText: {
    color: '#FFFFFF',
    fontSize: 14,
    fontWeight: '500',
    textAlign: 'center',
  },
  saveButtonText: {
    color: '#FFFFFF',
    fontSize: 14,
    fontWeight: '500',
    textAlign: 'center',
  },
  successModalContent: {
    backgroundColor: '#FFFFFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '92%',
    maxWidth: width * 0.95,
  },
  successModalTitle: {
    fontSize: 24,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 15,
  },
  successIcon: {
    marginVertical: 15,
  },
  successModalText: {
    fontSize: 18,
    color: '#333',
    marginBottom: 20,
  },
  successModalButtons: {
    flexDirection: 'row',
    justifyContent: 'center',
    gap: 10,
    width: '100%',
  },
  successModalButton: {
    paddingVertical: 5,
    paddingHorizontal: 10,
    borderRadius: 1,
    minWidth: 100,
    alignItems: 'center',
  },
  backButton: {
    backgroundColor: '#015185',
  },
  viewPDFButton: {
    backgroundColor: '#015185',
  },
  successModalButtonText: {
    color: '#FFFFFF',
    fontSize: 16,
    fontWeight: '400',
  },
    requestNumberRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.02,
  },
    requestNumberLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
    requestNumberValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
   scrollContent: {
    flexGrow: 1,
    paddingBottom: height * 0.1,
  },
   keyboardAvoidingView: {
    flex: 1,
  },
  successModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    alignItems: 'center',
    width: 340,
    maxWidth: '90%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  successModalTitleCustom: {
    fontSize: 24,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 4,
  },
  successModalTextCustom: {
    fontSize: 20,
    color: '#015185',
    marginTop: 2,
    textAlign: 'center',
    fontWeight: '400',
    marginBottom: 2,
  },
  successModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 4,
    alignItems: 'center',
  },
  successModalButtonCustom: {
    borderRadius: 4,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 0,
    marginRight: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  successModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '500',
  },
  loadingModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContentCustomSmall: {
    backgroundColor: '#fff',
    borderRadius: 16,
    paddingVertical: 18,
    paddingHorizontal: 18,
    alignItems: 'center',
    width: 140,
    maxWidth: '90%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  loadingIconSmall: {
    width: 38,
    height: 38,
    marginVertical: 8,
    tintColor: '#1A4C8B',
  },
  loaderModalTitleCustomSmall: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginTop: 2,
    marginBottom: 2,
  },
  loaderModalTextCustomSmall: {
    fontSize: 13,
    color: '#222',
    marginTop: 2,
    textAlign: 'center',
    fontWeight: '400',
  },
  processorRowCenter: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  processorModalText: {
    fontSize: 16,
    color: '#015185',
    marginLeft: 12,
    textAlign: 'center',
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  checklistItemContainer: {
    backgroundColor: '#000',
    borderRadius: 8,
    padding: 12,
    marginBottom: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  checklistItemHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  checklistItemText: {
    fontSize: 16,
    color: '#fff',
    flex: 1,
  },
  menuButton: {
    padding: 4,
  },
  assignmentBadge: {
    backgroundColor: '#e6f4ea',
    borderWidth: 1,
    borderColor: '#28a745',
    borderRadius: 4,
    padding: 8,
    marginTop: 4,
  },
  assignmentText: {
    color: '#28a745',
    fontSize: 12,
    fontWeight: '500',
  },
  assignmentDate: {
    color: '#666',
    fontSize: 11,
    marginTop: 2,
  },
  disabledAssignmentBadge: {
    backgroundColor: '#f0f0f0',
    borderColor: '#ddd',
  },
  disabledAssignmentText: {
    color: '#999',
  },
  userSelectionModal: {
    backgroundColor: '#fff',
    borderRadius: 8,
    width: '90%',
    maxHeight: '80%',
    padding: 16,
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 16,
  },
  userItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  userInfo: {
    flex: 1,
  },
  userName: {
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
  },
  userRole: {
    fontSize: 14,
    color: '#666',
    marginTop: 2,
  },
  disabledButton: {
    opacity: 0.5,
  },
  assignmentInfo: {
    marginTop: 8,
    padding: 8,
    backgroundColor: '#f8f9fa',
    borderRadius: 4,
  },
  rightSection: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  assignedUserText: {
    color: '#28a745',
    fontSize: 12,
    fontWeight: '500',
  },
  notAssignedText: {
    color: '#6c757d',
    fontSize: 12,
    fontWeight: '500',
  },
  dateInputButton: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.03,
  },
  dateInputText: {
    fontSize: width * 0.035,
    color: '#333',
    marginRight: width * 0.02,
  },
  laborModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  laborModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignItems: 'stretch',
    elevation: 8,
  },
  laborModalTitleCustom: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  laborInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 14,
    backgroundColor: '#fff',
  },
  laborHoursRowCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 14,
  },
  laborHoursLabelCustom: {
    fontSize: 15,
    color: '#015185',
    fontWeight: '500',
    marginRight: 10,
    minWidth: 80,
  },
  laborHoursInputCustom: {
    flex: 1,
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 10,
    fontSize: 16,
    marginRight: 10,
    backgroundColor: '#fff',
  },
  laborNoteInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 18,
    backgroundColor: '#fff',
    minHeight: 120,
    textAlignVertical: 'top',
  },
  laborModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
  },
  laborModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 10,
    minWidth: 70,
    alignItems: 'center',
  },
  laborModalCancelButtonCustom: {
    backgroundColor: '#015185',
  },
  laborModalSaveButtonCustom: {
    backgroundColor: '#015185',
  },
  laborModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  additionalCostModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  additionalCostModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignItems: 'stretch',
    elevation: 8,
  },
  additionalCostModalTitleCustom: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  additionalCostDescriptionInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 18,
    backgroundColor: '#fff',
    minHeight: 100,
    textAlignVertical: 'top',
  },
  additionalCostInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 18,
    backgroundColor: '#fff',
  },
  additionalCostModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
  },
  additionalCostModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 0,
    marginRight: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  additionalCostModalCancelButtonCustom: {
    // backgroundColor: '#015185',
  },
  additionalCostModalSaveButtonCustom: {
    // backgroundColor: '#015185',
  },
  additionalCostModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  taskModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  taskModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignItems: 'stretch',
    elevation: 8,
  },
  taskModalTitleCustom: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  taskModalRowCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 0,
    paddingVertical: 10,
   
    paddingHorizontal: 0,
  },
  taskModalLabelCustom: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '500',
    minWidth: 90,
  },
  taskModalInputCustom: {
    flex: 1,
    borderWidth: 0,
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    textAlign: 'left',
    marginLeft: 10,
  },
  taskModalTypeButtonRowCustom: {
    flexDirection: 'row',
    gap: 8,
  },
  taskModalTypeButtonCustom: {
    backgroundColor: '#e0e0e0',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 18,
    marginLeft: 8,
  },
  taskModalTypeButtonSelectedCustom: {
    backgroundColor: '#015185',
  },
  taskModalTypeButtonTextCustom: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
  },
  taskModalTypeButtonTextSelectedCustom: {
    color: '#fff',
    fontWeight: '600',
  },
  taskModalAssignButtonCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 12,
    marginLeft: 10,
  },
  taskModalAssignButtonTextCustom: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
    marginRight: 4,
  },
  taskModalFaultButtonCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 12,
    marginLeft: 10,
  },
  taskModalFaultButtonTextCustom: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
    marginRight: 4,
  },
  taskModalDividerCustom: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginVertical: 0,
  },
  taskModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: 18,
  },
  taskModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  taskModalCancelButtonCustom: {
    // backgroundColor: '#015185',
  },
  taskModalSaveButtonCustom: {
    // backgroundColor: '#015185',
  },
  taskModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  remarksModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  remarksModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 500,
    alignItems: 'stretch',
    elevation: 8,
  },
  remarksModalTitleCustom: {
    fontSize: 26,
    color: '#015185',
    fontWeight: '500',
    textAlign: 'center',
    marginBottom: 18,
  },
  remarksDescriptionInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 10,
    paddingVertical: 16,
    paddingHorizontal: 18,
    fontSize: 18,
    marginBottom: 24,
    backgroundColor: '#fff',
    minHeight: 120,
    textAlignVertical: 'top',
  },
  remarksModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
  },
  remarksModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 0,
    marginRight: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  remarksModalCancelButtonCustom: {
    // backgroundColor: '#015185',
  },
  remarksModalSaveButtonCustom: {
    // backgroundColor: '#015185',
  },
  remarksModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '400',
  },
  loaderModalOverlayCustom: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
    backgroundColor: 'rgba(0,0,0,0.2)',
    justifyContent: 'center',
    alignItems: 'center',
    zIndex: 9999,
  },
  loaderModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 32,
    alignItems: 'center',
    width: 140,
    maxWidth: '90%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 8,
  },
  loaderModalTitleCustom: {
    fontSize: 28,
    color: '#015185',
    fontWeight: '500',
    textAlign: 'center',
    marginBottom: 8,
  },
  loaderModalTextCustom: {
    fontSize: 22,
    color: '#222',
    marginTop: 8,
    textAlign: 'center',
    fontWeight: '400',
  },
  partsModalOverlayCustom: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.3)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  partsModalContentCustom: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 24,
    width: '90%',
    maxWidth: 400,
    alignItems: 'stretch',
    elevation: 8,
  },
  partsModalTitleCustom: {
    fontSize: 22,
    color: '#015185',
    fontWeight: '700',
    textAlign: 'center',
    marginBottom: 18,
  },
  partsInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 14,
    backgroundColor: '#fff',
  },
  partsUnitContainerCustom: {
    marginBottom: 14,
  },
  partsUnitLabelCustom: {
    fontSize: 15,
    color: '#015185',
    fontWeight: '500',
    marginBottom: 8,
  },
  partsUnitButtonRowCustom: {
    flexDirection: 'row',
    gap: 8,
  },
  partsUnitButtonCustom: {
    backgroundColor: '#e0e0e0',
    borderRadius: 5,
    paddingVertical: 6,
    paddingHorizontal: 18,
    marginRight: 8,
  },
  partsUnitButtonSelectedCustom: {
    backgroundColor: '#015185',
  },
  partsUnitButtonTextCustom: {
    color: '#015185',
    fontSize: 15,
    fontWeight: '500',
  },
  partsUnitButtonTextSelectedCustom: {
    color: '#fff',
    fontWeight: '600',
  },
  partsNoteInputCustom: {
    borderWidth: 2,
    borderColor: '#015185',
    borderRadius: 7,
    paddingVertical: 10,
    paddingHorizontal: 14,
    fontSize: 16,
    marginBottom: 18,
    backgroundColor: '#fff',
    minHeight: 120,
    textAlignVertical: 'top',
  },
  partsModalButtonRowCustom: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginTop: 10,
  },
  partsModalButtonCustom: {
    borderRadius: 5,
    paddingVertical: 8,
    paddingHorizontal: 16,
    marginLeft: 0,
    marginRight: 10,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#015185',
  },
  partsModalCancelButtonCustom: {
    // backgroundColor: '#015185',
  },
  partsModalSaveButtonCustom: {
    // backgroundColor: '#015185',
  },
  partsModalButtonTextCustom: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  selectEquipmentButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    borderRadius: 5,
    marginTop: 10,
  },
  selectEquipmentButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    textAlign: 'center',
    paddingHorizontal: 12,
  },
  // Inline Assign To Styles
  taskModalInlineAssignToContainer: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginVertical: 10,
    padding: 12,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  taskModalInlineAssignToHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  taskModalInlineAssignToTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
  },
  taskModalInlineSearchInput: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 8,
    fontSize: 14,
    marginBottom: 10,
  },
  taskModalInlineAssignToList: {
    maxHeight: 200,
  },
  taskModalInlineAssignToItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 8,
    paddingHorizontal: 4,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    backgroundColor: '#fff',
    marginBottom: 2,
    borderRadius: 4,
  },
  taskModalInlineProfilePic: {
    width: 32,
    height: 32,
    borderRadius: 16,
    marginRight: 10,
  },
  taskModalInlineProfilePicPlaceholder: {
    width: 32,
    height: 32,
    borderRadius: 16,
    backgroundColor: '#ddd',
    marginRight: 10,
    alignItems: 'center',
    justifyContent: 'center',
  },
  taskModalInlineUserInfo: {
    flex: 1,
  },
  taskModalInlineUserName: {
    fontWeight: '600',
    color: '#015185',
    fontSize: 14,
  },
  taskModalInlineUserDetails: {
    color: '#666',
    fontSize: 12,
  },
  // Assign To Modal Styles (matching TeamMemberHome)
  assignToModalItem: {
    flexDirection: 'row',
    padding: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    backgroundColor: '#fff',
    marginVertical: 2,
    borderRadius: 8,
    elevation: 1,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  assignToModalItemSelected: {
    backgroundColor: '#e6f0ff',
    borderColor: '#015185',
  },
  assignToModalItemLeft: {
    flex: 2,
    flexDirection: 'row',
    paddingRight: 12,
  },
  assignToModalItemRight: {
    flex: 1,
    alignItems: 'flex-end',
    justifyContent: 'space-between',
    flexDirection: 'column',
  },
  assignToModalProfilePic: {
    width: 40,
    height: 40,
    borderRadius: 20,
    marginRight: 10,
  },
  assignToModalProfilePicPlaceholder: {
    width: 40,
    height: 40,
    borderRadius: 20,
    backgroundColor: '#ddd',
    marginRight: 10,
    alignItems: 'center',
    justifyContent: 'center',
  },
  assignToModalUserInfo: {
    flex: 1,
  },
  assignToModalUserName: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 4,
  },
  assignToModalSelfText: {
    color: '#4CAF50',
    fontWeight: 'bold',
  },
  assignToModalUserDetails: {
    fontSize: 13,
    color: '#333',
    marginBottom: 2,
    lineHeight: 18,
  },
  assignToModalRoleBadge: {
    backgroundColor: '#4CAF50',
    paddingVertical: 6,
    paddingHorizontal: 12,
    borderRadius: 12,
    alignItems: 'center',
    marginBottom: 8,
  },
  assignToModalRoleText: {
    fontSize: 12,
    fontWeight: 'bold',
    color: '#fff',
  },
  // Inline Assign To Enhanced Styles
  taskModalInlineAssignToItemSelected: {
    backgroundColor: '#e6f0ff',
    borderColor: '#015185',
  },
  taskModalInlineAssignToItemLeft: {
    flex: 2,
    flexDirection: 'row',
    paddingRight: 8,
  },
  taskModalInlineAssignToItemRight: {
    flex: 1,
    alignItems: 'flex-end',
    justifyContent: 'space-between',
    flexDirection: 'column',
  },
  taskModalInlineSelfText: {
    color: '#4CAF50',
    fontWeight: 'bold',
  },
  taskModalInlineRoleBadge: {
    backgroundColor: '#4CAF50',
    paddingVertical: 4,
    paddingHorizontal: 8,
    borderRadius: 10,
    alignItems: 'center',
    marginBottom: 6,
  },
  taskModalInlineRoleText: {
    fontSize: 10,
    fontWeight: 'bold',
    color: '#fff',
  },
  labelText: {
    fontWeight: 'bold',
    color: '#333',
  },
  equipmentSelectButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    borderRadius: 5,
    marginTop: 10,
  },
  equipmentSelectButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    textAlign: 'center',
    paddingHorizontal: 12,
  },
  inspectionModal: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
  },
  permissionInfo: {
    backgroundColor: '#f8f9fa',
    padding: 10,
    borderRadius: 5,
    marginBottom: 10,
  },
  permissionText: {
    fontSize: 12,
    color: '#666',
    marginBottom: 2,
  },
  inspectionInfo: {
    marginBottom: 15,
  },
  inspectionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 5,
  },
  inspectionLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  inspectionValue: {
    fontSize: 14,
    color: '#666',
  },
  checkmarkContainer: {
    marginLeft: 10,
  },
  inspectionList: {
    maxHeight: 200,
    marginBottom: 15,
  },
  defaultText: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
  },
  inspectionOption: {
    paddingVertical: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  sectionHeader: {
    fontSize: 16,
    color: '#015185',
    fontWeight: 'bold',
    marginBottom: 10,
  },
  taskCard: {
    padding: 10,
    borderWidth: 2,
    borderColor: '#ddd',
    borderRadius: 5,
    marginBottom: 10,
  },
  taskTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    marginBottom: 5,
  },
  taskType: {
    fontSize: 14,
    color: '#666',
    marginBottom: 5,
  },
  taskNote: {
    fontSize: 14,
    color: '#777',
    marginBottom: 5,
  },
  taskStatus: {
    fontSize: 14,
    fontWeight: 'bold',
    color: '#333',
  },
  emptyText: {
    fontSize: 14,
    color: '#999',
    textAlign: 'center',
    marginVertical: 10,
  },
  // Add state for which task's status menu is open
  taskStatusMenuVisibleId: {
    // Add your styles here
  },
  dropdownList: {
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
  },
  dropdownMenu: {
    position: 'absolute',
    right: 10,
    top: 40,
    backgroundColor: '#fff',
    borderRadius: 6,
    elevation: 4,
    zIndex: 100,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  dropdownMenuItem: {
    padding: 12,
  },
  threeDotsButton: {
    padding: 8,
    marginLeft: 8,
  },
  
  // Inspection Modal Styles
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 20,
    paddingBottom: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  closeButton: {
    padding: 5,
  },
  loadingContainer: {
    alignItems: 'center',
    paddingVertical: 40,
  },
  loadingText: {
    marginTop: 10,
    color: '#666',
    fontSize: 16,
  },
  inspectionItem: {
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    backgroundColor: '#fff',
  },
  selectedInspectionItem: {
    backgroundColor: '#e3f2fd',
    borderLeftWidth: 4,
    borderLeftColor: '#015185',
  },
  inspectionItemHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  inspectionRequestNumber: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
  },
  inspectionItemDetails: {
    marginTop: 5,
  },
  inspectionEquipment: {
    fontSize: 14,
    color: '#333',
    marginBottom: 3,
  },
  inspectionDate: {
    fontSize: 14,
    color: '#666',
    marginBottom: 3,
  },
  inspectionFaulty: {
    fontSize: 14,
    color: '#d32f2f',
    fontWeight: '600',
  },
  emptyContainer: {
    alignItems: 'center',
    paddingVertical: 40,
  },
  emptyText: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  emptySubtext: {
    fontSize: 14,
    color: '#999',
    marginTop: 5,
  },
  selectedInspectionContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e8f5e8',
    padding: 12,
    borderRadius: 8,
    marginVertical: 10,
  },
  selectedInspectionLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#2e7d32',
    marginRight: 8,
  },
  selectedInspectionText: {
    flex: 1,
    fontSize: 14,
    color: '#2e7d32',
    fontWeight: '500',
  },
  removeInspectionButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#d32f2f',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 15,
  },
  removeInspectionText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    marginLeft: 4,
  },
  
  // Job Card Modal Styles
  jobCardModal: {
    maxHeight: '80%',
    width: '90%',
  },
  jobCardItem: {
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    backgroundColor: '#fff',
  },
  selectedJobCardItem: {
    backgroundColor: '#e3f2fd',
    borderLeftWidth: 4,
    borderLeftColor: '#015185',
  },
  jobCardItemHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  jobCardTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
  },
  jobCardItemDetails: {
    marginTop: 5,
  },
  jobCardRequestNumber: {
    fontSize: 14,
    color: '#333',
    marginBottom: 3,
  },
  jobCardStatus: {
    fontSize: 14,
    color: '#666',
    marginBottom: 3,
  },
  jobCardDate: {
    fontSize: 14,
    color: '#999',
  },
  selectedJobCardContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e8f5e8',
    padding: 12,
    borderRadius: 8,
    marginVertical: 10,
  },
  selectedJobCardLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#2e7d32',
    marginRight: 8,
  },
  selectedJobCardText: {
    flex: 1,
    fontSize: 14,
    color: '#2e7d32',
    fontWeight: '500',
  },
  removeJobCardButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#d32f2f',
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 15,
  },
  removeJobCardText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    marginLeft: 4,
  },
  linkedJobCardDetails: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    padding: 10,
    borderRadius: 6,
    marginTop: 8,
  },
  linkedJobCardText: {
    flex: 1,
    fontSize: 12,
    color: '#015185',
  },
  removeLinkedButton: {
    padding: 4,
    marginLeft: 8,
  },
  
});

export default CreateWorkOrder;         


