import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  StatusBar,
  ScrollView,
  Modal,
  Animated,
  Image,
  ToastAndroid,
  Platform,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import { Dropdown } from 'react-native-element-dropdown';
import Toast from 'react-native-toast-message';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';

// Utility to ensure fontSize is a number
const ensureFontSizeNumber = (style) => ({
  ...style,
  fontSize: Number(style?.fontSize) || 14,
});

// Constants
const TOTAL_HEADER_HEIGHT = 180;
const INITIAL_ITEMS_TO_LOAD = 15;
const ITEMS_PER_PAGE = 10;

const TeamMemberEquipHome = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [filteredEquipment, setFilteredEquipment] = useState([]);
  const [displayedEquipment, setDisplayedEquipment] = useState([]);
  const [page, setPage] = useState(1);
  const [isLoadingMore, setIsLoadingMore] = useState(false);
  const [isLoadingInitial, setIsLoadingInitial] = useState(true);
  const [hasMore, setHasMore] = useState(true);
  const [actionModalVisible, setActionModalVisible] = useState(false);
  const [statusModalVisible, setStatusModalVisible] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [newStatus, setNewStatus] = useState('Available');
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [selectedFilter, setSelectedFilter] = useState(null);
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [teamMemberId, setTeamMemberId] = useState(null);
  const [teamMemberUserId, setTeamMemberUserId] = useState(null);
  const isMounted = useRef(true);

  const statusItems = [
    { label: 'Available', value: 'Available' },
    { label: 'Maintenance Required', value: 'Maintenance Required' },
    { label: 'In Maintenance', value: 'In Maintenance' },
    { label: 'Breakdown', value: 'Breakdown' },
    { label: 'Deploy', value: 'Deploy' },
  ];

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (syncModalVisible) {
      Animated.loop(Animated.timing(rotateAnim, { toValue: 1, duration: 1000, useNativeDriver: true })).start();
    } else {
      rotateAnim.setValue(0);
    }
    return () => rotateAnim.setValue(0);
  }, [syncModalVisible]);

  useEffect(() => {
    return () => { isMounted.current = false; };
  }, []);

  const withRetry = useCallback(async (operation, maxAttempts = 3, delay = 1000) => {
    for (let attempt = 1; attempt <= maxAttempts; attempt++) {
      try {
        return await operation();
      } catch (error) {
        if (attempt === maxAttempts) throw error;
        await new Promise(resolve => setTimeout(resolve, delay));
      }
    }
  }, []);

  const checkLocalCredentials = async () => {
    try {
      const [storedTeamMemberUserId, storedTeamMemberId] = await AsyncStorage.multiGet(['teamMemberUserId', 'teamMemberId']);
      if (!storedTeamMemberUserId[1] || !storedTeamMemberId[1]) {
        showToast('error', 'Session Error', 'No active user found. Please log in again.');
        navigation.replace('TeamMemberLogin');
        return false;
      }

      setTeamMemberId(storedTeamMemberId[1]);
      setTeamMemberUserId(storedTeamMemberUserId[1]);
      return true;
    } catch (error) {
      console.error('Credential check error:', error);
      showToast('error', 'Session Error', 'Failed to validate credentials. Please log in again.');
      navigation.replace('TeamMemberLogin');
      return false;
    }
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Network check error:', error);
      return false;
    }
  };

  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    if (Platform.OS === 'android') {
      ToastAndroid.show(`${text1}: ${text2}`, ToastAndroid.LONG);
    } else {
      Toast.show({ type, text1, text2, position: 'bottom', visibilityTime: 3000 });
    }
  };

  const loadData = async (isRefresh = false) => {
    if (!(await checkLocalCredentials())) {
      setSyncModalVisible(false);
      setIsRefreshing(false);
      setIsLoadingInitial(false);
      return;
    }

    setSyncModalVisible(true);
    setIsLoadingInitial(!isRefresh);

    try {
      const isOnline = await checkNetworkStatus();
      let parsedData = [];

      if (isOnline) {
        const snapshot = await withRetry(() =>
          database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData').once('value')
        );
        let data = snapshot.val() || [];
        data = Array.isArray(data) ? data : [];
        parsedData = data
          .filter((item, index, self) => self.findIndex(t => t.id === item.id) === index)
          .map((item, index) => ({
            ...item,
            firebaseKey: item.firebaseKey || `equip_${index}`,
          }))
          .sort((a, b) => {
            if (a.teamMemberId === teamMemberId && b.teamMemberId !== teamMemberId) return -1;
            if (a.teamMemberId !== teamMemberId && b.teamMemberId === teamMemberId) return 1;
            const dateA = new Date(a.updatedAt || a.createdAt);
            const dateB = new Date(b.updatedAt || b.createdAt);
            return dateB - dateA;
          });

        await AsyncStorage.setItem('equipment', JSON.stringify(parsedData));
      } else {
        const savedData = await AsyncStorage.getItem('equipment');
        parsedData = savedData ? JSON.parse(savedData) : [];
        showToast('info', 'Offline Mode', 'Loaded from local storage.');
      }

      if (isMounted.current) {
        setFirebaseEquipment(parsedData);
        setFilteredEquipment(parsedData);
        setDisplayedEquipment(parsedData.slice(0, INITIAL_ITEMS_TO_LOAD));
        setHasMore(parsedData.length > INITIAL_ITEMS_TO_LOAD);
        const storedCountry = await AsyncStorage.getItem('selectedCountry');
        setSelectedCountry(storedCountry || 'Both Countries');
      }

      if (parsedData.length === 0) {
        showToast('info', 'No Equipment', isOnline ? 'No equipment found.' : 'No equipment available offline.');
      }
    } catch (error) {
      console.error('Load equipment error:', error);
      let errorMessage = 'Failed to load equipment.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (!isOnline) {
        errorMessage = 'Offline: Loaded from local storage.';
        const savedData = await AsyncStorage.getItem('equipment');
        let parsedData = savedData ? JSON.parse(savedData) : [];
        parsedData = parsedData.sort((a, b) => {
          if (a.teamMemberId === teamMemberId && b.teamMemberId !== teamMemberId) return -1;
          if (a.teamMemberId !== teamMemberId && b.teamMemberId === teamMemberId) return 1;
          const dateA = new Date(a.updatedAt || a.createdAt);
          const dateB = new Date(b.updatedAt || b.createdAt);
          return dateB - dateA;
        });
        if (isMounted.current) {
          setFirebaseEquipment(parsedData);
          setFilteredEquipment(parsedData);
          setDisplayedEquipment(parsedData.slice(0, INITIAL_ITEMS_TO_LOAD));
          setHasMore(parsedData.length > INITIAL_ITEMS_TO_LOAD);
        }
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setIsRefreshing(false);
        setIsLoadingInitial(false);
      }
    }
  };

  useEffect(() => {
    const initializeData = async () => {
      setSyncModalVisible(true);
      try {
        if (await checkLocalCredentials()) {
          await loadData(true);
        }
      } catch (error) {
        console.error('Initialization error:', error);
      } finally {
        if (isMounted.current) setSyncModalVisible(false);
      }
    };
    initializeData();
  }, []);

  useFocusEffect(
    useCallback(() => {
      handleRefresh();
    }, [teamMemberId])
  );

  useEffect(() => {
    const syncPendingChanges = async () => {
      if (!(await checkNetworkStatus()) || !teamMemberId || !teamMemberUserId) return;

      try {
        const equipment = JSON.parse(await AsyncStorage.getItem('equipment') || '[]');
        const pending = equipment.filter(item => item.pendingSync && item.teamMemberId === teamMemberId);
        if (pending.length === 0) return;

        setSyncModalVisible(true);
        const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
        const globalSnapshot = await withRetry(() => globalRef.once('value'));
        let globalData = globalSnapshot.val() || [];
        globalData = Array.isArray(globalData) ? globalData : [];

        const teamMemberRef = database().ref(`/GlobalTeamMemberInformation/${teamMemberId}/equipment`);
        const teamMemberSnapshot = await withRetry(() => teamMemberRef.once('value'));
        let teamMemberData = teamMemberSnapshot.val() || [];
        teamMemberData = Array.isArray(teamMemberData) ? teamMemberData : [];

        for (const item of pending) {
          const updatedItem = { ...item, pendingSync: false, pendingDelete: false };
          if (item.pendingDelete) {
            globalData = globalData.filter(g => g.id !== item.id);
            teamMemberData = teamMemberData.filter(t => t.id !== item.id);
          } else {
            const globalIndex = globalData.findIndex(g => g.id === item.id);
            if (globalIndex !== -1) globalData[globalIndex] = updatedItem;
            else globalData.push(updatedItem);
            const teamMemberIndex = teamMemberData.findIndex(t => t.id === item.id);
            if (teamMemberIndex !== -1) teamMemberData[teamMemberIndex] = updatedItem;
            else teamMemberData.push(updatedItem);
          }
        }

        await withRetry(() => globalRef.set(globalData));
        await withRetry(() => teamMemberRef.set(teamMemberData));

        const updatedEquipment = equipment
          .filter(item => !(item.pendingSync && item.pendingDelete && item.teamMemberId === teamMemberId))
          .map(item => item.pendingSync && item.teamMemberId === teamMemberId ? { ...item, pendingSync: false, pendingDelete: false } : item);
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipment));

        showToast('success', 'Success', 'Pending changes synced.');
        await loadData(true);
      } catch (error) {
        console.error('Sync error:', error);
        showToast('error', 'Error', 'Failed to sync pending changes.');
      } finally {
        if (isMounted.current) setSyncModalVisible(false);
      }
    };

    const unsubscribe = NetInfo.addEventListener(state => {
      if (isMounted.current && state.isConnected) syncPendingChanges();
    });
    return () => unsubscribe();
  }, [teamMemberId, teamMemberUserId]);

  const handleSearch = (query) => {
    setSearchQuery(query);
    setPage(1);
    filterEquipment(query, selectedFilter);
  };

  const filterEquipment = (query, filterType) => {
    let filtered = firebaseEquipment;

    if (selectedCountry && selectedCountry !== 'Both Countries') {
      filtered = filtered.filter(item => item.country === selectedCountry);
    }

    if (query) {
      filtered = filtered.filter(item => {
        switch (filterType) {
          case 'Created Date':
            return new Date(item.createdAt).toLocaleDateString().toLowerCase().includes(query.toLowerCase());
          case 'Equipment Number':
            return (item.equipmentNumber || '').toLowerCase().includes(query.toLowerCase());
          case 'Equipment Name':
            return (item.equipmentName || '').toLowerCase().includes(query.toLowerCase());
          case 'Equipment Categories':
            return (item.mainCategory || '').toLowerCase().includes(query.toLowerCase());
          case 'Meter Reading':
            return `${item.meterReading || ''}`.toLowerCase().includes(query.toLowerCase());
          default:
            return (
              (item.equipmentName || '').toLowerCase().includes(query.toLowerCase()) ||
              (item.equipmentNumber || '').toLowerCase().includes(query.toLowerCase()) ||
              (item.mainCategory || '').toLowerCase().includes(query.toLowerCase())
            );
        }
      });
    }

    if (filtered.length === 0 && query) {
      showToast('info', 'No Results', 'No equipment matches your search.');
    }

    setFilteredEquipment(filtered);
    setDisplayedEquipment(filtered.slice(0, INITIAL_ITEMS_TO_LOAD));
    setHasMore(filtered.length > INITIAL_ITEMS_TO_LOAD);
    setPage(1);
  };

  const handleFilterSelect = (filter) => {
    setSelectedFilter(filter);
    setFilterModalVisible(false);
    filterEquipment(searchQuery, filter);
  };

  const handleRefresh = async () => {
    setIsRefreshing(true);
    setPage(1);
    setHasMore(true);
    await loadData(true);
  };

  const handleScroll = ({ nativeEvent }) => {
    const { layoutMeasurement, contentOffset, contentSize } = nativeEvent;
    if (layoutMeasurement.height + contentOffset.y >= contentSize.height - 20 && hasMore && !isLoadingMore) {
      setIsLoadingMore(true);
      setPage(prev => prev + 1);
    }
  };

  useEffect(() => {
    const startIndex = (page - 1) * ITEMS_PER_PAGE;
    const endIndex = startIndex + ITEMS_PER_PAGE;
    setDisplayedEquipment(filteredEquipment.slice(0, INITIAL_ITEMS_TO_LOAD + endIndex));
    setHasMore(filteredEquipment.length > INITIAL_ITEMS_TO_LOAD + endIndex);
    setIsLoadingMore(false);
  }, [page, filteredEquipment]);

  const handleActionPress = (item) => {
    setSelectedEquipment(item);
    setActionModalVisible(true);
    setNewStatus(item.status || 'Available');
  };

  const handleStatusChange = async () => {
    if (!selectedEquipment || !(await checkLocalCredentials())) {
      setSyncModalVisible(false);
      setStatusModalVisible(false);
      setActionModalVisible(false);
      return;
    }

    const updatedEquipment = { ...selectedEquipment, status: newStatus, updatedAt: new Date().toISOString() };
    setSyncModalVisible(true);

    try {
      const isOnline = await checkNetworkStatus();
      let updatedFirebaseList = firebaseEquipment.map(item =>
        item.id === updatedEquipment.id ? updatedEquipment : item
      ).sort((a, b) => {
        if (a.teamMemberId === teamMemberId && b.teamMemberId !== teamMemberId) return -1;
        if (a.teamMemberId !== teamMemberId && b.teamMemberId === teamMemberId) return 1;
        const dateA = new Date(a.updatedAt || a.createdAt);
        const dateB = new Date(b.updatedAt || b.createdAt);
        return dateB - dateA;
      });

      if (isMounted.current) {
        setFirebaseEquipment(updatedFirebaseList);
        setFilteredEquipment(updatedFirebaseList);
        setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
      }

      if (isOnline) {
        const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
        const globalSnapshot = await withRetry(() => globalRef.once('value'));
        let globalData = globalSnapshot.val() || [];
        const globalIndex = globalData.findIndex(g => g.id === updatedEquipment.id);
        if (globalIndex !== -1) globalData[globalIndex] = updatedEquipment;
        else globalData.push(updatedEquipment);
        await withRetry(() => globalRef.set(globalData));

        const teamMemberRef = database().ref(`/GlobalTeamMemberInformation/${teamMemberId}/equipment`);
        const teamMemberSnapshot = await withRetry(() => teamMemberRef.once('value'));
        let teamMemberData = teamMemberSnapshot.val() || [];
        const teamMemberIndex = teamMemberData.findIndex(t => t.id === updatedEquipment.id);
        if (teamMemberIndex !== -1) teamMemberData[teamMemberIndex] = updatedEquipment;
        else teamMemberData.push(updatedEquipment);
        await withRetry(() => teamMemberRef.set(teamMemberData));

        await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
        showToast('success', 'Success', 'Status updated.');
      } else {
        updatedFirebaseList = updatedFirebaseList.map(item =>
          item.id === updatedEquipment.id ? { ...item, pendingSync: true } : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
        showToast('info', 'Offline', 'Status updated locally. Will sync when online.');
        if (isMounted.current) {
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
        }
      }
    } catch (error) {
      console.error('Status update error:', error);
      let errorMessage = 'Failed to update status.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (!isOnline) {
        errorMessage = 'Offline: Status updated locally, will sync when online.';
        const updatedFirebaseList = firebaseEquipment.map(item =>
          item.id === updatedEquipment.id ? { ...updatedEquipment, pendingSync: true } : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
        if (isMounted.current) {
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
        }
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setStatusModalVisible(false);
        setActionModalVisible(false);
      }
    }
  };

  const handleDelete = async () => {
    if (!selectedEquipment || selectedEquipment.teamMemberId !== teamMemberId) {
      showToast('error', 'Error', 'You can only delete equipment you created.');
      setShowDeleteConfirm(false);
      setActionModalVisible(false);
      return;
    }

    if (!(await checkLocalCredentials())) {
      setSyncModalVisible(false);
      setShowDeleteConfirm(false);
      setActionModalVisible(false);
      return;
    }

    setSyncModalVisible(true);

    try {
      const isOnline = await checkNetworkStatus();
      let updatedFirebaseList = firebaseEquipment.filter(item => item.id !== selectedEquipment.id);
      if (isMounted.current) {
        setFirebaseEquipment(updatedFirebaseList);
        setFilteredEquipment(updatedFirebaseList);
        setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
      }

      if (isOnline) {
        const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
        const globalSnapshot = await withRetry(() => globalRef.once('value'));
        let globalData = globalSnapshot.val() || [];
        globalData = globalData.filter(g => g.id !== selectedEquipment.id);
        await withRetry(() => globalRef.set(globalData));

        const teamMemberRef = database().ref(`/GlobalTeamMemberInformation/${teamMemberId}/equipment`);
        const teamMemberSnapshot = await withRetry(() => teamMemberRef.once('value'));
        let teamMemberData = teamMemberSnapshot.val() || [];
        teamMemberData = teamMemberData.filter(t => t.id !== selectedEquipment.id);
        await withRetry(() => teamMemberRef.set(teamMemberData));

        await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
        showToast('success', 'Success', 'Equipment deleted.');
      } else {
        updatedFirebaseList = firebaseEquipment.map(item =>
          item.id === selectedEquipment.id ? { ...item, pendingSync: true, pendingDelete: true } : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
        showToast('info', 'Offline', 'Marked for deletion locally. Will sync when online.');
        if (isMounted.current) {
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
        }
      }
    } catch (error) {
      console.error('Delete error:', error);
      let errorMessage = 'Failed to delete equipment.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (!isOnline) {
        errorMessage = 'Offline: Marked for deletion locally, will sync when online.';
        const updatedFirebaseList = firebaseEquipment.map(item =>
          item.id === selectedEquipment.id ? { ...item, pendingSync: true, pendingDelete: true } : item
        );
        await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
        if (isMounted.current) {
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
        }
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setShowDeleteConfirm(false);
        setActionModalVisible(false);
      }
    }
  };

  const getStatusColor = (status) => ({
    Available: '#2ecc71',
    'Maintenance Required': '#f1c40f',
    'In Maintenance': '#e67e22',
    Breakdown: '#e74c3c',
    Deploy: '#3498db',
  }[status] || '#666');

  const getDynamicFieldLabels = (mainCategory) => ({
    'Light Vehicles': { field1: 'Vehicle Number', field2: 'Registration Number', field3: 'VIN Number' },
    'Heavy Vehicles': { field1: 'Vehicle Number', field2: 'Registration Number', field3: 'VIN Number' },
    Rigs: { field1: 'Rig Type', field2: 'Chassis Number', field3: 'Serial Number' },
    'Heavy Plant Equipment': { field1: 'Equipment Number', field2: 'Equipment Name', field3: 'VIN Number' },
    'Auxiliary Equipment': { field1: 'Type', field2: 'Model', field3: 'Serial Number' },
    Lathes: { field1: 'Type', field2: 'Model', field3: 'Serial Number' },
    Trailers: { field1: 'Type', field2: 'Chassis Number', field3: 'Chassis Number' },
  }[mainCategory] || { field1: 'Field 1', field2: 'Field 2', field3: 'Field 3' });

  const renderAdditionalFields = (item) => {
    const fields = [];
    const { field1, field2, field3 } = getDynamicFieldLabels(item.mainCategory);
    const fieldMap = [
      { label: field1, key: field1.toLowerCase().replace(/\s/g, '') },
      { label: field2, key: field2.toLowerCase().replace(/\s/g, '') },
      { label: field3, key: field3.toLowerCase().replace(/\s/g, '') },
    ];
    fieldMap.forEach(({ label, key }) => {
      if (item[key] && item[key] !== 'N/A') fields.push({ label, value: item[key] });
    });
    return fields.map((field, index) => (
      <View key={index} style={styles.equipmentDetailItem}>
        <Text style={styles.detailLabel}>{field.label}</Text>
        <Text style={styles.detailValue}>{field.value}</Text>
      </View>
    ));
  };

  return (
    <View style={styles.container}>
      <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={24} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Equipments</Text>
          </View>
          <TouchableOpacity onPress={handleRefresh}>
            <Image source={require('../../../Images/adminhome/equp/cloud_1.png')} style={styles.icons12} />
          </TouchableOpacity>
        </View>
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder={`Search${selectedFilter ? ' by ' + selectedFilter : '...'}`}
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={20} color="#888" style={styles.searchIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton} onPress={() => setFilterModalVisible(true)}>
            <MaterialIcons name="filter-list" size={24} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {isLoadingInitial ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#3481BC" />
          <Text style={styles.loadingText}>Loading Equipment...</Text>
        </View>
      ) : (
        <ScrollView style={styles.mainContent} onScroll={handleScroll} scrollEventThrottle={16}>
          {displayedEquipment.length > 0 ? displayedEquipment.map(item => (
            <View key={item.id} style={styles.equipmentRow}>
              <View style={styles.equipmentInfo}>
                <View style={styles.equipmentDetailItemheader}>
                  <Text style={styles.detailLabelheader}>{item.equipmentName || 'Unnamed Equipment'}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Category</Text>
                  <Text style={styles.detailValue}>{item.mainCategory || 'N/A'}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Model</Text>
                  <Text style={styles.detailValue}>{item.model || 'N/A'}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Meter Reading</Text>
                  <Text style={styles.detailValue}>{item.meterReading ? `${item.meterReading} ${item.meterUnit}` : 'N/A'}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Created Date</Text>
                  <Text style={styles.detailValue}>{new Date(item.createdAt).toLocaleDateString()}</Text>
                </View>
                {renderAdditionalFields(item)}
                <View style={styles.equipmentDetailItemNoBorder}>
                  <Text style={[styles.detailLabelstatuss, { color: getStatusColor(item.status) }]}>{item.status || 'Available'}</Text>
                </View>
              </View>
              <TouchableOpacity style={styles.menuButton} onPress={() => handleActionPress(item)}>
                <MaterialIcons name="more-vert" size={24} color="#666" />
              </TouchableOpacity>
            </View>
          )) : (
            <Text style={styles.placeholderText}>{searchQuery ? 'No matching equipment found' : 'No equipment available'}</Text>
          )}
          {isLoadingMore && <ActivityIndicator size="large" color="#3481BC" style={{ marginVertical: 20 }} />}
        </ScrollView>
      )}

      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.mapButtonGradient}>
          <TouchableOpacity style={styles.mapButton}>
            <Image source={require('../../../Images/adminhome/equp/mapview_1.png')} style={styles.mapIcon} />
            <Text style={styles.addButtonTextmap}>Map View</Text>
          </TouchableOpacity>
        </LinearGradient>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.addButtonGradient}>
          <TouchableOpacity style={styles.addButton} onPress={() => navigation.navigate('TeamMemberEqAdd')}>
            <Ionicons name="add" size={24} color="#fff" />
            <Text style={styles.addButtonText}>Add Equipment</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      <Modal visible={actionModalVisible} transparent animationType="slide" onRequestClose={() => setActionModalVisible(false)}>
        <View style={styles.modalOverlay}>
          <View style={styles.actionModalContent}>
            <Text style={styles.actionModalTitle}>Actions</Text>
            <TouchableOpacity style={styles.actionItem} onPress={() => { setActionModalVisible(false); navigation.navigate('InsreportHeader', { equipment: selectedEquipment }); }}>
              <Text>Start Inspection</Text>
            </TouchableOpacity>
            <TouchableOpacity style={styles.actionItem} onPress={() => { setActionModalVisible(false); navigation.navigate('EquipmentOverview', { equipment: selectedEquipment }); }}>
              <Text>Equipment Overview</Text>
            </TouchableOpacity>
            <TouchableOpacity style={styles.actionItem} onPress={() => { setActionModalVisible(false); setStatusModalVisible(true); }}>
              <Text>Change Status</Text>
            </TouchableOpacity>
            <TouchableOpacity style={styles.actionItem} onPress={() => { setActionModalVisible(false); navigation.navigate('Add_equipment_update', { equipment: selectedEquipment }); }}>
              <Text>Update</Text>
            </TouchableOpacity>
            {selectedEquipment?.teamMemberId === teamMemberId && (
              <TouchableOpacity style={styles.actionItem} onPress={() => { setActionModalVisible(false); setShowDeleteConfirm(true); }}>
                <Text>Delete</Text>
              </TouchableOpacity>
            )}
            <TouchableOpacity style={styles.modalCloseButton} onPress={() => setActionModalVisible(false)}>
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal visible={statusModalVisible} transparent animationType="slide" onRequestClose={() => setStatusModalVisible(false)}>
        <View style={styles.modalOverlay}>
          <View style={styles.statusModalContent}>
            <LinearGradient colors={['#3481BC', '#025383']} style={styles.statusModalHeader}>
              <Text style={styles.statusModalTitle}>Change Status</Text>
            </LinearGradient>
            <View style={styles.statusDropdownWrapper}>
              <Dropdown
                style={styles.dropdown}
                containerStyle={styles.dropdownContainer}
                data={statusItems}
                labelField="label"
                valueField="value"
                placeholder="Select Status"
                value={newStatus}
                onChange={item => setNewStatus(item.value)}
                placeholderStyle={ensureFontSizeNumber(styles.dropdownPlaceholder)}
                selectedTextStyle={ensureFontSizeNumber(styles.dropdownSelectedText)}
                itemTextStyle={ensureFontSizeNumber(styles.dropdownItemText)}
                maxHeight={200}
              />
            </View>
            <TouchableOpacity style={styles.applyButton} onPress={handleStatusChange}>
              <Text style={styles.applyButtonText}>Apply</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal visible={filterModalVisible} transparent animationType="fade" onRequestClose={() => setFilterModalVisible(false)}>
        <View style={styles.modalOverlay}>
          <View style={styles.filterModalContent}>
            <Text style={styles.filterModalTitle}>Filter By</Text>
            {['Created Date', 'Equipment Number', 'Equipment Name', 'Equipment Categories', 'Meter Reading'].map(filter => (
              <TouchableOpacity key={filter} style={styles.filterItem} onPress={() => handleFilterSelect(filter)}>
                <Text style={styles.filterItemText}>{filter}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity style={styles.modalCloseButton} onPress={() => setFilterModalVisible(false)}>
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image source={require('../../../Images/adminlogin/sync.png')} style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]} />
              <Text style={styles.syncTextadd}>Processing Equipment Data</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={showDeleteConfirm} transparent animationType="fade" onRequestClose={() => setShowDeleteConfirm(false)}>
        <View style={styles.modalOverlay}>
          <View style={styles.deleteConfirmModalContent}>
            <Text style={styles.deleteConfirmTitle}>Confirm Delete</Text>
            <Text style={styles.deleteConfirmText}>Are you sure you want to delete this equipment?</Text>
            <View style={styles.deleteConfirmButtonContainer}>
              <TouchableOpacity style={[styles.deleteConfirmButton, { backgroundColor: '#666' }]} onPress={() => setShowDeleteConfirm(false)}>
                <Text style={styles.deleteConfirmButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.deleteConfirmButton, { backgroundColor: '#3481BC' }]} onPress={handleDelete}>
                <Text style={styles.deleteConfirmButtonText}>OK</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingTop: StatusBar.currentHeight || 44,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: 20,
    fontWeight: 'bold',
    marginLeft: 10,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 15,
    marginTop: 10,
  },
  searchWrapper: {
    flexDirection: 'row',
    flex: 1,
    backgroundColor: '#f0f0f0',
    borderRadius: 35,
    paddingHorizontal: 10,
    alignItems: 'center',
  },
  searchInput: {
    flex: 1,
    height: 40,
    fontSize: 16,
    color: '#000',
    paddingVertical: 0,
  },
  searchIcon: {
    marginLeft: 5,
  },
  filterButton: {
    marginLeft: 10,
    padding: 5,
  },
  mainContent: {
    flex: 1,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: 10,
    fontSize: 16,
    color: '#333',
  },
  placeholderText: {
    fontSize: 18,
    color: '#888',
    textAlign: 'center',
    marginTop: 20,
  },
  equipmentRow: {
    marginHorizontal: 15,
    marginVertical: '3%',
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#eee',
  },
  equipmentInfo: {
    flex: 1,
  },
  equipmentDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingHorizontal: '5%',
    paddingRight: '10%',
  },
  equipmentDetailItemNoBorder: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    paddingLeft: '5%',
  },
  equipmentDetailItemheader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: 5,
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: 3,
  },
  detailLabelstatuss: {
    fontSize: 14,
    fontWeight: '400',
    width: '60%',
    marginVertical: 3,
  },
  detailLabelheader: {
    fontSize: 18,
    color: '#015185',
    fontWeight: '600',
    width: '100%',
  },
  detailValue: {
    fontSize: 16,
    color: '#015185',
    width: '60%',
    textAlign: 'right',
    fontWeight: '500',
  },
  menuButton: {
    position: 'absolute',
    right: 10,
    bottom: 1,
    padding: 5,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  mapButtonGradient: {
    borderRadius: 5,
    overflow: 'hidden',
    marginBottom: 10,
  },
  mapButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 20,
  },
  mapIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  addButtonGradient: {
    borderRadius: 5,
    overflow: 'hidden',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 20,
  },
  addButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 5,
  },
  addButtonTextmap: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  actionModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
  },
  actionModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 15,
    color: '#333',
  },
  actionItem: {
    padding: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  modalCloseButton: {
    marginTop: 15,
    alignItems: 'center',
  },
  modalCloseText: {
    color: '#3481BC',
    fontSize: 16,
    fontWeight: 'bold',
  },
  statusModalContent: {
    width: '80%',
    height: 220,
    backgroundColor: '#fff',
    borderRadius: 10,
    overflow: 'hidden',
    alignSelf: 'center',
  },
  statusModalHeader: {
    padding: 10,
  },
  statusModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#fff',
    height: 40,
    textAlign: 'center',
    lineHeight: 40,
  },
  statusDropdownWrapper: {
    paddingHorizontal: 15,
    paddingVertical: 10,
    zIndex: 1000,
    alignItems: 'center',
  },
  dropdown: {
    backgroundColor: '#f0f0f0',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#ddd',
    height: 40,
    paddingHorizontal: 10,
    width: '100%',
  },
  dropdownContainer: {
    width: '65%',
    backgroundColor: '#fff',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#ddd',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  dropdownPlaceholder: {
    fontSize: 14,
    color: '#888',
  },
  dropdownSelectedText: {
    fontSize: 14,
    color: '#333',
  },
  dropdownItemText: {
    fontSize: 14,
    color: '#333',
  },
  applyButton: {
    backgroundColor: '#3481BC',
    padding: 8,
    borderRadius: 5,
    alignItems: 'center',
    marginHorizontal: 15,
    marginVertical: 10,
  },
  applyButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  filterModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 10,
  },
  filterModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 15,
    color: '#333',
  },
  filterItem: {
    padding: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  filterItemText: {
    fontSize: 16,
    color: '#333',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  icons12: {
    width: 24,
    height: 24,
  },
  deleteConfirmModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  deleteConfirmTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 15,
    color: '#333',
  },
  deleteConfirmText: {
    fontSize: 16,
    color: '#666',
    marginBottom: 20,
    textAlign: 'center',
  },
  deleteConfirmButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  deleteConfirmButton: {
    flex: 1,
    padding: 10,
    borderRadius: 5,
    marginHorizontal: 5,
    alignItems: 'center',
  },
  deleteConfirmButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export default TeamMemberEquipHome;