import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  FlatList,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';
import Modal from 'react-native-modal';

const { width, height } = Dimensions.get('window');

const TeamHRpreviousAppli = () => {
  const navigation = useNavigation();
  const [applications, setApplications] = useState([]);
  const [selectedApplication, setSelectedApplication] = useState(null);
  const [isModalVisible, setIsModalVisible] = useState(false);

  // Helper function to show toast messages
  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Helper function to format dates
  const formatDate = (isoString) => {
    if (!isoString) return 'N/A';
    const date = new Date(isoString);
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  // Fetch past applications from Firebase
  useEffect(() => {
    const fetchApplications = async () => {
      try {
        const teamMemberId = await AsyncStorage.getItem('teamMemberId');
        if (teamMemberId) {
          const applicationsRef = database().ref(`/GlobalTeamMemberInformation/${teamMemberId}/HrmoduleTeam`);
          applicationsRef.on('value', (snapshot) => {
            const data = snapshot.val();
            if (data) {
              // Convert object to array with application IDs
              const applicationsArray = Object.keys(data).map((key) => ({
                id: key,
                ...data[key],
              }));
              // Sort by submission date (newest first)
              applicationsArray.sort((a, b) => new Date(b.submittedAt) - new Date(a.submittedAt));
              setApplications(applicationsArray);
            } else {
              setApplications([]);
              showToast('info', 'Info', 'No past applications found.');
            }
          });

          // Cleanup listener on unmount
          return () => applicationsRef.off('value');
        } else {
          showToast('error', 'Error', 'Team Member ID not found in storage.');
        }
      } catch (error) {
        console.error('Error fetching past applications:', error);
        showToast('error', 'Error', 'Failed to load past applications.');
      }
    };

    fetchApplications();
  }, []);

  // Get application type label
  const getApplicationTypeLabel = (value) => {
    const applicationTypes = {
      leave_application: 'Leave Application',
      funeral_grant_form: 'Funeral Grant Form',
      salary_increase_application: 'Salary Advance Application',
      application_for_appeal: 'Application for Appeal',
    };
    return applicationTypes[value] || 'Unknown Application';
  };

  // Render application details based on type
  const renderApplicationDetails = (application) => {
    switch (application.applicationType) {
      case 'leave_application':
        return (
          <>
            <Text style={styles.detailText}>
              Leave Period: {application.leaveStartDate || 'N/A'} to {application.leaveEndDate || 'N/A'}
            </Text>
            <Text style={styles.detailText}>
              Reason: {application.reasonOfLeave || 'N/A'}
            </Text>
            <Text style={styles.detailText}>
              Report Date: {application.reportDate || 'N/A'}
            </Text>
          </>
        );
      case 'funeral_grant_form':
        return (
          <>
            <Text style={styles.detailText}>
              Deceased Name: {application.deceasedName || 'N/A'}
            </Text>
            <Text style={styles.detailText}>
              Relationship: {application.deceasedRelationship || 'N/A'}
            </Text>
            <Text style={styles.detailText}>
              Funeral Period: {application.funeralFromDate || 'N/A'} to {application.funeralToDate || 'N/A'}
            </Text>
          </>
        );
      case 'salary_increase_application':
        return (
          <>
            <Text style={styles.detailText}>
              Amount: K{application.salaryAdvanceAmount || 'N/A'}
            </Text>
            <Text style={styles.detailText}>
              Department: {application.salaryAdvanceDepartment || 'N/A'}
            </Text>
            <Text style={styles.detailText}>
              Recovery Period: {application.selectedRecoveryPeriod?.replace('_', ' ') || 'N/A'}
            </Text>
          </>
        );
      case 'application_for_appeal':
        return (
          <>
            <Text style={styles.detailText}>
              Offence: {application.appealOffence || 'N/A'}
            </Text>
            <Text style={styles.detailText}>
              Sanction: {application.appealSanction || 'N/A'}
            </Text>
            <Text style={styles.detailText}>
              Appeal Date: {application.appealDateOfAppeal || 'N/A'}
            </Text>
          </>
        );
      default:
        return <Text style={styles.detailText}>No additional details available.</Text>;
    }
  };

  // Handle three dots click
  const handleMorePress = (application) => {
    setSelectedApplication(application);
    setIsModalVisible(true);
  };

  // Render each application card
  const renderApplicationItem = ({ item }) => (
    <View style={styles.applicationCard}>
      <View style={styles.cardHeader}>
        <Text style={styles.applicationTitle}>
          {getApplicationTypeLabel(item.applicationType)}
        </Text>
        <View style={styles.headerRight}>
          <Text
            style={[
              styles.statusText,
              {
                color:
                  item.status === 'Approved'
                    ? '#4CAF50'
                    : item.status === 'Rejected'
                    ? '#F44336'
                    : '#888',
              },
            ]}
          >
            {item.status || 'Pending'}
          </Text>
          <TouchableOpacity onPress={() => handleMorePress(item)}>
            <MaterialIcons name="more-vert" size={width * 0.06} color="#666" />
          </TouchableOpacity>
        </View>
      </View>
      <View style={styles.cardBody}>
        <Text style={styles.detailText}>
          Applicant: {item.fullName || 'N/A'}
        </Text>
        {renderApplicationDetails(item)}
        <Text style={styles.detailText}>
          Submitted: {formatDate(item.submittedAt)}
        </Text>
      </View>
    </View>
  );

  // Render the modal content
  const renderModalContent = () => {
    if (!selectedApplication) return null;

    return (
      <View style={styles.modalContent}>
        <View style={styles.modalHeader}>
          <Text style={styles.modalTitle}>
            {getApplicationTypeLabel(selectedApplication.applicationType)}
          </Text>
          <TouchableOpacity onPress={() => setIsModalVisible(false)}>
            <MaterialIcons name="close" size={width * 0.06} color="#666" />
          </TouchableOpacity>
        </View>
        <View style={styles.modalBody}>
          <View style={styles.modalDetailRow}>
            <Text style={styles.modalLabel}>Status:</Text>
            <Text
              style={[
                styles.modalValue,
                {
                  color:
                    selectedApplication.status === 'Approved'
                      ? '#4CAF50'
                      : selectedApplication.status === 'Rejected'
                      ? '#F44336'
                      : '#888',
                },
              ]}
            >
              {selectedApplication.status || 'Pending'}
            </Text>
          </View>
          {selectedApplication.status === 'Approved' || selectedApplication.status === 'Rejected' ? (
            <>
              <View style={styles.modalDetailRow}>
                <Text style={styles.modalLabel}>Approved By:</Text>
                <Text style={styles.modalValue}>
                  {selectedApplication.approverName || 'N/A'}
                </Text>
              </View>
              <View style={styles.modalDetailRow}>
                <Text style={styles.modalLabel}>Approved Date:</Text>
                <Text style={styles.modalValue}>
                  {formatDate(selectedApplication.approvalDate)}
                </Text>
              </View>
              {selectedApplication.status === 'Rejected' && (
                <View style={styles.modalDetailRow}>
                  <Text style={styles.modalLabel}>Reason for Rejection:</Text>
                  <Text style={styles.modalValue}>
                    {selectedApplication.rejectionReason || 'N/A'}
                  </Text>
                </View>
              )}
            </>
          ) : (
            <Text style={styles.modalValue}>This application is still pending approval.</Text>
          )}
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <TouchableOpacity style={styles.backButton} onPress={() => navigation.goBack()}>
          <MaterialIcons name="arrow-back-ios" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>Past Applications</Text>
        <View style={styles.menuButtonPlaceholder} />
      </LinearGradient>

      {/* Main Content */}
      <View style={styles.content}>
        {applications.length === 0 ? (
          <View style={styles.emptyContainer}>
            <Text style={styles.emptyText}>No past applications found.</Text>
          </View>
        ) : (
          <FlatList
            data={applications}
            renderItem={renderApplicationItem}
            keyExtractor={(item) => item.id}
            contentContainerStyle={styles.listContent}
            showsVerticalScrollIndicator={false}
          />
        )}
      </View>

      {/* Bottom-to-Up Modal */}
      <Modal
        isVisible={isModalVisible}
        onBackdropPress={() => setIsModalVisible(false)}
        onSwipeComplete={() => setIsModalVisible(false)}
        swipeDirection="down"
        style={styles.modal}
        animationIn="slideInUp"
        animationOut="slideOutDown"
        backdropOpacity={0.3}
      >
        {renderModalContent()}
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f8f8',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? height * 0.05 : height * 0.02,
    paddingBottom: height * 0.04,
  },
  backButton: {
    padding: width * 0.02,
    top: '46%',
  },
  headerTitle: {
    fontSize: width * 0.05,
    color: '#fff',
    fontWeight: 'bold',
    top: '46%',
  },
  menuButtonPlaceholder: {
    width: width * 0.06,
    padding: width * 0.02,
  },
  content: {
    flex: 1,
    padding: width * 0.04,
  },
  listContent: {
    paddingBottom: width * 0.04,
  },
  applicationCard: {
    backgroundColor: '#fff',
    borderRadius: width * 0.03,
    marginBottom: width * 0.04,
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: width * 0.04,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  applicationTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
  },
  statusText: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    marginRight: width * 0.02,
  },
  cardBody: {
    padding: width * 0.04,
  },
  detailText: {
    fontSize: width * 0.038,
    color: '#333',
    marginBottom: width * 0.015,
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  emptyText: {
    fontSize: width * 0.045,
    color: '#666',
    textAlign: 'center',
  },
  // Modal Styles
  modal: {
    justifyContent: 'flex-end',
    margin: 0,
  },
  modalContent: {
    backgroundColor: '#fff',
    borderTopLeftRadius: width * 0.05,
    borderTopRightRadius: width * 0.05,
    padding: width * 0.05,
    maxHeight: height * 0.5,
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    paddingBottom: width * 0.03,
    marginBottom: width * 0.03,
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
  },
  modalBody: {
    paddingVertical: width * 0.02,
  },
  modalDetailRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.03,
  },
  modalLabel: {
    fontSize: width * 0.04,
    color: '#888',
    fontWeight: 'bold',
    width: '40%',
  },
  modalValue: {
    fontSize: width * 0.04,
    color: '#333',
    flex: 1,
    flexWrap: 'wrap',
  },
});

export default TeamHRpreviousAppli;