import React, { useRef, useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  Alert,
  Modal,
  Animated,
  Image,
  Platform,
} from 'react-native';
import SignatureCanvas from 'react-native-signature-canvas';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';

const { height, width } = Dimensions.get('window');

// SyncModal Component - Reusing from InspecSignature
const SyncModal = ({ visible }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <Text style={styles.modalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.modalTextadd}>Saving Signature</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const SignaturePadScreen = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { onSave, signatureTitle, initialSignature } = route.params || {};

  const signatureRef = useRef(null);
  const [signatureData, setSignatureData] = useState(initialSignature || null);
  const [showSyncModal, setShowSyncModal] = useState(false);

  useEffect(() => {
    if (signatureData) {
      // If there's an initial signature, load it into the canvas
      // This might require a specific method from react-native-signature-canvas
      // signatureRef.current?.drawSignature(signatureData);
    }
  }, [initialSignature]);

  const handleClear = () => {
    if (signatureRef.current) {
      signatureRef.current.clearSignature();
      setSignatureData(null);
      console.log('Signature cleared');
    } else {
      Alert.alert('Error', 'Signature canvas is not ready.');
      console.error('Signature canvas reference is null');
    }
  };

  const handleSave = () => {
    if (!signatureRef.current) {
      Alert.alert('Error', 'Signature canvas is not initialized.');
      console.error('Signature canvas reference is null');
      return;
    }

    setShowSyncModal(true);
    signatureRef.current.readSignature(); // This will trigger onOK
  };

  const handleOK = (signature) => {
    if (signature) {
      console.log('Signature captured via onOK:', signature);
      const processedSignature = signature.startsWith('data:image')
        ? signature
        : `data:image/png;base64,${signature}`;
      setSignatureData(processedSignature);

      if (typeof onSave === 'function') {
        onSave(processedSignature);
      } else {
        console.warn('onSave callback not provided to SignaturePadScreen.');
      }

      // Clear the canvas after saving if desired, or keep it for review
      // signatureRef.current.clearSignature();
      // setSignatureData(null);

      setShowSyncModal(false);
      navigation.goBack(); // Go back after saving

    } else {
      setShowSyncModal(false);
      Alert.alert('Warning', 'No signature detected. Please sign before saving.');
      console.log('No signature detected via onOK');
    }
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <TouchableOpacity style={styles.backButton} onPress={() => navigation.goBack()}>
          <MaterialIcons name="arrow-back-ios" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>{signatureTitle || 'Draw Signature'}</Text>
        <View style={styles.menuButtonPlaceholder} />{/* Placeholder for alignment */}
      </LinearGradient>

      <View style={styles.content}>
        <Text style={styles.label}>Draw your signature below</Text>
        <View style={styles.signatureContainer}>
          <SignatureCanvas
            ref={signatureRef}
            onOK={handleOK}
            descriptionText="Sign here"
            backgroundColor="#F5F5F5"
            penColor="#000000"
            canvasStyle={styles.signatureCanvas}
          />
        </View>

        <View style={styles.buttonContainer}>
          <TouchableOpacity onPress={handleClear} style={styles.clearButton}>
            <Text style={styles.clearButtonText}>Clear</Text>
          </TouchableOpacity>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.saveButton}>
            <TouchableOpacity
              onPress={handleSave}
              style={styles.saveButtonTouchable}
            >
              <Text style={styles.saveButtonText}>Save</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      </View>

      {/* Sync Modal */}
      <SyncModal visible={showSyncModal} />
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? height * 0.05 : height * 0.02,
    paddingBottom: height * 0.02,
  },
  backButton: {
    padding: width * 0.02,
  },
  headerTitle: {
    fontSize: width * 0.05,
    color: '#fff',
    fontWeight: 'bold',
  },
  menuButtonPlaceholder: {
    width: width * 0.06 + width * 0.04, // icon size + padding
  },
  content: { flex: 1, alignItems: 'center', padding: 20 },
  label: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 20,
    alignSelf: 'flex-start',
  },
  signatureContainer: {
    width: width * 0.9,
    height: height * 0.3,
    backgroundColor: '#F5F5F5',
    borderRadius: 10,
    overflow: 'hidden',
    marginBottom: 30,
  },
  signatureCanvas: { width: '100%', height: '100%' },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: width * 0.9,
  },
  clearButton: {
    width: '48%',
    padding: 12,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#025383',
    alignItems: 'center',
    justifyContent: 'center',
  },
  clearButtonText: { color: '#025383', fontSize: 16, fontWeight: '600' },
  saveButton: { width: '48%', borderRadius: 5, overflow: 'hidden' },
  saveButtonTouchable: { padding: 12, alignItems: 'center', justifyContent: 'center' },
  saveButtonText: { color: '#fff', fontSize: 16, fontWeight: '600' },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  modalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default SignaturePadScreen; 