import {
  View,
  Text,
  Image,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  Animated,
  ActivityIndicator,
  Platform,
  ToastAndroid,
  Modal,
  ScrollView,
  KeyboardAvoidingView,
} from 'react-native';
import React, { useState, useRef, useEffect } from 'react';
import { useNavigation } from '@react-navigation/native';
import LinearGradient from 'react-native-linear-gradient';
import { TextInput } from 'react-native-paper';
import Toast from 'react-native-toast-message';
import { MongoAPI } from '../ConfigMongo/MongoDBConfig';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import notifee, { AndroidImportance } from '@notifee/react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import messaging from '@react-native-firebase/messaging';
import DeviceInfo from 'react-native-device-info';
import PushNotificationAPI from '../ConfigMongo/PushNotificationConfig';

// Get FCM Token, save to MongoDB ONLY, send notification via MongoDB + Firebase
const getFCMTokenAndUpdate = async (userId, userName, userEmail) => {
  try {
    console.log('🔔 Requesting FCM permission...');
    
    // Request permission for notifications
    const authStatus = await messaging().requestPermission();
    const enabled =
      authStatus === messaging.AuthorizationStatus.AUTHORIZED ||
      authStatus === messaging.AuthorizationStatus.PROVISIONAL;

    if (enabled) {
      console.log('✅ FCM Authorization status:', authStatus);
      
      // Get FCM token from Firebase Messaging
      const fcmToken = await messaging().getToken();
      console.log('📱 FCM Token obtained:', fcmToken);

      // Get device info
      const deviceId = await DeviceInfo.getUniqueId();
      const platform = Platform.OS;

      // Save FCM token to MongoDB ONLY (no Firebase storage)
      console.log('💾 Saving FCM token to MongoDB...');
      const mongoResponse = await MongoAPI.updateFCMToken(userId, fcmToken, deviceId, platform);
      
      if (mongoResponse.success) {
        console.log('✅ FCM token saved to MongoDB');
        console.log(`📊 User now has ${mongoResponse.totalDevices || 1} device(s) registered`);
      } else {
        console.warn('⚠️ Failed to save FCM token to MongoDB:', mongoResponse.error);
        return; // Don't send notification if token save failed
      }

      // Display local notification immediately (using notifee)
      try {
        console.log('📱 Displaying local welcome notification...');
        
        // Create notification channel for Android
        const channelId = await notifee.createChannel({
          id: 'welcome',
          name: 'Welcome Notifications',
          importance: 4, // High importance
          sound: 'default',
        });

        // Display the notification
        await notifee.displayNotification({
          title: `Welcome ${userName}! 👋`,
          body: `You've successfully logged in to Titan Drilling. Have a productive day!`,
          android: {
            channelId,
            importance: 4,
            pressAction: {
              id: 'default',
            },
            sound: 'default',
            smallIcon: 'ic_launcher',
          },
          ios: {
            sound: 'default',
            foregroundPresentationOptions: {
              alert: true,
              badge: true,
              sound: true,
            },
          },
          data: {
            type: 'welcome',
            action: 'view_home',
            timestamp: new Date().toISOString(),
          },
        });

        console.log('✅ Local notification displayed successfully');
      } catch (localNotifError) {
        console.error('❌ Error displaying local notification:', localNotifError);
      }

      // Also send via backend (for other devices)
      try {
        console.log('📤 Sending push notification via backend to other devices...');
        
        const notificationResult = await PushNotificationAPI.sendWelcomeNotification(userId, userName);
        
        if (notificationResult.success) {
          console.log(`✅ Backend notification sent to ${notificationResult.successCount} device(s)`);
          console.log(`📊 Total tokens: ${notificationResult.totalTokens}`);
        } else {
          console.error('❌ Backend notification failed:', notificationResult.error);
        }
      } catch (notifError) {
        console.error('❌ Error sending backend notification:', notifError);
        // Don't fail login if notification fails
      }
    } else {
      console.log('⚠️ FCM permission not granted');
    }
  } catch (error) {
    console.error('❌ Error with FCM token:', error);
    // Don't fail login if FCM fails
  }
};


// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style, isLoading }) => {
  return (
    <LinearGradient colors={['#015185', '#3481BC']} style={[styles.gradientButton, style]}>
      <TouchableOpacity style={styles.buttonTouchable} onPress={onPress} activeOpacity={0.8} disabled={isLoading}>
        {isLoading ? (
          <ActivityIndicator size="small" color="#fff" />
        ) : (
          <Text style={styles.googleButtonText}>{title}</Text>
        )}
      </TouchableOpacity>
    </LinearGradient>
  );
};

// Custom Animated Input Component
const AnimatedInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  keyboardType,
  autoCapitalize,
  leftIcon,
  rightIcon,
  onRightIconPress,
  hasError,
  onBlur,
  leftIconName,
  rightIconName,
}) => {
  const [isFocused, setIsFocused] = useState(false);
  const animatedLabel = useRef(new Animated.Value(value ? 1 : 0)).current;
  const animatedBorderWidth = useRef(new Animated.Value(1)).current;
  const animatedBorderColor = useRef(new Animated.Value(0)).current;

  React.useEffect(() => {
    Animated.timing(animatedLabel, {
      toValue: isFocused || value ? 1 : 0,
      duration: 200,
      useNativeDriver: false,
    }).start();

    Animated.timing(animatedBorderWidth, {
      toValue: isFocused ? 2 : 1,
      duration: 200,
      useNativeDriver: false,
    }).start();

    Animated.timing(animatedBorderColor, {
      toValue: hasError ? 2 : isFocused || value ? 1 : 0,
      duration: 200,
      useNativeDriver: false,
    }).start();
  }, [isFocused, value, hasError]);

  const labelStyle = {
    position: 'absolute',
    left: 45,
    top: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [18, -8],
    }),
    fontSize: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [16, 12],
    }),
    color: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: ['#888', '#015185'],
    }),
    backgroundColor: '#fff',
    paddingHorizontal: 4,
    zIndex: 2,
  };

  const borderColor = animatedBorderColor.interpolate({
    inputRange: [0, 1, 2],
    outputRange: ['#ddd', '#015185', '#ff0000'],
  });

  return (
    <View style={styles.inputContainer}>
      <Animated.View
        style={[
          styles.inputGradient,
          {
            borderWidth: animatedBorderWidth,
            borderColor: borderColor,
          },
        ]}
      >
        <Animated.Text style={labelStyle}>{label}</Animated.Text>
        <TextInput
          mode="outlined"
          value={value}
          onChangeText={onChangeText}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          style={styles.input}
          onFocus={() => setIsFocused(true)}
          onBlur={() => {
            setIsFocused(false);
            onBlur && onBlur();
          }}
          left={
            leftIconName && (
              <TextInput.Icon
                icon={() => <MaterialIcons name={leftIconName} size={24} color="#015185" />}
              />
            )
          }
          right={
            rightIconName && (
              <TextInput.Icon
                icon={() => (
                  <TouchableOpacity onPress={onRightIconPress}>
                    <MaterialIcons name={rightIconName} size={24} color="#015185" />
                  </TouchableOpacity>
                )}
              />
            )
          }
          outlineStyle={{
            borderWidth: 0,
            borderRadius: 8,
          }}
          theme={{
            colors: {
              primary: 'transparent',
              text: '#333',
              placeholder: '#888',
            },
          }}
        />
      </Animated.View>
    </View>
  );
};

// SyncModal Component
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  React.useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[
                styles.syncIcon,
                { transform: [{ rotate: rotateInterpolate }] },
              ]}
            />
            <Text style={styles.syncModalTextadd}>Processing Login</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages based on platform
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.show(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

export default function TeamMemberLogin() {
  const navigation = useNavigation();
  const [identifier, setIdentifier] = useState('');
  const [password, setPassword] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [identifierError, setIdentifierError] = useState(false);
  const [identifierErrorMessage, setIdentifierErrorMessage] = useState('');
  const [passwordError, setPasswordError] = useState(false);
  const [passwordErrorMessage, setPasswordErrorMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  // Validate identifier (email or employee ID)
  const validateIdentifier = (identifier) => {
    const isEmpty = !identifier.trim();
    const isEmail = /\S+@\S+\.\S+/.test(identifier);
    const isEmployeeId = identifier.length >= 1; // Any text or number with at least 1 character
    const isTooLong = identifier.length > 100; // Increased limit for flexibility

    setIdentifierError(isEmpty || (!isEmail && !isEmployeeId) || isTooLong);
    if (isEmpty) {
      setIdentifierErrorMessage('Email/Employee ID is required');
    } else if (!isEmail && !isEmployeeId) {
      setIdentifierErrorMessage('Enter a valid email address or employee ID');
    } else if (isTooLong) {
      setIdentifierErrorMessage('Input cannot exceed 100 characters');
    } else {
      setIdentifierErrorMessage('');
    }
    return !(isEmpty || (!isEmail && !isEmployeeId) || isTooLong);
  };

  // Validate password
  const validatePassword = (password) => {
    const isEmpty = !password.trim();
    setPasswordError(isEmpty);
    if (isEmpty) {
      setPasswordErrorMessage('Password is required');
    } else {
      setPasswordErrorMessage('');
    }
    return !isEmpty;
  };

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      }
      return true;
    } catch (error) {
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

// Find user by email or employee ID using MongoDB
const findUserByEmailOrEmployeeId = async (input) => {
  try {
    console.log('🔍 Searching for user in MongoDB:', input);
    
    // Use search endpoint to find user without password verification
    const searchResult = await MongoAPI.searchUser(input, input);
    
    if (searchResult.success && searchResult.user) {
      console.log(`✅ User found: ${searchResult.user.name} (${searchResult.user.email})`);
      return { 
        userId: searchResult.user._id, 
        userData: searchResult.user 
      };
    } else {
      console.log('❌ User not found with:', input);
      return null;
    }
    
  } catch (error) {
    console.error('❌ Error finding user:', error);
    return null;
  }
};

  // Handle Login
  const handleLogin = async (input, pwd) => {
    const isIdentifierValid = validateIdentifier(input);
    const isPasswordValid = validatePassword(pwd);
    if (!isIdentifierValid || !isPasswordValid) {
      showToast('error', 'Validation Error', 'Please correct the errors in the form');
      return;
    }
    setIsLoading(true);
    setSyncModalVisible(true);
    try {
      if (!(await checkNetworkStatus())) {
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // First, try to find user by email or employee ID
      const userInfo = await findUserByEmailOrEmployeeId(input);
      
      if (!userInfo) {
        showToast('error', 'Login Error', 'User not found with provided email/employee ID');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      const { userId, userData } = userInfo;

      // Check if user is blocked
      if (userData.isBlocked === true) {
        showToast('error', 'Account Blocked', 'Your profile is blocked. Please contact with administrator.');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // Check user role - ProAdmin cannot login here
      if (userData.role === 'ProAdmin') {
        showToast('error', 'Access Denied', 'ProAdmin cannot login here');
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // Verify password using MongoDB fast login
      const loginResult = await MongoAPI.fastLogin(input, pwd, input);
      
      if (!loginResult.success) {
        const message =
          loginResult.error ||
          loginResult.message ||
          'Username and password not matched';
        showToast('error', 'Login Error', message);
        setIsLoading(false);
        setSyncModalVisible(false);
        return;
      }

      // Store session data immediately for fast navigation
      await Promise.all([
        AsyncStorage.setItem('userUid', userId),
        AsyncStorage.setItem('userEmail', userData.email),
        AsyncStorage.setItem('Loginsuccess-01', 'true'),
        AsyncStorage.setItem('TeamMemberLogined', 'true'),
        // Store complete user data for home screen
        AsyncStorage.setItem('userData', JSON.stringify(userData)),
      ]);

      // Remove any admin session flags
      await AsyncStorage.removeItem('ProAdminLogined').catch(() => {});

      // Update last login in MongoDB (background task)
      setTimeout(() => {
        (async () => {
          try {
            console.log('🔄 Updating last login in MongoDB...');
            await MongoAPI.updateUser(userId, {
              loginStatus: true,
              lastLogin: new Date().toISOString(),
            });
            console.log('✅ Last login updated in MongoDB');
          } catch (updateError) {
            console.warn('⚠️ Failed to update last login:', updateError);
          }
        })();
      }, 100);

      // Get FCM token and send welcome notification (background task)
      setTimeout(() => {
        getFCMTokenAndUpdate(userId, userData.name, userData.email);
      }, 500);

      // Navigate immediately for a faster perceived login
      showToast('success', 'Success', `Welcome ${userData.name}! Logged in successfully`);
      setIsLoading(false);
      setSyncModalVisible(false);
      navigation.replace('MainApp');
    } catch (error) {
      console.error('Login error:', error);
      const isCredentialError =
        error?.status === 401 ||
        error?.status === 404 ||
        /invalid password/i.test(error?.message || '') ||
        /user not found/i.test(error?.message || '');

      const friendlyMessage = isCredentialError
        ? 'Username and password not matched'
        : error?.message || 'An unexpected error occurred';

      showToast('error', 'Login Error', friendlyMessage);
      setIsLoading(false);
      setSyncModalVisible(false);
    }
  };

  const toggleShowPassword = () => {
    setShowPassword(!showPassword);
  };

  return (
    <SafeAreaView style={styles.container}>
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} />
      
      {/* Back Arrow Header */}
      <View style={styles.headerContainer}>
        <TouchableOpacity 
          style={styles.backButton}
          onPress={() => navigation.goBack()}
          activeOpacity={0.7}
        >
          <MaterialIcons 
            name="arrow-back" 
            size={24} 
            color="#015185" 
          />
        </TouchableOpacity>
      </View>
      
      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.keyboardAvoidingView}
      >
        <ScrollView contentContainerStyle={styles.scrollContent}>
          <View style={styles.contentContainer}>
            {/* Logo and Login Text */}
            <View style={styles.logoContainer}>
              <Image
                source={require('../../Images/adminlogin/logo.png')}
                style={styles.logo}
              />
              <Text style={styles.loginText}>Let’s Get You Started With Titan Drilling</Text>
            </View>
            {/* Form */}
            <View style={styles.formContainer}>
              {/* Animated Identifier Input */}
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Email/Employee ID"
                  value={identifier}
                  onChangeText={(text) => {
                    setIdentifier(text);
                    validateIdentifier(text);
                  }}
                  keyboardType="email-address"
                  autoCapitalize="none"
                  leftIconName="person"
                  hasError={identifierError}
                  onBlur={() => validateIdentifier(identifier)}
                />
                {identifierError && (
                  <Text style={styles.errorText}>{identifierErrorMessage}</Text>
                )}
              </View>
              {/* Animated Password Input */}
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  label="Password"
                  value={password}
                  onChangeText={(text) => {
                    setPassword(text);
                    validatePassword(text);
                  }}
                  secureTextEntry={!showPassword}
                  autoCapitalize="none"
                  leftIconName="lock"
                  rightIconName={showPassword ? 'visibility' : 'visibility-off'}
                  onRightIconPress={toggleShowPassword}
                  hasError={passwordError}
                  onBlur={() => validatePassword(password)}
                />
                {passwordError && (
                  <Text style={styles.errorText}>{passwordErrorMessage}</Text>
                )}
              </View>
              {/* Login Button */}
              <GradientButton
                title="Login"
                onPress={() => handleLogin(identifier, password)}
                isLoading={isLoading}
              />
            </View>
          </View>
        </ScrollView>
      </KeyboardAvoidingView>
      <Toast />
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
    paddingHorizontal: 10,
  },
  headerContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 10,
    paddingTop: Platform.OS === 'ios' ? 10 : 20,
  },
  backButton: {
    width: 40,
    height: 40,
    justifyContent: 'center',
    alignItems: 'center',
    borderRadius: 20,
    backgroundColor: 'rgba(1, 81, 133, 0.1)',
  },
  keyboardAvoidingView: {
    flex: 1,
  },
  scrollContent: {
    flexGrow: 1,
    justifyContent: 'center',
  },
  contentContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: 20,
  },
  logo: {
    width: '80%',
    height: 100,
    resizeMode: 'contain',
  },
  loginText: {
    fontSize: 16,
    color: '#015185',
    marginTop: 10,
    textAlign: 'center',
  },
  formContainer: {
    width: '100%',
  },
  inputContainer: {
    marginBottom: 30,
    marginHorizontal: 10,
  },
  inputGradient: {
    borderRadius: 8,
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  input: {
    backgroundColor: '#fff',
    height: 56,
    borderRadius: 8,
    fontSize: 16,
  },
  gradientButton: {
    borderRadius: 6,
    marginTop: 10,
    marginBottom: 10,
    width: '95%',
    justifyContent: 'center',
    alignSelf: 'center',
  },
  buttonTouchable: {
    alignItems: 'center',
    justifyContent: 'center',
    height: 48,
    borderRadius: 6,
    width: '100%',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  networkErrorBanner: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    backgroundColor: '#ff0000',
    paddingVertical: 10,
    alignItems: 'center',
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  errorText: {
    color: 'red',
    fontSize: 14,
    marginLeft: 10,
    marginTop: 4,
    marginBottom: 8,
    position: 'absolute',
    bottom: 0,
  },
  mainemailpass: {
    position: 'relative',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});