import React, { useCallback, useEffect, useRef, useState } from 'react';
import { Modal, View, Text, TouchableOpacity, StyleSheet, Platform, PermissionsAndroid, BackHandler } from 'react-native';
import VoiceToText, { VoiceToTextEvents } from '@appcitor/react-native-voice-to-text';
import LottieView from 'lottie-react-native';

const VoiceInputModal = ({
  visible,
  onClose = () => {},
  onAppendText = () => {},
  onError = () => {},
}) => {
  const [isListening, setIsListening] = useState(false);
  const [partialText, setPartialText] = useState('');
  const [volume, setVolume] = useState(0);
  const [error, setError] = useState(null);
  const [micPermissionGranted, setMicPermissionGranted] = useState(Platform.OS !== 'android');
  const [isVoiceModuleAvailable, setIsVoiceModuleAvailable] = useState(false);
  const [isRecognitionAvailable, setIsRecognitionAvailable] = useState(true);
  const [ignoreVoiceEvents, setIgnoreVoiceEvents] = useState(false);

  const lottieRef = useRef(null);
  const listenersRef = useRef([]);
  const isMountedRef = useRef(true);
  const isCleaningUpRef = useRef(false);
  const isListeningRef = useRef(false);
  const hasStoppedRecordingRef = useRef(false);
  const hasReceivedFinalResultsRef = useRef(false);
  const isRestartingRef = useRef(false);
  const lastFinalTextRef = useRef('');
  const cleanupTimeoutRef = useRef(null);
  const partialTextRef = useRef('');

  const reportError = useCallback(
    (message) => {
      setError(message);
      onError(message);
    },
    [onError]
  );

  const checkMicrophonePermission = useCallback(async () => {
    if (Platform.OS !== 'android') {
      return true;
    }
    try {
      const granted = await PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.RECORD_AUDIO);
      setMicPermissionGranted(granted);
      return granted;
    } catch (err) {
      reportError('Failed to check microphone permissions.');
      return false;
    }
  }, [reportError]);

  const requestMicrophonePermission = useCallback(async () => {
    if (Platform.OS !== 'android') {
      return true;
    }
    try {
      const granted = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.RECORD_AUDIO,
        {
          title: 'Microphone Permission',
          message: 'This app needs microphone access for speech recognition.',
          buttonPositive: 'OK',
        }
      );
      const isGranted = granted === PermissionsAndroid.RESULTS.GRANTED;
      setMicPermissionGranted(isGranted);
      if (!isGranted) {
        reportError('Microphone permission denied. Please enable it in app settings.');
      }
      return isGranted;
    } catch (err) {
      reportError('Failed to request microphone permission. Please try again.');
      return false;
    }
  }, [reportError]);

  const safeStopRecording = useCallback(
    async (shouldAddPartialText = false) => {
      if (hasStoppedRecordingRef.current || !isListeningRef.current) {
        return;
      }

      try {
        hasStoppedRecordingRef.current = true;
        isListeningRef.current = false;
        setIgnoreVoiceEvents(true);

        if (VoiceToText && isVoiceModuleAvailable) {
          const stopMethods = [
            () => VoiceToText.stopListening && VoiceToText.stopListening(),
            () => VoiceToText.stop && VoiceToText.stop(),
            () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
            () => VoiceToText.cancel && VoiceToText.cancel(),
            () => VoiceToText.destroy && VoiceToText.destroy(),
          ];

          for (const stopMethod of stopMethods) {
            try {
              if (typeof stopMethod === 'function') {
                await stopMethod();
                break;
              }
            } catch (err) {
              // Try next stop method
            }
          }
        }

        if (lottieRef.current) {
          lottieRef.current.pause();
          lottieRef.current.reset();
        }

        if (isMountedRef.current && !isCleaningUpRef.current) {
          setIsListening(false);
          if (cleanupTimeoutRef.current) {
            clearTimeout(cleanupTimeoutRef.current);
          }
          await new Promise((resolve) => {
            cleanupTimeoutRef.current = setTimeout(() => {
              if (!isMountedRef.current || isCleaningUpRef.current) {
                cleanupTimeoutRef.current = null;
                resolve();
                return;
              }

              if (shouldAddPartialText && !hasReceivedFinalResultsRef.current) {
                const cleanedPartial = partialTextRef.current.trim();
                if (cleanedPartial && lastFinalTextRef.current !== cleanedPartial) {
                  lastFinalTextRef.current = cleanedPartial;
                  onAppendText(cleanedPartial);
                }
              }

              setPartialText('');
              partialTextRef.current = '';
              setVolume(0);
              cleanupTimeoutRef.current = null;
              onClose();
              resolve();
            }, 500);
          });
        }
      } catch (err) {
        console.warn('Error in safeStopRecording:', err);
      }
    },
    [isVoiceModuleAvailable, onAppendText]
  );

  const cancelRecording = useCallback(async () => {
    await safeStopRecording(false);
  }, [safeStopRecording]);

  const finishRecording = useCallback(async () => {
    await safeStopRecording(true);
  }, [safeStopRecording]);

  const handleResults = useCallback(
    (event) => {
      if (!isMountedRef.current || isCleaningUpRef.current || ignoreVoiceEvents) {
        return;
      }

      hasReceivedFinalResultsRef.current = true;

      try {
        let payload = event?.value ?? event?.results ?? event?.text ?? event;
        let finalText = '';

        if (Array.isArray(payload)) {
          finalText = payload.join(' ');
        } else if (typeof payload === 'string') {
          finalText = payload;
        } else if (payload && typeof payload === 'object') {
          const extracted = Object.values(payload)
            .flatMap((v) => (Array.isArray(v) ? v : [v]))
            .filter((v) => typeof v === 'string');
          finalText = extracted.length ? extracted.join(' ') : '';
        } else {
          finalText = String(payload ?? '');
        }

        const trimmed = finalText.trim();
        if (!trimmed) {
          return;
        }

        if (lastFinalTextRef.current === trimmed) {
          setPartialText('');
          return;
        }

        lastFinalTextRef.current = trimmed;
        onAppendText(trimmed);
        partialTextRef.current = '';
        setPartialText('');
      } catch (err) {
        console.warn('handleResults parse error', err, event);
      }
    },
    [ignoreVoiceEvents, onAppendText]
  );

  const handlePartialResults = useCallback(
    (event) => {
      if (!isMountedRef.current || isCleaningUpRef.current || ignoreVoiceEvents) {
        return;
      }

      try {
        let payload = event?.value ?? event?.partialResults ?? event?.text ?? event;
        let partial = '';

        if (Array.isArray(payload)) {
          partial = payload.join(' ');
        } else if (typeof payload === 'string') {
          partial = payload;
        } else if (payload && typeof payload === 'object') {
          const extracted = Object.values(payload)
            .flatMap((v) => (Array.isArray(v) ? v : [v]))
            .filter((v) => typeof v === 'string');
          partial = extracted.length ? extracted.join(' ') : '';
        }

        partialTextRef.current = partial;
        setPartialText(partial);
      } catch (err) {
        console.warn('handlePartialResults parse error', err, event);
      }
    },
    [ignoreVoiceEvents]
  );

  const handleVolumeChange = useCallback(
    (event) => {
      if (!isMountedRef.current || isCleaningUpRef.current || ignoreVoiceEvents) {
        return;
      }
      setVolume(event?.value || 0);
    },
    [ignoreVoiceEvents]
  );

  const handleError = useCallback(
    (event) => {
      if (!isMountedRef.current || isCleaningUpRef.current) {
        return;
      }
      const message = event?.error?.message || event?.message || JSON.stringify(event);
      reportError(`Speech recognition error: ${message}`);
      safeStopRecording(false);
    },
    [reportError, safeStopRecording]
  );

  const handleStart = useCallback(() => {
    if (isMountedRef.current && !isCleaningUpRef.current) {
      isListeningRef.current = true;
      hasStoppedRecordingRef.current = false;
      hasReceivedFinalResultsRef.current = false;
      isRestartingRef.current = false;
      lastFinalTextRef.current = '';
      setIsListening(true);
      setPartialText('');
      setIgnoreVoiceEvents(false);
      if (lottieRef.current) {
        lottieRef.current.play();
      }
    }
  }, []);

  const handleEnd = useCallback(() => {
    if (
      isMountedRef.current &&
      !isCleaningUpRef.current &&
      !hasStoppedRecordingRef.current &&
      isListeningRef.current &&
      !isRestartingRef.current
    ) {
      isRestartingRef.current = true;

      setTimeout(async () => {
        if (
          isMountedRef.current &&
          !isCleaningUpRef.current &&
          !hasStoppedRecordingRef.current &&
          isListeningRef.current
        ) {
          try {
            if (VoiceToText && typeof VoiceToText.startListening === 'function') {
              await VoiceToText.startListening();
            } else if (VoiceToText && typeof VoiceToText.start === 'function') {
              await VoiceToText.start();
            }
          } catch (err) {
            console.warn('Failed to auto-restart recording:', err);
          }
        }
        isRestartingRef.current = false;
      }, 200);
    }
  }, []);

  const startRecording = useCallback(async () => {
    if (!isVoiceModuleAvailable) {
      reportError(
        'Voice module is not available. Ensure @appcitor/react-native-voice-to-text is installed and linked.'
      );
      return;
    }

    if (Platform.OS === 'android' && !micPermissionGranted) {
      const granted = await requestMicrophonePermission();
      if (!granted) {
        return;
      }
    }

    setError(null);
    setIgnoreVoiceEvents(false);
    setPartialText('');
    setVolume(0);
    lastFinalTextRef.current = '';

    if (Platform.OS === 'ios' && !isRecognitionAvailable) {
      reportError(
        'Speech recognition not available on this iOS device. Check Info.plist permissions and test on a real device.'
      );
      return;
    }

    try {
      if (typeof VoiceToText.setRecognitionLanguage === 'function') {
        await VoiceToText.setRecognitionLanguage('en-US');
      }

      hasStoppedRecordingRef.current = false;
      hasReceivedFinalResultsRef.current = false;
      isRestartingRef.current = false;
      isListeningRef.current = true;

      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      } else {
        throw new Error('startListening not available on VoiceToText');
      }
    } catch (err) {
      const message = err?.message || JSON.stringify(err);
      if (Platform.OS === 'ios' && message.includes('IsFormatSampleRateAndChannelCountValid')) {
        reportError(
          'iOS audio format error. Test on a real device and ensure Info.plist includes NSMicrophoneUsageDescription and NSSpeechRecognitionUsageDescription.'
        );
      } else {
        reportError(`Failed to start recording: ${message}`);
      }
      setIsListening(false);
    }
  }, [isRecognitionAvailable, isVoiceModuleAvailable, micPermissionGranted, reportError, requestMicrophonePermission]);

  useEffect(() => {
    if (!VoiceToText || typeof VoiceToText.addEventListener !== 'function') {
      reportError(
        'VoiceToText module is unavailable. Ensure @appcitor/react-native-voice-to-text is installed and linked correctly.'
      );
      return;
    }

    setIsVoiceModuleAvailable(true);

    const resultsListener = VoiceToText.addEventListener(VoiceToTextEvents.RESULTS, handleResults);
    const partialListener = VoiceToText.addEventListener(
      VoiceToTextEvents.PARTIAL_RESULTS,
      handlePartialResults
    );
    const volumeListener = VoiceToText.addEventListener(VoiceToTextEvents.VOLUME_CHANGED, handleVolumeChange);
    const errorListener = VoiceToText.addEventListener(VoiceToTextEvents.ERROR, handleError);
    const startListener = VoiceToText.addEventListener(VoiceToTextEvents.START, handleStart);
    const endListener = VoiceToText.addEventListener(VoiceToTextEvents.END, handleEnd);

    listenersRef.current = [
      resultsListener,
      partialListener,
      volumeListener,
      errorListener,
      startListener,
      endListener,
    ];

    (async () => {
      try {
        if (typeof VoiceToText.isRecognitionAvailable === 'function') {
          const available = await VoiceToText.isRecognitionAvailable();
          setIsRecognitionAvailable(available === true);
          if (!available && Platform.OS === 'ios') {
            reportError(
              'Speech recognition not available on this iOS device. Check Info.plist permissions and test on a real device.'
            );
          }
        }
      } catch (err) {
        console.warn('isRecognitionAvailable check failed', err);
      }
    })();

    if (Platform.OS === 'android') {
      checkMicrophonePermission();
    }

    return () => {
      isMountedRef.current = false;
      isCleaningUpRef.current = true;

      if (cleanupTimeoutRef.current) {
        clearTimeout(cleanupTimeoutRef.current);
        cleanupTimeoutRef.current = null;
      }

      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }

      const stopRecordingImmediately = async () => {
        try {
          if (isListeningRef.current && VoiceToText && !hasStoppedRecordingRef.current) {
            hasStoppedRecordingRef.current = true;
            const stopMethods = [
              () => VoiceToText.stopListening && VoiceToText.stopListening(),
              () => VoiceToText.stop && VoiceToText.stop(),
              () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
              () => VoiceToText.cancel && VoiceToText.cancel(),
              () => VoiceToText.destroy && VoiceToText.destroy(),
            ];

            for (const stopMethod of stopMethods) {
              try {
                if (typeof stopMethod === 'function') {
                  await stopMethod();
                  break;
                }
              } catch (err) {
                // continue
              }
            }
          }
        } catch (err) {
          console.warn('Error during voice cleanup (non-fatal):', err.message);
        }
      };

      stopRecordingImmediately();
      listenersRef.current = [];
    };
  }, [checkMicrophonePermission, handleEnd, handleError, handlePartialResults, handleResults, handleStart, handleVolumeChange, reportError]);

  useEffect(() => {
    if (!visible) {
      return;
    }

    isMountedRef.current = true;
    isCleaningUpRef.current = false;
    hasStoppedRecordingRef.current = false;
    hasReceivedFinalResultsRef.current = false;
    setError(null);
    partialTextRef.current = '';
    setPartialText('');
    setVolume(0);
    setIgnoreVoiceEvents(false);
    startRecording();

    return () => {
      isCleaningUpRef.current = true;
      if (cleanupTimeoutRef.current) {
        clearTimeout(cleanupTimeoutRef.current);
        cleanupTimeoutRef.current = null;
      }
      const stopRecordingSilently = async () => {
        try {
          if (isListeningRef.current && VoiceToText && !hasStoppedRecordingRef.current) {
            hasStoppedRecordingRef.current = true;
            const stopMethods = [
              () => VoiceToText.stopListening && VoiceToText.stopListening(),
              () => VoiceToText.stop && VoiceToText.stop(),
              () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
              () => VoiceToText.cancel && VoiceToText.cancel(),
              () => VoiceToText.destroy && VoiceToText.destroy(),
            ];

            for (const stopMethod of stopMethods) {
              try {
                if (typeof stopMethod === 'function') {
                  await stopMethod();
                  break;
                }
              } catch (err) {
                // continue
              }
            }
          }
        } catch (err) {
          console.warn('Error during voice cleanup (non-fatal):', err.message);
        } finally {
          isListeningRef.current = false;
        }
      };

      stopRecordingSilently();
      setIsListening(false);
      setPartialText('');
      partialTextRef.current = '';
      setVolume(0);
      setIgnoreVoiceEvents(false);
    };
  }, [startRecording, visible]);

  useEffect(() => {
    const backHandler = BackHandler.addEventListener('hardwareBackPress', () => {
      if (visible && isListeningRef.current) {
        cancelRecording();
        return true;
      }
      return false;
    });

    return () => backHandler.remove();
  }, [cancelRecording, visible]);

  if (!visible) {
    return null;
  }

  return (
    <Modal visible={visible} transparent animationType="fade" onRequestClose={cancelRecording}>
      <View style={styles.overlay}>
        <View style={styles.content}>
          <View style={styles.lottieContainer}>
            <LottieView
              ref={lottieRef}
              source={require('../../../Images/micanimation.json')}
              autoPlay={false}
              loop
              style={styles.lottieAnimation}
              resizeMode="contain"
            />
          </View>

          <Text style={styles.partialText}>
            {partialText
              ? partialText
              : isListening
              ? 'Listening... Speak now'
              : 'Preparing...'}
          </Text>

          {volume > 0 && (
            <View style={styles.volumeIndicator}>
              <View style={[styles.volumeBar, { width: `${Math.min(volume * 100, 100)}%` }]} />
            </View>
          )}

          {error ? <Text style={styles.errorText}>{error}</Text> : null}

          <View style={styles.buttonRow}>
            <TouchableOpacity style={[styles.button, styles.cancelButton]} onPress={cancelRecording}>
              <Text style={[styles.buttonText, { color: '#015185' }]}>Cancel</Text>
            </TouchableOpacity>
            <TouchableOpacity style={styles.button} onPress={finishRecording}>
              <Text style={styles.buttonText}>Done</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const styles = StyleSheet.create({
  overlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.7)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  content: {
    width: '85%',
    maxWidth: 400,
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
    elevation: 8,
  },
  lottieContainer: {
    width: 200,
    height: 200,
    justifyContent: 'center',
    alignItems: 'center',
    marginBottom: 20,
  },
  lottieAnimation: {
    width: '100%',
    height: '100%',
  },
  partialText: {
    minHeight: 50,
    textAlign: 'center',
    marginBottom: 20,
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
    paddingHorizontal: 16,
  },
  volumeIndicator: {
    width: '80%',
    height: 4,
    backgroundColor: '#e0e0e0',
    borderRadius: 2,
    marginBottom: 16,
    overflow: 'hidden',
  },
  volumeBar: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 2,
  },
  buttonRow: {
    flexDirection: 'row',
    width: '100%',
    justifyContent: 'space-between',
    gap: 12,
  },
  button: {
    flex: 1,
    paddingVertical: 12,
    backgroundColor: '#015185',
    borderRadius: 8,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  cancelButton: {
    backgroundColor: '#fff',
    borderWidth: 2,
    borderColor: '#015185',
  },
  buttonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 16,
  },
  errorText: {
    color: '#d32f2f',
    marginBottom: 12,
  },
});

export default VoiceInputModal;

