import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  FlatList,
  Modal,
  Animated,
  ActivityIndicator,
  Platform,
  ScrollView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useIsFocused } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import Toast from 'react-native-toast-message';
import InspectionAPI from '../../../ConfigMongo/InspectionMongoConfig';
import JobCardAPI from '../../../ConfigMongo/JobCardMongoConfig';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, message = 'Loading Data' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const EquipmentstatusKpiHome = () => {
  const navigation = useNavigation();
  const isFocused = useIsFocused();
  const isMounted = useRef(true);

  // Date states - Default to last 14 days
  const [fromDate, setFromDate] = useState(() => {
    const date = new Date();
    date.setDate(date.getDate() - 14);
    date.setHours(0, 0, 0, 0);
    return date;
  });
  const [toDate, setToDate] = useState(() => {
    const date = new Date();
    date.setHours(23, 59, 59, 999);
    return date;
  });
  const [showFromDatePicker, setShowFromDatePicker] = useState(false);
  const [showToDatePicker, setShowToDatePicker] = useState(false);

  // Data states
  const [allEquipment, setAllEquipment] = useState([]);
  const [maintenanceEquipment, setMaintenanceEquipment] = useState([]);
  const [breakdownEquipment, setBreakdownEquipment] = useState([]);
  const [activeEquipment, setActiveEquipment] = useState([]);
  
  // UI states
  const [activeTab, setActiveTab] = useState('all'); // 'all', 'maintenance', 'breakdown', 'active'
  const [loading, setLoading] = useState(false);
  const [userInfoModalVisible, setUserInfoModalVisible] = useState(false);
  const [userInfo, setUserInfo] = useState({ countries: [], projects: [] });

  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const formatDate = (date) => {
    if (!date) return 'N/A';
    const d = new Date(date);
    return d.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    });
  };

  // Fetch user info
  const fetchUserInfo = async () => {
    try {
      const cachedUserData = await AsyncStorage.getItem('userData');
      if (cachedUserData) {
        const userData = JSON.parse(cachedUserData);
        setUserInfo({
          countries: Array.isArray(userData.countries) ? userData.countries : [],
          projects: Array.isArray(userData.projects) ? userData.projects : [],
        });
      }
    } catch (error) {
      console.error('Error fetching user info:', error);
    }
  };

  // Load all equipment data with tags
  const loadEquipmentData = async () => {
    if (!isMounted.current) return;
    
    try {
      setLoading(true);
      
      console.log('🔄 Loading equipment with activity tags...');

      // Fetch all data in parallel
      const [
        equipmentResponse,
        inspectionsResponse,
        jobCardsResponse,
        serviceSchedulesResponse,
        requestMaintenanceResponse,
      ] = await Promise.all([
        // Equipment
        fetch('https://api.titandrillingzm.com:6001/EquipmentsDataJSON/GlobalEquipmentJsonData'),
        // Inspections
        InspectionAPI.getAllReports({ page: 1, limit: 10000 }),
        // Job Cards
        JobCardAPI.getAllJobCards({ page: 1, limit: 10000 }),
        // Service Schedules
        fetch('https://api.titandrillingzm.com:6010/service-schedules'),
        // Request Maintenance
        fetch('https://api.titandrillingzm.com:6009/request-maintenance'),
      ]);

      // Parse responses
      const equipmentData = await equipmentResponse.json();
      const serviceSchedulesData = await serviceSchedulesResponse.json();
      const requestMaintenanceData = await requestMaintenanceResponse.json();

      const equipment = equipmentData.success ? equipmentData.data : [];
      const inspections = inspectionsResponse.success ? inspectionsResponse.data : [];
      const jobCards = jobCardsResponse.success ? jobCardsResponse.data : [];
      const serviceSchedules = serviceSchedulesData.success ? serviceSchedulesData.data : [];
      const maintenanceRequests = requestMaintenanceData.success ? requestMaintenanceData.data : [];

      console.log(`📊 Loaded: ${equipment.length} equipment, ${inspections.length} inspections, ${jobCards.length} job cards, ${serviceSchedules.length} schedules, ${maintenanceRequests.length} maintenance`);

      // Build equipment map with tags
      const equipmentMap = {};

      // Helper function to create or get equipment entry
      const getOrCreateEquipment = (equipmentData) => {
        if (!equipmentData) return null;

        const equipmentId = equipmentData._id || equipmentData.id;
        const equipmentNumber = equipmentData.equipmentNumber || equipmentData.vehicleNumber || equipmentId;
        const equipmentName = equipmentData.equipmentName || equipmentData.name || null;

        // Try to find existing equipment by multiple keys
        const possibleKeys = [
          equipmentNumber,
          equipmentId,
          equipmentName,
        ].filter(Boolean);

        for (const key of possibleKeys) {
          if (equipmentMap[key]) {
            // Update name if the existing entry has "Unknown Equipment"
            const existing = equipmentMap[key];
            if (existing.equipmentName === 'Unknown Equipment' && equipmentName) {
              existing.equipmentName = equipmentName;
            }
            return existing;
          }
        }

        // Create new equipment entry if not found
        const equipmentEntry = {
          ...equipmentData,
          id: equipmentId,
          equipmentNumber,
          equipmentName: equipmentName || `Equipment #${equipmentNumber}`,
          tags: [],
          inspections: [],
          jobCards: [],
          serviceSchedules: [],
          maintenanceRequests: [],
          incidents: [],
        };

        // Add to map with multiple keys for better matching
        equipmentMap[equipmentNumber] = equipmentEntry;
        if (equipmentId && equipmentId !== equipmentNumber) {
          equipmentMap[equipmentId] = equipmentEntry;
        }
        if (equipmentName && equipmentName !== equipmentNumber) {
          equipmentMap[equipmentName] = equipmentEntry;
        }

        console.log(`✅ Created equipment: ${equipmentEntry.equipmentName} (ID: ${equipmentId}, Number: ${equipmentNumber})`);

        return equipmentEntry;
      };

      // Initialize all equipment from main equipment collection
      equipment.forEach((eq, index) => {
        getOrCreateEquipment(eq);
      });
      
      console.log(`📊 Initialized ${equipment.length} equipment items from main collection`);

      // Link inspections to equipment - Create equipment if doesn't exist
      inspections.forEach(inspection => {
        const selectedEquipment = inspection.step1?.selectedEquipment;
        if (!selectedEquipment) return;

        // Get or create equipment from inspection data
        const matchedEquipment = getOrCreateEquipment(selectedEquipment);
        
        if (matchedEquipment) {
          matchedEquipment.inspections.push({
            id: inspection._id || inspection.id,
            requestNumber: inspection.step1?.requestNumber || 'N/A',
            status: inspection.status || 'draft',
            createdAt: inspection.createdAt,
            faultyItems: inspection.step3?.faultyItems || 0,
            inspector: inspection.step1?.inspector,
            country: inspection.step1?.country,
            project: inspection.step1?.project,
          });

          // Add inspection tag if status is not completed
          if (inspection.status !== 'completed') {
            matchedEquipment.tags.push({
              type: 'Inspection',
              status: inspection.status || 'draft',
              reference: inspection.step1?.requestNumber,
              color: '#3498db',
            });
          }
          
          console.log(`🔗 Linked inspection ${inspection.step1?.requestNumber} to equipment ${matchedEquipment.equipmentName}`);
        }
      });

      // Link job cards to equipment - Create equipment if doesn't exist
      jobCards.forEach(jobCard => {
        const equipmentData = jobCard.equipment;
        if (!equipmentData) return;

        // Get or create equipment from job card data
        const matchedEquipment = getOrCreateEquipment(equipmentData);
        
        if (matchedEquipment) {
          matchedEquipment.jobCards.push({
            id: jobCard._id || jobCard.id,
            requestNumber: jobCard.requestNumber || jobCard.title || 'N/A',
            status: jobCard.status || 'Hold',
            priority: jobCard.priority || 'Medium',
            createdAt: jobCard.createdAt,
            dueDate: jobCard.dueDate,
            workOrderType: jobCard.workOrderType,
            assignedTo: jobCard.assignTo?.length || 0,
          });

          // Add job card tag if not complete
          if (jobCard.status && jobCard.status.toLowerCase() !== 'complete') {
            matchedEquipment.tags.push({
              type: 'Job Card',
              status: jobCard.status,
              reference: jobCard.requestNumber || jobCard.title,
              color: '#00B894',
            });
          }
          
          console.log(`🔗 Linked job card ${jobCard.requestNumber || jobCard.title} to equipment ${matchedEquipment.equipmentName}`);
        }
      });

      // Link service schedules to equipment - Create equipment if doesn't exist
      serviceSchedules.forEach(schedule => {
        const equipmentDetails = schedule.equipmentDetails;
        if (!equipmentDetails) return;

        // Get or create equipment from service schedule data
        const matchedEquipment = getOrCreateEquipment(equipmentDetails);
        
        if (matchedEquipment) {
          matchedEquipment.serviceSchedules.push({
            id: schedule._id || schedule.id,
            serviceOrderNumber: schedule.serviceOrderNumber || 'N/A',
            status: schedule.status || 'pending',
            createdAt: schedule.createdAt,
            repeatType: schedule.repeatType,
            lastServiceDate: schedule.lastServiceDate,
          });

          // Add service schedule tag if active
          if (schedule.status && schedule.status.toLowerCase() !== 'completed') {
            matchedEquipment.tags.push({
              type: 'Service Schedule',
              status: schedule.status,
              reference: schedule.serviceOrderNumber,
              color: '#9C27B0',
            });
          }
          
          console.log(`🔗 Linked service schedule ${schedule.serviceOrderNumber} to equipment ${matchedEquipment.equipmentName}`);
        }
      });

      // Link maintenance requests to equipment - Create equipment if doesn't exist
      maintenanceRequests.forEach(request => {
        // Request maintenance has equipment object directly
        const equipmentDetails = request.equipment || request.equipmentDetails;
        if (!equipmentDetails) {
          console.log(`⚠️ No equipment data in maintenance request ${request.requestNumber}`);
          return;
        }

        // Get or create equipment from maintenance request data
        const matchedEquipment = getOrCreateEquipment(equipmentDetails);
        
        if (matchedEquipment) {
          // Only add if status is pending/active (not rejected/completed)
          const status = (request.status || 'Pending').toLowerCase();
          const isPending = status !== 'completed' && 
                           status !== 'rejected' && 
                           status !== 'cancelled' &&
                           status !== 'closed';
          
          matchedEquipment.maintenanceRequests.push({
            id: request._id || request.id,
            requestNumber: request.requestNumber || 'N/A',
            status: request.status || 'Pending',
            priority: request.priority || 'Medium',
            createdAt: request.createdAt,
            title: request.workOrderTitle || request.title || 'N/A',
            country: request.country,
            project: request.project,
            requestType: request.requestType,
            assignedTo: Array.isArray(request.assignTo) ? request.assignTo.length : 0,
            tasks: Array.isArray(request.tasks) ? request.tasks.length : 0,
            isPending,
          });

          // Add maintenance request tag ONLY if pending/active
          if (isPending) {
            matchedEquipment.tags.push({
              type: 'Maintenance Request',
              status: request.status,
              reference: request.requestNumber,
              color: '#FF9800',
            });
          }
          
          console.log(`🔗 Linked maintenance request ${request.requestNumber} (${request.status}) to equipment ${matchedEquipment.equipmentName}`);
        }
      });

      // Convert to array and deduplicate (since we stored same equipment under multiple keys)
      const equipmentArrayRaw = Object.values(equipmentMap);
      const uniqueEquipmentMap = new Map();
      
      equipmentArrayRaw.forEach(eq => {
        const key = eq.equipmentNumber; // Use equipment number as the unique key
        if (!uniqueEquipmentMap.has(key)) {
          uniqueEquipmentMap.set(key, eq);
        }
      });
      
      const equipmentArray = Array.from(uniqueEquipmentMap.values());

      console.log(`📊 Total unique equipment: ${equipmentArray.length}`);
      console.log(`📊 Equipment with inspections: ${equipmentArray.filter(e => e.inspections.length > 0).length}`);
      console.log(`📊 Equipment with job cards: ${equipmentArray.filter(e => e.jobCards.length > 0).length}`);
      console.log(`📊 Equipment with service schedules: ${equipmentArray.filter(e => e.serviceSchedules.length > 0).length}`);
      console.log(`📊 Equipment with maintenance requests: ${equipmentArray.filter(e => e.maintenanceRequests.length > 0).length}`);

      // Filter by date range and ensure equipment has activity
      const filteredEquipment = equipmentArray.filter(eq => {
        // Must have at least one activity
        const hasAnyActivity = eq.inspections.length > 0 || 
                              eq.jobCards.length > 0 || 
                              eq.serviceSchedules.length > 0 || 
                              eq.maintenanceRequests.length > 0;

        if (!hasAnyActivity) return false;

        // Check if has activity in date range
        const hasActivityInDateRange = [
          ...eq.inspections,
          ...eq.jobCards,
          ...eq.serviceSchedules,
          ...eq.maintenanceRequests,
        ].some(item => {
          if (!item.createdAt) return false;
          const itemDate = new Date(item.createdAt);
          return itemDate >= fromDate && itemDate <= toDate;
        });

        return hasActivityInDateRange;
      });

      console.log(`📅 Equipment with activity in date range: ${filteredEquipment.length}`);

      // Categorize equipment
      const maintenance = filteredEquipment.filter(eq => {
        // Equipment with "Maintenance" in status OR has pending maintenance requests/service schedules
        const hasMaintenanceStatus = eq.status && eq.status.toLowerCase().includes('maintenance');
        const hasPendingMaintenance = eq.maintenanceRequests.some(r => r.isPending);
        const hasActiveSchedule = eq.tags.some(tag => tag.type === 'Service Schedule');
        
        return hasMaintenanceStatus || hasPendingMaintenance || hasActiveSchedule;
      });

      const breakdown = filteredEquipment.filter(eq => 
        eq.status && (
          eq.status.toLowerCase().includes('breakdown') ||
          eq.status.toLowerCase().includes('down') ||
          eq.status.toLowerCase().includes('not available') ||
          eq.tags.some(tag => tag.status && tag.status.toLowerCase().includes('emergency'))
        )
      );

      const active = filteredEquipment.filter(eq => 
        eq.status && (
          eq.status.toLowerCase().includes('active') ||
          eq.status.toLowerCase().includes('available') ||
          eq.status.toLowerCase().includes('operational')
        )
      );

      setAllEquipment(filteredEquipment);
      setMaintenanceEquipment(maintenance);
      setBreakdownEquipment(breakdown);
      setActiveEquipment(active);

      console.log(`✅ Equipment categorized: ${filteredEquipment.length} total, ${maintenance.length} maintenance, ${breakdown.length} breakdown, ${active.length} active`);
    } catch (error) {
      console.error('Error loading equipment data:', error);
      showToast('error', 'Error', 'Failed to load equipment data');
    } finally {
      if (isMounted.current) {
        setLoading(false);
      }
    }
  };

  // Initial load
  useEffect(() => {
    if (isFocused) {
      fetchUserInfo();
      loadEquipmentData();
    }

    return () => {
      isMounted.current = false;
    };
  }, [isFocused]);

  // Reload when dates change
  useEffect(() => {
    loadEquipmentData();
  }, [fromDate, toDate]);

  // Render equipment item
  const renderEquipmentItem = ({ item }) => {
    const totalActivity = item.inspections.length + item.jobCards.length + 
                         item.serviceSchedules.length + item.maintenanceRequests.length;

  return (
      <View style={styles.equipmentCard}>
        <View style={styles.equipmentHeader}>
          <View style={styles.equipmentTitleRow}>
            <MaterialIcons name="precision-manufacturing" size={24} color="#0055A5" />
            <View style={{ flex: 1, marginLeft: 8 }}>
              <Text style={styles.equipmentName}>{item.equipmentName || 'Unknown Equipment'}</Text>
              <Text style={styles.equipmentNumber}>#{item.equipmentNumber || 'N/A'}</Text>
            </View>
          </View>
          
          {/* Equipment Status Badge */}
          <View style={[styles.equipmentStatusBadge, { 
            backgroundColor: item.status?.toLowerCase().includes('available') ? '#2ecc40' :
                           item.status?.toLowerCase().includes('maintenance') ? '#f39c12' :
                           item.status?.toLowerCase().includes('breakdown') ? '#e74c3c' : '#666'
          }]}>
            <Text style={styles.equipmentStatusText}>{item.status || 'Unknown'}</Text>
          </View>
        </View>

        <View style={styles.equipmentInfo}>
          <View style={styles.infoRow}>
            <MaterialIcons name="category" size={16} color="#666" />
            <Text style={styles.infoText}>Category: {item.mainCategory || 'N/A'}</Text>
          </View>
          <View style={styles.infoRow}>
            <MaterialIcons name="build" size={16} color="#666" />
            <Text style={styles.infoText}>Model: {item.model || 'N/A'}</Text>
          </View>
          <View style={styles.infoRow}>
            <MaterialIcons name="location-on" size={16} color="#666" />
            <Text style={styles.infoText}>Location: {item.country || 'N/A'}</Text>
          </View>
          {item.meterReading && (
            <View style={styles.infoRow}>
              <MaterialIcons name="speed" size={16} color="#666" />
              <Text style={styles.infoText}>Meter Reading: {item.meterReading}</Text>
            </View>
          )}
        </View>

        {/* Active Tags Section - Shows what's currently running on this equipment */}
        {item.tags.length > 0 && (
          <>
            <View style={styles.divider} />
            <View style={styles.tagsSection}>
              <Text style={styles.tagsSectionTitle}>
                <MaterialIcons name="label" size={16} color="#0055A5" /> Active Tags ({item.tags.length})
              </Text>
              <View style={styles.tagsContainer}>
                {item.tags.map((tag, index) => (
                  <View key={index} style={[styles.tag, { backgroundColor: tag.color }]}>
                    <Text style={styles.tagType}>{tag.type}</Text>
                    <Text style={styles.tagReference}>{tag.reference}</Text>
                  </View>
                ))}
              </View>
            </View>
          </>
        )}

        {/* Activity Summary */}
        <View style={styles.divider} />
        <View style={styles.activitySummary}>
          <Text style={styles.activityTitle}>Activity Summary</Text>
          <View style={styles.activityRow}>
            <View style={styles.activityItem}>
              <MaterialIcons name="assignment" size={18} color="#3498db" />
              <Text style={styles.activityCount}>{item.inspections.length}</Text>
              <Text style={styles.activityLabel}>Inspections</Text>
            </View>
            <View style={styles.activityItem}>
              <MaterialIcons name="build" size={18} color="#00B894" />
              <Text style={styles.activityCount}>{item.jobCards.length}</Text>
              <Text style={styles.activityLabel}>Job Cards</Text>
            </View>
            <View style={styles.activityItem}>
              <MaterialIcons name="event" size={18} color="#9C27B0" />
              <Text style={styles.activityCount}>{item.serviceSchedules.length}</Text>
              <Text style={styles.activityLabel}>Schedules</Text>
            </View>
            <View style={styles.activityItem}>
              <MaterialIcons name="report-problem" size={18} color="#FF9800" />
              <Text style={styles.activityCount}>{item.maintenanceRequests.filter(r => r.isPending).length}</Text>
              <Text style={styles.activityLabel}>Pending Maint.</Text>
            </View>
          </View>
        </View>

        {/* Detailed Activity - Expandable sections */}
        {item.inspections.length > 0 && (
          <View style={styles.activityDetailsSection}>
            <Text style={styles.activityDetailsTitle}>
              <MaterialIcons name="assignment" size={16} color="#3498db" /> Inspections ({item.inspections.length})
            </Text>
            {item.inspections.slice(0, 3).map((inspection, idx) => (
              <View key={idx} style={[styles.activityDetailCard, { borderLeftColor: '#3498db' }]}>
                <View style={styles.activityDetailHeader}>
                  <Text style={styles.activityDetailText}>• {inspection.requestNumber}</Text>
                  <View style={[styles.activityStatusBadge, { 
                    backgroundColor: inspection.status === 'completed' ? '#2ecc40' : '#3498db'
                  }]}>
                    <Text style={styles.activityStatusText}>{inspection.status}</Text>
                  </View>
                </View>
                <Text style={styles.activityDetailSubtext}>
                  Inspector: {inspection.inspector} | Faulty: {inspection.faultyItems}
                </Text>
                <Text style={styles.activityDetailDate}>
                  {inspection.country} - {inspection.project} | {formatDate(inspection.createdAt)}
                </Text>
              </View>
            ))}
            {item.inspections.length > 3 && (
              <Text style={styles.moreText}>+{item.inspections.length - 3} more inspections</Text>
            )}
          </View>
        )}

        {item.jobCards.length > 0 && (
          <View style={styles.activityDetailsSection}>
            <Text style={styles.activityDetailsTitle}>
              <MaterialIcons name="build" size={16} color="#00B894" /> Job Cards ({item.jobCards.length})
            </Text>
            {item.jobCards.slice(0, 3).map((jobCard, idx) => (
              <View key={idx} style={styles.activityDetailCard}>
                <View style={styles.activityDetailHeader}>
                  <Text style={styles.activityDetailText}>• {jobCard.requestNumber}</Text>
                  <View style={[styles.activityStatusBadge, { 
                    backgroundColor: jobCard.status === 'complete' ? '#2ecc40' :
                                   jobCard.status === 'in_progress' ? '#f39c12' :
                                   jobCard.status === 'Hold' ? '#e74c3c' : '#3498db'
                  }]}>
                    <Text style={styles.activityStatusText}>{jobCard.status}</Text>
                  </View>
                </View>
                <Text style={styles.activityDetailSubtext}>
                  Type: {jobCard.workOrderType} | Priority: {jobCard.priority} | Workers: {jobCard.assignedTo}
                </Text>
                <Text style={styles.activityDetailDate}>Created: {formatDate(jobCard.createdAt)}</Text>
              </View>
            ))}
            {item.jobCards.length > 3 && (
              <Text style={styles.moreText}>+{item.jobCards.length - 3} more job cards</Text>
            )}
          </View>
        )}

        {item.serviceSchedules.length > 0 && (
          <View style={styles.activityDetailsSection}>
            <Text style={styles.activityDetailsTitle}>
              <MaterialIcons name="event" size={16} color="#9C27B0" /> Service Schedules ({item.serviceSchedules.length})
            </Text>
            {item.serviceSchedules.slice(0, 3).map((schedule, idx) => (
              <View key={idx} style={styles.activityDetailItem}>
                <Text style={styles.activityDetailText}>
                  • {schedule.serviceOrderNumber} - {schedule.status}
                </Text>
                <Text style={styles.activityDetailDate}>{formatDate(schedule.createdAt)}</Text>
              </View>
            ))}
            {item.serviceSchedules.length > 3 && (
              <Text style={styles.moreText}>+{item.serviceSchedules.length - 3} more</Text>
            )}
          </View>
        )}

        {item.maintenanceRequests.length > 0 && (
          <View style={styles.activityDetailsSection}>
            <Text style={styles.activityDetailsTitle}>
              <MaterialIcons name="report-problem" size={16} color="#FF9800" /> Maintenance Requests ({item.maintenanceRequests.filter(r => r.isPending).length} Pending, {item.maintenanceRequests.length} Total)
            </Text>
            {item.maintenanceRequests.slice(0, 3).map((request, idx) => (
              <View key={idx} style={[styles.activityDetailCard, { borderLeftColor: '#FF9800' }]}>
                <View style={styles.activityDetailHeader}>
                  <Text style={styles.activityDetailText}>• {request.requestNumber}</Text>
                  <View style={[styles.activityStatusBadge, { 
                    backgroundColor: request.isPending ? '#FF9800' : 
                                   request.status === 'Rejected' ? '#e74c3c' : '#95a5a6'
                  }]}>
                    <Text style={styles.activityStatusText}>{request.status}</Text>
                  </View>
                </View>
                <Text style={styles.activityDetailSubtext}>
                  {request.title} | Type: {request.requestType} | Priority: {request.priority}
                </Text>
                <Text style={styles.activityDetailSubtext}>
                  Workers: {request.assignedTo} | Tasks: {request.tasks}
                </Text>
                <Text style={styles.activityDetailDate}>
                  {request.country} - {request.project} | {formatDate(request.createdAt)}
                </Text>
                {request.isPending && (
                  <View style={styles.pendingIndicator}>
                    <MaterialIcons name="schedule" size={12} color="#FF9800" />
                    <Text style={styles.pendingText}>Pending Action Required</Text>
                  </View>
                )}
              </View>
            ))}
            {item.maintenanceRequests.length > 3 && (
              <Text style={styles.moreText}>+{item.maintenanceRequests.length - 3} more maintenance requests</Text>
            )}
          </View>
        )}
      </View>
    );
  };

  const currentData = activeTab === 'all' ? allEquipment : 
                      activeTab === 'maintenance' ? maintenanceEquipment : 
                      activeTab === 'breakdown' ? breakdownEquipment :
                      activeEquipment;

  return (
    <View style={styles.container}>
      <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()}>
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Equipment Status KPI</Text>
          <TouchableOpacity onPress={() => setUserInfoModalVisible(true)}>
            <MaterialIcons name="lock" size={24} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      <ScrollView 
        style={styles.scrollView}
        showsVerticalScrollIndicator={false}
        contentContainerStyle={styles.scrollContent}
      >
        {/* Date Range Selector */}
        <View style={styles.dateRangeContainer}>
        <Text style={styles.dateRangeTitle}>Date Range Filter</Text>
        
        <View style={styles.datePickersRow}>
          <View style={styles.datePickerContainer}>
            <Text style={styles.dateLabel}>From Date</Text>
            <TouchableOpacity 
              style={styles.dateButton}
              onPress={() => setShowFromDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#0055A5" />
              <Text style={styles.dateButtonText}>{formatDate(fromDate)}</Text>
          </TouchableOpacity>
          </View>

          <View style={styles.datePickerContainer}>
            <Text style={styles.dateLabel}>To Date</Text>
            <TouchableOpacity 
              style={styles.dateButton}
              onPress={() => setShowToDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#0055A5" />
              <Text style={styles.dateButtonText}>{formatDate(toDate)}</Text>
          </TouchableOpacity>
        </View>
        </View>

        {/* Date Pickers */}
        {showFromDatePicker && (
          <DateTimePicker
            value={fromDate}
            mode="date"
            display={Platform.OS === 'ios' ? 'spinner' : 'default'}
            onChange={(event, selectedDate) => {
              setShowFromDatePicker(false);
              if (selectedDate) {
                selectedDate.setHours(0, 0, 0, 0);
                setFromDate(selectedDate);
              }
            }}
          />
        )}

        {showToDatePicker && (
          <DateTimePicker
            value={toDate}
            mode="date"
            display={Platform.OS === 'ios' ? 'spinner' : 'default'}
            onChange={(event, selectedDate) => {
              setShowToDatePicker(false);
              if (selectedDate) {
                selectedDate.setHours(23, 59, 59, 999);
                setToDate(selectedDate);
              }
            }}
          />
        )}
      </View>

      {/* Tabs - Responsive 2-row layout */}
      <View style={styles.tabsContainer}>
        {/* First Row */}
        <View style={styles.tabRow}>
              <TouchableOpacity
            style={[styles.tab, activeTab === 'all' && styles.activeTab]}
            onPress={() => setActiveTab('all')}
          >
            <MaterialIcons 
              name="list" 
              size={20} 
              color={activeTab === 'all' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'all' && styles.activeTabText]}>
              All
            </Text>
            <Text style={[styles.tabCount, activeTab === 'all' && styles.activeTabCount]}>
              ({allEquipment.length})
            </Text>
          </TouchableOpacity>

              <TouchableOpacity
            style={[styles.tab, activeTab === 'maintenance' && styles.activeTab]}
            onPress={() => setActiveTab('maintenance')}
          >
            <MaterialIcons 
              name="build-circle" 
              size={20} 
              color={activeTab === 'maintenance' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'maintenance' && styles.activeTabText]}>
              Maintenance
            </Text>
            <Text style={[styles.tabCount, activeTab === 'maintenance' && styles.activeTabCount]}>
              ({maintenanceEquipment.length})
            </Text>
          </TouchableOpacity>
                </View>

        {/* Second Row */}
        <View style={styles.tabRow}>
          <TouchableOpacity
            style={[styles.tab, activeTab === 'breakdown' && styles.activeTab]}
            onPress={() => setActiveTab('breakdown')}
          >
            <MaterialIcons 
              name="warning" 
              size={20} 
              color={activeTab === 'breakdown' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'breakdown' && styles.activeTabText]}>
              Breakdown
            </Text>
            <Text style={[styles.tabCount, activeTab === 'breakdown' && styles.activeTabCount]}>
              ({breakdownEquipment.length})
            </Text>
              </TouchableOpacity>

          <TouchableOpacity
            style={[styles.tab, activeTab === 'active' && styles.activeTab]}
            onPress={() => setActiveTab('active')}
          >
            <MaterialIcons 
              name="check-circle" 
              size={20} 
              color={activeTab === 'active' ? '#fff' : '#666'} 
            />
            <Text style={[styles.tabText, activeTab === 'active' && styles.activeTabText]}>
              Active
            </Text>
            <Text style={[styles.tabCount, activeTab === 'active' && styles.activeTabCount]}>
              ({activeEquipment.length})
            </Text>
              </TouchableOpacity>
        </View>
      </View>

      {/* Results Summary */}
      <View style={styles.summaryContainer}>
        <MaterialIcons name="info-outline" size={18} color="#0055A5" />
        <Text style={styles.summaryText}>
          Showing {currentData.length} equipment with activity from {formatDate(fromDate)} to {formatDate(toDate)}
        </Text>
      </View>

      {/* Legend */}
      <View style={styles.legendContainer}>
        <Text style={styles.legendTitle}>Tag Types Legend:</Text>
        <ScrollView horizontal showsHorizontalScrollIndicator={false}>
          <View style={styles.legendRow}>
            <View style={styles.legendItem}>
              <View style={[styles.legendDot, { backgroundColor: '#3498db' }]} />
              <Text style={styles.legendText}>Inspection</Text>
            </View>
            <View style={styles.legendItem}>
              <View style={[styles.legendDot, { backgroundColor: '#00B894' }]} />
              <Text style={styles.legendText}>Job Card</Text>
            </View>
            <View style={styles.legendItem}>
              <View style={[styles.legendDot, { backgroundColor: '#9C27B0' }]} />
              <Text style={styles.legendText}>Service Schedule</Text>
            </View>
            <View style={styles.legendItem}>
              <View style={[styles.legendDot, { backgroundColor: '#FF9800' }]} />
              <Text style={styles.legendText}>Maintenance</Text>
            </View>
          </View>
        </ScrollView>
      </View>

        {/* Content */}
        {loading ? (
          <View style={styles.loadingContainer}>
            <ActivityIndicator size="large" color="#0055A5" />
            <Text style={styles.loadingText}>Loading equipment data...</Text>
          </View>
        ) : currentData.length > 0 ? (
          currentData.map((item, index) => (
            <View key={`${item.equipmentNumber}-${item.id}-${index}`}>
              {renderEquipmentItem({ item })}
            </View>
          ))
        ) : (
          <View style={styles.emptyContainer}>
            <MaterialIcons 
              name={activeTab === 'maintenance' ? 'build-circle' :
                    activeTab === 'breakdown' ? 'warning' :
                    activeTab === 'active' ? 'check-circle' : 'precision-manufacturing'} 
              size={60} 
              color="#ccc" 
            />
            <Text style={styles.emptyText}>
              No equipment found in this category
            </Text>
            <Text style={styles.emptySubtext}>
              Try adjusting the date range or check another tab
            </Text>
          </View>
        )}
      </ScrollView>

      {/* User Info Modal */}
      <Modal visible={userInfoModalVisible} transparent animationType="fade" onRequestClose={() => setUserInfoModalVisible(false)}>
        <View style={styles.modalContainer}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>User Information</Text>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Countries:</Text>
              <Text style={styles.userInfoValue}>
                {userInfo.countries.length > 0 ? userInfo.countries.join(', ') : 'N/A'}
              </Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Projects:</Text>
              <Text style={styles.userInfoValue}>
                {userInfo.projects.length > 0 ? userInfo.projects.join(', ') : 'N/A'}
              </Text>
            </View>
            <TouchableOpacity style={styles.closeButton} onPress={() => setUserInfoModalVisible(false)}>
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <SyncModal visible={loading} message="Loading Equipment Data..." />
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  header: {
    paddingTop: StatusBar.currentHeight || 40,
    paddingBottom: 16,
    paddingHorizontal: 16,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  headerTitle: {
    color: '#fff',
    fontSize: 20,
    fontWeight: 'bold',
  },
  dateRangeContainer: {
    backgroundColor: '#fff',
    padding: 16,
    marginHorizontal: 16,
    marginTop: 16,
    marginBottom: 0,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  dateRangeTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 12,
  },
  datePickersRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  datePickerContainer: {
    flex: 1,
    marginHorizontal: 4,
  },
  dateLabel: {
    fontSize: 12,
    color: '#666',
    marginBottom: 6,
    fontWeight: '500',
  },
  dateButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    paddingVertical: 10,
    paddingHorizontal: 12,
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#0055A5',
  },
  dateButtonText: {
    marginLeft: 8,
    fontSize: 14,
    color: '#0055A5',
    fontWeight: '500',
  },
  tabsContainer: {
    backgroundColor: '#fff',
    marginHorizontal: 16,
    marginTop: 12,
    borderRadius: 8,
    overflow: 'hidden',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  tabRow: {
    flexDirection: 'row',
    width: '100%',
  },
  tab: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 4,
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#f5f5f5',
    borderWidth: 0.5,
    borderColor: '#e0e0e0',
    minHeight: 60,
  },
  activeTab: {
    backgroundColor: '#0055A5',
  },
  tabContent: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
  },
  tabText: {
    fontSize: 12,
    color: '#666',
    fontWeight: '600',
    marginTop: 4,
    textAlign: 'center',
  },
  activeTabText: {
    color: '#fff',
  },
  tabCount: {
    fontSize: 11,
    color: '#999',
    fontWeight: '500',
    marginTop: 2,
  },
  activeTabCount: {
    color: '#fff',
  },
  summaryContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    marginHorizontal: 16,
    marginTop: 12,
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderRadius: 6,
    borderLeftWidth: 4,
    borderLeftColor: '#ffd205',
  },
  summaryText: {
    fontSize: 12,
    color: '#666',
    fontWeight: '500',
    marginLeft: 8,
    flex: 1,
  },
  legendContainer: {
    backgroundColor: '#fff',
    marginHorizontal: 16,
    marginTop: 12,
    marginBottom: 16,
    padding: 12,
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  legendTitle: {
    fontSize: 12,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
  },
  legendRow: {
    flexDirection: 'row',
    gap: 12,
  },
  legendItem: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  legendDot: {
    width: 10,
    height: 10,
    borderRadius: 5,
    marginRight: 4,
  },
  legendText: {
    fontSize: 11,
    color: '#666',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 20,
  },
  equipmentCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    marginBottom: 16,
    marginHorizontal: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    overflow: 'hidden',
  },
  equipmentHeader: {
    backgroundColor: '#E9E9E9',
    paddingHorizontal: 16,
    paddingVertical: 12,
  },
  equipmentTitleRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  equipmentName: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
  },
  equipmentNumber: {
    fontSize: 12,
    color: '#666',
    marginTop: 2,
  },
  equipmentStatusBadge: {
    paddingHorizontal: 10,
    paddingVertical: 4,
    borderRadius: 4,
    alignSelf: 'flex-start',
    marginTop: 4,
  },
  equipmentStatusText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
  },
  equipmentInfo: {
    padding: 16,
    gap: 8,
  },
  infoRow: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 8,
  },
  infoText: {
    fontSize: 14,
    color: '#666',
  },
  divider: {
    height: 1,
    backgroundColor: '#ddd',
    marginHorizontal: 16,
  },
  tagsSection: {
    padding: 16,
  },
  tagsSectionTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 10,
  },
  tagsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    gap: 8,
  },
  tag: {
    paddingHorizontal: 12,
    paddingVertical: 6,
    borderRadius: 16,
    flexDirection: 'column',
    alignItems: 'center',
  },
  tagType: {
    color: '#fff',
    fontSize: 11,
    fontWeight: '600',
  },
  tagReference: {
    color: '#fff',
    fontSize: 10,
    marginTop: 2,
  },
  activitySummary: {
    padding: 16,
  },
  activityTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 10,
  },
  activityRow: {
    flexDirection: 'row',
    justifyContent: 'space-around',
  },
  activityItem: {
    alignItems: 'center',
  },
  activityCount: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 4,
  },
  activityLabel: {
    fontSize: 10,
    color: '#666',
    marginTop: 2,
  },
  activityDetailsSection: {
    paddingHorizontal: 16,
    paddingVertical: 8,
    backgroundColor: '#f9f9f9',
  },
  activityDetailsTitle: {
    fontSize: 13,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
  },
  activityDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 4,
    paddingLeft: 16,
  },
  activityDetailCard: {
    backgroundColor: '#fff',
    borderRadius: 6,
    padding: 10,
    marginBottom: 8,
    borderLeftWidth: 3,
    borderLeftColor: '#00B894',
  },
  activityDetailHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 4,
  },
  activityDetailText: {
    fontSize: 12,
    color: '#015185',
    fontWeight: '600',
    flex: 1,
  },
  activityStatusBadge: {
    paddingHorizontal: 6,
    paddingVertical: 3,
    borderRadius: 4,
  },
  activityStatusText: {
    color: '#fff',
    fontSize: 10,
    fontWeight: '600',
  },
  activityDetailSubtext: {
    fontSize: 11,
    color: '#666',
    marginBottom: 2,
  },
  activityDetailDate: {
    fontSize: 10,
    color: '#999',
  },
  moreText: {
    fontSize: 11,
    color: '#0055A5',
    fontStyle: 'italic',
    marginTop: 4,
    paddingLeft: 16,
  },
  pendingIndicator: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: 6,
    paddingTop: 6,
    borderTopWidth: 1,
    borderTopColor: '#FFE0B2',
    gap: 4,
  },
  pendingText: {
    fontSize: 11,
    color: '#FF9800',
    fontWeight: '600',
  },
  loadingContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    paddingVertical: 60,
    minHeight: 300,
  },
  loadingText: {
    marginTop: 12,
    fontSize: 14,
    color: '#666',
  },
  emptyContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 40,
    paddingVertical: 60,
    minHeight: 300,
  },
  emptyText: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    marginTop: 16,
    fontWeight: '500',
  },
  emptySubtext: {
    fontSize: 14,
    color: '#999',
    textAlign: 'center',
    marginTop: 8,
  },
  modalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 16,
  },
  userInfoItem: {
    width: '100%',
    paddingVertical: 8,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  userInfoLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
  userInfoValue: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
    marginTop: 4,
  },
  closeButton: {
    backgroundColor: '#0055A5',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 6,
    marginTop: 16,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default EquipmentstatusKpiHome;
