import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, StatusBar, Dimensions, ScrollView, Modal, ActivityIndicator, Animated, Image } from 'react-native';
import Svg, { Circle } from 'react-native-svg';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import DropDownPicker from 'react-native-dropdown-picker';
import DateTimePickerModal from 'react-native-modal-datetime-picker';
import { BarChart } from 'react-native-gifted-charts';
import { LineChart } from 'react-native-chart-kit';
import API_ENDPOINTS from '../../../../config/mongodb.endpoints';

const { width } = Dimensions.get('window');
const HEADER_HEIGHT = width * 0.32;

// Full list of roles from Basic_Details_TeamTab1.js
const ALL_ROLES = [
  'Operations Manager',
  'Health & Safety Manager',
  'Project Manager',
  'Site Manager',
  'Supervisor',
  'HrManager',
  'FinanceManager',
  'Driller',
  'Trainee Driller',
  'Junior Driller',
  'Coremarker',
  'Assistant',
  'Workshop Manager',
  'Workshop Foreman',
  'Maintenance Manager',
  'Mechanical Engineer',
  'Boilermaker',
  'Welder',
  'Mechanic',
  'Machinist',
  'Electrician',
  'Housekeeper',
  'Cook',
  'Stores Officer',
  'General Worker',
];

const MODULE_TABS = [
  { key: 'inspections', label: 'Total Inspections' },
  { key: 'jobCards', label: 'Job Cards' },
  { key: 'maintenance', label: 'Maintenance' },
  { key: 'serviceOrders', label: 'Service Orders' },
];

// Custom Concentric Circular Progress Chart Component
const ConcentricProgressChart = ({ data, size = 150 }) => {
  const strokeWidth = 12;
  const radius = (size - strokeWidth) / 2;
  const center = size / 2;
  
  const renderCircle = (progress, strokeColor, strokeWidth, radius, index) => {
    const circumference = 2 * Math.PI * radius;
    const strokeDasharray = circumference;
    const strokeDashoffset = circumference - (progress / 100) * circumference;
    
    return (
      <View key={index} style={{ position: 'absolute' }}>
        <Svg width={size} height={size}>
          {/* Background circle */}
          <Circle
            cx={center}
            cy={center}
            r={radius}
            stroke="#E0E0E0"
            strokeWidth={strokeWidth}
            fill="transparent"
          />
          {/* Progress circle */}
          <Circle
            cx={center}
            cy={center}
            r={radius}
            stroke={strokeColor}
            strokeWidth={strokeWidth}
            fill="transparent"
            strokeDasharray={strokeDasharray}
            strokeDashoffset={strokeDashoffset}
            strokeLinecap="round"
            transform={`rotate(-90 ${center} ${center})`}
          />
        </Svg>
      </View>
    );
  };

  return (
    <View style={{ alignItems: 'center', justifyContent: 'center' }}>
      <View style={{ position: 'relative', width: size, height: size }}>
        {data.map((item, index) => {
          const currentRadius = radius - (index * (strokeWidth + 4));
          return renderCircle(
            item.percentage,
            item.color,
            strokeWidth,
            currentRadius,
            index
          );
        })}
      </View>
      <View style={styles.legendContainer}>
        {data.map((item, idx) => (
          <View key={idx} style={styles.legendItem}>
            <View style={[styles.legendColor, { backgroundColor: item.color }]} />
            <Text style={[styles.legendText, { color: item.color }]}>
              {item.text}: {item.count}
            </Text>
          </View>
        ))}
      </View>
    </View>
  );
};

// Loader Modal (SyncModal) from EmployeesKpi.js
const SyncModal = ({ visible, message = 'Loading Roles' }) => {
  const rotateAnim = React.useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  React.useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);
  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const SupervisorKpi = () => {
  const navigation = useNavigation();
  const [allUsers, setAllUsers] = useState([]); // All admins from Firebase
  const [filteredUsers, setFilteredUsers] = useState([]); // Filtered by country/project
  const [totalMembers, setTotalMembers] = useState(0);
  const [supervisorCount, setSupervisorCount] = useState(0);
  const [operatorCount, setOperatorCount] = useState(0);
  const [teamMembers, setTeamMembers] = useState([]); // Added from EmployeesKpi.js
  const [totalInspections, setTotalInspections] = useState(0); // Added from EmployeesKpi.js
  const [modalVisible, setModalVisible] = useState(false);
  const [modalLoading, setModalLoading] = useState(false);
  const [modalError, setModalError] = useState(null);
  const [userDetails, setUserDetails] = useState(null);
  const [myCountry, setMyCountry] = useState('');
  const [myProject, setMyProject] = useState('');
  const [roleCounts, setRoleCounts] = useState({});
  const [managerCount, setManagerCount] = useState(0);
  const [dropdownOpen, setDropdownOpen] = useState(false);
  const [selectedRole, setSelectedRole] = useState('');
  const [dropdownItems, setDropdownItems] = useState([]);
  const [userStats, setUserStats] = useState({}); // { userId: { inspections: n, jobCards: n, dvir: n, incidents: n, maintenance: n, serviceSchedule: n } }
  const [loadingStats, setLoadingStats] = useState(false);
  const [roleLoading, setRoleLoading] = useState(false); // Added from EmployeesKpi.js
  const [kpiMode, setKpiMode] = useState('role'); // 'role' or 'module'
  const [moduleTab, setModuleTab] = useState('inspections');
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage] = useState(20);
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [dateRange, setDateRange] = useState({ from: null, to: null });
  const [datePickerMode, setDatePickerMode] = useState('from');
  const [barData, setBarData] = useState([]);
  const [donutData, setDonutData] = useState([]);
  const [donutCenter, setDonutCenter] = useState({ value: 0, label: '' });
  const [inspectionReports, setInspectionReports] = useState([]);
  const [jobCards, setJobCards] = useState([]);
  const [maintenanceReports, setMaintenanceReports] = useState([]);
  const [serviceOrders, setServiceOrders] = useState([]);
  const [userDetailsList, setUserDetailsList] = useState([]);
  const [jobCardUserDetails, setJobCardUserDetails] = useState([]);
  const [maintenanceUserDetails, setMaintenanceUserDetails] = useState([]);
  const [serviceOrderUserDetails, setServiceOrderUserDetails] = useState([]);
  const [inspectionEquipmentCount, setInspectionEquipmentCount] = useState(0);
  const [jobCardEquipmentCount, setJobCardEquipmentCount] = useState(0);
  const [maintenanceEquipmentCount, setMaintenanceEquipmentCount] = useState(0);
  const [serviceOrderEquipmentCount, setServiceOrderEquipmentCount] = useState(0);
  const [loadingInspectionsSummary, setLoadingInspectionsSummary] = useState(false);
  const [loadingJobCardsSummary, setLoadingJobCardsSummary] = useState(false);
  const [loadingMaintenanceSummary, setLoadingMaintenanceSummary] = useState(false);
  const [loadingServiceOrdersSummary, setLoadingServiceOrdersSummary] = useState(false);
  const [totalInspectionsModule, setTotalInspectionsModule] = useState(null); // Renamed to avoid conflict
  const [loadingTotalInspections, setLoadingTotalInspections] = useState(false);
  const [allInspectionDateCounts, setAllInspectionDateCounts] = useState([]);
  const [loadingAllInspectionDates, setLoadingAllInspectionDates] = useState(false);
  const [selectedBar, setSelectedBar] = useState(null);

  const showToast = (type, title, message) => {
    // Only show info and success toasts, skip error toasts
    if (type !== 'error') {
      Toast.show({
        type: type,
        text1: title,
        text2: message,
        position: 'bottom',
      });
    }
  };

  // Fetch all admins and current user's country/project from MongoDB
  useEffect(() => {
    const fetchAllUsers = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          showToast('error', 'Error', 'User ID not found. Please log in again.');
          return;
        }

        // Fetch current user's details from MongoDB
        const userResponse = await fetch(API_ENDPOINTS.USERS.GET_BY_ID(userUid));
        const userData = await userResponse.json();
        
        if (!userData.success || !userData.data) {
          showToast('error', 'Error', 'User details not found.');
          return;
        }

        const myData = userData.data;
        let myCountries = Array.isArray(myData?.countries) ? myData.countries : (myData?.countries ? [myData.countries] : []);
        let myProjects = Array.isArray(myData?.projects) ? myData.projects : (myData?.projects ? [myData.projects] : []);
        setMyCountry(myCountries[0] || '');
        setMyProject(myProjects[0] || '');

        // Fetch all users from MongoDB with pagination
        const allUsersResponse = await fetch(`${API_ENDPOINTS.USERS.GET_TEAM_MANAGEMENT}?page=1&limit=500`);
        const allUsersData = await allUsersResponse.json();

        if (!allUsersData.success || !allUsersData.data) {
          setAllUsers([]);
          setFilteredUsers([]);
          setTotalMembers(0);
          setSupervisorCount(0);
          setOperatorCount(0);
          return;
        }

        const usersArr = allUsersData.data.map((user) => ({
          id: user._id || user.uid,
          ...user,
        }));

        setAllUsers(usersArr);

        // Filter users by country/project
        const filtered = usersArr.filter((user) => {
          const userCountries = Array.isArray(user.countries) ? user.countries : (user.countries ? [user.countries] : []);
          const userProjects = Array.isArray(user.projects) ? user.projects : (user.projects ? [user.projects] : []);
          const countryMatch = myCountries.some((c) => userCountries.includes(c));
          const projectMatch = myProjects.some((p) => userProjects.includes(p));
          return countryMatch && projectMatch;
        });

        setFilteredUsers(filtered);
        setTotalMembers(filtered.length);

        // Count by role
        const counts = {};
        let managerTotal = 0;
        ALL_ROLES.forEach(role => { counts[role] = 0; });
        filtered.forEach(u => {
          const role = u.role || '';
          if (ALL_ROLES.includes(role)) {
            counts[role] = (counts[role] || 0) + 1;
          }
          if (role.endsWith('Manager')) {
            managerTotal++;
          }
        });

        setRoleCounts(counts);
        setManagerCount(managerTotal);
        setSupervisorCount(filtered.filter(u => (u.role || '').toLowerCase() === 'supervisor').length);
        setOperatorCount(filtered.filter(u => !['supervisor', 'admin', 'manager'].includes((u.role || '').toLowerCase())).length);

        // Build dropdown items with "All" option
        const roleItems = ALL_ROLES
          .filter(role => (counts[role] || 0) > 0)
          .map(role => ({ label: `${role} (${counts[role] || 0})`, value: role }));
        
        const allItem = { label: `All (${filtered.length})`, value: 'All' };
        const items = [allItem, ...roleItems];
        
        setDropdownItems(items);
        setSelectedRole('All'); // Default to "All"
      } catch (error) {
        console.error('Error fetching all users:', error);
        showToast('error', 'Error', 'Failed to fetch users.');
      }
    };
    fetchAllUsers();
  }, []);

  // Removed fetchTeamMembers - using allUsers from MongoDB instead

  // Fetch all module counts for each user from MongoDB
  useEffect(() => {
    const fetchStats = async () => {
      setLoadingStats(true);
      try {
        if (!filteredUsers.length) {
          setUserStats({});
          setLoadingStats(false);
          return;
        }

        const stats = {};
        
        // Fetch all data from MongoDB in parallel
        const [
          inspectionsResponse,
          jobCardsResponse,
          dvirResponse,
          incidentReportsResponse,
          maintenanceResponse,
          serviceSchedulesResponse
        ] = await Promise.all([
          fetch(`${API_ENDPOINTS.INSPECTIONS.GET_ALL}?page=1&limit=10000`),
          fetch(`${API_ENDPOINTS.JOB_CARDS.GET_ALL}?page=1&limit=10000`),
          fetch(`${API_ENDPOINTS.DVIR.GET_ALL}?limit=10000`),
          fetch(API_ENDPOINTS.INCIDENT_REPORTS.GET_ALL),
          fetch(API_ENDPOINTS.MAINTENANCE.GET_ALL),
          fetch(API_ENDPOINTS.SERVICE_SCHEDULES.GET_ALL)
        ]);

        const [
          inspectionsData,
          jobCardsData,
          dvirData,
          incidentReportsData,
          maintenanceData,
          serviceSchedulesData
        ] = await Promise.all([
          inspectionsResponse.json(),
          jobCardsResponse.json(),
          dvirResponse.json(),
          incidentReportsResponse.json(),
          maintenanceResponse.json(),
          serviceSchedulesResponse.json()
        ]);

        const allInspections = inspectionsData.success ? inspectionsData.data : [];
        const allJobCards = jobCardsData.success ? jobCardsData.data : [];
        const allDVIR = dvirData.success ? dvirData.data : [];
        const allIncidentReports = incidentReportsData.success ? incidentReportsData.data : [];
        const allMaintenance = maintenanceData.success ? maintenanceData.data : [];
        const allServiceSchedules = serviceSchedulesData.success ? serviceSchedulesData.data : [];

        // Count all modules per user
        filteredUsers.forEach(user => {
          const userId = user.id;
          const inspections = allInspections.filter(insp => 
            insp.userId === userId || insp.createdBy === userId
          ).length;
          
          const jobCards = allJobCards.filter(jc => 
            jc.adminUid === userId || jc.createdBy === userId
          ).length;
          
          const dvir = allDVIR.filter(d => 
            d.userUid === userId || d.createdBy === userId
          ).length;
          
          const incidents = allIncidentReports.filter(ir => 
            ir.adminUid === userId || ir.createdBy === userId
          ).length;
          
          const maintenance = allMaintenance.filter(m => 
            m.adminUid === userId || m.userId === userId
          ).length;
          
          const serviceSchedule = allServiceSchedules.filter(ss => 
            ss.adminUid === userId || ss.createdBy === userId
          ).length;
          
          stats[userId] = { 
            inspections, 
            jobCards, 
            dvir, 
            incidents, 
            maintenance, 
            serviceSchedule 
          };
          
          console.log('Stats for', userId, '- Inspections:', inspections, 'Job Cards:', jobCards, 'DVIR:', dvir, 'Incidents:', incidents, 'Maintenance:', maintenance, 'Service Schedule:', serviceSchedule);
        });

        setUserStats(stats);
      } catch (e) {
        setUserStats({});
        console.log('Error in fetchStats:', e);
      }
      setLoadingStats(false);
    };
    fetchStats();
  }, [filteredUsers]);

  // Handle role change with loading state
  useEffect(() => {
    if (selectedRole) {
      setRoleLoading(true);
      setCurrentPage(1); // Reset to first page when role changes
      const timer = setTimeout(() => {
        setRoleLoading(false);
      }, 500); // 500ms delay to simulate loading
      return () => clearTimeout(timer);
    }
  }, [selectedRole]);

  // Fetch inspection reports based on date range from MongoDB
  useEffect(() => {
    if (kpiMode !== 'module' || !dateRange.from || !dateRange.to || moduleTab !== 'inspections') {
      setInspectionReports([]);
      setUserDetailsList([]);
      setInspectionEquipmentCount(0);
      setLoadingInspectionsSummary(false);
      return;
    }
    setLoadingInspectionsSummary(true);
    const fetchData = async () => {
      try {
        // Fetch all inspections from MongoDB
        const inspectionsResponse = await fetch(`${API_ENDPOINTS.INSPECTIONS.GET_ALL}?page=1&limit=10000`);
        const inspectionsData = await inspectionsResponse.json();
        const allInspectionsData = inspectionsData.success ? inspectionsData.data : [];

        const reports = [];
        const equipmentSet = new Set();
        
        // Normalize date range for comparison
        const fromDateNormalized = new Date(dateRange.from);
        fromDateNormalized.setHours(0, 0, 0, 0);
        const toDateNormalized = new Date(dateRange.to);
        toDateNormalized.setHours(23, 59, 59, 999);
        
        allInspectionsData.forEach(report => {
          const createdAt = report?.step3?.createdAt || report?.createdAt;
          if (createdAt) {
            const reportDate = new Date(createdAt);
            if (reportDate >= fromDateNormalized && reportDate <= toDateNormalized) {
              const userId = report.userId || report.createdBy;
              reports.push({ 
                date: reportDate, 
                status: report.status || 'Completed', 
                userId 
              });
              const eqNum = report?.step1?.selectedEquipment?.equipmentNumber;
              if (eqNum) equipmentSet.add(eqNum);
            }
          }
        });

        console.log('=== INSPECTION REPORTS DEBUG ===');
        console.log('Date Range:', fromDateNormalized.toLocaleDateString(), 'to', toDateNormalized.toLocaleDateString());
        console.log('Total Inspections Fetched:', allInspectionsData.length);
        console.log('Filtered Inspections in Date Range:', reports.length);
        console.log('Sample Report Dates:', reports.slice(0, 5).map(r => r.date.toLocaleDateString()));
        
        setInspectionReports(reports);
        setInspectionEquipmentCount(equipmentSet.size);

        // Count reports per user
        const userReportCounts = {};
        reports.forEach(r => {
          if (!userReportCounts[r.userId]) userReportCounts[r.userId] = 0;
          userReportCounts[r.userId]++;
        });

        const userIds = Object.keys(userReportCounts);
        
        // Get user details from already loaded allUsers
        const details = userIds.map(uid => {
          const user = allUsers.find(u => u.id === uid || u._id === uid);
          return {
            id: uid,
            name: user?.fullName || user?.name || user?.email || uid,
            role: user?.role || 'Unknown',
            count: userReportCounts[uid]
          };
        });

        setUserDetailsList(details);
        setLoadingInspectionsSummary(false);
      } catch (e) {
        console.error('Error fetching inspection reports:', e);
        setInspectionReports([]);
        setUserDetailsList([]);
        setInspectionEquipmentCount(0);
        setLoadingInspectionsSummary(false);
      }
    };
    fetchData();
  }, [kpiMode, dateRange, moduleTab, allUsers]);

  // Fetch job cards from MongoDB
  useEffect(() => {
    if (kpiMode !== 'module' || !dateRange.from || !dateRange.to || moduleTab !== 'jobCards') {
      setJobCards([]);
      setJobCardUserDetails([]);
      setJobCardEquipmentCount(0);
      setLoadingJobCardsSummary(false);
      return;
    }
    setLoadingJobCardsSummary(true);
    const fetchJobCards = async () => {
      try {
        // Fetch all job cards from MongoDB
        const jobCardsResponse = await fetch(`${API_ENDPOINTS.JOB_CARDS.GET_ALL}?page=1&limit=10000`);
        const jobCardsData = await jobCardsResponse.json();
        const allJobCardsData = jobCardsData.success ? jobCardsData.data : [];

        const jobCardsArr = [];
        const equipmentSet = new Set();

        // Normalize date range for comparison
        const fromDateNormalized = new Date(dateRange.from);
        fromDateNormalized.setHours(0, 0, 0, 0);
        const toDateNormalized = new Date(dateRange.to);
        toDateNormalized.setHours(23, 59, 59, 999);

        allJobCardsData.forEach(card => {
          const createdAt = card?.createdAt;
          if (createdAt) {
            const cardDate = new Date(createdAt);
            if (cardDate >= fromDateNormalized && cardDate <= toDateNormalized) {
              const userId = card.adminUid || card.createdBy;
              jobCardsArr.push({ date: cardDate, userId });
              const eqNum = card?.equipment?.equipmentNumber;
              if (eqNum) equipmentSet.add(eqNum);
            }
          }
        });

        setJobCards(jobCardsArr);
        setJobCardEquipmentCount(equipmentSet.size);

        // Count job cards per user
        const userJobCardCounts = {};
        jobCardsArr.forEach(jc => {
          if (!userJobCardCounts[jc.userId]) userJobCardCounts[jc.userId] = 0;
          userJobCardCounts[jc.userId]++;
        });

        const userIds = Object.keys(userJobCardCounts);
        
        // Get user details from already loaded allUsers
        const details = userIds.map(uid => {
          const user = allUsers.find(u => u.id === uid || u._id === uid);
          return {
            id: uid,
            name: user?.fullName || user?.name || user?.email || uid,
            role: user?.role || 'Unknown',
            count: userJobCardCounts[uid]
          };
        });

        setJobCardUserDetails(details);
        setLoadingJobCardsSummary(false);
      } catch (e) {
        console.error('Error fetching job cards:', e);
        setJobCards([]);
        setJobCardUserDetails([]);
        setJobCardEquipmentCount(0);
        setLoadingJobCardsSummary(false);
      }
    };
    fetchJobCards();
  }, [kpiMode, dateRange, moduleTab, allUsers]);

  // Fetch maintenance reports from MongoDB
  useEffect(() => {
    if (kpiMode !== 'module' || !dateRange.from || !dateRange.to || moduleTab !== 'maintenance') {
      setMaintenanceReports([]);
      setMaintenanceUserDetails([]);
      setMaintenanceEquipmentCount(0);
      setLoadingMaintenanceSummary(false);
      return;
    }
    setLoadingMaintenanceSummary(true);
    const fetchMaintenance = async () => {
      try {
        // Fetch all maintenance reports from MongoDB
        const maintenanceResponse = await fetch(API_ENDPOINTS.MAINTENANCE.GET_ALL);
        const maintenanceData = await maintenanceResponse.json();
        const allMaintenanceData = maintenanceData.success ? maintenanceData.data : [];

        const reportsArr = [];
        const equipmentSet = new Set();

        // Normalize date range for comparison
        const fromDateNormalized = new Date(dateRange.from);
        fromDateNormalized.setHours(0, 0, 0, 0);
        const toDateNormalized = new Date(dateRange.to);
        toDateNormalized.setHours(23, 59, 59, 999);

        allMaintenanceData.forEach(report => {
          const createdAt = report?.createdAt;
          if (createdAt) {
            const reportDate = new Date(createdAt);
            if (reportDate >= fromDateNormalized && reportDate <= toDateNormalized) {
              const userId = report.adminUid || report.userId;
              reportsArr.push({ date: reportDate, userId });
              const eqNum = report?.equipment?.equipmentNumber;
              if (eqNum) equipmentSet.add(eqNum);
            }
          }
        });

        setMaintenanceReports(reportsArr);
        setMaintenanceEquipmentCount(equipmentSet.size);

        // Count reports per user
        const userMaintCounts = {};
        reportsArr.forEach(r => {
          if (!userMaintCounts[r.userId]) userMaintCounts[r.userId] = 0;
          userMaintCounts[r.userId]++;
        });

        const userIds = Object.keys(userMaintCounts);
        
        // Get user details from already loaded allUsers
        const details = userIds.map(uid => {
          const user = allUsers.find(u => u.id === uid || u._id === uid);
          return {
            id: uid,
            name: user?.fullName || user?.name || user?.email || uid,
            role: user?.role || 'Unknown',
            count: userMaintCounts[uid]
          };
        });

        setMaintenanceUserDetails(details);
        setLoadingMaintenanceSummary(false);
      } catch (e) {
        console.error('Error fetching maintenance reports:', e);
        setMaintenanceReports([]);
        setMaintenanceUserDetails([]);
        setMaintenanceEquipmentCount(0);
        setLoadingMaintenanceSummary(false);
      }
    };
    fetchMaintenance();
  }, [kpiMode, dateRange, moduleTab, allUsers]);

  // Fetch service orders from MongoDB
  useEffect(() => {
    if (kpiMode !== 'module' || !dateRange.from || !dateRange.to || moduleTab !== 'serviceOrders') {
      setServiceOrders([]);
      setServiceOrderUserDetails([]);
      setServiceOrderEquipmentCount(0);
      setLoadingServiceOrdersSummary(false);
      return;
    }
    setLoadingServiceOrdersSummary(true);
    const fetchServiceOrders = async () => {
      try {
        // Fetch all service schedules from MongoDB
        const serviceResponse = await fetch(API_ENDPOINTS.SERVICE_SCHEDULES.GET_ALL);
        const serviceData = await serviceResponse.json();
        const allServiceData = serviceData.success ? serviceData.data : [];

        const ordersArr = [];
        const equipmentSet = new Set();

        // Normalize date range for comparison
        const fromDateNormalized = new Date(dateRange.from);
        fromDateNormalized.setHours(0, 0, 0, 0);
        const toDateNormalized = new Date(dateRange.to);
        toDateNormalized.setHours(23, 59, 59, 999);

        allServiceData.forEach(order => {
          const createdAt = order?.createdAt;
          if (createdAt) {
            const orderDate = new Date(createdAt);
            if (orderDate >= fromDateNormalized && orderDate <= toDateNormalized) {
              const userId = order.adminUid;
              ordersArr.push({ date: orderDate, userId });
              const eqName = order?.equipment?.equipmentName;
              if (eqName) equipmentSet.add(eqName);
            }
          }
        });

        setServiceOrders(ordersArr);
        setServiceOrderEquipmentCount(equipmentSet.size);

        // Count orders per user
        const userOrderCounts = {};
        ordersArr.forEach(r => {
          if (!userOrderCounts[r.userId]) userOrderCounts[r.userId] = 0;
          userOrderCounts[r.userId]++;
        });

        const userIds = Object.keys(userOrderCounts);
        
        // Get user details from already loaded allUsers
        const details = userIds.map(uid => {
          const user = allUsers.find(u => u.id === uid || u._id === uid);
          return {
            id: uid,
            name: user?.fullName || user?.name || user?.email || uid,
            role: user?.role || 'Unknown',
            count: userOrderCounts[uid]
          };
        });

        setServiceOrderUserDetails(details);
        setLoadingServiceOrdersSummary(false);
      } catch (e) {
        console.error('Error fetching service orders:', e);
        setServiceOrders([]);
        setServiceOrderUserDetails([]);
        setServiceOrderEquipmentCount(0);
        setLoadingServiceOrdersSummary(false);
      }
    };
    fetchServiceOrders();
  }, [kpiMode, dateRange, moduleTab, allUsers]);

  // Fetch total inspection reports from MongoDB
  useEffect(() => {
    if (kpiMode !== 'module') return;
    setLoadingTotalInspections(true);
    setTotalInspectionsModule(null);
    const fetchTotal = async () => {
      try {
        // Fetch count from MongoDB
        const countResponse = await fetch(API_ENDPOINTS.INSPECTIONS.COUNT);
        const countData = await countResponse.json();
        const total = countData.success ? countData.total : 0;
        setTotalInspectionsModule(total);
      } catch (e) {
        console.error('Error fetching total inspections:', e);
        setTotalInspectionsModule('Error');
      }
      setLoadingTotalInspections(false);
    };
    fetchTotal();
  }, [kpiMode]);

  // Fetch all inspection report dates and counts from MongoDB
  useEffect(() => {
    if (kpiMode !== 'module') return;
    setLoadingAllInspectionDates(true);
    const fetchDates = async () => {
      try {
        // Fetch all inspections from MongoDB
        const inspectionsResponse = await fetch(`${API_ENDPOINTS.INSPECTIONS.GET_ALL}?page=1&limit=10000`);
        const inspectionsData = await inspectionsResponse.json();
        const allInspectionsData = inspectionsData.success ? inspectionsData.data : [];

        const dateCounts = {};
        allInspectionsData.forEach(report => {
          const createdAt = report?.step3?.createdAt || report?.createdAt;
          if (createdAt) {
            const dateStr = new Date(createdAt).toLocaleDateString();
            dateCounts[dateStr] = (dateCounts[dateStr] || 0) + 1;
          }
        });

        const arr = Object.entries(dateCounts).map(([date, count]) => ({ date, count }));
        arr.sort((a, b) => new Date(a.date) - new Date(b.date));
        setAllInspectionDateCounts(arr);
      } catch (e) {
        console.error('Error fetching inspection dates:', e);
        setAllInspectionDateCounts([]);
      }
      setLoadingAllInspectionDates(false);
    };
    fetchDates();
  }, [kpiMode]);

  // Update chart data
  useEffect(() => {
    if (kpiMode !== 'module' || !dateRange.from || !dateRange.to) {
      setBarData([]);
      setDonutData([]);
      setDonutCenter({ value: 0, label: '' });
      return;
    }

    const start = new Date(dateRange.from);
    const end = new Date(dateRange.to);
    const dates = getDateArray(start, end);
    let data = [];
    let donutBreakdown = [];

    // Helper function to normalize date to start of day
    const normalizeDate = (date) => {
      const d = new Date(date);
      d.setHours(0, 0, 0, 0);
      return d.getTime();
    };

    if (moduleTab === 'inspections') {
      console.log('=== CHART GENERATION FOR INSPECTIONS ===');
      console.log('Total Inspection Reports:', inspectionReports.length);
      console.log('Date Range:', dates.length, 'days');
      
      data = dates.map(d => {
        const targetDate = normalizeDate(d);
        const count = inspectionReports.filter(r => {
          const reportDate = normalizeDate(r.date);
          return reportDate === targetDate;
        }).length;
        
        if (count > 0) {
          console.log(`${d.toLocaleDateString()}: ${count} reports`);
        }
        
        return {
          value: count,
          label: `${d.getDate()}/${d.getMonth() + 1}`,
          date: d,
          frontColor: '#A18AFF',
          sideColor: '#8000FF',
          barWidth: 22,
          topLabelComponent: () => null,
        };
      });
      
      const total = inspectionReports.length;
      const sumOfDailyCounts = data.reduce((sum, d) => sum + d.value, 0);
      console.log('Total from inspectionReports.length:', total);
      console.log('Sum of daily counts:', sumOfDailyCounts);
      console.log('Match:', total === sumOfDailyCounts ? 'YES ✓' : 'NO ✗');
      
      donutBreakdown = [{ value: total, color: '#A18AFF', text: 'Total Reports' }];
      setDonutCenter({ value: total, label: 'Total' });
    } else if (moduleTab === 'jobCards') {
      data = dates.map(d => {
        const targetDate = normalizeDate(d);
        const count = jobCards.filter(j => {
          const cardDate = normalizeDate(j.date);
          return cardDate === targetDate;
        }).length;
        return {
          value: count,
          label: `${d.getDate()}/${d.getMonth() + 1}`,
          date: d,
          frontColor: '#A18AFF',
          sideColor: '#8000FF',
          barWidth: 22,
          topLabelComponent: () => null,
        };
      });
      const total = jobCards.length;
      donutBreakdown = [{ value: total, color: '#A18AFF', text: 'Total Job Cards' }];
      setDonutCenter({ value: total, label: 'Total' });
    } else if (moduleTab === 'maintenance') {
      data = dates.map(d => {
        const targetDate = normalizeDate(d);
        const count = maintenanceReports.filter(r => {
          const reportDate = normalizeDate(r.date);
          return reportDate === targetDate;
        }).length;
        return {
          value: count,
          label: `${d.getDate()}/${d.getMonth() + 1}`,
          date: d,
          frontColor: '#A18AFF',
          sideColor: '#8000FF',
          barWidth: 22,
          topLabelComponent: () => null,
        };
      });
      const total = maintenanceReports.length;
      donutBreakdown = [{ value: total, color: '#A18AFF', text: 'Total Maintenance' }];
      setDonutCenter({ value: total, label: 'Total' });
    } else if (moduleTab === 'serviceOrders') {
      data = dates.map(d => {
        const targetDate = normalizeDate(d);
        const count = serviceOrders.filter(r => {
          const reportDate = normalizeDate(r.date);
          return reportDate === targetDate;
        }).length;
        return {
          value: count,
          label: `${d.getDate()}/${d.getMonth() + 1}`,
          date: d,
          frontColor: '#A18AFF',
          sideColor: '#8000FF',
          barWidth: 22,
          topLabelComponent: () => null,
        };
      });
      const total = serviceOrders.length;
      donutBreakdown = [{ value: total, color: '#A18AFF', text: 'Total Service Orders' }];
      setDonutCenter({ value: total, label: 'Total' });
    }

    setBarData(data);
    setDonutData(donutBreakdown);
  }, [kpiMode, moduleTab, dateRange, inspectionReports, jobCards, maintenanceReports, serviceOrders]);

  // Set dateRange to last 7 days when switching to Module KPI
  useEffect(() => {
    if (kpiMode === 'module') {
      const today = new Date();
      const from = new Date(today);
      from.setDate(today.getDate() - 6); // last 7 days including today
      setDateRange({ from, to: today });
    }
  }, [kpiMode]);

  const filterByCountryProject = (member) => {
    const allowedCountries = Array.isArray(myCountry) ? myCountry : [myCountry];
    const allowedProjects = Array.isArray(myProject) ? myProject : [myProject];
    const countryMatch = allowedCountries.length === 0 || (member.countries && member.countries.some(c => allowedCountries.includes(c)));
    const projectMatch = allowedProjects.length === 0 || (member.projects && member.projects.some(p => allowedProjects.includes(p)));
    return countryMatch && projectMatch;
  };

  const processTeamMembers = (members) => {
    const filtered = members.filter(filterByCountryProject);
    let total = 0;
    let supervisors = 0;
    let operators = 0;
    const membersArray = [];
    filtered.forEach((memberData) => {
      const inspectionCount = memberData.inspections || 0;
      const role = memberData.role || 'Employee';
      const name = memberData.fullName || memberData.name || 'Unknown';
      membersArray.push({
        id: memberData.id,
        name,
        role,
        inspections: inspectionCount,
      });
      total += inspectionCount;
      if (role === 'Supervisor') supervisors++;
      if (!['Supervisor', 'Admin', 'Manager'].includes(role)) operators++;
    });
    membersArray.sort((a, b) => a.name.localeCompare(b.name));
    setTeamMembers(membersArray);
    setTotalInspections(total);
    setSupervisorCount(supervisors);
    setOperatorCount(operators);
  };

  const handleLockPress = async () => {
    setModalVisible(true);
    setModalLoading(true);
    setModalError(null);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setModalError('User ID not found. Please log in again.');
        setModalLoading(false);
        return;
      }

      // Fetch user details from MongoDB
      const userResponse = await fetch(API_ENDPOINTS.USERS.GET_BY_ID(userUid));
      const userData = await userResponse.json();
      
      if (!userData.success || !userData.data) {
        setModalError('User details not found.');
      } else {
        const data = userData.data;
        const country = Array.isArray(data.countries) && data.countries.length > 0
          ? data.countries.join(', ')
          : (data.selectedCountry || data.country || 'N/A');
        const project = Array.isArray(data.projects) && data.projects.length > 0
          ? data.projects.join(', ')
          : (data.selectedProject || data.project || 'N/A');
        let teamEmployPerms = [];
        if (Array.isArray(data.modules)) {
          const teamEmployModule = data.modules.find(m => m.module === 'teamEmploy');
          if (teamEmployModule && teamEmployModule.permissions) {
            teamEmployPerms = teamEmployModule.permissions;
          }
        }
        setUserDetails({ country, project, teamEmployPermissions: teamEmployPerms });
      }
    } catch (err) {
      console.error('Error fetching user details:', err);
      setModalError('Failed to fetch user details.');
    }
    setModalLoading(false);
  };

  const renderPermissionModal = () => (
    <Modal visible={modalVisible} transparent animationType="fade">
      <View style={styles.modalContainer}>
        <View style={styles.modalContent}>
          <Text style={styles.modalTitle}>User Permissions</Text>
          {modalLoading ? (
            <ActivityIndicator size="large" color="#015185" />
          ) : modalError ? (
            <Text style={styles.errorText}>{modalError}</Text>
          ) : userDetails ? (
            <View style={styles.permissionContainer}>
              <Text style={styles.permissionLabel}>Country:</Text>
              <Text style={styles.permissionValue}>{userDetails.country}</Text>
              <Text style={styles.permissionLabel}>Project:</Text>
              <Text style={styles.permissionValue}>{userDetails.project}</Text>
              <Text style={styles.permissionLabel}>Team Employ Permissions:</Text>
              <Text style={styles.permissionValue}>
                {userDetails.teamEmployPermissions && userDetails.teamEmployPermissions.length > 0
                  ? userDetails.teamEmployPermissions.join(', ')
                  : 'None'}
              </Text>
            </View>
          ) : null}
          <TouchableOpacity
            style={styles.closeButton}
            onPress={() => setModalVisible(false)}
          >
            <Text style={styles.closeButtonText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  function getDateArray(start, end) {
    const arr = [];
    let dt = new Date(start);
    while (dt <= end) {
      arr.push(new Date(dt));
      dt.setDate(dt.getDate() + 1);
    }
    return arr;
  }

  const maxY = barData.length > 0 ? Math.max(2, Math.ceil(Math.max(...barData.map(d => d.value)) / 2) * 2) : 2;
  const yAxisLabels = Array.from({ length: maxY / 2 }, (_, i) => (i + 1) * 2);

  // Table data based on selectedRole (all users for the role)
  const allDisplayedUsers = selectedRole === 'All'
    ? filteredUsers
    : filteredUsers.filter((user) => (user.role || '') === selectedRole);

  // Pagination calculations
  const totalUsers = allDisplayedUsers.length;
  const totalPages = Math.ceil(totalUsers / itemsPerPage);
  const startIndex = (currentPage - 1) * itemsPerPage;
  const endIndex = startIndex + itemsPerPage;
  const displayedUsers = allDisplayedUsers.slice(startIndex, endIndex);

  // Pagination handlers
  const handlePreviousPage = () => {
    if (currentPage > 1) {
      setCurrentPage(currentPage - 1);
    }
  };

  const handleNextPage = () => {
    if (currentPage < totalPages) {
      setCurrentPage(currentPage + 1);
    }
  };



  // Update donutData for the concentric progress chart
  useEffect(() => {
    if (kpiMode !== 'module' || !dateRange.from || !dateRange.to) {
      setBarData([]);
      setDonutData([]);
      setDonutCenter({ value: 0, label: '' });
      return;
    }
    let totalReports = 0, totalEquipment = 0, totalUsers = 0;
    if (moduleTab === 'inspections') {
      totalReports = inspectionReports.length;
      totalEquipment = inspectionEquipmentCount;
      totalUsers = userDetailsList.length;
    } else if (moduleTab === 'jobCards') {
      totalReports = jobCards.length;
      totalEquipment = jobCardEquipmentCount;
      totalUsers = jobCardUserDetails.length;
    } else if (moduleTab === 'maintenance') {
      totalReports = maintenanceReports.length;
      totalEquipment = maintenanceEquipmentCount;
      totalUsers = maintenanceUserDetails.length;
    } else if (moduleTab === 'serviceOrders') {
      totalReports = serviceOrders.length;
      totalEquipment = serviceOrderEquipmentCount;
      totalUsers = serviceOrderUserDetails.length;
    }
    
    // Calculate percentages for concentric chart (similar to the image)
    const maxValue = Math.max(totalReports, totalEquipment, totalUsers, 1);
    const reportsPercentage = Math.round((totalReports / maxValue) * 100);
    const equipmentPercentage = Math.round((totalEquipment / maxValue) * 100);
    const usersPercentage = Math.round((totalUsers / maxValue) * 100);
    
    const donutBreakdown = [
      { percentage: reportsPercentage, count: totalReports, color: '#2267A2', text: 'Reports' },
      { percentage: equipmentPercentage, count: totalEquipment, color: '#D81B60', text: 'Equipment' },
      { percentage: usersPercentage, count: totalUsers, color: '#2ECC40', text: 'Users' },
    ];
    setDonutData(donutBreakdown);
    setDonutCenter({ value: totalReports + totalEquipment + totalUsers, label: 'Total' });
  }, [kpiMode, moduleTab, dateRange, inspectionReports, jobCards, maintenanceReports, serviceOrders, inspectionEquipmentCount, jobCardEquipmentCount, maintenanceEquipmentCount, serviceOrderEquipmentCount, userDetailsList, jobCardUserDetails, maintenanceUserDetails, serviceOrderUserDetails]);

  return (
    <View style={styles.container}>
      <SyncModal visible={roleLoading || (filteredUsers.length === 0 && allUsers.length === 0)} message="Loading..." />
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeftIcon}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Supervisor KPI</Text>
          <View style={styles.headerRightIcons}>
            <TouchableOpacity onPress={() => alert('Cloud sync pressed')} style={styles.headerIconButton}>
              <MaterialIcons name="cloud" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <TouchableOpacity onPress={handleLockPress} style={styles.headerIconButton} accessibilityLabel="Show user permissions" accessibilityRole="button">
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>
      <ScrollView contentContainerStyle={[styles.body, { paddingTop: 0, paddingBottom: 0 }]}>
        {/* Replace the old modeSelector/tabSelector with the new pill-shaped tab bar */}
        <View style={styles.tabSelectorBar}>
          <TouchableOpacity
            style={[
              styles.tabButtonBar,
              kpiMode === 'role' && styles.activeTabButtonBar,
            ]}
            onPress={() => setKpiMode('role')}
          >
            <Text style={[
              styles.tabButtonTextBar,
              kpiMode === 'role' && styles.activeTabButtonTextBar,
            ]}>
              Role KPI
            </Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[
              styles.tabButtonBar,
              kpiMode === 'module' && styles.activeTabButtonBar,
            ]}
            onPress={() => setKpiMode('module')}
          >
            <Text style={[
              styles.tabButtonTextBar,
              kpiMode === 'module' && styles.activeTabButtonTextBar,
            ]}>
              Module KPI
            </Text>
          </TouchableOpacity>
        </View>
        {kpiMode === 'role' ? (
          <>
            {/* Filter by Role Dropdown */}
            <View style={{ width: '92%', alignSelf: 'center', marginTop: 8, marginBottom: 8 }}>
              <Text style={{ fontSize: width * 0.045, fontWeight: '600', color: '#025383', marginBottom: width * 0.01 }}>Filter by Role:</Text>
              <DropDownPicker
                open={dropdownOpen}
                value={selectedRole}
                items={dropdownItems}
                setOpen={setDropdownOpen}
                setValue={setSelectedRole}
                setItems={setDropdownItems}
                placeholder="Select Role"
                style={{ borderColor: '#015185', borderRadius: 8, minHeight: 48 }}
                containerStyle={{ marginBottom: 0 }}
                dropDownContainerStyle={{ borderColor: '#015185', borderRadius: 8 }}
                textStyle={{ fontSize: width * 0.04, color: '#015185' }}
                placeholderStyle={{ color: '#888', fontSize: width * 0.04 }}
                listMode="SCROLLVIEW"
                zIndex={10000}
              />
            </View>
            {/* Table */}
            <ScrollView horizontal showsHorizontalScrollIndicator={false} style={{ width: '100%' }}>
              <View style={[styles.tableWrapperEnhanced, { minWidth: 700, marginTop: 0, marginBottom: 8 }]}>
                <LinearGradient
                  colors={['#3481BC', '#025383']}
                  start={{ x: 0, y: 0 }}
                  end={{ x: 1, y: 0 }}
                  style={styles.tableHeaderRowEnhanced}
                >
                  <Text style={[styles.tableHeaderCellEnhanced, { width: 60, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>S.No</Text>
                  <Text style={[styles.tableHeaderCellEnhanced, { width: 150, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>Employee Number</Text>
                  <Text style={[styles.tableHeaderCellEnhanced, { width: 200, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>Name</Text>
                  <Text style={[styles.tableHeaderCellEnhanced, { width: 250 }]}>Email</Text>
                </LinearGradient>
                {roleLoading ? (
                  <View style={styles.emptyRow}><Text style={styles.emptyText}>Loading...</Text></View>
                ) : displayedUsers.length > 0 ? (
                  displayedUsers.map((user, idx) => (
                    <TouchableOpacity
                      key={user.id}
                      activeOpacity={0.85}
                      style={[
                        styles.tableRowEnhanced,
                        idx % 2 === 0 ? styles.tableRowEvenEnhanced : styles.tableRowOddEnhanced,
                      ]}
                    >
                      <Text style={[styles.tableCellEnhanced, { width: 60, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>{startIndex + idx + 1}</Text>
                      <Text style={[styles.tableCellEnhanced, { width: 150, borderRightWidth: 1, borderColor: '#d0d7e2' }]}>{user.employeeNumber || '-'}</Text>
                      <Text style={[styles.tableCellEnhanced, { width: 200, color: '#2267A2', fontWeight: '700', borderRightWidth: 1, borderColor: '#d0d7e2' }]}>{user.fullName || user.name || 'Unknown'}</Text>
                      <Text style={[styles.tableCellEnhanced, { width: 250, color: '#025383', fontWeight: '500' }]}>{user.email || '-'}</Text>
                    </TouchableOpacity>
                  ))
                ) : (
                  <View style={styles.emptyRow}>
                    <Text style={styles.emptyText}>No users found for this role.</Text>
                  </View>
                )}
              </View>
            </ScrollView>
            
            {/* Pagination Controls */}
            {totalUsers > 0 && (
              <View style={{
                width: '92%',
                alignSelf: 'center',
                marginTop: 16,
                flexDirection: 'row',
                justifyContent: 'space-between',
                alignItems: 'center',
                paddingHorizontal: 8,
              }}>
                {/* Page Info */}
                <Text style={{ fontSize: width * 0.038, color: '#025383', fontWeight: '600' }}>
                  Showing {startIndex + 1}-{Math.min(endIndex, totalUsers)} of {totalUsers}
                </Text>
                
                {/* Navigation Buttons */}
                <View style={{ flexDirection: 'row', alignItems: 'center', gap: 12 }}>
                  <TouchableOpacity
                    onPress={handlePreviousPage}
                    disabled={currentPage === 1}
                    style={{
                      paddingHorizontal: 16,
                      paddingVertical: 8,
                      backgroundColor: currentPage === 1 ? '#d0d0d0' : '#2267A2',
                      borderRadius: 6,
                      minWidth: 80,
                      alignItems: 'center',
                    }}
                  >
                    <Text style={{
                      color: currentPage === 1 ? '#888' : '#fff',
                      fontWeight: '600',
                      fontSize: width * 0.038,
                    }}>
                      Previous
                    </Text>
                  </TouchableOpacity>

                  <Text style={{ fontSize: width * 0.04, color: '#025383', fontWeight: '700' }}>
                    Page {currentPage} of {totalPages}
                  </Text>

                  <TouchableOpacity
                    onPress={handleNextPage}
                    disabled={currentPage === totalPages}
                    style={{
                      paddingHorizontal: 16,
                      paddingVertical: 8,
                      backgroundColor: currentPage === totalPages ? '#d0d0d0' : '#2267A2',
                      borderRadius: 6,
                      minWidth: 80,
                      alignItems: 'center',
                    }}
                  >
                    <Text style={{
                      color: currentPage === totalPages ? '#888' : '#fff',
                      fontWeight: '600',
                      fontSize: width * 0.038,
                    }}>
                      Next
                    </Text>
                  </TouchableOpacity>
                </View>
              </View>
            )}
          </>
        ) : (
          <View style={styles.moduleContent}>
            <ScrollView 
              horizontal 
              showsHorizontalScrollIndicator={false}
              contentContainerStyle={styles.tabSelector}
            >
              {MODULE_TABS.map(tab => (
                <TouchableOpacity
                  key={tab.key}
                  style={[styles.tabButton, moduleTab === tab.key && styles.activeTabButton]}
                  onPress={() => setModuleTab(tab.key)}
                >
                  <Text style={[styles.tabText, moduleTab === tab.key && styles.activeTabText]}>{tab.label}</Text>
                </TouchableOpacity>
              ))}
            </ScrollView>
            <View style={styles.datePickerContainer}>
              <TouchableOpacity
                style={styles.dateButton}
                onPress={() => { setDatePickerMode('from'); setDatePickerVisible(true); }}
              >
                <Text style={styles.dateText}>
                  {dateRange.from ? `From: ${dateRange.from.toLocaleDateString()}` : 'Select From Date'}
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.dateButton, !dateRange.from && styles.disabledDateButton]}
                onPress={() => { setDatePickerMode('to'); setDatePickerVisible(true); }}
                disabled={!dateRange.from}
              >
                <Text style={[styles.dateText, !dateRange.from && styles.disabledDateText]}>
                  {dateRange.to ? `To: ${dateRange.to.toLocaleDateString()}` : 'Select To Date'}
                </Text>
              </TouchableOpacity>
              {(dateRange.from || dateRange.to) && (
                <TouchableOpacity style={styles.clearButton} onPress={() => setDateRange({ from: null, to: null })}>
                  <Text style={styles.clearText}>Clear</Text>
                </TouchableOpacity>
              )}
            </View>
            <View style={{ marginBottom: 16, backgroundColor: '#F5F7FA', borderRadius: 8, padding: 12, alignItems: 'center', borderWidth: 1, borderColor: '#E0E0E0' }}>
              {moduleTab === 'inspections' && (
                loadingInspectionsSummary ? (
                  <ActivityIndicator size="small" color="#015185" style={{ marginVertical: 12 }} />
                ) : (
                  <>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 16 }}>
                      Total Inspection Reports: {inspectionReports.length}
                    </Text>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 15 }}>
                      Total Equipment Used for Inspection: {inspectionEquipmentCount}
                    </Text>
                    <Text style={{ color: '#2267A2', fontWeight: '700', fontSize: 15 }}>
                      Total Users: {userDetailsList.length} ({Array.from(new Set(userDetailsList.map(u => u.role))).join(', ')})
                    </Text>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 15, marginTop: 4 }}>
                      Users who created reports in selected range: {userDetailsList.length}
                    </Text>
                    <View style={{ marginTop: 2, alignItems: 'flex-start' }}>
                      {userDetailsList.map((user, idx) => (
                        <Text key={user.id} style={{ color: '#444', fontSize: 13 }}>
                          {idx + 1}. {user.name} ({user.role}): {user.count} reports
                        </Text>
                      ))}
                    </View>
                  </>
                )
              )}
              {moduleTab === 'jobCards' && (
                loadingJobCardsSummary ? (
                  <ActivityIndicator size="small" color="#015185" style={{ marginVertical: 12 }} />
                ) : (
                  <>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 16 }}>
                      Total Job Cards: {jobCards.length}
                    </Text>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 15 }}>
                      Total Equipment Used for Job Cards: {jobCardEquipmentCount}
                    </Text>
                    <Text style={{ color: '#2267A2', fontWeight: '700', fontSize: 15 }}>
                      Total Users: {jobCardUserDetails.length} ({Array.from(new Set(jobCardUserDetails.map(u => u.role))).join(', ')})
                    </Text>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 15, marginTop: 4 }}>
                      Users who created job cards in selected range: {jobCardUserDetails.length}
                    </Text>
                    <View style={{ marginTop: 2, alignItems: 'flex-start' }}>
                      {jobCardUserDetails.map((user, idx) => (
                        <Text key={user.id} style={{ color: '#444', fontSize: 13 }}>
                          {idx + 1}. {user.name} ({user.role}): {user.count} job cards
                        </Text>
                      ))}
                    </View>
                  </>
                )
              )}
              {moduleTab === 'maintenance' && (
                loadingMaintenanceSummary ? (
                  <ActivityIndicator size="small" color="#015185" style={{ marginVertical: 12 }} />
                ) : (
                  <>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 16 }}>
                      Total Maintenance Reports: {maintenanceReports.length}
                    </Text>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 15 }}>
                      Total Equipment Used for Maintenance: {maintenanceEquipmentCount}
                    </Text>
                    <Text style={{ color: '#2267A2', fontWeight: '700', fontSize: 15 }}>
                      Total Users: {maintenanceUserDetails.length} ({Array.from(new Set(maintenanceUserDetails.map(u => u.role))).join(', ')})
                    </Text>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 15, marginTop: 4 }}>
                      Users who created maintenance reports in selected range: {maintenanceUserDetails.length}
                    </Text>
                    <View style={{ marginTop: 2, alignItems: 'flex-start' }}>
                      {maintenanceUserDetails.map((user, idx) => (
                        <Text key={user.id} style={{ color: '#444', fontSize: 13 }}>
                          {idx + 1}. {user.name} ({user.role}): {user.count} reports
                        </Text>
                      ))}
                    </View>
                  </>
                )
              )}
              {moduleTab === 'serviceOrders' && (
                loadingServiceOrdersSummary ? (
                  <ActivityIndicator size="small" color="#015185" style={{ marginVertical: 12 }} />
                ) : (
                  <>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 16 }}>
                      Total Service Orders: {serviceOrders.length}
                    </Text>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 15 }}>
                      Total Equipment Used for Service Orders: {serviceOrderEquipmentCount}
                    </Text>
                    <Text style={{ color: '#2267A2', fontWeight: '700', fontSize: 15 }}>
                      Total Users: {serviceOrderUserDetails.length} ({Array.from(new Set(serviceOrderUserDetails.map(u => u.role))).join(', ')})
                    </Text>
                    <Text style={{ color: '#015185', fontWeight: '700', fontSize: 15, marginTop: 4 }}>
                      Users who created service orders in selected range: {serviceOrderUserDetails.length}
                    </Text>
                    <View style={{ marginTop: 2, alignItems: 'flex-start' }}>
                      {serviceOrderUserDetails.map((user, idx) => (
                        <Text key={user.id} style={{ color: '#444', fontSize: 13 }}>
                          {idx + 1}. {user.name} ({user.role}): {user.count} service orders
                        </Text>
                      ))}
                    </View>
                  </>
                )
              )}
            </View>
            <View style={styles.chartSection}>
              <View style={styles.chartCard}>
                <Text style={styles.chartTitle}>{MODULE_TABS.find(t => t.key === moduleTab)?.label}</Text>
                <Text style={styles.chartSubtitle}>{dateRange.from && dateRange.to ? `${dateRange.from.toLocaleDateString()} - ${dateRange.to.toLocaleDateString()}` : ''}</Text>
                <ScrollView horizontal showsHorizontalScrollIndicator={false}>
                  {barData.length > 0 ? (
                    <BarChart
                      data={barData}
                      barWidth={22}
                      spacing={18}
                      roundedTop
                      roundedBottom
                      yAxisThickness={1}
                      xAxisColor="#888"
                      yAxisTextStyle={{ color: '#888' }}
                      xAxisLabelTextStyle={{ color: '#888' }}
                      height={220}
                      noOfSections={yAxisLabels.length}
                      maxValue={maxY}
                      yAxisLabelTexts={yAxisLabels.map(String)}
                      showLine
                      lineConfig={{
                        color: '#8000FF',
                        thickness: 2,
                        curved: true,
                        data: barData.map(d => ({ value: d.value })),
                        showDataPoints: true,
                        dataPointsColor: '#A18AFF',
                        dataPointsRadius: 5,
                      }}
                      frontColor="#A18AFF"
                      gradientColor="#E0C3FC"
                      style={{ minWidth: Math.max(barData.length * 40, 320) }}
                      xAxisLabelTexts={barData.map(d => d.label)}
                      onPress={(bar, index) => {
                        if (selectedBar && selectedBar.index === index) {
                          setSelectedBar(null);
                        } else {
                          setSelectedBar({ ...bar, index, date: bar.date || new Date(bar.label.split('/').reverse().join('/')) });
                        }
                      }}
                    />
                  ) : <Text style={styles.noDataText}>No data to display</Text>}
                  {selectedBar && (
                    <Text style={styles.selectedBarText}>
                      On {selectedBar.date.toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric' })}, total inspections: {selectedBar.value}
                    </Text>
                  )}
                </ScrollView>
              </View>
              <View style={styles.chartCard}>
                <Text style={styles.chartTitle}>Performance Overview</Text>
                <Text style={styles.chartSubtitle}>Progress indicators for Reports, Equipment, and Users</Text>
                {donutData.length > 0 ? (
                  <View style={styles.concentricChartContainer}>
                    <ConcentricProgressChart data={donutData} size={180} />
                  </View>
                ) : <Text style={styles.noDataText}>No data to display</Text>}
              </View>
            </View>
          </View>
        )}
      </ScrollView>
      {renderPermissionModal()}
      <Toast />
      <DateTimePickerModal
        isVisible={datePickerVisible}
        mode="date"
        onConfirm={date => {
          if (datePickerMode === 'from') {
            setDateRange(r => ({ ...r, from: date, to: null }));
          } else {
            setDateRange(r => ({ ...r, to: date }));
          }
          setDatePickerVisible(false);
        }}
        onCancel={() => setDatePickerVisible(false)}
      />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#F5F7FA',
  },
  header: {
    width: '100%',
    justifyContent: 'flex-end',
    paddingBottom: width * 0.04,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.05,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeftIcon: {
    paddingVertical: width * 0.02,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '700',
    flex: 1,
    marginLeft: width * 0.03,
  },
  headerRightIcons: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerIconButton: {
    marginLeft: width * 0.03,
    padding: width * 0.02,
  },
  body: {
    flexGrow: 1,
    alignItems: 'center',
    justifyContent: 'flex-start',
    paddingTop: 0,
    paddingBottom: 0,
    backgroundColor: '#fff',
  },
  modeSelector: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginBottom: 16,
  },
  modeButton: {
    paddingVertical: 10,
    paddingHorizontal: 24,
    borderRadius: 24,
    backgroundColor: '#E0E0E0',
    marginHorizontal: 8,
  },
  activeModeButton: {
    backgroundColor: '#015185',
    elevation: 2,
  },
  modeText: {
    fontSize: 16,
    fontWeight: '700',
    color: '#015185',
  },
  activeModeText: {
    color: '#FFF',
  },
  tableContainer: {
    width: '100%',
    marginBottom: 16,
  },
  tableHeader: {
    flexDirection: 'row',
    backgroundColor: '#F5F5F5',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
  },
  tableHeaderCell: {
    fontSize: 16,
    fontWeight: '700',
    color: '#444',
    textAlign: 'center',
    flex: 1,
  },
  tableRow: {
    flexDirection: 'row',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#F0F0F0',
  },
  tableCell: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    flex: 1,
  },
  supervisorRole: {
    color: '#2267A2',
    fontWeight: '600',
  },
  operatorRole: {
    color: '#D81B60',
    fontWeight: '600',
  },
  defaultRole: {
    color: '#444',
    fontWeight: '500',
  },
  tableWrapperEnhanced: {
    width: '92%',
    backgroundColor: '#f8f8f8',
    alignSelf: 'flex-start',
    marginTop: 16,
    marginBottom: 16,
    overflow: 'hidden',
    borderWidth: 1,
    borderColor: '#d0d7e2',
    shadowColor: '#3481BC',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.12,
    shadowRadius: 8,
    elevation: 6,
  },
  tableHeaderRowEnhanced: {
    flexDirection: 'row',
    paddingVertical: width * 0.035,
    alignItems: 'center',
    justifyContent: 'center',
    borderBottomWidth: 1,
    borderColor: '#d0d7e2',
  },
  tableHeaderCellEnhanced: {
    fontWeight: 'bold',
    color: '#fff',
    fontSize: width * 0.043,
    textAlign: 'center',
    letterSpacing: 0.5,
    paddingVertical: 6,
    paddingHorizontal: 4,
  },
  tableRowEnhanced: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.032,
    borderBottomWidth: 1,
    borderBottomColor: '#d0d7e2',
    backgroundColor: '#fafdff',
  },
  tableRowEvenEnhanced: {
    backgroundColor: '#fafdff',
  },
  tableRowOddEnhanced: {
    backgroundColor: '#eaf3fa',
  },
  tableCellEnhanced: {
    textAlign: 'center',
    fontSize: width * 0.041,
    color: '#333',
    paddingVertical: 8,
    paddingHorizontal: 6,
  },
  emptyRow: {
    padding: width * 0.04,
    alignItems: 'center',
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    maxWidth: 400,
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  tabSelector: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginBottom: 16,
    paddingHorizontal: 16,
  },
  tabButton: {
    paddingVertical: 8,
    paddingHorizontal: 16,
    borderRadius: 18,
    backgroundColor: '#E0E0E0',
    marginHorizontal: 4,
    minWidth: 100,
    alignItems: 'center',
  },
  activeTabButton: {
    backgroundColor: '#015185',
    elevation: 2,
  },
  tabText: {
    fontSize: 14,
    fontWeight: '700',
    color: '#015185',
  },
  activeTabText: {
    color: '#FFF',
  },
  datePickerContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 16,
  },
  dateButton: {
    flex: 1,
    backgroundColor: '#F0F0F0',
    borderRadius: 8,
    padding: 12,
    marginRight: 8,
    alignItems: 'center',
  },
  disabledDateButton: {
    opacity: 0.5,
  },
  dateText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  disabledDateText: {
    color: '#AAA',
  },
  clearButton: {
    padding: 12,
    backgroundColor: '#D81B60',
    borderRadius: 8,
  },
  clearText: {
    fontSize: 14,
    color: '#FFF',
    fontWeight: '600',
  },
  chartSection: {
    width: '92%',
    alignSelf: 'center',
  },
  chartCard: {
    backgroundColor: '#FFF',
    borderRadius: 8,
    padding: 16,
    marginBottom: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  chartTitle: {
    fontSize: 18,
    fontWeight: '700',
    color: '#333',
    marginBottom: 8,
  },
  chartSubtitle: {
    fontSize: 14,
    color: '#888',
    marginBottom: 12,
  },
  noDataText: {
    textAlign: 'center',
    color: '#888',
    marginTop: 20,
  },
  centerLabel: {
    alignItems: 'center',
  },
  centerValue: {
    fontSize: 20,
    fontWeight: '700',
    color: '#015185',
  },
  centerLabelText: {
    fontSize: 14,
    fontWeight: '500',
    color: '#015185',
  },
  legendContainer: {
    marginTop: 12,
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'center',
  },
  legendItem: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 8,
    marginBottom: 4,
  },
  legendColor: {
    width: 12,
    height: 12,
    borderRadius: 6,
    marginRight: 6,
  },
  legendText: {
    fontSize: 12,
    fontWeight: '600',
  },
  moduleContent: {
    width: '100%',
  },
  modalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '70%',
    backgroundColor: '#FFF',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    maxWidth: 400,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '700',
    color: '#333',
    marginBottom: 16,
  },
  errorText: {
    color: '#FF6B6B',
    fontSize: 14,
    textAlign: 'center',
    marginBottom: 12,
  },
  permissionContainer: {
    width: '100%',
  },
  permissionLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 4,
  },
  permissionValue: {
    fontSize: 14,
    color: '#333',
    marginBottom: 8,
    textAlign: 'center',
  },
  closeButton: {
    backgroundColor: '#3481BC',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 8,
    marginTop: 16,
  },
  closeButtonText: {
    fontSize: 16,
    fontWeight: '700',
    color: '#FFF',
  },
  selectedBarText: {
    textAlign: 'center',
    color: '#015185',
    fontWeight: '700',
    marginTop: 8,
    fontSize: 16,
  },
  concentricChartContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    position: 'relative',
    marginVertical: 20,
  },

  // Add modern tab bar styles
  tabSelectorBar: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'center',
    marginVertical: 16,
    backgroundColor: '#F0F4F8',
    borderRadius: 24,
    padding: 4,
    alignSelf: 'center',
  },
  tabButtonBar: {
    flex: 1,
    paddingVertical: 10,
    paddingHorizontal: 24,
    borderRadius: 20,
    backgroundColor: 'transparent',
    marginHorizontal: 4,
    alignItems: 'center',
    justifyContent: 'center',
  },
  activeTabButtonBar: {
    backgroundColor: '#015185',
    shadowColor: '#015185',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 4,
    elevation: 2,
  },
  tabButtonTextBar: {
    color: '#015185',
    fontWeight: '700',
    fontSize: 16,
  },
  activeTabButtonTextBar: {
    color: '#fff',
  },
});

export default SupervisorKpi;