import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  StatusBar,
  Modal,
  FlatList,
  TouchableWithoutFeedback,
  Animated,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import DateTimePicker from '@react-native-community/datetimepicker';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import { Dropdown } from 'react-native-element-dropdown';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;

// MongoDB API URLs
const MONGODB_SERVICE_SCHEDULE_API_URL = 'https://api.titandrillingzm.com:6010';
const MONGODB_ADMIN_USERS_API_URL = 'https://api.titandrillingzm.com:6003';

const CustomInputField = ({ label, value, onChangeText, isRequired = false, showError = false, errorMessage = '', style, ...props }) => (
  <View style={[inputStyles.container, style]}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
    </Text>
    <TextInput
      style={[inputStyles.input, showError ? inputStyles.inputError : null]}
      value={value}
      onChangeText={onChangeText}
      placeholderTextColor="#888"
      {...props}
    />
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

const UNIT_OPTIONS = ['Miles', 'KM', 'Hours'];
const TIME_UNIT_OPTIONS = ['Day', 'Week', 'Month', 'Year'];

const AddSchedule = () => {
  const navigation = useNavigation();
  const [repeatType, setRepeatType] = useState('Repeat');
  const [intervals, setIntervals] = useState([
    { checked: true, value: '', unit: 'Miles' },
    { checked: false, value: '', unit: 'Miles' },
  ]);
  const [intervalUnit, setIntervalUnit] = useState('Day');
  const [notifyBefore, setNotifyBefore] = useState({ value: '', unit: 'Miles', time: 'Day' });
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [scheduleTitle, setScheduleTitle] = useState('');
  const [note, setNote] = useState('');
  const [lastServiceDate, setLastServiceDate] = useState(new Date());
  const [tempDate, setTempDate] = useState(lastServiceDate);
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [scheduleId, setScheduleId] = useState('');
  const [showErrors, setShowErrors] = useState(false);
  const [meterReading, setMeterReading] = useState('');
  const [meterUnit, setMeterUnit] = useState('Miles');
  const [tasks, setTasks] = useState([]);
  const [taskModalVisible, setTaskModalVisible] = useState(false);
  const [newTask, setNewTask] = useState('');
  const [intervalDropdownIndex, setIntervalDropdownIndex] = useState(null);
  const [meterUnitDropdownVisible, setMeterUnitDropdownVisible] = useState(false);

  const [isLoading, setIsLoading] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const animationRef = useRef(null);

  // New state for country and project dropdowns
  const [selectedCountry, setSelectedCountry] = useState('');
  const [selectedProject, setSelectedProject] = useState('');
  const [userCountries, setUserCountries] = useState([]);
  const [userProjects, setUserProjects] = useState([]);

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Project mapping for all possible countries
  const projectMap = {
    Zambia: [
      { label: 'Kansanshi', value: 'Kansanshi' },
      { label: 'Kalumbila', value: 'Kalumbila' },
      { label: 'Kimteto', value: 'Kimteto' },
      { label: 'Kobold', value: 'Kobold' },
      { label: 'FQM Exploration', value: 'FQM Exploration' },
      { label: 'Mimosa', value: 'Mimosa' },
    ],
    DRC: [
      { label: 'Lubumbashi', value: 'Lubumbashi' },
      { label: 'Musompo', value: 'Musompo' },
      { label: 'IME', value: 'IME' },
      { label: 'Kamoa', value: 'Kamoa' },
    ],
  };

  // Generate country options from user's countries
  const getCountryOptions = () => {
    console.log('getCountryOptions called with userCountries:', userCountries);
    const options = userCountries.map(country => ({
      label: country,
      value: country
    }));
    console.log('Country options generated:', options);
    return options;
  };

  const getProjectOptions = () => {
    if (!selectedCountry) {
      console.log('getProjectOptions: No country selected');
      return [];
    }
    
    console.log('getProjectOptions called with selectedCountry:', selectedCountry, 'userProjects:', userProjects);
    
    // Filter projects based on user's projects for the selected country
    const allProjectsForCountry = projectMap[selectedCountry] || [];
    console.log('All projects for country', selectedCountry, ':', allProjectsForCountry);
    
    const userProjectsForCountry = allProjectsForCountry.filter(project => 
      userProjects.includes(project.value)
    );
    
    console.log('Filtered user projects for country:', userProjectsForCountry);
    return userProjectsForCountry;
  };

  // Fetch user data and generate schedule ID
  useEffect(() => {
    const fetchUserData = async () => {
      try {
        const uid = await AsyncStorage.getItem('userUid');
        if (!uid) {
          showToast('error', 'Error', 'User not authenticated.');
          navigation.navigate('AdminLogin');
          return;
        }

        console.log('Fetching user data for uid:', uid);
        
        // Use MongoAPI like ServiceScheduleHome
        const response = await MongoAPI.getUserById(uid);
        console.log('User response:', response);
        
        if (!response.success || !response.data) {
          console.log('User not found in database');
          showToast('error', 'Error', 'User data not found.');
          return;
        }

        const userData = response.data;
        console.log('User data received:', {
          name: userData.name,
          email: userData.email,
          countries: userData.countries,
          projects: userData.projects
        });
        
        // Normalize countries and projects to arrays
        const countries = Array.isArray(userData.countries) ? userData.countries : 
                        (userData.countries ? Object.values(userData.countries) : []);
        const projects = Array.isArray(userData.projects) ? userData.projects : 
                        (userData.projects ? Object.values(userData.projects) : []);
        
        console.log('User countries from MongoDB:', countries);
        console.log('User projects from MongoDB:', projects);
        
        setUserCountries(countries);
        setUserProjects(projects);
        
        // Set default country and project if user has only one
        if (countries.length === 1) {
          setSelectedCountry(countries[0]);
          console.log('Auto-selected country:', countries[0]);
        }
        if (projects.length === 1) {
          setSelectedProject(projects[0]);
          console.log('Auto-selected project:', projects[0]);
        }
        
        // Debug: Show what options will be available
        console.log('Country options will be:', getCountryOptions());
        console.log('Project options will be (after country selection):', getProjectOptions());
      } catch (error) {
        console.error('Error fetching user data:', error);
        showToast('error', 'Error', 'Failed to load user data');
      }
    };

    fetchUserData();

    const unsubscribeNetInfo = NetInfo.addEventListener((state) => {
      if (state.isConnected && state.isInternetReachable) {
        syncOfflineSchedules();
      }
    });

    return () => {
      unsubscribeNetInfo();
    };
  }, []);

  // Generate schedule ID preview when country and project are selected (without incrementing counter)
  useEffect(() => {
    if (selectedCountry && selectedProject) {
      generateScheduleIdPreview();
    }
  }, [selectedCountry, selectedProject]);

  // Helper functions for country and project codes
  const getCountryCode = (country) => {
    const countryCodes = {
      'Zambia': 'ZMB',
      'DRC': 'DRC',
    };
    return countryCodes[country] || country.substring(0, 3).toUpperCase();
  };

  const getProjectCode = (project) => {
    // Take first 3 characters of the project name and convert to uppercase
    if (!project) return 'XXX';
    return project.substring(0, 3).toUpperCase();
  };

  // Generate schedule ID preview (without incrementing counter)
  const generateScheduleIdPreview = async () => {
    try {
      // Get current counter from MongoDB (without incrementing)
      const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/counter/get`);
      
      if (!response.ok) {
        throw new Error(`Failed to get counter: ${response.status}`);
      }
      
      const result = await response.json();
      const currentCounter = result.success ? result.counter : 0;
      const nextCounter = currentCounter + 1; // Show next number but don't save it

      // Generate schedule ID preview
      const countryCode = getCountryCode(selectedCountry);
      const projectCode = getProjectCode(selectedProject);
      const formattedCounter = nextCounter.toString().padStart(3, '0');
      const previewScheduleId = `${countryCode}-${projectCode}-SS-${formattedCounter}`;
      
      console.log('Schedule ID Preview:', {
        country: selectedCountry,
        countryCode,
        project: selectedProject,
        projectCode,
        nextCounter,
        scheduleId: previewScheduleId
      });
      
      setScheduleId(previewScheduleId);
    } catch (error) {
      console.error('Error generating schedule ID preview:', error);
      // Fallback preview ID if there's an error
      const countryCode = selectedCountry ? selectedCountry.substring(0, 3).toUpperCase() : 'XXX';
      const projectCode = selectedProject ? selectedProject.substring(0, 3).toUpperCase() : 'XXX';
      setScheduleId(`${countryCode}-${projectCode}-SS-XXX`);
    }
  };

  // Generate actual schedule ID (increments counter) - called only when creating schedule
  const generateScheduleId = async () => {
    try {
      // Increment counter in MongoDB
      const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules/counter/increment`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      if (!response.ok) {
        throw new Error(`Failed to increment counter: ${response.status}`);
      }
      
      const result = await response.json();
      const newCounter = result.success ? result.counter : Date.now();

      // Generate actual schedule ID
      const countryCode = getCountryCode(selectedCountry);
      const projectCode = getProjectCode(selectedProject);
      const formattedCounter = newCounter.toString().padStart(3, '0');
      const newScheduleId = `${countryCode}-${projectCode}-SS-${formattedCounter}`;
      
      console.log('Generated Actual Schedule ID:', {
        country: selectedCountry,
        countryCode,
        project: selectedProject,
        projectCode,
        counter: newCounter,
        scheduleId: newScheduleId
      });
      
      return newScheduleId;
    } catch (error) {
      console.error('Error generating actual schedule ID:', error);
      // Fallback ID if there's an error
      const countryCode = selectedCountry ? selectedCountry.substring(0, 3).toUpperCase() : 'XXX';
      const projectCode = selectedProject ? selectedProject.substring(0, 3).toUpperCase() : 'XXX';
      return `${countryCode}-${projectCode}-SS-${Date.now()}`;
    }
  };

  const handleEquipmentSelect = (equipmentData) => {
    console.log('Equipment selected in AddSchedule:', equipmentData);
    
    // Ensure equipmentNumber is always set if possible
    let equipmentNumber = equipmentData.equipmentNumber;
    if (!equipmentNumber && equipmentData.entryNumber) {
      equipmentNumber = equipmentData.entryNumber;
    }
    
    // Create a comprehensive equipment object with all available data
    const selectedEquipmentData = {
      ...equipmentData,
      equipmentNumber,
      // Ensure we have the equipment name from various possible fields
      equipmentName: equipmentData.equipmentName || equipmentData.rigName || equipmentData.vehicleNumber || 'Equipment',
      // Ensure meter reading is properly formatted
      meterReading: equipmentData.meterReading || 0,
      meterUnit: equipmentData.meterUnit || 'Miles',
    };
    
    console.log('Processed equipment data:', selectedEquipmentData);
    
    setSelectedEquipment(selectedEquipmentData);
    setMeterReading(selectedEquipmentData.meterReading.toString());
    setMeterUnit(selectedEquipmentData.meterUnit);
    setNotifyBefore({ ...notifyBefore, unit: selectedEquipmentData.meterUnit });
    intervals.forEach((_, idx) => {
      handleUnitSelect(selectedEquipmentData.meterUnit, idx);
    });
  };

  const formatDate = (date) => {
    if (!date || isNaN(date.getTime())) return 'Select Date';
    return date.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: 'short',
      year: 'numeric',
    }).replace(/ /g, '-');
  };

  const openDatePicker = () => {
    setTempDate(lastServiceDate || new Date());
    setDatePickerVisible(true);
  };

  const handleDateCancel = () => {
    setDatePickerVisible(false);
  };

  const openTaskModal = () => {
    setNewTask('');
    setTaskModalVisible(true);
  };

  const handleTaskConfirm = () => {
    if (newTask.trim()) {
      setTasks([...tasks, newTask.trim()]);
    }
    setTaskModalVisible(false);
    setNewTask('');
  };

  const handleTaskCancel = () => {
    setTaskModalVisible(false);
    setNewTask('');
  };

  const handleUnitSelect = (unit, index) => {
    setIntervals(intervals.map((item, i) => (i === index ? { ...item, unit } : item)));
    setIntervalDropdownIndex(null);
  };



  const handleMeterUnitSelect = (unit) => {
    setMeterUnit(unit);
    setNotifyBefore({ ...notifyBefore, unit });
    intervals.forEach((_, idx) => {
      handleUnitSelect(unit, idx);
    });
    setMeterUnitDropdownVisible(false);
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Function to send service schedule notifications (optional - can be implemented later)
  const sendServiceScheduleNotifications = async (scheduleData) => {
    try {
      console.log('📧 Service schedule notifications will be implemented in future updates');
      // TODO: Implement MongoDB-based notification system
    } catch (error) {
      console.error('❌ Error in sendServiceScheduleNotifications:', error);
      // Don't throw error - notifications are optional
    }
  };

  const storeScheduleOffline = async (scheduleData) => {
    try {
      const offlineSchedules = await AsyncStorage.getItem('offlineSchedules');
      const schedules = offlineSchedules ? JSON.parse(offlineSchedules) : [];
      schedules.push(scheduleData);
      await AsyncStorage.setItem('offlineSchedules', JSON.stringify(schedules));
      showToast('success', 'Success', 'Schedule stored offline');
      return true;
    } catch (error) {
      console.error('Error storing schedule offline:', error);
      showToast('error', 'Error', 'Failed to store schedule offline');
      return false;
    }
  };

  const syncOfflineSchedules = async () => {
    try {
      const offlineSchedules = await AsyncStorage.getItem('offlineSchedules');
      if (!offlineSchedules) return;
      const schedules = JSON.parse(offlineSchedules);
      if (schedules.length === 0) return;

      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) {
        console.error('User not authenticated during sync');
        return;
      }

      console.log(`🔄 Syncing ${schedules.length} offline schedules to MongoDB...`);

      for (const schedule of schedules) {
        try {
          const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify({ ...schedule, adminUid: uid }),
          });

          if (!response.ok) {
            console.error(`Failed to sync schedule ${schedule.id}: ${response.status}`);
          } else {
            console.log(`✅ Synced schedule ${schedule.id}`);
          }
        } catch (error) {
          console.error(`Error syncing schedule ${schedule.id}:`, error);
        }
      }

      await AsyncStorage.setItem('offlineSchedules', JSON.stringify([]));
      showToast('success', 'Success', 'Offline schedules synced to database');
    } catch (error) {
      console.error('Error syncing offline schedules:', error);
      showToast('error', 'Error', 'Failed to sync offline schedules');
    }
  };

  const handleCreateSchedule = async () => {
    setShowErrors(true);

    // Fix: Validate intervals for repeating schedules
    const validIntervals = intervals.filter((interval) => interval.checked && interval.value);
    if (
      !selectedCountry ||
      !selectedProject ||
      !scheduleTitle ||
      !note ||
      !selectedEquipment ||
      !meterReading ||
      isNaN(parseFloat(meterReading)) ||
      !lastServiceDate ||
      isNaN(lastServiceDate.getTime()) ||
      !tasks.length ||
      !repeatType ||
      (repeatType === 'Repeat' && validIntervals.length === 0)
    ) {
      showToast('error', 'Validation Error', 'Please fill all required fields, including Country, Project, and at least one interval for repeating schedules.');
      setIsLoading(false);
      return;
    }

    setIsLoading(true);
    animationRef.current = Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    );
    animationRef.current.start();

    // Generate actual schedule ID (increments counter)
    const actualScheduleId = await generateScheduleId();
    
    const scheduleData = {
      id: Date.now().toString(),
      scheduleId: actualScheduleId,
      country: selectedCountry,
      project: selectedProject,
      title: scheduleTitle,
      equipment: {
        entryNumber: selectedEquipment.entryNumber || '',
        equipmentName: selectedEquipment.equipmentName || 'TYRE - TYRE',
        equipmentNumber: selectedEquipment.equipmentNumber || '', // Ensure equipmentNumber is always set
        meterReading: parseFloat(meterReading) || 0,
        meterUnit: meterUnit,
      },
      note,
      lastServiceDate: formatDate(lastServiceDate),
      meterReading: parseFloat(meterReading) || 0,
      meterUnit,
      repeatType,
      intervals: validIntervals,
      intervalUnit,
      notifyBefore,
      tasks,
      status: 'upcoming',
      createdAt: new Date().toISOString(),
      dueHours: '12',
      hoursToGo: '10',
      daysToGo: '69',
    };

    console.log('📝 Schedule data to be saved:', {
      scheduleId: actualScheduleId,
      country: selectedCountry,
      project: selectedProject,
      title: scheduleTitle,
      equipment: selectedEquipment?.equipmentName,
      tasksCount: tasks.length,
      intervalsCount: validIntervals.length
    });

    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) {
        showToast('error', 'Error', 'User not authenticated. Please log in again.');
        navigation.navigate('AdminLogin');
        setIsLoading(false);
        return;
      }

      // Add adminUid to scheduleData
      const fullScheduleData = {
        ...scheduleData,
        adminUid: uid
      };

      const state = await NetInfo.fetch();
      if (state.isConnected && state.isInternetReachable) {
        // Save to MongoDB
        console.log('📝 Saving schedule to MongoDB...');
        const response = await fetch(`${MONGODB_SERVICE_SCHEDULE_API_URL}/service-schedules`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(fullScheduleData),
        });

        if (!response.ok) {
          throw new Error(`Failed to create schedule: ${response.status}`);
        }

        const result = await response.json();
        
        if (result.success) {
          console.log('✅ Schedule created successfully in MongoDB!');
          
          // Save to local storage
          await AsyncStorage.setItem('globalServiceSchedules', JSON.stringify([fullScheduleData]));

          // Send service schedule notifications
          try {
            console.log('📱 Sending service schedule notifications...');
            await sendServiceScheduleNotifications({
              scheduleId: scheduleData.id,
              scheduleNumber: actualScheduleId,
              title: scheduleTitle,
              country: selectedCountry,
              project: selectedProject,
              equipment: selectedEquipment,
              repeatType: repeatType,
              tasks: tasks,
              creatorId: uid
            });
            console.log('✅ Service schedule notifications sent successfully');
          } catch (notificationError) {
            console.error('❌ Error sending service schedule notifications:', notificationError);
            // Don't fail the schedule creation if notifications fail
          }

          showToast('success', 'Success', 'Schedule created successfully.');
          navigation.replace('ServiceScheduleHome');
        } else {
          throw new Error(result.error || 'Failed to create schedule');
        }
      } else {
        const success = await storeScheduleOffline(fullScheduleData);
        if (success) {
          showToast('info', 'Offline', 'Schedule stored offline. Will sync when online.');
          navigation.replace('ServiceScheduleHome');
        }
      }
    } catch (error) {
      console.error('Error saving schedule:', error);
      const uid = await AsyncStorage.getItem('userUid');
      const fullScheduleData = { ...scheduleData, adminUid: uid };
      const success = await storeScheduleOffline(fullScheduleData);
      if (success) {
        showToast('info', 'Offline', 'Schedule stored offline due to error. Will sync when online.');
        navigation.replace('ServiceScheduleHome');
      } else {
        showToast('error', 'Error', 'Failed to create schedule.');
      }
    } finally {
      setIsLoading(false);
      if (animationRef.current) {
        animationRef.current.stop();
        animationRef.current = null;
      }
      rotateAnim.setValue(0);
    }
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Add Schedule</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleCreateSchedule}>
              <Text style={styles.createButtonText}>Create</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView style={styles.mainContent} keyboardShouldPersistTaps="handled">
        <View style={styles.scheduleIdRow}>
          <Text style={styles.scheduleIdLabel}>Schedule ID</Text>
          <Text style={styles.scheduleIdValue}>{scheduleId}</Text>
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Country
            <Text style={inputStyles.requiredError}> *</Text>
          </Text>
          <Dropdown
            style={[inputStyles.input, showErrors && !selectedCountry ? inputStyles.inputError : null]}
            placeholderStyle={inputStyles.placeholderStyle}
            selectedTextStyle={inputStyles.selectedTextStyle}
            data={getCountryOptions()}
            maxHeight={200}
            labelField="label"
            valueField="value"
            placeholder={userCountries.length > 0 ? "Select Country" : "Loading countries..."}
            value={selectedCountry}
            onChange={(item) => {
              setSelectedCountry(item.value);
              setSelectedProject(''); // Reset project when country changes
              setScheduleId(''); // Reset schedule ID preview
            }}
            disable={userCountries.length === 0}
            renderLeftIcon={() => (
              <MaterialIcons 
                style={inputStyles.iconStyle} 
                color="#015185" 
                name="public" 
                size={20} 
              />
            )}
            renderRightIcon={() => (
              <MaterialIcons 
                style={inputStyles.iconStyle} 
                color="#015185" 
                name="arrow-drop-down" 
                size={20} 
              />
            )}
          />
          {showErrors && !selectedCountry && (
            <Text style={inputStyles.errorText}>Country is required</Text>
          )}
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Project
            <Text style={inputStyles.requiredError}> *</Text>
          </Text>
          <Dropdown
            style={[inputStyles.input, showErrors && !selectedProject ? inputStyles.inputError : null]}
            placeholderStyle={inputStyles.placeholderStyle}
            selectedTextStyle={inputStyles.selectedTextStyle}
            data={getProjectOptions()}
            maxHeight={200}
            labelField="label"
            valueField="value"
            placeholder={selectedCountry ? (getProjectOptions().length > 0 ? "Select Project" : "No projects available") : "Select Country first"}
            value={selectedProject}
            onChange={(item) => {
              setSelectedProject(item.value);
            }}
            disable={!selectedCountry || getProjectOptions().length === 0}
            renderLeftIcon={() => (
              <MaterialIcons 
                style={inputStyles.iconStyle} 
                color="#015185" 
                name="business" 
                size={20} 
              />
            )}
            renderRightIcon={() => (
              <MaterialIcons 
                style={inputStyles.iconStyle} 
                color="#015185" 
                name="arrow-drop-down" 
                size={20} 
              />
            )}
          />
          {showErrors && !selectedProject && (
            <Text style={inputStyles.errorText}>Project is required</Text>
          )}
        </View>

        <CustomInputField
          label="Schedule Title"
          value={scheduleTitle}
          onChangeText={setScheduleTitle}
          isRequired
          showError={showErrors && !scheduleTitle}
          errorMessage="Schedule Title is required"
        />

        <CustomInputField
          label={<Text>Note<Text style={inputStyles.requiredError}> *</Text></Text>}
          value={note}
          onChangeText={setNote}
          isRequired
          showError={showErrors && !note}
          errorMessage="Note is required"
        />

        <View style={inputStyles.container}>
          <View style={styles.rowBetween}>
            <Text style={inputStyles.label}>
              Equipments
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <TouchableOpacity
              onPress={() => navigation.navigate('Service_select_equipment_SS', { source: 'AddSchedule', onSelect: handleEquipmentSelect })}
            >
              <Text style={styles.selectText}>Select</Text>
            </TouchableOpacity>
          </View>
          {showErrors && !selectedEquipment && (
            <Text style={inputStyles.errorText}>Equipment is required</Text>
          )}
          {selectedEquipment ? (
            <View style={styles.selectedEquipmentContainer}>
              <Text style={styles.selectedEquipmentText}>{selectedEquipment.equipmentName || selectedEquipment.rigName || selectedEquipment.vehicleNumber || 'Equipment'}</Text>
              <View style={styles.equipmentDetailsGrid}>
                <Text style={styles.selectedEquipmentDetail}>Entry Number: {selectedEquipment.entryNumber || 'N/A'}</Text>
                <Text style={styles.selectedEquipmentDetail}>Equipment Number: {selectedEquipment.equipmentNumber || 'N/A'}</Text>
                <Text style={styles.selectedEquipmentDetail}>Meter: {selectedEquipment.meterReading || '0'} {selectedEquipment.meterUnit || 'Miles'}</Text>
                {selectedEquipment.vehicleNumber && (
                  <Text style={styles.selectedEquipmentDetail}>Vehicle Number: {selectedEquipment.vehicleNumber}</Text>
                )}
                {selectedEquipment.registrationNumber && (
                  <Text style={styles.selectedEquipmentDetail}>Registration: {selectedEquipment.registrationNumber}</Text>
                )}
                {selectedEquipment.make && (
                  <Text style={styles.selectedEquipmentDetail}>Make: {selectedEquipment.make}</Text>
                )}
                {selectedEquipment.model && (
                  <Text style={styles.selectedEquipmentDetail}>Model: {selectedEquipment.model}</Text>
                )}
                {selectedEquipment.mainCategory && (
                  <Text style={styles.selectedEquipmentDetail}>Category: {selectedEquipment.mainCategory}</Text>
                )}
                {selectedEquipment.status && (
                  <Text style={[styles.selectedEquipmentDetail, { color: getStatusColor(selectedEquipment.status) }]}>
                    Status: {selectedEquipment.status}
                  </Text>
                )}
              </View>
            </View>
          ) : (
            <Text style={styles.placeholderText}>No equipment selected</Text>
          )}
          <View style={quantityStyles.row}>
            <CustomInputField
              label={<Text>Last Service Meter Reading<Text style={inputStyles.requiredError}> *</Text></Text>}
              value={meterReading}
              onChangeText={(text) => {
                // Fix: Allow only one decimal point and up to 2 decimal places
                const sanitizedText = text.replace(/[^0-9.]|^(.*\..*)\./g, '$1').replace(/(\.\d{2})\d+/, '$1');
                setMeterReading(sanitizedText);
              }}
              keyboardType="numeric"
              isRequired
              showError={showErrors && (!meterReading || isNaN(parseFloat(meterReading)))}
              errorMessage="Valid meter reading is required"
              style={quantityStyles.input}
            />
            <View style={quantityStyles.dropdownWrapper}>
              <Dropdown
                style={quantityStyles.unitBox}
                placeholderStyle={quantityStyles.unitText}
                selectedTextStyle={quantityStyles.unitText}
                data={UNIT_OPTIONS.map(unit => ({ label: unit, value: unit }))}
                maxHeight={150}
                labelField="label"
                valueField="value"
                placeholder="Select Unit"
                value={meterUnit}
                onChange={(item) => handleMeterUnitSelect(item.value)}
                renderRightIcon={() => (
                  <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
                )}
              />
            </View>
          </View>
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Last Service Date
            <Text style={inputStyles.requiredError}> *</Text>
          </Text>
          <View style={quantityStyles.row}>
            <TouchableOpacity
              style={[quantityStyles.input1, { justifyContent: 'center' }]}
              onPress={openDatePicker}
            >
              <Text style={{ color: lastServiceDate && !isNaN(lastServiceDate.getTime()) ? '#222' : '#888', fontSize: width * 0.04 }}>
                {formatDate(lastServiceDate)}
              </Text>
            </TouchableOpacity>
            <TouchableOpacity onPress={openDatePicker}>
              <MaterialIcons name="calendar-today" size={22} color="#015185" style={{ marginLeft: 8 }} />
            </TouchableOpacity>
          </View>
          {showErrors && (!lastServiceDate || isNaN(lastServiceDate.getTime())) && (
            <Text style={inputStyles.errorText}>Last Service Date is required</Text>
          )}
        </View>

        {datePickerVisible && (
          <DateTimePicker
            value={tempDate || new Date()}
            mode="date"
            display="default"
            onChange={(event, selectedDate) => {
              if (event.type === 'set' && selectedDate) {
                setTempDate(selectedDate);
                setLastServiceDate(selectedDate);
                setDatePickerVisible(false);
              } else if (event.type === 'dismissed') {
                setDatePickerVisible(false);
              }
            }}
          />
        )}

        <View style={inputStyles.container}>
          <View style={styles.taskHeader}>
            <Text style={inputStyles.label}>
              Task
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <TouchableOpacity style={styles.addTaskButton} onPress={openTaskModal}>
              <Text style={styles.addTaskText}>+ Add Task</Text>
            </TouchableOpacity>
          </View>
          {showErrors && !tasks.length && (
            <Text style={inputStyles.errorText}>Task is required</Text>
          )}
          {tasks.length > 0 ? (
            tasks.map((task, idx) => (
              <Text key={idx} style={styles.taskItem}>- {task}</Text>
            ))
          ) : (
            <Text style={styles.placeholderText}>No tasks added</Text>
          )}
        </View>

        <Modal visible={taskModalVisible} transparent animationType="fade" onRequestClose={handleTaskCancel}>
          <View style={styles.modalOverlay}>
            <View style={styles.taskModalContainer}>
              <Text style={styles.modalTitle}>Add New Task</Text>
              <CustomInputField
                label="Task Name"
                value={newTask}
                onChangeText={setNewTask}
                style={{ width: '100%' }}
              />
              <View style={styles.modalButtons}>
                <TouchableOpacity style={[styles.modalButton, styles.cancelButton]} onPress={handleTaskCancel}>
                  <Text style={styles.modalButtonText}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity style={[styles.modalButton, styles.doneButton]} onPress={handleTaskConfirm}>
                  <Text style={styles.modalButtonText}>Done</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>

        <View style={inputStyles.container}>
          <View style={[styles.rowBetween, { marginBottom: 0 }]}>
            <Text style={inputStyles.label}>
              Repeat Type
              <Text style={inputStyles.requiredError}> *</Text>
            </Text>
            <View style={styles.repeatTypeRow}>
              <TouchableOpacity
                style={[styles.pillButton, repeatType === 'Repeat' && styles.pillButtonActive]}
                onPress={() => setRepeatType('Repeat')}
              >
                <Text style={[styles.pillButtonText, repeatType === 'Repeat' && styles.pillButtonTextActive]}>Repeat</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.pillButton, repeatType === 'One time' && styles.pillButtonActive]}
                onPress={() => setRepeatType('One time')}
              >
                <Text style={[styles.pillButtonText, repeatType === 'One time' && styles.pillButtonTextActive]}>One time</Text>
              </TouchableOpacity>
            </View>
          </View>
          {showErrors && !repeatType && (
            <Text style={inputStyles.errorText}>Repeat Type is required</Text>
          )}
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.labelSmall}>Intervals (whichever comes first)</Text>
          {intervals.map((interval, idx) => (
            <View key={idx} style={quantityStyles.row}>
              <TouchableOpacity
                style={styles.checkbox}
                onPress={() => {
                  setIntervals(intervals.map((item, i) => (i === idx ? { ...item, checked: !item.checked } : item)));
                }}
              >
                <MaterialIcons
                  name={interval.checked ? 'check-box' : 'check-box-outline-blank'}
                  size={22}
                  color={interval.checked ? '#015185' : '#A0A0A0'}
                />
              </TouchableOpacity>
              <Text style={styles.everyText}>Every</Text>
              <CustomInputField
                label=""
                value={interval.value}
                onChangeText={(text) => {
                  const numericText = text.replace(/[^0-9]/g, '');
                  if (numericText.length <= 10) {
                    setIntervals(intervals.map((item, i) => (i === idx ? { ...item, value: numericText } : item)));
                  }
                }}
                keyboardType="numeric"
                placeholder="0"
                editable={interval.checked}
                maxLength={10}
                style={[quantityStyles.input, { width: 70 }, !interval.checked && inputStyles.inputDisabled]}
              />
              <View style={quantityStyles.dropdownWrapper}>
                <Dropdown
                  style={quantityStyles.unitBox}
                  placeholderStyle={quantityStyles.unitText}
                  selectedTextStyle={quantityStyles.unitText}
                  data={UNIT_OPTIONS.map(unit => ({ label: unit, value: unit }))}
                  maxHeight={150}
                  labelField="label"
                  valueField="value"
                  placeholder="Select Unit"
                  value={interval.unit}
                  onChange={(item) => handleUnitSelect(item.value, idx)}
                  renderRightIcon={() => (
                    <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
                  )}
                />
              </View>
            </View>
          ))}
          {showErrors && repeatType === 'Repeat' && !intervals.some((interval) => interval.checked && interval.value) && (
            <Text style={inputStyles.errorText}>At least one interval is required for repeating schedules</Text>
          )}
          <View style={styles.pillButtonGroup}>
            {['Day', 'Week', 'Month', 'Year'].map((unit) => (
              <TouchableOpacity
                key={unit}
                style={[styles.pillButton, intervalUnit === unit && styles.pillButtonActive]}
                onPress={() => setIntervalUnit(unit)}
              >
                <Text style={[styles.pillButtonText, intervalUnit === unit && styles.pillButtonTextActive]}>{unit}</Text>
              </TouchableOpacity>
            ))}
          </View>
        </View>



        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>Notify Before</Text>
          <View style={quantityStyles.row}>
            <CustomInputField
              label=""
              value={notifyBefore.value}
              onChangeText={(text) => setNotifyBefore({ ...notifyBefore, value: text.replace(/[^0-9]/g, '') })}
              keyboardType="numeric"
              placeholder="0"
              style={[quantityStyles.input, { width: 70 }]}
            />
            <View style={quantityStyles.dropdownWrapper}>
              <Dropdown
                style={quantityStyles.unitBox}
                placeholderStyle={quantityStyles.unitText}
                selectedTextStyle={quantityStyles.unitText}
                data={UNIT_OPTIONS.map(unit => ({ label: unit, value: unit }))}
                maxHeight={150}
                labelField="label"
                valueField="value"
                placeholder="Select Unit"
                value={notifyBefore.unit}
                onChange={(item) => setNotifyBefore({ ...notifyBefore, unit: item.value })}
                renderRightIcon={() => (
                  <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
                )}
              />
            </View>
          </View>
          <View style={styles.pillButtonGroup}>
            {['Day', 'Week', 'Month'].map((unit) => (
              <TouchableOpacity
                key={unit}
                style={[styles.pillButton, notifyBefore.time === unit && styles.pillButtonActive]}
                onPress={() => setNotifyBefore({ ...notifyBefore, time: unit })}
              >
                <Text style={[styles.pillButtonText, notifyBefore.time === unit && styles.pillButtonTextActive]}>{unit}</Text>
              </TouchableOpacity>
            ))}
          </View>
          <Text style={styles.infoText}>
            The schedule will be marked as 'Due soon' once any of the above threshold is crossed.
          </Text>
        </View>



        <Modal visible={isLoading} transparent animationType="fade">
          <View style={styles.syncModalContainer}>
            <View style={styles.syncModalContent}>
              <Text style={styles.syncModalText}>Please Wait</Text>
              <View style={styles.syncIconDiv}>
                {/* <Animated.Image
                  source={require('../../assets/sync.png')} // Fix: Update path to match your project
                  style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                /> */}
                <Text style={styles.syncModalTextAdd}>Creating Schedule</Text>
              </View>
            </View>
          </View>
        </Modal>

        <Toast />
      </ScrollView>
    </View>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  inputDisabled: {
    color: '#A0A0A0',
    borderColor: '#B0C4DE',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
  placeholderStyle: {
    fontSize: width * 0.04,
    color: '#888',
  },
  selectedTextStyle: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  iconStyle: {
    marginRight: 5,
  },
});

const quantityStyles = StyleSheet.create({
  row: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  input: {
    flex: 1,
    marginRight: 8,
  },
  input1: {
    flex: 1,
    marginRight: 8,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  unitBox: {
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    backgroundColor: '#fff',
    width: 100,
    justifyContent: 'space-between',
  },
  unitText: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginRight: 2,
  },
  dropdownWrapper: {
    position: 'relative',
    zIndex: 1,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.2)',
  },
  dropdownContainer: {
    position: 'absolute',
    top: width * 0.5,
    right: width * 0.12,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
    width: 120,
    zIndex: 1000,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
  },
  dropdownItem: {
    paddingVertical: 12,
    paddingHorizontal: 16,
  },
  dropdownText: {
    color: '#015185',
    fontSize: width * 0.038,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.019,
    backgroundColor: '#fff',
    borderRadius: width * 0.01,
  },
  createButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  rowBetween: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 12,
  },
  row: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 12,
  },
  labelSmall: {
    color: '#015185',
    marginTop: 10,
    marginBottom: 6,
    fontSize: width * 0.034,
  },
  selectText: {
    fontSize: width * 0.034,
    color: '#025383',
  },
  everyText: {
    color: '#015185',
    fontSize: 15,
    marginHorizontal: 8,
  },
  checkbox: {
    marginRight: 4,
  },
  taskHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 6,
  },
  addTaskButton: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 18,
    paddingVertical: 8,
  },
  addTaskText: {
    color: '#fff',
    fontSize: width * 0.034,
  },
  repeatTypeRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: 180,
  },
  pillButtonGroup: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    alignItems: 'center',
    marginTop: 4,
    marginBottom: 8,
  },
  pillButton: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 18,
    paddingVertical: 7,
    marginRight: 8,
    backgroundColor: '#fff',
  },
  pillButtonActive: {
    backgroundColor: '#015185',
  },
  pillButtonText: {
    color: '#015185',
    fontSize: width * 0.034,
    fontWeight: '500',
  },
  pillButtonTextActive: {
    color: '#fff',
  },
  infoText: {
    color: '#015185',
    fontSize: 13,
    marginTop: 10,
  },
  selectedEquipmentContainer: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 12,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  selectedEquipmentText: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 8,
    fontWeight: 'bold',
  },
  equipmentDetailsGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
  },
  selectedEquipmentDetail: {
    fontSize: 13,
    color: '#015185',
    marginBottom: 4,
    width: '48%',
    lineHeight: 18,
  },
  placeholderText: {
    fontSize: 15,
    color: '#A0A0A0',
    marginBottom: 12,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  datePickerContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  datePickerButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    marginTop: 10,
  },
  datePickerButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  taskModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#025383',
    marginBottom: 15,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  modalButton: {
    flex: 1,
    borderRadius: 5,
    paddingVertical: 10,
    alignItems: 'center',
    marginHorizontal: 5,
  },
  cancelButton: {
    backgroundColor: '#A0A0A0',
  },
  doneButton: {
    backgroundColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  taskItem: {
    fontSize: 15,
    color: '#333',
    marginBottom: 4,
  },
  scheduleIdRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.04,
  },
  scheduleIdLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  scheduleIdValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)', // Fix: Corrected syntax
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextAdd: {
    fontSize: 14,
    color: '#333',
  },
  syncIconDiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default AddSchedule;