import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  FlatList,
  ScrollView,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { getFilteredTeamMembers } from '../Utils/teamMemberUtils';
import { Dropdown } from 'react-native-element-dropdown';
import uuid from 'react-native-uuid';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;
const DROPDOWN_HEIGHT = 2 * 44; // 2 items, 44px each
const MARGIN = 8;

const SyncModal = ({ visible, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    let animation;
    if (visible) {
      animation = Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      );
      animation.start();
    } else {
      rotateAnim.setValue(0);
      if (animation) animation.stop && animation.stop();
    }
    return () => {
      if (animation) animation.stop && animation.stop();
    };
  }, [visible]);

  if (!visible) return null;
  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={{
        flex: 1,
        backgroundColor: 'rgba(0,0,0,0.5)',
        justifyContent: 'center',
        alignItems: 'center',
      }}>
        <View style={{
          width: '70%',
          backgroundColor: 'white',
          borderRadius: 10,
          padding: 30,
          alignItems: 'center',
        }}>
          <Text style={{ fontSize: 20, marginBottom: 18, color: '#015185', fontWeight: 'bold' }}>Please Wait</Text>
          <Animated.Image
            source={require('../../../Images/adminlogin/sync.png')}
            style={{
              width: 48,
              height: 48,
              marginBottom: 16,
              transform: [{ rotate: rotateInterpolate }],
            }}
          />
          <Text style={{ fontSize: 16, color: '#333', textAlign: 'center' }}>{message}</Text>
        </View>
      </View>
    </Modal>
  );
};

const CustomInputField = ({
  label,
  value,
  onChangeText,
  isRequired = false,
  showError = false,
  errorMessage = '',
  icon,
  onIconPress,
  editable = true,
  showSelectButton = false,
  onSelectPress,
  noBackground = false,
  multiline = false,
  tags = [],
  onRemoveTag,
  noBorder = false,
  ...props
}) => (
  <View style={inputStyles.container}>
    <View style={inputStyles.labelContainer}>
      <Text style={inputStyles.label}>
        {label}
        {isRequired && <Text style={inputStyles.requiredError}> *</Text>}
      </Text>
      {showSelectButton && (
        <TouchableOpacity style={inputStyles.labelSelectButton} onPress={onSelectPress}>
          <Text style={inputStyles.selectText}>Select</Text>
          <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
        </TouchableOpacity>
      )}
    </View>
    <View style={inputStyles.inputContainer}>
      {tags.length > 0 && (
        <View style={inputStyles.tagsContainer}>
          {tags.map((item, index) => (
            <View key={index} style={inputStyles.tag}>
              <Text style={inputStyles.tagText}>{item.name}</Text>
              {onRemoveTag && (
                <TouchableOpacity
                  style={inputStyles.tagCloseButton}
                  onPress={() => onRemoveTag(item)}
                >
                  <View style={inputStyles.tagCloseCircle}>
                    <MaterialIcons name="close" size={width * 0.035} color="#fff" />
                  </View>
                </TouchableOpacity>
              )}
            </View>
          ))}
        </View>
      )}
      <TextInput
        style={[
          inputStyles.input,
          showError ? inputStyles.inputError : null,
          !editable && inputStyles.disabledInput,
          noBackground && inputStyles.noBackground,
          multiline && inputStyles.multilineInput,
          tags.length > 0 && inputStyles.inputWithTags,
          noBorder && inputStyles.noBorder,
        ]}
        value={value}
        onChangeText={onChangeText}
        editable={editable}
        placeholderTextColor="#888"
        multiline={multiline}
        {...props}
      />
      {icon && (
        <TouchableOpacity style={inputStyles.inputIcon} onPress={onIconPress}>
          {icon}
        </TouchableOpacity>
      )}
    </View>
    {showError && <Text style={inputStyles.errorText}>{errorMessage}</Text>}
  </View>
);

const UpdateRequestMaintenance = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { requestId } = route.params || {};
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [workOrderTitle, setWorkOrderTitle] = useState('');
  const [memo, setMemo] = useState('');
  const [priority, setPriority] = useState(null);
  const [assignTo, setAssignTo] = useState([]);
  const [originalAssignTo, setOriginalAssignTo] = useState([]);
  const [assignToModalVisible, setAssignToModalVisible] = useState(false);
  const [requestTypeModalVisible, setRequestTypeModalVisible] = useState(false);
  const [selectedRequestType, setSelectedRequestType] = useState(null);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [selectedChecklist, setSelectedChecklist] = useState(null);
  const [searchQuery, setSearchQuery] = useState('');
  const [taskModalVisible, setTaskModalVisible] = useState(false);
  const [taskTitle, setTaskTitle] = useState('');
  const [taskDescription, setTaskDescription] = useState('');
  const [tasks, setTasks] = useState([]);
  const [requestNumber, setRequestNumber] = useState('');
  const [errors, setErrors] = useState({ workOrderTitle: '', country: '', project: '', taskTitle: '' });
  const [userId, setUserId] = useState(null);
  const [successModalVisible, setSuccessModalVisible] = useState(false);
  const [taskDropdownVisible, setTaskDropdownVisible] = useState(false);
  const [taskDropdownPosition, setTaskDropdownPosition] = useState({ x: 0, y: 0 });
  const [selectedTask, setSelectedTask] = useState(null);
  const [taskMenuButtonWidth, setTaskMenuButtonWidth] = useState(30);
  const [userPermissions, setUserPermissions] = useState([]);
  const [adminProfile, setAdminProfile] = useState(null);
  const [teamMembers, setTeamMembers] = useState([]);
  const [country, setCountry] = useState(null);
  const [countryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);
  const [project, setProject] = useState(null);
  const [projectItems, setProjectItems] = useState([]);

  // Add state for fault list modal and data
  const [faultListModalVisible, setFaultListModalVisible] = useState(false);
  const [selectedFaults, setSelectedFaults] = useState([]);
  const faultListData = [];

  // --- Add state for per-task assignees and modal ---
  const [taskAssignees, setTaskAssignees] = useState([]);
  const [taskAssignToModalVisible, setTaskAssignToModalVisible] = useState(false);
  const [taskAssignSearchQuery, setTaskAssignSearchQuery] = useState('');
  const [taskNote, setTaskNote] = useState('');

  const requestTypes = [
    { id: '1', name: 'General' },
    { id: '2', name: 'Preventive' },
    { id: '3', name: 'Breakdown' },
    { id: '4', name: 'Corrective' },
    { id: '5', name: 'Safety' },
    { id: '6', name: 'Upgrade' },
    { id: '7', name: 'Cleaning' },
    { id: '8', name: 'Testing' },
  ];

  const priorityOptions = [
    { label: 'Low', image: require('../../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../../Images/adminhome/inspreport/Priority4.png') },
  ];

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected ?? false;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const validateField = (field, value, errorKey, errorMessage) => {
    setErrors(prev => ({
      ...prev,
      [errorKey]: (typeof value === 'string' ? value.trim() : '').length > 0 ? '' : errorMessage,
    }));
  };

  const generateRequestNumber = (country, project, nextNumber) => {
    if (country && project) {
      const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
      const projectCode = project.slice(0, 3).toUpperCase();
      return `${countryCode}-${projectCode}-RM-${nextNumber.toString().padStart(3, '0')}`;
    }
    return `RM${nextNumber.toString().padStart(3, '0')}`;
  };

  const extractNumberFromRequestNumber = (requestNumber) => {
    if (!requestNumber) return 1;
    const newFormatMatch = requestNumber.match(/-RM-(\d+)$/);
    if (newFormatMatch) return parseInt(newFormatMatch[1], 10);
    const oldFormatMatch = requestNumber.match(/RM(\d+)/);
    if (oldFormatMatch) return parseInt(oldFormatMatch[1], 10);
    return 1;
  };

  useEffect(() => {
    const loadUserIdAndRequest = async () => {
      try {
        const uid = await AsyncStorage.getItem('userUid');
          setUserId(uid);
          if (requestId) {
            setSyncModalVisible(true);
            const isOnline = await checkNetworkStatus();
            let request = null;
            if (isOnline) {
            // Fetch from all admins globally
              const snapshot = await database()
              .ref('GlobalRequestMaintenancereport/admins')
                .once('value');
            const allAdmins = snapshot.val();
            let found = null;
            if (allAdmins) {
              for (const adminKey of Object.keys(allAdmins)) {
                if (adminKey === 'lastRequestNumber') continue;
                const adminRequests = allAdmins[adminKey];
                if (adminRequests && adminRequests[requestId]) {
                  found = adminRequests[requestId];
                  break;
                }
              }
            }
            request = found;
              if (request) {
                const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
                let requests = [];
                try {
                  requests = storedRequests ? JSON.parse(storedRequests) : [];
                  if (!Array.isArray(requests)) {
                    requests = [];
                    console.warn('Invalid stored requests, resetting to empty array.');
                  }
                } catch (error) {
                  console.error('AsyncStorage parse error:', error);
                  showToast('error', 'Error', 'Failed to load local data.');
                }
                requests = requests.filter(req => req.id !== requestId);
                requests.push(request);
                await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));
                if (request.checklistId) {
                  const checklistSnapshot = await database()
                    .ref(`/InspectionChecklistCreatebyusers/admin/${request.checklistId}`)
                    .once('value');
                  const checklist = checklistSnapshot.val();
                  setSelectedChecklist(checklist ? { id: request.checklistId, ...checklist } : null);
                }
              }
            } else {
              const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
              let requests = [];
              try {
                requests = storedRequests ? JSON.parse(storedRequests) : [];
                if (!Array.isArray(requests)) {
                  requests = [];
                  console.warn('Invalid stored requests, resetting to empty array.');
                }
              } catch (error) {
                console.error('AsyncStorage parse error:', error);
                showToast('error', 'Error', 'Failed to load local data.');
              }
              request = requests.find(req => req.id === requestId);
              showToast('info', 'Offline', 'Using offline data. Will sync when online.');
            }
            if (request) {
              setWorkOrderTitle(request.workOrderTitle || '');
              setMemo(request.memo || '');
              setPriority(request.priority || null);
              setAssignTo(request.assignTo || []);
              setOriginalAssignTo(request.assignTo || []);
              setSelectedRequestType(request.requestType || null);
              setSelectedEquipment(request.equipment || null);
              setTasks(request.tasks || []);
              setRequestNumber(request.requestNumber || '');
              setCountry(request.country || null);
              setProject(request.project || null);
              setSelectedChecklist(request.checklistId ? { id: request.checklistId, title: request.checklistTitle || 'Checklist' } : null);
            } else {
              showToast('error', 'Error', 'Maintenance request not found.');
              navigation.goBack();
            }
            setSyncModalVisible(false);
          } else {
            showToast('error', 'Error', 'No request ID provided.');
            navigation.goBack();
        }
      } catch (error) {
        console.error('Error fetching request data:', error);
        showToast('error', 'Error', 'Failed to load request data.');
        setSyncModalVisible(false);
        navigation.goBack();
      }
    };
    loadUserIdAndRequest();
  }, [navigation, requestId]);

  useEffect(() => {
    const updateRequestNumber = async () => {
      if (!requestId || !country || !project) return;
      try {
        const lastRequestRef = database().ref(`/GlobalRequestMaintenancereport/admins/lastRequestNumber`);
        const snapshot = await lastRequestRef.once('value');
        const lastNumber = snapshot.val();
        const currentNumber = extractNumberFromRequestNumber(requestNumber);
        const newRequestNumber = generateRequestNumber(country, project, currentNumber);
        setRequestNumber(newRequestNumber);
      } catch (error) {
        console.error('Error updating request number:', error);
      }
    };
    updateRequestNumber();
  }, [country, project, requestNumber]);

  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    setTimeout(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
    }, 2000);
  };

  const handlePrioritySelect = (selectedPriority) => {
    setPriority(selectedPriority);
  };

  const handleRequestTypeSelect = (requestType) => {
    setSelectedRequestType(requestType.name);
    setRequestTypeModalVisible(false);
  };

  const handleEquipmentSelect = () => {
    navigation.navigate('SelectAllInOneEquipment', {
      selectedEquipment,
      onSelect: (equipment) => {
        setSelectedEquipment(equipment);
      },
    });
  };

  const handleChecklistSelect = () => {
    navigation.navigate('SelectChecklist', {
      selectedChecklist,
      onSelect: (checklist) => {
        setSelectedChecklist(checklist);
      },
    });
  };

  const handleAssignToSelect = (item) => {
    const isSelected = assignTo.some(selected => selected.id === item.id);
    if (isSelected) {
      setAssignTo(assignTo.filter(selected => selected.id !== item.id));
    } else {
      setAssignTo([...assignTo, item]);
    }
  };

  const handleSaveAssignTo = () => {
    setAssignToModalVisible(false);
    setSearchQuery('');
  };

  const handleCancelAssignTo = () => {
    setAssignTo(originalAssignTo);
    setAssignToModalVisible(false);
    setSearchQuery('');
  };

  const handleRemoveAssignTo = (itemToRemove) => {
    setAssignTo(assignTo.filter(item => item.id !== itemToRemove.id));
  };

  const handleAddTask = () => {
    if (!taskTitle.trim()) {
      setErrors({ ...errors, taskTitle: 'Task Title is required' });
      return;
    }
    setTasks([
      ...tasks,
      {
        id: uuid.v4(),
        title: taskTitle,
        description: taskDescription,
        assignees: taskAssignees,
        note: taskNote,
      },
    ]);
    setTaskTitle('');
    setTaskDescription('Repair');
    setTaskAssignees([]);
    setTaskNote('');
    setErrors({ ...errors, taskTitle: '' });
    setTaskModalVisible(false);
  };

  const handleEditTask = (task) => {
    setTaskTitle(task.title);
    setTaskDescription(task.description);
    setTaskAssignees(task.assignees || []);
    setTaskNote(task.note || '');
    setSelectedTask(task);
    setTaskModalVisible(true);
  };

  const handleSaveEditedTask = () => {
    if (!taskTitle.trim()) {
      setErrors({ ...errors, taskTitle: 'Task Title is required' });
      return;
    }
    setTasks(tasks.map(task =>
      task.id === selectedTask.id
        ? { ...task, title: taskTitle, description: taskDescription, assignees: taskAssignees, note: taskNote }
        : task
    ));
    setTaskTitle('');
    setTaskDescription('Repair');
    setTaskAssignees([]);
    setTaskNote('');
    setErrors({ ...errors, taskTitle: '' });
    setTaskModalVisible(false);
    setSelectedTask(null);
  };

  const handleDeleteTask = (taskId) => {
    setTasks(tasks.filter(task => task.id !== taskId));
    setTaskDropdownVisible(false);
  };

  const saveMaintenanceRequest = async () => {
    console.log('saveMaintenanceRequest called');
    if (!userId) {
      showToast('error', 'Error', 'User not logged in. Please log in again.');
      navigation.replace('AdminLogin');
      return;
    }

    if (!workOrderTitle.trim()) {
      setErrors({ ...errors, workOrderTitle: 'Work Order Title is required' });
      showToast('error', 'Error', 'Work Order Title is required');
      return;
    }

    if (!country) {
      setErrors({ ...errors, country: 'Country is required' });
      showToast('error', 'Error', 'Country is required');
      return;
    }

    if (!project) {
      setErrors({ ...errors, project: 'Project is required' });
      showToast('error', 'Error', 'Project is required');
      return;
    }

    setSyncModalVisible(true);
    // Fallback: always hide sync modal after 5 seconds
    setTimeout(() => {
      setSyncModalVisible(false);
    }, 5000);
    try {
      const isOnline = await checkNetworkStatus();
      const finalRequestNumber = requestNumber;
      const updatedRequest = {
        id: requestId,
        requestNumber: finalRequestNumber,
        workOrderTitle,
        memo,
        priority,
        country,
        project,
        assignTo,
        requestType: selectedRequestType,
        equipment: selectedEquipment ? {
          equipmentName: selectedEquipment.equipmentName,
          equipmentNumber: selectedEquipment.equipmentNumber,
          model: selectedEquipment.model,
          meterReading: selectedEquipment.meterReading,
          meterUnit: selectedEquipment.meterUnit,
          status: selectedEquipment.status,
          id: selectedEquipment.id,
        } : null,
        tasks,
        checklistId: selectedChecklist?.id || null,
        checklistTitle: selectedChecklist?.title || null,
        createdAt: new Date().toISOString(),
        userId,
      };

      const storedRequests = await AsyncStorage.getItem('globalMaintenanceRequests');
      let requests = [];
      try {
        requests = storedRequests ? JSON.parse(storedRequests) : [];
        if (!Array.isArray(requests)) {
          requests = [];
          console.warn('Invalid stored requests, resetting to empty array.');
        }
      } catch (error) {
        console.error('AsyncStorage parse error:', error);
        showToast('error', 'Error', 'Failed to load local data.');
      }
      requests = requests.filter(req => req.id !== requestId);
      requests.push(updatedRequest);
      await AsyncStorage.setItem('globalMaintenanceRequests', JSON.stringify(requests));

      if (isOnline) {
        await database()
          .ref(`/GlobalRequestMaintenancereport/admins/${userId}/${requestId}`)
          .set(updatedRequest);
        showToast('success', 'Success', 'Maintenance request updated successfully!');
      } else {
        showToast('info', 'Offline', 'Request updated locally. Will sync when online.');
      }

      setSyncModalVisible(false);
      setTimeout(() => {
        console.log('Setting success modal visible');
        setSuccessModalVisible(true);
      }, 100);
    } catch (error) {
      console.error('Error updating maintenance request:', error);
      let errorMessage = 'Failed to update maintenance request.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      } else if (error.code === 'NETWORK_ERROR' || !(await checkNetworkStatus())) {
        errorMessage = 'Offline mode. Request updated locally and will sync when online.';
      }
      showToast('error', 'Error', errorMessage);
      setSyncModalVisible(false);
    }
  };

  const handleSuccessModalClose = () => {
    console.log('handleSuccessModalClose called');
    setSuccessModalVisible(false);
    setSyncModalVisible(false);
    setAssignToModalVisible(false);
    setRequestTypeModalVisible(false);
    setTaskModalVisible(false);
    setFaultListModalVisible(false);
    setTimeout(() => {
      if (navigation && typeof navigation.replace === 'function') {
        navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'RequestmaintenanceHome' },
          ],
        });
      } else if (navigation && typeof navigation.goBack === 'function') {
        navigation.goBack();
      } else {
        console.warn('Navigation not available');
      }
    }, 200);
  };

  const fetchFilteredTeamMembers = async () => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        setTeamMembers([]);
        return;
      }

      const adminRef = database().ref(`/Globalusers/admins/${adminUid}`);
      const snapshot = await adminRef.once('value');
      if (!snapshot.exists()) {
        setTeamMembers([]);
        return;
      }

      const userData = snapshot.val();
      const userCountry = userData.countries || [];
      const userProject = userData.projects || [];
      const teamEmployModule = userData.modules?.find(m => m.module === 'teamEmploy');
      const permissions = teamEmployModule?.permissions || [];

      setUserPermissions(permissions);
      setAdminProfile({
        id: adminUid,
        fullName: userData.name || userData.fullName || 'Admin',
        employeeNumber: userData.employeeNumber || 'N/A',
        role: userData.role || 'Admin',
        profilePic: userData.profilePic,
        projects: userData.projects || [],
        countries: userData.countries || [],
        isBlocked: userData.isBlocked || false,
      });

      const filtered = await getFilteredTeamMembers(adminUid, userCountry, userProject, permissions);
      setTeamMembers(filtered);
    } catch (error) {
      console.error('Error fetching team members:', error);
      setTeamMembers([]);
      showToast('error', 'Error', 'Failed to load team members.');
    }
  };

  const hasOnlyMineView = Array.isArray(userPermissions)
    ? userPermissions.some(p => p && typeof p === 'string' && p.toLowerCase() === 'onlymineview')
    : false;

  useEffect(() => {
    if (assignToModalVisible) {
      fetchFilteredTeamMembers();
      setOriginalAssignTo([...assignTo]);
    }
  }, [assignToModalVisible]);

  useEffect(() => {
    let defaultProjects = [];
    if (country === 'Zambia') {
      defaultProjects = [
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ];
    } else if (country === 'DRC') {
      defaultProjects = [
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ];
    }
    // If the current project is not in the list, add it
    if (project && !defaultProjects.some((item) => item.value === project)) {
      defaultProjects = [...defaultProjects, { label: project, value: project }];
    }
    setProjectItems(defaultProjects);
    if (!country || (project && !defaultProjects.some((item) => item.value === project))) {
      setProject(null);
    }
  }, [country, project]);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  useEffect(() => {
    let animation;
    if (syncModalVisible) {
      animation = Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      );
      animation.start();
    } else {
      rotateAnim.setValue(0);
    }
    return () => {
      if (animation) animation.stop();
    };
  }, [syncModalVisible]);

  useEffect(() => {
    if (taskAssignToModalVisible) {
      fetchFilteredTeamMembers();
    }
  }, [taskAssignToModalVisible]);

  return (
    <KeyboardAvoidingView
      style={styles.container}
      behavior={Platform.OS === 'ios' ? 'padding' : 'padding'}
      keyboardVerticalOffset={0}
    >
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Update RM</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={saveMaintenanceRequest}>
              <Text style={styles.createButtonText}>Update</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView
        style={styles.mainContent}
        keyboardShouldPersistTaps="handled"
        contentContainerStyle={{ paddingBottom: width * 0.1 }}
      >
        <View style={styles.formGroup}>
          <View style={styles.requestNumberRow}>
            <Text style={styles.requestNumberLabel}>Request #</Text>
            <Text style={styles.requestNumberValue}>{requestNumber}</Text>
          </View>
        </View>

        <View style={styles.formGroup}>
          <CustomInputField
            label="Work Order Title"
            value={workOrderTitle}
            onChangeText={(text) => {
              setWorkOrderTitle(text);
              setErrors({ ...errors, workOrderTitle: text.trim() ? '' : 'Work Order Title is required' });
            }}
            isRequired
            showError={!!errors.workOrderTitle}
            errorMessage={errors.workOrderTitle}
            placeholder="Write Request Title"
          />
        </View>

        <View style={styles.formGroup}>
          <Text style={{ color: '#015185', fontWeight: '500', fontSize: width * 0.04, marginBottom: width * 0.02 }}>
            Country
            <Text style={{ color: 'red', fontWeight: 'bold' }}> *</Text>
          </Text>
          <Dropdown
            style={[
              styles.inputmain,
              errors.country && styles.inputError,
            ]}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            data={countryItems}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select Country"
            value={country}
            onChange={item => {
              setCountry(item.value);
              setProject(null);
              // Set project items based on country
              if (item.value === 'Zambia') {
                setProjectItems([
                  { label: 'Kobold', value: 'Kobold' },
                  { label: 'Kalumbila', value: 'Kalumbila' },
                  { label: 'Kansanshi', value: 'Kansanshi' },
                  { label: 'Kimiteto', value: 'Kimiteto' },
                  { label: 'Mimosa', value: 'Mimosa' },
                ]);
              } else if (item.value === 'DRC') {
                setProjectItems([
                  { label: 'Musompo', value: 'Musompo' },
                  { label: 'IME', value: 'IME' },
                ]);
              } else {
                setProjectItems([]);
              }
            }}
            containerStyle={styles.dropdownContainer}
            itemTextStyle={styles.dropdownText}
            activeColor="#e0e0e0"
          />
          {errors.country && <Text style={styles.errorText}>Country is required</Text>}
        </View>

        <View style={styles.formGroup}>
          <Text style={{ color: '#015185', fontWeight: '500', fontSize: width * 0.04, marginBottom: width * 0.02 }}>
            Project
            <Text style={{ color: 'red', fontWeight: 'bold' }}> *</Text>
          </Text>
          <Dropdown
            style={[
              styles.inputmain,
              !country && styles.disabledDropdown,
              errors.project && styles.inputError,
            ]}
            placeholderStyle={styles.placeholderStyle}
            selectedTextStyle={styles.dropdownText}
            data={projectItems}
            maxHeight={300}
            labelField="label"
            valueField="value"
            placeholder="Select Project"
            value={project}
            onChange={item => setProject(item.value)}
            containerStyle={styles.dropdownContainer}
            itemTextStyle={styles.dropdownText}
            activeColor="#e0e0e0"
            disable={!country}
          />
          {errors.project && <Text style={styles.errorText}>Project is required</Text>}
        </View>

        <View style={styles.formGroup}>
          <CustomInputField
            label="Memo"
            value={memo}
            onChangeText={setMemo}
            multiline
            placeholder="Write Memo"
          />
        </View>

        <View style={styles.formGroup}>
          <Text style={styles.label}>Priority</Text>
          <View style={styles.priorityContainer}>
            {priorityOptions.map((option) => {
              const isSelected = priority === option.label;
              return (
                <View key={option.label} style={styles.priorityItem}>
                  <TouchableOpacity
                    style={[
                      styles.priorityButton,
                      isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                    ]}
                    onPress={() => handlePrioritySelect(option.label)}
                  >
                    <Image source={option.image} style={styles.priorityImage} />
                    <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                      {option.label}
                    </Text>
                  </TouchableOpacity>
                </View>
              );
            })}
          </View>
        </View>

        <View style={styles.formGroup}>
          <TouchableOpacity style={styles.selectRow} onPress={handleEquipmentSelect}>
            <Text style={styles.selectLabel}>Vehicle</Text>
            <View style={styles.selectRight}>
              <Text style={styles.selectText}>Select</Text>
              <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
            </View>
          </TouchableOpacity>
          {selectedEquipment ? (
            <View style={{backgroundColor:'#f0f0f0', borderRadius:8, padding:10, marginTop:6}}>
              <Text style={{fontWeight:'bold', color:'#015185', fontSize:16}}>{selectedEquipment.equipmentName || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Number: {selectedEquipment.equipmentNumber || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Model: {selectedEquipment.model || 'N/A'}</Text>
              <Text style={{color:'#333'}}>Meter: {selectedEquipment.meterReading} {selectedEquipment.meterUnit}</Text>
              <Text style={{color:'#333'}}>Status: {selectedEquipment.status || 'N/A'}</Text>
            </View>
          ) : (
            <Text style={styles.selectedText}>No Vehicle Selected</Text>
          )}
        </View>

        <View style={styles.formGroup}>
          <TouchableOpacity style={styles.selectRow} onPress={() => setAssignToModalVisible(true)}>
            <Text style={styles.selectLabel}>Assign to</Text>
            <View style={styles.selectRight}>
              <Text style={styles.selectText}>Select</Text>
              <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
            </View>
          </TouchableOpacity>
          {assignTo.length > 0 ? (
            <View style={{ marginTop: width * 0.01 }}>
              <ScrollView
                horizontal={true}
                showsHorizontalScrollIndicator={false}
                contentContainerStyle={{ flexDirection: 'row', alignItems: 'center' }}
              >
                {assignTo.map((item, index) => (
                  <View key={index} style={{
                    flexDirection: 'row',
                    alignItems: 'center',
                    backgroundColor: '#e6f0ff',
                    borderRadius: 8,
                    paddingVertical: 6,
                    paddingHorizontal: 10,
                    marginRight: 8,
                    marginBottom: 4,
                  }}>
                    {item.profilePic ? (
                      <Image
                        source={{ uri: item.profilePic.url }}
                        style={{ width: 32, height: 32, borderRadius: 16, marginRight: 8 }}
                      />
                    ) : (
                      <View style={{
                        width: 32,
                        height: 32,
                        borderRadius: 16,
                        backgroundColor: '#ddd',
                        marginRight: 8,
                        alignItems: 'center',
                        justifyContent: 'center',
                      }}>
                        <MaterialIcons name="person" size={18} color="#888" />
                      </View>
                    )}
                    <View style={{ marginRight: 6 }}>
                      <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 14 }}>{item.fullName}</Text>
                      <Text style={{ color: '#333', fontSize: 11 }}>Emp#: {item.employeeNumber} | Role: {item.role}</Text>
                    </View>
                    <TouchableOpacity
                      style={{ marginLeft: 2, padding: 2 }}
                      onPress={() => handleRemoveAssignTo(item)}
                    >
                      <View style={{
                        width: 18,
                        height: 18,
                        borderRadius: 9,
                        backgroundColor: '#d32f2f',
                        justifyContent: 'center',
                        alignItems: 'center',
                      }}>
                        <MaterialIcons name="close" size={13} color="#fff" />
                      </View>
                    </TouchableOpacity>
                  </View>
                ))}
              </ScrollView>
            </View>
          ) : (
            <Text style={styles.selectedText}>No one assigned</Text>
          )}
        </View>

        <View style={styles.formGroup}>
          <TouchableOpacity style={styles.selectRow} onPress={() => setRequestTypeModalVisible(true)}>
            <Text style={styles.selectLabel}>Request Type</Text>
            <View style={styles.selectRight}>
              <Text style={styles.selectText}>{selectedRequestType || 'Select'}</Text>
              <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
            </View>
          </TouchableOpacity>
        </View>

        <View style={styles.formGroup}>
          <View style={styles.taskListHeader}>
            <Text style={styles.taskListTitle}>Task List</Text>
            <TouchableOpacity style={styles.addTaskButton} onPress={() => setTaskModalVisible(true)}>
              <MaterialIcons name="add" size={width * 0.04} color="#fff" />
              <Text style={styles.addTaskButtonText}>Add Task</Text>
            </TouchableOpacity>
          </View>
          {tasks.length > 0 ? (
            <View style={styles.taskListContainer}>
              {tasks.map((task, index) => (
                <View key={task.id || index} style={styles.taskCard}>
                  <View style={styles.taskCardHeader}>
                    <Text style={styles.taskCardTitle}>{task.title}</Text>
                    <TouchableOpacity
                      onPress={(event) => {
                        setSelectedTask(task);
                        const { pageX, pageY } = event.nativeEvent;
                        const screenHeight = Dimensions.get('window').height;
                        let top = pageY;
                        if (pageY + DROPDOWN_HEIGHT + MARGIN > screenHeight) {
                          top = pageY - DROPDOWN_HEIGHT - MARGIN;
                        }
                        setTaskDropdownPosition({ x: pageX - 120, y: top });
                        setTaskDropdownVisible(true);
                      }}
                    >
                      <MaterialIcons name="more-vert" size={22} color="#222" />
                    </TouchableOpacity>
                  </View>
                  <Text style={styles.taskCardType}>{task.description}</Text>
                  <MaterialIcons name="person" size={22} color="#888" style={{ marginTop: 2 }} />
                </View>
              ))}
              <Modal
                visible={taskDropdownVisible}
                transparent
                animationType="fade"
                onRequestClose={() => setTaskDropdownVisible(false)}
              >
                <TouchableOpacity
                  style={styles.taskDropdownOverlay}
                  onPress={() => setTaskDropdownVisible(false)}
                  activeOpacity={1}
                >
                  <View
                    style={[
                      styles.taskDropdownContainer,
                      {
                        position: 'absolute',
                        top: taskDropdownPosition.y,
                        left: taskDropdownPosition.x,
                      },
                    ]}
                  >
                    <TouchableOpacity
                      style={styles.taskDropdownItem}
                      onPress={() => {
                        handleEditTask(selectedTask);
                        setTaskDropdownVisible(false);
                      }}
                    >
                      <MaterialIcons name="edit" size={20} color="#3481BC" style={{ marginRight: 8 }} />
                      <Text style={styles.taskDropdownText}>Edit</Text>
                    </TouchableOpacity>
                    <TouchableOpacity
                      style={styles.taskDropdownItem}
                      onPress={() => handleDeleteTask(selectedTask.id)}
                    >
                      <MaterialIcons name="delete" size={20} color="#C2185B" style={{ marginRight: 8 }} />
                      <Text style={styles.taskDropdownText}>Delete</Text>
                    </TouchableOpacity>
                  </View>
                </TouchableOpacity>
              </Modal>
            </View>
          ) : (
            <Text style={styles.noTasksText}>No tasks added</Text>
          )}
        </View>
      </ScrollView>

      {/* Add Task Modal */}
      <Modal
        visible={taskModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setTaskModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.addTaskModalContentNew}>
            <Text style={styles.addTaskTitleNew}>{selectedTask ? 'Edit Task' : 'Add Task'}</Text>
            {/* Task Row */}
            <View style={styles.addTaskRowNew}>
              <Text style={styles.addTaskLabelNew}>Task</Text>
              <TextInput
                style={styles.addTaskInputNew}
                value={taskTitle}
                onChangeText={(text) => {
                  setTaskTitle(text);
                  setErrors({ ...errors, taskTitle: text.trim() ? '' : 'Task Title is required' });
                }}
                // No placeholder
              />
            </View>
            {/* Type Row */}
            <View style={styles.addTaskRowNew}>
              <Text style={styles.addTaskLabelNew}>Type</Text>
              <View style={styles.addTaskTypeButtonsNew}>
                <TouchableOpacity
                  style={[
                    styles.addTaskTypeButtonNew,
                    taskDescription === 'Repair' ? styles.addTaskTypeButtonSelectedNew : styles.addTaskTypeButtonUnselectedNew,
                  ]}
                  onPress={() => setTaskDescription('Repair')}
                >
                  <Text style={taskDescription === 'Repair' ? styles.addTaskTypeTextSelectedNew : styles.addTaskTypeTextNew}>
                    Repair
                  </Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[
                    styles.addTaskTypeButtonNew,
                    taskDescription === 'Replace' ? styles.addTaskTypeButtonSelectedNew : styles.addTaskTypeButtonUnselectedNew,
                  ]}
                  onPress={() => setTaskDescription('Replace')}
                >
                  <Text style={taskDescription === 'Replace' ? styles.addTaskTypeTextSelectedNew : styles.addTaskTypeTextNew}>
                    Replace
                  </Text>
                </TouchableOpacity>
              </View>
            </View>
            {/* Assign to Row */}
            <View style={styles.addTaskRowNew}>
              <Text style={styles.addTaskLabelNew}>Assign to</Text>
              <TouchableOpacity
                style={styles.addTaskSetRightNew}
                onPress={() => {
                  setTaskModalVisible(false);
                  setTimeout(() => setTaskAssignToModalVisible(true), 300);
                }}
              >
                <Text style={styles.addTaskSetTextNew}>Set</Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View>
            {taskAssignees.length > 0 && (
              <View style={{ marginTop: width * 0.01 }}>
                <ScrollView
                  horizontal={true}
                  showsHorizontalScrollIndicator={false}
                  contentContainerStyle={{ flexDirection: 'row', alignItems: 'center' }}
                >
                  {taskAssignees.map((item, index) => (
                    <View key={index} style={{
                      flexDirection: 'row',
                      alignItems: 'center',
                      backgroundColor: '#e6f0ff',
                      borderRadius: 8,
                      paddingVertical: 6,
                      paddingHorizontal: 10,
                      marginRight: 8,
                      marginBottom: 4,
                    }}>
                      {item.profilePic ? (
                        <Image
                          source={{ uri: item.profilePic.url }}
                          style={{ width: 32, height: 32, borderRadius: 16, marginRight: 8 }}
                        />
                      ) : (
                        <View style={{
                          width: 32,
                          height: 32,
                          borderRadius: 16,
                          backgroundColor: '#ddd',
                          marginRight: 8,
                          alignItems: 'center',
                          justifyContent: 'center',
                        }}>
                          <MaterialIcons name="person" size={18} color="#888" />
                        </View>
                      )}
                      <View style={{ marginRight: 6 }}>
                        <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 14 }}>{item.fullName}</Text>
                        <Text style={{ color: '#333', fontSize: 11 }}>Emp#: {item.employeeNumber} | Role: {item.role}</Text>
                      </View>
                      <TouchableOpacity
                        style={{ marginLeft: 2, padding: 2 }}
                        onPress={() => setTaskAssignees(taskAssignees.filter(u => u.id !== item.id))}
                      >
                        <View style={{
                          width: 18,
                          height: 18,
                          borderRadius: 9,
                          backgroundColor: '#d32f2f',
                          justifyContent: 'center',
                          alignItems: 'center',
                        }}>
                          <MaterialIcons name="close" size={13} color="#fff" />
                        </View>
                      </TouchableOpacity>
                    </View>
                  ))}
                </ScrollView>
              </View>
            )}
            {/* Note Row */}
            <View style={styles.addTaskRowNew}>
              <Text style={styles.addTaskLabelNew}>Note</Text>
              <TextInput
                style={styles.addTaskInputNew}
                value={taskNote}
                onChangeText={setTaskNote}
              />
            </View>
            {/* Fault List Row */}
            <View style={styles.addTaskRowNew}>
              <Text style={styles.addTaskLabelNew}>Fault List</Text>
              <TouchableOpacity
                style={styles.addTaskSetRightNew}
                onPress={() => setFaultListModalVisible(true)}
              >
                <Text style={styles.addTaskSetTextNew}>Select</Text>
                <MaterialIcons name="chevron-right" size={width * 0.05} color="#015185" />
              </TouchableOpacity>
            </View>
            {/* Button Row */}
            <View style={styles.addTaskButtonRowNew}>
              <TouchableOpacity
                style={[styles.addTaskActionButtonNew, styles.addTaskCloseButtonNew]}
                onPress={() => {
                  setTaskModalVisible(false);
                  setSelectedTask(null);
                  setTaskTitle('');
                  setTaskDescription('Repair');
                }}
              >
                <Text style={styles.addTaskCloseButtonTextNew}>Close</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.addTaskActionButtonNew, styles.addTaskSaveButtonNew]}
                onPress={selectedTask ? handleSaveEditedTask : handleAddTask}
              >
                <Text style={styles.addTaskSaveButtonTextNew}>{selectedTask ? 'Update' : 'Save'}</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Fault List Modal */}
      <Modal
        visible={faultListModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setFaultListModalVisible(false)}
      >
        <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.2)' }}>
          <View style={{ width: '90%', backgroundColor: '#fff', borderRadius: 12, overflow: 'hidden' }}>
            <View style={{ padding: 16 }}>
              <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: width * 0.05, textAlign: 'center', marginBottom: 16 }}>
                Equipments fault list
              </Text>
              {faultListData.length > 0 ? (
                <FlatList
                  data={faultListData}
                  keyExtractor={(item, index) => index.toString()}
                  renderItem={({ item }) => (
                    <View style={{ padding: 12, borderBottomWidth: 1, borderBottomColor: '#f0f0f0' }}>
                      <Text style={{ color: '#015185', fontWeight: 'bold', fontSize: 16 }}>{item.title}</Text>
                      <Text style={{ color: '#666', fontSize: 12, marginTop: 2 }}>{item.date}</Text>
                      <Text style={{ color: '#015185', fontSize: 14, textAlign: 'right', marginTop: 4 }}>{item.code}</Text>
                    </View>
                  )}
                />
              ) : (
                <Text style={{ textAlign: 'center', color: '#888', fontSize: 16, padding: 20 }}>
                  No faults available.
                </Text>
              )}
            </View>
            <View style={{ flexDirection: 'row', justifyContent: 'center', padding: 16, gap: 12 }}>
              <TouchableOpacity
                style={{ backgroundColor: '#015185', paddingVertical: 8, paddingHorizontal: 20, borderRadius: 4 }}
                onPress={() => setFaultListModalVisible(false)}
              >
                <Text style={{ color: '#fff', fontWeight: '500' }}>Close</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={{ backgroundColor: '#015185', paddingVertical: 8, paddingHorizontal: 20, borderRadius: 4 }}
                onPress={() => setFaultListModalVisible(false)}
              >
                <Text style={{ color: '#fff', fontWeight: '500' }}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[
                  styles.syncIcon,
                  { transform: [{ rotate: rotateInterpolate }] },
                ]}
              />
              <Text style={styles.syncModalTextadd}>
                Updating Maintenance Request...
              </Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={successModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setSuccessModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalText}>Success</Text>
            <Text style={styles.successModalSubText}>
              Maintenance request updated successfully!
            </Text>
            <TouchableOpacity
              style={styles.successModalButton}
              onPress={handleSuccessModalClose}
            >
              <Text style={styles.successModalButtonText}>OK</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={requestTypeModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setRequestTypeModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.requestTypeModalContent}>
            {requestTypes.map((item, idx) => {
              const isSelected = selectedRequestType === item.name;
              return (
                <React.Fragment key={item.id}>
                  <TouchableOpacity
                    style={styles.requestTypeRow}
                    onPress={() => handleRequestTypeSelect(item)}
                  >
                    <Text style={[styles.requestTypeLabel, isSelected && styles.requestTypeLabelSelected]}>
                      {item.name}
                    </Text>
                    <View style={styles.requestTypeRadioOuter}>
                      {isSelected ? <View style={styles.requestTypeRadioInner} /> : null}
                    </View>
                  </TouchableOpacity>
                  {idx !== requestTypes.length - 1 && <View style={styles.requestTypeDivider} />}
                </React.Fragment>
              );
            })}
            <View style={styles.requestTypeButtonRow}>
              <TouchableOpacity
                style={styles.requestTypeCloseButton}
                onPress={() => setRequestTypeModalVisible(false)}
              >
                <Text style={styles.requestTypeCloseButtonText}>Close</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={assignToModalVisible}
        transparent
        animationType="slide"
        onRequestClose={handleCancelAssignTo}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, { maxHeight: '80%' }]}> 
            <Text style={styles.modalTitle}>Assign To</Text>
            <TextInput
              style={styles.searchInput}
              placeholder="Search team members..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <FlatList
              data={
                hasOnlyMineView
                  ? [adminProfile ? { ...adminProfile, isSelf: true } : null].filter(Boolean)
                  : teamMembers.filter(member =>
                      (member.fullName?.toLowerCase() || '').includes(searchQuery.toLowerCase()) ||
                      (member.employeeNumber?.toLowerCase() || '').includes(searchQuery.toLowerCase()) ||
                      (member.role?.toLowerCase() || '').includes(searchQuery.toLowerCase())
                    )
              }
              keyExtractor={item => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={{
                    flexDirection: 'row',
                    alignItems: 'center',
                    paddingVertical: 10,
                    borderBottomWidth: 1,
                    borderBottomColor: '#eee',
                    backgroundColor: assignTo.some(selected => selected.id === item.id) ? '#e6f0ff' : '#fff',
                  }}
                  onPress={() => handleAssignToSelect(item)}
                >
                  {item.profilePic ? (
                    <Image
                      source={{ uri: item.profilePic.url }}
                      style={{ width: 40, height: 40, borderRadius: 20, marginRight: 10 }}
                    />
                  ) : (
                    <View style={{
                      width: 40,
                      height: 40,
                      borderRadius: 20,
                      backgroundColor: '#ddd',
                      marginRight: 10,
                      alignItems: 'center',
                      justifyContent: 'center',
                    }}>
                      <MaterialIcons name="person" size={24} color="#888" />
                    </View>
                  )}
                  <View style={{ flex: 1 }}>
                    <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 16 }}>{item.fullName}</Text>
                    <Text style={{ color: '#333', fontSize: 13 }}>
                      Emp#: {item.employeeNumber} | Role: {item.role}
                    </Text>
                  </View>
                  {assignTo.some(selected => selected.id === item.id) && (
                    <MaterialIcons name="check" size={24} color="#015185" style={{ marginLeft: 8 }} />
                  )}
                </TouchableOpacity>
              )}
              style={{ marginBottom: 10 }}
              ListEmptyComponent={<Text style={styles.defaultText}>No team members found</Text>}
            />
            <View style={{ alignItems: 'center', marginTop: 10 }}>
              <TouchableOpacity
                style={styles.saveAssignButton}
                onPress={handleSaveAssignTo}
              >
                <Text style={styles.saveAssignButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Add Task Assign To Modal */}
      <Modal
        visible={taskAssignToModalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setTaskAssignToModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={[styles.modalContent, { maxHeight: '80%' }]}> 
            <Text style={styles.modalTitle}>Assign To (Task)</Text>
            <TextInput
              style={styles.searchInput}
              placeholder="Search team members..."
              placeholderTextColor="#888"
              value={taskAssignSearchQuery}
              onChangeText={setTaskAssignSearchQuery}
            />
            <FlatList
              data={teamMembers.filter(member =>
                (member.fullName?.toLowerCase() || '').includes(taskAssignSearchQuery.toLowerCase()) ||
                (member.employeeNumber?.toLowerCase() || '').includes(taskAssignSearchQuery.toLowerCase()) ||
                (member.role?.toLowerCase() || '').includes(taskAssignSearchQuery.toLowerCase())
              )}
              keyExtractor={item => item.id}
              renderItem={({ item }) => (
                <TouchableOpacity
                  style={{
                    flexDirection: 'row',
                    alignItems: 'center',
                    paddingVertical: 10,
                    borderBottomWidth: 1,
                    borderBottomColor: '#eee',
                    backgroundColor: taskAssignees.some(selected => selected.id === item.id) ? '#e6f0ff' : '#fff',
                  }}
                  onPress={() => {
                    if (taskAssignees.some(u => u.id === item.id)) {
                      setTaskAssignees(taskAssignees.filter(u => u.id !== item.id));
                    } else {
                      setTaskAssignees([...taskAssignees, item]);
                    }
                  }}
                >
                  {item.profilePic ? (
                    <Image
                      source={{ uri: item.profilePic.url }}
                      style={{ width: 40, height: 40, borderRadius: 20, marginRight: 10 }}
                    />
                  ) : (
                    <View style={{
                      width: 40,
                      height: 40,
                      borderRadius: 20,
                      backgroundColor: '#ddd',
                      marginRight: 10,
                      alignItems: 'center',
                      justifyContent: 'center',
                    }}>
                      <MaterialIcons name="person" size={24} color="#888" />
                    </View>
                  )}
                  <View style={{ flex: 1 }}>
                    <Text style={{ fontWeight: 'bold', color: '#015185', fontSize: 16 }}>{item.fullName}</Text>
                    <Text style={{ color: '#333', fontSize: 13 }}>
                      Emp#: {item.employeeNumber} | Role: {item.role}
                    </Text>
                  </View>
                  {taskAssignees.some(selected => selected.id === item.id) && (
                    <MaterialIcons name="check" size={24} color="#015185" style={{ marginLeft: 8 }} />
                  )}
                </TouchableOpacity>
              )}
              style={{ marginBottom: 10 }}
              ListEmptyComponent={<Text style={styles.defaultText}>No team members found</Text>}
            />
            <View style={{ alignItems: 'center', marginTop: 10 }}>
              <TouchableOpacity
                style={styles.saveAssignButton}
                onPress={() => {
                  setTaskAssignToModalVisible(false);
                  setTimeout(() => setTaskModalVisible(true), 300);
                }}
              >
                <Text style={styles.saveAssignButtonText}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </KeyboardAvoidingView>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.03,
  },
  labelContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: width * 0.02,
  },
  label: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  inputContainer: {
    position: 'relative',
  },
  tagsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginBottom: width * 0.01,
  },
  tag: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e0e0e0',
    borderRadius: width * 0.05,
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.02,
    marginRight: width * 0.02,
    marginBottom: width * 0.01,
  },
  tagText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginRight: width * 0.01,
  },
  tagCloseButton: {
    padding: width * 0.005,
  },
  tagCloseCircle: {
    width: width * 0.035,
    height: width * 0.035,
    borderRadius: (width * 0.035) / 2,
    backgroundColor: '#d32f2f',
    justifyContent: 'center',
    alignItems: 'center',
  },
  input: {
    flex: 1,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  multilineInput: {
    height: width * 0.2,
    textAlignVertical: 'top',
  },
  inputError: {
    borderColor: 'red',
  },
  disabledInput: {
    backgroundColor: '#f5f5f5',
  },
  noBackground: {
    backgroundColor: 'transparent',
  },
  noBorder: {
    borderWidth: 0,
  },
  inputWithTags: {
    paddingTop: 10,
    paddingBottom: 10,
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
  inputIcon: {
    position: 'absolute',
    right: 12,
    top: '50%',
    transform: [{ translateY: -width * 0.03 }],
  },
  labelSelectButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.01,
  },
  selectText: {
    fontSize: width * 0.038,
    color: '#015185',
    marginRight: width * 0.01,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
    backgroundColor: '#fff',
  },
  createButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  mainContent: {
    flex: 1,
  },
  formGroup: {
    marginBottom: width * 0.03,
    paddingHorizontal: width * 0.03,
    paddingVertical: width * 0.015,
  },
  label: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
    marginBottom: width * 0.02,
  },
  requestNumberRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F2F2F2',
    borderRadius: 6,
    paddingHorizontal: 16,
    paddingVertical: 14,
    marginBottom: width * 0.02,
  },
  requestNumberLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
  },
  requestNumberValue: {
    fontSize: width * 0.038,
    color: '#025383',
    fontWeight: 'bold',
  },
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: width * 0.02,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  row: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
    maxHeight: '80%',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '70%',
    alignItems: 'center',
  },
  successModalText: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.025,
  },
  successModalSubText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  successModalButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.08,
    borderRadius: width * 0.02,
  },
  successModalButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  modalTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  taskListHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: width * 0.02,
  },
  taskListTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
  },
  taskListContainer: {
    marginTop: width * 0.02,
  },
  taskCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: width * 0.03,
    marginBottom: width * 0.02,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  taskCardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: width * 0.01,
  },
  taskCardTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
  },
  taskCardType: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: width * 0.01,
  },
  taskDropdownOverlay: {
    flex: 1,
    backgroundColor: 'transparent',
  },
  taskDropdownContainer: {
    backgroundColor: '#fff',
    borderRadius: 8,
    elevation: 8,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 6,
    width: 120,
    zIndex: 999,
  },
  taskDropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 14,
    paddingHorizontal: 18,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    backgroundColor: '#fff',
  },
  taskDropdownText: {
    fontSize: width * 0.04,
    color: '#222',
  },
  addTaskButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#015185',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
  },
  addTaskButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
  },
  selectRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F5F5F5',
    borderRadius: 4,
    paddingVertical: width * 0.035,
    paddingHorizontal: width * 0.03,
    marginBottom: width * 0.01,
  },
  selectLabel: {
    fontSize: width * 0.038,
    color: '#707173',
    fontWeight: '500',
  },
  selectRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectText: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginRight: 2,
  },
  selectedText: {
    fontSize: width * 0.035,
    color: '#333',
    marginTop: width * 0.01,
    textAlign: 'left',
  },
  noTasksText: {
    fontSize: width * 0.038,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.02,
  },
  addTaskModalContentNew: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: width * 0.03,
    width: '90%',
    alignItems: 'stretch',
    alignSelf: 'center',
    minWidth: 300,
  },
  addTaskTitleNew: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  addTaskRowNew: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#f5f5f5',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingVertical: width * 0.035,
    paddingHorizontal: width * 0.03,
  },
  addTaskLabelNew: {
    fontSize: width * 0.038,
    color: '#015185',
    fontWeight: '500',
    flex: 1,
  },
  addTaskInputNew: {
    flex: 2,
    fontSize: width * 0.038,
    color: '#222',
    backgroundColor: 'transparent',
    borderWidth: 0,
    textAlign: 'right',
  },
  addTaskTypeButtonsNew: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: width * 0.02,
  },
  addTaskTypeButtonNew: {
    borderWidth: 1,
    borderRadius: 4,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.04,
    marginHorizontal: width * 0.01,
  },
  addTaskTypeButtonSelectedNew: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  addTaskTypeButtonUnselectedNew: {
    backgroundColor: '#fff',
    borderColor: '#015185',
  },
  addTaskTypeTextNew: {
    color: '#015185',
    fontWeight: '500',
    fontSize: width * 0.038,
  },
  addTaskTypeTextSelectedNew: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.038,
  },
  addTaskSetRightNew: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  addTaskSetTextNew: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginRight: 2,
  },
  addTaskValueNew: {
    color: '#222',
    fontSize: width * 0.038,
    fontWeight: '400',
  },
  addTaskButtonRowNew: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: width * 0.04,
    gap: width * 0.03,
    backgroundColor: '#fff',
    borderBottomLeftRadius: 12,
    borderBottomRightRadius: 12,
    paddingBottom: width * 0.02,
  },
  addTaskActionButtonNew: {
    borderRadius: 4,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.07,
    marginLeft: width * 0.02,
  },
  addTaskCloseButtonNew: {
    backgroundColor: '#015185',
  },
  addTaskCloseButtonTextNew: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  addTaskSaveButtonNew: {
    backgroundColor: '#015185',
  },
  addTaskSaveButtonTextNew: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  requestTypeModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.04,
    padding: width * 0.03,
    width: '90%',
    alignSelf: 'center',
    marginTop: width * 0.15,
  },
  requestTypeRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingVertical: width * 0.045,
    paddingHorizontal: width * 0.01,
    backgroundColor: 'transparent',
  },
  requestTypeLabel: {
    color: '#015185',
    fontSize: width * 0.045,
    fontWeight: '400',
  },
  requestTypeLabelSelected: {
    fontWeight: 'bold',
  },
  requestTypeRadioOuter: {
    width: width * 0.055,
    height: width * 0.055,
    borderRadius: width * 0.0275,
    borderWidth: 2,
    borderColor: '#015185',
    alignItems: 'center',
    justifyContent: 'center',
  },
  requestTypeRadioInner: {
    width: width * 0.03,
    height: width * 0.03,
    borderRadius: width * 0.015,
    backgroundColor: '#015185',
  },
  requestTypeDivider: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginLeft: 0,
    marginRight: 0,
  },
  requestTypeButtonRow: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: width * 0.04,
  },
  requestTypeCloseButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.07,
  },
  requestTypeCloseButtonText: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  searchInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 3,
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.025,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  selectedItemsContainer: {
    marginTop: width * 0.01,
  },
  selectedItemsRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectedItem: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e0e0e0',
    borderRadius: width * 0.05,
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.02,
    marginRight: width * 0.02,
    marginBottom: width * 0.01,
  },
  selectedItemText: {
    fontSize: width * 0.035,
    color: '#015185',
    marginRight: width * 0.02,
  },
  closeButton: {
    padding: width * 0.005,
  },
  closeCircle: {
    width: width * 0.04,
    height: width * 0.04,
    borderRadius: (width * 0.04) / 2,
    backgroundColor: '#d32f2f',
    justifyContent: 'center',
    alignItems: 'center',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderColor: '#ddd',
    borderRadius: 5,
    marginTop: 4,
  },
  dropdownText: {
    fontSize: 16,
    color: '#333',
  },
  placeholderStyle: {
    fontSize: 16,
    color: '#888',
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  inputmain: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  saveAssignButton: {
    backgroundColor: '#015185',
    borderRadius: width * 0.02,
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.05,
    alignItems: 'center',
    justifyContent: 'center',
    alignSelf: 'center',
  },
  saveAssignButtonText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: '500',
  },
  assignModalContentImproved: {
    backgroundColor: '#fff',
    borderRadius: 14,
    padding: width * 0.06,
    width: '92%',
    maxWidth: 400,
    alignItems: 'stretch',
    alignSelf: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.15,
    shadowRadius: 8,
    elevation: 8,
  },
  assignModalTitleImproved: {
    fontSize: width * 0.052,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: width * 0.04,
  },
  assignSearchRowImproved: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F5F5F5',
    borderRadius: 8,
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.03,
    marginBottom: width * 0.03,
  },
  assignSearchInputImproved: {
    flex: 1,
    fontSize: width * 0.038,
    color: '#222',
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  assignUserCardImproved: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: width * 0.035,
    marginBottom: width * 0.02,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.07,
    shadowRadius: 3,
    elevation: 2,
  },
  assignUserCardSelectedImproved: {
    borderWidth: 2,
    borderColor: '#015185',
    backgroundColor: '#e6f0fa',
  },
  assignUserRowImproved: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: width * 0.01,
  },
  assignUserLabelImproved: {
    fontSize: width * 0.037,
    color: '#707173',
    fontWeight: '500',
  },
  assignUserValueImproved: {
    fontSize: width * 0.037,
    color: '#222',
    marginTop: 2,
  },
  assignRolePillImproved: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingVertical: 4,
    paddingHorizontal: 12,
    alignItems: 'center',
    justifyContent: 'center',
    marginLeft: 8,
  },
  assignRoleTextImproved: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: '500',
  },
  assignSelectButtonImproved: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 18,
    alignSelf: 'flex-end',
  },
  assignSelectButtonSelectedImproved: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  assignSelectButtonTextImproved: {
    color: '#015185',
    fontWeight: '600',
    fontSize: width * 0.038,
  },
  assignSelectButtonTextSelectedImproved: {
    color: '#fff',
    fontWeight: '600',
    fontSize: width * 0.038,
  },
  assignUserDividerImproved: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginVertical: width * 0.01,
  },
  assignButtonRowImproved: {
    flexDirection: 'row',
    justifyContent: 'flex-end',
    marginTop: width * 0.04,
    gap: width * 0.03,
  },
  assignActionButtonImproved: {
    borderRadius: 6,
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.07,
    marginLeft: width * 0.02,
  },
  assignCancelButtonImproved: {
    borderWidth: 1,
    borderColor: '#015185',
    backgroundColor: '#fff',
  },
  assignCancelButtonTextImproved: {
    color: '#333',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
  assignSaveButtonImproved: {
    backgroundColor: '#015185',
  },
  assignSaveButtonTextImproved: {
    color: '#fff',
    fontWeight: '500',
    fontSize: width * 0.04,
  },
});

export default UpdateRequestMaintenance;