/**
 * OfflineRequestMaintenanceHelper.js
 * Helper module for managing offline data for Request Maintenance
 */

import AsyncStorage from '@react-native-async-storage/async-storage';

class OfflineRequestMaintenanceHelper {
  /**
   * Save request maintenance locally for offline sync
   * @param {Object} request - Complete request maintenance data
   * @returns {Promise<boolean>} Success status
   */
  static async saveOfflineRequest(request) {
    try {
      // Get existing offline requests
      const offlineRequestsRaw = await AsyncStorage.getItem('offlineMaintenanceRequests');
      const offlineRequests = offlineRequestsRaw ? JSON.parse(offlineRequestsRaw) : [];
      
      // Add new request with offline flag
      const offlineRequest = {
        ...request,
        offlineCreated: true,
        offlineSyncPending: true,
        offlineCreatedAt: new Date().toISOString(),
      };
      
      offlineRequests.push(offlineRequest);
      
      // Save to AsyncStorage
      await AsyncStorage.setItem('offlineMaintenanceRequests', JSON.stringify(offlineRequests));
      
      console.log('✅ Saved offline maintenance request:', request.id);
      console.log('📊 Total offline requests:', offlineRequests.length);
      
      return true;
    } catch (error) {
      console.error('❌ Error saving offline request:', error);
      return false;
    }
  }

  /**
   * Get all offline requests waiting to be synced
   * @returns {Promise<Array>} Array of offline requests
   */
  static async getOfflineRequests() {
    try {
      const offlineRequestsRaw = await AsyncStorage.getItem('offlineMaintenanceRequests');
      const offlineRequests = offlineRequestsRaw ? JSON.parse(offlineRequestsRaw) : [];
      return offlineRequests;
    } catch (error) {
      console.error('❌ Error getting offline requests:', error.message);
      return [];
    }
  }

  /**
   * Get all requests (both online cached and offline)
   * @returns {Promise<Array>} Array of all requests
   */
  static async getAllRequests() {
    try {
      // Get cached online requests
      const cachedRequestsRaw = await AsyncStorage.getItem('globalMaintenanceRequests');
      const cachedRequests = cachedRequestsRaw ? JSON.parse(cachedRequestsRaw) : [];
      
      // Get offline requests
      const offlineRequests = await this.getOfflineRequests();
      
      // Combine both arrays
      const allRequests = [...cachedRequests, ...offlineRequests];
      
      return allRequests;
    } catch (error) {
      console.error('❌ Error getting all requests:', error.message);
      return [];
    }
  }

  /**
   * Sync offline requests to MongoDB
   * @param {Function} uploadCallback - Function to upload request to MongoDB
   * @returns {Promise<Object>} Sync results
   */
  static async syncOfflineRequests(uploadCallback) {
    try {
      const offlineRequests = await this.getOfflineRequests();
      
      if (offlineRequests.length === 0) {
        console.log('✅ No offline requests to sync');
        return { success: true, synced: 0, failed: 0 };
      }
      
      console.log('🔄 Starting sync of', offlineRequests.length, 'offline requests...');
      
      let synced = 0;
      let failed = 0;
      const failedRequests = [];
      
      for (const request of offlineRequests) {
        try {
          // Call the upload callback function
          const result = await uploadCallback(request);
          
          if (result && result.success) {
            synced++;
            console.log('✅ Synced request:', request.id);
          } else {
            failed++;
            failedRequests.push(request);
            console.error('❌ Failed to sync request:', request.id);
          }
        } catch (error) {
          failed++;
          failedRequests.push(request);
          console.error('❌ Error syncing request:', request.id, error);
        }
      }
      
      // Update offline requests (keep only failed ones)
      await AsyncStorage.setItem('offlineMaintenanceRequests', JSON.stringify(failedRequests));
      
      console.log('✅ Sync complete - Synced:', synced, 'Failed:', failed);
      
      return {
        success: true,
        synced,
        failed,
        total: offlineRequests.length,
      };
    } catch (error) {
      console.error('❌ Error syncing offline requests:', error);
      return {
        success: false,
        synced: 0,
        failed: 0,
        error: error.message,
      };
    }
  }

  /**
   * Clear all offline data (use with caution)
   */
  static async clearOfflineData() {
    try {
      await AsyncStorage.removeItem('offlineMaintenanceRequests');
      await AsyncStorage.removeItem('offlineMaintenanceUpdates');
      console.log('✅ Cleared offline maintenance requests and updates');
      return true;
    } catch (error) {
      console.error('❌ Error clearing offline data:', error);
      return false;
    }
  }

  // Save an offline update (approve/reject or any PUT)
  static async saveOfflineUpdate(update) {
    try {
      const raw = await AsyncStorage.getItem('offlineMaintenanceUpdates');
      const updates = raw ? JSON.parse(raw) : [];
      const filtered = updates.filter((u) => u.id !== update.id);
      filtered.push({ ...update, queuedAt: new Date().toISOString() });
      await AsyncStorage.setItem('offlineMaintenanceUpdates', JSON.stringify(filtered));
      console.log('✅ Queued offline maintenance update:', update.id, update.action);
      return true;
    } catch (error) {
      console.error('❌ Error saving offline maintenance update:', error);
      return false;
    }
  }

  // Get offline updates queue
  static async getOfflineUpdates() {
    try {
      const raw = await AsyncStorage.getItem('offlineMaintenanceUpdates');
      return raw ? JSON.parse(raw) : [];
    } catch (error) {
      console.error('❌ Error getting offline maintenance updates:', error);
      return [];
    }
  }

  // Sync offline updates with provided uploader (PUT)
  static async syncOfflineUpdates(uploadCallback) {
    try {
      const updates = await this.getOfflineUpdates();
      if (!updates.length) {
        console.log('✅ No offline maintenance updates to sync');
        return { success: true, synced: 0, failed: 0 };
      }

      console.log('🔄 Starting sync of', updates.length, 'offline maintenance updates...');
      let synced = 0;
      let failed = 0;
      const failedUpdates = [];

      for (const upd of updates) {
        try {
          const result = await uploadCallback(upd);
          if (result && result.success) {
            synced++;
            console.log('✅ Synced maintenance update:', upd.id, upd.action);
          } else {
            failed++;
            failedUpdates.push(upd);
            console.error('❌ Failed to sync maintenance update:', upd.id, upd.action);
          }
        } catch (err) {
          failed++;
          failedUpdates.push(upd);
          console.error('❌ Error syncing maintenance update:', upd.id, err.message);
        }
      }

      await AsyncStorage.setItem('offlineMaintenanceUpdates', JSON.stringify(failedUpdates));
      console.log('✅ Updates sync complete - Synced:', synced, 'Failed:', failed);
      return { success: true, synced, failed, total: updates.length };
    } catch (error) {
      console.error('❌ Error syncing offline maintenance updates:', error);
      return { success: false, synced: 0, failed: 0, error: error.message };
    }
  }
}

export default OfflineRequestMaintenanceHelper;

