import React, { useState, useRef } from 'react';
import {
  View,
  Text, 
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  Animated, 
  Modal,
  ScrollView,
  TextInput,
  Alert,
  Modal as RNModal,
  TouchableWithoutFeedback,
  FlatList,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;

const CustomInputField = ({ label, value, onChangeText, isRequired = false, showError = false, errorMessage = '', ...props }) => (
  <View style={inputStyles.container}>
    <Text style={inputStyles.label}>
      {label}
      {isRequired && (
        <Text style={inputStyles.requiredError}> *</Text>
      )}
    </Text>
    <TextInput
      style={[
        inputStyles.input,
        showError ? inputStyles.inputError : null,
      ]}
      value={value}
      onChangeText={onChangeText}
      {...props}
      placeholderTextColor="#888"
    />
    {showError && (
      <Text style={inputStyles.errorText}>{errorMessage}</Text>
    )}
  </View>
);

const UNIT_OPTIONS = ['Piece', 'Kg', 'Gram', 'Gallon'];

const AddPartInventory = () => {
  const navigation = useNavigation();
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  const [formData, setFormData] = useState({
    partNumber: '',
    partName: '',
    quantity: '',
    quantityUnit: 'Piece',
    costPerItem: '',
    manufacturer: '',
    location: '',
    itemDescription: '',
    itemType: '',
    supplier: '',
  });
  const [showErrors, setShowErrors] = useState(false);
  const [unitDropdownVisible, setUnitDropdownVisible] = useState(false);
  const requiredFields = ['partNumber', 'partName', 'quantity', 'costPerItem', 'manufacturer'];
  const handleChange = (field, value) => {
    setFormData({ ...formData, [field]: value });
  };
  const handleUnitSelect = (unit) => {
    setFormData({ ...formData, quantityUnit: unit });
    setUnitDropdownVisible(false);
  };
  const validateForm = () => {
    for (let field of requiredFields) {
      if (!formData[field]) {
        return false;
      }
    }
    return true;
  };
  const handleCreate = async () => {
    setShowErrors(true);
    if (!validateForm()) {
      return;
    }
    try {
      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
      const newPart = {
        id: `SKU${Math.floor(100000 + Math.random() * 900000)}`,
        partNumber: formData.partNumber,
        partName: formData.partName,
        quantity: formData.quantity,
        quantityUnit: formData.quantityUnit,
        costPerItem: formData.costPerItem,
        manufacturer: formData.manufacturer,
        location: formData.location,
        itemDescription: formData.itemDescription,
        itemType: formData.itemType,
        supplier: formData.supplier,
      };
      const existingParts = await AsyncStorage.getItem('partsInventory');
      const parts = existingParts ? JSON.parse(existingParts) : [];
      parts.push(newPart);
      await AsyncStorage.setItem('partsInventory', JSON.stringify(parts));
      setTimeout(() => {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        navigation.navigate('PartInventoryHome');
      }, 2000);
    } catch (error) {
      console.error('Error saving part:', error);
      Alert.alert('Error', 'Failed to save part. Please try again.');
      setSyncModalVisible(false);
    }
  };
  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Parts Entry</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={handleCreate}>
              <Text style={styles.createButtonText}>Create</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>
      <ScrollView style={styles.mainContent} keyboardShouldPersistTaps="handled">
        <CustomInputField
          label="Part Number"
          value={formData.partNumber}
          onChangeText={(text) => handleChange('partNumber', text)}
          isRequired
          showError={showErrors && !formData.partNumber}
          errorMessage="Part Number is required"
        />
        <CustomInputField
          label="Part Name"
          value={formData.partName}
          onChangeText={(text) => handleChange('partName', text)}
          isRequired
          showError={showErrors && !formData.partName}
          errorMessage="Part Name is required"
        />
        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Quantity
            <Text style={inputStyles.requiredError}> *</Text>
          </Text>
          <View style={quantityStyles.row}>
            <TextInput
              style={[
                inputStyles.input,
                quantityStyles.input,
                showErrors && !formData.quantity ? inputStyles.inputError : null,
              ]}
              value={formData.quantity}
              onChangeText={(text) => handleChange('quantity', text)}
              keyboardType="numeric"
              //placeholder="Last Service Meter Reading"
              placeholderTextColor="#888"
            />
            <TouchableOpacity
              style={quantityStyles.unitBox}
              onPress={() => setUnitDropdownVisible(true)}
              activeOpacity={0.7}
            >
              <Text style={quantityStyles.unitText}>{formData.quantityUnit}</Text>
              <MaterialIcons name="arrow-drop-down" size={22} color="#015185" />
            </TouchableOpacity>
          </View>
          {showErrors && !formData.quantity && (
            <Text style={inputStyles.errorText}>Quantity is required</Text>
          )}
        </View>
        <CustomInputField
          label="Cost Per Item"
          value={formData.costPerItem}
          onChangeText={(text) => handleChange('costPerItem', text)}
          isRequired
          keyboardType="numeric"
          showError={showErrors && !formData.costPerItem}
          errorMessage="Cost Per Item is required"
        />
        <CustomInputField
          label="Manufacturer"
          value={formData.manufacturer}
          onChangeText={(text) => handleChange('manufacturer', text)}
          isRequired
          showError={showErrors && !formData.manufacturer}
          errorMessage="Manufacturer is required"
        />
        <CustomInputField
          label="Location"
          value={formData.location}
          onChangeText={(text) => handleChange('location', text)}
        />
        <CustomInputField
          label="Item Description"
          value={formData.itemDescription}
          onChangeText={(text) => handleChange('itemDescription', text)}
        />
        <CustomInputField
          label="Item Type"
          value={formData.itemType}
          onChangeText={(text) => handleChange('itemType', text)}
        />
        <CustomInputField
          label="Supplier"
          value={formData.supplier}
          onChangeText={(text) => handleChange('supplier', text)}
        />
      </ScrollView>
      <RNModal
        visible={unitDropdownVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setUnitDropdownVisible(false)}
      >
        <TouchableWithoutFeedback onPress={() => setUnitDropdownVisible(false)}>
          <View style={quantityStyles.dropdownOverlay} />
        </TouchableWithoutFeedback>
        <View style={quantityStyles.dropdownContainer}>
          <FlatList
            data={UNIT_OPTIONS}
            keyExtractor={(item) => item}
            renderItem={({ item }) => (
              <TouchableOpacity
                style={quantityStyles.dropdownItem}
                onPress={() => handleUnitSelect(item)}
              >
                <Text style={quantityStyles.dropdownText}>{item}</Text>
              </TouchableOpacity>
            )}
          />
        </View>
      </RNModal>
      <View style={{ height: 8, width: '100%', backgroundColor: 'transparent' }}>
        <LinearGradient
          colors={['#fff', '#025383']}
          style={{
            height: 8,
            width: '80%',
            alignSelf: 'center',
            borderRadius: 4,
            marginBottom: 8,
          }}
          start={{ x: 0, y: 0 }}
          end={{ x: 1, y: 0 }}
        />
      </View>
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  required: {
    color: '#015185',
    fontWeight: 'bold',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
});

const quantityStyles = StyleSheet.create({
  row: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  input: {
    flex: 1,
    marginRight: 8,
  },
  unitBox: {
    flexDirection: 'row',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    backgroundColor: '#fff',
    minWidth: 70,
    justifyContent: 'space-between',
  },
  unitText: {
    color: '#015185',
    fontSize: width * 0.038,
    fontWeight: '500',
    marginRight: 2,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.2)',
  },
  dropdownContainer: {
    position: 'absolute',
    top: width * 0.5,
    right: width * 0.12,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
    width: 120,
    zIndex: 1000,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
  },
  dropdownItem: {
    paddingVertical: 12,
    paddingHorizontal: 16,
  },
  dropdownText: {
    color: '#015185',
    fontSize: width * 0.038,
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.019,
    marginLeft: width * 0.025,
    backgroundColor: '#fff',
    borderRadius: width * 0.01,
  },
  createButtonText: {
    color: '#015185',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default AddPartInventory;