import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  TextInput,
  ScrollView,
  Alert,
  Animated,
  Modal,
  Platform,
  KeyboardAvoidingView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import { Dropdown } from 'react-native-element-dropdown';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

// MongoDB API URL
const MONGODB_LOGBOOK_API_URL = 'https://api.titandrillingzm.com:6012';

const LogBookEntry = () => {
  const navigation = useNavigation();
  const route = useRoute();

  const [entryNumber, setEntryNumber] = useState(Math.floor(Math.random() * 1000000000).toString());
  
  // Country and Project states
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [selectedProject, setSelectedProject] = useState(null);
  const [countryItems, setCountryItems] = useState([]);
  const [projectItems, setProjectItems] = useState([]);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    logBookPermissions: [],
  });

  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [equipmentId, setEquipmentId] = useState(null);
  const [currentMeterReading, setCurrentMeterReading] = useState(null);
  const [equipmentStatus, setEquipmentStatus] = useState(null);
  const [startReading, setStartReading] = useState('');
  const [endReading, setEndReading] = useState('');
  const [startDateTime, setStartDateTime] = useState(null);
  const [endDateTime, setEndDateTime] = useState(null);
  const [showStartPicker, setShowStartPicker] = useState(false);
  const [showEndPicker, setShowEndPicker] = useState(false);
  const [pickerMode, setPickerMode] = useState('date');
  const [fuelOpening, setFuelOpening] = useState('');
  const [fuelConsumption, setFuelConsumption] = useState('');
  const [fuelUnit, setFuelUnit] = useState('KG');
  const [itemsFuelUnit] = useState([
    { label: 'KG', value: 'KG' },
    { label: 'Piece', value: 'Piece' },
    { label: 'Gram', value: 'Gram' },
    { label: 'Gallon', value: 'Gallon' },
    { label: 'Quart', value: 'Quart' },
    { label: 'Liter', value: 'Liter' },
    { label: 'ML', value: 'ML' },
    { label: 'Meter', value: 'Meter' },
  ]);
  const [operatorName, setOperatorName] = useState('');
  const [workLocation, setWorkLocation] = useState('');
  const [workType, setWorkType] = useState('');
  const [workDescription, setWorkDescription] = useState('');
  const [materialDescription, setMaterialDescription] = useState('');
  const [materialName, setMaterialName] = useState('');
  const [materialQuantity, setMaterialQuantity] = useState('');
  const [materialUnit, setMaterialUnit] = useState('Ton');
  const [showMaterialFields, setShowMaterialFields] = useState(false);
  const [itemsMaterialUnit] = useState([
    { label: 'Ton', value: 'Ton' },
    { label: 'KG', value: 'KG' },
    { label: 'Piece', value: 'Piece' },
    { label: 'Gram', value: 'Gram' },
    { label: 'Gallon', value: 'Gallon' },
    { label: 'Quart', value: 'Quart' },
    { label: 'Liter', value: 'Liter' },
    { label: 'ML', value: 'ML' },
    { label: 'Meter', value: 'Meter' },
  ]);
  const [issueReport, setIssueReport] = useState('');
  const [issueName, setIssueName] = useState('');
  const [issueDetails, setIssueDetails] = useState('');
  const [showIssueFields, setShowIssueFields] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const [tempDate, setTempDate] = useState(null);
  const [showCustomPicker, setShowCustomPicker] = useState(false);
  const [pickerType, setPickerType] = useState('start'); // 'start' or 'end'
  const [pickerStep, setPickerStep] = useState('date'); // 'date' or 'time'

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const equipmentStatuses = ['Available', 'Deployed', 'In Maintenance', 'Breakdown'];

  // Generate custom entry number based on country and project
  const generateEntryNumber = async (country, project) => {
    if (!country || !project) {
      return 'LBE-###';
    }

    try {
      const response = await fetch(`${MONGODB_LOGBOOK_API_URL}/logbook/next-entry-number?country=${encodeURIComponent(country)}&project=${encodeURIComponent(project)}`);
      const result = await response.json();
      
      if (result.success && result.entryNumber) {
        return result.entryNumber;
      }
      
      // Fallback
      return 'LBE-###';
    } catch (error) {
      console.error('Error generating entry number:', error);
      return 'LBE-###';
    }
  };

  // Update entry number when country or project changes
  const updateEntryNumber = async () => {
    if (selectedCountry && selectedProject) {
      const newEntryNumber = await generateEntryNumber(selectedCountry, selectedProject);
      setEntryNumber(newEntryNumber);
    }
  };

  // Fetch user data and set up country/project dropdowns
  const fetchUserData = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      console.log('Fetching user data for uid:', userUid);
      
      // Use MongoAPI to get user data
      const response = await MongoAPI.getUserById(userUid);
      console.log('User response:', response);
      
      if (!response.success || !response.data) {
        console.log('User not found in database');
        showToast('error', 'Error', 'User data not found.');
        return;
      }

      const userDataFromDB = response.data;
      console.log('User data received:', {
        name: userDataFromDB.name,
        email: userDataFromDB.email,
        countries: userDataFromDB.countries,
        projects: userDataFromDB.projects,
        modules: userDataFromDB.modules
      });

      // Normalize modules to array and extract Logbook permissions
      const modules = Array.isArray(userDataFromDB.modules) ? userDataFromDB.modules : 
                      (userDataFromDB.modules ? Object.values(userDataFromDB.modules) : []);
      
      // Find Logbook module
      const logBookModule = modules.find(module => 
        module.module && (
          module.module === 'logbook' ||
          module.module === 'Logbook' ||
          module.module.toLowerCase() === 'logbook'
        )
      );
      
      console.log('Logbook module found:', logBookModule);
      console.log('All available modules:', modules.map(m => ({ module: m.module, permissions: m.permissions })));
      
      const permissions = logBookModule ? (Array.isArray(logBookModule.permissions) ? logBookModule.permissions : []) : [];
      
      const userDataObj = {
        countries: userDataFromDB.countries || [],
        projects: userDataFromDB.projects || [],
        logBookPermissions: permissions,
      };
      
      setUserData(userDataObj);
      console.log('Logbook Permissions:', permissions);
      
      // Set up country dropdown items
      const countryDropdownItems = userDataObj.countries.map(country => ({
        label: country,
        value: country,
      }));
      setCountryItems(countryDropdownItems);
      
      // Set up project dropdown items
      const projectDropdownItems = userDataObj.projects.map(project => ({
        label: project,
        value: project,
      }));
      setProjectItems(projectDropdownItems);
      
      // Set default values if editing an existing entry
      if (route.params?.entry) {
        const entry = route.params.entry;
        if (entry.country && userDataObj.countries.includes(entry.country)) {
          setSelectedCountry(entry.country);
        }
        if (entry.project && userDataObj.projects.includes(entry.project)) {
          setSelectedProject(entry.project);
        }
      }
    } catch (error) {
      console.error('Error fetching user data:', error);
      showToast('error', 'Error', 'Failed to load user data.');
    }
  };

  // Update project items based on country selection
  const updateProjectItems = (country) => {
    if (country === 'Zambia') {
      const zambiaProjects = userData.projects.filter(project => 
        ['Kobold', 'Kalumbila', 'Kansanshi', 'Kimiteto', 'Mimosa'].includes(project)
      );
      setProjectItems(zambiaProjects.map(project => ({ label: project, value: project })));
    } else if (country === 'DRC') {
      const drcProjects = userData.projects.filter(project => 
        ['Musompo', 'IME'].includes(project)
      );
      setProjectItems(drcProjects.map(project => ({ label: project, value: project })));
    } else {
      // If no country selected or "Both Countries", show all projects
      setProjectItems(userData.projects.map(project => ({ label: project, value: project })));
    }
    
    // Reset project selection when country changes
    setSelectedProject(null);
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected);
    });

    checkNetworkStatus().then((connected) => {
      setIsOnline(connected);
    });

    if (route.params?.entry) {
      const entry = route.params.entry;
      setEntryNumber(entry.entryNumber || Math.floor(Math.random() * 1000000000).toString());
      setSelectedCountry(entry.country || null);
      setSelectedProject(entry.project || null);
      setSelectedEquipment(entry.selectedEquipment || null);
      setEquipmentId(entry.equipmentId || null);
      setCurrentMeterReading(entry.currentMeterReading || null);
      setEquipmentStatus(entry.equipmentStatus || null);
      setStartReading(entry.startReading || '');
      setEndReading(entry.endReading || '');
      setStartDateTime(entry.startDateTime ? new Date(entry.startDateTime) : null);
      setEndDateTime(entry.endDateTime ? new Date(entry.endDateTime) : null);
      setFuelOpening(entry.fuelOpening || '');
      setFuelConsumption(entry.fuelConsumption || '');
      setFuelUnit(entry.fuelUnit || 'KG');
      setOperatorName(entry.operatorName || '');
      setWorkLocation(entry.workLocation || '');
      setWorkType(entry.workType || '');
      setWorkDescription(entry.workDescription || '');
      setMaterialDescription(entry.materialDescription || '');
      setMaterialName(entry.materialName || '');
      setMaterialQuantity(entry.materialQuantity || '');
      setMaterialUnit(entry.materialUnit || 'Ton');
      setIssueReport(entry.issueReport || '');
      setIssueName(entry.issueName || '');
      setIssueDetails(entry.issueDetails || '');
      setShowMaterialFields(!!entry.materialName || !!entry.materialQuantity);
      setShowIssueFields(!!entry.issueReport || !!entry.issueName || !!entry.issueDetails);
    } else if (route.params?.selectedEquipment) {
      const { equipmentName, meterReading, id } = route.params.selectedEquipment;
      setSelectedEquipment(equipmentName || null);
      setEquipmentId(id || null);
      setCurrentMeterReading(meterReading || null);
    }

    return () => unsubscribe();
  }, [route.params]);

  // Fetch user data after component mounts
  useEffect(() => {
    fetchUserData();
  }, []);

  // Update project items when country changes
  useEffect(() => {
    if (selectedCountry) {
      updateProjectItems(selectedCountry);
    }
  }, [selectedCountry, userData.projects]);

  // Update entry number when both country and project are selected
  useEffect(() => {
    if (selectedCountry && selectedProject) {
      updateEntryNumber().catch(error => {
        console.error('Error updating entry number:', error);
      });
    }
  }, [selectedCountry, selectedProject]);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showToast = (type, text1, text2) => {
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const queueOperation = async (operation) => {
    try {
      const pendingOperationsRaw = await AsyncStorage.getItem('pendingLogBookOperations');
      const pendingOperations = pendingOperationsRaw ? JSON.parse(pendingOperationsRaw) : [];
      pendingOperations.push(operation);
      await AsyncStorage.setItem('pendingLogBookOperations', JSON.stringify(pendingOperations));
      showToast('info', 'Offline Mode', 'Operation queued for sync when online.');
    } catch (error) {
      console.error('Error queuing operation:', error);
      showToast('error', 'Error', 'Failed to queue operation.');
    }
  };

  const saveLogBookEntry = async () => {
    // Validate required fields
    if (!selectedCountry) {
      Alert.alert('Validation Error', 'Country selection is required.');
      return;
    }
    if (!selectedProject) {
      Alert.alert('Validation Error', 'Project selection is required.');
      return;
    }
    if (!selectedEquipment) {
      Alert.alert('Validation Error', 'Equipment selection is required.');
      return;
    }
    if (!equipmentStatus) {
      Alert.alert('Validation Error', 'Equipment Status is required.');
      return;
    }
    if (!startReading) {
      Alert.alert('Validation Error', 'Start Reading is required.');
      return;
    }
    if (!endReading) {
      Alert.alert('Validation Error', 'End Reading is required.');
      return;
    }
    if (!startDateTime) {
      Alert.alert('Validation Error', 'Start Date and Time is required.');
      return;
    }
    if (!endDateTime) {
      Alert.alert('Validation Error', 'End Date and Time is required.');
      return;
    }
    if (endDateTime <= startDateTime) {
      Alert.alert('Validation Error', 'End Date and Time must be after Start Date and Time.');
      return;
    }

    setSyncModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setSyncModalVisible(false);
        showToast('error', 'Authentication Error', 'No user is logged in.');
        navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'LogbookList' },
          ],
        });
        return;
      }

      const logBookEntry = {
        id: route.params?.entry?.id || Date.now().toString(),
        entryNumber,
        country: selectedCountry,
        project: selectedProject,
        selectedEquipment,
        equipmentId,
        currentMeterReading,
        equipmentStatus,
        startReading,
        endReading,
        startDateTime: startDateTime.toISOString(),
        endDateTime: endDateTime.toISOString(),
        fuelOpening,
        fuelConsumption,
        fuelUnit,
        operatorName,
        workLocation,
        workType,
        workDescription,
        materialDescription,
        materialName,
        materialQuantity,
        materialUnit,
        issueReport,
        issueName,
        issueDetails,
        adminUid: userUid,
        createdAt: route.params?.entry?.createdAt || new Date().toISOString(),
        updatedAt: new Date().toISOString(),
      };

      console.log('📝 Saving logbook entry to MongoDB...', {
        entryNumber: logBookEntry.entryNumber,
        country: logBookEntry.country,
        project: logBookEntry.project
      });

      // Save to MongoDB
      const method = route.params?.entry?.id ? 'PUT' : 'POST';
      const url = route.params?.entry?.id 
        ? `${MONGODB_LOGBOOK_API_URL}/logbook/${logBookEntry.id}`
        : `${MONGODB_LOGBOOK_API_URL}/logbook`;

      const response = await fetch(url, {
        method,
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(logBookEntry),
      });

      if (!response.ok) {
        throw new Error(`Failed to save logbook entry: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.success) {
        console.log('✅ Logbook entry saved successfully:', result.data.entryNumber);
        
        // Store in AsyncStorage for offline access
        await AsyncStorage.setItem(`logbook_${logBookEntry.id}`, JSON.stringify(logBookEntry));
        
        showToast('success', 'Success', `Logbook entry ${result.data.entryNumber} saved successfully!`);
        
        setTimeout(() => {
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          navigation.reset({
            index: 1,
            routes: [
              { name: 'MainApp' },
              { name: 'LogbookList' },
            ],
          });
        }, 2000);
      } else {
        throw new Error(result.message || 'Failed to save logbook entry');
      }
    } catch (error) {
      console.error('Error saving logbook entry to MongoDB:', error);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      let errorMessage = 'Failed to save logbook entry.';
      if (error.message.includes('Failed to fetch')) {
        errorMessage = 'Network error. Please check your connection.';
      }
      showToast('error', 'Error', errorMessage);
    }
  };

  const handleStatusSelect = (status) => {
    setEquipmentStatus(status);
  };

  const handleEquipmentSelected = useCallback((equipmentData) => {
    console.log('Equipment selected in LogBookEntry:', equipmentData);
    
    // Extract equipment information
    const equipmentName = equipmentData.equipmentName || 
                          equipmentData.rigName || 
                          equipmentData.vehicleNumber || 
                          'Unknown Equipment';
    const equipmentNumber = equipmentData.equipmentNumber || 
                           equipmentData.id || 
                           equipmentData.equipmentNo || 
                           'N/A';
    const meterReading = equipmentData.meterReading || 
                        equipmentData.currentMeterReading || 
                        0;
    
    console.log('Setting equipment:', {
      name: equipmentName,
      id: equipmentData.id,
      meterReading: meterReading
    });
    
    setSelectedEquipment(equipmentName);
    setEquipmentId(equipmentData.id || null);
    setCurrentMeterReading(meterReading);
  }, []);

  const handleSelectEquipment = () => {
    console.log('Navigating to Logbook_Equipment with callback');
    navigation.navigate('Logbook_Equipment', {
      source: 'LogBookEntry',
      onSelect: handleEquipmentSelected,
    });
  };

  const handleStartPickerChange = (event, selectedDate) => {
    setShowStartPicker(Platform.OS === 'ios');
    if (selectedDate) {
      if (pickerMode === 'date') {
        setStartDateTime(selectedDate);
        setPickerMode('time');
        setShowStartPicker(true);
      } else {
        setStartDateTime(selectedDate);
        setPickerMode('date');
      }
    }
  };

  const handleEndPickerChange = (event, selectedDate) => {
    setShowEndPicker(Platform.OS === 'ios');
    if (selectedDate) {
      if (pickerMode === 'date') {
        setEndDateTime(selectedDate);
        setPickerMode('time');
        setShowEndPicker(true);
      } else {
        setEndDateTime(selectedDate);
        setPickerMode('date');
      }
    }
  };

  const showStartDateTimePicker = () => {
    setPickerMode('date');
    setShowStartPicker(true);
  };

  const showEndDateTimePicker = () => {
    setPickerMode('date');
    setShowEndPicker(true);
  };

  const showDateTimePicker = (type) => {
    setPickerType(type); // 'start' or 'end'
    setPickerStep('date');
    setTempDate(type === 'start' ? startDateTime || new Date() : endDateTime || new Date());
    setShowCustomPicker(true);
  };

  const formatDateTime = (date) => {
    if (!date) return '';
    return date.toLocaleString('en-US', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
      hour: '2-digit',
      minute: '2-digit',
      hour12: true,
    });
  };

  return (
    <KeyboardAvoidingView
      style={styles.container}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
    >
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity style={{padding: width * 0.012}} onPress={() => navigation.navigate('LogbookList')}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>New Logbook Entry</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.saveButton} onPress={saveLogBookEntry}>
              <Text style={styles.saveButtonText}>Create</Text>
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <ScrollView contentContainerStyle={styles.content}>
        <View style={styles.entryRow}>
          <Text style={styles.entryLabel}>Entry #</Text>
          <TextInput
            style={styles.entryNumberInput}
            value={entryNumber}
            onChangeText={setEntryNumber}
            editable={false}
          />
        </View>

        <View style={styles.countrySelectContainer}>
          <Text style={styles.countryText}>
            Country <Text style={styles.required}>*</Text>
          </Text>
                                               <Dropdown
               data={countryItems}
               labelField="label"
               valueField="value"
               value={selectedCountry}
               placeholder="Select Country"
               placeholderStyle={styles.dropdownPlaceholder}
               selectedTextStyle={styles.dropdownSelectedText}
               onChange={item => {
                 setSelectedCountry(item.value);
                 updateProjectItems(item.value);
                 // Update entry number when country changes
                 setTimeout(() => {
                   updateEntryNumber().catch(error => {
                     console.error('Error updating entry number:', error);
                   });
                 }, 100);
               }}
               maxHeight={300}
               style={styles.dropdown}
               containerStyle={styles.dropdownContainer}
               activeColor="#025383"
               itemContainerStyle={styles.dropdownItemContainer}
               renderItem={(item, selected) => (
                 <View style={[
                   styles.dropdownItemContainer,
                   selected && styles.dropdownItemContainerSelected
                 ]}>
                   <Text style={[
                     styles.dropdownItemText,
                     selected && { color: '#fff', fontWeight: '600' }
                   ]}>
                     {item.label}
                   </Text>
                 </View>
               )}
               renderRightIcon={() => (
                 <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
               )}
               showsVerticalScrollIndicator={false}
             />
        </View>
        <View style={styles.projectSelectContainer}>
          <Text style={styles.projectText}>
            Project <Text style={styles.required}>*</Text>
          </Text>
                                    <Dropdown
               data={projectItems}
               labelField="label"
               valueField="value"
               value={selectedProject}
               placeholder="Select Project"
               placeholderStyle={styles.dropdownPlaceholder}
               selectedTextStyle={styles.dropdownSelectedText}
               onChange={item => {
                 setSelectedProject(item.value);
                 // Update entry number when project changes
                 setTimeout(() => {
                   updateEntryNumber().catch(error => {
                     console.error('Error updating entry number:', error);
                   });
                 }, 100);
               }}
               maxHeight={300}
               style={styles.dropdown}
               containerStyle={styles.dropdownContainer}
               activeColor="#025383"
               itemContainerStyle={styles.dropdownItemContainer}
               renderItem={(item, selected) => (
                 <View style={[
                   styles.dropdownItemContainer,
                   selected && styles.dropdownItemContainerSelected
                 ]}>
                   <Text style={[
                     styles.dropdownItemText,
                     selected && { color: '#fff', fontWeight: '600' }
                   ]}>
                     {item.label}
                   </Text>
                 </View>
               )}
               renderRightIcon={() => (
                 <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
               )}
               showsVerticalScrollIndicator={false}
             />
        </View>

        <View style={styles.equipmentSelectContainer}>
          <Text style={styles.equipmentText}>
            Equipment <Text style={styles.required}>*</Text>
          </Text>
          <TouchableOpacity onPress={handleSelectEquipment} style={styles.selectContainer}>
            <Text style={styles.selectText}>{selectedEquipment || 'Select'}</Text>
            <MaterialIcons name="keyboard-arrow-down" size={width * 0.05} color="#015185" />
          </TouchableOpacity>
        </View>
        {selectedEquipment && (
          <>
            <Text style={styles.equipmentNameText}>Selected: {selectedEquipment}</Text>
            <Text style={styles.meterReadingText}>
              Current Meter Reading: {currentMeterReading ? `${currentMeterReading}` : 'N/A'}
            </Text>
          </>
        )}
        <Text style={styles.label}>
          Equipment Status <Text style={styles.required}>*</Text>
        </Text>
        <View style={[styles.statusContainer, equipmentStatus && styles.statusContainerSelected]}>
          {equipmentStatuses.map((status) => (
            <TouchableOpacity
              key={status}
              style={[
                styles.statusButton,
                equipmentStatus === status && styles.statusButtonSelected,
              ]}
              onPress={() => handleStatusSelect(status)}
            >
              <Text
                style={[
                  styles.statusButtonText,
                  equipmentStatus === status && styles.statusButtonTextSelected,
                ]}
              >
                {status}
              </Text>
            </TouchableOpacity>
          ))}
        </View>
        <View style={styles.row}>
          <View style={styles.column}>
            <Text style={styles.label}>
              Start Reading <Text style={styles.required}>*</Text>
            </Text>
            <TextInput
              style={[styles.input, styles.halfInput]}
              value={startReading}
              onChangeText={setStartReading}
              keyboardType="numeric"
              placeholder="Miles"
            />
          </View>
          <View style={styles.column}>
            <Text style={styles.label}>
              End Reading <Text style={styles.required}>*</Text>
            </Text>
            <TextInput
              style={[styles.input, styles.halfInput]}
              value={endReading}
              onChangeText={setEndReading}
              keyboardType="numeric"
              placeholder="Miles"
            />
          </View>
        </View>
        <View style={styles.row}>
          <View style={styles.column}>
            <Text style={styles.label}>
              Start Date & Time <Text style={styles.required}>*</Text>
            </Text>
            <TouchableOpacity
              style={[styles.input, styles.halfInput]}
              onPress={() => Platform.OS === 'ios' ? showDateTimePicker('start') : showStartDateTimePicker()}
            >
              <Text style={styles.dateTimeText}>
                {startDateTime ? formatDateTime(startDateTime) : 'Select Date & Time'}
              </Text>
            </TouchableOpacity>
            {Platform.OS === 'android' && showStartPicker && (
              <DateTimePicker
                value={startDateTime || new Date()}
                mode={pickerMode}
                display="default"
                onChange={handleStartPickerChange}
                minimumDate={new Date()}
              />
            )}
          </View>
          <View style={styles.column}>
            <Text style={styles.label}>
              End Date & Time <Text style={styles.required}>*</Text>
            </Text>
            <TouchableOpacity
              style={[styles.input, styles.halfInput]}
              onPress={() => Platform.OS === 'ios' ? showDateTimePicker('end') : showEndDateTimePicker()}
            >
              <Text style={styles.dateTimeText}>
                {endDateTime ? formatDateTime(endDateTime) : 'Select Date & Time'}
              </Text>
            </TouchableOpacity>
            {Platform.OS === 'android' && showEndPicker && (
              <DateTimePicker
                value={endDateTime || (startDateTime || new Date())}
                mode={pickerMode}
                display="default"
                onChange={handleEndPickerChange}
                minimumDate={startDateTime || new Date()}
              />
            )}
          </View>
        </View>
        <View style={styles.row}>
          <View style={styles.column}>
            <Text style={styles.label}>Fuel Opening</Text>
            <TextInput
              style={[styles.input, styles.halfInput]}
              value={fuelOpening}
              onChangeText={setFuelOpening}
              keyboardType="numeric"
            />
          </View>
          <View style={styles.column}>
            <Text style={styles.label}>Fuel Closing</Text>
            <TextInput
              style={[styles.input, styles.halfInput]}
              value={fuelConsumption}
              onChangeText={setFuelConsumption}
              keyboardType="numeric"
            />
          </View>
        </View>
        <TextInput
          style={styles.inputNoLabel}
          value={operatorName}
          onChangeText={setOperatorName}
          placeholder="Operator Name"
        />
        <TextInput
          style={styles.inputNoLabel}
          value={workLocation}
          onChangeText={setWorkLocation}
          placeholder="Work Location"
        />
        <TextInput
          style={styles.inputNoLabel}
          value={workType}
          onChangeText={setWorkType}
          placeholder="Work Type"
        />
        <TextInput
          style={[styles.inputNoLabel, styles.noteInput]}
          value={workDescription}
          onChangeText={setWorkDescription}
          placeholder="Work Description"
        />
        <View style={styles.materialDescriptionContainer}>
          <Text style={styles.materialDescriptionInput}>Material Description</Text>
          <TouchableOpacity onPress={() => setShowMaterialFields(!showMaterialFields)}>
            <MaterialIcons
              name={showMaterialFields ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
              size={width * 0.05}
              color="#015185"
            />
          </TouchableOpacity>
        </View>
        {showMaterialFields && (
          <>
            <TextInput
              style={styles.inputNoBorder}
              value={materialName}
              onChangeText={setMaterialName}
              placeholder="Material Name"
            />
            <View style={styles.row}>
              <View style={styles.quantityColumn}>
                <TextInput
                  style={styles.inputNoBorder}
                  value={materialQuantity}
                  onChangeText={setMaterialQuantity}
                  placeholder="Material Quantity"
                  keyboardType="numeric"
                />
              </View>
              <View style={styles.unitColumn}>
                                                  <Dropdown
                   data={itemsMaterialUnit}
                   labelField="label"
                   valueField="value"
                   value={materialUnit}
                   placeholder="Ton"
                   placeholderStyle={styles.dropdownPlaceholder}
                   selectedTextStyle={styles.dropdownSelectedText}
                   inputSearchStyle={styles.dropdownInputSearch}
                   onChange={item => {
                     setMaterialUnit(item.value);
                   }}
                   search
                   maxHeight={300}
                   style={styles.dropdown}
                   containerStyle={styles.dropdownContainer}
                   activeColor="#025383"
                   itemContainerStyle={styles.dropdownItemContainer}
                   searchPlaceholder="Search unit..."
                   searchInputStyle={styles.dropdownSearchInput}
                   searchQuery={(search) => {
                     const query = search.toLowerCase();
                     return itemsMaterialUnit.filter(item => {
                       const itemLabel = item.label.toLowerCase();
                       return itemLabel.includes(query);
                     });
                   }}
                   renderItem={(item, selected) => (
                     <View style={[
                       styles.dropdownItemContainer,
                       selected && styles.dropdownItemContainerSelected
                     ]}>
                       <Text style={[
                         styles.dropdownItemText,
                         selected && { color: '#fff', fontWeight: '600' }
                       ]}>
                         {item.label}
                       </Text>
                     </View>
                   )}
                   renderRightIcon={() => (
                     <MaterialIcons name="keyboard-arrow-down" size={20} color="#015185" />
                   )}
                   showsVerticalScrollIndicator={false}
                 />
              </View>
            </View>
          </>
        )}
        <View style={styles.issueReportContainer}>
          <Text style={styles.issueReportInput}>Issue Report</Text>
          <TouchableOpacity onPress={() => setShowIssueFields(!showIssueFields)}>
            <MaterialIcons
              name={showIssueFields ? 'keyboard-arrow-up' : 'keyboard-arrow-down'}
              size={width * 0.05}
              color="#015185"
            />
          </TouchableOpacity>
        </View>
        {showIssueFields && (
          <>
            <TextInput
              style={styles.inputNoBorder}
              value={issueReport}
              onChangeText={setIssueReport}
              placeholder="Issue Report"
            />
            <TextInput
              style={styles.inputNoBorder}
              value={issueName}
              onChangeText={setIssueName}
              placeholder="Issue Name"
            />
            <TextInput
              style={[styles.inputNoBorder, styles.noteInput]}
              value={issueDetails}
              onChangeText={setIssueDetails}
              placeholder="Issue Details"
              multiline
            />
          </>
        )}
      </ScrollView>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.loadingModalContainer}>
          <View style={styles.loadingModalContent}>
            <Text style={styles.loadingModalText}>Please Wait</Text>
            <View style={styles.loadingIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.loadingIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.loadingModalTextadd}>Saving Logbook Entry</Text>
            </View>
          </View>
        </View>
      </Modal>
      {Platform.OS === 'ios' && showCustomPicker && (
        <Modal visible={showCustomPicker} transparent animationType="fade">
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center', backgroundColor: 'rgba(0,0,0,0.4)' }}>
            <View style={{ backgroundColor: '#fff', borderRadius: 12, padding: 20, width: '80%', alignItems: 'center' }}>
              <Text style={{ fontSize: 18, fontWeight: 'bold', marginBottom: 10 }}>
                {pickerStep === 'date' ? 'Select Date' : 'Select Time'}
              </Text>
              <DateTimePicker
                value={tempDate || new Date()}
                mode={pickerStep}
                display="spinner"
                onChange={(event, selectedDate) => {
                  if (selectedDate) setTempDate(selectedDate);
                }}
                style={{ width: '100%' }}
              />
              <View style={{ flexDirection: 'row', marginTop: 20 }}>
                {pickerStep === 'date' ? (
                  <TouchableOpacity
                    style={{ marginHorizontal: 10, padding: 10, backgroundColor: '#025383', borderRadius: 6 }}
                    onPress={() => setPickerStep('time')}
                  >
                    <Text style={{ color: '#fff', fontWeight: 'bold' }}>Next: Time</Text>
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity
                    style={{ marginHorizontal: 10, padding: 10, backgroundColor: '#025383', borderRadius: 6 }}
                    onPress={() => {
                      setShowCustomPicker(false);
                      if (pickerType === 'start') setStartDateTime(tempDate);
                      else setEndDateTime(tempDate);
                    }}
                  >
                    <Text style={{ color: '#fff', fontWeight: 'bold' }}>Confirm</Text>
                  </TouchableOpacity>
                )}
                <TouchableOpacity
                  style={{ marginHorizontal: 10, padding: 10, backgroundColor: '#888', borderRadius: 6 }}
                  onPress={() => setShowCustomPicker(false)}
                >
                  <Text style={{ color: '#fff' }}>Cancel</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </Modal>
      )}
      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: '600',
    marginLeft: width * 0.025,
  },
  saveButton: {
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    backgroundColor: '#fff',
    borderRadius: width * 0.015,
  },
  saveButtonText: {
    color: '#025383',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  content: {
    padding: width * 0.04,
    paddingBottom: width * 0.2,
  },
  entryRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: '#F1F1F1',
    padding: width * 0.025,
    borderRadius: width * 0.015,
    marginBottom: width * 0.02,
  },
  entryLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  entryNumberInput: {
    fontSize: width * 0.034,
    color: '#333',
    textAlign: 'right',
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  required: {
    color: '#FF0000',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputNoLabel: {
    padding: width * 0.025,
    fontSize: width * 0.04,
    marginBottom: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#ccc',
  },
  inputNoBorder: {
    padding: width * 0.025,
    fontSize: width * 0.04,
    marginBottom: width * 0.02,
    height: width * 0.12,
    borderBottomWidth: 1,
    borderBottomColor: '#ccc',
  },
  materialDescriptionContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ccc',
    marginBottom: width * 0.02,
  },
  materialDescriptionInput: {
    flex: 1,
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  issueReportContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ccc',
    marginBottom: width * 0.02,
  },
  issueReportInput: {
    flex: 1,
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  halfInput: {
    width: '100%',
  },
  noteInput: {
    textAlignVertical: 'top',
  },
  dropdown: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    marginBottom: width * 0.02,
    height: width * 0.12,
    backgroundColor: '#fff',
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    backgroundColor: '#fff',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  dropdownPlaceholder: {
    fontSize: width * 0.04,
    color: '#888',
    fontWeight: '400',
  },
  dropdownSelectedText: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '600',
  },
  dropdownInputSearch: {
    fontSize: width * 0.04,
    color: '#000',
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownItemText: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '400',
  },
  dropdownItemContainer: {
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    paddingVertical: 8,
    paddingHorizontal: 12,
  },
  dropdownItemContainerSelected: {
    backgroundColor: '#025383',
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    paddingVertical: 8,
    paddingHorizontal: 12,
  },
  dropdownSearchInput: {
    fontSize: width * 0.04,
    color: '#000',
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
    paddingVertical: 8,
  },
  row: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: width * 0.02,
  },
  quantityColumn: {
    flex: 2,
    marginRight: width * 0.03,
  },
  unitColumn: {
    flex: 1,
  },
  column: {
    flex: 1,
    marginRight: width * 0.03,
  },
  statusContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    padding: width * 0.01,
    borderRadius: width * 0.015,
    marginBottom: width * 0.02,
  },
  statusContainerSelected: {
    // borderWidth: 1,
    // borderColor: '#025383',
  },
  statusButton: {
    flex: 1,
    paddingVertical: width * 0.03,
    marginHorizontal: width * 0.006,
    backgroundColor: '#E6E6E6',
    borderRadius: width * 0.01,
    alignItems: 'center',
  },
  statusButtonSelected: {
    backgroundColor: '#025383',
  },
  statusButtonText: {
    fontSize: width * 0.028,
    color: '#025383',
    textAlign: 'center',
    fontWeight: '500',
  },
  statusButtonTextSelected: {
    color: '#FFFFFF',
    fontWeight: '600',
  },
  equipmentSelectContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: width * 0.02,
  },
  equipmentText: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  equipmentNameText: {
    fontSize: width * 0.034,
    color: '#000',
    marginBottom: width * 0.01,
  },
  meterReadingText: {
    fontSize: width * 0.034,
    color: '#000',
    marginBottom: width * 0.028,
  },
  selectContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  selectText: {
    fontSize: width * 0.034,
    color: '#015185',
    paddingVertical: width * 0.025,
    marginRight: width * 0.02,
  },
  countrySelectContainer: {
    marginBottom: width * 0.02,
  },
  countryText: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  projectSelectContainer: {
    marginBottom: width * 0.02,
  },
  projectText: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  loadingModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  loadingModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  loadingIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  loadingModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  loadingIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  dateTimeText: {
    fontSize: width * 0.04,
    color: '#333',
  },
});

export default LogBookEntry;