# 📋 Inspection Report Module - Offline & MongoDB Sync

## Overview
The Inspection Report module supports **full offline functionality** with automatic MongoDB synchronization. Create, view, and manage inspection reports with or without internet connection.

---

## 🎯 Key Features

✅ **Full Offline Mode** - Create reports without internet  
✅ **Auto Request Numbers** - Sequential numbering based on MongoDB count  
✅ **PDF Storage** - Local PDFs viewable offline  
✅ **Manual Sync Control** - User controls when to sync (no auto-sync)  
✅ **Duplicate Prevention** - Smart deduplication by report ID  
✅ **Permission Caching** - Access controls work offline  

---

## 🔄 How It Works

### **1. Online Mode (Internet Available)**

```
User Creates Report → MongoDB → Assigned Request Number
                                 ↓
                         DRC-MUS-IR-001
                                 ↓
                    Cached Locally for Offline Access
```

**Request Number Format:** `{COUNTRY}-{PROJECT}-IR-{NUMBER}`
- Example: `DRC-MUS-IR-001`, `ZAM-KAL-IR-002`
- Number is based on **total count in MongoDB**

### **2. Offline Mode (No Internet)**

```
User Creates Report → Offline Storage → Temporary Request Number
                                        ↓
                                DRC-MUS-IR-001-OFFLINE
                                        ↓
                               Local PDF Generated
                                        ↓
                            Stored in AsyncStorage
```

**Features:**
- Reports saved with `-OFFLINE` suffix
- PDFs stored locally using React Native FS
- Yellow **SYNC** badge shown on report
- Viewable offline with all details

### **3. Sync Process (Manual)**

```
User Clicks Cloud Button → Fetch MongoDB Total Count
                                    ↓
                            Calculate Next Number
                                    ↓
                         Remove -OFFLINE Suffix
                                    ↓
                    Create Report in MongoDB with Clean Number
                         (DRC-MUS-IR-002)
                                    ↓
                         Upload Local PDF to MongoDB
                                    ↓
                         Remove from Offline Storage
                                    ↓
                          UI Refreshes Automatically
                           (SYNC badge removed)
```

---

## 📁 Storage Architecture

### **AsyncStorage Keys**

| Key | Purpose | Format |
|-----|---------|--------|
| `inspectionReports` | Cached reports from MongoDB | Array of reports with permissions |
| `offlineReports` | Reports created offline (pending sync) | Array of offline reports |
| `offlinePDFs` | Local PDF file paths | `{ reportId: { pdfPath, savedAt } }` |
| `inspectionModuleUserInfo` | User permissions & access | `{ countries, projects, inspectionPermissions }` |
| `inspectionReportsLastSync` | Last sync timestamp | ISO date string |
| `report_{reportId}` | Individual report cache | Full report object |

### **Offline Report Structure**

```javascript
{
  _id: "offline_1698598320123",
  requestNumber: "DRC-MUS-IR-001-OFFLINE",
  step1: {
    country: "DRC",
    project: "Musompo",
    inspector: "John Doe",
    requestNumber: "DRC-MUS-IR-001-OFFLINE"
  },
  step2: { /* checklist selections */ },
  step3: {
    pdfLocalPath: "/data/.../inspection_offline_1698598320123.pdf",
    createdAt: "2025-10-29T20:12:00.000Z"
  },
  offlineCreated: true,
  offlineSyncPending: true,
  userPermissions: ["create", "view"],
  userCountries: ["DRC", "Zambia"],
  userProjects: ["Musompo", "Lubumbashi"]
}
```

---

## 🚀 Creating an Offline Report

### **Step 1: Start New Inspection**
```javascript
// User presses "Start New Inspection" button
// Works online OR offline
navigation.navigate('InsreportHeader');
```

### **Step 2: Fill Report Details (Step 1)**
- Select Country (DRC, Zambia)
- Select Project (Musompo, Lubumbashi, etc.)
- Select Equipment
- Enter Inspector name
- **Offline mode:** Data saved locally in `inspectionStep1`

### **Step 3: Complete Checklist (Step 2)**
- Select checklist type
- Mark items as OK/Faulty
- Add photos (stored locally)
- **Offline mode:** Data saved in `inspectionStep2`

### **Step 4: Generate Report (Step 3)**
```javascript
// Generate PDF
const pdfPath = await generatePDF(reportData);

// Save offline
const offlineReport = {
  _id: `offline_${Date.now()}`,
  requestNumber: `${countryCode}-${projectCode}-IR-001-OFFLINE`,
  step1: step1Data,
  step2: step2Data,
  step3: {
    pdfLocalPath: pdfPath,
    createdAt: new Date().toISOString()
  },
  offlineCreated: true,
  offlineSyncPending: true
};

// Store in AsyncStorage
await OfflineDataHelper.saveOfflineReport(offlineReport);
```

**Result:**
- Report appears with yellow **SYNC** badge
- PDF viewable offline
- No internet required

---

## 🔄 Syncing Offline Reports

### **Manual Sync (Recommended)**

1. **User clicks Cloud button** (top-right corner)
2. **System checks for pending offline reports**
3. **For each offline report:**
   - Fetches MongoDB total count
   - Generates clean request number (removes `-OFFLINE`)
   - Creates report in MongoDB
   - Uploads local PDF
   - Updates report with PDF URL
   - Removes from offline storage
4. **UI auto-refreshes**
   - SYNC badges disappear
   - Reports show as synced
   - Counts update

### **Sync Code Flow**

```javascript
// 1. Get MongoDB count
const countResponse = await InspectionAPI.getTotalCount();
const nextNumber = countResponse.total + 1;

// 2. Generate clean request number
// Before: "DRC-MUS-IR-001-OFFLINE"
// After:  "DRC-MUS-IR-002" (based on MongoDB total)
const newRequestNumber = `${countryCode}-${projectCode}-IR-${String(nextNumber).padStart(3, '0')}`;

// 3. Update report
report.requestNumber = newRequestNumber;
report.step1.requestNumber = newRequestNumber;

// 4. Create in MongoDB
await InspectionAPI.createInspection(userId, report);

// 5. Upload PDF
const formData = new FormData();
formData.append('pdf', {
  uri: report.step3.pdfLocalPath,
  type: 'application/pdf',
  name: `inspection_${report._id}.pdf`
});
const pdfResponse = await InspectionAPI.uploadPdf(userId, formData);

// 6. Update with PDF URL
await InspectionAPI.updateInspection(userId, reportId, {
  'step3.pdfDownloadUrl': pdfResponse.downloadUrl
});

// 7. Remove from offline storage (automatic via OfflineDataHelper)
// 8. Reload UI to remove SYNC badge
await loadReports(false);
await updateReportCounts();
```

---

## 📊 Request Number Generation

### **MongoDB Count-Based Numbering**

```javascript
// Current MongoDB count: 5 reports exist
const countResponse = await InspectionAPI.getTotalCount();
// → { success: true, total: 5 }

// Next report gets number 6
const nextNumber = countResponse.total + 1; // → 6

// Generate request number
const requestNumber = `DRC-MUS-IR-006`;
```

### **Country Codes**
```javascript
const countryMap = {
  'Zambia': 'ZAM',
  'DRC': 'DRC'
};
```

### **Project Codes**
```javascript
const projectMap = {
  'lubumbashi': 'LUB',
  'musompo': 'MUS',
  'kansanshi': 'KAN',
  'kalumbila': 'KAL',
  'kimteto': 'KIM',
  'kobold': 'KOB',
  'fqm exploration': 'FQM',
  'mimosa': 'MIM',
  'ime': 'IME',
  'kamoa': 'KAM'
};
```

### **Examples**
```
DRC-MUS-IR-001  → DRC, Musompo, Inspection Report #1
ZAM-KAL-IR-002  → Zambia, Kalumbila, Inspection Report #2
DRC-LUB-IR-003  → DRC, Lubumbashi, Inspection Report #3
```

---

## 🔐 Permissions & Offline Access

### **Permission Types**
- `create` - Create new inspections
- `view` - View all inspections (filtered by country/project)
- `onlyMineView` - View only own inspections
- `delete` - Delete inspections

### **Offline Permission Caching**

Permissions are stored with **each cached report** to enable offline filtering:

```javascript
const report = {
  // ... report data ...
  userPermissions: ["create", "view"],
  userCountries: ["DRC", "Zambia"],
  userProjects: ["Musompo", "Lubumbashi"],
  syncedAt: "2025-10-29T20:00:00.000Z",
  syncedBy: "user_123"
};
```

**Offline Benefits:**
- User can view reports based on their permissions
- Country/project filtering works offline
- No need to re-authenticate

---

## 📱 UI Indicators

### **Offline Report Badges**

```javascript
{item.offlineSyncPending && (
  <View style={styles.offlineBadge}>
    <MaterialIcons name="cloud-upload" size={12} color="#fff" />
    <Text style={styles.offlineBadgeText}>SYNC</Text>
  </View>
)}
```

**Yellow SYNC Badge** = Report created offline, needs sync

### **Status Counter Cards**

| Card | Description | Tap Action |
|------|-------------|------------|
| **Offline Stored** | Total reports available offline (cached + pending) | Shows breakdown of cached vs pending |
| **Online in DB** | Reports synced to MongoDB | Shows sync metadata |
| **Storage** | Total AsyncStorage size | Shows storage details |

---

## 🛠️ Important Functions

### **OfflineDataHelper.js**

```javascript
// Save offline report
static async saveOfflineReport(report)

// Get all offline reports
static async getOfflineReports()

// Sync offline reports (with callback for each)
static async syncOfflineReports(uploadCallback)

// Save PDF locally
static async saveOfflinePDF(reportId, pdfPath)

// Get offline PDF path
static async getOfflinePDF(reportId)

// Clear all offline data
static async clearOfflineData()
```

### **Inspection_Report_Home.js**

```javascript
// Load reports (cache first, then MongoDB if online)
const loadReports = async (showModal = false)

// Sync offline reports to MongoDB
const syncOfflineReports = async ()

// Manual cloud sync (user-triggered)
const handleCloudSync = async ()

// Update offline/online counts
const updateReportCounts = async ()

// Fetch user permissions (cached offline)
const fetchUserInfo = async ()
```

---

## 🚫 What's Disabled (No Auto-Sync)

### **Removed Auto-Sync Triggers:**

❌ **No sync on screen focus** - Prevents duplicate uploads  
❌ **No sync on app load** - User controls when to sync  
❌ **No sync on connection restore** - Prevents accidental syncs  

### **Why Manual Sync?**

✅ **User control** - Sync only when ready  
✅ **No duplicates** - Single sync operation per click  
✅ **Network efficiency** - Sync during good connection  
✅ **Battery saving** - No background sync operations  

---

## 🐛 Troubleshooting

### **SYNC Badge Not Disappearing**
- Press **Cloud button** to sync
- UI auto-refreshes after successful sync
- Check console logs for sync errors

### **Request Numbers Not Sequential**
- Request numbers based on **MongoDB total count**
- If count is 5, next report is #6
- Offline suffix removed during sync

### **Duplicate Reports**
- Deduplication by report ID
- Each report ID appears only once
- Map-based filtering prevents duplicates

### **Offline Reports Not Showing**
- Check `offlineReports` in AsyncStorage
- Press **Refresh** button to reload cache
- Check permissions are cached

---

## 📝 Example Workflow

### **Scenario: Field Inspection (No Internet)**

1. **8:00 AM** - Technician arrives at site (no internet)
2. **8:05 AM** - Presses "Start New Inspection"
3. **8:10 AM** - Fills Step 1 (Location: DRC, Project: Musompo)
4. **8:30 AM** - Completes checklist (Step 2) - 2 faults found
5. **8:45 AM** - Generates PDF → `DRC-MUS-IR-001-OFFLINE`
6. **9:00 AM** - Returns to office (internet available)
7. **9:05 AM** - Presses **Cloud button**
8. **9:06 AM** - Report synced → `DRC-MUS-IR-002` (based on MongoDB count)
9. **9:07 AM** - SYNC badge disappears, PDF uploaded to MongoDB

**Result:**
- Report accessible in MongoDB
- PDF URL: `https://mongodb.com/pdfs/inspection_DRC-MUS-IR-002.pdf`
- Clean request number (no `-OFFLINE`)
- Available for other users to view

---

## 🎯 Best Practices

### **For Users**

1. **Create reports offline** - Works anywhere
2. **Sync regularly** - Click cloud button when online
3. **Check SYNC badges** - Yellow badge = needs sync
4. **Verify counts** - Tap counter cards to see details

### **For Developers**

1. **Always cache permissions** - Store with each report
2. **Use MongoDB count** - For sequential numbering
3. **Clean offline suffix** - Remove `-OFFLINE` before MongoDB
4. **Deduplicate reports** - Use Map by report ID
5. **Manual sync only** - No auto-sync triggers
6. **Refresh UI after sync** - Remove SYNC badges

---

## 📚 Related Files

| File | Purpose |
|------|---------|
| `Inspection_Report_Home.js` | Main listing, sync, offline logic |
| `OfflineDataHelper.js` | Offline storage utilities |
| `InsInfostep1.js` | Step 1 - Basic info |
| `Doinsstep2.js` | Step 2 - Checklist |
| `Summerystep3.js` | Step 3 - Summary & PDF generation |
| `InspectionMongoConfig.js` | MongoDB API endpoints |

---

## 🔑 Key Takeaways

✅ **Offline-First Design** - Works without internet  
✅ **Manual Sync Control** - User decides when to sync  
✅ **Sequential Numbering** - Based on MongoDB total count  
✅ **Clean Request Numbers** - No `-OFFLINE` in MongoDB  
✅ **Full Permission Caching** - Access control works offline  
✅ **Automatic UI Updates** - SYNC badges removed after sync  
✅ **Duplicate Prevention** - Smart deduplication by ID  

---

**Last Updated:** October 29, 2025  
**Module:** Inspection Report  
**Storage:** AsyncStorage + MongoDB  
**PDF Storage:** React Native FS (offline) + MongoDB (online)

