import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';

class InspectionEmailService {
  constructor() {
    this.emailQueue = [];
    this.isProcessing = false;
  }

  /**
   * Send inspection report emails to creator and supervisors
   * @param {Object} reportData - The inspection report data
   * @param {string} pdfUrl - PDF download URL
   * @param {string} signatureUrl - Signature image URL
   */
  async sendInspectionEmails(reportData, pdfUrl, signatureUrl) {
    try {
      console.log('Starting email notification process for inspection:', reportData.id);
      
      // Get current user info
      const currentUser = await this.getCurrentUser();
      if (!currentUser) {
        throw new Error('Current user not found');
      }

      // Get all supervisors for the same country and project
      const supervisors = await this.getSupervisorsByCountryAndProject(
        reportData.step1?.country,
        reportData.step1?.project
      );

      // Prepare email data
      const emailData = {
        reportId: reportData.id,
        requestNumber: reportData.step1?.requestNumber,
        country: reportData.step1?.country,
        project: reportData.step1?.project,
        inspector: reportData.step1?.inspector,
        equipment: reportData.step1?.selectedEquipment?.equipmentName,
        status: reportData.step3?.equipmentStatus,
        faultyItems: reportData.step3?.faultyItems,
        createdAt: reportData.step3?.createdAt,
        pdfUrl: pdfUrl,
        signatureUrl: signatureUrl,
        creator: {
          name: currentUser.name,
          email: currentUser.email,
          employeeNumber: currentUser.employeeNumber
        }
      };

      // Send email to creator
      await this.sendCreatorEmail(emailData);

      // Send emails to all supervisors
      await this.sendSupervisorEmails(emailData, supervisors);

      // Save email records to GlobalInspectionEmails
      await this.saveEmailRecords(emailData, supervisors);

      console.log('Email notification process completed successfully');
      Toast.show({
        type: 'success',
        text1: 'Email Notifications Sent',
        text2: `Emails sent to ${supervisors.length + 1} recipients`,
        position: 'top'
      });

    } catch (error) {
      console.error('Error sending inspection emails:', error);
      Toast.show({
        type: 'error',
        text1: 'Email Error',
        text2: 'Failed to send email notifications',
        position: 'top'
      });
      throw error;
    }
  }

  /**
   * Get current user information
   */
  async getCurrentUser() {
    try {
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        throw new Error('User email not found');
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');

      const userData = userSnapshot.val();
      if (!userData) {
        throw new Error('User not found in database');
      }

      const userId = Object.keys(userData)[0];
      const user = userData[userId];

      return {
        id: userId,
        name: user.name || 'Unknown User',
        email: user.email,
        employeeNumber: user.employeeNumber || '',
        role: user.role || 'user',
        countries: user.countries || [],
        projects: user.projects || []
      };
    } catch (error) {
      console.error('Error getting current user:', error);
      throw error;
    }
  }

  /**
   * Get all supervisors for a specific country and project
   */
  async getSupervisorsByCountryAndProject(country, project) {
    try {
      if (!country || !project) {
        console.warn('Country or project not provided for supervisor lookup');
        return [];
      }

      const usersSnapshot = await database()
        .ref('/Globalusers/admins')
        .once('value');

      const usersData = usersSnapshot.val();
      if (!usersData) {
        return [];
      }

      const supervisors = [];
      Object.entries(usersData).forEach(([userId, userData]) => {
        // Check if user has supervisor role
        if (userData.role === 'supervisor' || 
            userData.role === 'Supervisor' || 
            userData.role === 'SUPERVISOR') {
          
          // Check if user has access to the same country and project
          const hasCountryAccess = userData.countries && 
            Array.isArray(userData.countries) && 
            userData.countries.includes(country);
          
          const hasProjectAccess = userData.projects && 
            Array.isArray(userData.projects) && 
            userData.projects.includes(project);

          if (hasCountryAccess && hasProjectAccess) {
            supervisors.push({
              id: userId,
              name: userData.name || 'Unknown Supervisor',
              email: userData.email,
              employeeNumber: userData.employeeNumber || '',
              role: userData.role,
              countries: userData.countries || [],
              projects: userData.projects || []
            });
          }
        }
      });

      console.log(`Found ${supervisors.length} supervisors for ${country}/${project}`);
      return supervisors;
    } catch (error) {
      console.error('Error getting supervisors:', error);
      return [];
    }
  }

  /**
   * Send email to the inspection creator
   */
  async sendCreatorEmail(emailData) {
    try {
      const emailContent = this.generateCreatorEmailContent(emailData);
      
      // Save to email queue for processing
      await this.queueEmail({
        to: emailData.creator.email,
        subject: `Inspection Report Created - ${emailData.requestNumber}`,
        content: emailContent,
        type: 'creator',
        reportId: emailData.reportId,
        recipientId: emailData.creator.id,
        recipientRole: 'creator'
      });

      console.log('Creator email queued successfully');
    } catch (error) {
      console.error('Error sending creator email:', error);
      throw error;
    }
  }

  /**
   * Send emails to all supervisors
   */
  async sendSupervisorEmails(emailData, supervisors) {
    try {
      for (const supervisor of supervisors) {
        const emailContent = this.generateSupervisorEmailContent(emailData, supervisor);
        
        await this.queueEmail({
          to: supervisor.email,
          subject: `New Inspection Report - ${emailData.requestNumber}`,
          content: emailContent,
          type: 'supervisor',
          reportId: emailData.reportId,
          recipientId: supervisor.id,
          recipientRole: 'supervisor'
        });
      }

      console.log(`Supervisor emails queued for ${supervisors.length} recipients`);
    } catch (error) {
      console.error('Error sending supervisor emails:', error);
      throw error;
    }
  }

  /**
   * Generate email content for creator
   */
  generateCreatorEmailContent(emailData) {
    return `
      <html>
        <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
          <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
            <h2 style="color: #015185;">Inspection Report Created Successfully</h2>
            
            <p>Dear ${emailData.creator.name},</p>
            
            <p>Your inspection report has been created and submitted successfully.</p>
            
            <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px; margin: 20px 0;">
              <h3 style="margin-top: 0; color: #015185;">Report Details</h3>
              <p><strong>Request Number:</strong> ${emailData.requestNumber}</p>
              <p><strong>Country:</strong> ${emailData.country}</p>
              <p><strong>Project:</strong> ${emailData.project}</p>
              <p><strong>Equipment:</strong> ${emailData.equipment}</p>
              <p><strong>Status:</strong> ${emailData.status}</p>
              <p><strong>Faulty Items:</strong> ${emailData.faultyItems}</p>
              <p><strong>Created:</strong> ${new Date(emailData.createdAt).toLocaleString()}</p>
            </div>
            
            <p>Your report has been automatically shared with all supervisors in the ${emailData.country} - ${emailData.project} region.</p>
            
            <p>Best regards,<br>Titan Drilling System</p>
          </div>
        </body>
      </html>
    `;
  }

  /**
   * Generate email content for supervisors
   */
  generateSupervisorEmailContent(emailData, supervisor) {
    return `
      <html>
        <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
          <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
            <h2 style="color: #015185;">New Inspection Report Available</h2>
            
            <p>Dear ${supervisor.name},</p>
            
            <p>A new inspection report has been created in your jurisdiction.</p>
            
            <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px; margin: 20px 0;">
              <h3 style="margin-top: 0; color: #015185;">Report Details</h3>
              <p><strong>Request Number:</strong> ${emailData.requestNumber}</p>
              <p><strong>Country:</strong> ${emailData.country}</p>
              <p><strong>Project:</strong> ${emailData.project}</p>
              <p><strong>Inspector:</strong> ${emailData.inspector}</p>
              <p><strong>Equipment:</strong> ${emailData.equipment}</p>
              <p><strong>Status:</strong> ${emailData.status}</p>
              <p><strong>Faulty Items:</strong> ${emailData.faultyItems}</p>
              <p><strong>Created:</strong> ${new Date(emailData.createdAt).toLocaleString()}</p>
              <p><strong>Created By:</strong> ${emailData.creator.name} (${emailData.creator.employeeNumber})</p>
            </div>
            
            <p>Please review this report and take any necessary actions.</p>
            
            <p>Best regards,<br>Titan Drilling System</p>
          </div>
        </body>
      </html>
    `;
  }

  /**
   * Queue email for processing
   */
  async queueEmail(emailData) {
    try {
      const emailRecord = {
        id: Date.now().toString(),
        ...emailData,
        status: 'queued',
        createdAt: new Date().toISOString(),
        attempts: 0,
        lastAttempt: null
      };

      // Save to local queue
      this.emailQueue.push(emailRecord);
      
      // Save to AsyncStorage for persistence
      const existingQueue = JSON.parse(await AsyncStorage.getItem('emailQueue') || '[]');
      existingQueue.push(emailRecord);
      await AsyncStorage.setItem('emailQueue', JSON.stringify(existingQueue));

      // Process queue if not already processing
      if (!this.isProcessing) {
        this.processEmailQueue();
      }

      return emailRecord;
    } catch (error) {
      console.error('Error queuing email:', error);
      throw error;
    }
  }

  /**
   * Process the email queue
   */
  async processEmailQueue() {
    if (this.isProcessing || this.emailQueue.length === 0) {
      return;
    }

    this.isProcessing = true;
    console.log('Processing email queue...');

    try {
      while (this.emailQueue.length > 0) {
        const emailRecord = this.emailQueue.shift();
        
        try {
          await this.sendEmail(emailRecord);
          await this.updateEmailStatus(emailRecord.id, 'sent');
          console.log(`Email sent successfully to ${emailRecord.to}`);
        } catch (error) {
          console.error(`Failed to send email to ${emailRecord.to}:`, error);
          emailRecord.attempts += 1;
          emailRecord.lastAttempt = new Date().toISOString();
          
          if (emailRecord.attempts < 3) {
            // Re-queue for retry
            this.emailQueue.push(emailRecord);
          } else {
            await this.updateEmailStatus(emailRecord.id, 'failed');
          }
        }
      }
    } finally {
      this.isProcessing = false;
      console.log('Email queue processing completed');
    }
  }

  /**
   * Send individual email (placeholder for actual email service integration)
   */
  async sendEmail(emailRecord) {
    // This is a placeholder - you would integrate with your actual email service here
    // Examples: SendGrid, AWS SES, Nodemailer, etc.
    
    console.log('Sending email:', {
      to: emailRecord.to,
      subject: emailRecord.subject,
      type: emailRecord.type
    });

    // Simulate email sending delay
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    // For now, we'll just log the email content
    // In production, replace this with actual email service call
    console.log('Email content:', emailRecord.content);
    
    return true;
  }

  /**
   * Update email status in database
   */
  async updateEmailStatus(emailId, status) {
    try {
      const emailRef = database().ref(`/GlobalInspectionEmails/${emailId}`);
      await emailRef.update({
        status: status,
        updatedAt: new Date().toISOString()
      });
    } catch (error) {
      console.error('Error updating email status:', error);
    }
  }

  /**
   * Save email records to GlobalInspectionEmails database
   */
  async saveEmailRecords(emailData, supervisors) {
    try {
      const currentUser = await this.getCurrentUser();
      const emailRecords = [];

      // Create email record for creator
      emailRecords.push({
        id: `creator_${emailData.reportId}`,
        reportId: emailData.reportId,
        requestNumber: emailData.requestNumber,
        recipientId: currentUser.id,
        recipientEmail: currentUser.email,
        recipientName: currentUser.name,
        recipientRole: 'creator',
        emailType: 'inspection_created',
        status: 'queued',
        country: emailData.country,
        project: emailData.project,
        createdAt: new Date().toISOString(),
        metadata: {
          equipment: emailData.equipment,
          inspector: emailData.inspector,
          status: emailData.status,
          faultyItems: emailData.faultyItems
        }
      });

      // Create email records for supervisors
      supervisors.forEach(supervisor => {
        emailRecords.push({
          id: `supervisor_${emailData.reportId}_${supervisor.id}`,
          reportId: emailData.reportId,
          requestNumber: emailData.requestNumber,
          recipientId: supervisor.id,
          recipientEmail: supervisor.email,
          recipientName: supervisor.name,
          recipientRole: 'supervisor',
          emailType: 'inspection_notification',
          status: 'queued',
          country: emailData.country,
          project: emailData.project,
          createdAt: new Date().toISOString(),
          metadata: {
            equipment: emailData.equipment,
            inspector: emailData.inspector,
            status: emailData.status,
            faultyItems: emailData.faultyItems,
            creatorName: currentUser.name,
            creatorEmployeeNumber: currentUser.employeeNumber
          }
        });
      });

      // Save all records to database
      const batchUpdates = {};
      emailRecords.forEach(record => {
        batchUpdates[record.id] = record;
      });

      await database()
        .ref('/GlobalInspectionEmails')
        .update(batchUpdates);

      console.log(`Saved ${emailRecords.length} email records to database`);
    } catch (error) {
      console.error('Error saving email records:', error);
      throw error;
    }
  }

  /**
   * Get email statistics
   */
  async getEmailStatistics() {
    try {
      const snapshot = await database()
        .ref('/GlobalInspectionEmails')
        .once('value');

      const emails = snapshot.val() || {};
      const total = Object.keys(emails).length;
      const sent = Object.values(emails).filter(email => email.status === 'sent').length;
      const failed = Object.values(emails).filter(email => email.status === 'failed').length;
      const queued = Object.values(emails).filter(email => email.status === 'queued').length;

      return {
        total,
        sent,
        failed,
        queued,
        successRate: total > 0 ? ((sent / total) * 100).toFixed(2) : 0
      };
    } catch (error) {
      console.error('Error getting email statistics:', error);
      return { total: 0, sent: 0, failed: 0, queued: 0, successRate: 0 };
    }
  }

  /**
   * Retry failed emails
   */
  async retryFailedEmails() {
    try {
      const snapshot = await database()
        .ref('/GlobalInspectionEmails')
        .orderByChild('status')
        .equalTo('failed')
        .once('value');

      const failedEmails = snapshot.val() || {};
      let retryCount = 0;

      for (const [emailId, emailData] of Object.entries(failedEmails)) {
        if (emailData.attempts < 3) {
          await this.queueEmail({
            to: emailData.recipientEmail,
            subject: emailData.emailType === 'creator' ? 
              `Inspection Report Created - ${emailData.requestNumber}` :
              `New Inspection Report - ${emailData.requestNumber}`,
            content: this.generateEmailContent(emailData),
            type: emailData.recipientRole,
            reportId: emailData.reportId,
            recipientId: emailData.recipientId,
            recipientRole: emailData.recipientRole
          });
          retryCount++;
        }
      }

      console.log(`Retried ${retryCount} failed emails`);
      return retryCount;
    } catch (error) {
      console.error('Error retrying failed emails:', error);
      return 0;
    }
  }

  /**
   * Generate email content based on email data
   */
  generateEmailContent(emailData) {
    if (emailData.recipientRole === 'creator') {
      return this.generateCreatorEmailContent({
        id: emailData.reportId,
        step1: {
          requestNumber: emailData.requestNumber,
          country: emailData.country,
          project: emailData.project,
          inspector: emailData.metadata?.inspector,
          selectedEquipment: { equipmentName: emailData.metadata?.equipment }
        },
        step3: {
          equipmentStatus: emailData.metadata?.status,
          faultyItems: emailData.metadata?.faultyItems,
          createdAt: emailData.createdAt
        },
        creator: {
          name: emailData.recipientName,
          email: emailData.recipientEmail,
          employeeNumber: emailData.metadata?.creatorEmployeeNumber
        }
      });
    } else {
      return this.generateSupervisorEmailContent({
        id: emailData.reportId,
        step1: {
          requestNumber: emailData.requestNumber,
          country: emailData.country,
          project: emailData.project,
          inspector: emailData.metadata?.inspector,
          selectedEquipment: { equipmentName: emailData.metadata?.equipment }
        },
        step3: {
          equipmentStatus: emailData.metadata?.status,
          faultyItems: emailData.metadata?.faultyItems,
          createdAt: emailData.createdAt
        },
        creator: {
          name: emailData.metadata?.creatorName,
          email: emailData.metadata?.creatorEmail,
          employeeNumber: emailData.metadata?.creatorEmployeeNumber
        }
      }, {
        name: emailData.recipientName,
        email: emailData.recipientEmail,
        employeeNumber: emailData.metadata?.recipientEmployeeNumber
      });
    }
  }
}

export default new InspectionEmailService();
