import React, { useState, useEffect, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Alert,
  Image,
  TextInput,
  Platform,
  ToastAndroid,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AntDesign from 'react-native-vector-icons/AntDesign';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import checklistData from '../Checklist/Checklistfile.json';
import { useNavigation, useFocusEffect, useRoute } from '@react-navigation/native';
import DateTimePicker from '@react-native-community/datetimepicker';
import InspectionAPI from '../../ConfigMongo/InspectionMongoConfig.js';

const Doinsstep2 = ({ setActiveStep }) => {
  const [sections, setSections] = useState([]);
  const [expandedSections, setExpandedSections] = useState({});
  const [selections, setSelections] = useState({});
  const [selectedFiles, setSelectedFiles] = useState({});
  const [numericInputs, setNumericInputs] = useState({});
  const [checklistTitle, setChecklistTitle] = useState('');
  const [notes, setNotes] = useState({});
  const [inspectionId, setInspectionId] = useState(null);
  const [headerInputs, setHeaderInputs] = useState({});
  const [footerInputs, setFooterInputs] = useState({});
  const [headerInputValues, setHeaderInputValues] = useState({});
  const [footerInputValues, setFooterInputValues] = useState({});
  const [headerInputErrors, setHeaderInputErrors] = useState({});
  const [showHeaderErrors, setShowHeaderErrors] = useState(false);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [currentDateField, setCurrentDateField] = useState('');
  const [selectedDate, setSelectedDate] = useState(new Date());

  const navigation = useNavigation();
  const route = useRoute();

  useEffect(() => {
    loadInspectionData();
  }, []);

  // Reload notes and selections when screen is focused
  useFocusEffect(
    useCallback(() => {
      loadNotes();
      loadSelections();
      loadHeaderFooterInputs();
    }, [inspectionId])
  );

  // Load inspection ID and checklist data
  const loadInspectionData = async () => {
    try {
      const inspectionData = await AsyncStorage.getItem('inspectionStep1');
      if (inspectionData) {
        const parsedData = JSON.parse(inspectionData);
        setInspectionId(parsedData.id);
        await loadChecklistData(parsedData.id);
      } else {
        setChecklistTitle('No Checklist Selected');
        setSections([]);
      }
    } catch (error) {
      console.error('Error loading inspection data:', error);
      setChecklistTitle('Error Loading Data');
      setSections([]);
    }
  };

  // Load notes from AsyncStorage
  const loadNotes = async () => {
    try {
      const savedNotes = JSON.parse(await AsyncStorage.getItem(`inspectionNotes_${inspectionId}`) || '{}');
      setNotes(savedNotes);
    } catch (error) {
      console.error('Error loading notes:', error);
    }
  };

  // Load selections for the current inspection
  const loadSelections = async () => {
    try {
      const savedResults = JSON.parse(
        await AsyncStorage.getItem(`inspectionResults_${inspectionId}`) || '{}'
      );
      setSelections(savedResults.selections || {});
      setSelectedFiles(savedResults.selectedFiles || {});
      setNumericInputs(savedResults.numericInputs || {});
    } catch (error) {
      console.error('Error loading selections:', error);
    }
  };

  // Load header and footer inputs
  const loadHeaderFooterInputs = async () => {
    try {
      const savedHeaderInputs = JSON.parse(
        await AsyncStorage.getItem(`checklistHeaderInputs_${inspectionId}`) || '{}'
      );
      const savedFooterInputs = JSON.parse(
        await AsyncStorage.getItem(`checklistFooterInputs_${inspectionId}`) || '{}'
      );
      const savedHeaderValues = JSON.parse(
        await AsyncStorage.getItem(`headerInputValues_${inspectionId}`) || '{}'
      );
      const savedFooterValues = JSON.parse(
        await AsyncStorage.getItem(`footerInputValues_${inspectionId}`) || '{}'
      );
      
      setHeaderInputs(savedHeaderInputs);
      setFooterInputs(savedFooterInputs);
      
      // Initialize today's date for date fields if not already set
      const today = new Date().toLocaleDateString();
      const initializedHeaderValues = { ...savedHeaderValues };
      
      Object.keys(savedHeaderInputs).forEach(key => {
        const isDateField = key.toLowerCase().includes('date') || savedHeaderInputs[key].toLowerCase().includes('date');
        if (isDateField && !initializedHeaderValues[key]) {
          initializedHeaderValues[key] = today;
        }
      });
      
      setHeaderInputValues(initializedHeaderValues);
      setFooterInputValues(savedFooterValues);
      
      // Save the initialized values
      await AsyncStorage.setItem(`headerInputValues_${inspectionId}`, JSON.stringify(initializedHeaderValues));
      
      // Auto-fill equipment data from Step 1
      await autoFillEquipmentData();
    } catch (error) {
      console.error('Error loading header/footer inputs:', error);
    }
  };

  // Handle header input changes
  const handleHeaderInputChange = async (key, value) => {
    const newValues = { ...headerInputValues, [key]: value };
    setHeaderInputValues(newValues);
    
    // Validate in real-time
    const isEmpty = !value || value.trim() === '';
    setHeaderInputErrors(prev => ({
      ...prev,
      [key]: isEmpty
    }));
    
    await AsyncStorage.setItem(`headerInputValues_${inspectionId}`, JSON.stringify(newValues));
  };

  // Handle footer input changes
  const handleFooterInputChange = async (key, value) => {
    const newValues = { ...footerInputValues, [key]: value };
    setFooterInputValues(newValues);
    await AsyncStorage.setItem(`footerInputValues_${inspectionId}`, JSON.stringify(newValues));
  };

  // Handle date picker
  const handleDatePress = (fieldKey) => {
    setCurrentDateField(fieldKey);
    setShowDatePicker(true);
  };

  // Format header label (capitalize and replace underscores with spaces)
  const formatHeaderLabel = (label) => {
    if (!label) return '';
    return label
      .replace(/_/g, ' ') // Replace underscores with spaces
      .split(' ')
      .map(word => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
      .join(' ');
  };

  // Format section title: title case and replace underscores with spaces
  const formatSectionTitle = (title) => {
    if (!title) return '';
    return title
      .toString()
      .replace(/_/g, ' ')
      .split(' ')
      .map(word => word ? word.charAt(0).toUpperCase() + word.slice(1).toLowerCase() : '')
      .join(' ');
  };

  // Format item name: uppercase and replace underscores with spaces
  const formatItemNameUpper = (name) => {
    if (!name && name !== 0) return '';
    return String(name).replace(/_/g, ' ').toUpperCase();
  };

  // Helpers to detect special header fields
  const isDateFieldKeyOrLabel = (key, label) => {
    const lowerKey = (key || '').toLowerCase();
    const lowerLabel = (label || '').toLowerCase();
    return lowerKey.includes('date') || lowerLabel.includes('date');
  };

  const isProjectOrSiteField = (key, label) => {
    const lowerKey = (key || '').toLowerCase();
    const lowerLabel = (label || '').toLowerCase();
    return (
      lowerKey.includes('project') ||
      lowerLabel.includes('project') ||
      lowerKey.includes('site') ||
      lowerLabel.includes('site') ||
      lowerKey.includes('location') ||
      lowerLabel.includes('location')
    );
  };

  const isOdometerField = (key, label) => {
    const lowerKey = (key || '').toLowerCase();
    const lowerLabel = (label || '').toLowerCase();
    return lowerKey.includes('odometer') || lowerLabel.includes('odometer');
  };

  const getBestFleetValue = (equipment) => {
    if (!equipment) return '';
    return (
      equipment.fleetNumber ||
      equipment.fleet_no ||
      equipment.fleetNo ||
      equipment.fleet ||
      equipment.vehicleNumber ||
      equipment.registrationNumber ||
      equipment.registration ||
      equipment.assetNumber ||
      equipment.equipmentId ||
      equipment.id ||
      ''
    );
  };

  // Auto-fill equipment data from Step 1
  const autoFillEquipmentData = async () => {
    try {
      const inspectionData = await AsyncStorage.getItem('inspectionStep1');
      if (inspectionData) {
        const parsedData = JSON.parse(inspectionData);
        const selectedEquipment = parsedData.selectedEquipment;
        
        // Build a single working copy to avoid overwriting with stale state
        let workingValues = { ...headerInputValues };

        if (selectedEquipment) {
          // Auto-fill based on field names and equipment data
          Object.keys(headerInputs).forEach(key => {
            const lowerKey = key.toLowerCase();
            const lowerLabel = (headerInputs[key] || '').toLowerCase();
            
            // Auto-fill model
            if (lowerKey.includes('model') || lowerLabel.includes('model')) {
              workingValues[key] = selectedEquipment.model || selectedEquipment.vehicleNumber || '';
            }
            
            // Auto-fill rig
            if (lowerKey.includes('rig') || lowerLabel.includes('rig')) {
              workingValues[key] = selectedEquipment.rig || selectedEquipment.equipmentName || '';
            }
            
            // Auto-fill fleet number
            if (lowerKey.includes('fleet') || lowerLabel.includes('fleet')) {
              workingValues[key] = getBestFleetValue(selectedEquipment);
            }
            
            // Auto-fill meter/odometer reading
            if (lowerKey.includes('meter') || lowerKey.includes('odometer') || lowerLabel.includes('meter') || lowerLabel.includes('odometer')) {
              workingValues[key] = selectedEquipment.meterReading?.toString() || '';
            }
            
            // Auto-fill equipment name
            if (lowerKey.includes('equipment') || lowerLabel.includes('equipment')) {
              workingValues[key] = selectedEquipment.equipmentName || '';
            }
          });
        }

        // Auto-fill project/site/location related fields from Step 1 project
        if (parsedData.project) {
          Object.keys(headerInputs).forEach(key => {
            const label = headerInputs[key] || '';
            if (isProjectOrSiteField(key, label)) {
              workingValues[key] = parsedData.project;
            }
          });
        }

        // Ensure date and fleet defaults if still empty
        const today = new Date().toLocaleDateString();
        Object.keys(headerInputs).forEach(key => {
          const label = headerInputs[key] || '';
          if (isDateFieldKeyOrLabel(key, label) && !workingValues[key]) {
            workingValues[key] = today;
          }
          if ((key.toLowerCase().includes('fleet') || label.toLowerCase().includes('fleet')) && !workingValues[key]) {
            workingValues[key] = getBestFleetValue(selectedEquipment || parsedData.selectedEquipment);
          }
        });

        setHeaderInputValues(workingValues);
        await AsyncStorage.setItem(`headerInputValues_${inspectionId}`, JSON.stringify(workingValues));
      }
    } catch (error) {
      console.error('Error auto-filling equipment data:', error);
    }
  };

  const handleDateChange = (event, date) => {
    if (Platform.OS === 'android') {
      setShowDatePicker(false);
    }
    
    if (date) {
      setSelectedDate(date);
      const formattedDate = date.toLocaleDateString();
      handleHeaderInputChange(currentDateField, formattedDate);
    }
  };

  const handleDateConfirm = () => {
    if (Platform.OS === 'ios') {
      setShowDatePicker(false);
      const formattedDate = selectedDate.toLocaleDateString();
      handleHeaderInputChange(currentDateField, formattedDate);
    }
  };

  const handleDateCancel = () => {
    setShowDatePicker(false);
  };

  // Ensure project/location/date fields auto-filled when headerInputs are available
  useEffect(() => {
    const ensureHeaderDefaults = async () => {
      try {
        if (!inspectionId) return;
        if (!headerInputs || Object.keys(headerInputs).length === 0) return;
        const saved = await AsyncStorage.getItem('inspectionStep1');
        const parsed = saved ? JSON.parse(saved) : {};
        const today = new Date().toLocaleDateString();
        let updated = { ...headerInputValues };
        let changed = false;
        Object.keys(headerInputs).forEach(key => {
          const label = headerInputs[key] || '';
          if (isDateFieldKeyOrLabel(key, label)) {
            if (!updated[key]) { updated[key] = today; changed = true; }
          }
          if (isProjectOrSiteField(key, label) && parsed.project) {
            if (updated[key] !== parsed.project) { updated[key] = parsed.project; changed = true; }
          }
          if ((key.toLowerCase().includes('fleet') || label.toLowerCase().includes('fleet'))) {
            const bestFleet = getBestFleetValue(parsed.selectedEquipment);
            if (bestFleet && updated[key] !== bestFleet) { updated[key] = bestFleet; changed = true; }
          }
        });
        if (changed) {
          setHeaderInputValues(updated);
          await AsyncStorage.setItem(`headerInputValues_${inspectionId}`, JSON.stringify(updated));
        }
      } catch (e) {
        console.warn('ensureHeaderDefaults error', e?.message);
      }
    };
    ensureHeaderDefaults();
  }, [headerInputs, inspectionId]);

  // Load checklist data for the current inspection
  const loadChecklistData = async (currentInspectionId) => {
    try {
      const selectedChecklistIds = JSON.parse(
        await AsyncStorage.getItem('selectedChecklistIds') || '[]'
      );

      if (selectedChecklistIds.length === 0) {
        console.warn('No checklist IDs selected');
        setChecklistTitle('No Checklist Selected');
        setSections([]);
        return;
      }

      const checklistId = selectedChecklistIds[0];
      let checklist;

      const storedChecklist = await AsyncStorage.getItem(`checklist_${checklistId}`);
      console.log('Stored checklist from AsyncStorage:', storedChecklist);
      if (storedChecklist) {
        checklist = JSON.parse(storedChecklist);
        console.log('Parsed checklist data:', checklist);
      } else {
        console.log('No stored checklist found, checking stored checklists');
        const storedChecklists = JSON.parse(
          await AsyncStorage.getItem('checklists') || '[]'
        );
        checklist = storedChecklists.find((c) => selectedChecklistIds.includes(c.id));
        console.log('Found checklist from stored checklists:', checklist);
        if (!checklist) {
          console.log('Checking JSON checklist data');
          const jsonChecklist = checklistData.Checklistfile.documents.find(
            (doc) => doc.document_id === checklistId
          );
          console.log('Found JSON checklist:', jsonChecklist);
          if (jsonChecklist) {
            checklist = {
              id: jsonChecklist.document_id,
              title: jsonChecklist.document_name,
              description: 'General Checklist',
              checklist: jsonChecklist.checklist,
              header_inputs: jsonChecklist.header_inputs || {},
              footer_inputs: jsonChecklist.footer_inputs || {},
            };
            console.log('Created checklist from JSON:', checklist);
          }
        }
      }

      if (!checklist || (!checklist.checklist && !checklist.items)) {
        setChecklistTitle(checklist?.title || 'No Checklist Selected');
        setSections([]);
        return;
      }

      setChecklistTitle(checklist.title || 'Untitled Checklist');

      // Fetch header inputs from MongoDB for the selected checklist
      try {
        console.log('Fetching header inputs for checklistId:', checklistId);
        
        // Try to fetch from MongoDB first
        const checklistResponse = await InspectionAPI.getChecklist(checklistId);
        
        if (checklistResponse.success && checklistResponse.data) {
          const targetDocument = checklistResponse.data;
          console.log('Found target document from MongoDB:', targetDocument);
          
          if (targetDocument.header_inputs) {
            console.log('Setting header inputs from MongoDB:', targetDocument.header_inputs);
            setHeaderInputs(targetDocument.header_inputs);
            await AsyncStorage.setItem(`checklistHeaderInputs_${currentInspectionId}`, JSON.stringify(targetDocument.header_inputs));
          } else if (checklist.header_inputs) {
            setHeaderInputs(checklist.header_inputs);
            await AsyncStorage.setItem(`checklistHeaderInputs_${currentInspectionId}`, JSON.stringify(checklist.header_inputs));
          }
        } else {
          console.log('No document found in MongoDB, using stored checklist');
          if (checklist.header_inputs) {
            setHeaderInputs(checklist.header_inputs);
            await AsyncStorage.setItem(`checklistHeaderInputs_${currentInspectionId}`, JSON.stringify(checklist.header_inputs));
          }
        }
      } catch (mongoError) {
        console.error('Error fetching header inputs from MongoDB:', mongoError);
        // Fallback to stored header inputs
        if (checklist.header_inputs) {
          setHeaderInputs(checklist.header_inputs);
          await AsyncStorage.setItem(`checklistHeaderInputs_${currentInspectionId}`, JSON.stringify(checklist.header_inputs));
        }
      }

      // Process checklist items (either from 'checklist' or 'items')
      let dynamicSections = [];
      if (checklist.items) {
        // Handle user-created checklists with 'items' structure
        dynamicSections = checklist.items.map((item, idx) => {
          if (item.text && item.subItems) {
            return {
              key: `${checklist.id}_${item.text}_${idx}`,
              name: item.text,
              subItems: item.subItems.map((subItem, subIdx) => ({
                key: subItem.key || `${checklist.id}_${item.text}_${subIdx}`,
                itemName: subItem.itemName || 'Unnamed Inspection Item',
                category: item.text,
                description: subItem.description || '',
                inputType: subItem.inputType || 'Okay/Faulty/N/A',
                instruction: subItem.instruction || '',
                sortOrder: subItem.sortOrder !== undefined ? subItem.sortOrder : subIdx,
                isInspectionRequired: subItem.isInspectionRequired || false,
              })),
            };
          }
          return null;
        }).filter((section) => section && Array.isArray(section.subItems) && section.subItems.length > 0);
      } else if (checklist.checklist) {
        // Handle global checklists with 'checklist' structure (like your JSON)
        const sectionEntries = Object.entries(checklist.checklist);
        console.log('Processing checklist sections:', sectionEntries);
        
        dynamicSections = sectionEntries
          .map(([sectionKey, sectionValue], idx) => {
            console.log(`Processing section ${sectionKey}:`, sectionValue);
            
            // Handle array of strings (like your A, B, C sections)
            if (Array.isArray(sectionValue) && sectionValue.length > 0 && typeof sectionValue[0] === 'string') {
              const subItems = sectionValue.map((item, i) => ({
                key: `${checklist.id}_${sectionKey}_${i}`,
                itemName: item,
                category: sectionKey, // This is crucial for PDF classification
                inputType: 'Okay/Faulty/N/A',
                sortOrder: i,
                isInspectionRequired: false,
                description: '',
                instruction: '',
              }));
              
              console.log(`Created ${subItems.length} items for section ${sectionKey}:`, subItems);
              
              return {
                key: `${checklist.id}_${sectionKey}`,
                name: sectionKey, // Use actual section key instead of "Section A"
                subItems,
              };
            }
            // Handle array of objects
            if (Array.isArray(sectionValue) && sectionValue.length > 0 && typeof sectionValue[0] === 'object') {
              const subItems = sectionValue.map((item, i) => ({
                key: `${checklist.id}_${sectionKey}_${i}`,
                itemName: item.itemName || item,
                category: sectionKey, // This is crucial for PDF classification
                inputType: item.inputType || 'Okay/Faulty/N/A',
                sortOrder: i,
                isInspectionRequired: item.isInspectionRequired || false,
                description: item.description || '',
                instruction: item.instruction || '',
              }));
              
              console.log(`Created ${subItems.length} object items for section ${sectionKey}:`, subItems);
              
              return {
                key: `${checklist.id}_${sectionKey}`,
                name: sectionKey === 'items' ? 'Checklist Items' : sectionKey, // Use actual section key
                subItems,
              };
            }
            // Handle nested objects
            if (typeof sectionValue === 'object' && sectionValue !== null) {
              const subItems = Object.values(sectionValue).flat().map((item, i) => ({
                key: `${checklist.id}_${sectionKey}_${i}`,
                itemName: item.itemName || item,
                category: sectionKey, // This is crucial for PDF classification
                inputType: item.inputType || 'Okay/Faulty/N/A',
                sortOrder: i,
                isInspectionRequired: item.isInspectionRequired || false,
                description: item.description || '',
                instruction: item.instruction || '',
              }));
              if (subItems.length > 0) {
                console.log(`Created ${subItems.length} nested items for section ${sectionKey}:`, subItems);
                return {
                  key: `${checklist.id}_${sectionKey}`,
                  name: sectionKey, // Use actual section key
                  subItems,
                };
              }
            }
            return null;
          })
          .filter((section) => section && Array.isArray(section.subItems) && section.subItems.length > 0);
      }

      console.log('Final dynamic sections:', dynamicSections);

      // Sort sections: A, B, C first (in order), then the rest
      const sectionOrder = ['A', 'B', 'C'];
      const orderedSections = [];
      sectionOrder.forEach((key) => {
        const idx = dynamicSections.findIndex((s) => s.name === key);
        if (idx !== -1) {
          orderedSections.push(dynamicSections[idx]);
          dynamicSections.splice(idx, 1);
        }
      });
      const finalSections = [...orderedSections, ...dynamicSections];

      setSections(finalSections);

      // Initialize expanded sections
      const initialExpanded = {};
      finalSections.forEach((section) => {
        initialExpanded[section.key] = false;
      });
      setExpandedSections(initialExpanded);

      // Initialize selections, files, and numeric inputs
      const savedResults = JSON.parse(
        await AsyncStorage.getItem(`inspectionResults_${currentInspectionId}`) || '{}'
      );
      const initialSelections = {};
      const initialFiles = {};
      const initialNumericInputs = {};
      finalSections
        .flatMap((section) => section.subItems)
        .forEach((item) => {
          initialSelections[item.key] = savedResults.selections?.[item.key] || null;
          initialFiles[item.key] = savedResults.selectedFiles?.[item.key] || null;
          initialNumericInputs[item.key] = savedResults.numericInputs?.[item.key] || '';
        });

      setSelections(initialSelections);
      setSelectedFiles(initialFiles);
      setNumericInputs(initialNumericInputs);

      // Save footer inputs for rendering
      if (checklist.footer_inputs) {
        await AsyncStorage.setItem(`checklistFooterInputs_${currentInspectionId}`, JSON.stringify(checklist.footer_inputs));
      }
    } catch (error) {
      console.error('Error loading checklist data:', error);
      setSections([]);
      setChecklistTitle('Error Loading Checklist');
    }
  };

  // Updated getOptionsForInputType to handle various input types
  const getOptionsForInputType = (inputType) => {
    switch (inputType) {
      case 'Okay/Faulty/N/A':
        return ['Okay', 'Faulty', 'N/A'];
      case 'Ok/Reject/N/A':
        return ['Ok', 'Reject', 'N/A'];
      case 'Ok/Not Ok':
        return ['Ok', 'Not Ok'];
      case 'Numeric':
        return [];
      default:
        if (typeof inputType === 'string' && inputType.includes('/')) {
          const options = inputType.split('/').map((opt) => opt.trim()).filter((opt) => opt);
          if (options.length > 0) {
            return options;
          }
        }
        return ['Okay', 'Faulty', 'N/A'];
    }
  };

  const handleSelectOption = async (itemKey, option) => {
    setSelections((prev) => {
      const updatedSelections = {
        ...prev,
        [itemKey]: option,
      };
      saveInspectionResults(updatedSelections, selectedFiles, numericInputs);
      return updatedSelections;
    });
    if (Platform.OS === 'android') {
      ToastAndroid.show(`Selected: ${option}`, ToastAndroid.SHORT);
    }
  };

  const handleNumericInput = async (itemKey, value) => {
    setNumericInputs((prev) => {
      const updatedNumericInputs = {
        ...prev,
        [itemKey]: value.replace(/[^0-9]/g, ''),
      };
      setSelections((prevSelections) => {
        const updatedSelections = {
          ...prevSelections,
          [itemKey]: value ? value : null,
        };
        saveInspectionResults(updatedSelections, selectedFiles, updatedNumericInputs);
        return updatedSelections;
      });
      return updatedNumericInputs;
    });
  };

  const handleFilePick = (itemKey) => {
    const selection = selections[itemKey];
    if (!selection) {
      if (Platform.OS === 'android') {
        ToastAndroid.show('Please make a selection first', ToastAndroid.SHORT);
      } else {
        Alert.alert('Selection Required', 'Please make a selection first');
      }
      return;
    }

    Alert.alert(
      'Select Image',
      'Choose an option',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'Take Picture', onPress: () => pickImage(itemKey, 'camera') },
        { text: 'Choose from Gallery', onPress: () => pickImage(itemKey, 'gallery') },
      ],
      { cancelable: true }
    );
  };

  const handleNotePress = async (itemKey) => {
    const selection = selections[itemKey];
    if (!selection) {
      if (Platform.OS === 'android') {
        ToastAndroid.show('Please make a selection first', ToastAndroid.SHORT);
      } else {
        Alert.alert('Selection Required', 'Please make a selection first');
      }
      return;
    }

    await saveInspectionResults(selections, selectedFiles, numericInputs);
    navigation.navigate('DoinspecNoteadd', { itemKey, inspectionId });
  };

  const handleBackPress = () => {
    Alert.alert(
      'Exit Step',
      'Do you want to exit this step? Your data will be saved.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'OK',
          onPress: async () => {
            await saveInspectionResults(selections, selectedFiles, numericInputs);
            navigation.goBack();
          },
          style: 'default',
        },
      ],
      { cancelable: false }
    );
  };

  const pickImage = async (itemKey, source) => {
    const options = {
      mediaType: 'photo',
      maxWidth: 500,
      maxHeight: 500,
      quality: 0.8,
    };

    const callback = async (response) => {
      if (response.didCancel) {
        console.log('User cancelled image picker');
      } else if (response.errorCode) {
        console.error('Image Picker Error: ', response.errorMessage);
        Alert.alert('Error', response.errorMessage || 'Failed to pick image');
      } else {
        const uri = response.assets[0].uri;
        setSelectedFiles((prev) => {
          const updatedFiles = {
            ...prev,
            [itemKey]: uri,
          };
          saveInspectionResults(selections, updatedFiles, numericInputs);
          return updatedFiles;
        });
      }
    };

    if (source === 'camera') {
      launchCamera(options, callback);
    } else {
      launchImageLibrary(options, callback);
    }
  };

  const calculateFaultyItems = () => {
    const allItems = sections.flatMap((section) => section.subItems);
    return allItems.reduce((count, item) => {
      const selection = selections[item.key];
      if (selection && (selection === 'Faulty' || selection === 'Reject' || selection === 'Not Ok') && item.inputType !== 'Numeric') {
        return count + 1;
      }
      return count;
    }, 0);
  };

  const validateSelections = () => {
    const requiredItems = sections
      .flatMap((section) => section.subItems)
      .filter((item) => item.isInspectionRequired);

    const missingRequiredItems = requiredItems.filter(
      (item) => !selections[item.key] || selections[item.key] === ''
    );

    if (missingRequiredItems.length > 0) {
      Alert.alert(
        'Required Items Missing',
        'Please complete all required inspection items before proceeding.',
        [{ text: 'OK' }]
      );
      return false;
    }

    return true;
  };

  const validateHeaderInputs = () => {
    const headerInputKeys = Object.keys(headerInputs);
    const errors = {};
    let hasErrors = false;

    headerInputKeys.forEach(key => {
      const label = headerInputs[key] || '';
      if (isOdometerField(key, label)) {
        errors[key] = false;
        return;
      }
      const isEmpty = !headerInputValues[key] || headerInputValues[key].trim() === '';
      errors[key] = isEmpty;
      if (isEmpty) hasErrors = true;
    });

    setHeaderInputErrors(errors);
    setShowHeaderErrors(true);

    if (hasErrors) {
      Alert.alert(
        'Required Fields Missing',
        'Please fill in all header information fields before proceeding.',
        [{ text: 'OK' }]
      );
      return false;
    }

    return true;
  };

  const saveInspectionResults = async (selectionsToSave, filesToSave, numericInputsToSave) => {
    try {
      const savedNotes = JSON.parse(await AsyncStorage.getItem(`inspectionNotes_${inspectionId}`) || '{}');
      const inspectionResults = {
        selections: selectionsToSave,
        selectedFiles: filesToSave,
        numericInputs: numericInputsToSave,
        notes: savedNotes,
        sections,
        checklistTitle,
        faultyItems: calculateFaultyItems(),
        timestamp: new Date().toISOString(),
      };
      await AsyncStorage.setItem(
        `inspectionResults_${inspectionId}`,
        JSON.stringify(inspectionResults)
      );
    } catch (error) {
      console.error('Error saving inspection results:', error);
    }
  };

  const handleNext = async () => {
    // Validate header inputs are filled
    if (!validateHeaderInputs()) {
      return;
    }

    if (validateSelections()) {
      try {
        // Log section-wise data before saving
        console.log('=== SECTION-WISE INSPECTION DATA ===');
        sections.forEach((section) => {
          const sectionItems = section.subItems || [];
          const selectedItems = sectionItems.filter(item => selections[item.key]);
          const faultyItems = sectionItems.filter(item => 
            selections[item.key] === 'Faulty' || 
            selections[item.key] === 'Reject' || 
            selections[item.key] === 'Not Ok'
          );
          
          console.log(`Section: ${section.name}`);
          console.log(`- Total Items: ${sectionItems.length}`);
          console.log(`- Selected Items: ${selectedItems.length}`);
          console.log(`- Faulty Items: ${faultyItems.length}`);
          console.log(`- Items:`, sectionItems.map(item => ({
            name: item.itemName,
            selection: selections[item.key],
            category: item.category
          })));
          console.log('---');
        });
        
        console.log('=== HEADER INPUTS ===');
        console.log('Header Input Values:', headerInputValues);
        console.log('=== FOOTER INPUTS ===');
        console.log('Footer Input Values:', footerInputValues);
        
        await saveInspectionResults(selections, selectedFiles, numericInputs);
        const existingResults = JSON.parse(
          await AsyncStorage.getItem('inspectionResults') || '[]'
        );
        existingResults.push({
          inspectionId,
          selections,
          selectedFiles,
          numericInputs,
          notes,
          sections,
          checklistTitle,
          headerInputValues,
          footerInputValues,
          faultyItems: calculateFaultyItems(),
          timestamp: new Date().toISOString(),
        });
        await AsyncStorage.setItem('inspectionResults', JSON.stringify(existingResults));
        setActiveStep(3);
      } catch (error) {
        console.error('Error saving inspection results:', error);
        Alert.alert('Error', 'Failed to save inspection data.');
      }
    }
  };

  const toggleSection = (sectionKey) => {
    setExpandedSections((prev) => ({
      ...prev,
      [sectionKey]: !prev[sectionKey],
    }));
  };

  const renderItem = (item, index) => {
    const selectedOption = selections[item.key];
    const selectedFileUri = selectedFiles[item.key];
    const numericValue = numericInputs[item.key];
    const itemNote = notes[item.key] || '';
    const options = getOptionsForInputType(item.inputType);

    return (
      <View key={item.key} style={styles.centeredItem}>
        <View style={styles.itemContainer}>
          <Text style={styles.itemText}>
            {index + 1}. {formatItemNameUpper(item.itemName)}
            {item.isInspectionRequired && <Text style={styles.requiredText}> (Required)</Text>}
          </Text>
          {item.description && (
            <Text style={styles.detailText}>Description: {item.description}</Text>
          )}
          {item.instruction && (
            <Text style={styles.detailText}>Instruction: {item.instruction}</Text>
          )}
          <View style={styles.itemRow}>
            {item.inputType === 'Numeric' ? (
              <TextInput
                style={styles.numericInput}
                value={numericValue}
                onChangeText={(text) => handleNumericInput(item.key, text)}
                placeholder="Enter number"
                keyboardType="numeric"
              />
            ) : (
              <View style={styles.optionsContainer}>
                {options.map((option) => {
                  const isSelected = selectedOption === option;
                  return (
                    <TouchableOpacity
                      key={option}
                      style={[
                        styles.optionButton,
                        isSelected && styles.selectedOptionButton,
                        isSelected && (option === 'Okay' || option === 'Ok') && styles.okButton,
                        isSelected && (option === 'Faulty' || option === 'Reject' || option === 'Not Ok') && styles.rejectButton,
                        isSelected && option === 'N/A' && styles.naButton,
                        isSelected && !['Okay', 'Ok', 'Faulty', 'Reject', 'Not Ok', 'N/A'].includes(option) && styles.customOptionButton,
                      ]}
                      onPress={() => handleSelectOption(item.key, option)}
                    >
                      <Text
                        style={[
                          styles.optionText,
                          isSelected && styles.optionTextSelected,
                        ]}
                      >
                        {option}
                      </Text>
                    </TouchableOpacity>
                  );
                })}
              </View>
            )}
            <View style={styles.iconsContainer}>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => handleNotePress(item.key)}
              >
                <MaterialIcons name="note-add" size={24} color="#333" />
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => handleFilePick(item.key)}
              >
                <MaterialIcons name="camera-alt" size={24} color="#333" />
              </TouchableOpacity>
            </View>
          </View>
          {itemNote && (
            <Text style={styles.noteText}>
              Note: {itemNote}
            </Text>
          )}
        </View>
        {selectedFileUri && (
          <Image
            source={{ uri: selectedFileUri }}
            style={styles.selectedImage}
            onError={(e) => console.error('Image error:', e.nativeEvent.error)}
          />
        )}
        <View style={styles.itemBorder} />
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <Text style={styles.checklistTitle}>{checklistTitle}</Text>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <View style={styles.headingContainer}>
          <Text style={styles.headingText}>
            Tap on section to expand
          </Text>
        </View>

        {/* Header Inputs Section */}
        <View style={styles.headerInputsContainer}>
          <View style={styles.headerInputsHeader}>
            <Text style={styles.headerInputsTitle}>Header Information <Text style={styles.requiredText}>* Required</Text></Text>
            <TouchableOpacity 
              style={styles.autoFillButton}
              onPress={autoFillEquipmentData}
            >
              <MaterialIcons name="refresh" size={16} color="#0078D4" />
              <Text style={styles.autoFillButtonText}>Auto Fill</Text>
            </TouchableOpacity>
          </View>
          {console.log('Header inputs to render:', headerInputs)}
          {console.log('Header inputs keys:', Object.keys(headerInputs))}
          {Object.keys(headerInputs).length > 0 ? (
            Object.entries(headerInputs).map(([key, label]) => {
              console.log('Rendering header input:', key, label);
              const hasError = showHeaderErrors && headerInputErrors[key];
              const isDateField = isDateFieldKeyOrLabel(key, label);
              const currentValue = headerInputValues[key] || '';
              const formattedLabel = formatHeaderLabel(label || key);
              const isProjectSite = isProjectOrSiteField(key, label);
              const isOdometer = isOdometerField(key, label);
              
              // Skip rendering date fields entirely; they are auto-filled with today's date
              if (isDateField) {
                return null;
              }

              return (
                <View key={key} style={styles.inputFieldContainer}>
                  <Text style={styles.inputLabel}>
                    {formattedLabel}
                    {!isOdometer && <Text style={styles.requiredText}> *</Text>}
                  </Text>
                  <TextInput
                    style={[
                      styles.inputField,
                      hasError && styles.inputFieldError,
                      isProjectSite && styles.disabledInput
                    ]}
                    value={currentValue}
                    onChangeText={(text) => handleHeaderInputChange(key, text)}
                    placeholder={`Enter ${formattedLabel.toLowerCase()}`}
                    placeholderTextColor="#888"
                    editable={!isProjectSite}
                  />
                  {hasError && (
                    <Text style={styles.errorText}>This field is required</Text>
                  )}
                </View>
              );
            })
          ) : (
            <Text style={styles.emptySectionText}>No header inputs found for this checklist</Text>
          )}
        </View>

        {sections.length > 0 ? (
          sections.map((section) => {
            // Calculate selected items count for this section
            const sectionItems = section.subItems || [];
            const selectedItemsCount = sectionItems.filter(item => selections[item.key]).length;
            const totalItemsCount = sectionItems.length;
            
            return (
              <View key={section.key}>
                <TouchableOpacity
                  style={styles.sectionHeader}
                  onPress={() => toggleSection(section.key)}
                >
                  <View style={styles.sectionHeaderContent}>
                    <AntDesign
                      name={expandedSections[section.key] ? 'caretup' : 'caretdown'}
                      size={20}
                      color={expandedSections[section.key] ? 'red' : 'black'}
                    />
                    <View style={styles.sectionHeaderTextContainer}>
                      <Text style={styles.sectionHeaderText}>{formatSectionTitle(section.name)}</Text>
                      <Text style={styles.sectionItemCount}>
                        ({selectedItemsCount}/{totalItemsCount} selected)
                      </Text>
                    </View>
                  </View>
                </TouchableOpacity>
                {expandedSections[section.key] && (
                  <View style={styles.sectionContent}>
                    {section.subItems.length > 0 ? (
                      section.subItems.map((item, index) => renderItem(item, index))
                    ) : (
                      <Text style={styles.emptySectionText}>No inspection items</Text>
                    )}
                  </View>
                )}
              </View>
            );
          })
        ) : (
          <Text style={styles.emptySectionText}>No sections available</Text>
        )}

        {/* Footer Inputs are intentionally hidden */}
      </ScrollView>

      {/* Date Picker Modal */}
      {showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleDateChange}
          onConfirm={handleDateConfirm}
          onCancel={handleDateCancel}
        />
      )}

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNext}
        >
          <Text style={styles.nextButtonText}>Inspection Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  checklistTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  headingContainer: {
    paddingVertical: 10,
    marginHorizontal: '5%',
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionHeaderTextContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  sectionItemCount: {
    fontSize: 14,
    color: '#666',
    marginLeft: 5,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  requiredText: {
    fontSize: 14,
    color: '#e74c3c',
    fontWeight: '600',
  },
  detailText: {
    fontSize: 14,
    color: '#666',
    marginBottom: 4,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  rejectButton: {
    backgroundColor: '#FF4B2B',
  },
  naButton: {
    backgroundColor: '#2193b0',
  },
  customOptionButton: {
    backgroundColor: '#6c757d',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  numericInput: {
    flex: 1,
    height: 40,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    fontSize: 14,
    marginRight: 10,
  },
  iconsContainer: {
    flexDirection: 'row',
  },
  iconButton: {
    padding: 10,
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    marginTop: 10,
  },
  selectedImage: {
    width: 40,
    height: 40,
    marginTop: 5,
    alignSelf: 'center',
  },
  noteText: {
    fontSize: 14,
    color: '#333',
    marginTop: 8,
    marginBottom: 8,
    fontStyle: 'italic',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  emptySectionText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    padding: 20,
  },
  headerInputsContainer: {
    marginHorizontal: '5%',
    marginBottom: 15,
    padding: 15,
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
  },
  headerInputsHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  headerInputsTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    flex: 1,
    marginRight: 10,
  },
  autoFillButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#E0E0E0',
    paddingVertical: 5,
    paddingHorizontal: 10,
    borderRadius: 5,
  },
  autoFillButtonText: {
    fontSize: 14,
    color: '#0078D4',
    marginLeft: 5,
  },
  inputFieldContainer: {
    marginBottom: 10,
  },
  inputLabel: {
    fontSize: 14,
    color: '#555',
    marginBottom: 5,
  },
  inputField: {
    height: 40,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    fontSize: 14,
    color: '#333',
  },
  disabledInput: {
    backgroundColor: '#E0E0E0',
  },
  inputFieldError: {
    borderColor: '#FF4B2B',
    borderWidth: 1,
  },
  errorText: {
    color: '#FF4B2B',
    fontSize: 12,
    marginTop: 5,
  },
  footerInputsContainer: {
    marginHorizontal: '5%',
    marginTop: 15,
    padding: 15,
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
  },
  footerInputsTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 10,
    textAlign: 'center',
  },
  emptySectionText: {
    fontSize: 14,
    color: '#888',
    textAlign: 'center',
    paddingVertical: 20,
    fontStyle: 'italic',
  },
  dateInputField: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    height: 40,
    borderWidth: 1,
    borderColor: '#ccc',
    borderRadius: 5,
    paddingHorizontal: 10,
    fontSize: 14,
    color: '#333',
  },
  dateInputText: {
    flex: 1,
    fontSize: 14,
    color: '#333',
  },
  datePlaceholderText: {
    color: '#888',
  },
});

export default Doinsstep2;