import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Platform,
  Alert,
  StatusBar,
  Dimensions,
  Modal,
  PermissionsAndroid,
  BackHandler,
  ActivityIndicator,
} from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation, useRoute, useFocusEffect } from '@react-navigation/native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import LinearGradient from 'react-native-linear-gradient';
import VoiceToText, { VoiceToTextEvents } from '@appcitor/react-native-voice-to-text';
import LottieView from 'lottie-react-native';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.24;
// test
const DoinspecNoteadd = () => {
  const [note, setNote] = useState('');
  const navigation = useNavigation();
  const route = useRoute();
  const { itemKey, inspectionId } = route.params || {};

  const [isListening, setIsListening] = useState(false);
  const [error, setError] = useState(null);
  const [micPermissionGranted, setMicPermissionGranted] = useState(false);
  const [isVoiceModuleAvailable, setIsVoiceModuleAvailable] = useState(false);
  const [isRecognitionAvailable, setIsRecognitionAvailable] = useState(true);
  const [modalVisible, setModalVisible] = useState(false);
  const [partialText, setPartialText] = useState('');
  const [volume, setVolume] = useState(0);
  const [ignoreVoiceEvents, setIgnoreVoiceEvents] = useState(false);
  const [isStopping, setIsStopping] = useState(false);
  const lottieRef = useRef(null);
  const isMountedRef = useRef(true);
  const listenersRef = useRef([]);
  const isCleaningUpRef = useRef(false);
  const isListeningRef = useRef(false);
  const cleanupTimeoutRef = useRef(null);
  const stopFallbackTimeoutRef = useRef(null);
  const pendingStopShouldAddRef = useRef(false);
  const hasStoppedRecordingRef = useRef(false);
  const hasReceivedFinalResultsRef = useRef(false);
  const recordingStartTimeRef = useRef(null);
  const isRestartingRef = useRef(false);
  const lastFinalTextRef = useRef('');

  useEffect(() => {
    loadNote();
  }, [inspectionId, itemKey]);

  const loadNote = async () => {
    try {
      const savedNotes = JSON.parse(
        await AsyncStorage.getItem(`inspectionNotes_${inspectionId}`) || '{}'
      );
      if (savedNotes[itemKey]) {
        const existingNote = savedNotes[itemKey] || '';
        setNote(existingNote);
      }
    } catch (error) {
      console.error('Error loading note:', error);
    }
  };

  const saveNote = async () => {
    try {
      const savedNotes = JSON.parse(
        await AsyncStorage.getItem(`inspectionNotes_${inspectionId}`) || '{}'
      );
      savedNotes[itemKey] = note;
      await AsyncStorage.setItem(
        `inspectionNotes_${inspectionId}`,
        JSON.stringify(savedNotes)
      );
      navigation.goBack();
    } catch (error) {
      console.error('Error saving note:', error);
      Alert.alert('Error', 'Failed to save note.');
    }
  };

  const appendVoiceTextToNote = useCallback((incomingText) => {
    const trimmedFinal = (incomingText || '').trim();
    if (!trimmedFinal) {
      return;
    }

    setNote((prevValue) => {
      const currentValue = typeof prevValue === 'string' ? prevValue : '';
      const currentTrimmed = currentValue.trim();

      if (!currentTrimmed) {
        lastFinalTextRef.current = trimmedFinal;
        return trimmedFinal;
      }

      if (lastFinalTextRef.current === trimmedFinal) {
        return currentValue;
      }

      if (currentTrimmed.endsWith(trimmedFinal)) {
        return currentValue;
      }

      const lastFewWords = currentTrimmed.split(' ').slice(-3).join(' ');
      const firstFewWords = trimmedFinal.split(' ').slice(0, 3).join(' ');
      if (lastFewWords && firstFewWords && lastFewWords === firstFewWords && lastFewWords.length > 5) {
        return currentValue;
      }

      if (trimmedFinal.length > 15 && currentTrimmed.includes(trimmedFinal)) {
        return currentValue;
      }

      const updated = currentTrimmed ? `${currentTrimmed} ${trimmedFinal}` : trimmedFinal;
      lastFinalTextRef.current = trimmedFinal;
      return updated;
    });
  }, []);

  const onSpeechResults = (event) => {
    if (!isMountedRef.current || isCleaningUpRef.current || ignoreVoiceEvents) {
      return;
    }

    hasReceivedFinalResultsRef.current = true;

    try {
      let payload = event?.value ?? event?.results ?? event?.text ?? event;
      let finalText = '';

      if (Array.isArray(payload)) {
        if (payload.length) {
          finalText = payload.join(' ');
        }
      } else if (typeof payload === 'string') {
        finalText = payload;
      } else if (payload && typeof payload === 'object') {
        const extracted = Object.values(payload)
          .flatMap((v) => (Array.isArray(v) ? v : [v]))
          .filter((v) => typeof v === 'string');
        if (extracted.length) {
          finalText = extracted.join(' ');
        } else {
          finalText = String(payload ?? '');
        }
      } else {
        finalText = String(payload ?? '');
      }

      if (finalText && finalText.trim()) {
        const trimmedFinal = finalText.trim();

        if (lastFinalTextRef.current === trimmedFinal) {
          setPartialText('');
          return;
        }

        appendVoiceTextToNote(trimmedFinal);
        setPartialText('');
      }
    } catch (err) {
      console.warn('onSpeechResults parse error', err, event);
    }
  };

  const onSpeechError = (event) => {
    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }

    const message = event?.error?.message || event?.message || JSON.stringify(event);
    const lowerMessage = String(message).toLowerCase();
    const code = String(event?.error?.code ?? event?.code ?? '');
    const isSilenceError =
      lowerMessage.includes('no match') ||
      lowerMessage.includes('timeout') ||
      lowerMessage.includes('11') ||
      code.includes('11');

    if (isSilenceError) {
      setError(null);
      setIgnoreVoiceEvents(false);

      if (
        isMountedRef.current &&
        !isCleaningUpRef.current &&
        !hasStoppedRecordingRef.current &&
        isListeningRef.current &&
        !isRestartingRef.current
      ) {
        isRestartingRef.current = true;
        setTimeout(restartListening, 200);
      }
      return;
    }

    setError(`Speech recognition error: ${message}`);
    isListeningRef.current = false;
    setIsListening(false);
    setModalVisible(false);
  };

  const checkMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.RECORD_AUDIO);
      setMicPermissionGranted(granted);
      return granted;
    } catch (err) {
      console.warn('Permission check error:', err);
      return false;
    }
  };

  const requestMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.RECORD_AUDIO,
        {
          title: 'Microphone Permission',
          message: 'The app needs microphone access to capture your note by voice.',
          buttonPositive: 'OK',
        }
      );
      const isGranted = granted === PermissionsAndroid.RESULTS.GRANTED;
      setMicPermissionGranted(isGranted);
      return isGranted;
    } catch (err) {
      console.warn('Permission request error:', err);
      return false;
    }
  };

  const restartListening = useCallback(async () => {
    if (
      !VoiceToText ||
      !isMountedRef.current ||
      isCleaningUpRef.current ||
      hasStoppedRecordingRef.current
    ) {
      isRestartingRef.current = false;
      return;
    }

    try {
      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      }
    } catch (restartError) {
      console.warn('Failed to auto-restart recording:', restartError);
    } finally {
      isRestartingRef.current = false;
    }
  }, []);

  const startRecording = async () => {
    if (!isVoiceModuleAvailable) {
      setError('Speech engine unavailable. Please reinstall or relink the voice module.');
      return;
    }

    setError(null);

    if (Platform.OS === 'android' && !micPermissionGranted) {
      const granted = await requestMicrophonePermission();
      if (!granted) {
        setError('Microphone permission denied. Enable it in System Settings to use voice notes.');
        return;
      }
    }

    try {
      if (Platform.OS === 'ios' && !isRecognitionAvailable) {
        setError('Speech recognition not available on this iOS device.');
        return;
      }

      if (typeof VoiceToText.setRecognitionLanguage === 'function') {
        await VoiceToText.setRecognitionLanguage('en-US');
      }

      hasStoppedRecordingRef.current = false;
      hasReceivedFinalResultsRef.current = false;
      isRestartingRef.current = false;
      isListeningRef.current = true;
      setIgnoreVoiceEvents(false);
      setIsStopping(false);
      pendingStopShouldAddRef.current = false;
      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }
      setPartialText('');
      recordingStartTimeRef.current = Date.now();
      lastFinalTextRef.current = '';

      setModalVisible(true);
      setIsListening(true);

      if (lottieRef.current) {
        lottieRef.current.play();
      }

      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      } else {
        throw new Error('startListening not available on VoiceToText');
      }
    } catch (e) {
      const message = e?.message || JSON.stringify(e);
      setError(`Failed to start voice note: ${message}`);
      setIsListening(false);
      setModalVisible(false);
      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }
    }
  };

  const finalizeStop = useCallback(
    (shouldAddPartialText = false) => {
      setIgnoreVoiceEvents(true);
      isListeningRef.current = false;
      hasStoppedRecordingRef.current = true;
      pendingStopShouldAddRef.current = false;

      if (!isMountedRef.current || isCleaningUpRef.current) {
        return;
      }

      if (shouldAddPartialText && !hasReceivedFinalResultsRef.current) {
        setPartialText((currentPartial) => {
          if (currentPartial && currentPartial.trim()) {
            appendVoiceTextToNote(currentPartial);
          }
          return '';
        });
      } else {
        setPartialText('');
      }

      setIsListening(false);
      setModalVisible(false);
      setIsStopping(false);

      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }
    },
    [appendVoiceTextToNote]
  );

  const safeStopRecording = useCallback(
    ({ shouldAddPartialText = false, showProcessing = false } = {}) => {
      pendingStopShouldAddRef.current = shouldAddPartialText;

      if (hasStoppedRecordingRef.current) {
        isListeningRef.current = false;
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          setIsStopping(true);
        }
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 200 : 0);
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          if (stopFallbackTimeoutRef.current) {
            clearTimeout(stopFallbackTimeoutRef.current);
          }
          stopFallbackTimeoutRef.current = setTimeout(() => {
            finalizeStop(pendingStopShouldAddRef.current);
          }, 3000);
        }
        return;
      }

      if (!isListeningRef.current) {
        isListeningRef.current = false;
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 200 : 0);
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          if (stopFallbackTimeoutRef.current) {
            clearTimeout(stopFallbackTimeoutRef.current);
          }
          stopFallbackTimeoutRef.current = setTimeout(() => {
            finalizeStop(pendingStopShouldAddRef.current);
          }, 3000);
        }
        return;
      }

      if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
        setIsStopping(true);
        if (stopFallbackTimeoutRef.current) {
          clearTimeout(stopFallbackTimeoutRef.current);
        }
        stopFallbackTimeoutRef.current = setTimeout(() => {
          finalizeStop(pendingStopShouldAddRef.current);
        }, 3000);
      }

      try {
        hasStoppedRecordingRef.current = true;
        isListeningRef.current = false;

        if (VoiceToText && isVoiceModuleAvailable) {
          let stopFunc = null;
          if (typeof VoiceToText.stopListening === 'function') {
            stopFunc = VoiceToText.stopListening;
          } else if (typeof VoiceToText.stop === 'function') {
            stopFunc = VoiceToText.stop;
          } else if (typeof VoiceToText.cancelListening === 'function') {
            stopFunc = VoiceToText.cancelListening;
          } else if (typeof VoiceToText.cancel === 'function') {
            stopFunc = VoiceToText.cancel;
          }

          if (stopFunc) {
            try {
              const result = stopFunc();
              if (result && typeof result.then === 'function') {
                result.catch((stopError) => console.warn('⚠️ Error stopping recording:', stopError));
              }
            } catch (stopError) {
              console.warn('⚠️ Error stopping recording:', stopError);
            }
          }
        }

        if (lottieRef.current) {
          lottieRef.current.pause();
          lottieRef.current.reset();
        }

        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }

        const delay = showProcessing ? 550 : 400;
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, delay);
      } catch (e) {
        console.warn('⚠️ Error in safeStopRecording:', e);
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 300 : 0);
      }
    },
    [finalizeStop, isVoiceModuleAvailable]
  );

  const stopRecording = async () => {
    if (isStopping) {
      return;
    }
    await safeStopRecording({ showProcessing: true });
  };

  const cancelRecording = async () => {
    if (isStopping) {
      return;
    }

    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }

    try {
      await safeStopRecording({ shouldAddPartialText: false, showProcessing: true });
    } catch (e) {
      if (isMountedRef.current && !isCleaningUpRef.current) {
        setError(`Failed to cancel recording: ${e.message || JSON.stringify(e)}`);
      }
    }
  };

  const onDone = async () => {
    if (isStopping) {
      return;
    }

    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }

    try {
      await safeStopRecording({ shouldAddPartialText: true, showProcessing: true });
    } catch (err) {
      console.warn('onDone stop error', err);
    }
  };

  useEffect(() => {
    if (!VoiceToText || typeof VoiceToText.addEventListener !== 'function') {
      const msg =
        'VoiceToText module is unavailable. Ensure @appcitor/react-native-voice-to-text is installed and linked correctly.';
      console.error(msg);
      setError(msg);
      return;
    }

    setIsVoiceModuleAvailable(true);

    const resultsListener = VoiceToText.addEventListener(VoiceToTextEvents.RESULTS, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEvents) {
        onSpeechResults(event);
      }
    });
    const partialListener = VoiceToText.addEventListener(
      VoiceToTextEvents.PARTIAL_RESULTS,
      (event) => {
        if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEvents) {
          if (isListeningRef.current) {
            setPartialText(event?.value || '');
          }
        }
      }
    );
    const volumeListener = VoiceToText.addEventListener(VoiceToTextEvents.VOLUME_CHANGED, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEvents) {
        setVolume(event?.value || 0);
      }
    });
    const errorListener = VoiceToText.addEventListener(VoiceToTextEvents.ERROR, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEvents) {
        onSpeechError(event);
      }
    });
    const startListener = VoiceToText.addEventListener(VoiceToTextEvents.START, () => {
      if (isMountedRef.current && !isCleaningUpRef.current) {
        isListeningRef.current = true;
        hasStoppedRecordingRef.current = false;
        hasReceivedFinalResultsRef.current = false;
        isRestartingRef.current = false;
        recordingStartTimeRef.current = Date.now();
        setIsListening(true);
        setModalVisible(true);
        setPartialText('');
        setIgnoreVoiceEvents(false);
        if (lottieRef.current) {
          lottieRef.current.play();
        }
      }
    });
    const endListener = VoiceToText.addEventListener(VoiceToTextEvents.END, () => {
      if (
        isMountedRef.current &&
        !isCleaningUpRef.current &&
        !hasStoppedRecordingRef.current &&
        isListeningRef.current &&
        !isRestartingRef.current
      ) {
        isRestartingRef.current = true;
        setTimeout(async () => {
          if (
            isMountedRef.current &&
            !isCleaningUpRef.current &&
            !hasStoppedRecordingRef.current &&
            isListeningRef.current
          ) {
            await restartListening();
          } else {
            isRestartingRef.current = false;
          }
        }, 200);
      }
    });

    listenersRef.current = [
      resultsListener,
      partialListener,
      volumeListener,
      errorListener,
      startListener,
      endListener,
    ];

    (async () => {
      try {
        if (typeof VoiceToText.isRecognitionAvailable === 'function') {
          const avail = await VoiceToText.isRecognitionAvailable();
          setIsRecognitionAvailable(avail === true);
          if (!avail && Platform.OS === 'ios') {
            setError(
              'Speech recognition not available on this device. Test on a real device and verify Info.plist permissions.'
            );
          }
        }
      } catch (err) {
        console.warn('isRecognitionAvailable check failed', err);
      }
    })();

    const checkPermissions = async () => {
      if (Platform.OS === 'android') {
        try {
          const granted = await checkMicrophonePermission();
          if (!granted) {
            const requested = await requestMicrophonePermission();
            if (!requested) {
              setError('Microphone permission denied. Please enable it in app settings.');
            }
          }
        } catch (err) {
          console.warn('Permission check error:', err);
          setError('Failed to check microphone permissions.');
        }
      }
    };

    checkPermissions();

    return () => {
      isMountedRef.current = false;
      isCleaningUpRef.current = true;

      if (cleanupTimeoutRef.current) {
        clearTimeout(cleanupTimeoutRef.current);
      }
      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }

      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }

      const stopRecordingImmediately = async () => {
        try {
          if (isListeningRef.current && VoiceToText && !hasStoppedRecordingRef.current) {
            hasStoppedRecordingRef.current = true;

            const stopMethods = [
              () => VoiceToText.stopListening && VoiceToText.stopListening(),
              () => VoiceToText.stop && VoiceToText.stop(),
              () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
              () => VoiceToText.cancel && VoiceToText.cancel(),
              () => VoiceToText.destroy && VoiceToText.destroy(),
            ];

            for (const stopMethod of stopMethods) {
              try {
                if (typeof stopMethod === 'function') {
                  await stopMethod();
                  break;
                }
              } catch (e) {
                // continue
              }
            }
          }
        } catch (e) {
          console.warn('Error during cleanup (non-fatal):', e.message);
        }
      };

      stopRecordingImmediately();
      listenersRef.current = [];
    };
  }, [appendVoiceTextToNote, restartListening]);

  useFocusEffect(
    useCallback(() => {
      hasStoppedRecordingRef.current = false;
      isCleaningUpRef.current = false;

      return () => {
        safeStopRecording();
      };
    }, [safeStopRecording])
  );

  useEffect(() => {
    const backHandler = BackHandler.addEventListener('hardwareBackPress', () => {
      if (isListeningRef.current) {
        safeStopRecording();
        return true;
      }
      return false;
    });

    return () => backHandler.remove();
  }, [safeStopRecording]);

  const shouldShowError =
    error && !String(error).toLowerCase().includes('unknown error: 11');

  useEffect(() => {
    const unsubscribe = navigation.addListener('beforeRemove', () => {
      if (isListeningRef.current) {
        safeStopRecording();
      }
    });

    return unsubscribe;
  }, [navigation, safeStopRecording]);

  useEffect(() => {
    if (modalVisible && isListening && lottieRef.current) {
      const timer = setTimeout(() => {
        if (lottieRef.current) {
          lottieRef.current.play();
        }
      }, 100);
      return () => clearTimeout(timer);
    } else if (!isListening && lottieRef.current) {
      lottieRef.current.pause();
      lottieRef.current.reset();
    }
  }, [modalVisible, isListening]);

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity
              style={styles.backButton}
              onPress={() => navigation.goBack()}
              activeOpacity={0.8}
            >
              <MaterialIcons name="arrow-back" size={24} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Add Note</Text>
          </View>
          <View style={styles.headerRight} />
        </View>
      </LinearGradient>
      <Text style={styles.status}>Status: {isListening ? 'Listening...' : 'Stopped'}</Text>
      {shouldShowError ? <Text style={styles.errorText}>Error: {error}</Text> : null}

      <LinearGradient colors={['#3481BC', '#025383']} style={styles.noteContainer}>
        <Text style={styles.noteLabel}>Note</Text>
        <View style={styles.inputContainer}>
          <TextInput
            style={styles.noteInput}
            value={note}
            onChangeText={(text) => {
              setNote(text);
            }}
            placeholder="Enter your note here"
            placeholderTextColor="#e0e0e0"
            multiline
            numberOfLines={5}
          />
          <TouchableOpacity
            style={styles.micButton}
            onPress={isListening ? stopRecording : startRecording}
            disabled={isListening || !isVoiceModuleAvailable}
          >
            <MaterialIcons
              name={isListening ? 'mic-off' : 'mic'}
              size={24}
              color={isListening ? '#FF4B2B' : '#035484'}
            />
          </TouchableOpacity>
        </View>
      </LinearGradient>
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => navigation.goBack()}>
          <Text style={styles.buttonText}>CANCEL</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.doneButton} onPress={saveNote}>
          <Text style={styles.doneButtonText}>DONE</Text>
        </TouchableOpacity>
      </View>

      <Modal
        visible={modalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => {
          if (isListening) {
            cancelRecording();
          }
        }}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            {isStopping ? (
              <View style={styles.processingContainer}>
                <ActivityIndicator size="large" color="#015185" />
                <Text style={styles.processingText}>Finishing up...</Text>
              </View>
            ) : (
              <>
                <View style={styles.lottieContainer}>
                  <LottieView
                    ref={lottieRef}
                    source={require('../../../Images/micanimation.json')}
                    autoPlay={false}
                    loop={true}
                    style={styles.lottieAnimation}
                    resizeMode="contain"
                  />
                </View>

                <Text style={styles.partial}>
                  {partialText
                    ? partialText
                    : isListening
                    ? 'Listening... Speak now'
                    : 'Preparing...'}
                </Text>

                {volume > 0 && (
                  <View style={styles.volumeIndicator}>
                    <View style={[styles.volumeBar, { width: `${Math.min(volume * 100, 100)}%` }]} />
                  </View>
                )}

                <View style={styles.modalButtons}>
                  <TouchableOpacity
                    style={[styles.modalButton, styles.cancelBtn]}
                    onPress={cancelRecording}
                    activeOpacity={0.7}
                    disabled={isStopping}
                  >
                    <Text style={[styles.modalButtonText, { color: '#015185' }]}>Cancel</Text>
                  </TouchableOpacity>
                  <TouchableOpacity
                    style={styles.modalButton}
                    onPress={onDone}
                    activeOpacity={0.7}
                    disabled={isStopping}
                  >
                    <Text style={styles.modalButtonText}>Done</Text>
                  </TouchableOpacity>
                </View>
              </>
            )}
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f0f0f0',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? width * 0.08 : width * 0.04,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
    width: width * 0.07,
    height: width * 0.07,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  noteContainer: {
    width: '94%',
    alignSelf: 'center',
    paddingHorizontal: 15,
    paddingVertical: 10,
    borderRadius: 12,
    marginTop: 20,
    marginBottom: 12,
    backgroundColor: '#3481BC',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  status: {
    marginTop: 16,
    marginHorizontal: 16,
    color: '#666',
    fontSize: 14,
  },
  errorText: {
    marginHorizontal: 16,
    marginTop: 4,
    color: '#d32f2f',
    fontSize: 14,
    backgroundColor: '#ffebee',
    padding: 8,
    borderRadius: 6,
  },
  noteLabel: {
    fontSize: 16,
    color: '#fff',
    marginBottom: 5,
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    position: 'relative',
  },
  noteInput: {
    flex: 1,
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 12,
    paddingRight: 40,
    fontSize: 16,
    minHeight: 100,
    textAlignVertical: 'top',
    color: '#222',
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  micButton: {
    position: 'absolute',
    right: 10,
    bottom: 10,
    padding: 5,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    paddingVertical: 10,
  },
  cancelButton: {
    paddingVertical: 10,
    paddingHorizontal: 28,
    marginHorizontal: 10,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#035484',
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 2,
  },
  doneButton: {
    paddingVertical: 10,
    paddingHorizontal: 20,
    marginHorizontal: 10,
    backgroundColor: '#035484',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 2,
  },
  buttonText: {
    fontSize: 16,
    color: '#035484',
    fontWeight: 'bold',
  },
  doneButtonText: {
    fontSize: 16,
    color: '#fff',
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.7)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '85%',
    maxWidth: 400,
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
    elevation: 8,
  },
  lottieContainer: {
    width: 200,
    height: 200,
    justifyContent: 'center',
    alignItems: 'center',
    marginBottom: 20,
  },
  lottieAnimation: {
    width: '100%',
    height: '100%',
  },
  partial: {
    minHeight: 50,
    textAlign: 'center',
    marginBottom: 20,
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
    paddingHorizontal: 16,
  },
  modalButtons: {
    flexDirection: 'row',
    width: '100%',
    justifyContent: 'space-between',
    gap: 12,
  },
  modalButton: {
    flex: 1,
    paddingVertical: 12,
    backgroundColor: '#015185',
    borderRadius: 8,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  cancelBtn: {
    backgroundColor: '#fff',
    borderWidth: 2,
    borderColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 16,
  },
  processingContainer: {
    width: '100%',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 24,
  },
  processingText: {
    marginTop: 20,
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    textAlign: 'center',
  },
  volumeIndicator: {
    width: '80%',
    height: 4,
    backgroundColor: '#e0e0e0',
    borderRadius: 2,
    marginBottom: 16,
    overflow: 'hidden',
  },
  volumeBar: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 2,
  },
  backButton: {
    padding: 5,
  },

});

export default DoinspecNoteadd;