import React, { useRef, useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  Dimensions,
  Alert,
  Modal,
  Animated,
  Image,
} from 'react-native';
import SignatureCanvas from 'react-native-signature-canvas';
import LinearGradient from 'react-native-linear-gradient';
import AsyncStorage from '@react-native-async-storage/async-storage';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';

const { height, width } = Dimensions.get('window');

const SyncModal = ({ visible }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.modalOverlay}>
        <View style={styles.modalContent}>
          <Text style={styles.modalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.modalTextadd}>Saving Signature</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const OperatorSignaturedo = ({ onSave, onBack, setHasUnsavedSignature, report }) => {
  const signatureRef = useRef(null);
  const [signatureData, setSignatureData] = useState(null);
  const [showSyncModal, setShowSyncModal] = useState(false);

  useEffect(() => {
    if (!onSave || typeof onSave !== 'function') {
      console.error('OperatorSignaturedo: onSave prop is missing or not a function');
      Alert.alert('Error', 'Unable to save the signature due to a configuration issue. Please contact support.');
    }
    if (!onBack || typeof onBack !== 'function') {
      console.error('OperatorSignaturedo: onBack prop is missing or not a function');
      Alert.alert('Error', 'Navigation is not configured properly. Please try again or contact support.');
    }
    if (!setHasUnsavedSignature || typeof setHasUnsavedSignature !== 'function') {
      console.error('OperatorSignaturedo: setHasUnsavedSignature prop is missing or not a function');
      Alert.alert('Warning', 'Signature change tracking is not available. You can still proceed.');
    }
    if (!report?.id) {
      console.warn('OperatorSignaturedo: No report ID provided');
      Alert.alert('Warning', 'Report ID is missing. Signature will not be saved to storage.');
    }
  }, [onSave, onBack, setHasUnsavedSignature, report]);

  useEffect(() => {
    const loadSignature = async () => {
      if (!report?.id) {
        console.warn('OperatorSignaturedo: No report ID provided for loading signature');
        return;
      }
      try {
        const storedData = await AsyncStorage.getItem('inspectionReports');
        if (!storedData) {
          console.log('OperatorSignaturedo: No stored inspection reports found');
          return;
        }
        const reports = JSON.parse(storedData);
        if (!Array.isArray(reports)) {
          console.warn('OperatorSignaturedo: Invalid stored reports format');
          return;
        }
        const currentReport = reports.find((r) => r.id === report.id);
        if (currentReport?.correctiveAction?.operatorSignatureUri) {
          console.log('Loaded operator signature from AsyncStorage:', currentReport.correctiveAction.operatorSignatureUri);
          setSignatureData(currentReport.correctiveAction.operatorSignatureUri);
          if (setHasUnsavedSignature) {
            setHasUnsavedSignature(false);
          }
        }
      } catch (error) {
        console.error('OperatorSignaturedo: Error loading operator signature:', error);
        Alert.alert('Warning', 'Failed to load existing signature. You can still proceed with a new signature.');
      }
    };
    loadSignature();
  }, [report, setHasUnsavedSignature]);

  const saveSignatureToAsyncStorage = async (processedSignature) => {
    if (!report?.id) {
      console.warn('OperatorSignaturedo: No report ID, skipping AsyncStorage save');
      return;
    }

    try {
      let storedReports = [];
      const storedData = await AsyncStorage.getItem('inspectionReports');
      storedReports = storedData ? JSON.parse(storedData) : [];
      if (!Array.isArray(storedReports)) {
        console.warn('OperatorSignaturedo: Invalid stored reports format, resetting to empty array');
        storedReports = [];
      }

      const reportExists = storedReports.some((r) => r.id === report.id);
      let updatedReports;
      if (reportExists) {
        updatedReports = storedReports.map((r) =>
          r.id === report.id
            ? {
                ...r,
                correctiveAction: {
                  ...r.correctiveAction,
                  operatorSignatureUri: processedSignature,
                },
              }
            : r
        );
      } else {
        updatedReports = [
          ...storedReports,
          {
            ...report,
            correctiveAction: {
              ...report.correctiveAction,
              operatorSignatureUri: processedSignature,
            },
          },
        ];
      }

      await AsyncStorage.setItem('inspectionReports', JSON.stringify(updatedReports));
      console.log('Operator signature saved to AsyncStorage:', processedSignature);
    } catch (error) {
      console.error('OperatorSignaturedo: Error saving operator signature to AsyncStorage:', error);
      Alert.alert('Error', 'Failed to save signature to storage. The signature is still saved for this session.');
    }
  };

  const handleClear = () => {
    if (!signatureRef.current) {
      Alert.alert('Error', 'Signature canvas is not ready. Please try again.');
      return;
    }
    signatureRef.current.clearSignature();
    setSignatureData(null);
    if (setHasUnsavedSignature) {
      setHasUnsavedSignature(false);
    }
    console.log('Operator signature cleared');
  };

  const handleSave = () => {
    if (!signatureRef.current) {
      Alert.alert('Error', 'Signature canvas is not initialized. Please try again.');
      return;
    }
    setShowSyncModal(true);
    signatureRef.current.readSignature();
  };

  const handleOK = (signature) => {
    if (!signature) {
      setShowSyncModal(false);
      Alert.alert('Warning', 'No signature detected. Please sign before saving.');
      console.log('OperatorSignaturedo: No operator signature detected');
      return;
    }

    console.log('Operator signature captured:', signature);
    const processedSignature = signature.startsWith('data:image')
      ? signature
      : `data:image/png;base64,${signature}`;
    setSignatureData(processedSignature);

    if (!onSave || typeof onSave !== 'function') {
      setShowSyncModal(false);
      Alert.alert('Error', 'Unable to save the signature due to a configuration issue.');
      console.error('onSave is not a function');
      return;
    }

    console.log('Saving operator signature data:', processedSignature);
    onSave(processedSignature); // Pass string instead of object

    saveSignatureToAsyncStorage(processedSignature);

    if (signatureRef.current) {
      signatureRef.current.clearSignature();
      setSignatureData(null);
    }

    if (setHasUnsavedSignature) {
      setHasUnsavedSignature(false);
    }

    setShowSyncModal(false);

    if (typeof onBack === 'function') {
      onBack();
      console.log('Navigating back after save');
    } else {
      console.log('onBack is not a function, no navigation performed');
    }
  };

  return (
    <View style={styles.container}>
      <View style={styles.content}>
        <Text style={styles.label}>Operator Signature</Text>
        <View style={styles.signatureContainer}>
          <SignatureCanvas
            ref={signatureRef}
            onOK={handleOK}
            descriptionText="Sign here"
            backgroundColor="#F5F5F5"
            penColor="#000000"
            canvasProps={{ style: styles.signatureCanvas }}
          />
        </View>

        <View style={styles.buttonContainer}>
          <TouchableOpacity onPress={handleClear} style={styles.clearButton}>
            <Text style={styles.clearButtonText}>Clear</Text>
          </TouchableOpacity>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.saveButton}>
            <TouchableOpacity
              onPress={handleSave}
              style={styles.saveButtonTouchable}
            >
              <Text style={styles.saveButtonText}>Save</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      </View>

      <SyncModal visible={showSyncModal} />
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  content: { flex: 1, alignItems: 'center', padding: 20 },
  label: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 20,
    alignSelf: 'flex-start',
  },
  signatureContainer: {
    width: width * 0.9,
    height: height / 5,
    backgroundColor: '#F5F5F5',
    borderRadius: 10,
    overflow: 'hidden',
    marginBottom: 30,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  signatureCanvas: { width: '100%', height: '100%' },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: width * 0.9,
  },
  clearButton: {
    width: '48%',
    padding: 12,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#025383',
    alignItems: 'center',
    justifyContent: 'center',
  },
  clearButtonText: { color: '#025383', fontSize: 16, fontWeight: '600' },
  saveButton: { width: '48%', borderRadius: 5, overflow: 'hidden' },
  saveButtonTouchable: { padding: 12, alignItems: 'center', justifyContent: 'center' },
  saveButtonText: { color: '#fff', fontSize: 16, fontWeight: '600' },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  modalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  modalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default OperatorSignaturedo;