import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  FlatList,
  ActivityIndicator,
  ScrollView,
  Alert,
  Modal,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const InspectionListView = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const type = route.params?.type === 'hse' ? 'hse' : 'inspection';
  const [items, setItems] = useState([]);
  const [loading, setLoading] = useState(true);
  const isMounted = useRef(true);
  const [selectionMode, setSelectionMode] = useState(false);
  const [selectedIds, setSelectedIds] = useState([]);
  const [filteredItems, setFilteredItems] = useState([]);
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [availableCountries, setAvailableCountries] = useState([]);
  const [availableProjects, setAvailableProjects] = useState([]);
  const [availableInspectors, setAvailableInspectors] = useState([]);
  const [availableCreators, setAvailableCreators] = useState([]);
  const [adminIdToName, setAdminIdToName] = useState({});
  const [activeFilters, setActiveFilters] = useState({
    dateRange: 'all', countries: [], projects: [], inspectors: [], creators: []
  });
  const [tempFilters, setTempFilters] = useState(() => ({ ...activeFilters, countries: [...activeFilters.countries], projects: [...activeFilters.projects], inspectors: [...activeFilters.inspectors], creators: [...activeFilters.creators] }));
  const [actionLoadingId, setActionLoadingId] = useState(null);

  const CACHE_KEYS = {
    inspection: 'inspection_reports_cache_v1',
    hse: 'hseInspectionsCache',
  };

  const isReportEmpty = (r) => {
    if (!r) return true;
    // consider non-empty if any meaningful field exists
    const hasStep1 = !!(r.step1 && (r.step1.requestNumber || r.step1.country || r.step1.project || r.step1.inspector || (r.step1.selectedEquipment && r.step1.selectedEquipment.equipmentName)));
    const hasStep3Pdf = !!(r.step3 && (r.step3.pdfDownloadUrl || r.step3.pdfPath));
    const hasWorkOrderPdf = !!(r.workOrder && r.workOrder.pdfDownloadUrl);
    const hasCorrectivePdf = !!(r.correctiveAction && (r.correctiveAction.pdfDownloadUrl || r.correctiveAction.pdfPath));
    const hasStep2 = !!(r.step2 && (Array.isArray(r.step2.sections) ? r.step2.sections.length > 0 : Object.keys(r.step2 || {}).length > 0));
    return !(hasStep1 || hasStep3Pdf || hasWorkOrderPdf || hasCorrectivePdf || hasStep2);
  };

  const showToast = (type, t1, t2) => {
    Toast.show({ type, text1: t1, text2: t2, position: 'bottom', visibilityTime: 2500 });
  };

  const checkNetwork = async () => {
    try {
      const s = await NetInfo.fetch();
      return s.isConnected && s.isInternetReachable;
    } catch (e) { return false; }
  };

  const loadFromCache = async () => {
    try {
      const raw = await AsyncStorage.getItem(CACHE_KEYS[type]);
      if (!raw) return [];
      const parsed = JSON.parse(raw);
      return Array.isArray(parsed) ? parsed : [];
    } catch (e) {
      console.warn('Failed to read cache', e);
      return [];
    }
  };

  const saveToCache = async (arr) => {
    try {
      await AsyncStorage.setItem(CACHE_KEYS[type], JSON.stringify(arr.slice(0, 2000)));
    } catch (e) {
      console.warn('Failed to save cache', e);
    }
  };

  // Helper to obtain user's countries/projects permissions (tries AsyncStorage first, then Firebase)
  const getUserPermissions = async () => {
    try {
      const cachedUserInfo = await AsyncStorage.getItem('userInfo');
      if (cachedUserInfo) {
        const ui = JSON.parse(cachedUserInfo);
        return { countries: ui.countries || [], projects: ui.projects || [], userId: ui.userUid || null };
      }

      // fallback to userUid or userEmail
      let userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        const userEmail = await AsyncStorage.getItem('userEmail');
        if (userEmail) {
          const snap = await database().ref('/Globalusers/admins').orderByChild('email').equalTo(userEmail).once('value');
          const val = snap.val();
          if (val) userId = Object.keys(val)[0];
        }
      }

      if (userId) {
        const snap = await database().ref(`/Globalusers/admins/${userId}`).once('value');
        const val = snap.val() || {};
        return { countries: val.countries || [], projects: val.projects || [], userId };
      }
    } catch (err) {
      console.warn('Failed to load user permissions', err);
    }
    return { countries: [], projects: [], userId: null };
  };

  const fetchOnce = async () => {
    setLoading(true);
    try {
      const online = await checkNetwork();
      const perms = await getUserPermissions();

      if (!online) {
        const cached = await loadFromCache();
        let filteredCached = Array.isArray(cached) ? cached.filter(r => !isReportEmpty(r)) : [];
        // apply permission filtering when available
        if (perms.countries.length || perms.projects.length) {
          filteredCached = filteredCached.filter(r => {
            const country = r.step1?.country || r.country || '';
            const project = r.step1?.project || r.project || '';
            const okCountry = perms.countries.length ? perms.countries.includes(country) : true;
            const okProject = perms.projects.length ? perms.projects.includes(project) : true;
            return okCountry && okProject;
          });
        }
        if (isMounted.current) setItems(filteredCached);
        showToast('info', 'Offline', 'Showing cached data');
        return;
      }

      if (type === 'hse') {
        const snap = await database().ref('/GlobalHSEInspections/admins').once('value');
        const data = snap.val() || {};
        let arr = [];
        Object.entries(data).forEach(([adminId, reports]) => {
          if (!reports) return;
          Object.entries(reports).forEach(([id, r]) => {
            if (id === 'Reportidnum') return;
            arr.push({ id, adminId, ...r, createdAt: r.createdAt || r.step3?.createdAt });
          });
        });

        // remove empty placeholder reports
        arr = arr.filter(r => !isReportEmpty(r));

        // apply permission filtering if user has countries/projects
        if (perms.countries.length || perms.projects.length) {
          arr = arr.filter(r => {
            const country = r.step1?.country || r.country || '';
            const project = r.step1?.project || r.project || '';
            const okCountry = perms.countries.length ? perms.countries.includes(country) : true;
            const okProject = perms.projects.length ? perms.projects.includes(project) : true;
            return okCountry && okProject;
          });
        }

        arr.sort((a,b) => new Date(b.createdAt) - new Date(a.createdAt));
        if (isMounted.current) { setItems(arr); setFilteredItems(arr); }
        await saveToCache(arr);
      } else {
        const snap = await database().ref('/GlobalInspectionReport/admins').once('value');
        const data = snap.val() || {};
        let arr = [];
        Object.entries(data).forEach(([adminId, reportsObj]) => {
          if (!reportsObj) return;
          Object.entries(reportsObj).forEach(([reportId, report]) => {
            if (reportId === 'lastRequestNumber') return;
            arr.push({ id: reportId, adminId, ...report, createdAt: report.step3?.createdAt || report.createdAt });
          });
        });

        // remove empty placeholder reports
        arr = arr.filter(r => !isReportEmpty(r));

        // apply permission filtering if user has countries/projects
        if (perms.countries.length || perms.projects.length) {
          arr = arr.filter(r => {
            const country = r.step1?.country || r.country || '';
            const project = r.step1?.project || r.project || '';
            const okCountry = perms.countries.length ? perms.countries.includes(country) : true;
            const okProject = perms.projects.length ? perms.projects.includes(project) : true;
            return okCountry && okProject;
          });
        }

        arr.sort((a,b) => new Date(b.createdAt) - new Date(a.createdAt));
        if (isMounted.current) { setItems(arr); setFilteredItems(arr); }
        await saveToCache(arr);
      }
    } catch (e) {
      console.error('Failed to fetch items', e);
      const cached = await loadFromCache();
      if (isMounted.current) { setItems(cached); setFilteredItems(cached); }
      showToast('error', 'Failed', 'Could not fetch latest data; showing cache if available');
    } finally {
      if (isMounted.current) setLoading(false);
    }
  };

  const confirmDeleteAll = () => {
    const title = type === 'hse' ? 'Delete All HSE Reports' : 'Delete All Inspection Reports';
    const message = 'Are you sure you want to delete ALL reports? This action cannot be undone.';
    Alert.alert(title, message, [
      { text: 'Cancel', style: 'cancel' },
      { text: 'Delete', style: 'destructive', onPress: performDeleteAll },
    ], { cancelable: true });
  };

  const performDeleteAll = async () => {
    setLoading(true);
    try {
      let userId = await AsyncStorage.getItem('userUid');
      if (!userId) {
        const userEmail = await AsyncStorage.getItem('userEmail');
        if (!userEmail) throw new Error('User not authenticated');
        const userSnapshot = await database().ref('/Globalusers/admins').orderByChild('email').equalTo(userEmail).once('value');
        const userData = userSnapshot.val();
        if (!userData) throw new Error('User not found');
        userId = Object.keys(userData)[0];
      }

      if (type === 'hse') {
        await database().ref(`/GlobalHSEInspections/admins/${userId}`).remove();
      } else {
        await database().ref(`/GlobalInspectionReport/admins/${userId}`).remove();
      }

      // remove cached items and per-report cache
      try {
        await AsyncStorage.removeItem(CACHE_KEYS[type]);
        for (const it of items || []) {
          if (it && it.id) await AsyncStorage.removeItem(`report_${it.id}`).catch(() => {});
        }
      } catch (e) { /* ignore */ }

      if (isMounted.current) setItems([]);
      showToast('success', 'Deleted', 'All reports deleted successfully.');
    } catch (e) {
      console.error('Failed to delete all reports', e);
      showToast('error', 'Error', e.message || 'Failed to delete reports');
    } finally {
      if (isMounted.current) setLoading(false);
    }
  };

  const toggleSelection = (id) => {
    setSelectedIds(prev => {
      const exists = prev.includes(id);
      if (exists) return prev.filter(x => x !== id);
      return [...prev, id];
    });
  };

  const performBulkDelete = async () => {
    if (!selectedIds || selectedIds.length === 0) {
      showToast('info', 'No selection', 'Please select reports to delete');
      return;
    }

    Alert.alert('Delete Selected', `Delete ${selectedIds.length} selected reports?`, [
      { text: 'Cancel', style: 'cancel' },
      { text: 'Delete', style: 'destructive', onPress: async () => {
        setLoading(true);
        try {
          // remove each selected item by its adminId (as rendered)
          for (const id of selectedIds) {
            const it = items.find(i => i.id === id);
            if (!it) continue;
            const adminId = it.adminId || (await AsyncStorage.getItem('userUid'));
            if (type === 'hse') {
              await database().ref(`/GlobalHSEInspections/admins/${adminId}/${id}`).remove();
            } else {
              await database().ref(`/GlobalInspectionReport/admins/${adminId}/${id}`).remove();
            }
            await AsyncStorage.removeItem(`report_${id}`).catch(() => {});
          }

          const remaining = items.filter(i => !selectedIds.includes(i.id));
          await AsyncStorage.setItem(CACHE_KEYS[type], JSON.stringify(remaining.slice(0,2000))).catch(() => {});
          if (isMounted.current) setItems(remaining);
          showToast('success', 'Deleted', `${selectedIds.length} reports deleted`);
        } catch (e) {
          console.error('Bulk delete failed', e);
          showToast('error', 'Error', 'Failed to delete selected reports');
        } finally {
          if (isMounted.current) {
            setLoading(false);
            setSelectionMode(false);
            setSelectedIds([]);
          }
        }
      }}
    ], { cancelable: true });
  };

  useEffect(() => {
    (async () => {
      const cached = await loadFromCache();
      if (isMounted.current && cached.length > 0) { setItems(cached); setFilteredItems(cached); }
      // fetch fresh once
      await fetchOnce();
    })();
    return () => { isMounted.current = false; };
  }, [type]);

  // when filter modal opens, deep-copy activeFilters into tempFilters so Cancel can discard
  useEffect(() => {
    if (filterModalVisible) {
      setTempFilters({
        dateRange: activeFilters.dateRange,
        countries: Array.isArray(activeFilters.countries) ? [...activeFilters.countries] : [],
        projects: Array.isArray(activeFilters.projects) ? [...activeFilters.projects] : [],
        inspectors: Array.isArray(activeFilters.inspectors) ? [...activeFilters.inspectors] : [],
        creators: Array.isArray(activeFilters.creators) ? [...activeFilters.creators] : [],
      });
    }
  }, [filterModalVisible, activeFilters]);

  // recompute filteredItems whenever items or activeFilters change
  useEffect(() => {
    // reuse logic from applyFilters but operate on current activeFilters
    const filters = activeFilters;
    let out = [...items];
    const { dateRange, countries, projects, inspectors, creators } = filters;
    if (dateRange && dateRange !== 'all') {
      const now = new Date();
      const cutoff = new Date();
      if (dateRange === 'today') cutoff.setHours(0,0,0,0);
      else if (dateRange === '7days') cutoff.setDate(now.getDate() - 7);
      else if (dateRange === '30days') cutoff.setDate(now.getDate() - 30);
      out = out.filter(r => {
        const d = r.createdAt ? new Date(r.createdAt) : null;
        return d && d >= cutoff;
      });
    }
    if (countries && countries.length) out = out.filter(r => countries.includes(r.step1?.country));
    if (projects && projects.length) out = out.filter(r => projects.includes(r.step1?.project));
    if (inspectors && inspectors.length) out = out.filter(r => inspectors.includes(r.step1?.inspector));
    if (creators && creators.length) out = out.filter(r => creators.includes(r.adminId));
    if (isMounted.current) setFilteredItems(out);
  }, [items, activeFilters]);

  // derive available filter values from items
  useEffect(() => {
    const countries = new Set();
    const projects = new Set();
    const inspectors = new Set();
    const creators = new Set();
    items.forEach(it => {
      const c = it.step1?.country;
      const p = it.step1?.project;
      const ins = it.step1?.inspector;
      if (c) countries.add(c);
      if (p) projects.add(p);
      if (ins) inspectors.add(ins);
      if (it.adminId) creators.add(it.adminId);
    });
    setAvailableCountries(Array.from(countries).sort());
    setAvailableProjects(Array.from(projects).sort());
    setAvailableInspectors(Array.from(inspectors).sort());

    const adminIds = Array.from(creators);
    if (adminIds.length === 0) {
      setAvailableCreators([]);
      setAdminIdToName({});
    } else {
      (async () => {
        const map = {};
        await Promise.all(adminIds.map(async (id) => {
          try {
            const snap = await database().ref(`/Globalusers/admins/${id}`).once('value');
            const val = snap.val();
            map[id] = val ? (val.name || val.email || id) : id;
          } catch (e) {
            map[id] = id;
          }
        }));
        if (isMounted.current) {
          setAdminIdToName(prev => ({ ...prev, ...map }));
          setAvailableCreators(adminIds.map(id => ({ id, name: map[id] || id })));
        }
      })().catch(() => {
        if (isMounted.current) setAvailableCreators(adminIds.map(id => ({ id, name: id })));
      });
    }
  }, [items]);

  const applyFilters = (filtersArg) => {
    const filters = filtersArg || activeFilters;
    let out = [...items];
    const { dateRange, countries, projects, inspectors, creators } = filters;
    if (dateRange && dateRange !== 'all') {
      const now = new Date();
      const cutoff = new Date();
      if (dateRange === 'today') cutoff.setHours(0,0,0,0);
      else if (dateRange === '7days') cutoff.setDate(now.getDate() - 7);
      else if (dateRange === '30days') cutoff.setDate(now.getDate() - 30);
      out = out.filter(r => {
        const d = r.createdAt ? new Date(r.createdAt) : null;
        return d && d >= cutoff;
      });
    }
    if (countries.length) out = out.filter(r => countries.includes(r.step1?.country));
    if (projects.length) out = out.filter(r => projects.includes(r.step1?.project));
    if (inspectors.length) out = out.filter(r => inspectors.includes(r.step1?.inspector));
    if (creators.length) out = out.filter(r => creators.includes(r.adminId));
    setFilteredItems(out);
    setActiveFilters(filters);
    setFilterModalVisible(false);
  };

  const getActiveFilterLabels = () => {
    const labels = [];
    if (activeFilters.dateRange && activeFilters.dateRange !== 'all') {
      labels.push(activeFilters.dateRange === 'today' ? 'Today' : activeFilters.dateRange === '7days' ? 'Last 7 Days' : activeFilters.dateRange === '30days' ? 'Last 30 Days' : 'Date');
    }
    activeFilters.countries.forEach(c => labels.push(c));
    activeFilters.projects.forEach(p => labels.push(p));
    activeFilters.inspectors.forEach(i => labels.push(i));
    if (activeFilters.creators && activeFilters.creators.length) labels.push(`${activeFilters.creators.length} creators`);
    return labels;
  };

  const clearFilters = () => {
    setActiveFilters({ dateRange: 'all', countries: [], projects: [], inspectors: [], creators: [] });
    setFilteredItems(items);
  };

  const clearTempFilters = () => {
    setTempFilters({ dateRange: 'all', countries: [], projects: [], inspectors: [], creators: [] });
  };

  const renderItem = ({ item }) => (
    <TouchableOpacity
      onPress={() => selectionMode ? toggleSelection(item.id) : null}
      onLongPress={() => setSelectionMode(true)}
      style={[styles.row, selectedIds.includes(item.id) && { backgroundColor: '#ffecec' }]}
    >
      <View style={{ flex: 1 }}>
        <Text style={styles.idText}>#{item.step1?.requestNumber || item.id}</Text>
        <Text style={styles.metaText}>{item.step1?.country || ''} • {item.step1?.project || ''}</Text>
        <Text style={styles.creatorText}>Created by: {adminIdToName[item.adminId] || item.adminId || 'Unknown'}</Text>
      </View>
      <View style={{ alignItems: 'flex-end' }}>
        <Text style={styles.dateText}>{item.createdAt ? new Date(item.createdAt).toLocaleString() : 'N/A'}</Text>
        {/* show buttons only when corresponding PDFs exist */}
        <View style={styles.actionContainer}>
          { (item.step3?.pdfDownloadUrl || item.step3?.pdfPath) ? (
            <TouchableOpacity disabled={actionLoadingId === item.id} onPress={() => handleViewReport(item)} style={styles.actionButton}>
              <Text style={styles.actionText}>View</Text>
            </TouchableOpacity>
          ) : null }

          { item.workOrder?.pdfDownloadUrl ? (
            <TouchableOpacity disabled={actionLoadingId === item.id} onPress={() => handleViewWorkOrder(item)} style={styles.actionButton}>
              <Text style={styles.actionText}>Job Card</Text>
            </TouchableOpacity>
          ) : null }

          { (item.correctiveAction?.pdfDownloadUrl || item.correctiveAction?.pdfPath) ? (
            <TouchableOpacity disabled={actionLoadingId === item.id} onPress={() => handleViewCorrective(item)} style={styles.actionButton}>
              <Text style={styles.actionText}>Corrective</Text>
            </TouchableOpacity>
          ) : null }
        </View>
      </View>
    </TouchableOpacity>
  );

  // Open inspection report PDF (online first, fallback to cached)
  const handleViewReport = async (report) => {
    if (!report) { showToast('error', 'Error', 'No report selected.'); return; }
    setActionLoadingId(report.id);
    try {
      const online = await checkNetwork();
      let pdfUrl = null;
      if (online) {
        const snap = await database().ref(`/GlobalInspectionReport/admins/${report.adminId}/${report.id}`).once('value');
        const val = snap.val();
        pdfUrl = val?.step3?.pdfDownloadUrl || val?.step3?.pdfPath || report.step3?.pdfDownloadUrl;
      } else {
        const raw = await AsyncStorage.getItem(`report_${report.id}`);
        if (raw) {
          const r = JSON.parse(raw);
          pdfUrl = r.step3?.pdfPath || r.step3?.pdfDownloadUrl;
        }
      }
      if (!pdfUrl) {
        showToast('error', 'Error', online ? 'PDF URL not available.' : 'PDF not available offline.');
        return;
      }
      navigation.navigate('PdfviewPage', { pdfUrl, onGoBack: () => {} });
    } catch (e) {
      console.error('Error opening report PDF', e);
      showToast('error', 'Error', 'Failed to open report PDF.');
    } finally {
      setActionLoadingId(null);
    }
  };

  // Open related work order PDF if exists
  const handleViewWorkOrder = async (report) => {
    if (!report) { showToast('error', 'Error', 'No report selected.'); return; }
    setActionLoadingId(report.id);
    try {
      const online = await checkNetwork();
      if (!online) { showToast('error', 'Error', 'Connect to internet to view Job Card.'); return; }
      const workOrdersSnap = await database().ref(`/GlobalWorkOrders/admins`).once('value');
      const workAdmins = workOrdersSnap.val() || {};
      let found = null;
      Object.entries(workAdmins).forEach(([adminId, wos]) => {
        if (!wos) return;
        Object.entries(wos).forEach(([woId, wo]) => {
          if (wo.linkInspection && wo.linkInspection.id === report.id) {
            found = { ...wo, id: woId, adminId };
          }
        });
      });
      if (!found || !found.pdfDownloadUrl) { showToast('info', 'Not Found', 'No Job Card PDF found for this inspection.'); return; }
      navigation.navigate('PdfviewPage', { pdfUrl: found.pdfDownloadUrl, onGoBack: () => {} });
    } catch (e) {
      console.error('Error fetching work order', e);
      showToast('error', 'Error', 'Failed to load Job Card.');
    } finally {
      setActionLoadingId(null);
    }
  };

  // Open corrective action PDF if exists
  const handleViewCorrective = async (report) => {
    if (!report) { showToast('error', 'Error', 'No report selected.'); return; }
    setActionLoadingId(report.id);
    try {
      const online = await checkNetwork();
      if (!online) { showToast('error', 'Error', 'Connect to internet to view Corrective Action.'); return; }
      const snap = await database().ref(`/GlobalCorrectiveActions/admins/${report.adminId}/${report.id}`).once('value');
      const val = snap.val();
      const pdfUrl = val?.pdfDownloadUrl || report.correctiveAction?.pdfDownloadUrl;
      if (!pdfUrl) { showToast('info', 'Not Found', 'No Corrective Action PDF found for this inspection.'); return; }
      navigation.navigate('PdfviewPage', { pdfUrl, onGoBack: () => {} });
    } catch (e) {
      console.error('Error fetching corrective action', e);
      showToast('error', 'Error', 'Failed to load Corrective Action.');
    } finally {
      setActionLoadingId(null);
    }
  };

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
      <View style={styles.header}>
        <View style={{ flexDirection: 'row', alignItems: 'center' }}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={{ padding: 8 }}>
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={[styles.headerTitle, { marginLeft: 12 }]}>{type === 'hse' ? 'HSE Reports' : 'Inspection Reports'}</Text>
        </View>
        <View style={{ flexDirection: 'row', alignItems: 'center' }}>
          <Text style={{ color: '#fff', fontSize: 12, marginRight: 8 }}>Total: {filteredItems.length}</Text>
          <TouchableOpacity onPress={() => setFilterModalVisible(true)} style={{ padding: 8 }}>
            <MaterialIcons name="filter-list" size={22} color="#fff" />
          </TouchableOpacity>
          <TouchableOpacity onPress={fetchOnce} style={{ padding: 8 }}>
            <MaterialIcons name="refresh" size={22} color="#fff" />
          </TouchableOpacity>
          <TouchableOpacity onPress={confirmDeleteAll} style={{ padding: 8, marginLeft: 8 }}>
            <MaterialIcons name="delete-sweep" size={22} color="#fff" />
          </TouchableOpacity>
        </View>
      </View>

      {/* Active filter summary */}
      {getActiveFilterLabels().length > 0 && (
        <View style={styles.filterSummary}>
          <ScrollView horizontal showsHorizontalScrollIndicator={false} contentContainerStyle={{ alignItems: 'center' }}>
            {getActiveFilterLabels().map((lbl, idx) => (
              <View key={idx} style={styles.filterChipSmall}>
                <Text style={styles.filterChipTextSmall}>{lbl}</Text>
              </View>
            ))}
          </ScrollView>
          <TouchableOpacity onPress={clearFilters} style={{ marginLeft: 8 }}>
            <Text style={styles.clearFilterText}>Clear</Text>
          </TouchableOpacity>
        </View>
      )}

      <View style={styles.content}>
        {selectionMode && (
          <View style={{ flexDirection: 'row', justifyContent: 'space-between', marginBottom: 12 }}>
            <Text style={{ fontSize: 14 }}>{selectedIds.length} selected</Text>
            <View style={{ flexDirection: 'row' }}>
              <TouchableOpacity onPress={() => { setSelectionMode(false); setSelectedIds([]); }} style={{ marginRight: 12 }}>
                <Text style={{ color: '#0078D4' }}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={performBulkDelete}>
                <Text style={{ color: '#e74c3c' }}>Delete Selected</Text>
              </TouchableOpacity>
            </View>
          </View>
        )}
        {loading ? (
          <ActivityIndicator size="large" color="#015185" style={{ marginTop: 20 }} />
        ) : (
          <>
            <FlatList
              data={filteredItems}
              keyExtractor={(it) => it.id}
              renderItem={renderItem}
              contentContainerStyle={{ paddingBottom: 120 }}
              ListEmptyComponent={<Text style={{ textAlign: 'center', color: '#666', marginTop: 24 }}>No records found</Text>}
            />
          </>
        )}
      </View>
      
      <Modal visible={filterModalVisible} transparent animationType="slide" onRequestClose={() => setFilterModalVisible(false)}>
        <View style={[styles.filterModalOverlay, { alignItems: 'stretch' }]}>
          <View style={[styles.filterModalContainer, { width: '100%' }]}>
            <View style={styles.filterModalHeader}>
              <Text style={styles.filterModalTitle}>Filter Inspection Reports</Text>
              <TouchableOpacity onPress={() => { setFilterModalVisible(false); setTempFilters(activeFilters); }}>
                <MaterialIcons name="close" size={24} color="#666" />
              </TouchableOpacity>
            </View>
            <ScrollView style={styles.filterModalContent}>
              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>Date Range</Text>
                <View style={styles.filterChipsContainer}>
                  {[
                    { label: 'All Time', value: 'all' },
                    { label: 'Today', value: 'today' },
                    { label: 'Last 7 Days', value: '7days' },
                    { label: 'Last 30 Days', value: '30days' },
                  ].map(range => (
                    <TouchableOpacity
                      key={range.value}
                      style={[styles.filterChip, tempFilters.dateRange === range.value && styles.filterChipActive]}
                      onPress={() => setTempFilters(prev => ({ ...prev, dateRange: range.value }))}
                    >
                      <Text style={[styles.filterChipText, tempFilters.dateRange === range.value && styles.filterChipTextActive]}>{range.label}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>

              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>Countries</Text>
                <View style={styles.filterChipsContainer}>
                  {availableCountries.length === 0 ? <Text style={styles.noFilterOptions}>No countries</Text> : availableCountries.map(c => (
                    <TouchableOpacity key={c} style={[styles.filterChip, tempFilters.countries.includes(c) && styles.filterChipActive]} onPress={() => setTempFilters(prev => ({ ...prev, countries: prev.countries.includes(c) ? prev.countries.filter(x => x !== c) : [...prev.countries, c] }))}>
                      <Text style={[styles.filterChipText, tempFilters.countries.includes(c) && styles.filterChipTextActive]}>{c}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>

              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>Projects</Text>
                <View style={styles.filterChipsContainer}>
                  {availableProjects.length === 0 ? <Text style={styles.noFilterOptions}>No projects</Text> : availableProjects.map(p => (
                    <TouchableOpacity key={p} style={[styles.filterChip, tempFilters.projects.includes(p) && styles.filterChipActive]} onPress={() => setTempFilters(prev => ({ ...prev, projects: prev.projects.includes(p) ? prev.projects.filter(x => x !== p) : [...prev.projects, p] }))}>
                      <Text style={[styles.filterChipText, tempFilters.projects.includes(p) && styles.filterChipTextActive]}>{p}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>

              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>Inspectors</Text>
                <View style={styles.filterChipsContainer}>
                  {availableInspectors.length === 0 ? <Text style={styles.noFilterOptions}>No inspectors</Text> : availableInspectors.map(i => (
                    <TouchableOpacity key={i} style={[styles.filterChip, tempFilters.inspectors.includes(i) && styles.filterChipActive]} onPress={() => setTempFilters(prev => ({ ...prev, inspectors: prev.inspectors.includes(i) ? prev.inspectors.filter(x => x !== i) : [...prev.inspectors, i] }))}>
                      <Text style={[styles.filterChipText, tempFilters.inspectors.includes(i) && styles.filterChipTextActive]}>{i}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>

              <View style={styles.filterSection}>
                <Text style={styles.filterSectionTitle}>Creators</Text>
                <View style={styles.filterChipsContainer}>
                  {availableCreators.length === 0 ? <Text style={styles.noFilterOptions}>No creators</Text> : availableCreators.map(c => (
                    <TouchableOpacity key={c.id} style={[styles.filterChip, tempFilters.creators.includes(c.id) && styles.filterChipActive]} onPress={() => setTempFilters(prev => ({ ...prev, creators: prev.creators.includes(c.id) ? prev.creators.filter(x => x !== c.id) : [...prev.creators, c.id] }))}>
                      <Text style={[styles.filterChipText, tempFilters.creators.includes(c.id) && styles.filterChipTextActive]}>{c.name}</Text>
                    </TouchableOpacity>
                  ))}
                </View>
              </View>
            </ScrollView>

            <View style={styles.filterModalFooter}>
              <TouchableOpacity style={styles.clearFiltersButton} onPress={() => { clearTempFilters(); }}>
                <Text style={styles.clearFiltersButtonText}>Clear All</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.applyFiltersButton} onPress={() => { applyFilters(tempFilters); }}>
                <Text style={styles.applyFiltersButtonText}>Apply Filters</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  header: {
    backgroundColor: '#015185',
    paddingTop: StatusBar.currentHeight || 44,
    paddingHorizontal: 12,
    paddingBottom: 12,
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  headerTitle: { color: '#fff', fontSize: 18, fontWeight: '700', marginLeft: 8 },
  content: { flex: 1, padding: 12 },
  countText: { fontSize: 32, fontWeight: '700', color: '#015185', textAlign: 'center', marginVertical: 12 },
  countRow: { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', paddingHorizontal: 12, marginTop: 12 },
  smallBox: { width: 40, height: 30, borderWidth: 1, borderColor: '#e74c3c', borderRadius: 4 },
  bigCount: { fontSize: 36, fontWeight: '700', color: '#015185' },
  filterSummary: { flexDirection: 'row', alignItems: 'center', paddingHorizontal: 12, paddingVertical: 8, borderBottomWidth: 1, borderBottomColor: '#eee', backgroundColor: '#fff' },
  filterChipSmall: { backgroundColor: '#e3f2fd', paddingHorizontal: 8, paddingVertical: 6, borderRadius: 12, marginRight: 8 },
  filterChipTextSmall: { color: '#015185', fontSize: 12, fontWeight: '600' },
  clearFilterText: { color: '#e74c3c', fontSize: 12, fontWeight: '600' },
  // Filter modal styles
  filterModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'flex-end',
    alignItems: 'stretch',
  },
  filterModalContainer: {
    backgroundColor: '#fff',
    borderTopLeftRadius: 20,
    borderTopRightRadius: 20,
    maxHeight: (Dimensions.get('window').height || 800) * 0.85,
    alignSelf: 'stretch',
    paddingBottom: 12,
  },
  filterModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  filterModalTitle: {
    fontSize: 16,
    fontWeight: '700',
    color: '#015185',
  },
  filterModalContent: {
    paddingHorizontal: 16,
    paddingVertical: 12,
  },
  filterSection: {
    marginBottom: 16,
  },
  filterSectionTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
  },
  filterChipsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  filterChip: {
    paddingVertical: 8,
    paddingHorizontal: 12,
    borderRadius: 18,
    backgroundColor: '#f0f0f0',
    borderWidth: 1,
    borderColor: '#ddd',
    marginRight: 8,
    marginBottom: 8,
  },
  filterChipActive: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  filterChipText: {
    fontSize: 13,
    color: '#333',
    fontWeight: '500',
  },
  filterChipTextActive: {
    color: '#fff',
    fontWeight: '600',
  },
  noFilterOptions: {
    fontSize: 14,
    color: '#999',
    fontStyle: 'italic',
    textAlign: 'center',
    paddingVertical: 8,
  },
  filterModalFooter: {
    flexDirection: 'row',
    padding: 12,
    borderTopWidth: 1,
    borderTopColor: '#e9ecef',
  },
  clearFiltersButton: {
    flex: 1,
    paddingVertical: 10,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
    alignItems: 'center',
    marginRight: 8,
  },
  clearFiltersButtonText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
  applyFiltersButton: {
    flex: 1,
    paddingVertical: 10,
    borderRadius: 8,
    backgroundColor: '#015185',
    alignItems: 'center',
  },
  applyFiltersButtonText: {
    fontSize: 14,
    color: '#fff',
    fontWeight: '600',
  },
  row: { flexDirection: 'row', padding: 12, backgroundColor: '#f7f7f7', borderRadius: 8, marginBottom: 8, alignItems: 'center' },
  idText: { color: '#015185', fontWeight: '700' },
  metaText: { color: '#666', marginTop: 4, fontSize: 12 },
  dateText: { color: '#666', fontSize: 12, marginLeft: 8 },
  actionContainer: { flexDirection: 'row', marginTop: 8 },
  actionButton: { backgroundColor: '#015185', paddingHorizontal: 8, paddingVertical: 4, borderRadius: 6, marginLeft: 6 },
  actionText: { color: '#fff', fontSize: 12 },
  creatorText: { color: '#666', fontSize: 12, marginTop: 6 },
});

export default InspectionListView;


