import React, { useState, useImperativeHandle, forwardRef, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  Alert,
  Image,
  KeyboardAvoidingView,
  Platform,
  Modal,
  Animated,
  PermissionsAndroid,
  BackHandler,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
// Removed Firebase imports - now using MongoDB API
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import Toast from 'react-native-toast-message';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import NetInfo from '@react-native-community/netinfo';
import OfflineIncidentHelper from './OfflineIncidentHelper';
import VoiceToText, { VoiceToTextEvents } from '@appcitor/react-native-voice-to-text';
import LottieView from 'lottie-react-native';

const { width } = Dimensions.get('window');
const IMAGE_BOX_GAP = 8;
const IMAGE_BOX_HORIZONTAL_PADDING = 50;
const IMAGE_BOX_SIZE = (width - IMAGE_BOX_HORIZONTAL_PADDING - 2 * IMAGE_BOX_GAP - 12) / 3;

// Incident Notifications API URL
const INCIDENT_NOTIFICATIONS_API_URL = 'https://api.titandrillingzm.com:6021';

// Helper function for fetch with timeout
const fetchWithTimeout = (url, options = {}, timeout = 10000) => {
  return Promise.race([
    fetch(url, options),
    new Promise((_, reject) =>
      setTimeout(() => reject(new Error('Request timeout - please check your internet connection')), timeout)
    ),
  ]);
};

// Helper function to check network connectivity
const checkNetworkConnection = async () => {
  try {
    const netInfo = await NetInfo.fetch();
    console.log('Network status:', netInfo);
    
    if (!netInfo.isConnected) {
      throw new Error('No internet connection. Please check your network settings.');
    }
    
    if (netInfo.isInternetReachable === false) {
      throw new Error('Internet not reachable. Please check your connection.');
    }
    
    return true;
  } catch (error) {
    console.error('Network check error:', error);
    throw error;
  }
};

// SyncModal Component for Animated Loading
const SyncModal = ({ visible }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Generating PDF...</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// SuccessModal Component
const SuccessModal = ({ visible, onClose, onNavigateBack, onViewPDF, pdfUrl }) => {
  return (
    <Modal visible={visible} animationType="fade" transparent onRequestClose={onClose}>
      <View style={styles.successOverlay}>
        <View style={styles.successCard}>
          <View style={styles.successIconWrap}>
            <MaterialIcons name="check-circle" size={54} color="#00B894" />
          </View>
          <Text style={styles.successTitle}>Incident Report Saved</Text>
          <Text style={styles.successMessage}>
            Your incident report has been generated and stored successfully. You can head back or
            open the PDF now.
          </Text>
          <View style={styles.successActions}>
            <TouchableOpacity
              style={[styles.successButton, styles.successButtonSecondary]}
              onPress={() => onNavigateBack && onNavigateBack()}
              activeOpacity={0.8}
            >
              <Text style={[styles.successButtonText, styles.successButtonSecondaryText]}>Back</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.successButton}
              onPress={() => onViewPDF && onViewPDF(pdfUrl)}
              activeOpacity={0.8}
            >
              <Text style={styles.successButtonText}>View PDF</Text>
            </TouchableOpacity>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const VoiceToTextModal = ({
  visible,
  initialValue,
  onCancel,
  onComplete,
}) => {
  const [isVoiceModuleAvailable, setIsVoiceModuleAvailable] = useState(false);
  const [isRecognitionAvailable, setIsRecognitionAvailable] = useState(true);
  const [micPermissionGranted, setMicPermissionGranted] = useState(Platform.OS !== 'android');
  const [isListening, setIsListening] = useState(false);
  const [transcript, setTranscript] = useState(initialValue || '');
  const [partialText, setPartialText] = useState('');
  const [volume, setVolume] = useState(0);
  const [error, setError] = useState(null);

  const lottieRef = useRef(null);
  const listenersRef = useRef([]);
  const isMountedRef = useRef(true);
  const isCleaningUpRef = useRef(false);
  const isListeningRef = useRef(false);
  const hasStoppedRecordingRef = useRef(false);
  const hasReceivedFinalResultsRef = useRef(false);
  const isRestartingRef = useRef(false);
  const ignoreVoiceEventsRef = useRef(false);
  const cleanupTimeoutRef = useRef(null);
  const lastFinalTextRef = useRef('');
  const transcriptRef = useRef(initialValue || '');

  const ensureInitialised = useRef(false);
  const visibleRef = useRef(visible);

  useEffect(() => {
    visibleRef.current = visible;
  }, [visible]);

  useEffect(() => {
    transcriptRef.current = initialValue || '';
    if (!visible) {
      setTranscript(initialValue || '');
    }
  }, [initialValue, visible]);

  useEffect(() => {
    isMountedRef.current = true;
    if (ensureInitialised.current) {
      return;
    }

    if (!VoiceToText || typeof VoiceToText.addEventListener !== 'function') {
      const msg = 'VoiceToText module is unavailable. Ensure @appcitor/react-native-voice-to-text is installed and linked correctly.';
      console.error(msg);
      setError(msg);
    } else {
      setIsVoiceModuleAvailable(true);

      const resultsListener = VoiceToText.addEventListener(VoiceToTextEvents.RESULTS, (event) => {
        if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current && visibleRef.current) {
          onSpeechResults(event);
        }
      });
      const partialListener = VoiceToText.addEventListener(VoiceToTextEvents.PARTIAL_RESULTS, (event) => {
        if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current && visibleRef.current) {
          if (isListeningRef.current) {
            setPartialText(event?.value || '');
          }
        }
      });
      const volumeListener = VoiceToText.addEventListener(VoiceToTextEvents.VOLUME_CHANGED, (event) => {
        if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current && visibleRef.current) {
          setVolume(event?.value || 0);
        }
      });
      const errorListener = VoiceToText.addEventListener(VoiceToTextEvents.ERROR, (event) => {
        if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current && visibleRef.current) {
          onSpeechError(event);
        }
      });
      const startListener = VoiceToText.addEventListener(VoiceToTextEvents.START, () => {
        if (isMountedRef.current && !isCleaningUpRef.current && visibleRef.current) {
          console.log('🎤 Recording started event received');
          isListeningRef.current = true;
          hasStoppedRecordingRef.current = false;
          hasReceivedFinalResultsRef.current = false;
          isRestartingRef.current = false;
          setIsListening(true);
          setPartialText('');
          ignoreVoiceEventsRef.current = false;
          if (lottieRef.current) {
            lottieRef.current.play();
          }
        }
      });
      const endListener = VoiceToText.addEventListener(VoiceToTextEvents.END, () => {
        console.log('⚠️ END event received - recording session ended');
        if (
          isMountedRef.current &&
          !isCleaningUpRef.current &&
          !hasStoppedRecordingRef.current &&
          isListeningRef.current &&
          !isRestartingRef.current &&
          visibleRef.current
        ) {
          console.log('🔄 Auto-restarting recording for continuous mode...');
          isRestartingRef.current = true;
          setTimeout(async () => {
            if (
              isMountedRef.current &&
              !isCleaningUpRef.current &&
              !hasStoppedRecordingRef.current &&
              isListeningRef.current &&
          visibleRef.current
            ) {
              try {
                if (VoiceToText && typeof VoiceToText.startListening === 'function') {
                  await VoiceToText.startListening();
                } else if (VoiceToText && typeof VoiceToText.start === 'function') {
                  await VoiceToText.start();
                }
                console.log('✅ Recording restarted automatically');
                isRestartingRef.current = false;
              } catch (restartError) {
                console.warn('⚠️ Failed to auto-restart recording:', restartError);
                isRestartingRef.current = false;
              }
            } else {
              isRestartingRef.current = false;
            }
          }, 200);
        } else {
          console.log('ℹ️ END event ignored (user stopped or cleaning up)');
        }
      });

      listenersRef.current = [
        resultsListener,
        partialListener,
        volumeListener,
        errorListener,
        startListener,
        endListener,
      ];
    }

    ensureInitialised.current = true;

    (async () => {
      try {
        if (typeof VoiceToText.isRecognitionAvailable === 'function') {
          const avail = await VoiceToText.isRecognitionAvailable();
          setIsRecognitionAvailable(avail === true);
          if (!avail && Platform.OS === 'ios') {
            setError('Speech recognition not available on this device. Test on a real device and ensure permissions are set in Info.plist.');
          }
        }
      } catch (err) {
        console.warn('isRecognitionAvailable check failed', err);
      }
    })();

    const checkPermissions = async () => {
      if (Platform.OS === 'android') {
        try {
          const granted = await PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.RECORD_AUDIO);
          setMicPermissionGranted(granted);
          if (!granted) {
            const requested = await PermissionsAndroid.request(
              PermissionsAndroid.PERMISSIONS.RECORD_AUDIO,
              {
                title: 'Microphone Permission',
                message: 'This app needs microphone access for speech recognition.',
                buttonPositive: 'OK',
              }
            );
            const isGranted = requested === PermissionsAndroid.RESULTS.GRANTED;
            setMicPermissionGranted(isGranted);
            if (!isGranted) {
              setError('Microphone permission denied. Please enable it in app settings.');
            }
          }
        } catch (err) {
          console.warn('Permission check error:', err);
          setError('Failed to check microphone permissions.');
        }
      }
    };

    checkPermissions();

    return () => {
      console.log('🧹 Voice modal unmount cleanup');
      isMountedRef.current = false;
      isCleaningUpRef.current = true;
      if (cleanupTimeoutRef.current) {
        clearTimeout(cleanupTimeoutRef.current);
      }
      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }
      stopRecordingImmediately();
      listenersRef.current = [];
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    if (visible) {
      hasStoppedRecordingRef.current = false;
      isCleaningUpRef.current = false;
      ignoreVoiceEventsRef.current = false;
      setError(null);
      setPartialText('');
      setVolume(0);
      setTranscript(initialValue || '');
      transcriptRef.current = initialValue || '';
      lastFinalTextRef.current = '';

      startRecording();
    } else {
      safeStopRecording(false);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [visible]);

  useEffect(() => {
    const backHandler = BackHandler.addEventListener('hardwareBackPress', () => {
      if (visible && isListeningRef.current) {
        cancelRecording();
        return true;
      }
      return false;
    });
    return () => backHandler.remove();
  }, [visible]);

  const onSpeechResults = (e) => {
    if (!isMountedRef.current || isCleaningUpRef.current || ignoreVoiceEventsRef.current) {
      return;
    }
    hasReceivedFinalResultsRef.current = true;
    try {
      let payload = e?.value ?? e?.results ?? e?.text ?? e;
      let finalText = '';
      if (Array.isArray(payload)) {
        if (payload.length) {
          finalText = payload.join(' ');
        }
      } else if (typeof payload === 'string') {
        finalText = payload;
      } else if (payload && typeof payload === 'object') {
        const extracted = Object.values(payload)
          .flatMap((v) => (Array.isArray(v) ? v : [v]))
          .filter((v) => typeof v === 'string');
        if (extracted.length) {
          finalText = extracted.join(' ');
        } else {
          finalText = String(payload ?? '');
        }
      } else {
        finalText = String(payload ?? '');
      }

      if (finalText && finalText.trim() && isMountedRef.current) {
        const trimmedFinal = finalText.trim();
        if (lastFinalTextRef.current === trimmedFinal) {
          console.log('⚠️ Duplicate final text detected, skipping');
          setPartialText('');
          return;
        }
        setTranscript((prev) => {
          const prevTrimmed = (prev || '').trim();
          if (!prevTrimmed) {
            lastFinalTextRef.current = trimmedFinal;
            transcriptRef.current = trimmedFinal;
            return trimmedFinal;
          }
          if (prevTrimmed.endsWith(trimmedFinal)) {
            return prevTrimmed;
          }
          const lastFewWords = prevTrimmed.split(' ').slice(-3).join(' ');
          const firstFewWords = trimmedFinal.split(' ').slice(0, 3).join(' ');
          if (lastFewWords === firstFewWords && lastFewWords.length > 5) {
            return prev;
          }
          if (trimmedFinal.length > 15 && prevTrimmed.includes(trimmedFinal)) {
            return prev;
          }
          const updated = `${prevTrimmed} ${trimmedFinal}`.trim();
          lastFinalTextRef.current = trimmedFinal;
          transcriptRef.current = updated;
          return updated;
        });
        setPartialText('');
      }
    } catch (err) {
      console.warn('onSpeechResults parse error', err, e);
    }
  };

  const onSpeechError = (e) => {
    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }
    const message = e?.error?.message || e?.message || JSON.stringify(e);
    setError(`Speech recognition error: ${message}`);
    isListeningRef.current = false;
    setIsListening(false);
  };

  const startRecording = async () => {
    if (!visible) return;
    if (!isVoiceModuleAvailable) {
      setError('Voice module is not available. Please check installation.');
      return;
    }
    if (Platform.OS === 'android' && !micPermissionGranted) {
      setError('Microphone permission denied.');
      return;
    }
    try {
      if (Platform.OS === 'ios' && !isRecognitionAvailable) {
        setError('Speech recognition not available on this device or permission denied.');
        return;
      }
      if (typeof VoiceToText.setRecognitionLanguage === 'function') {
        await VoiceToText.setRecognitionLanguage('en-US');
      }
      hasStoppedRecordingRef.current = false;
      hasReceivedFinalResultsRef.current = false;
      isRestartingRef.current = false;
      isListeningRef.current = true;
      ignoreVoiceEventsRef.current = false;
      setPartialText('');
      lastFinalTextRef.current = '';
      setIsListening(true);
      if (lottieRef.current) {
        lottieRef.current.play();
      }
      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      } else {
        throw new Error('startListening not available on VoiceToText');
      }
      console.log('✅ Recording started successfully');
    } catch (e) {
      const message = e?.message || JSON.stringify(e);
      if (Platform.OS === 'ios' && message.includes('IsFormatSampleRateAndChannelCountValid')) {
        setError('iOS audio format error. Test on a real device and ensure Info.plist includes NSMicrophoneUsageDescription and NSSpeechRecognitionUsageDescription.');
      } else {
        setError(`Failed to start recording: ${message}`);
      }
      setIsListening(false);
      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }
    }
  };

  const safeStopRecording = useCallback(
    (shouldAddPartialText = false) =>
      new Promise((resolve) => {
        if (hasStoppedRecordingRef.current || !isListeningRef.current) {
          resolve(transcriptRef.current || '');
          return;
        }
        hasStoppedRecordingRef.current = true;
        isListeningRef.current = false;
        ignoreVoiceEventsRef.current = true;

        const finalize = (finalTranscript) => {
          resolve(finalTranscript?.trim() || '');
        };

        (async () => {
          if (VoiceToText && isVoiceModuleAvailable) {
            const stopMethods = [
              () => VoiceToText.stopListening && VoiceToText.stopListening(),
              () => VoiceToText.stop && VoiceToText.stop(),
              () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
              () => VoiceToText.cancel && VoiceToText.cancel(),
            ];
            for (const stopMethod of stopMethods) {
              try {
                if (typeof stopMethod === 'function') {
                  const maybePromise = stopMethod();
                  if (maybePromise && typeof maybePromise.then === 'function') {
                    await maybePromise;
                  }
                  break;
                }
              } catch (stopError) {
                console.warn('⚠️ Error stopping recording:', stopError);
              }
            }
          }
          if (lottieRef.current) {
            lottieRef.current.pause();
            lottieRef.current.reset();
          }
          if (isMountedRef.current && !isCleaningUpRef.current) {
            setIsListening(false);
            cleanupTimeoutRef.current = setTimeout(() => {
              if (!isMountedRef.current || isCleaningUpRef.current) {
                finalize(transcriptRef.current);
                return;
              }
              if (shouldAddPartialText && !hasReceivedFinalResultsRef.current) {
                setPartialText((currentPartial) => {
                  let finalValue = transcriptRef.current;
                  if (currentPartial && currentPartial.trim()) {
                    setTranscript((prev) => {
                      const prevTrimmed = (prev || '').trim();
                      const trimmedPartial = currentPartial.trim();
                      if (prevTrimmed.endsWith(trimmedPartial)) {
                        finalValue = prevTrimmed;
                        transcriptRef.current = prevTrimmed;
                        finalize(finalValue);
                        return prevTrimmed;
                      }
                      if (prevTrimmed.includes(trimmedPartial) && trimmedPartial.length > 5) {
                        finalValue = prevTrimmed;
                        transcriptRef.current = prevTrimmed;
                        finalize(finalValue);
                        return prevTrimmed;
                      }
                      const updated = prevTrimmed ? `${prevTrimmed} ${trimmedPartial}`.trim() : trimmedPartial;
                      transcriptRef.current = updated;
                      finalValue = updated;
                      finalize(updated);
                      return updated;
                    });
                  } else {
                    finalize(transcriptRef.current);
                  }
                  return '';
                });
              } else {
                setPartialText('');
                finalize(transcriptRef.current);
              }
            }, 500);
          } else {
            finalize(transcriptRef.current);
          }
        })();
      }),
    [isVoiceModuleAvailable]
  );

  const stopRecordingImmediately = async () => {
    try {
      if (isListeningRef.current && VoiceToText && !hasStoppedRecordingRef.current) {
        hasStoppedRecordingRef.current = true;
        isListeningRef.current = false;
        ignoreVoiceEventsRef.current = true;
        const stopMethods = [
          () => VoiceToText.stopListening && VoiceToText.stopListening(),
          () => VoiceToText.stop && VoiceToText.stop(),
          () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
          () => VoiceToText.cancel && VoiceToText.cancel(),
          () => VoiceToText.destroy && VoiceToText.destroy(),
        ];
        for (const stopMethod of stopMethods) {
          try {
            if (typeof stopMethod === 'function') {
              const maybePromise = stopMethod();
              if (maybePromise && typeof maybePromise.then === 'function') {
                await maybePromise;
              }
            }
          } catch (e) {
            console.warn('⚠️ Error during cleanup stop:', e);
          }
        }
      }
    } catch (e) {
      console.warn('⚠️ Error stopping recording immediately:', e);
    }
  };

  const cancelRecording = async () => {
    const finalText = await safeStopRecording(false);
    onCancel(finalText);
  };

  const handleDone = async () => {
    const finalText = await safeStopRecording(true);
    onComplete(finalText);
  };

  return (
    <Modal
      visible={visible}
      transparent
      animationType="fade"
      onRequestClose={() => {
        cancelRecording();
      }}
    >
      <View style={styles.voiceModalOverlay}>
        <View style={styles.voiceModalContent}>
          <View style={styles.voiceLottieContainer}>
            <LottieView
              ref={lottieRef}
              source={require('../../../Images/micanimation.json')}
              autoPlay={false}
              loop
              style={styles.voiceLottie}
              resizeMode="contain"
            />
          </View>
          <Text style={styles.voiceStatusText}>
            {partialText
              ? partialText
              : isListening
              ? 'Listening... Speak now'
              : error
              ? error
              : 'Preparing...'}
          </Text>
          {volume > 0 && (
            <View style={styles.voiceVolumeIndicator}>
              <View style={[styles.voiceVolumeBar, { width: `${Math.min(volume * 100, 100)}%` }]} />
            </View>
          )}
          <View style={styles.voiceModalButtons}>
            <TouchableOpacity style={[styles.voiceModalButton, styles.voiceCancelButton]} onPress={cancelRecording}>
              <Text style={[styles.voiceModalButtonText, { color: '#015185' }]}>Cancel</Text>
            </TouchableOpacity>
            <TouchableOpacity style={styles.voiceModalButton} onPress={handleDone}>
              <Text style={styles.voiceModalButtonText}>Done</Text>
            </TouchableOpacity>
          </View>
          <View style={styles.voiceTranscriptContainer}>
            <Text style={styles.voiceTranscriptLabel}>Captured Text</Text>
            <ScrollView style={styles.voiceTranscriptBox}>
              <Text style={styles.voiceTranscriptText}>{transcript || 'No text yet'}</Text>
            </ScrollView>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const IncidentReportStep2 = forwardRef(({ currentStep, formData, selectedIncidents, involvesEquipment }, ref) => {
  const navigation = useNavigation();
  const [incidentImages, setIncidentImages] = useState(Array(9).fill(null));
  const [incidentCause, setIncidentCause] = useState('');
  const [equipmentDamage, setEquipmentDamage] = useState('');
  const [additionalComment, setAdditionalComment] = useState('');
  const [immediateCorrectiveAction, setImmediateCorrectiveAction] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [successModalVisible, setSuccessModalVisible] = useState(false);
  const [pdfUrl, setPdfUrl] = useState(null);
  const [reportId, setReportId] = useState(null);
  const [voiceModalVisible, setVoiceModalVisible] = useState(false);
  const [activeVoiceField, setActiveVoiceField] = useState(null);
  const [voiceInitialValue, setVoiceInitialValue] = useState('');

  useEffect(() => {
    loadPersistedState();
  }, []);

  const loadPersistedState = async () => {
    try {
      const state = await AsyncStorage.getItem('IncidentReportStep2State');
      if (state) {
        const parsed = JSON.parse(state);
        setIncidentImages(parsed.incidentImages || Array(9).fill(null));
        setIncidentCause(parsed.incidentCause || '');
        setEquipmentDamage(parsed.equipmentDamage || '');
        setAdditionalComment(parsed.additionalComment || '');
        setImmediateCorrectiveAction(parsed.immediateCorrectiveAction || '');
      }
    } catch (error) {
      console.error('Error loading persisted state:', error);
    }
  };

  const saveState = async (overrides = {}) => {
    try {
      const state = {
        incidentImages: overrides.incidentImages ?? incidentImages,
        incidentCause: overrides.incidentCause ?? incidentCause,
        equipmentDamage: overrides.equipmentDamage ?? equipmentDamage,
        additionalComment: overrides.additionalComment ?? additionalComment,
        immediateCorrectiveAction:
          overrides.immediateCorrectiveAction ?? immediateCorrectiveAction,
      };
      await AsyncStorage.setItem('IncidentReportStep2State', JSON.stringify(state));
    } catch (error) {
      console.error('Error saving state:', error);
    }
  };

  const getVoiceFieldValue = useCallback(
    (fieldKey) => {
      switch (fieldKey) {
        case 'incidentCause':
          return incidentCause;
        case 'equipmentDamage':
          return equipmentDamage;
        case 'additionalComment':
          return additionalComment;
        case 'immediateCorrectiveAction':
          return immediateCorrectiveAction;
        default:
          return '';
      }
    },
    [incidentCause, equipmentDamage, additionalComment, immediateCorrectiveAction]
  );

  const applyVoiceResultToField = useCallback(
    (fieldKey, value) => {
      switch (fieldKey) {
        case 'incidentCause':
          setIncidentCause(value);
          saveState({ incidentCause: value });
          break;
        case 'equipmentDamage':
          setEquipmentDamage(value);
          saveState({ equipmentDamage: value });
          break;
        case 'additionalComment':
          setAdditionalComment(value);
          saveState({ additionalComment: value });
          break;
        case 'immediateCorrectiveAction':
          setImmediateCorrectiveAction(value);
          saveState({ immediateCorrectiveAction: value });
          break;
        default:
          break;
      }
    },
    [saveState]
  );

  const handleOpenVoiceModal = useCallback(
    (fieldKey) => {
      const existingValue = getVoiceFieldValue(fieldKey) || '';
      setActiveVoiceField(fieldKey);
      setVoiceInitialValue(existingValue);
      setVoiceModalVisible(true);
    },
    [getVoiceFieldValue]
  );

  const handleVoiceCancel = useCallback(() => {
    setVoiceModalVisible(false);
    setActiveVoiceField(null);
  }, []);

  const handleVoiceComplete = useCallback(
    (finalText) => {
      if (activeVoiceField) {
        applyVoiceResultToField(activeVoiceField, finalText || '');
      }
      setVoiceModalVisible(false);
      setActiveVoiceField(null);
    },
    [activeVoiceField, applyVoiceResultToField]
  );



  const validateStep2 = () => {
    saveState();
    return true; // No required fields in Step 2
  };

  const requestStoragePermission = async () => {
    try {
      // Android 13+ uses Photo Picker which doesn't require READ_MEDIA_IMAGES
      // Only request legacy storage permissions for Android 12 and below
      const permissions = Platform.OS === 'android'
        ? Platform.Version < 33
          ? [PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE, PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE]
          : []
        : [];

      if (permissions.length === 0) return true;

      let allGranted = true;
      for (const permission of permissions) {
        const result = await check(permission);
        if (result === RESULTS.DENIED) {
          const requestResult = await request(permission);
          if (requestResult !== RESULTS.GRANTED) {
            allGranted = false;
            Alert.alert('Permission Denied', 'Storage permission is required to save the report.');
          }
        } else if (result === RESULTS.BLOCKED) {
          allGranted = false;
          Alert.alert('Permission Blocked', 'Please enable storage permission in Settings.');
        }
      }
      return allGranted;
    } catch (error) {
      console.error('Error requesting storage permission:', error);
      Alert.alert('Error', 'Failed to request storage permission.');
      return false;
    }
  };

const requestCameraPermission = async () => {
  if (Platform.OS !== 'android') return true;
  try {
    const alreadyGranted = await PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.CAMERA);
    if (alreadyGranted) {
      return true;
    }

    const status = await PermissionsAndroid.request(PermissionsAndroid.PERMISSIONS.CAMERA, {
      title: 'Camera Permission Required',
      message: 'Titan Drilling needs access to your camera to capture incident photos.',
      buttonPositive: 'OK',
      buttonNegative: 'Cancel',
    });

    if (status === PermissionsAndroid.RESULTS.GRANTED) {
      return true;
    }

    if (status === PermissionsAndroid.RESULTS.NEVER_ASK_AGAIN) {
      Alert.alert(
        'Permission Required',
        'Camera permission has been permanently denied. Please enable it from Settings > Apps > Titan Drilling > Permissions.'
      );
    } else {
      Alert.alert('Permission Denied', 'Camera permission is required to take incident pictures.');
    }
    return false;
  } catch (error) {
    console.error('Error requesting camera permission:', error);
    Alert.alert('Error', 'Unable to request camera permission. Please try again.');
      return false;
    }
  };

  const uploadImagesToServer = async (imageUris, reportId) => {
    try {
      console.log('Starting image upload to server...');
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) throw new Error('User not authenticated');

      const formData = new FormData();
      
      // Add all images to formData with Android-specific handling
      imageUris.forEach((uri, index) => {
        if (uri) {
          // For Android, ensure proper file URI format
          let fileUri = uri;
          if (Platform.OS === 'android' && !uri.startsWith('file://')) {
            fileUri = 'file://' + uri;
          }
          
          formData.append('images', {
            uri: fileUri,
            type: 'image/jpeg',
            name: `incident_image_${index}_${Date.now()}.jpg`,
          });
        }
      });

      console.log(`Uploading ${imageUris.length} images to server...`);
      console.log('Platform:', Platform.OS);
      
      // Use fetchWithTimeout with 30 second timeout for image upload (larger files)
      const response = await fetchWithTimeout(
        `https://api.titandrillingzm.com:6007/incident-reports/${userUid}/${reportId}/upload-images`,
        {
          method: 'POST',
          body: formData,
          // Don't set Content-Type header - let fetch set it automatically with boundary
        },
        30000 // 30 second timeout for images
      );

      console.log('Image upload response status:', response.status);
      
      if (!response.ok) {
        throw new Error(`Server responded with status ${response.status}`);
      }

      const result = await response.json();
      console.log('Image upload result:', result);
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to upload images');
      }

      return result.imageUrls; // Return all uploaded image URLs
    } catch (error) {
      console.error('Error uploading images:', error);
      console.error('Error details:', {
        message: error.message,
        name: error.name,
        stack: error.stack
      });
      
      if (error.message.includes('timeout')) {
        throw new Error('Image upload timeout. Please check your internet connection and try again.');
      }
      
      if (error.message.includes('Network request failed')) {
        throw new Error('Network error during image upload. Please check:\n1. Internet connection\n2. Server is accessible\n3. File permissions');
      }
      
      throw error;
    }
  };

const preparePdfImageSources = async (imageUris = []) => {
  if (!Array.isArray(imageUris) || imageUris.length === 0) return [];

  const results = await Promise.all(
    imageUris.map(async (uri) => {
      if (!uri || typeof uri !== 'string') return null;
      try {
        let path = uri;
        if (path.startsWith('content://')) {
          try {
            const statResult = await RNFS.stat(path);
            path =
              statResult?.path ||
              statResult?.originalFilepath ||
              statResult?.target ||
              path;
          } catch (statError) {
            console.warn('Unable to resolve content URI for PDF image:', statError?.message || statError);
          }
        }

        if (path.startsWith('file://')) {
          path = path.replace('file://', '');
        }

        const base64Data = await RNFS.readFile(path, 'base64');
        return `data:image/jpeg;base64,${base64Data}`;
      } catch (error) {
        console.warn('Failed to convert image for PDF:', uri, error?.message || error);
        return null;
      }
    })
  );

  return results;
};

const generatePDFWithHtml = async (report, pdfImageSources = []) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const selectedIncidents = report.selectedIncidents || {};

      const formatDate = (date) => {
        if (!date) return 'N/A';
        return new Date(date).toLocaleDateString('en-GB');
      };

      const formatTime = (date) => {
        if (!date) return 'N/A';
        return new Date(date).toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit' });
      };

      // Types of Incident checkboxes (4 columns per row)
      const incidentTypes = [
        'Brake Failure', 'Chemicals Spill', 'Collision', 'Drowned in Water',
        'Electrical Failure', 'Engine Failure', 'Equipment Failure', 'Explosion/fire',
        'Fall Attachments', 'Fall from elevation', 'Leakage in Vehicle', 'Loss of Stability',
        'Oil Spill', 'Person hit by Vehicle', 'Steering Failure', 'Stuck',
        'Swamping', 'Tire Burst', 'Turned Over', 'Other'
      ];
      let incidentTypesHtml = '';
      for (let i = 0; i < incidentTypes.length; i += 4) {
        incidentTypesHtml += '<tr>';
        for (let j = 0; j < 4; j++) {
          const idx = i + j;
          if (idx < incidentTypes.length) {
            const type = incidentTypes[idx];
            incidentTypesHtml += `<td style="padding:2px 6px;"><input type="checkbox" ${selectedIncidents[type] ? 'checked' : ''}/> ${type}</td>`;
          } else {
            incidentTypesHtml += '<td></td>';
          }
        }
        incidentTypesHtml += '</tr>';
      }

      // Severity checkboxes
      const severityOptions = ['High', 'Medium', 'Low'];
      let severityHtml = '<tr>';
      severityOptions.forEach((option) => {
        severityHtml += `<td style="padding:2px 6px;"><input type="checkbox" ${step1.severity === option ? 'checked' : ''}/> ${option}</td>`;
      });
      severityHtml += '</tr>';

      // Images
      let imagesHtml = '';
      const imageUrls = step2.incidentImages || [];
      const base64Images = Array.isArray(pdfImageSources) ? pdfImageSources : [];
      if ((base64Images.length > 0 && base64Images.some(Boolean)) || (imageUrls.length > 0 && imageUrls.some(Boolean))) {
        imagesHtml = '<div style="display:flex;flex-wrap:wrap;gap:8px;">';
        for (let index = 0; index < Math.max(imageUrls.length, base64Images.length); index += 1) {
          const dataUri = base64Images[index];
          const remoteUri = imageUrls[index];
          const resolvedSrc = dataUri || remoteUri;
          if (resolvedSrc) {
            imagesHtml += `<div style="width:120px;height:90px;border:1px solid #ccc;display:flex;align-items:center;justify-content:center;margin:4px;"><img src="${resolvedSrc}" style="max-width:100%;max-height:100%;object-fit:contain;"/></div>`;
          }
        }
        imagesHtml += '</div>';
      } else {
        imagesHtml = '<div style="color:#888;">No Images</div>';
      }

      const userName = await AsyncStorage.getItem('userName') || 'N/A';

      // Helper to show dash instead of N/A or empty
      const dashIfNA = (val) => {
        if (val === undefined || val === null || val === '' || val === 'N/A') return '-';
        return val;
      };

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Incident Report</title>
          <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; }
            .container { max-width: 900px; margin: 0 auto; padding: 12px; }
            .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px; }
            .header h1 { font-size: 22px; font-weight: bold; margin: 0; }
            .logo { font-size: 22px; font-weight: bold; color: #fff;  border-radius: 50%; width: 150px; height: 48px; display: flex; align-items: center; justify-content: center; }
            .section-title { font-weight: bold; font-size: 14px; margin: 10px 0 4px; background-color: #e6e6e6; padding: 4px; border: 1px solid #ccc; }
            table { width: 100%; border-collapse: collapse; margin-bottom: 8px; }
            th, td { border: 1px solid #ccc; padding: 6px; text-align: left; font-size: 13px; }
            th { background-color: #f2f2f2; font-weight: bold; }
            .checkbox-table td { border: none; padding: 2px 6px; font-size: 13px; }
            .checkbox-table input[type="checkbox"] { margin-right: 4px; }
            .text-section { border: 1px solid #ccc; padding: 8px; min-height: 32px; font-size: 13px; margin-bottom: 8px; }
            .footer { text-align: center; margin-top: 10px; font-size: 11px; color: #000; border-top: 1px solid #000; padding-top: 6px; }
            .reported-by { text-align: right; font-size: 13px; margin-top: 8px; }
            .reported-by span { border-bottom: 1px solid #000; padding-bottom: 2px; }
            .logo img { width: 100%; height: 100%; object-fit: contain; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Incident Report</h1>
              <div class="logo">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo"/></div>
            </div>
            <table>
              <tr>
                <th colspan="2">Vehicle Details</th>
                <th colspan="2">Incident Details</th>
              </tr>
              <tr>
                <td>Vehicle Category</td>
                <td>${dashIfNA(step1.vehicleName)}</td>
                <td>Incident Number</td>
                <td>${dashIfNA(step1.incidentNumber)}</td>
              </tr>
              <tr>
                <td>Vehicle Name</td>
                <td>${dashIfNA(step1.vehicleName)}</td>
                <td>Incident Date</td>
                <td>${dashIfNA(formatDate(step1.incidentDate))}</td>
              </tr>
              <tr>
                <td>Vehicle #</td>
                <td>${dashIfNA(step1.vehicleId)}</td>
                <td>Incident Time</td>
                <td>${dashIfNA(formatTime(step1.incidentDate))}</td>
              </tr>
              <tr>
                <td>VIN #</td>
                <td>-</td>
                <td>Location</td>
                <td>${dashIfNA(step1.project)}</td>
              </tr>
              <tr>
                <td>Operator</td>
                <td>${dashIfNA(step1.incidentDate2)}</td>
                <td>Incident Area</td>
                <td>${dashIfNA(step1.incidentArea)}</td>
              </tr>
              <tr>
                <td>GPS Coordinates</td>
                <td colspan="3">${dashIfNA(step1.coordinates)}</td>
              </tr>
              <tr>
                <td>GPS Address</td>
                <td colspan="3">${dashIfNA(step1.gpsAddress)}</td>
              </tr>
            </table>
            <div class="section-title">Types of Incident</div>
            <table class="checkbox-table">${incidentTypesHtml}</table>
            <div class="section-title">Incident Severity Rating</div>
            <table class="checkbox-table">${severityHtml}</table>
            <div class="section-title">How incident occurred and what were the causes?</div>
            <div class="text-section">${step2.incidentCause || '-'}</div>
            <div class="section-title">What broke on the equipment, list all damages?</div>
            <div class="text-section">${step2.equipmentDamage || '-'}</div>
            <div class="section-title">Additional Comment</div>
            <div class="text-section">${step2.additionalComment || '-'}</div>
            <div class="section-title">Incident Images</div>
            ${imagesHtml}
            <div class="reported-by">
              <span>${userName}</span><br>
              Reported By
            </div>
            <div class="footer">
              Prepared By: Titan Drilling            </div>
          </div>
        </body>
        </html>
      `;

      const safeIncidentId = (report?.step1?.incidentNumber || report?.incidentNumber || 'unknown').toString().replace(/[^a-zA-Z0-9-_]/g, '_');
      const options = {
        html,
        fileName: `Incident_Report_${safeIncidentId}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const saveReportToMongoDB = async (userUid, report, pdfDownloadUrl) => {
    try {
      const reportData = {
        ...report,
        step2: {
          ...report.step2,
          ...(pdfDownloadUrl && { pdfDownloadUrl }),
        },
        // Ensure adminUid is set for permission filtering
        adminUid: userUid,
      };

      console.log('Saving report to MongoDB:', {
        id: reportData.id,
        incidentNumber: reportData.incidentNumber,
        adminUid: reportData.adminUid
      });

      // Use fetchWithTimeout with 15 second timeout
      const response = await fetchWithTimeout(
        `https://api.titandrillingzm.com:6007/incident-reports`,
        {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(reportData),
        },
        15000 // 15 second timeout
      );

      console.log('MongoDB save response status:', response.status);
      
      if (!response.ok) {
        throw new Error(`Server responded with status ${response.status}`);
      }
      
      const result = await response.json();
      console.log('MongoDB save result:', result);
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to save incident report');
      }
      
      console.log('Incident report saved successfully to MongoDB');
      
      if (pdfDownloadUrl) {
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Incident report saved successfully',
        });
      }
    } catch (error) {
      console.error('Error saving report to MongoDB:', error);
      if (error.message.includes('timeout')) {
        throw new Error('Database save timeout. Please check your internet connection and try again.');
      }
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: error.message || 'Failed to save incident report',
      });
      throw error;
    }
  };

  const updateReportWithFiles = async (userUid, incidentNumber, imageUrls, pdfDownloadUrl) => {
    try {
    const updateData = {};

    if (Array.isArray(imageUrls) && imageUrls.some((url) => typeof url === 'string' && url.trim().length > 0)) {
      const sanitizedImageUrls = imageUrls.filter((url) => typeof url === 'string' && url.trim().length > 0);
      if (sanitizedImageUrls.length > 0) {
        updateData['step2.incidentImages'] = sanitizedImageUrls;
      }
    }

    if (pdfDownloadUrl) {
      updateData.pdfDownloadUrl = pdfDownloadUrl;
      updateData['step2.pdfDownloadUrl'] = pdfDownloadUrl;
    }

      console.log('Updating report with files:', {
        incidentNumber,
        imageCount: imageUrls.filter(url => url).length,
        hasPdf: !!pdfDownloadUrl,
        pdfUrl: pdfDownloadUrl
      });

      // Use fetchWithTimeout with 15 second timeout
      const response = await fetchWithTimeout(
        `https://api.titandrillingzm.com:6007/incident-reports/${userUid}/${incidentNumber}`,
        {
          method: 'PUT',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(updateData),
        },
        15000 // 15 second timeout
      );

      console.log('Update response status:', response.status);
      
      if (!response.ok) {
        throw new Error(`Server responded with status ${response.status}`);
      }
      
      const result = await response.json();
      console.log('Update result:', result);
      
      if (!result.success) {
        throw new Error(result.error || 'Failed to update incident report');
      }
      
      console.log('Incident report updated successfully with files. PDF URL:', pdfDownloadUrl);
    } catch (error) {
      console.error('Error updating report with files:', error);
      if (error.message.includes('timeout')) {
        throw new Error('Report update timeout. Please check your internet connection and try again.');
      }
      throw error;
    }
  };

  // Send Incident notifications (push, in-app, email)
  const sendIncidentNotifications = async (incidentData, createdByName, reporterEmail, pdfUrl) => {
    try {
      console.log('📤 Sending incident notifications...');
      console.log('📋 Incident Number:', incidentData.incidentNumber);
      console.log('👤 Reporter:', createdByName);
      console.log('📧 Reporter Email:', reporterEmail);
      console.log('📎 PDF URL:', pdfUrl);
      
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout
      
      const response = await fetch(`${INCIDENT_NOTIFICATIONS_API_URL}/api/incident-notifications/new-incident`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          incidentData,
          createdByName,
          reporterEmail,
          pdfUrl,
        }),
        signal: controller.signal,
      });
      
      clearTimeout(timeoutId);
      
      if (response.ok) {
        const result = await response.json();
        console.log('✅ Incident notifications sent successfully:', result);
        console.log(`📊 Recipients: ${result.recipients?.total || 0} users notified`);
        console.log(`📱 Push: ${result.results?.push?.sent || 0} sent`);
        console.log(`📧 Emails: ${result.results?.emails?.sent || 0} sent`);
        return result;
      } else {
        const errorText = await response.text();
        console.error('❌ Incident notifications failed:', errorText);
        return { success: false, error: errorText };
      }
    } catch (error) {
      if (error.name === 'AbortError') {
        console.error('❌ Incident notifications request timed out');
      } else {
        console.error('❌ Error sending incident notifications:', error);
      }
      // Don't fail the report generation if notifications fail
      return { success: false, error: error.message };
    }
  };



  // Generate offline report (save locally without MongoDB)
  const generateOfflineReport = async () => {
    try {
      // Get user authentication
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        throw new Error('User not authenticated. Please log in again.');
      }

      // Load step 1 data from AsyncStorage
      const step1DataStr = await AsyncStorage.getItem('IncidentReportStep1Data');
      if (!step1DataStr) {
        throw new Error('Incident details not found. Please restart the report.');
      }
      const step1Data = JSON.parse(step1DataStr);
      const incidentNumber = step1Data.incidentNumber;
      if (!incidentNumber) {
        throw new Error('Incident number not available.');
      }

      // Load user permissions for offline storage
      let userPermissions = {};
      try {
        const cachedPermissions = await AsyncStorage.getItem('incidentModuleUserInfo');
        if (cachedPermissions) {
          userPermissions = JSON.parse(cachedPermissions);
        }
      } catch (permError) {
        console.warn('⚠️ Could not load permissions for offline report:', permError);
      }

      // Create flattened report structure
      const report = {
        id: incidentNumber,
        _id: incidentNumber,
        incidentNumber: step1Data.incidentNumber,
        incidentDate: step1Data.incidentDate,
        incidentArea: step1Data.incidentArea,
        severity: step1Data.severity,
        incidentCategory: step1Data.incidentCategory,
        country: step1Data.country,
        project: step1Data.project,
        vehicleId: step1Data.vehicleId,
        vehicleName: step1Data.vehicleName,
        personInvolved: step1Data.personInvolved,
        coordinates: step1Data.coordinates,
        gpsAddress: step1Data.gpsAddress,
        selectedIncidents: step1Data.selectedIncidents,
        involvesEquipment: step1Data.involvesEquipment,
        otherIncidentNote: step1Data.otherIncidentNote,
        createdAt: step1Data.createdAt || new Date().toISOString(),
        adminUid: userUid,
        // CRITICAL: Store permissions for offline access
        userPermissions: userPermissions.incidentReportPermissions || [],
        userCountries: userPermissions.countries || [],
        userProjects: userPermissions.projects || [],
        step1: step1Data,
        step2: {
          incidentImages: incidentImages,
          incidentCause,
          equipmentDamage,
          additionalComment,
          immediateCorrectiveAction,
        },
      };

      // Generate PDF locally
      console.log('📄 Generating PDF locally...');
      const pdfImageSources = await preparePdfImageSources(report.step2.incidentImages || []);
      const pdfPath = await generatePDFWithHtml({ 
        step1: step1Data, 
        step2: report.step2, 
        selectedIncidents: report.selectedIncidents 
      }, pdfImageSources);
      
      // Save PDF path in report
      report.step2.pdfLocalPath = pdfPath;
      // Persist local PDF path for offline access (store Android-safe URI with file:// prefix)
      try {
        const storedPath = Platform.OS === 'android' && !pdfPath.startsWith('file://') ? `file://${pdfPath}` : pdfPath;
        await AsyncStorage.setItem(`incident_pdf_local_${incidentNumber}`, storedPath);
      } catch (e) {
        console.warn('⚠️ Failed to persist local PDF path:', e?.message || e);
      }
      
      // Save to offline storage
      console.log('💾 Saving to offline storage...');
      console.log('   Incident Number:', incidentNumber);
      console.log('   Permissions stored:', report.userPermissions);
      console.log('   Countries stored:', report.userCountries);
      console.log('   Projects stored:', report.userProjects?.length || 0, 'projects');
      
      await OfflineIncidentHelper.saveOfflineReport(report);
      await OfflineIncidentHelper.saveOfflinePDF(incidentNumber, pdfPath);

      // Clear temporary data
      await AsyncStorage.removeItem('IncidentReportStep1Data');
      await AsyncStorage.removeItem('IncidentReportStep2State');

      console.log('✅ Offline incident report saved with permissions:', incidentNumber);

      // Show success message
      Toast.show({
        type: 'success',
        text1: 'Saved Offline',
        text2: `Incident ${incidentNumber} saved locally. Will sync when online.`,
        position: 'top',
        visibilityTime: 4000,
      });

      setTimeout(() => {
        Alert.alert(
          '📴 Saved Offline',
          `Incident report ${incidentNumber} has been saved locally.\n\nIt will automatically sync to MongoDB when you have internet connection.\n\nYou can view the PDF offline.`,
          [
            {
              text: 'View PDF',
              onPress: () => {
                const openUri = Platform.OS === 'android' && !pdfPath.startsWith('file://') ? `file://${pdfPath}` : pdfPath;
                navigation.navigate('PdfviewPage', {
                  pdfUrl: openUri,
                });
              }
            },
            {
              text: 'Go Back',
              onPress: () => {
                navigation.reset({
                  index: 1,
                  routes: [
                    { name: 'MainApp' },
                    { name: 'IncodentReport' },
                  ],
                });
              }
            }
          ]
        );
      }, 500);

    } catch (error) {
      console.error('❌ Error generating offline report:', error);
      throw error;
    }
  };

  const generateReport = async () => {
    if (!validateStep2()) return;

    const hasPermission = await requestStoragePermission();
    if (!hasPermission) return;

    setSyncModalVisible(true);
    
    // Step 1: Check network connectivity FIRST
    console.log('Checking network connectivity...');
    const netInfo = await NetInfo.fetch();
    const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
    
    if (!isOnline) {
      // OFFLINE MODE: Save locally
      console.log('📴 Offline mode detected - saving report locally...');
      try {
        await generateOfflineReport();
        setSyncModalVisible(false);
        return;
      } catch (offlineError) {
        console.error('❌ Offline save failed:', offlineError);
        setSyncModalVisible(false);
        Alert.alert(
          'Error',
          'Failed to save offline report. Please try again.',
          [{ text: 'OK' }]
        );
        return;
      }
    }
    
    // ONLINE MODE: Continue with MongoDB save
    try {
      try {
        await checkNetworkConnection();
        console.log('Network connection verified');
      } catch (networkError) {
        setSyncModalVisible(false);
        Alert.alert(
          'Network Error',
          networkError.message || 'No internet connection. Please check your network settings and try again.',
          [{ text: 'OK' }]
        );
        return;
      }

      // Step 2: Test server connection
      console.log('Testing server connection...');
      const serverTestUrl = 'https://api.titandrillingzm.com:6007/';
      const maxAttempts = 2;
      let serverReachable = false;
      let lastServerError = null;

      for (let attempt = 1; attempt <= maxAttempts; attempt += 1) {
        try {
          console.log(`🔍 Server connectivity test attempt ${attempt}/${maxAttempts}: ${serverTestUrl}`);
        const testResponse = await fetchWithTimeout(
            serverTestUrl,
          { method: 'GET' },
            15000
        );

        console.log('Server test response status:', testResponse.status);
        
          if (testResponse.status >= 500) {
            throw new Error(`Server responded with status ${testResponse.status}`);
        }

          serverReachable = true;
          break;
      } catch (serverError) {
          lastServerError = serverError;
          console.warn(`⚠️ Server connectivity test attempt ${attempt} failed:`, serverError?.message || serverError);
          await new Promise((resolve) => setTimeout(resolve, 1000 * attempt));
        }
      }

      const proceedOnlineFlow = async () => {
      // Step 3: Get user authentication
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        setSyncModalVisible(false);
        return;
      }

      // Load step 1 data from AsyncStorage
      const step1DataStr = await AsyncStorage.getItem('IncidentReportStep1Data');
      if (!step1DataStr) {
        Alert.alert('Error', 'Incident details not found. Please restart the report.');
        setSyncModalVisible(false);
        return;
      }
      const step1Data = JSON.parse(step1DataStr);
      const incidentNumber = step1Data.incidentNumber;
      if (!incidentNumber) {
        Alert.alert('Error', 'Incident number not available.');
        setSyncModalVisible(false);
        return;
      }
      setReportId(incidentNumber);

      // Create flattened report structure for home page compatibility
      const report = {
        id: incidentNumber,
        _id: incidentNumber,
        // Flatten step1 data to top level for home page compatibility
        incidentNumber: step1Data.incidentNumber,
        incidentDate: step1Data.incidentDate,
        incidentArea: step1Data.incidentArea,
        severity: step1Data.severity,
        incidentCategory: step1Data.incidentCategory,
        country: step1Data.country,
        project: step1Data.project,
        vehicleId: step1Data.vehicleId,
        vehicleName: step1Data.vehicleName,
        personInvolved: step1Data.personInvolved,
        coordinates: step1Data.coordinates,
        gpsAddress: step1Data.gpsAddress,
        selectedIncidents: step1Data.selectedIncidents,
        involvesEquipment: step1Data.involvesEquipment,
        otherIncidentNote: step1Data.otherIncidentNote,
        createdAt: step1Data.createdAt || new Date().toISOString(),
        // Keep nested structure for PDF generation
        step1: step1Data,
        step2: {
          incidentImages: Array(9).fill(null), // Initialize empty, will be updated after upload
          incidentCause,
          equipmentDamage,
          additionalComment,
          immediateCorrectiveAction,
        },
      };

      // First, save the report to MongoDB (without files)
      console.log('Saving initial report to MongoDB...');
      await saveReportToMongoDB(userUid, report, null);

      // Upload images to server (only upload non-null images)
      const imagesToUpload = incidentImages.filter(uri => uri !== null);
      let imageUrls = Array(9).fill(null);
      
      if (imagesToUpload.length > 0) {
        console.log('Uploading images to server...');
        const uploadedUrls = await uploadImagesToServer(imagesToUpload, incidentNumber);
        // Map uploaded URLs back to their original positions
        let urlIndex = 0;
        incidentImages.forEach((uri, index) => {
          if (uri) {
            imageUrls[index] = uploadedUrls[urlIndex];
            urlIndex++;
          }
        });
      }

      // Generate PDF
      console.log('Generating PDF...');
      const pdfImageSources = await preparePdfImageSources(incidentImages);
      const pdfPath = await generatePDFWithHtml({ 
        step1: step1Data, 
        step2: { ...report.step2, incidentImages: imageUrls }, 
        selectedIncidents: report.selectedIncidents 
      }, pdfImageSources);
      // Persist local PDF path for offline access (even in online flow)
      try {
        const storedPath = Platform.OS === 'android' && !pdfPath.startsWith('file://') ? `file://${pdfPath}` : pdfPath;
        await AsyncStorage.setItem(`incident_pdf_local_${incidentNumber}`, storedPath);
      } catch (e) {
        console.warn('⚠️ Failed to persist local PDF path (online flow):', e?.message || e);
      }
      
      // Upload PDF to server
      console.log('Uploading PDF to server...');
      console.log('PDF Path:', pdfPath);
      console.log('Platform:', Platform.OS);
      
      const formData = new FormData();
      
      // For Android, ensure proper file URI format
      let pdfUri = pdfPath;
      if (Platform.OS === 'android') {
        // Android needs file:// prefix
        if (!pdfPath.startsWith('file://')) {
          pdfUri = 'file://' + pdfPath;
        }
        console.log('Android PDF URI:', pdfUri);
      }
      
      formData.append('pdf', {
        uri: pdfUri,
        type: 'application/pdf',
        name: `IncidentReport_${incidentNumber}_${Date.now()}.pdf`,
      });

      console.log('FormData prepared for upload');

      // Use fetchWithTimeout with 30 second timeout for PDF upload
      const pdfResponse = await fetchWithTimeout(
        `https://api.titandrillingzm.com:6007/incident-reports/${userUid}/${incidentNumber}/upload-pdf`,
        {
          method: 'POST',
          body: formData,
          // Don't set Content-Type header - let the browser/fetch set it automatically with boundary
        },
        30000 // 30 second timeout for PDF
      );

      console.log('PDF upload response status:', pdfResponse.status);
      console.log('PDF upload response OK:', pdfResponse.ok);
      
      if (!pdfResponse.ok) {
        const errorText = await pdfResponse.text().catch(() => 'Unable to read error response');
        console.error('PDF upload error response:', errorText);
        throw new Error(`PDF upload failed with status ${pdfResponse.status}: ${errorText}`);
      }
      
      const pdfResult = await pdfResponse.json();
      console.log('PDF upload result:', pdfResult);
      
      if (!pdfResult.success) {
        throw new Error(pdfResult.error || 'Failed to upload PDF');
      }

      const pdfDownloadUrl =
        pdfResult.pdfUrl ||
        pdfResult.data?.pdfDownloadUrl ||
        pdfResult.data?.step2?.pdfDownloadUrl;

      if (!pdfDownloadUrl) {
        throw new Error('PDF upload succeeded but no download URL was returned by the server.');
      }
      console.log('PDF uploaded successfully. URL:', pdfDownloadUrl);

      // Update the report with final file URLs
      console.log('Updating report with file URLs...');
      await updateReportWithFiles(userUid, incidentNumber, imageUrls, pdfDownloadUrl);

      // Keep local PDF so it can be opened offline later
      // await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));

      // Send notifications after successful save (fire-and-forget)
      setTimeout(async () => {
        try {
          // Get reporter name and email
          const reporterName = step1Data.personInvolved || 'Reporter';
          
          // Get current user's email for confirmation
          let reporterEmail = '';
          try {
            const userData = await AsyncStorage.getItem('userData');
            if (userData) {
              const user = JSON.parse(userData);
              reporterEmail = user.email || '';
            }
          } catch (emailError) {
            console.warn('⚠️ Could not get reporter email:', emailError);
          }
          
          // Prepare incident data for notifications
          const incidentNotificationData = {
            _id: incidentNumber,
            id: incidentNumber,
            incidentNumber: incidentNumber,
            severity: step1Data.severity,
            incidentCategory: step1Data.incidentCategory,
            country: step1Data.country,
            project: step1Data.project,
            incidentArea: step1Data.incidentArea,
          };
          
          console.log('🔔 Sending incident notifications in background...');
          const notifResult = await sendIncidentNotifications(
            incidentNotificationData,
            reporterName,
            reporterEmail,
            pdfDownloadUrl
          );
          
          if (notifResult.success) {
            console.log(`✅ Incident notifications sent to ${notifResult.recipients?.total || 0} users`);
          } else {
            console.warn('⚠️ Incident notifications failed:', notifResult.error);
          }
        } catch (notifError) {
          console.error('❌ Background notification error:', notifError);
        }
      }, 500); // 500ms delay for background execution

      setSyncModalVisible(false);
      
      // Show success toast
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: 'Incident report generated and saved successfully',
      });
      
      setTimeout(() => {
        setPdfUrl(pdfDownloadUrl);
        setSuccessModalVisible(true);
      }, 200);

      // Clear persisted state
      await AsyncStorage.removeItem('IncidentReportStep1Data');
      await AsyncStorage.removeItem('IncidentReportStep2State');
      // Do NOT auto-navigate here; only show modal
      };

      if (!serverReachable) {
        console.error('❌ Server connectivity test failed after retries:', lastServerError);

        const errorMessage =
          lastServerError?.message ||
          'Cannot connect to server. Please ensure the API endpoint is reachable and try again.';

        setSyncModalVisible(false);

        Alert.alert(
          'Server Connection Issue',
          `${errorMessage}\n\nWould you like to retry?`,
          [
            {
              text: 'Cancel',
              style: 'cancel',
            },
            {
              text: 'Retry',
              onPress: () => {
                setTimeout(() => {
                  generateReport();
                }, 300);
              },
            },
          ]
        );
        return;
      }

      await proceedOnlineFlow();
    } catch (error) {
      console.error('Error generating report:', error);
      console.error('Error details:', {
        message: error.message,
        name: error.name,
        stack: error.stack,
        platform: Platform.OS
      });
      setSyncModalVisible(false);
      
      // Provide detailed error messages based on error type
      let errorTitle = 'Error';
      let errorMessage = 'Failed to generate or save report.';
      
      if (error.message.includes('timeout')) {
        errorTitle = 'Timeout Error';
        errorMessage = 'The operation timed out. Please check your internet connection and try again.';
      } else if (error.message.includes('Network request failed')) {
        errorTitle = 'Network Error';
        if (Platform.OS === 'android') {
          errorMessage = 'Network error on Android. Please check:\n1. Internet connection\n2. App permissions\n3. Server is accessible at https://api.titandrillingzm.com:6007\n4. Try restarting the app';
        } else {
          errorMessage = 'Network connection failed. Please check your internet and try again.';
        }
      } else if (error.message.includes('Network') || error.message.includes('internet')) {
        errorTitle = 'Network Error';
        errorMessage = error.message || 'Network connection issue. Please check your internet and try again.';
      } else if (error.message.includes('Server')) {
        errorTitle = 'Server Error';
        errorMessage = 'Cannot connect to server. Please try again later.';
      } else if (error.message.includes('authenticated')) {
        errorTitle = 'Authentication Error';
        errorMessage = 'Your session has expired. Please log in again.';
      } else {
        errorMessage = error.message || 'An unexpected error occurred. Please try again.';
      }
      
      Alert.alert(
        errorTitle,
        errorMessage,
        [
          { 
            text: 'OK',
            onPress: () => {
              // If it's a network error, keep the modal open so user can retry
              if (error.message.includes('timeout') || error.message.includes('Network')) {
                console.log('Network error - user can retry');
              }
            }
          }
        ]
      );
      
      Toast.show({
        type: 'error',
        text1: errorTitle,
        text2: errorMessage,
        position: 'top',
        visibilityTime: 4000,
      });
    } finally {
      setSyncModalVisible(false);
    }
  };

  // Only allow navigation from the modal's Back or View PDF buttons
  const navigateToIncidentHome = useCallback(() => {
    navigation.reset({
      index: 1,
      routes: [
        { name: 'MainApp' },
        { name: 'IncodentReport' },
      ],
    });
  }, [navigation]);

  const handleViewPDF = async (pdfUrl) => {
    if (!pdfUrl || !reportId) {
      Alert.alert('Error', 'PDF or Report ID not available.');
      return;
    }
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        return;
      }

      let targetUrl = pdfUrl;

      // If the remote PDF URL is missing or not reachable, fall back to the locally stored PDF path
      if (!targetUrl) {
        const localPath = await AsyncStorage.getItem(`incident_pdf_local_${reportId}`);
        if (localPath) {
          targetUrl = localPath;
        }
      }

      if (!targetUrl) {
        Alert.alert('Error', 'PDF file is not available. Please try regenerating the report.');
        return;
      }

      setSuccessModalVisible(false);

      navigation.navigate('PdfviewPage', {
        pdfUrl: targetUrl,
        originalPdfUrl: pdfUrl,
        reportId,
        onGoBack: () => {
          navigateToIncidentHome();
        },
      });
    } catch (error) {
      console.error('Error viewing PDF:', error);
      Alert.alert('Error', 'Could not view the PDF. Please try again.');
    }
  };

  const handleNavigateBack = () => {
    setSuccessModalVisible(false);
    navigateToIncidentHome();
  };

  useImperativeHandle(ref, () => ({
    validateStep2,
    getStep2Data: () => ({
      incidentImages,
      incidentCause,
      equipmentDamage,
      additionalComment,
      immediateCorrectiveAction,
    }),
    generatePDF: generateReport,
  }));

  const handleRemoveImage = (index) => {
    Alert.alert(
      'Remove Image',
      'Are you sure you want to remove this image?',
      [
        { text: 'Cancel', style: 'cancel' },
        { 
          text: 'Remove', 
          style: 'destructive',
          onPress: () => {
            const newImages = [...incidentImages];
            newImages[index] = null;
            setIncidentImages(newImages);
            saveState({ incidentImages: newImages });
          }
        },
      ],
      { cancelable: true }
    );
  };

  const handleFilePick = (index) => {
    Alert.alert(
      'Select Image',
      'Choose an option',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'Take Picture', onPress: () => pickImage(index, 'camera') },
        { text: 'Choose from Gallery', onPress: () => pickImage(index, 'gallery') },
      ],
      { cancelable: true }
    );
  };

  const pickImage = async (index, source) => {
    const options = {
      mediaType: 'photo',
      maxWidth: 500,
      maxHeight: 500,
      quality: 0.8,
    };

    const callback = (response) => {
      if (response.didCancel) {
        console.log('User cancelled image picker');
      } else if (response.errorCode) {
        console.error('Image Picker Error:', response.errorCode, response.errorMessage);
        if (response.errorCode === 'permission') {
          Alert.alert(
            'Permission Error',
            'Photo library access is denied. Please enable it in Settings > Privacy > Photos.'
          );
        } else if (response.errorCode === 'library') {
          Alert.alert('No Photos Available', 'No photos are available in your library.');
        } else {
          Alert.alert('Error', response.errorMessage || 'Failed to pick image');
        }
      } else if (response.assets && response.assets.length > 0) {
        const uri = response.assets[0].uri;
        const newImages = [...incidentImages];
        newImages[index] = uri;
        setIncidentImages(newImages);
        saveState({ incidentImages: newImages });
      } else {
        Alert.alert('No Selection', 'No image was selected.');
      }
    };

    if (source === 'camera') {
      const hasPermission = await requestCameraPermission();
      if (!hasPermission) {
        return;
      }
      launchCamera(options, callback);
    } else {
      launchImageLibrary(options, callback);
    }
  };

  const chunkArray = (arr, size) => {
    const result = [];
    for (let i = 0; i < arr.length; i += size) {
      result.push(arr.slice(i, i + size));
    }
    return result;
  };

  const renderStep2 = () => (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
      enabled
    >
      <ScrollView
        style={styles.formContainer}
        contentContainerStyle={{ paddingBottom: 100, flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={true}
        bounces={true}
        automaticallyAdjustKeyboardInsets={true}
      >
        <TouchableOpacity style={styles.incidentImagesButton}>
          <Text style={styles.incidentImagesButtonText}>Click Incident Images</Text>
        </TouchableOpacity>
        <View style={styles.incidentImagesGridWrapper}>
          <View style={styles.incidentImagesGrid}>
            {incidentImages.map((img, idx) => (
              <View key={idx} style={styles.incidentImageBox}>
                {img ? (
                  <View style={styles.imageContainer}>
                    <Image
                      source={{ uri: img }}
                      style={styles.incidentImage}
                      onError={(e) => console.log('Image error:', e.nativeEvent.error)}
                    />
                    <TouchableOpacity 
                      style={styles.removeImageButton} 
                      onPress={() => handleRemoveImage(idx)}
                    >
                      <MaterialIcons name="close" size={16} color="#fff" />
                    </TouchableOpacity>
                    <TouchableOpacity 
                      style={styles.cameraOverlay} 
                      onPress={() => handleFilePick(idx)}
                    >
                      <MaterialIcons name="camera-alt" size={24} color="#fff" />
                    </TouchableOpacity>
                  </View>
                ) : (
                  <TouchableOpacity style={styles.emptyImageBox} onPress={() => handleFilePick(idx)}>
                    <MaterialIcons name="camera-alt" size={36} color="#015185" />
                  </TouchableOpacity>
                )}
              </View>
            ))}
          </View>
        </View>
      </ScrollView>
    </KeyboardAvoidingView>
  );

  const renderStep3 = () => (
    <KeyboardAvoidingView
      style={{ flex: 1 }}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 64 : 0}
      enabled
    >
      <ScrollView
        style={{ flex: 1, backgroundColor: '#F5F6FA' }}
        contentContainerStyle={{ paddingBottom: 100, flexGrow: 1 }}
        keyboardShouldPersistTaps="handled"
        showsVerticalScrollIndicator={true}
        bounces={true}
        automaticallyAdjustKeyboardInsets={true}
      >
        <View style={styles.summaryCard}>
          <View style={styles.incidentCauseGroup}>
            <Text style={styles.incidentCauseLabel}>Description of incident</Text>
            <View style={styles.incidentCauseRelative}>
              <TextInput
                style={styles.incidentCauseInput}
                multiline
                value={incidentCause}
                onChangeText={(text) => { setIncidentCause(text); saveState({ incidentCause: text }); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <TouchableOpacity
                style={styles.incidentCauseMic}
                onPress={() => handleOpenVoiceModal('incidentCause')}
              >
                <MaterialIcons name="mic" size={24} color="#015185" />
              </TouchableOpacity>
            </View>
          </View>
          <View style={styles.equipmentDamageGroup}>
            <Text style={styles.equipmentDamageLabel}>Details of damage / injury / loss</Text>
            <View style={styles.equipmentDamageRelative}>
              <TextInput
                style={styles.equipmentDamageInput}
                multiline
                value={equipmentDamage}
                onChangeText={(text) => { setEquipmentDamage(text); saveState({ equipmentDamage: text }); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <TouchableOpacity
                style={styles.equipmentDamageMic}
                onPress={() => handleOpenVoiceModal('equipmentDamage')}
              >
                <MaterialIcons name="mic" size={24} color="#015185" />
              </TouchableOpacity>
            </View>
          </View>
          <View style={styles.additionalCommentGroup}>
            <Text style={styles.additionalCommentLabel}>Possible direct causes</Text>
            <View style={styles.additionalCommentRelative}>
              <TextInput
                style={styles.additionalCommentInput}
                multiline
                value={additionalComment}
                onChangeText={(text) => { setAdditionalComment(text); saveState({ additionalComment: text }); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <TouchableOpacity
                style={styles.additionalCommentMic}
                onPress={() => handleOpenVoiceModal('additionalComment')}
              >
                <MaterialIcons name="mic" size={24} color="#015185" />
              </TouchableOpacity>
            </View>
          </View>
          <View style={styles.immediateCorrectiveActionGroup}>
            <Text style={styles.immediateCorrectiveActionLabel}>Immediate Corrective Action taken</Text>
            <View style={styles.immediateCorrectiveActionRelative}>
              <TextInput
                style={styles.immediateCorrectiveActionInput}
                multiline
                value={immediateCorrectiveAction}
                onChangeText={(text) => { setImmediateCorrectiveAction(text); saveState({ immediateCorrectiveAction: text }); }}
                placeholder="Type here..."
                placeholderTextColor="#A0A0A0"
                textAlignVertical="top"
              />
              <TouchableOpacity
                style={styles.immediateCorrectiveActionMic}
                onPress={() => handleOpenVoiceModal('immediateCorrectiveAction')}
              >
                <MaterialIcons name="mic" size={24} color="#015185" />
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </ScrollView>
    </KeyboardAvoidingView>
  );

  return (
    <View style={styles.container}>
      {currentStep === 2 && renderStep3()}
      {currentStep === 3 && renderStep2()}
      <SyncModal visible={syncModalVisible} />
      <SuccessModal
        visible={successModalVisible}
        onClose={handleNavigateBack}
        onNavigateBack={handleNavigateBack}
        onViewPDF={handleViewPDF}
        pdfUrl={pdfUrl}
      />
      <VoiceToTextModal
        visible={voiceModalVisible}
        initialValue={voiceInitialValue}
        onCancel={handleVoiceCancel}
        onComplete={handleVoiceComplete}
      />
      <Toast />
    </View>
  );
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  formContainer: {
    flex: 1,
    backgroundColor: '#fff',
    padding: 15,
  },
  incidentImagesButton: {
    backgroundColor: '#015185',
    borderRadius: 4,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 8,
    marginBottom: 18,
    marginHorizontal: 15,
  },
  incidentImagesButtonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 14,
  },
  incidentImagesGridWrapper: {
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
  },
  incidentImagesGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'center',
    alignItems: 'center',
    width: 320, // or use width: '90%' for responsiveness
  },
  incidentImageBox: {
    width: 90,
    height: 90,
    margin: 8,
    backgroundColor: '#F5F8FA',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#B0C4DE',
    position: 'relative',
    overflow: 'hidden',
  },
  imageContainer: {
    position: 'relative',
    width: '100%',
    height: '100%',
  },
  incidentImage: {
    width: '100%',
    height: '100%',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#B0C4DE',
  },
  removeImageButton: {
    position: 'absolute',
    top: 5,
    right: 5,
    backgroundColor: 'rgba(255, 0, 0, 0.8)',
    borderRadius: 12,
    width: 24,
    height: 24,
    alignItems: 'center',
    justifyContent: 'center',
    zIndex: 2,
  },
  cameraOverlay: {
    position: 'absolute',
    bottom: 5,
    right: 5,
    backgroundColor: 'rgba(0, 0, 0, 0.6)',
    borderRadius: 12,
    padding: 5,
    zIndex: 1,
  },
  emptyImageBox: {
    width: '100%',
    height: '100%',
    backgroundColor: '#F5F8FA',
    borderRadius: 8,
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#B0C4DE',
  },
  summaryCard: {
    backgroundColor: '#fff',
    borderRadius: 14,
    margin: 18,
    padding: 18,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.08,
    shadowRadius: 8,
    elevation: 4,
  },
  incidentCauseGroup: {
    marginBottom: 18,
  },
  incidentCauseLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  incidentCauseRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  incidentCauseInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 16,
    paddingVertical: 14,
    fontSize: 15,
    color: '#222',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  incidentCauseMic: {
    position: 'absolute',
    right: 16,
    top: 14,
    padding: 6,
    borderRadius: 20,
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOpacity: 0.12,
    shadowRadius: 4,
    shadowOffset: { width: 0, height: 2 },
  },
  equipmentDamageGroup: {
    marginBottom: 18,
  },
  equipmentDamageLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  equipmentDamageRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  equipmentDamageInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 15,
    paddingVertical: 13,
    fontSize: 15,
    color: '#222',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  equipmentDamageMic: {
    position: 'absolute',
    right: 15,
    top: 13,
    padding: 6,
    borderRadius: 20,
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOpacity: 0.12,
    shadowRadius: 4,
    shadowOffset: { width: 0, height: 2 },
  },
  additionalCommentGroup: {
    marginBottom: 18,
  },
  additionalCommentLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  additionalCommentRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  additionalCommentInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 14,
    paddingVertical: 14,
    fontSize: 14,
    color: '#333',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  additionalCommentMic: {
    position: 'absolute',
    right: 14,
    top: 14,
    padding: 6,
    borderRadius: 20,
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOpacity: 0.12,
    shadowRadius: 4,
    shadowOffset: { width: 0, height: 2 },
  },
  immediateCorrectiveActionGroup: {
    marginBottom: 18,
  },
  immediateCorrectiveActionLabel: {
    fontSize: 15,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  immediateCorrectiveActionRelative: {
    position: 'relative',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 12,
    marginTop: 6,
    minHeight: 120,
  },
  immediateCorrectiveActionInput: {
    flex: 1,
    minHeight: 120,
    maxHeight: 150,
    paddingHorizontal: 14,
    paddingVertical: 14,
    fontSize: 14,
    color: '#333',
    backgroundColor: 'transparent',
    borderRadius: 12,
    textAlignVertical: 'top',
  },
  immediateCorrectiveActionMic: {
    position: 'absolute',
    right: 14,
    top: 14,
    padding: 6,
    borderRadius: 20,
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOpacity: 0.12,
    shadowRadius: 4,
    shadowOffset: { width: 0, height: 2 },
  },
  successOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.55)',
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 24,
  },
  successCard: {
    width: '90%',
    maxWidth: 420,
    backgroundColor: '#ffffff',
    borderRadius: 20,
    paddingHorizontal: 24,
    paddingVertical: 28,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 12 },
    shadowOpacity: 0.18,
    shadowRadius: 16,
    elevation: 12,
  },
  successIconWrap: {
    width: 84,
    height: 84,
    borderRadius: 42,
    backgroundColor: 'rgba(0, 184, 148, 0.12)',
    justifyContent: 'center',
    alignItems: 'center',
    marginBottom: 18,
  },
  successTitle: {
    fontSize: 20,
    fontWeight: '700',
    color: '#0055A5',
    textAlign: 'center',
    marginBottom: 8,
  },
  successMessage: {
    fontSize: 14,
    color: '#4A4A4A',
    textAlign: 'center',
    lineHeight: 20,
    marginBottom: 26,
    paddingHorizontal: 8,
  },
  successActions: {
    width: '100%',
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    gap: 12,
  },
  successButton: {
    flex: 1,
    backgroundColor: '#0055A5',
    borderRadius: 12,
    paddingVertical: 12,
    alignItems: 'center',
    justifyContent: 'center',
  },
  successButtonSecondary: {
    backgroundColor: '#ffffff',
    borderWidth: 1.5,
    borderColor: '#0055A5',
  },
  successButtonText: {
    fontSize: 15,
    fontWeight: '600',
    color: '#ffffff',
  },
  successButtonSecondaryText: {
    color: '#0055A5',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  voiceModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.7)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  voiceModalContent: {
    width: '85%',
    maxWidth: 420,
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
    elevation: 8,
  },
  voiceLottieContainer: {
    width: 200,
    height: 200,
    justifyContent: 'center',
    alignItems: 'center',
    marginBottom: 20,
  },
  voiceLottie: {
    width: '100%',
    height: '100%',
  },
  voiceStatusText: {
    minHeight: 50,
    textAlign: 'center',
    marginBottom: 20,
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
    paddingHorizontal: 16,
  },
  voiceVolumeIndicator: {
    width: '80%',
    height: 4,
    backgroundColor: '#e0e0e0',
    borderRadius: 2,
    marginBottom: 16,
    overflow: 'hidden',
  },
  voiceVolumeBar: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 2,
  },
  voiceModalButtons: {
    flexDirection: 'row',
    width: '100%',
    justifyContent: 'space-between',
    gap: 12,
    marginBottom: 16,
  },
  voiceModalButton: {
    flex: 1,
    paddingVertical: 12,
    backgroundColor: '#015185',
    borderRadius: 8,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  voiceCancelButton: {
    backgroundColor: '#fff',
    borderWidth: 2,
    borderColor: '#015185',
  },
  voiceModalButtonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 16,
  },
  voiceTranscriptContainer: {
    width: '100%',
    marginTop: 12,
  },
  voiceTranscriptLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 6,
  },
  voiceTranscriptBox: {
    maxHeight: 120,
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 10,
    padding: 12,
    backgroundColor: '#f8f9fa',
  },
  voiceTranscriptText: {
    fontSize: 14,
    color: '#333',
    lineHeight: 20,
  },
});

export default IncidentReportStep2;