import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  Dimensions,
  TouchableOpacity,
  ScrollView,
  Platform,
  Image,
  Modal,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import { MongoAPI } from '../../../ConfigMongo/MongoDBConfig';

const { width, height } = Dimensions.get('window');

const AdminHrApplyHome = () => {
  const navigation = useNavigation();
  const [userName, setUserName] = useState('Team Member');
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [hrPermissions, setHrPermissions] = useState([]);
  const [userCountries, setUserCountries] = useState([]);
  const [userProjects, setUserProjects] = useState([]);
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);

  // Fetch user data from Firebase and handle network status
  useEffect(() => {
    let unsubscribeNetInfo;

    const initialize = async () => {
      try {
        await checkNetworkStatus();
        await fetchUserData();

        unsubscribeNetInfo = NetInfo.addEventListener((state) => {
          const newOnlineStatus = state.isConnected || false;
          setIsOnline(newOnlineStatus);

          if (!newOnlineStatus) {
            showToast('info', 'Offline Mode', 'Please connect to the internet');
          } else {
            showToast('success', 'Online', 'Connected to the internet');
            fetchUserData(); // Re-fetch data when back online
          }
        });
      } catch (error) {
        console.error('Error during initialization:', error);
        showToast('error', 'Initialization Error', 'Failed to initialize application.');
      }
    };

    initialize();

    return () => {
      if (unsubscribeNetInfo) {
        unsubscribeNetInfo();
      }
    };
  }, []);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      setIsOnline(netInfo.isConnected || false);
      return netInfo.isConnected || false;
    } catch (error) {
      console.error('Error checking network status:', error);
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

  const fetchUserData = async () => {
    try {
      // Retrieve adminUid from AsyncStorage
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Authentication Error', 'User ID not found. Please log in again.');
        navigation.reset({
          index: 0,
          routes: [{ name: 'Login' }],
        });
        return;
      }

      if (!isOnline) {
        // Load cached user data when offline
        const cachedUserData = await AsyncStorage.getItem('userData');
        if (cachedUserData) {
          const userData = JSON.parse(cachedUserData);
          setUserName(userData.name || 'Team Member');
          setHrPermissions(userData.hrPermissions || []);
          setUserCountries(userData.countries || []);
          setUserProjects(userData.projects || []);
          showToast('info', 'Offline Mode', 'Loaded cached user data');
        } else {
          showToast('info', 'Offline Mode', 'No cached user data available');
        }
        return;
      }

      // Fetch from MongoDB
      const response = await MongoAPI.getUserById(adminUid);
      console.log('📡 MongoDB user response:', response);

      if (response.success && response.data) {
        const userData = response.data;
        
        setUserName(userData.name || 'Team Member');
        setUserCountries(userData.countries || []);
        setUserProjects(userData.projects || []);

        // Normalize modules to array
        const modules = Array.isArray(userData.modules) ? userData.modules : 
                        (userData.modules ? Object.values(userData.modules) : []);

        console.log('📦 All modules:', modules);

        // Find hrForm or hrApplications module
        const hrModule = modules.find((mod) => 
          mod.module && (
            mod.module === 'hrForm' || 
            mod.module === 'hrApplications' ||
            mod.module.toLowerCase() === 'hrform' ||
            mod.module.toLowerCase() === 'hrapplications'
          )
        );

        console.log('📋 HR module found:', hrModule);

        const permissions = hrModule?.permissions || [];
        setHrPermissions(permissions);

        console.log('✅ HR Permissions loaded:', permissions);

        // Cache data in AsyncStorage
        await AsyncStorage.setItem('userData', JSON.stringify({
          name: userData.name,
          employeeNumber: userData.employeeNumber,
          countries: userData.countries,
          projects: userData.projects,
          hrPermissions: permissions,
          department: userData.department,
          nrcIdNumber: userData.nrcIdNumber,
        }));
      } else {
        showToast('info', 'Data Warning', 'User data not found in database.');
        setUserName('Team Member');
      }
    } catch (error) {
      console.error('Error fetching user data:', error);
      showToast('error', 'Error', 'Failed to load user data.');
    }
  };

  const handleLogout = async () => {
    try {
      await AsyncStorage.multiRemove(['userUid', 'userData']);
      showToast('success', 'Success', 'Logged out successfully.');
      navigation.reset({
        index: 0,
        routes: [{ name: 'AdminLogin' }],
      });
    } catch (error) {
      console.error('Error during logout:', error);
      showToast('error', 'Error', 'Failed to log out. Please try again.');
    }
  };

  const showToast = (type, title, message) => {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <TouchableOpacity style={styles.backButton} onPress={() => navigation.goBack()}>
          <MaterialIcons name="arrow-back-ios" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>HR Application</Text>
        <TouchableOpacity 
          style={styles.menuButton} 
          onPress={() => setPermissionModalVisible(true)}
        >
          <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
      </LinearGradient>

      {/* Menu Dropdown */}
      {isMenuOpen && (
        <View style={styles.menuDropdown}>
          <TouchableOpacity
            style={styles.menuItem}
            onPress={() => {
              setIsMenuOpen(false);
              navigation.navigate('AdminHrAppliApply');
            }}
          >
            <Text style={styles.menuItemText}>Apply for Leave/Grant</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.menuItem}
            onPress={() => {
              setIsMenuOpen(false);
              // navigation.navigate('TeamHRpreviousAppli');
               navigation.navigate('AdminHrAppliPastView');
              
            }}
          >
            <Text style={styles.menuItemText}>View Past Applications</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.menuItem, styles.logoutItem]}
            onPress={() => {
              setIsMenuOpen(false);
              handleLogout();
            }}
          >
            <Text style={[styles.menuItemText, styles.logoutText]}>Logout</Text>
          </TouchableOpacity>
        </View>
      )}

      {/* Main Content */}
      <LinearGradient
        colors={['#f8f8f8', '#e8ecef']}
        style={styles.contentContainer}
        start={{ x: 0, y: 0 }}
        end={{ x: 0, y: 1 }}
      >
        <ScrollView contentContainerStyle={styles.content}>
          {/* Welcome Section */}
          <View style={styles.welcomeContainer}>
            <Text style={styles.welcomeText}>Welcome, {userName}!</Text>
            <Text style={styles.subText}>
              Manage your HR applications with Team HR.
            </Text>
          </View>

          {/* Quick Actions */}
          <View style={styles.actionsContainer}>
            <Text style={styles.sectionTitle}>Quick Actions</Text>
            <View style={styles.actionsGrid}>
              <TouchableOpacity
                style={styles.actionCard}
                onPress={() => navigation.navigate('AdminHrAppliApply')}
              >
                <MaterialIcons name="note-add" size={width * 0.08} color="#025383" />
                <Text style={styles.actionText}>Apply for Leave/Grant</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.actionCard}
                // onPress={() => navigation.navigate('TeamHRpreviousAppli')}
                onPress={() => navigation.navigate('AdminHrAppliPastView')}
                
              >
                <MaterialIcons name="history" size={width * 0.08} color="#025383" />
                <Text style={styles.actionText}>View Past Applications</Text>
              </TouchableOpacity>
            </View>
          </View>

          {/* HR Announcements */}
          <View style={styles.announcementsContainer}>
            <Text style={styles.sectionTitle}>HR Announcements</Text>
            <View style={styles.announcementCard}>
              <Text style={styles.announcementText}>
                Reminder: Submit your annual leave applications in testing.
              </Text>
            </View>
            <View style={styles.announcementCard}>
              <Text style={styles.announcementText}>
                New Policy Update: Funeral grant applications now require additional documentation.
              </Text>
            </View>
          </View>

          {/* Footer */}
          <View style={styles.footer}>
            <Text style={styles.footerText}>Team HR © 2025</Text>
          </View>
        </ScrollView>
      </LinearGradient>

      {/* Permission Modal */}
      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <View style={styles.permissionModalOverlay}>
          <View style={styles.permissionModalContent}>
            <Text style={styles.permissionModalTitle}>User Permissions</Text>
            
            <Text style={styles.permissionModalLabel}>HR Application Permissions:</Text>
            <Text style={styles.permissionModalValue}>
              {hrPermissions && hrPermissions.length > 0 ? hrPermissions.join(', ') : 'None'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Countries:</Text>
            <Text style={styles.permissionModalValue}>
              {userCountries && userCountries.length > 0 ? userCountries.join(', ') : 'None'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Projects:</Text>
            <Text style={styles.permissionModalValue}>
              {userProjects && userProjects.length > 0 ? userProjects.join(', ') : 'None'}
            </Text>
            
            <TouchableOpacity
              style={styles.permissionCloseButton}
              onPress={() => setPermissionModalVisible(false)}
            >
              <Text style={styles.permissionCloseButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f8f8',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? height * 0.05 : height * 0.02,
    paddingBottom: height * 0.02,
  },
  backButton: {
    padding: width * 0.02,
    top:"28%"
  },
  headerTitle: {
    fontSize: width * 0.05,
    color: '#fff',
    fontWeight: 'bold',
    top:"28%"
  },
  menuButton: {
    padding: width * 0.02,
  },
  menuDropdown: {
    position: 'absolute',
    top: Platform.OS === 'ios' ? height * 0.12 : height * 0.09,
    right: width * 0.04,
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    zIndex: 1000,
  },
  menuItem: {
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.05,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  menuItemText: {
    fontSize: width * 0.04,
    color: '#333',
    fontWeight: '500',
  },
  logoutItem: {
    borderBottomWidth: 0,
  },
  logoutText: {
    color: '#d32f2f',
  },
  contentContainer: {
    flex: 1,
  },
  content: {
    flexGrow: 1,
    padding: width * 0.04,
  },
  welcomeContainer: {
    alignItems: 'center',
    marginBottom: width * 0.06,
    marginTop: width * 0.04,
  },
  welcomeText: {
    fontSize: width * 0.06,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.02,
  },
  subText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    paddingHorizontal: width * 0.05,
  },
  actionsContainer: {
    marginBottom: width * 0.06,
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.03,
  },
  actionsGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
  },
  actionCard: {
    width: '48%',
    backgroundColor: '#fff',
    borderRadius: width * 0.03,
    padding: width * 0.04,
    alignItems: 'center',
    marginBottom: width * 0.04,
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  actionText: {
    fontSize: width * 0.04,
    color: '#333',
    marginTop: width * 0.02,
    textAlign: 'center',
    fontWeight: '500',
  },
  announcementsContainer: {
    marginBottom: width * 0.06,
  },
  announcementCard: {
    backgroundColor: '#fff',
    borderRadius: width * 0.03,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 3,
  },
  announcementText: {
    fontSize: width * 0.038,
    color: '#333',
    lineHeight: width * 0.055,
  },
  footer: {
    alignItems: 'center',
    marginTop: width * 0.06,
    marginBottom: width * 0.04,
  },
  footerText: {
    fontSize: width * 0.035,
    color: '#666',
  },
  permissionModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  permissionModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '80%',
    alignItems: 'flex-start',
    elevation: 5,
  },
  permissionModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 16,
    alignSelf: 'center',
    width: '100%',
    textAlign: 'center',
  },
  permissionModalLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
    marginTop: 10,
  },
  permissionModalValue: {
    fontSize: 14,
    color: '#333',
    marginBottom: 4,
  },
  permissionCloseButton: {
    alignSelf: 'center',
    marginTop: 18,
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 24,
    paddingVertical: 8,
  },
  permissionCloseButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
});

export default AdminHrApplyHome;