import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  FlatList,
  TouchableOpacity,
  ActivityIndicator,
  Dimensions,
  SafeAreaView,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import moment from 'moment';
import NetInfo from '@react-native-community/netinfo';

const { width, height } = Dimensions.get('window');

const AdminHrAppliPastView = () => {
  const navigation = useNavigation();
  const [applications, setApplications] = useState([]);
  const [loading, setLoading] = useState(true);
  const [isOnline, setIsOnline] = useState(true);

  // Helper function to show toast messages
  const showToast = (type, title, message) => {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Monitor network status
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected || false);
    });
    return () => unsubscribe();
  }, []);

  // Fetch applications from Firebase and AsyncStorage
  useEffect(() => {
    const fetchApplications = async () => {
      setLoading(true);
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          showToast('error', 'Authentication Error', 'User ID not found. Please log in again.');
          navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
          return;
        }

        const currentUser = auth().currentUser;
        if (!currentUser || currentUser.uid !== userUid) {
          showToast('error', 'Authentication Error', 'Invalid session. Please log in again.');
          await AsyncStorage.multiRemove(['userUid', 'userData']);
          navigation.reset({ index: 0, routes: [{ name: 'Login' }] });
          return;
        }

        let allApplications = [];

        if (!isOnline) {
          const cachedApplications = await AsyncStorage.getItem('offlineApplications');
          if (cachedApplications) {
            const parsedApplications = JSON.parse(cachedApplications);
            allApplications = parsedApplications.filter((app) => app.userUid === userUid);
            showToast('info', 'Offline Mode', 'Loaded cached applications.');
          } else {
            showToast('warning', 'Offline Mode', 'No cached applications available.');
          }
        } else {
          const applicationTypes = [
            'leave_application',
            'funeral_grant_form',
            'salary_advance_application',
            'application_for_appeal',
            'grievance_record',
            'hospital_attendance_form',
          ];

          for (const appType of applicationTypes) {
            const applicationRef = database().ref(`/GlobalHrApplications/admins/${appType}/${userUid}`);
            const snapshot = await applicationRef.once('value');
            const data = snapshot.val();
            if (data) {
              const appList = Object.entries(data).map(([key, value]) => ({
                id: key,
                applicationType: appType,
                ...value,
              }));
              allApplications = [...allApplications, ...appList];
            }
          }

          // Cache applications
          await AsyncStorage.setItem('offlineApplications', JSON.stringify(allApplications));
        }

        // Sort applications by submission date (newest first)
        allApplications.sort((a, b) => moment(b.submittedAt).valueOf() - moment(a.submittedAt).valueOf());
        setApplications(allApplications);

        if (allApplications.length === 0) {
          showToast('info', 'No Applications', 'No past applications found.');
        }
      } catch (error) {
        console.error('Error fetching applications:', error);
        const cachedApplications = await AsyncStorage.getItem('offlineApplications');
        if (cachedApplications) {
          const parsedApplications = JSON.parse(cachedApplications);
          setApplications(parsedApplications.filter((app) => app.userUid === userUid));
          showToast('warning', 'Offline Mode', 'Loaded cached applications due to network error.');
        } else {
          showToast('error', 'Error', 'Failed to load applications. Please check your connection.');
        }
      } finally {
        setLoading(false);
      }
    };

    fetchApplications();
  }, [isOnline, navigation]);

  // Group applications by time period
  const groupApplicationsByTime = () => {
    const today = moment().startOf('day');
    const yesterday = moment().subtract(1, 'days').startOf('day');
    const startOfWeek = moment().startOf('week');
    const startOfPreviousWeeks = moment().subtract(1, 'weeks').startOf('week');

    const grouped = {
      Today: [],
      Yesterday: [],
      'This Week': [],
      'Previous Weeks': [],
    };

    applications.forEach((app) => {
      const appDate = moment(app.submittedAt);
      if (appDate.isSame(today, 'day')) {
        grouped.Today.push(app);
      } else if (appDate.isSame(yesterday, 'day')) {
        grouped.Yesterday.push(app);
      } else if (appDate.isSameOrAfter(startOfWeek)) {
        grouped['This Week'].push(app);
      } else {
        grouped['Previous Weeks'].push(app);
      }
    });

    return grouped;
  };

  const groupedApplications = groupApplicationsByTime();

  const applicationTypeLabels = {
    leave_application: 'Leave Application',
    funeral_grant_form: 'Funeral Grant Form',
    salary_advance_application: 'Salary Advance Application',
    application_for_appeal: 'Application for Appeal',
    grievance_record: 'Grievance Record',
    hospital_attendance_form: 'Hospital Attendance Form',
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Approved':
        return '#34C759';
      case 'Rejected':
        return '#FF3B30';
      default:
        return '#0288D1';
    }
  };

  const formatDate = (dateString) => {
    if (!dateString) return 'N/A';
    return moment(dateString).format('MMMM Do YYYY, h:mm a');
  };

  const renderApplicationItem = ({ item }) => (
    <TouchableOpacity
      style={styles.applicationCard}
      onPress={() =>
        navigation.navigate('ApplicationDetail', {
          application: item,
          applicationType: item.applicationType,
        })
      }
      accessibilityLabel={`View ${applicationTypeLabels[item.applicationType] || item.applicationType} application`}
      accessibilityRole="button"
    >
      <View style={styles.cardHeader}>
        <Text style={styles.cardTitle}>
          {applicationTypeLabels[item.applicationType] || item.applicationType}
        </Text>
        <View style={[styles.statusBadge, { backgroundColor: getStatusColor(item.status) }]}>
          <Text style={styles.cardStatus}>{item.status || 'Pending'}</Text>
        </View>
      </View>
      <Text style={styles.cardDate}>{formatDate(item.submittedAt)}</Text>
      <View style={styles.cardDetails}>
        <Text style={styles.cardDetailText}>Employee: {item.fullName || 'N/A'}</Text>
        <Text style={styles.cardDetailText}>Employee No: {item.employeeNo || 'N/A'}</Text>
        {item.applicationType === 'leave_application' && (
          <>
            <Text style={styles.cardDetailText}>Start: {formatDate(item.leaveStartDate)}</Text>
            <Text style={styles.cardDetailText}>End: {formatDate(item.leaveEndDate)}</Text>
            <Text style={styles.cardDetailText}>Reason: {item.reasonOfLeave || 'N/A'}</Text>
          </>
        )}
        {item.applicationType === 'funeral_grant_form' && (
          <>
            <Text style={styles.cardDetailText}>Deceased: {item.deceasedName || 'N/A'}</Text>
            <Text style={styles.cardDetailText}>Relationship: {item.deceasedRelationship || 'N/A'}</Text>
            <Text style={styles.cardDetailText}>
              Total: K{' '}
              {item.selectedFuneralRelationshipType === 'spouse_child' ||
              item.selectedFuneralRelationshipType === 'both'
                ? (item.funeralSpouseChildData
                    ?.reduce((sum, curr) => sum + (parseFloat(curr.amount) || 0), 0)
                    .toFixed(2) || '0.00')
                : item.selectedFuneralRelationshipType === 'mother_father' ||
                  item.selectedFuneralRelationshipType === 'both'
                ? (item.funeralMotherFatherData
                    ?.reduce((sum, curr) => sum + (parseFloat(curr.amount) || 0), 0)
                    .toFixed(2) || '0.00')
                : '0.00'}
            </Text>
          </>
        )}
        {item.applicationType === 'salary_advance_application' && (
          <>
            <Text style={styles.cardDetailText}>Amount: K {item.salaryAdvanceAmount || 'N/A'}</Text>
            <Text style={styles.cardDetailText}>Sites: {item.selectedSites?.join(', ') || 'N/A'}</Text>
            <Text style={styles.cardDetailText}>Reason: {item.selectedReasons?.join(', ') || 'N/A'}</Text>
          </>
        )}
        {item.applicationType === 'application_for_appeal' && (
          <>
            <Text style={styles.cardDetailText}>Offence: {item.appealOffence || 'N/A'}</Text>
            <Text style={styles.cardDetailText}>Sanction: {item.appealSanction || 'N/A'}</Text>
          </>
        )}
        {item.applicationType === 'grievance_record' && (
          <>
            <Text style={styles.cardDetailText}>
              Complainant: {item.grievanceComplaints?.[0]?.name || 'N/A'}
            </Text>
            <Text style={styles.cardDetailText}>Place: {item.grievancePlaceOfWork || 'N/A'}</Text>
          </>
        )}
        {item.applicationType === 'hospital_attendance_form' && (
          <>
            <Text style={styles.cardDetailText}>Clinics: {item.selectedClinics?.join(', ') || 'N/A'}</Text>
            <Text style={styles.cardDetailText}>Shift: {item.selectedShift || 'N/A'}</Text>
          </>
        )}
      </View>
    </TouchableOpacity>
  );

  const renderSection = (title, data) => {
    if (data.length === 0) return null;
    return (
      <View style={styles.sectionContainer}>
        <Text style={styles.sectionTitle}>{title}</Text>
        <FlatList
          data={data}
          renderItem={renderApplicationItem}
          keyExtractor={(item) => item.id}
          showsVerticalScrollIndicator={false}
          nestedScrollEnabled
        />
      </View>
    );
  };

  return (
    <SafeAreaView style={styles.container}>
      <LinearGradient
        colors={['#0288D1', '#01579B']}
        style={styles.header}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <TouchableOpacity
          style={styles.backButton}
          onPress={() => navigation.goBack()}
          accessibilityLabel="Go back"
          accessibilityRole="button"
        >
          <MaterialIcons name="arrow-back-ios" size={width * 0.06} color="#fff" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>All Applications</Text>
        <View style={styles.placeholder} />
      </LinearGradient>

      {loading ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#0288D1" />
          <Text style={styles.loadingText}>Loading applications...</Text>
        </View>
      ) : applications.length === 0 ? (
        <View style={styles.emptyContainer}>
          <MaterialIcons name="inbox" size={width * 0.15} color="#6B7280" />
          <Text style={styles.emptyText}>No applications found</Text>
        </View>
      ) : (
        <FlatList
          contentContainerStyle={styles.content}
          data={[{ id: 'sections' }]}
          renderItem={() => (
            <>
              {renderSection('Today', groupedApplications.Today)}
              {renderSection('Yesterday', groupedApplications.Yesterday)}
              {renderSection('This Week', groupedApplications['This Week'])}
              {renderSection('Previous Weeks', groupedApplications['Previous Weeks'])}
            </>
          )}
          keyExtractor={(item) => item.id}
          showsVerticalScrollIndicator={false}
        />
      )}
    </SafeAreaView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#F4F6F8',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingVertical: height * 0.02,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  headerTitle: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: '700',
    color: '#fff',
    textAlign: 'center',
  },
  backButton: {
    padding: width * 0.02,
  },
  placeholder: {
    width: width * 0.06,
  },
  content: {
    padding: width * 0.04,
    paddingBottom: height * 0.1,
  },
  sectionContainer: {
    marginBottom: height * 0.03,
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: '700',
    color: '#1A1A1A',
    marginBottom: height * 0.015,
  },
  applicationCard: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: width * 0.04,
    marginBottom: height * 0.02,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  cardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: height * 0.01,
  },
  cardTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#1A1A1A',
    flex: 1,
  },
  statusBadge: {
    borderRadius: 12,
    paddingVertical: height * 0.005,
    paddingHorizontal: width * 0.03,
  },
  cardStatus: {
    fontSize: width * 0.035,
    fontWeight: '600',
    color: '#fff',
  },
  cardDate: {
    fontSize: width * 0.035,
    fontWeight: '400',
    color: '#6B7280',
    marginBottom: height * 0.01,
  },
  cardDetails: {
    borderTopWidth: 1,
    borderTopColor: '#E0E0E0',
    paddingTop: height * 0.01,
  },
  cardDetailText: {
    fontSize: width * 0.038,
    fontWeight: '400',
    color: '#1A1A1A',
    marginBottom: height * 0.005,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    fontSize: width * 0.04,
    fontWeight: '400',
    color: '#4A4A4A',
    marginTop: height * 0.02,
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  emptyText: {
    fontSize: width * 0.045,
    fontWeight: '400',
    color: '#6B7280',
    marginTop: height * 0.02,
  },
});

export default AdminHrAppliPastView;