import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, Dimensions, SafeAreaView, StatusBar, Platform, ScrollView } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import ApplicationList from './ApplicationList';

const { width } = Dimensions.get('window');

// MongoDB API URL
const MONGODB_HRFORM_API_URL = 'https://api.titandrillingzm.com:6013';

const AdminSalaryAdv = () => {
  const navigation = useNavigation();
  const [applications, setApplications] = useState([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isOnline, setIsOnline] = useState(true);
  const [allowedCountries, setAllowedCountries] = useState([]);
  const [allowedProjects, setAllowedProjects] = useState([]);
  const [hrPermissions, setHrPermissions] = useState([]);
  const [permissionsLoaded, setPermissionsLoaded] = useState(false);

  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Fetch user permissions on mount
  useEffect(() => {
    const fetchUserPermissions = async () => {
      try {
        // Get user data from AsyncStorage (set during login)
        const userData = await AsyncStorage.getItem('userData');
        if (!userData) {
          console.log('❌ No user data found in AsyncStorage');
          return;
        }

        const user = JSON.parse(userData);
        console.log('👤 User data from AsyncStorage for permissions:', user);

        // Get userUid from AsyncStorage or use user._id
        let userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          userUid = user._id || user.id;
          if (userUid) {
            await AsyncStorage.setItem('userUid', userUid);
          }
        }

        if (!userUid) {
          console.log('❌ No userUid found');
          return;
        }

        console.log('🔍 Using userUid for permissions:', userUid);
        
        setAllowedCountries(Array.isArray(user.countries) ? user.countries : (user.countries ? [user.countries] : []));
        setAllowedProjects(Array.isArray(user.projects) ? user.projects : (user.projects ? [user.projects] : []));
        
        // Normalize modules to array
        const modules = Array.isArray(user.modules) ? user.modules : 
                        (user.modules ? Object.values(user.modules) : []);

        console.log('📦 All modules:', modules);

        // Get HR permissions
        const hrModule = modules.find((mod) => 
          mod.module && (mod.module === 'hrForm' || mod.module.toLowerCase() === 'hrform')
        );
        
        console.log('📋 HR Form module:', hrModule);
        
        const permissions = hrModule?.permissions || [];
        setHrPermissions(permissions);
        
        // Debug log
        console.log('✅ HR Permissions loaded:', permissions);
        console.log('🔒 onlyMineView present:', permissions.includes('onlyMineView'));
        console.log('👁️ view present:', permissions.includes('view'));
        console.log('🌟 all present:', permissions.includes('all'));
        
        // Mark permissions as loaded
        setPermissionsLoaded(true);
      } catch (e) {
        console.error('❌ Error fetching user permissions:', e);
        // Still mark as loaded to prevent infinite waiting
        setPermissionsLoaded(true);
      }
    };
    fetchUserPermissions();
  }, []);

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected || false);
    });

    return () => unsubscribe();
  }, []);

  useEffect(() => {
    const fetchApplications = async () => {
      // Wait for permissions to be loaded
      if (!permissionsLoaded) {
        return; // Don't fetch until permissions are loaded
      }
      
      setIsLoading(true);
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          showToast('error', 'Authentication Error', 'User ID not found. Please log in again.');
          setIsLoading(false);
          return;
        }

        console.log('📋 Fetching salary advance applications from MongoDB...');
        console.log('🔍 Filters:', {
          countries: allowedCountries,
          projects: allowedProjects,
          permissions: hrPermissions
        });

        if (!isOnline) {
          const cachedApplications = await AsyncStorage.getItem('offlineApplications');
          if (cachedApplications) {
            const parsedApplications = JSON.parse(cachedApplications);
            let filteredApplications = parsedApplications.filter(
              (app) => app.applicationType === 'salary_advance_application'
            );

            // Apply onlyMineView filter
            if (hrPermissions.includes('onlyMineView')) {
              console.log('Filtering salary advance applications: onlyMineView mode');
              filteredApplications = filteredApplications.filter(app => app.userUid === userUid);
            } else {
              console.log('Filtering salary advance applications: country/project mode');
              // Apply country/project filter
              filteredApplications = filteredApplications.filter(app => 
                (allowedCountries.length === 0 || allowedCountries.includes(app.country)) &&
                (allowedProjects.length === 0 || allowedProjects.includes(app.project))
              );
            }

            setApplications(filteredApplications);
            showToast('info', 'Offline Mode', 'Loaded cached salary advance applications.');
          } else {
            showToast('info', 'Offline Mode', 'No cached salary advance applications available.');
          }
          setIsLoading(false);
          return;
        }

        // Fetch applications from MongoDB
        const queryParams = new URLSearchParams({
          countries: allowedCountries.join(','),
          projects: allowedProjects.join(','),
          onlyMineView: hrPermissions.includes('onlyMineView').toString()
        });

        const response = await fetch(`${MONGODB_HRFORM_API_URL}/hr-applications/salary_advance_application/${userUid}?${queryParams}`);
        
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();
        console.log('✅ MongoDB salary advance applications response:', result);

        if (result.success && result.data) {
          setApplications(result.data);
          
          // Cache for offline use
          await AsyncStorage.setItem('offlineApplications', JSON.stringify(result.data));
          
          console.log('📋 Salary advance applications loaded:', result.data.length);
        }
        
        setIsLoading(false);
      } catch (error) {
        console.error('Error fetching salary advance applications:', error);
        showToast('error', 'Error', 'Failed to load salary advance applications.');
        const cachedApplications = await AsyncStorage.getItem('offlineApplications');
        if (cachedApplications) {
          const parsedApplications = JSON.parse(cachedApplications);
          let filteredApplications = parsedApplications.filter(
            (app) => app.applicationType === 'salary_advance_application'
          );

          // Apply onlyMineView filter
          if (hrPermissions.includes('onlyMineView')) {
            filteredApplications = filteredApplications.filter(app => app.userUid === userUid);
          } else {
            // Apply country/project filter
            filteredApplications = filteredApplications.filter(app => 
              (allowedCountries.length === 0 || allowedCountries.includes(app.country)) &&
              (allowedProjects.length === 0 || allowedProjects.includes(app.project))
            );
          }

          setApplications(filteredApplications);
          showToast('warning', 'Offline Mode', 'Loaded cached salary advance applications due to error.');
        }
        setIsLoading(false);
      }
    };

    fetchApplications();
  }, [isOnline, allowedCountries, allowedProjects, hrPermissions, permissionsLoaded]);

  return (
    <View style={styles.flex1}>
      <LinearGradient
        colors={['#025383', '#015185']}
        style={styles.headerGradient}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <SafeAreaView style={styles.safeAreaHeader}>
          <View style={styles.headerContentFixed}>
            <TouchableOpacity
              style={styles.backButton}
              onPress={() => navigation.goBack()}
              accessibilityLabel="Go back"
              accessibilityRole="button"
            >
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitleFixed}>Salary Advance Applications</Text>
            <View style={{ width: width * 0.06 }} />
          </View>
        </SafeAreaView>
      </LinearGradient>
      <ScrollView contentContainerStyle={styles.scrollContent} bounces={false} showsVerticalScrollIndicator={false}>
        <ApplicationList
          applications={applications}
          isLoading={isLoading}
          applicationType="salary_advance_application"
          navigation={navigation}
        />
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  flex1: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  headerGradient: {
    width: '100%',
    paddingBottom: Platform.OS === 'android' ? StatusBar.currentHeight || 24 : 0,
  },
  safeAreaHeader: {
    backgroundColor: 'transparent',
  },
  headerContentFixed: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'android' ? ((StatusBar.currentHeight || width * 0.4) - 30) : 0,
    paddingBottom: width * 0.03,
    minHeight: Platform.OS === 'android' ? width * 0.01 - 10 : width * 0.01,
  },
  headerTitleFixed: {
    flex: 1,
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#fff',
    textAlign: 'center',
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  backButton: {
    padding: width * 0.02,
    ...(Platform.OS === 'android' ? { top: 30 } : {}),
  },
  scrollContent: {
    flexGrow: 1,
    paddingBottom: width * 0.04,
  },
});

export default AdminSalaryAdv;