import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  FlatList,
  Modal,
  TouchableWithoutFeedback,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useIsFocused } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

// MongoDB API URL
const MONGODB_FUEL_LOG_API_URL = 'https://api.titandrillingzm.com:6011';

const Fuel_Log_List = () => {
  const navigation = useNavigation();
  const isFocused = useIsFocused();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [fuelLogs, setFuelLogs] = useState([]);
  const [filteredLogs, setFilteredLogs] = useState([]);
  const [dropdownVisible, setDropdownVisible] = useState(null);
  const [selectedItem, setSelectedItem] = useState(null);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0, openUp: false });
  const [sortBy, setSortBy] = useState('createdAt');
  const [userData, setUserData] = useState(null);
  const [fuelLogPermissions, setFuelLogPermissions] = useState([]);
  const [isLoadingPermissions, setIsLoadingPermissions] = useState(true);
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const [localUserUid, setLocalUserUid] = useState(null);
  const isMounted = useRef(true);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const DROPDOWN_WIDTH = 220;
  const DROPDOWN_ITEM_HEIGHT = width * 0.13;
  const DROPDOWN_ICON_SIZE = 22;

  // Check if user has specific permission
  const hasPermission = (permission) => {
    const hasSpecificPermission = fuelLogPermissions.includes(permission);
    const hasAllPermission = fuelLogPermissions.includes('all');
    const result = hasSpecificPermission || hasAllPermission;
    
    console.log(`Permission check for "${permission}":`, {
      fuelLogPermissions,
      hasSpecificPermission,
      hasAllPermission,
      result
    });
    
    return result;
  };

  // Check if user can create fuel logs
  const canCreate = () => hasPermission('create');
  
  // Check if user can delete fuel logs
  const canDelete = () => hasPermission('delete');
  
  // Check if user can view fuel logs
  const canView = () => hasPermission('view');

  const startRotation = useCallback(() => {
    rotateAnim.setValue(0);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();
  }, [rotateAnim]);

  const stopRotation = useCallback(() => {
    rotateAnim.stopAnimation(() => rotateAnim.setValue(0));
  }, [rotateAnim]);

  const fetchUserData = useCallback(async (showLoader = true) => {
    try {
      const uid = await AsyncStorage.getItem('userUid');
      if (!uid) {
        showToast('error', 'Error', 'User not authenticated. Please log in again.');
        navigation.navigate('AdminLogin');
        return;
      }

      if (showLoader) {
        setSyncModalVisible(true);
        startRotation();
      }

      console.log('Fetching user data for uid:', uid);
      
      // Use MongoAPI like inspection module
      const response = await MongoAPI.getUserById(uid);
      console.log('User response:', response);
      
      if (!response.success || !response.data) {
        console.log('User not found in database');
        showToast('error', 'Error', 'User data not found.');
        setIsLoadingPermissions(false);
        return;
      }

      const data = response.data;
      console.log('User data received:', {
        name: data.name,
        email: data.email,
        countries: data.countries,
        projects: data.projects,
        modules: data.modules
      });
      
      // Normalize modules to array and extract Fuel Log permissions
      const modules = Array.isArray(data.modules) ? data.modules : (data.modules ? Object.values(data.modules) : []);
      
      // Find Fuel Log module - try multiple possible names like inspection module
      const fuelLogModule = modules.find(module => 
        module.module && (
          module.module === 'fuelLog' ||
          module.module === 'Fuel Log' ||
          module.module.toLowerCase() === 'fuel log' ||
          module.module.toLowerCase() === 'fuellog' ||
          module.module.toLowerCase() === 'fuel_log'
        )
      );
      
      console.log('Fuel Log module found:', fuelLogModule);
      console.log('All available modules:', modules.map(m => ({ module: m.module, permissions: m.permissions })));
      
      const permissions = fuelLogModule ? (Array.isArray(fuelLogModule.permissions) ? fuelLogModule.permissions : []) : [];
      setFuelLogPermissions(permissions);
      setUserData({ ...data, modules, fuelLogPermissions: permissions });
      setIsLoadingPermissions(false);
      
      console.log('Fuel Log Permissions:', permissions);
      console.log('All modules:', modules);
    } catch (error) {
      console.error('Error fetching user data:', error);
      showToast('error', 'Error', 'Failed to fetch user data.');
      setIsLoadingPermissions(false);
    } finally {
      if (showLoader) {
        setSyncModalVisible(false);
        stopRotation();
      }
    }
  }, [navigation, showToast, startRotation, stopRotation]);

  useEffect(() => {
    // Load UID from AsyncStorage and fetch data
    const initializeComponent = async () => {
      const uid = await AsyncStorage.getItem('userUid');
      setLocalUserUid(uid);
      
      // Fetch user permissions first
      await fetchUserData(true);
      await fetchFuelLogs(true);
    };
    
    initializeComponent();
    
    return () => {
      isMounted.current = false;
    };
  }, [fetchFuelLogs, fetchUserData]);

  useEffect(() => {
    if (isFocused) {
      fetchFuelLogs(true);
    }
  }, [fetchFuelLogs, isFocused]);

  const checkNetworkStatus = useCallback(async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) throw new Error('Internet connection not available');
      return true;
    } catch (error) {
      throw new Error('Failed to check network status: ' + error.message);
    }
  }, []);

  const showToast = useCallback((type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  }, []);

  const fetchFuelLogs = useCallback(async (showLoader = true) => {
    if (showLoader) {
      setSyncModalVisible(true);
      startRotation();
    }
    setIsRefreshing(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) throw new Error('No user is logged in');

      if (!(await checkNetworkStatus())) throw new Error('Internet connection not available');

      console.log('📋 Fetching fuel logs from MongoDB...');

      // Fetch fuel logs from MongoDB
      const response = await fetch(`${MONGODB_FUEL_LOG_API_URL}/fuel-logs?adminUid=${userUid}`);
      
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      
      const result = await response.json();
      
      if (result.success && result.data) {
        const logsArray = result.data;
        console.log(`✅ Fetched ${logsArray.length} fuel logs from MongoDB`);
        
        setFuelLogs(logsArray);
        setFilteredLogs(logsArray);
        showToast('success', 'Success', `Loaded ${logsArray.length} fuel logs.`);
      } else {
        setFuelLogs([]);
        setFilteredLogs([]);
        showToast('info', 'No Data', 'No fuel logs found.');
      }
    } catch (error) {
      console.error('Error fetching fuel logs from MongoDB:', error);
      let errorMessage = 'Failed to load data.';
      if (error.message.includes('Internet connection not available')) {
        errorMessage = 'No internet connection. Please connect to the internet.';
      } else if (error.message.includes('No user is logged in')) {
        errorMessage = 'User not authenticated. Please log in.';
        navigation.navigate('AdminLogin');
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      if (isMounted.current) {
        if (showLoader) {
          setSyncModalVisible(false);
          stopRotation();
        }
        setIsRefreshing(false);
      }
    }
  }, [checkNetworkStatus, isMounted, navigation, showToast, startRotation, stopRotation]);

  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text) {
      const filtered = fuelLogs.filter(
        (log) =>
          (log.equipmentName?.toLowerCase().includes(text.toLowerCase()) ||
          log.fuelType?.toLowerCase().includes(text.toLowerCase()) ||
          log.equipmentNo?.toLowerCase().includes(text.toLowerCase()) ||
          log.requestNumber?.toLowerCase().includes(text.toLowerCase()))
      );
      setFilteredLogs(filtered);
    } else {
      setFilteredLogs(fuelLogs);
    }
  };

  const handleSort = (criteria) => {
    setSortBy(criteria);
    let sortedLogs = [...filteredLogs];
    switch (criteria) {
      case 'createdAt':
        sortedLogs.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        break;
      case 'equipmentNo':
        sortedLogs.sort((a, b) => a.equipmentNo.localeCompare(b.equipmentNo));
        break;
      case 'equipmentName':
        sortedLogs.sort((a, b) => a.equipmentName.localeCompare(b.equipmentName));
        break;
      case 'fuelQuantity':
        sortedLogs.sort((a, b) => parseFloat(b.fuelQuantity) - parseFloat(a.fuelQuantity));
        break;
      case 'fuelCost':
        sortedLogs.sort((a, b) => parseFloat(b.fuelCost || 0) - parseFloat(a.fuelCost || 0));
        break;
      default:
        break;
    }
    setFilteredLogs(sortedLogs);
    setFilterModalVisible(false);
  };

  const handleRefresh = () => {
    fetchFuelLogs(true);
  };

  const handleActionPress = (item, event) => {
    setSelectedItem(item);
    setDropdownVisible(true);

    const { pageX, pageY } = event.nativeEvent;
    const iconHeight = 22;
    const dropdownHeight = DROPDOWN_ITEM_HEIGHT * 1; // Only delete option
    const screenHeight = Dimensions.get('window').height;

    let top = pageY + 2;
    let openUp = false;
    if (pageY + 2 + dropdownHeight + 20 > screenHeight) {
      top = pageY - dropdownHeight - 2;
      openUp = true;
    }

    setDropdownPosition({ x: pageX - DROPDOWN_WIDTH + iconHeight, y: top, openUp });
  };

  const handleDelete = async () => {
    if (!canDelete()) {
      showToast('error', 'Permission Denied', 'You do not have permission to delete fuel logs.');
      setDropdownVisible(false);
      setShowDeleteConfirm(false);
      return;
    }

    if (!selectedItem) {
      showToast('error', 'Error', 'No item selected to delete.');
      return;
    }

    setSyncModalVisible(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) throw new Error('No user is logged in');
      if (!(await checkNetworkStatus())) throw new Error('Internet connection not available');

      console.log(`📋 Deleting fuel log ${selectedItem._id} from MongoDB...`);

      // Delete from MongoDB
      const response = await fetch(`${MONGODB_FUEL_LOG_API_URL}/fuel-logs/${selectedItem._id}`, {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`Failed to delete fuel log: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.success) {
        console.log('✅ Fuel log deleted successfully');
        const updatedLogs = fuelLogs.filter(log => log._id !== selectedItem._id);
        setFuelLogs(updatedLogs);
        setFilteredLogs(updatedLogs);
        showToast('success', 'Success', 'Fuel log deleted successfully.');
      } else {
        throw new Error(result.message || 'Failed to delete fuel log');
      }
    } catch (error) {
      console.error('Error deleting fuel log:', error);
      let errorMessage = 'Failed to delete fuel log.';
      if (error.message.includes('Internet connection not available')) {
        errorMessage = 'No internet connection. Please try again later.';
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      setSyncModalVisible(false);
      setDropdownVisible(null);
      setShowDeleteConfirm(false);
    }
  };

  const getDropdownData = () => {
    const data = [];
    
    if (canDelete()) {
      data.push({ label: 'Delete', icon: 'delete', iconColor: '#F44336', action: 'handleDelete' });
    }
    
    return data.length > 0 ? data : [{ label: 'No permissions', icon: 'block', iconColor: '#999', action: 'noAction' }];
  };

  const renderFuelLog = ({ item }) => (
    <View style={styles.logCard}>
      <View style={styles.logTable}>
        {/* Request Number Row */}
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={[styles.tableHeader, { color: '#015185', fontWeight: 'bold' }]}>Request#</Text>
          <Text style={[styles.tableData, { color: '#015185', fontWeight: 'bold' }]}>{item.requestNumber || 'N/A'}</Text>
        </View>
        {/* Equipment Number Row */}
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Equipment #</Text>
          <Text style={styles.tableData}>{item.equipmentNo || 'N/A'}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Equipment Name</Text>
          <Text style={styles.tableData}>{item.equipmentName || 'N/A'}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Fuel Type</Text>
          <Text style={styles.tableData}>{item.fuelType || 'N/A'}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Fuel Quantity</Text>
          <Text style={styles.tableData}>{item.fuelQuantity} {item.fuelUnit || ''}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Meter Reading</Text>
          <Text style={styles.tableData}>{item.meterReading} {item.meterUnit || ''}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Total</Text>
          <Text style={styles.tableData}>{item.fuelCost || 'N/A'}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Date</Text>
          <Text style={styles.tableData}>
            {new Date(item.createdAt).toLocaleString('en-GB', {
              day: '2-digit',
              month: 'short',
              year: 'numeric',
              hour: '2-digit',
              minute: '2-digit',
              hour12: true,
            })}
          </Text>
        </View>
      </View>
      {/* Three dots icon for dropdown */}
      {canDelete() && (
        <View style={{ alignItems: 'flex-end', marginTop: 8 }}>
          <TouchableOpacity
            onPress={(event) => handleActionPress(item, event)}
            style={styles.threeDotsContainer}
          >
            <MaterialIcons name="more-vert" size={width * 0.06} color="#015185" />
          </TouchableOpacity>
        </View>
      )}
    </View>
  );

  const effectiveSyncModalVisible = syncModalVisible || isLoadingPermissions;
  const syncTitleText = isLoadingPermissions ? 'Loading Permissions' : 'Please Wait';
  const syncSubtitleText = isLoadingPermissions ? 'Fetching access settings' : 'Data syncing';

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Fuel Log</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={handleRefresh} style={styles.refreshButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={() => setPermissionModalVisible(true)}
            >
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton} onPress={() => setFilterModalVisible(true)}>
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      <FlatList
        data={filteredLogs}
        renderItem={renderFuelLog}
        keyExtractor={(item) => item._id}
        contentContainerStyle={styles.mainContent}
        ListEmptyComponent={
          <Text style={styles.defaultText}>No fuel logs found.</Text>
        }
        refreshing={isRefreshing}
        onRefresh={handleRefresh}
      />

      {dropdownVisible && (
        <Modal
          visible={dropdownVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setDropdownVisible(false)}
        >
          <TouchableOpacity
            style={styles.dropdownOverlay}
            onPress={() => setDropdownVisible(false)}
            activeOpacity={1}
          >
            <View
              style={[
                styles.dropdownContainer,
                {
                  position: 'absolute',
                  top: dropdownPosition.y + 8,
                  left: dropdownPosition.x,
                },
              ]}
            >
              {getDropdownData().map((option, index, array) => (
                <TouchableOpacity
                  key={index}
                  style={[
                    styles.dropdownItem,
                    index === array.length - 1 && { borderBottomWidth: 0 },
                    option.action === 'noAction' && styles.disabledDropdownItem,
                  ]}
                  onPress={() => {
                    if (option.action === 'noAction') {
                      setDropdownVisible(false);
                      return;
                    }
                    
                    setDropdownVisible(false);
                    setShowDeleteConfirm(true);
                  }}
                >
                  <MaterialIcons name={option.icon} size={DROPDOWN_ICON_SIZE} color={option.iconColor} />
                  <Text style={[styles.dropdownText, option.action === 'noAction' && styles.disabledDropdownText]}>{option.label}</Text>
                </TouchableOpacity>
              ))}
            </View>
          </TouchableOpacity>
        </Modal>
      )}

      {/* Delete confirmation modal */}
      <Modal
        visible={!!showDeleteConfirm}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setShowDeleteConfirm(false)}
      >
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Confirm Delete</Text>
            <Text style={styles.syncModalTextadd}>Are you sure you want to delete this fuel log?</Text>
            <View style={{ flexDirection: 'row', marginTop: 20 }}>
              <TouchableOpacity
                style={[styles.closeButton, { marginRight: 10, backgroundColor: '#888' }]}
                onPress={() => setShowDeleteConfirm(false)}
              >
                <Text style={styles.closeButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.closeButton, { backgroundColor: '#F44336' }]}
                onPress={async () => {
                  await handleDelete();
                }}
              >
                <Text style={styles.closeButtonText}>Delete</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {canCreate() && (
        <View style={styles.buttonContainer}>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('CreateFuelLog')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.buttonText}>Fuel Log Entry</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      )}

      <Modal visible={effectiveSyncModalVisible} transparent animationType="fade">
        <View style={styles.loadingModalContainer}>
          <View style={styles.loadingModalContent}>
            <Text style={styles.loadingModalText}>{syncTitleText}</Text>
            <View style={styles.loadingIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.loadingIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.loadingModalTextadd}>{syncSubtitleText}</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={filterModalVisible} transparent animationType="fade">
        <TouchableWithoutFeedback onPress={() => setFilterModalVisible(false)}>
          <View style={styles.filterModalContainer}>
            <View style={styles.filterModalContent}>
              <TouchableOpacity onPress={() => handleSort('createdAt')}>
                <Text style={styles.filterText}>Created Date</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={() => handleSort('equipmentNo')}>
                <Text style={styles.filterText}>Equipment Number</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={() => handleSort('equipmentName')}>
                <Text style={styles.filterText}>Equipment Name</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={() => handleSort('fuelQuantity')}>
                <Text style={styles.filterText}>Fuel Quantity</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={() => handleSort('fuelCost')}>
                <Text style={styles.filterText}>Total Cost</Text>
              </TouchableOpacity>
              <View style={styles.buttonWrapper}>
                <TouchableOpacity
                  style={styles.closeButton}
                  onPress={() => setFilterModalVisible(false)}
                >
                  <Text style={styles.closeButtonText}>Close</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </TouchableWithoutFeedback>
      </Modal>

      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <View style={styles.permissionModalOverlay}>
          <View style={styles.permissionModalContent}>
            <Text style={styles.permissionModalTitle}>User Permissions</Text>
            
            <Text style={styles.permissionModalLabel}>Fuel Log Permissions:</Text>
            <Text style={styles.permissionModalValue}>
              {fuelLogPermissions.length > 0 ? fuelLogPermissions.join(', ') : 'None'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Countries:</Text>
            <Text style={styles.permissionModalValue}>
              {userData && userData.countries && userData.countries.length > 0 ? userData.countries.join(', ') : 'None'}
            </Text>
            
            <Text style={styles.permissionModalLabel}>Projects:</Text>
            <Text style={styles.permissionModalValue}>
              {userData && userData.projects && userData.projects.length > 0 ? userData.projects.join(', ') : 'None'}
            </Text>
            
            <TouchableOpacity
              style={styles.permissionCloseButton}
              onPress={() => setPermissionModalVisible(false)}
            >
              <Text style={styles.permissionCloseButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#ECF0F3',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  mainContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.2,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  logCard: {
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  logTable: {
    backgroundColor: '#fff',
  },
  tableRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: width * 0.015,
  },
  tableRowWithBorder: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingBottom: width * 0.015,
  },
  tableHeader: {
    fontSize: width * 0.035,
    color: '#025383',
    fontWeight: 'bold',
    width: '40%',
  },
  tableData: {
    fontSize: width * 0.035,
    color: '#333',
    width: '60%',
    textAlign: 'right',
  },
  threeDotsContainer: {
    padding: width * 0.01,
    marginLeft: width * 0.02,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.2)',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#eee',
    width: 220,
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.12,
    shadowRadius: 8,
    paddingVertical: 4,
    zIndex: 999,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    backgroundColor: '#fff',
  },
  dropdownText: {
    fontSize: 16,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
  disabledDropdownItem: {
    opacity: 0.5,
  },
  disabledDropdownText: {
    color: '#999',
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  loadingModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  loadingModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  loadingIcon: {
    width: width * 0.05,
    height: width * 0.05,
    marginRight: width * 0.02,
  },
  loadingModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  loadingIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  filterModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  filterModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'flex-start',
  },
  filterText: {
    fontSize: width * 0.04,
    color: '#025383',
    marginBottom: 10,
  },
  buttonWrapper: {
    width: '100%',
    alignItems: 'flex-end',
    marginTop: 10,
  },
  closeButton: {
    backgroundColor: '#025383',
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  modalTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#025383',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  permissionSection: {
    marginBottom: width * 0.04,
    width: '100%',
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#025383',
    marginBottom: width * 0.02,
  },
  permissionText: {
    fontSize: width * 0.04,
    color: '#333',
    marginLeft: width * 0.02,
    marginBottom: width * 0.01,
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#888',
    fontStyle: 'italic',
  },
  permissionModalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  permissionModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 24,
    width: '80%',
    alignItems: 'flex-start',
    elevation: 5,
  },
  permissionModalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 16,
    alignSelf: 'center',
    width: '100%',
    textAlign: 'center',
  },
  permissionModalLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
    marginTop: 10,
  },
  permissionModalValue: {
    fontSize: 14,
    color: '#333',
    marginBottom: 4,
  },
  permissionCloseButton: {
    alignSelf: 'center',
    marginTop: 18,
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 24,
    paddingVertical: 8,
  },
  permissionCloseButtonText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  loadingText: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: 'bold',
  },
});

export default Fuel_Log_List;
