import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  StatusBar,
  ScrollView,
  Modal,
  Animated,
  Image,
  ToastAndroid,
  Platform,
  ActivityIndicator,
  Dimensions,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import { Dropdown } from 'react-native-element-dropdown';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import { MongoAPI, USERS_API } from '../../ConfigMongo/MongoDBConfig';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;
const INITIAL_ITEMS_TO_LOAD = 15;
const ITEMS_PER_PAGE = 10;

const EquipmentOne = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [filteredEquipment, setFilteredEquipment] = useState([]);
  const [displayedEquipment, setDisplayedEquipment] = useState([]);
  const [page, setPage] = useState(1);
  const [isLoadingMore, setIsLoadingMore] = useState(false);
  const [isLoadingInitial, setIsLoadingInitial] = useState(true);
  const [hasMore, setHasMore] = useState(true);
  const [actionModalVisible, setActionModalVisible] = useState(false);
  const [statusModalVisible, setStatusModalVisible] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const [selectedEquipment, setSelectedEquipment] = useState(null);
  const [newStatus, setNewStatus] = useState('Available');
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [selectedFilter, setSelectedFilter] = useState(null);
  const [selectedCountry, setSelectedCountry] = useState('All Countries');
  const [countryDropdownVisible, setCountryDropdownVisible] = useState(false);
  const [countryDropdownPosition, setCountryDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [selectedVehicleType, setSelectedVehicleType] = useState('All Vehicle Types');
  const [vehicleTypeDropdownVisible, setVehicleTypeDropdownVisible] = useState(false);
  const [vehicleTypeDropdownPosition, setVehicleTypeDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [userInfoModalVisible, setUserInfoModalVisible] = useState(false);
  const [userInfo, setUserInfo] = useState({
    countries: [],
    projects: [],
    equipmentPermissions: [],
    employeeNumber: '',
    name: '',
    userUid: '',
  });
  const [isOnline, setIsOnline] = useState(true);
  const [lastSyncTime, setLastSyncTime] = useState(null);
  const [isSyncing, setIsSyncing] = useState(false);
  const [forceOfflineMode, setForceOfflineMode] = useState(false);
  const [pendingChanges, setPendingChanges] = useState(false);
  const countryButtonRef = useRef(null);
  const vehicleTypeButtonRef = useRef(null);
  const permissionsButtonRef = useRef(null);
  const isMounted = useRef(true);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const checkNetworkStatus = async () => {
    try {
      if (forceOfflineMode) {
        return false; // Force offline
      }
      const netInfo = await NetInfo.fetch();
      // Check both isConnected and isInternetReachable for better accuracy
      const hasConnection = netInfo.isConnected && netInfo.isInternetReachable !== false;
      console.log('🌐 Equipment Network Check:', {
        isConnected: netInfo.isConnected,
        isInternetReachable: netInfo.isInternetReachable,
        hasConnection
      });
      return hasConnection;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Check for pending changes
  const checkPendingChanges = async () => {
    try {
      const existing = await AsyncStorage.getItem('equipment');
      const equipment = existing ? JSON.parse(existing) : [];
      const pending = equipment.filter((item) => item.pendingSync || item.pendingDelete);
      setPendingChanges(pending.length > 0);
      return pending.length;
    } catch (error) {
      console.error('Error checking pending changes:', error);
      return 0;
    }
  };

  // Toggle online/offline mode
  const toggleOfflineMode = async () => {
    const newOfflineMode = !forceOfflineMode;
    
    if (newOfflineMode) {
      // Switching to offline
      setForceOfflineMode(true);
      await AsyncStorage.setItem('forceOfflineMode_equipment', JSON.stringify(true));
      setIsOnline(false);
      showToast('info', 'Offline Mode', 'Working offline now');
      console.log('📴 Forced offline mode enabled');
    } else {
      // Trying to switch to online - check actual network first
      console.log('🔍 Checking actual internet connectivity...');
      
      try {
        const netInfo = await NetInfo.fetch();
        const hasInternet = netInfo.isConnected && netInfo.isInternetReachable !== false;
        
        console.log('📡 Equipment Network Info:', {
          isConnected: netInfo.isConnected,
          isInternetReachable: netInfo.isInternetReachable,
          type: netInfo.type,
          hasInternet
        });
        
        if (hasInternet) {
          // Network is available - switch to online
          setForceOfflineMode(false);
          await AsyncStorage.setItem('forceOfflineMode_equipment', JSON.stringify(false));
          setIsOnline(true);
          showToast('success', 'Online Mode', 'Connected - Syncing...');
          console.log('✅ Network available - Switching to online mode');
          
          // Auto-sync data and pending changes
          await handleRefresh();
        } else {
          // No network available - keep offline mode ON
          setForceOfflineMode(true);
          await AsyncStorage.setItem('forceOfflineMode_equipment', JSON.stringify(true));
          setIsOnline(false);
          showToast('error', 'No Internet', 'No network connection. Staying offline.');
          console.log('❌ No internet connection - Staying offline');
        }
      } catch (error) {
        console.error('❌ Error checking network:', error);
        // On error, stay offline
        setForceOfflineMode(true);
        await AsyncStorage.setItem('forceOfflineMode_equipment', JSON.stringify(true));
        setIsOnline(false);
        showToast('error', 'Network Error', 'Cannot detect network. Staying offline.');
      }
    }
  };

  const showToast = (type, text1, text2) => {
    if (Platform.OS === 'android') {
      ToastAndroid.show(`${text1}: ${text2}`, ToastAndroid.LONG);
    } else {
      Toast.show({ type, text1, text2, position: 'bottom' });
    }
  };

  const fetchUserInfo = async () => {
    try {
      // Get user data from AsyncStorage (set during login)
      const userData = await AsyncStorage.getItem('userData');
      if (!userData) {
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return {
          countries: [],
          projects: [],
          equipmentPermissions: [],
          employeeNumber: '',
          name: '',
          userUid: '',
        };
      }
      const user = JSON.parse(userData);

      let userUid = await AsyncStorage.getItem('userUid');
      let userDataFromStorage = null;

      if (userUid) {
        // Use the user data already parsed from AsyncStorage
        userDataFromStorage = user;
      }

      if (!userDataFromStorage) {
        // Use the user data already parsed from AsyncStorage
        userDataFromStorage = user;
        if (userDataFromStorage && userDataFromStorage.userUid) {
          userUid = userDataFromStorage.userUid;
          await AsyncStorage.setItem('userUid', userUid);
        } else {
          userUid = user.email; // Fallback to email if no userUid
          await AsyncStorage.setItem('userUid', userUid);
        }
      }

      if (!userDataFromStorage) {
        showToast('error', 'Error', 'User data not found.');
        return {
          countries: [],
          projects: [],
          equipmentPermissions: [],
          employeeNumber: '',
          name: '',
          userUid: userUid || user.uid,
        };
      }

      const equipmentModule = userDataFromStorage.modules?.find(
        (module) => module.module?.toLowerCase() === 'equipments'
      ) || { permissions: [] };

      const userInfo = {
        countries: Array.isArray(userDataFromStorage.countries) ? userDataFromStorage.countries : [],
        projects: Array.isArray(userDataFromStorage.projects) ? userDataFromStorage.projects : [],
        equipmentPermissions: Array.isArray(equipmentModule.permissions) ? equipmentModule.permissions : [],
        employeeNumber: userDataFromStorage.employeeNumber || '',
        name: userDataFromStorage.name || '',
        userUid: userUid || user.uid,
      };

      await AsyncStorage.setItem('userInfo', JSON.stringify(userInfo));
      setUserInfo(userInfo);
      return userInfo;
    } catch (error) {
      console.error('Error fetching user info:', error);
      showToast('error', 'Error', 'Failed to fetch user permissions.');
      return {
        countries: [],
        projects: [],
        equipmentPermissions: [],
        employeeNumber: '',
        name: '',
        userUid: '',
      };
    }
  };

  const filterEquipmentByPermission = useCallback((equipmentArr, userInfoObj) => {
    if (!equipmentArr || !userInfoObj) {
      console.log('❌ filterEquipmentByPermission: No equipment array or user info');
      return [];
    }

    console.log('🔍 Filtering equipment:', {
      totalEquipment: equipmentArr.length,
      userCountries: userInfoObj.countries,
      userPermissions: userInfoObj.equipmentPermissions,
      userUid: userInfoObj.userUid
    });

    // More permissive filtering
    if (userInfoObj.equipmentPermissions.includes('onlyMineView')) {
      const creatorIdentifier = userInfoObj.userUid;
      const filtered = equipmentArr
        .filter((item) => {
          // Show if created by user OR if in user's countries (more permissive)
          const isCreator = item.createdBy === creatorIdentifier || !item.createdBy;
          const isInCountry = userInfoObj.countries.includes(item.country);
          
          // Be more permissive: show if either condition is true
          return isCreator || isInCountry;
        })
        .map((item) => ({
          ...item,
          createdBy: item.createdBy || userInfoObj.userUid,
          equipmentName: item.equipmentName
            ? item.equipmentName.charAt(0).toUpperCase() + item.equipmentName.slice(1).toLowerCase()
            : item.equipmentName,
        }));
      
      console.log('✅ Filtered with onlyMineView:', filtered.length);
      return filtered;
    } else if (userInfoObj.equipmentPermissions.includes('view')) {
      const filtered = equipmentArr
        .filter((item) => {
          const isInCountry = userInfoObj.countries.includes(item.country);
          return isInCountry;
        })
        .map((item) => ({
          ...item,
          equipmentName: item.equipmentName
            ? item.equipmentName.charAt(0).toUpperCase() + item.equipmentName.slice(1).toLowerCase()
            : item.equipmentName,
        }));
      
      console.log('✅ Filtered with view permission:', filtered.length);
      return filtered;
    } else if (userInfoObj.equipmentPermissions.includes('all')) {
      // If user has 'all' permission, show everything
      console.log('✅ User has "all" permission, showing all equipment');
      return equipmentArr.map((item) => ({
        ...item,
        equipmentName: item.equipmentName
          ? item.equipmentName.charAt(0).toUpperCase() + item.equipmentName.slice(1).toLowerCase()
          : item.equipmentName,
      }));
    } else {
      console.log('❌ No valid permissions found:', userInfoObj.equipmentPermissions);
      return [];
    }
  }, []);

  const loadData = async (isRefresh = false, showModal = true) => {
    if (!isMounted.current) return;
    
    if (showModal) {
      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    }
    
    setIsSyncing(true);

    try {
      // Get user data from AsyncStorage (set during login)
      const userData = await AsyncStorage.getItem('userData');
      if (!userData) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        setIsSyncing(false);
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return;
      }
      const user = JSON.parse(userData);

      const userInfoData = await fetchUserInfo();
      if (!userInfoData) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        setIsSyncing(false);
        return;
      }

      console.log('📊 User Info:', {
        countries: userInfoData.countries,
        projects: userInfoData.projects,
        equipmentPermissions: userInfoData.equipmentPermissions,
        userUid: userInfoData.userUid
      });
      const onlineStatus = await checkNetworkStatus();
      setIsOnline(onlineStatus);
      let parsedData = [];

      if (onlineStatus) {
        try {
          // Fetch equipment data from MongoDB API
          console.log('🌐 Fetching equipment from MongoDB API...');
          
          const response = await fetch('https://api.titandrillingzm.com:6001/EquipmentsDataJSON/GlobalEquipmentJsonData', {
            method: 'GET',
            headers: {
              'Content-Type': 'application/json',
            },
          });

          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }

          const responseData = await response.json();
          console.log('✅ MongoDB API Response:', responseData);

          // Extract data from response
          if (responseData.success && responseData.data && Array.isArray(responseData.data)) {
            parsedData = responseData.data.map((item, index) => ({
              ...item,
              id: item.id || item._id || `equip_${index}`,
              firebaseKey: item.firebaseKey || `equip_${index}`,
              mainCategory: item.mainCategory || item.category || item.equipmentType || 'Unknown',
              country: item.country || 'DRC',
              equipmentName: item.equipmentName || item.rigName || item.vehicleNumber || 'Equipment',
            }));
          } else if (Array.isArray(responseData)) {
            // Handle direct array response
            parsedData = responseData.map((item, index) => ({
              ...item,
              id: item.id || item._id || `equip_${index}`,
              firebaseKey: item.firebaseKey || `equip_${index}`,
              mainCategory: item.mainCategory || item.category || item.equipmentType || 'Unknown',
              country: item.country || 'DRC',
              equipmentName: item.equipmentName || item.rigName || item.vehicleNumber || 'Equipment',
            }));
          }

          console.log('📦 Parsed MongoDB Data:', parsedData.length, 'items');
          if (parsedData.length > 0) {
            console.log('📋 First item structure:', parsedData[0]);
            console.log('🔑 All fields in first item:', Object.keys(parsedData[0]));
            console.log('🌍 Countries in data:', [...new Set(parsedData.map(item => item.country))]);
          } else {
            console.log('⚠️ No equipment data received from MongoDB!');
          }
        } catch (fetchError) {
          console.error('❌ Error fetching from MongoDB:', fetchError);
          
          // Try to load from cache as fallback
          console.log('📦 Loading from cache as fallback...');
          const savedData = await AsyncStorage.getItem('equipment');
          const savedSyncTime = await AsyncStorage.getItem('equipmentLastSync');
          
          if (savedSyncTime) {
            setLastSyncTime(savedSyncTime);
          }
          
          if (savedData) {
            parsedData = JSON.parse(savedData);
            showToast('info', 'Offline Mode', 'Using cached equipment data');
          } else {
            showToast('error', 'Error', 'No offline data available');
            parsedData = [];
          }
        }

        parsedData = parsedData
          .filter((item, index, self) => self.findIndex((t) => t.id === item.id) === index)
          .sort((a, b) => {
            const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
            const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
            return dateB - dateA;
          });

        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        console.log('Data after filterEquipmentByPermission:', filteredData); // Debug: Check final filtered data

        // Debug: Log the first equipment item to see all available fields
        if (filteredData.length > 0) {
          console.log('First equipment item structure:', filteredData[0]);
          console.log('All available fields:', Object.keys(filteredData[0]));
        }

        parsedData = filteredData;
        
        // Store with timestamp
        await AsyncStorage.setItem('equipment', JSON.stringify(parsedData));
        const syncTime = new Date().toISOString();
        await AsyncStorage.setItem('equipmentLastSync', syncTime);
        setLastSyncTime(syncTime);
        
        if (isRefresh) {
          showToast('success', 'Synced', 'Equipment data synced successfully');
        }
        
        // Check for pending changes after sync
        await checkPendingChanges();
      } else {
        console.log('📴 Offline - Loading from local storage...');
        const savedData = await AsyncStorage.getItem('equipment');
        const savedSyncTime = await AsyncStorage.getItem('equipmentLastSync');
        
        if (savedSyncTime) {
          setLastSyncTime(savedSyncTime);
        }
        
        parsedData = savedData ? JSON.parse(savedData) : [];
        
        console.log('📦 Loaded from AsyncStorage:', parsedData.length, 'items');
        console.log('📋 Pending items:', parsedData.filter(i => i.pendingSync).length);
        
        // Normalize offline data fields
        parsedData = parsedData.map(item => ({
          ...item,
          mainCategory: item.mainCategory || item.category || item.equipmentType || 'Unknown',
          country: item.country || 'DRC',
          equipmentName: item.equipmentName || item.rigName || item.vehicleNumber || 'Equipment',
        }));
        
        // Filter by permissions
        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        console.log('📊 Offline Data after filtering:', filteredData.length, 'items');
        console.log('📋 Items with pendingSync:', filteredData.filter(i => i.pendingSync).length);
        
        parsedData = filteredData;
        
        // Sort by date (most recent first, including offline saves)
        parsedData.sort((a, b) => {
          const dateA = a.offlineSaveTime 
            ? new Date(a.offlineSaveTime) 
            : (a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt));
          const dateB = b.offlineSaveTime 
            ? new Date(b.offlineSaveTime) 
            : (b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt));
          return dateB - dateA;
        });
        
        if (isRefresh) {
          showToast('info', 'Offline', 'Using locally cached equipment data');
        }
      }

      if (isRefresh) {
        setFirebaseEquipment(parsedData);
      } else {
        setFirebaseEquipment((prev) => [...prev, ...parsedData]);
      }

      setFilteredEquipment(parsedData);
      setDisplayedEquipment(parsedData.slice(0, INITIAL_ITEMS_TO_LOAD));
      setHasMore(parsedData.length > INITIAL_ITEMS_TO_LOAD);

      const storedCountry = await AsyncStorage.getItem('selectedCountry');
      // Convert 'Both Countries' to 'All Countries' to prevent showing 'Both Countries' in UI
      const countryToSet = storedCountry === 'Both Countries' ? 'All Countries' : (storedCountry || 'All Countries');
      setSelectedCountry(countryToSet);
      setSelectedVehicleType('All Vehicle Types');
    } catch (error) {
      console.error('Error loading equipment:', error.code, error.message);
      let errorMessage = 'Failed to load equipment.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Check Firebase security rules.';
      } else if (error.code === 'NETWORK_ERROR') {
        errorMessage = 'Offline: Loading from local storage.';
        const savedData = await AsyncStorage.getItem('equipment');
        const parsedData = savedData ? JSON.parse(savedData) : [];
        const userInfoData = await AsyncStorage.getItem('userInfo');
        const userInfoParsed = userInfoData ? JSON.parse(userInfoData) : userInfo;
        
        // Normalize error handling data fields
        const normalizedData = parsedData.map(item => ({
          ...item,
          mainCategory: item.mainCategory || item.category || item.equipmentType || 'Unknown',
          country: item.country || 'DRC',
        }));
        
        normalizedData.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        const filteredData = filterEquipmentByPermission(normalizedData, userInfoParsed);
        setFirebaseEquipment(filteredData);
        setFilteredEquipment(filteredData);
        setDisplayedEquipment(filteredData.slice(0, INITIAL_ITEMS_TO_LOAD));
        setHasMore(filteredData.length > INITIAL_ITEMS_TO_LOAD);
      }
      showToast(
        error.code === 'NETWORK_ERROR' ? 'info' : 'error',
        error.code === 'NETWORK_ERROR' ? 'Offline' : 'Error',
        errorMessage
      );
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        setIsRefreshing(false);
        setIsLoadingMore(false);
        setIsLoadingInitial(false);
        setIsSyncing(false);
      }
    }
  };

  // Format last sync time
  const getLastSyncText = () => {
    if (!lastSyncTime) return 'Never synced';
    
    const now = new Date();
    const syncDate = new Date(lastSyncTime);
    const diffInMinutes = Math.floor((now - syncDate) / (1000 * 60));
    
    if (diffInMinutes < 1) return 'Just now';
    if (diffInMinutes < 60) return `${diffInMinutes}m ago`;
    
    const diffInHours = Math.floor(diffInMinutes / 60);
    if (diffInHours < 24) return `${diffInHours}h ago`;
    
    const diffInDays = Math.floor(diffInHours / 24);
    return `${diffInDays}d ago`;
  };

  // Initial load
  useEffect(() => {
    const initializeData = async () => {
      // Load force offline mode preference
      const savedOfflineMode = await AsyncStorage.getItem('forceOfflineMode_equipment');
      if (savedOfflineMode) {
        setForceOfflineMode(JSON.parse(savedOfflineMode));
      }
      
      // Check actual network status with detailed info
      const netInfo = await NetInfo.fetch();
      const hasInternet = netInfo.isConnected && netInfo.isInternetReachable !== false;
      const effectiveOnline = savedOfflineMode ? false : hasInternet;
      
      console.log('📡 Equipment Initial Network Status:', {
        isConnected: netInfo.isConnected,
        isInternetReachable: netInfo.isInternetReachable,
        type: netInfo.type,
        effectiveOnline,
        forceOffline: savedOfflineMode
      });
      
      setIsOnline(effectiveOnline);
      
      // Auto-set offline mode if no internet
      if (!hasInternet && !savedOfflineMode) {
        console.log('⚠️ No internet detected - Auto-switching to offline mode');
        setForceOfflineMode(true);
        await AsyncStorage.setItem('forceOfflineMode_equipment', JSON.stringify(true));
      }
      
      // Check for pending changes
      await checkPendingChanges();
      
      // Load last sync time from storage
      const syncTime = await AsyncStorage.getItem('equipmentLastSync');
      if (syncTime) {
        setLastSyncTime(syncTime);
      }
      
      // Load data
      await loadData(true, false);
    };
    
    initializeData();
    
    return () => {
      isMounted.current = false;
    };
  }, []);

  // Network status listener for auto-sync
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (!isMounted.current || forceOfflineMode) return;
      
      const wasOffline = !isOnline;
      // Check both isConnected and isInternetReachable
      const isNowOnline = state.isConnected && state.isInternetReachable !== false;
      
      console.log('📡 Equipment Network State Changed:', {
        isConnected: state.isConnected,
        isInternetReachable: state.isInternetReachable,
        type: state.type,
        effectiveOnline: isNowOnline
      });
      
      setIsOnline(isNowOnline);
      
      // Auto-sync when connection is restored
      if (wasOffline && isNowOnline) {
        console.log('🌐 Connection restored - Auto-syncing equipment...');
        showToast('success', 'Online', 'Connection restored, syncing...');
        
        // Auto-sync equipment data and pending changes
        setTimeout(async () => {
          try {
            await loadData(true, true);
            await checkPendingChanges();
          } catch (error) {
            console.error('Auto-sync failed:', error);
            showToast('error', 'Sync Failed', 'Could not sync equipment data');
          }
        }, 1000);
      } else if (!isNowOnline && wasOffline !== isNowOnline) {
        console.log('📴 Connection lost - Switching to offline mode');
        showToast('info', 'Offline', 'No internet connection');
      }
    });
    
    return () => {
      if (unsubscribe) unsubscribe();
    };
  }, [isOnline, forceOfflineMode]);

  useFocusEffect(
    useCallback(() => {
      // Re-check network status and reload data when screen comes into focus
      const recheckAndReload = async () => {
        if (!forceOfflineMode) {
          const netInfo = await NetInfo.fetch();
          const hasInternet = netInfo.isConnected && netInfo.isInternetReachable !== false;
          setIsOnline(hasInternet);
          
          console.log('🔄 Equipment Screen focused - Network recheck:', {
            isConnected: netInfo.isConnected,
            isInternetReachable: netInfo.isInternetReachable,
            hasInternet
          });
          
          // Auto-set offline if no internet
          if (!hasInternet) {
            setForceOfflineMode(true);
            await AsyncStorage.setItem('forceOfflineMode_equipment', JSON.stringify(true));
          }
        }
        
        // Always reload data from AsyncStorage to show new offline additions
        console.log('🔄 Reloading equipment list (including offline items)...');
        await loadData(true, false); // Reload without showing modal
        await checkPendingChanges(); // Update pending count
      };
      
      recheckAndReload();
    }, [forceOfflineMode])
  );

  useEffect(() => {
    const syncPendingChanges = async () => {
      const isOnline = await checkNetworkStatus();
      if (!isOnline) {
        console.log('📴 Not online, skipping pending sync');
        return;
      }

      // Get user data from AsyncStorage (set during login)
      const userData = await AsyncStorage.getItem('userData');
      if (!userData) return;
      const user = JSON.parse(userData);

      try {
        const existing = await AsyncStorage.getItem('equipment');
        const equipment = existing ? JSON.parse(existing) : [];
        const pending = equipment.filter((item) => item.pendingSync || item.pendingDelete);

        if (pending.length > 0) {
          console.log('🔄 Syncing', pending.length, 'pending equipment changes...');
          setSyncModalVisible(true);
          Animated.loop(
            Animated.timing(rotateAnim, {
              toValue: 1,
              duration: 1000,
              useNativeDriver: true,
            })
          ).start();

          let successCount = 0;
          let failCount = 0;

          // Sync to MongoDB via API
          for (const item of pending) {
            try {
              if (item.pendingDelete) {
                // Delete from MongoDB
                await fetch(`https://api.titandrillingzm.com:6001/equipment/${item.id}`, {
                  method: 'DELETE',
                  headers: { 'Content-Type': 'application/json' },
                });
                successCount++;
                console.log('✅ Deleted from MongoDB:', item.id);
              } else if (item.savedOffline) {
                // New equipment created offline - use POST
                const { savedOffline, offlineSaveTime, pendingSync, ...cleanItem } = item;
                
                await fetch('https://api.titandrillingzm.com:6001/add', {
                  method: 'POST',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify(cleanItem),
                });
                successCount++;
                console.log('✅ Created in MongoDB:', item.id);
              } else {
                // Updated existing equipment - use PUT
                const { savedOffline, offlineSaveTime, pendingSync, ...cleanItem } = item;
                
                await fetch(`https://api.titandrillingzm.com:6001/equipment/${item.id}`, {
                  method: 'PUT',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify(cleanItem),
                });
                successCount++;
                console.log('✅ Updated in MongoDB:', item.id);
              }
            } catch (syncError) {
              failCount++;
              console.error('❌ Error syncing item:', item.id, syncError);
            }
          }

          const updatedEquipment = equipment
            .filter((item) => !item.pendingDelete)
            .map((item) => ({
              ...item,
              pendingSync: false,
              pendingDelete: false,
              savedOffline: false,
            }));
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipment));
          
          // Update sync timestamp
          await AsyncStorage.setItem('equipmentLastSync', new Date().toISOString());
          setLastSyncTime(new Date().toISOString());

          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          
          if (successCount > 0) {
            showToast('success', 'Synced', `${successCount} changes synced to database.`);
            console.log('✅ Successfully synced', successCount, 'equipment changes');
          }
          
          if (failCount > 0) {
            showToast('warning', 'Partial Sync', `${failCount} items failed to sync.`);
          }
          
          // Update pending changes status
          await checkPendingChanges();
          
          // Reload data to show updated status
          await loadData(true, false);
        }
      } catch (error) {
        console.error('❌ Error syncing pending changes:', error);
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'Failed to sync pending changes.');
      }
    };

    const unsubscribe = NetInfo.addEventListener((state) => {
      const hasInternet = state.isConnected && state.isInternetReachable !== false;
      
      console.log('📡 Equipment_one - Network changed:', {
        isConnected: state.isConnected,
        isInternetReachable: state.isInternetReachable,
        hasInternet
      });
      
      if (hasInternet) {
        console.log('🌐 Connection available - Auto-syncing pending changes...');
        syncPendingChanges();
      }
    });

    return () => unsubscribe();
  }, []);

  const handleSearch = (query) => {
    setSearchQuery(query);
    setPage(1);
    filterEquipment(query, selectedFilter, selectedCountry, selectedVehicleType);
  };

  const filterEquipment = (query, filterType, country, vehicleType = selectedVehicleType) => {
    let filtered = filterEquipmentByPermission(firebaseEquipment, userInfo);

    if (country && country !== 'All Countries') {
      filtered = filtered.filter((item) => item.country === country);
    }

    if (vehicleType && vehicleType !== 'All Vehicle Types') {
      filtered = filtered.filter((item) => getVehicleTypeFromCategory(item) === vehicleType);
    }

    if (query) {
      filtered = filtered.filter((item) => {
        switch (filterType) {
          case 'Created Date':
            return new Date(item.createdAt)
              .toLocaleDateString()
              .toLowerCase()
              .includes(query.toLowerCase());
          case 'Equipment Number':
            return (item.equipmentNumber?.toLowerCase() || '').includes(query.toLowerCase());
          case 'Equipment Name':
            return (item.equipmentName?.toLowerCase() || '').includes(query.toLowerCase());
          case 'Equipment Categories':
            return (item.mainCategory?.toLowerCase() || '').includes(query.toLowerCase());
          case 'Meter Reading':
            return `${item?.meterReading || ''}`.toLowerCase().includes(query.toLowerCase());
          case 'Status':
            return (item.status?.toLowerCase() || '').includes(query.toLowerCase());
          default:
            return (
              (item?.equipmentName?.toLowerCase() || '').includes(query.toLowerCase()) ||
              (item?.equipmentNumber?.toLowerCase() || '').includes(query.toLowerCase()) ||
              (item?.mainCategory?.toLowerCase() || '').includes(query.toLowerCase()) ||
              (item?.status?.toLowerCase() || '').includes(query.toLowerCase())
            );
        }
      });
    }

    setFilteredEquipment(filtered);
    setDisplayedEquipment(filtered.slice(0, INITIAL_ITEMS_TO_LOAD));
    setHasMore(filtered.length > INITIAL_ITEMS_TO_LOAD);
    setPage(1);
  };

  const handleCountryFilterPress = () => {
    if (countryButtonRef.current) {
      countryButtonRef.current.measureInWindow((x, y, width, height) => {
        setCountryDropdownPosition({ x, y: y + height + 4, width });
        setCountryDropdownVisible(true);
      });
    }
  };

  const handleCountrySelect = (country) => {
    // Never save 'Both Countries' to AsyncStorage, use 'All Countries' instead
    const countryToSave = country === 'Both Countries' ? 'All Countries' : country;
    setSelectedCountry(countryToSave);
    AsyncStorage.setItem('selectedCountry', countryToSave);
    filterEquipment(searchQuery, selectedFilter, countryToSave);
    setCountryDropdownVisible(false);
  };

  const handleVehicleTypeFilterPress = () => {
    if (vehicleTypeButtonRef.current) {
      vehicleTypeButtonRef.current.measureInWindow((x, y, width, height) => {
        setVehicleTypeDropdownPosition({ x, y: y + height + 4, width });
        setVehicleTypeDropdownVisible(true);
      });
    }
  };

  const handleVehicleTypeSelect = (vehicleType) => {
    setSelectedVehicleType(vehicleType);
    filterEquipment(searchQuery, selectedFilter, selectedCountry, vehicleType);
    setVehicleTypeDropdownVisible(false);
  };

  const handleFilterSelect = (filter) => {
    setSelectedFilter(filter);
    setFilterModalVisible(false);
    filterEquipment(searchQuery, filter, selectedCountry, selectedVehicleType);
  };

  const handleRefresh = async () => {
    if (!isMounted.current) return;
    
    const onlineStatus = await checkNetworkStatus();
    if (!onlineStatus) {
      showToast('error', 'Offline', 'Cannot sync while offline');
      return;
    }
    
    setIsRefreshing(true);
    setPage(1);
    setHasMore(true);
    setSelectedVehicleType('All Vehicle Types');
    await loadData(true, true);
    
    // Check for pending changes after sync
    await checkPendingChanges();
  };

  const handleScroll = ({ nativeEvent }) => {
    const { layoutMeasurement, contentOffset, contentSize } = nativeEvent;
    const isBottom = layoutMeasurement.height + contentOffset.y >= contentSize.height - 20;
    if (isBottom && hasMore && !isLoadingMore) {
      setIsLoadingMore(true);
      setPage((prev) => prev + 1);
    }
  };

  useEffect(() => {
    const startIndex = (page - 1) * ITEMS_PER_PAGE;
    const endIndex = startIndex + ITEMS_PER_PAGE;
    setDisplayedEquipment(filteredEquipment.slice(0, INITIAL_ITEMS_TO_LOAD + endIndex));
    setHasMore(filteredEquipment.length > INITIAL_ITEMS_TO_LOAD + endIndex);
  }, [page, filteredEquipment]);

  const handleActionPress = (item) => {
    if (!userInfo.equipmentPermissions.includes('view') && !userInfo.equipmentPermissions.includes('onlyMineView')) {
      showToast('error', 'Permission Denied', 'You do not have permission to view equipment.');
      return;
    }
    setSelectedEquipment(item);
    setActionModalVisible(true);
    setNewStatus(item.status || 'Available');
  };

  const handleStatusChange = async () => {
    if (!userInfo.equipmentPermissions.includes('update')) {
      showToast('error', 'Permission Denied', 'You do not have permission to update equipment.');
      return;
    }
    if (selectedEquipment) {
      const updatedEquipment = {
        ...selectedEquipment,
        status: newStatus,
        updatedAt: new Date().toISOString(),
        createdBy: userInfo.userUid, // Ensure createdBy is set during updates
      };
      let updatedFirebaseList;

      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      try {
        // Get user data from AsyncStorage (set during login)
        const userData = await AsyncStorage.getItem('userData');
        if (!userData) {
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('error', 'Error', 'User not authenticated. Please log in.');
          navigation.replace('AdminLogin');
          return;
        }

        const isOnline = await checkNetworkStatus();

        updatedFirebaseList = firebaseEquipment.map((item) =>
          item.id === updatedEquipment.id ? updatedEquipment : item
        );
        updatedFirebaseList.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        setFirebaseEquipment(updatedFirebaseList);
        setFilteredEquipment(updatedFirebaseList);
        setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));

        if (isOnline) {
          // Update MongoDB via API
          const response = await fetch(`https://api.titandrillingzm.com:6001/equipment/${updatedEquipment.id}`, {
            method: 'PUT',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(updatedEquipment),
          });

          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }

          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          showToast('success', 'Success', 'Status updated in MongoDB.');
        } else {
          updatedFirebaseList = updatedFirebaseList.map((item) =>
            item.id === updatedEquipment.id ? { ...item, pendingSync: true } : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          showToast('info', 'Offline', 'Status updated locally. Will sync when online.');
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
        }
      } catch (error) {
        console.error('Error updating status:', error.code, error.message);
        let errorMessage = 'Failed to update status.';
        if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Check Firebase security rules.';
        } else if (error.code === 'NETWORK_ERROR') {
          errorMessage = 'Offline: Status updated locally, will sync when online.';
          updatedFirebaseList = firebaseEquipment.map((item) =>
            item.id === updatedEquipment.id ? { ...updatedEquipment, pendingSync: true } : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
          showToast('info', 'Offline', errorMessage);
          return;
        }
        showToast('error', 'Error', errorMessage);
      } finally {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        setStatusModalVisible(false);
        setActionModalVisible(false);
        
        // Check for pending changes
        await checkPendingChanges();
      }
    }
  };

  const handleDelete = async () => {
    if (!userInfo.equipmentPermissions.includes('delete')) {
      showToast('error', 'Permission Denied', 'You do not have permission to delete equipment.');
      return;
    }
    if (selectedEquipment) {
      let updatedFirebaseList;

      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      try {
        // Get user data from AsyncStorage (set during login)
        const userData = await AsyncStorage.getItem('userData');
        if (!userData) {
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('error', 'Error', 'User not authenticated. Please log in.');
          navigation.replace('AdminLogin');
          return;
        }

        const isOnline = await checkNetworkStatus();

        updatedFirebaseList = firebaseEquipment.filter((item) => item.id !== selectedEquipment.id);
        setFirebaseEquipment(updatedFirebaseList);
        setFilteredEquipment(updatedFirebaseList);
        setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));

        if (isOnline) {
          // Delete from MongoDB via API
          const response = await fetch(`https://api.titandrillingzm.com:6001/equipment/${selectedEquipment.id}`, {
            method: 'DELETE',
            headers: {
              'Content-Type': 'application/json',
            },
          });

          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }

          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          showToast('success', 'Success', 'Equipment deleted from MongoDB.');
        } else {
          updatedFirebaseList = firebaseEquipment.map((item) =>
            item.id === selectedEquipment.id
              ? { ...item, pendingSync: true, pendingDelete: true }
              : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          showToast('info', 'Offline', 'Equipment deleted locally. Will sync when online.');
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
        }
      } catch (error) {
        console.error('Error deleting equipment:', error.code, error.message);
        let errorMessage = 'Failed to delete equipment.';
        if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Check Firebase security rules.';
        } else if (error.code === 'NETWORK_ERROR') {
          errorMessage = 'Offline: Equipment deleted locally, will sync when online.';
          updatedFirebaseList = firebaseEquipment.map((item) =>
            item.id === selectedEquipment.id
              ? { ...item, pendingSync: true, pendingDelete: true }
              : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
          showToast('info', 'Offline', errorMessage);
          return;
        }
        showToast('error', 'Error', errorMessage);
      } finally {
        setSyncModalVisible(false);
        setShowDeleteConfirm(false);
        rotateAnim.setValue(0);
        setActionModalVisible(false);
        
        // Check for pending changes
        await checkPendingChanges();
      }
    }
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  const getDynamicFieldLabels = (mainCategory) => {
    switch (mainCategory) {
      case 'Light Vehicles TDC':
      case 'Heavy Vehicles TDC':
        return { field1: 'Vehicle Number', field2: 'Registration', field3: 'VIN Number' };
      case 'TDC Rigs':
        return { field1: 'Rig Name', field2: 'Truck Type', field3: 'Chassis Number' };
      case 'Machinery':
        return { field1: 'Equipment Number', field2: 'Equipment Name', field3: 'VIN Number' };
      case 'Mud Mixer Trailers':
        return { field1: 'Number', field2: 'Serial Number', field3: 'Type' };
      case 'Trailers':
        return { field1: 'Power', field2: 'Type', field3: 'Equipment' };
      case 'Appliances':
      case 'Generators':
        return { field1: 'Type', field2: 'Type', field3: 'Serial' };
      case 'Workshop Machinery':
      case 'Pumps':
      case 'Tents':
        return { field1: 'Type', field2: 'Model', field3: 'Serial' };
      case 'Containers':
        return { field1: 'Purpose', field2: 'Container Type', field3: 'Container' };
      case 'Lightning Shelters':
        return { field1: 'Shelter', field2: 'Shelter', field3: 'Shelter' };
      default:
        return { field1: 'Field 1', field2: 'Field 2', field3: 'Field 3' };
    }
  };

  const getVehicleTypeFromCategory = (item) => {
    const { mainCategory } = item;
    if (mainCategory && typeof mainCategory === 'string') {
      if (mainCategory.toLowerCase().includes('light vehicle')) {
        return 'Light Vehicle';
      } else if (mainCategory.toLowerCase().includes('heavy vehicle')) {
        return 'Heavy Vehicle';
      } else if (mainCategory.toLowerCase().includes('drill rig') || mainCategory.toLowerCase().includes('rig')) {
        return 'Drill Rig';
      } else if (mainCategory.toLowerCase().includes('heavy plant') || mainCategory.toLowerCase().includes('plant equipment')) {
        return 'Heavy Plant Equipment';
      } else if (mainCategory.toLowerCase().includes('trailer')) {
        return 'Trailers';
      } else if (mainCategory.toLowerCase().includes('auxiliary')) {
        return 'Auxiliary Equipment';
      } else if (mainCategory.toLowerCase().includes('machine shop')) {
        return 'Machine Shop';
      } else if (mainCategory.toLowerCase().includes('boilershop') || mainCategory.toLowerCase().includes('boiler shop')) {
        return 'Boilershop';
      }
    }
    return mainCategory || 'Other';
  };

  const getFieldsToDisplay = (item) => {
    const { mainCategory, country } = item;
    
    // Debug: Log the item structure for the first few items
    if (item.id === firebaseEquipment[0]?.id || item.id === firebaseEquipment[1]?.id) {
      console.log('Equipment item structure:', {
        id: item.id,
        mainCategory: item.mainCategory,
        category: item.category,
        equipmentType: item.equipmentType,
        country: item.country,
        allFields: Object.keys(item)
      });
    }
    
    // Define field mappings based on category and country
    const fieldMappings = {
      'Light Vehicle': {
        DRC: [
          { label: 'VEHICLE NUMBER', value: item.vehicleNumber || item.equipmentNumber || 'N/A' },
          { label: 'REGISTRATION NUMBER', value: item.registrationNumber || item.registration || 'N/A' },
          { label: 'MAKE', value: item.make || item.manufacturer || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' }
        ],
        Zambia: [
          { label: 'VEHICLE NUMBER', value: item.vehicleNumber || item.equipmentNumber || 'N/A' },
          { label: 'REGISTRATION NUMBER', value: item.registrationNumber || item.registration || 'N/A' },
          { label: 'MAKE', value: item.make || item.manufacturer || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' }
        ]
      },
      'Heavy Vehicle': {
        DRC: [
          { label: 'VEHICLE NUMBER', value: item.vehicleNumber || item.equipmentNumber || 'N/A' },
          { label: 'REGISTRATION NUMBER', value: item.registrationNumber || item.registration || 'N/A' },
          { label: 'MAKE', value: item.make || item.manufacturer || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' }
        ],
        Zambia: [
          { label: 'VEHICLE NUMBER', value: item.vehicleNumber || item.equipmentNumber || 'N/A' },
          { label: 'REGISTRATION NUMBER', value: item.registrationNumber || item.registration || 'N/A' },
          { label: 'MAKE', value: item.make || item.manufacturer || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' }
        ]
      },
      'Drill Rig': {
        DRC: [
          { label: 'RIG NAME', value: item.rigName || item.equipmentName || 'N/A' },
          { label: 'RIG TYPE', value: item.rigType || item.equipmentType || 'N/A' },
          { label: 'RIG CHASIS NUMBER', value: item.rigChasisNumber || item.chasisNumber || item.chassisNumber || 'N/A' },
          { label: 'RIG ENGINE NUMBER', value: item.rigEngineNumber || item.engineNumber || 'N/A' }
        ],
        Zambia: [
          { label: 'RIG NAME', value: item.rigName || item.equipmentName || 'N/A' },
          { label: 'RIG TYPE', value: item.rigType || item.equipmentType || 'N/A' },
          { label: 'RIG CHASIS NUMBER', value: item.rigChasisNumber || item.chasisNumber || item.chassisNumber || 'N/A' },
          { label: 'RIG ENGINE NUMBER', value: item.rigEngineNumber || item.engineNumber || 'N/A' }
        ]
      },
      'Heavy Plant Equipment': {
        DRC: [
          { label: 'EQUIPMENT NAME', value: item.equipmentName || 'N/A' },
          { label: 'EQUIPMENT TYPE', value: item.equipmentType || item.mainCategory || 'N/A' },
          { label: 'CHASIS NUMBER', value: item.chasisNumber || item.chassisNumber || 'N/A' },
          { label: 'ENGINE NUMBER', value: item.engineNumber || 'N/A' }
        ],
        Zambia: [
          { label: 'EQUIPMENT NAME', value: item.equipmentName || 'N/A' },
          { label: 'EQUIPMENT TYPE', value: item.equipmentType || item.mainCategory || 'N/A' },
          { label: 'CHASIS NUMBER', value: item.chasisNumber || item.chassisNumber || 'N/A' },
          { label: 'ENGINE NUMBER', value: item.engineNumber || 'N/A' }
        ]
      },
      'Trailers': {
        DRC: [
          { label: 'TRAILER NUMBER', value: item.trailerNumber || item.equipmentNumber || 'N/A' },
          { label: 'REGISTRATION NUMBER', value: item.registrationNumber || item.registration || 'N/A' },
          { label: 'CHASIS NUMBER', value: item.chasisNumber || item.chassisNumber || 'N/A' },
          { label: 'TRAILER TYPE', value: item.trailerType || item.equipmentType || 'N/A' }
        ],
        Zambia: [
          { label: 'TRAILER NUMBER', value: item.trailerNumber || item.equipmentNumber || 'N/A' },
          { label: 'REGISTRATION NUMBER', value: item.registrationNumber || item.registration || 'N/A' },
          { label: 'CHASIS NUMBER', value: item.chasisNumber || item.chassisNumber || 'N/A' },
          { label: 'TRAILER TYPE', value: item.trailerType || item.equipmentType || 'N/A' }
        ]
      },
      'Auxiliary Equipment': {
        DRC: [
          { label: 'MACHINERY TYPE', value: item.machineryType || item.equipmentType || item.mainCategory || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' },
          { label: 'SERIAL NUMBER', value: item.serialNumber || item.serial || 'N/A' },
          { label: 'EQUIPMENT NUMBER', value: item.equipmentNumber || item.id || 'N/A' }
        ],
        Zambia: [
          { label: 'MACHINERY TYPE', value: item.machineryType || item.equipmentType || item.mainCategory || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' },
          { label: 'SERIAL NUMBER', value: item.serialNumber || item.serial || 'N/A' },
          { label: 'EQUIPMENT NUMBER', value: item.equipmentNumber || item.id || 'N/A' }
        ]
      },
      'Machine Shop': {
        DRC: [
          { label: 'MACHINERY TYPE', value: item.machineryType || item.equipmentType || item.mainCategory || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' },
          { label: 'SERIAL NUMBER', value: item.serialNumber || item.serial || 'N/A' },
          { label: 'EQUIPMENT NUMBER', value: item.equipmentNumber || item.id || 'N/A' }
        ],
        Zambia: [
          { label: 'MACHINERY TYPE', value: item.machineryType || item.equipmentType || item.mainCategory || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' },
          { label: 'SERIAL NUMBER', value: item.serialNumber || item.serial || 'N/A' },
          { label: 'EQUIPMENT NUMBER', value: item.equipmentNumber || item.id || 'N/A' }
        ]
      },
      'Boilershop': {
        DRC: [
          { label: 'MACHINERY TYPE', value: item.machineryType || item.equipmentType || item.mainCategory || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' },
          { label: 'SERIAL NUMBER', value: item.serialNumber || item.serial || 'N/A' },
          { label: 'EQUIPMENT NUMBER', value: item.equipmentNumber || item.id || 'N/A' }
        ],
        Zambia: [
          { label: 'MACHINERY TYPE', value: item.machineryType || item.equipmentType || item.mainCategory || 'N/A' },
          { label: 'MODEL', value: item.model || 'N/A' },
          { label: 'SERIAL NUMBER', value: item.serialNumber || item.serial || 'N/A' },
          { label: 'EQUIPMENT NUMBER', value: item.equipmentNumber || item.id || 'N/A' }
        ]
      }
    };

    // Get the category key (normalize the mainCategory)
    let categoryKey = mainCategory;
    if (mainCategory && typeof mainCategory === 'string') {
      // Handle variations in category names
      if (mainCategory.toLowerCase().includes('light vehicle')) {
        categoryKey = 'Light Vehicle';
      } else if (mainCategory.toLowerCase().includes('heavy vehicle')) {
        categoryKey = 'Heavy Vehicle';
      } else if (mainCategory.toLowerCase().includes('drill rig') || mainCategory.toLowerCase().includes('rig')) {
        categoryKey = 'Drill Rig';
      } else if (mainCategory.toLowerCase().includes('heavy plant') || mainCategory.toLowerCase().includes('plant equipment')) {
        categoryKey = 'Heavy Plant Equipment';
      } else if (mainCategory.toLowerCase().includes('trailer')) {
        categoryKey = 'Trailers';
      } else if (mainCategory.toLowerCase().includes('auxiliary')) {
        categoryKey = 'Auxiliary Equipment';
      } else if (mainCategory.toLowerCase().includes('machine shop')) {
        categoryKey = 'Machine Shop';
      } else if (mainCategory.toLowerCase().includes('boilershop') || mainCategory.toLowerCase().includes('boiler shop')) {
        categoryKey = 'Boilershop';
      }
    }

    // Get the country key
    const countryKey = country || 'DRC'; // Default to DRC if no country specified

    // Debug: Log the mapping decision
    if (item.id === firebaseEquipment[0]?.id) {
      console.log('Field mapping decision:', {
        originalCategory: mainCategory,
        normalizedCategory: categoryKey,
        country: countryKey,
        hasMapping: !!(fieldMappings[categoryKey] && fieldMappings[categoryKey][countryKey])
      });
    }

    // Return the appropriate fields or default fields
    if (fieldMappings[categoryKey] && fieldMappings[categoryKey][countryKey]) {
      return fieldMappings[categoryKey][countryKey];
    }

    // Return default fields if no specific mapping found
    return [
      { label: 'EQUIPMENT ID', value: item.id || item.equipmentId || 'N/A' },
      { label: 'EQUIPMENT NAME', value: item.equipmentName || 'N/A' },
      { label: 'CATEGORY', value: item.mainCategory || 'N/A' },
      { label: 'STATUS', value: item.status || 'N/A' }
    ];
  };

  const renderAdditionalFields = (item) => {
    const fields = [];
    
    // Debug: Log all available fields for the first item to see database structure
    if (item.id === firebaseEquipment[0]?.id) {
      console.log('All available fields for equipment:', Object.keys(item));
      console.log('Sample equipment data:', item);
    }
    
    // Get all available fields from the item
    const allFields = Object.keys(item);
    
    // Define fields to exclude from display (already shown elsewhere or not relevant)
    const excludeFields = [
      'id', 'firebaseKey', 'createdAt', 'updatedAt', 'createdBy', 'pendingSync', 'pendingDelete',
      'equipmentId', 'equipmentNumber', 'equipmentName', 'mainCategory', 'model', 'meterReading', 
      'meterUnit', 'status', 'country', 'serialNumber', 'serial', 'manufacturer', 'make', 'year', 
      'manufactureYear', 'dynamicField1', 'dynamicField2', 'dynamicField3'
    ];
    
    // Get remaining fields to display
    const remainingFields = allFields.filter(field => 
      !excludeFields.includes(field) && 
      item[field] !== null && 
      item[field] !== undefined && 
      item[field] !== '' && 
      item[field] !== 'N/A'
    );
    
    // Add remaining fields to display
    remainingFields.forEach(field => {
      if (item[field]) {
        // Convert field name to readable format
        const readableFieldName = field
          .replace(/([A-Z])/g, ' $1') // Add space before capital letters
          .replace(/^./, str => str.toUpperCase()) // Capitalize first letter
          .replace(/([A-Z])/g, ' $1') // Handle consecutive capitals
          .trim();
        
        // Handle different data types
        let displayValue = item[field];
        if (typeof displayValue === 'boolean') {
          displayValue = displayValue ? 'Yes' : 'No';
        } else if (typeof displayValue === 'object') {
          displayValue = JSON.stringify(displayValue);
        }
        
        fields.push({ 
          label: readableFieldName, 
          value: displayValue 
        });
      }
    });
    
    // Add the original dynamic fields if they exist and aren't already included
    const { field1, field2, field3 } = getDynamicFieldLabels(item.mainCategory);
    if (item.dynamicField1 && item.dynamicField1 !== 'N/A' && !fields.find(f => f.label === field1)) {
      fields.push({ label: field1, value: item.dynamicField1 });
    }
    if (item.dynamicField2 && item.dynamicField2 !== 'N/A' && !fields.find(f => f.label === field2)) {
      fields.push({ label: field2, value: item.dynamicField2 });
    }
    if (item.dynamicField3 && item.dynamicField3 !== '' && !fields.find(f => f.label === field3)) {
      fields.push({ label: field3, value: item.dynamicField3 });
    }

    return fields.map((field, index) => (
      <View key={index} style={styles.equipmentDetailItem}>
        <Text style={styles.detailLabel}>{field.label}</Text>
        <Text style={styles.detailValue}>{field.value}</Text>
      </View>
    ));
  };

  const getCountryOptions = () => {
    // Remove 'Both Countries' from the options
    const countries = ['All Countries', ...new Set(firebaseEquipment.map(item => item.country).filter(Boolean))].filter(c => c !== 'Both Countries');
    const countryCounts = countries.reduce((acc, country) => {
      if (country === 'All Countries') {
        acc[country] = firebaseEquipment.length;
      } else {
        acc[country] = firebaseEquipment.filter(item => item.country === country).length;
      }
      return acc;
    }, {});
    return { countries, countryCounts };
  };

  const getStatusOptions = () => [
    { label: 'Available', value: 'Available' },
    { label: 'In Maintenance', value: 'In Maintenance' },
    { label: 'Maintenance Required', value: 'Maintenance Required' },
    { label: 'Breakdown', value: 'Breakdown' },
    { label: 'Discontinue', value: 'Discontinue' },
    { label: 'Deployed', value: 'Deployed' },
    { label: 'Idle', value: 'Idle' },
  ];

  const getActionOptions = () => {
    const baseOptions = [];
    if (userInfo.equipmentPermissions.includes('view') || userInfo.equipmentPermissions.includes('onlyMineView')) {
      baseOptions.push(
        { label: 'Start Inspection', action: () => navigation.navigate('InsreportHeader', { equipment: selectedEquipment }) },
        { label: 'Equipment Overview', action: () => navigation.navigate('EquipmentOverview', { equipment: selectedEquipment }) }
      );
    }
    if (userInfo.equipmentPermissions.includes('update')) {
      baseOptions.push({
        label: 'Update',
        action: () => navigation.navigate('Add_equipment_update', { equipment: selectedEquipment }),
      });
      baseOptions.push({
        label: 'Change Status',
        action: () => {
          setStatusModalVisible(true);
          setNewStatus(selectedEquipment.status || 'Available');
        },
      });
    }
    if (userInfo.equipmentPermissions.includes('delete')) {
      baseOptions.push({
        label: 'Delete',
        action: () => setShowDeleteConfirm(true),
      });
    }
    return baseOptions;
  };

  const renderCountryDropdown = () => {
    const { countries, countryCounts } = getCountryOptions();
    return (
      <Modal visible={countryDropdownVisible} transparent animationType="none">
        <TouchableOpacity
          style={styles.dropdownOverlay}
          onPress={() => setCountryDropdownVisible(false)}
          activeOpacity={1}
        >
          <View
            style={[
              styles.dropdownContainer,
              {
                top: countryDropdownPosition.y,
                left: countryDropdownPosition.x,
                width: countryDropdownPosition.width,
                minWidth: 80,
                maxWidth: 300,
              },
            ]}
          >
            <ScrollView
              nestedScrollEnabled={true}
              showsVerticalScrollIndicator={true}
              style={{ maxHeight: 200 }}
            >
              {countries.map((country) => (
                <TouchableOpacity
                  key={country}
                  style={[
                    styles.dropdownItem,
                    selectedCountry === country && styles.dropdownItemActive,
                  ]}
                  onPress={() => handleCountrySelect(country)}
                >
                  <MaterialIcons
                    name="public"
                    size={width * 0.05}
                    color={selectedCountry === country ? '#3481BC' : '#333'}
                  />
                  <Text
                    style={[
                      styles.dropdownText,
                      selectedCountry === country && styles.dropdownTextActive,
                    ]}
                  >
                    {country === 'All Countries' ? 'All Countries' : country} ({countryCounts[country] || 0})
                  </Text>
                </TouchableOpacity>
              ))}
            </ScrollView>
          </View>
        </TouchableOpacity>
      </Modal>
    );
  };

  const renderVehicleTypeDropdown = () => {

    const availableVehicleTypes = firebaseEquipment
      .map(item => getVehicleTypeFromCategory(item))
      .filter(Boolean)
      .filter(type => type !== 'N/A' && type !== '');
    
    const vehicleTypes = ['All Vehicle Types', ...new Set(availableVehicleTypes)];
    const vehicleTypeCounts = vehicleTypes.reduce((acc, type) => {
      if (type === 'All Vehicle Types') {
        acc[type] = firebaseEquipment.length;
      } else {
        acc[type] = firebaseEquipment.filter(item => getVehicleTypeFromCategory(item) === type).length;
      }
      return acc;
    }, {});

    return (
      <Modal visible={vehicleTypeDropdownVisible} transparent animationType="none">
        <TouchableOpacity
          style={styles.dropdownOverlay}
          onPress={() => setVehicleTypeDropdownVisible(false)}
          activeOpacity={1}
        >
          <View
            style={[
              styles.dropdownContainer,
              {
                top: vehicleTypeDropdownPosition.y,
                left: vehicleTypeDropdownPosition.x,
                width: vehicleTypeDropdownPosition.width,
                minWidth: 120,
                maxWidth: 300,
              },
            ]}
          >
            <ScrollView
              nestedScrollEnabled={true}
              showsVerticalScrollIndicator={true}
              style={{ maxHeight: 200 }}
            >
              {vehicleTypes.map((type) => (
                <TouchableOpacity
                  key={type}
                  style={[
                    styles.dropdownItem,
                    selectedVehicleType === type && styles.dropdownItemActive,
                  ]}
                  onPress={() => handleVehicleTypeSelect(type)}
                >
                  <MaterialIcons
                    name="directions-car"
                    size={width * 0.05}
                    color={selectedVehicleType === type ? '#3481BC' : '#333'}
                  />
                  <Text
                    style={[
                      styles.dropdownText,
                      selectedVehicleType === type && styles.dropdownTextActive,
                    ]}
                  >
                    {type === 'All Vehicle Types' ? 'All Vehicle Types' : type} ({vehicleTypeCounts[type] || 0})
                  </Text>
                </TouchableOpacity>
              ))}
            </ScrollView>
          </View>
        </TouchableOpacity>
      </Modal>
    );
  };

  const renderFilterBar = () => {
    const { countryCounts } = getCountryOptions();
    let label = selectedCountry;
    let count = countryCounts[selectedCountry] || 0;
    if (selectedCountry === 'All Countries') {
      // If both Zambia and DRC exist, show 'All Countries (count)'
      const uniqueCountries = new Set(firebaseEquipment.map(item => item.country).filter(Boolean));
      if (uniqueCountries.has('Zambia') && uniqueCountries.has('DRC')) {
        label = 'All Countries';
        count = countryCounts['All Countries'] || 0;
      } else if (uniqueCountries.has('Zambia')) {
        label = 'Zambia';
        count = countryCounts['Zambia'] || 0;
      } else if (uniqueCountries.has('DRC')) {
        label = 'DRC';
        count = countryCounts['DRC'] || 0;
      }
    }

    // Get vehicle type count
    let vehicleTypeCount = firebaseEquipment.length;
    if (selectedVehicleType !== 'All Vehicle Types') {
      vehicleTypeCount = firebaseEquipment.filter(item => getVehicleTypeFromCategory(item) === selectedVehicleType).length;
    }
    
    // Check if vehicle types are available
    const hasVehicleTypes = firebaseEquipment.some(item => {
      const { mainCategory } = item;
      return mainCategory && mainCategory !== 'N/A' && mainCategory !== '';
    });
    return (
      <View style={{ backgroundColor: '#f5f5f5' }}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.filterBar}
        >
          <TouchableOpacity
            ref={countryButtonRef}
            style={styles.filterButton}
            onPress={handleCountryFilterPress}
          >
            <MaterialIcons name="public" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>
              Country: {label} ({count})
            </Text>
            <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
          </TouchableOpacity>
          {hasVehicleTypes && (
            <TouchableOpacity
              ref={vehicleTypeButtonRef}
              style={styles.filterButton}
              onPress={handleVehicleTypeFilterPress}
            >
              <MaterialIcons name="directions-car" size={width * 0.05} color="#333" style={styles.filterIcon} />
              <Text style={styles.filterButtonText}>
                Vehicle Type: {selectedVehicleType} ({vehicleTypeCount})
              </Text>
              <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
            </TouchableOpacity>
          )}
          <TouchableOpacity
            ref={permissionsButtonRef}
            style={styles.filterButton}
            onPress={() => setUserInfoModalVisible(true)}
          >
            <MaterialIcons name="lock" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>Permissions</Text>
          </TouchableOpacity>
        </ScrollView>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Equipments</Text>
          </View>
          <TouchableOpacity onPress={handleRefresh} style={styles.refreshButton}>
            <Image
              source={require('../../../Images/adminhome/equp/cloud.png')}
              style={styles.refreshIcon}
            />
          </TouchableOpacity>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder={`Search${selectedFilter ? ' by ' + selectedFilter : '...'}`}
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons
              name="search"
              size={width * 0.05}
              color="#888"
              style={styles.bladeIcon}
            />
          </View>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => setFilterModalVisible(true)}
          >
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Online/Offline Toggle & Sync Status Bar */}
      <View style={{
        flexDirection: 'row',
        alignItems: 'center',
        justifyContent: 'space-between',
        paddingHorizontal: 10,
        paddingVertical: 8,
        backgroundColor: isOnline ? '#E8F5E9' : '#FFF3E0',
        borderBottomWidth: 1,
        borderBottomColor: isOnline ? '#4CAF50' : '#FF9800',
      }}>
        <View style={{ flexDirection: 'row', alignItems: 'center', flex: 1 }}>
          {/* Status Indicator */}
          <View style={{
            width: 8,
            height: 8,
            borderRadius: 4,
            backgroundColor: isOnline ? '#4CAF50' : '#FF9800',
            marginRight: 8,
          }} />
          <View style={{ flex: 1 }}>
            <Text style={{ fontSize: 12, color: '#555', fontWeight: 'bold' }}>
              {isOnline ? '🌐 Online' : '📴 Offline'}
            </Text>
            <Text style={{ fontSize: 10, color: '#777' }}>
              {lastSyncTime ? `Synced ${getLastSyncText()}` : 'Not synced'}
              {pendingChanges && ' • Changes pending'}
            </Text>
          </View>
        </View>

        {/* Online/Offline Toggle */}
        <TouchableOpacity
          onPress={toggleOfflineMode}
          style={{
            width: 50,
            height: 28,
            borderRadius: 14,
            backgroundColor: forceOfflineMode ? '#ccc' : '#4CAF50',
            justifyContent: 'center',
            padding: 2,
          }}
          activeOpacity={0.8}
        >
          <View style={{
            width: 24,
            height: 24,
            borderRadius: 12,
            backgroundColor: '#fff',
            alignSelf: forceOfflineMode ? 'flex-start' : 'flex-end',
            shadowColor: '#000',
            shadowOpacity: 0.3,
            shadowRadius: 2,
            shadowOffset: { width: 0, height: 1 },
            elevation: 3,
          }} />
        </TouchableOpacity>
      </View>

      {renderFilterBar()}
      {renderCountryDropdown()}
      {firebaseEquipment.some(item => {
        const { mainCategory } = item;
        return mainCategory && mainCategory !== 'N/A' && mainCategory !== '';
      }) && renderVehicleTypeDropdown()}

      {isLoadingInitial ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#3481BC" />
          <Text style={styles.loadingText}>Loading Equipment...</Text>
        </View>
      ) : (
        <ScrollView
          style={styles.mainContent}
          onScroll={handleScroll}
          scrollEventThrottle={16}
        >
          {displayedEquipment.length > 0 ? (
            displayedEquipment.map((item) => (
                <View key={item.id} style={[
                  styles.equipmentRow,
                  item.pendingSync && { borderLeftWidth: 4, borderLeftColor: '#FF9800' }
                ]}>
                  <View style={styles.equipmentInfo}>
                    <View style={styles.equipmentDetailItemheader}>
                      <Text style={styles.detailLabelheader}>
                        {item.equipmentName || item.rigName || item.vehicleNumber || 'Equipment'}
                        {item.pendingSync && <Text style={{ color: '#FF9800', fontSize: width * 0.03 }}> • Pending Sync</Text>}
                      </Text>
                    </View>
                    
                    {/* Render dynamic fields based on category and country */}
                    {getFieldsToDisplay(item).map((field, index) => (
                      <View key={index} style={styles.equipmentDetailItem}>
                        <Text style={styles.detailLabel}>{field.label}</Text>
                        <Text style={styles.detailValue}>
                          {field.value}
                        </Text>
                      </View>
                    ))}
                    
                    {/* Status field */}
                    <View style={styles.equipmentDetailItemNoBorder}>
                      <Text
                        style={[
                          styles.detailLabelstatuss,
                          { color: getStatusColor(item.status) },
                        ]}
                      >
                        {item.status}
                      </Text>
                    </View>
                  </View>
                  {(userInfo.equipmentPermissions.includes('view') || userInfo.equipmentPermissions.includes('onlyMineView')) && (
                    <TouchableOpacity
                      style={styles.menuButton}
                      onPress={() => handleActionPress(item)}
                    >
                      <MaterialIcons name="more-vert" size={width * 0.06} color="#666" />
                    </TouchableOpacity>
                  )}
                </View>
            ))
          ) : (
            <Text style={styles.placeholderText}>
              {searchQuery
                ? 'No matching equipment found'
                : 'No equipment available'}
            </Text>
          )}
          {isLoadingMore && (
            <ActivityIndicator size="large" color="#3481BC" style={{ marginVertical: 20 }} />
          )}
        </ScrollView>
      )}

      <View style={styles.buttonContainer}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={styles.mapButtonGradient}
        >
          <TouchableOpacity style={styles.mapButton} onPress={() => navigation.navigate('map_view_equipment')}>
            <Image
              source={require('../../../Images/adminhome/equp/mapview_1.png')}
              style={styles.mapIcon}
            />
            <Text style={styles.addButtonTextmap}>Map View</Text>
          </TouchableOpacity>
        </LinearGradient>
        {userInfo.equipmentPermissions.includes('create') && (
          <LinearGradient
            colors={['#3481BC', '#025383']}
            style={styles.addButtonGradient}
          >
            <TouchableOpacity
              style={styles.addButton}
              onPress={() => navigation.navigate('Add_equipment')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.addButtonText}>Add Equipment</Text>
            </TouchableOpacity>
          </LinearGradient>
        )}
      </View>

      <Modal
        visible={actionModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setActionModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.actionModalContent}>
            <Text style={styles.actionModalTitle}>Actions</Text>
            {getActionOptions().map((option, index) => (
              <TouchableOpacity
                key={index}
                style={styles.actionItem}
                onPress={() => {
                  setActionModalVisible(false);
                  option.action();
                }}
              >
                <Text>{option.label}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setActionModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={statusModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setStatusModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.statusModalContent}>
            <LinearGradient
              colors={['#3481BC', '#025383']}
              style={styles.statusModalHeader}
            >
              <Text style={styles.statusModalTitle}>Change Status</Text>
            </LinearGradient>
            <View style={styles.statusDropdownWrapper}>
              <Dropdown
                style={styles.dropdown}
                containerStyle={styles.dropdownContainer}
                data={getStatusOptions()}
                labelField="label"
                valueField="value"
                placeholder="Select Status"
                value={newStatus}
                onChange={(item) => setNewStatus(item.value)}
                placeholderStyle={styles.dropdownPlaceholder}
                selectedTextStyle={styles.dropdownSelectedText}
                itemTextStyle={styles.dropdownItemText}
                maxHeight={200}
              />
            </View>
            <TouchableOpacity
              style={styles.applyButton}
              onPress={handleStatusChange}
            >
              <Text style={styles.applyButtonText}>Apply</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={filterModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setFilterModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.filterModalContent}>
            <Text style={styles.filterModalTitle}>Filter By</Text>
            {[
              'Created Date',
              'Equipment Number',
              'Equipment Name',
              'Equipment Categories',
              'Meter Reading',
              'Status',
            ].map((filter) => (
              <TouchableOpacity
                key={filter}
                style={styles.filterItem}
                onPress={() => handleFilterSelect(filter)}
              >
                <Text style={styles.filterItemText}>{filter}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setFilterModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={userInfoModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setUserInfoModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.userInfoModalContent}>
            <Text style={styles.filterModalTitle}>User Permissions</Text>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Countries:</Text>
              <Text style={styles.userInfoValue}>{(userInfo.countries && userInfo.countries.length > 0) ? userInfo.countries.join(', ') : 'None'}</Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Projects:</Text>
              <Text style={styles.userInfoValue}>{(userInfo.projects && userInfo.projects.length > 0) ? userInfo.projects.join(', ') : 'None'}</Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Equipment Permissions:</Text>
              <Text style={styles.userInfoValue}>{(userInfo.equipmentPermissions && userInfo.equipmentPermissions.length > 0) ? userInfo.equipmentPermissions.join(', ') : 'None'}</Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Employee Number:</Text>
              <Text style={styles.userInfoValue}>{userInfo.employeeNumber || 'N/A'}</Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Name:</Text>
              <Text style={styles.userInfoValue}>{userInfo.name || 'N/A'}</Text>
            </View>
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setUserInfoModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={syncModalVisible}
        transparent
        animationType="fade"
      >
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Processing Equipment Data</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={showDeleteConfirm}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setShowDeleteConfirm(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.deleteConfirmModalContent}>
            <Text style={styles.deleteConfirmTitle}>Confirm Delete</Text>
            <Text style={styles.deleteConfirmText}>
              Are you sure you want to delete this equipment?
            </Text>
            <View style={styles.deleteConfirmButtonContainer}>
              <TouchableOpacity
                style={[styles.deleteConfirmButton, { backgroundColor: '#666' }]}
                onPress={() => setShowDeleteConfirm(false)}
              >
                <Text style={styles.deleteConfirmButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.deleteConfirmButton, { backgroundColor: '#3481BC' }]}
                onPress={handleDelete}
              >
                <Text style={styles.deleteConfirmButtonText}>OK</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    fontSize: width * 0.04,
    color: '#333',
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    marginLeft: width * 0.025,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  filterBar: {
    flexDirection: 'row',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
  },
  filterButtonText: {
    fontSize: width * 0.035,
    color: '#333',
    marginLeft: width * 0.015,
    marginRight: width * 0.01,
  },
  filterIcon: {
    marginRight: width * 0.01,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'transparent',
  },
  dropdownContainer: {
    position: 'absolute',
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 5,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownItemActive: {
    backgroundColor: '#e6f0ff',
  },
  dropdownText: {
    marginLeft: width * 0.025,
    fontSize: width * 0.04,
    color: '#333',
  },
  dropdownTextActive: {
    color: '#3481BC',
    fontWeight: 'bold',
  },
  mainContent: {
    flex: 1,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: 10,
    fontSize: width * 0.04,
    color: '#333',
  },
  placeholderText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  equipmentRow: {
    marginHorizontal: width * 0.04,
    marginVertical: width * 0.01,
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#eee',
  },
  equipmentInfo: {
    flex: 1,
  },
  equipmentDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingHorizontal: width * 0.05,
    paddingRight: width * 0.1,
  },
  equipmentDetailItemNoBorder: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    paddingLeft: width * 0.05,
  },
  equipmentDetailItemheader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: 5,
  },
  detailLabel: {
    fontSize: width * 0.035,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: 3,
  },
  detailLabelstatuss: {
    fontSize: width * 0.035,
    fontWeight: '400',
    width: '60%',
    marginVertical: 3,
  },
  detailLabelheader: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: '600',
    width: '100%',
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#015185',
    width: '60%',
    textAlign: 'right',
    fontWeight: '500',
  },
  menuButton: {
    position: 'absolute',
    right: 10,
    bottom: 1,
    padding: 5,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  mapButtonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
    marginBottom: 10,
  },
  mapButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  mapIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.025,
  },
  addButtonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  addButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  addButtonTextmap: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  actionModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
  },
  actionModalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    marginBottom: width * 0.04,
    color: '#333',
  },
  actionItem: {
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  modalCloseButton: {
    marginTop: width * 0.04,
    alignItems: 'center',
  },
  modalCloseText: {
    color: '#3481BC',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  statusModalContent: {
    width: '80%',
    height: 220,
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    overflow: 'hidden',
    alignSelf: 'center',
  },
  statusModalHeader: {
    padding: 10,
  },
  statusModalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#fff',
    height: 40,
    textAlign: 'center',
    lineHeight: 40,
  },
  statusDropdownWrapper: {
    paddingHorizontal: width * 0.04,
    paddingVertical: 10,
    zIndex: 1000,
    alignItems: 'center',
  },
  dropdown: {
    backgroundColor: '#f0f0f0',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#ddd',
    height: 40,
    paddingHorizontal: 10,
    width: '100%',
  },
  dropdownContainer: {
    width: '65%',
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#ddd',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  dropdownPlaceholder: {
    fontSize: width * 0.035,
    color: '#888',
  },
  dropdownSelectedText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  dropdownItemText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  applyButton: {
    backgroundColor: '#3481BC',
    padding: width * 0.025,
    borderRadius: width * 0.012,
    alignItems: 'center',
    marginHorizontal: width * 0.04,
    marginVertical: 10,
  },
  applyButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  filterModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
  },
  filterModalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    marginBottom: width * 0.04,
    color: '#333',
  },
  filterItem: {
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  filterItemText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  userInfoModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
  },
  userInfoItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  userInfoLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '500',
    width: '40%',
  },
  userInfoValue: {
    fontSize: width * 0.04,
    color: '#333',
    width: '60%',
    textAlign: 'right',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: width * 0.025,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  icons12: {
    width: width * 0.06,
    height: width * 0.06,
  },
  deleteConfirmModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  deleteConfirmTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    marginBottom: width * 0.04,
    color: '#333',
  },
  deleteConfirmText: {
    fontSize: width * 0.04,
    color: '#666',
    marginBottom: width * 0.05,
    textAlign: 'center',
  },
  deleteConfirmButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  deleteConfirmButton: {
    flex: 1,
    padding: width * 0.025,
    borderRadius: width * 0.012,
    marginHorizontal: width * 0.012,
    alignItems: 'center',
  },
  deleteConfirmButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
});

export default EquipmentOne;