import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import AsyncStorage from '@react-native-async-storage/async-storage';
import NetInfo from '@react-native-community/netinfo';
import equipmentData from './EquipmentCategory.json';

// Normalize JSON data and add dynamic fields
const normalizeEquipmentData = () => {
  const normalized = [];
  const currentDate = new Date();
  let nameCounter = 1;

  const randomDate = () => new Date(currentDate - Math.random() * 30 * 24 * 60 * 60 * 1000).toISOString();

  // Helper function to create dynamic fields
  const createDynamicFields = (item, category) => {
    switch (category) {
      case 'Light Vehicles TDC':
      case 'Heavy Vehicles TDC':
        return [
          { key: 'Vehicle Number', value: item['Vehicle Number'] || 'N/A' },
          { key: 'Registration', value: item['Reg Num'] || 'N/A' },
          { key: 'VIN Number', value: item['VIN Number'] || item['VIN Number/Chas#'] || 'N/A' },
        ];
      case 'Containers':
        return [
          { key: 'Purpose', value: item['Purpose'] || 'N/A' },
          { key: 'Container Type', value: item['Container Type'] || 'N/A' },
        ];
      case 'TDC Rigs':
        return [
          { key: 'Rig Name', value: item['Rig Name'] || 'N/A' },
          { key: 'Truck Type', value: item['Truck Type'] || 'N/A' },
          { key: 'Chassis Number', value: item['Chassis Number'] || 'N/A' },
        ];
      case 'Machinery':
        return [
          { key: 'Equipment Number', value: item['Equipment Num'] || 'N/A' },
          { key: 'Equipment Name', value: item['Equipment Name'] || 'N/A' },
          { key: 'VIN Number', value: item['VIN Number/Chas#'] || 'N/A' },
        ];
      case 'Mud Mixer Trailers':
        return [
          { key: 'Number', value: item['Number'] || 'N/A' },
          { key: 'Serial Number', value: item['Serial Number'] || 'N/A' },
          { key: 'Type', value: item['Type'] || 'N/A' },
        ];
      case 'Lightning Shelters':
        return [
          { key: 'Shelter Name', value: item['Shelter'] || 'N/A' },
        ];
      case 'Trailers':
        return [
          { key: 'Type', value: item['Type'] || 'N/A' },
          { key: 'Chassis Number', value: item['Chassis#'] || 'N/A' },
        ];
      case 'Appliances':
      case 'Generators':
        return [
          { key: 'Type', value: item['Type'] || 'N/A' },
          { key: 'Model', value: item['Model'] || 'N/A' },
          { key: 'Serial Number', value: item['Model/Serial Num'] || 'N/A' },
        ];
      case 'Workshop Machinery':
      case 'Pumps':
      case 'Tents':
        return [
          { key: 'Type', value: item['Type'] || 'N/A' },
          { key: 'Model', value: item['Model'] || 'N/A' },
          { key: 'Serial Number', value: item['Serial Num'] || 'N/A' },
        ];
      default:
        return [];
    }
  };

  // Light Vehicles TDC
  equipmentData['Light Vehicles TDC']?.forEach((item, index) => {
    normalized.push({
      id: `lv-${index}`,
      equipmentNumber: item['Vehicle Number'] || `LV-${index + 1}`,
      equipmentName: `Vehicle ${nameCounter++}`,
      equipmentCategory: 'Light Vehicle',
      mainCategory: 'Light Vehicles TDC',
      model: item['Vehicle Type'] || 'N/A',
      meterReading: Math.floor(Math.random() * 10000),
      meterUnit: 'Miles',
      status: 'Available',
      createdAt: randomDate(),
      country: item['Vehicle Number']?.includes('ZAMBIA') ? 'Zambia' : 'Namibia',
      vin: item['VIN Number'] || 'N/A',
      regNum: item['Reg Num'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Light Vehicles TDC'),
    });
  });

  // Heavy Vehicles TDC
  equipmentData['Heavy Vehicles TDC']?.forEach((item, index) => {
    normalized.push({
      id: `hv-${index}`,
      equipmentNumber: item['Vehicle Number'] || `HV-${index + 1}`,
      equipmentName: `Vehicle ${nameCounter++}`,
      equipmentCategory: 'Heavy Vehicle',
      mainCategory: 'Heavy Vehicles TDC',
      model: item['Vehicle Type'] || 'N/A',
      meterReading: Math.floor(Math.random() * 10000),
      meterUnit: 'Miles',
      status: 'Available',
      createdAt: randomDate(),
      country: item['Vehicle Number']?.includes('ZAMBIA') ? 'Zambia' : 'Namibia',
      vin: item['VIN Number/Chas#'] || 'N/A',
      regNum: item['Reg Num'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Heavy Vehicles TDC'),
    });
  });

  // Containers
  equipmentData['Containers']?.forEach((item, index) => {
    normalized.push({
      id: `cont-${index}`,
      equipmentNumber: `CONT-${index + 1}`,
      equipmentName: `Container ${nameCounter++}`,
      equipmentCategory: 'Container',
      mainCategory: 'Containers',
      model: item['Container Type'] || 'N/A',
      meterReading: 0,
      meterUnit: 'N/A',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      purpose: item['Purpose'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Containers'),
    });
  });

  // TDC Rigs
  equipmentData['TDC Rigs']?.forEach((item, index) => {
    normalized.push({
      id: `rig-${index}`,
      equipmentNumber: item['Rig Name'] || `RIG-${index + 1}`,
      equipmentName: `Rig ${nameCounter++}`,
      equipmentCategory: 'Rig',
      mainCategory: 'TDC Rigs',
      model: item['Rig Type'] || 'N/A',
      meterReading: Math.floor(Math.random() * 5000),
      meterUnit: 'Hours',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      truckType: item['Truck Type'] || 'N/A',
      chassisNumber: item['Chassis Number'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'TDC Rigs'),
    });
  });

  // Machinery
  equipmentData['Machinery']?.forEach((item, index) => {
    normalized.push({
      id: `mach-${index}`,
      equipmentNumber: item['Equipment Num'] || `MACH-${index + 1}`,
      equipmentName: `Machinery ${nameCounter++}`,
      equipmentCategory: 'Machinery',
      mainCategory: 'Machinery',
      model: item['Equipment Type'] || 'N/A',
      meterReading: Math.floor(Math.random() * 5000),
      meterUnit: 'Hours',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      vin: item['VIN Number/Chas#'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Machinery'),
    });
  });

  // Mud Mixer Trailers
  equipmentData['Mud Mixer Trailers']?.forEach((item, index) => {
    normalized.push({
      id: `trailer-${index}`,
      equipmentNumber: `TRAILER-${item['Number'] || index + 1}`,
      equipmentName: `Trailer ${nameCounter++}`,
      equipmentCategory: 'Mud Mixer Trailer',
      mainCategory: 'Mud Mixer Trailers',
      model: item['Type'] || 'N/A',
      meterReading: 0,
      meterUnit: 'N/A',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      serialNumber: item['Serial Number'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Mud Mixer Trailers'),
    });
  });

  // Lightning Shelters
  equipmentData['Lightning Shelters']?.forEach((item, index) => {
    normalized.push({
      id: `shelter-${index}`,
      equipmentNumber: `SHELTER-${index + 1}`,
      equipmentName: `Shelter ${nameCounter++}`,
      equipmentCategory: 'Lightning Shelter',
      mainCategory: 'Lightning Shelters',
      model: 'Shelter',
      meterReading: 0,
      meterUnit: 'N/A',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      shelterName: item['Shelter'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Lightning Shelters'),
    });
  });

  // Trailers
  equipmentData['Trailers']?.forEach((item, index) => {
    normalized.push({
      id: `rod-trailer-${index}`,
      equipmentNumber: `ROD-TRAILER-${index + 1}`,
      equipmentName: `Trailer ${nameCounter++}`,
      equipmentCategory: 'Trailer',
      mainCategory: 'Trailers',
      model: item['Type'] || 'N/A',
      meterReading: 0,
      meterUnit: 'N/A',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      chassisNumber: item['Chassis#'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Trailers'),
    });
  });

  // Appliances
  equipmentData['Appliances']?.forEach((item, index) => {
    normalized.push({
      id: `appliance-${index}`,
      equipmentNumber: `APP-${index + 1}`,
      equipmentName: `Appliance ${nameCounter++}`,
      equipmentCategory: 'Appliance',
      mainCategory: 'Appliances',
      model: item['Model'] || 'N/A',
      meterReading: 0,
      meterUnit: 'N/A',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      serialNumber: item['Model/Serial Num'] || 'N/A',
      applianceType: item['Type'] || 'N/A',
      dynamic : createDynamicFields(item, 'Appliances'),
    });
  });

  // Generators
  equipmentData['Generators']?.forEach((item, index) => {
    normalized.push({
      id: `gen-${index}`,
      equipmentNumber: `GEN-${index + 1}`,
      equipmentName: `Generator ${nameCounter++}`,
      equipmentCategory: 'Generator',
      mainCategory: 'Generators',
      model: item['Model'] || 'N/A',
      meterReading: Math.floor(Math.random() * 2000),
      meterUnit: 'Hours',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      serialNumber: item['Model/Serial Num'] || 'N/A',
      generatorType: item['Type'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Generators'),
    });
  });

  // Workshop Machinery
  equipmentData['Workshop Machinery']?.forEach((item, index) => {
    normalized.push({
      id: `wmach-${index}`,
      equipmentNumber: `WMACH-${index + 1}`,
      equipmentName: `Machinery ${nameCounter++}`,
      equipmentCategory: 'Workshop Machinery',
      mainCategory: 'Workshop Machinery',
      model: item['Model'] || 'N/A',
      meterReading: Math.floor(Math.random() * 1000),
      meterUnit: 'Hours',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      serialNumber: item['Serial Num'] || 'N/A',
      machineryType: item['Type'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Workshop Machinery'),
    });
  });

  // Pumps
  equipmentData['Pumps']?.forEach((item, index) => {
    normalized.push({
      id: `pump-${index}`,
      equipmentNumber: `PUMP-${index + 1}`,
      equipmentName: `Pump ${nameCounter++}`,
      equipmentCategory: 'Pump',
      mainCategory: 'Pumps',
      model: item['Model'] || 'N/A',
      meterReading: Math.floor(Math.random() * 1000),
      meterUnit: 'Hours',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      serialNumber: item['Serial Num'] || 'N/A',
      pumpType: item['Type'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Pumps'),
    });
  });

  // Tents
  equipmentData['Tents']?.forEach((item, index) => {
    normalized.push({
      id: `tent-${index}`,
      equipmentNumber: `TENT-${index + 1}`,
      equipmentName: `Tent ${nameCounter++}`,
      equipmentCategory: 'Tent',
      mainCategory: 'Tents',
      model: item['Model'] || 'N/A',
      meterReading: 0,
      meterUnit: 'N/A',
      status: 'Available',
      createdAt: randomDate(),
      country: 'Namibia',
      serialNumber: item['Serial Num'] || 'N/A',
      tentType: item['Type'] || 'N/A',
      dynamicFields: createDynamicFields(item, 'Tents'),
    });
  });

  return normalized.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
};

// Upload JSON data to Firebase
export const uploadEquipmentToDatabase = async (userId) => {
  try {
    const isOnline = await checkNetworkStatus();
    if (!isOnline) {
      console.error('No internet connection. Cannot upload data.');
      return false;
    }

    const equipmentRef = database().ref(`/equipmentsvehicles/${userId}`);
    const snapshot = await equipmentRef.once('value');
    const existingData = snapshot.val();

    if (!existingData) {
      const normalizedData = normalizeEquipmentData();
      const equipmentMap = {};
      normalizedData.forEach((item) => {
        equipmentMap[item.id] = item;
      });
      await equipmentRef.set(equipmentMap);
      await AsyncStorage.setItem('equipmentsvehicles', JSON.stringify(normalizedData));
      console.log('Equipment data uploaded to Firebase.');
      return true;
    } else {
      console.log('Equipment data already exists in Firebase.');
      return false;
    }
  } catch (error) {
    console.error('Error uploading equipment to database:', error);
    return false;
  }
};

// Fetch equipment data from Firebase
export const fetchEquipmentFromDatabase = async (userId) => {
  try {
    const isOnline = await checkNetworkStatus();
    let equipmentData = [];

    if (isOnline) {
      const equipmentRef = database().ref(`/equipmentsvehicles/${userId}`);
      const snapshot = await equipmentRef.once('value');
      const data = snapshot.val();
      equipmentData = data
        ? Object.values(data).sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt))
        : [];
      await AsyncStorage.setItem('equipmentsvehicles', JSON.stringify(equipmentData));
    } else {
      const savedData = await AsyncStorage.getItem('equipmentsvehicles');
      equipmentData = savedData ? JSON.parse(savedData) : [];
    }

    return equipmentData;
  } catch (error) {
    console.error('Error fetching equipment from database:', error);
    const savedData = await AsyncStorage.getItem('equipmentsvehicles');
    return savedData ? JSON.parse(savedData) : [];
  }
};

// Update equipment status
export const updateEquipmentStatus = async (userId, equipmentId, newStatus) => {
  try {
    const isOnline = await checkNetworkStatus();
    const equipmentRef = database().ref(`/equipmentsvehicles/${userId}/${equipmentId}`);

    if (isOnline) {
      await equipmentRef.update({ status: newStatus });
      const savedData = await AsyncStorage.getItem('equipmentsvehicles');
      let equipmentData = savedData ? JSON.parse(savedData) : [];
      equipmentData = equipmentData.map((item) =>
        item.id === equipmentId ? { ...item, status: newStatus } : item
      );
      await AsyncStorage.setItem('equipmentsvehicles', JSON.stringify(equipmentData));
      return true;
    } else {
      let savedData = await AsyncStorage.getItem('equipmentsvehicles');
      let equipmentData = savedData ? JSON.parse(savedData) : [];
      equipmentData = equipmentData.map((item) =>
        item.id === equipmentId ? { ...item, status: newStatus, pendingSync: true } : item
      );
      await AsyncStorage.setItem('equipmentsvehicles', JSON.stringify(equipmentData));
      return false;
    }
  } catch (error) {
    console.error('Error updating equipment status:', error);
    return false;
  }
};

// Delete equipment
export const deleteEquipment = async (userId, equipmentId) => {
  try {
    const isOnline = await checkNetworkStatus();
    const equipmentRef = database().ref(`/equipmentsvehicles/${userId}/${equipmentId}`);

    if (isOnline) {
      await equipmentRef.remove();
      const savedData = await AsyncStorage.getItem('equipmentsvehicles');
      let equipmentData = savedData ? JSON.parse(savedData) : [];
      equipmentData = equipmentData.filter((item) => item.id !== equipmentId);
      await AsyncStorage.setItem('equipmentsvehicles', JSON.stringify(equipmentData));
      return true;
    } else {
      let savedData = await AsyncStorage.getItem('equipmentsvehicles');
      let equipmentData = savedData ? JSON.parse(savedData) : [];
      equipmentData = equipmentData
        .filter((item) => item.id !== equipmentId)
        .map((item) => ({ ...item, pendingSync: true }));
      await AsyncStorage.setItem('equipmentsvehicles', JSON.stringify(equipmentData));
      return false;
    }
  } catch (error) {
    console.error('Error deleting equipment:', error);
    return false;
  }
};

// Sync pending changes
export const syncPendingChanges = async (userId) => {
  try {
    const isOnline = await checkNetworkStatus();
    if (!isOnline) return false;

    let savedData = await AsyncStorage.getItem('equipmentsvehicles');
    let equipmentData = savedData ? JSON.parse(savedData) : [];
    const pending = equipmentData.filter((item) => item.pendingSync);

    if (pending.length > 0) {
      const equipmentRef = database().ref(`/equipmentsvehicles/${userId}`);
      for (const item of pending) {
        await equipmentRef.child(item.id).set({ ...item, pendingSync: false });
      }
      equipmentData = equipmentData.map((item) =>
        item.pendingSync ? { ...item, pendingSync: false } : item
      );
      await AsyncStorage.setItem('equipmentsvehicles', JSON.stringify(equipmentData));
      return true;
    }
    return false;
  } catch (error) {
    console.error('Error syncing pending changes:', error);
    return false;
  }
};

// Check network status
const checkNetworkStatus = async () => {
  try {
    const netInfo = await NetInfo.fetch();
    return netInfo.isConnected;
  } catch (error) {
    console.error('Error checking network status:', error);
    return false;
  }
};