import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  TextInput,
  Dimensions,
  Image,
  Modal,
  Alert,
  Share,
  Platform,
  Linking,
  Animated,
  KeyboardAvoidingView,
  PermissionsAndroid,
  BackHandler,
  ActivityIndicator,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import { useNavigation } from '@react-navigation/native';
import NetInfo from '@react-native-community/netinfo';
import { Buffer } from 'buffer';
// Firebase imports removed - using MongoDB only
import { check, request, PERMISSIONS, RESULTS } from 'react-native-permissions';
import VoiceToText, { VoiceToTextEvents } from '@appcitor/react-native-voice-to-text';
import LottieView from 'lottie-react-native';

// Placeholder for DvirPChecklist
const DvirPChecklist = {
  requirements: {
    label: "Requirements",
    items: [
      { id: "req1", label: "I hold a valid driver's licence" },
      { id: "req2", label: "I will obey speed limits and road signs" },
      { id: "req3", label: "I will not overload the vehicle and ensure the load is secure" },
      { id: "req4", label: "I have completed the vehicle pre check list and satisfied all is order for my trip" },
      { id: "req5", label: "I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)" },
      { id: "req6", label: "I have read and understand the Titan driving procedure, National driving rules and cliental rules" },
      { id: "req7", label: "I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)" },
    ],
  },
  hazards: {
    label: "Hazards – Are you aware of:",
    items: [
      { id: "haz1", label: "Pedestrians crossing the road, school children and drunkards" },
      { id: "haz2", label: "Motor bikes and taxis stopping suddenly or pulling in to the road suddenly" },
      { id: "haz3", label: "Oncoming vehicles overtaking on blind rises or corners" },
      { id: "haz4", label: "Heavy trucks or plant taking up width of the road" },
      { id: "haz5", label: "Riots or unrest" },
      { id: "haz6", label: "Gravel/Sandy roads can be slippery – Loss of control" },
      { id: "haz7", label: "Tyre blowouts or punctures – Loss of control" },
    ],
  },
};

const { height, width } = Dimensions.get('window');

// MongoDB DVIR API URL
const MONGODB_DVIR_API_URL = 'https://api.titandrillingzm.com:6006';

// DVIR Notifications API URL
const DVIR_NOTIFICATIONS_API_URL = 'https://api.titandrillingzm.com:6020';

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, onClose }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')} // Ensure this path is correct
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>Generating Report</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const DvirSummeryStep3 = ({
  setActiveStep = () => {},
  signatureStatus = false,
  signatureData = null,
  onSignaturePress = () => {},
}) => {
  const navigation = useNavigation();
  const [equipmentStatus, setEquipmentStatus] = useState(null);
  const [priority, setPriority] = useState(null);
  const [safeToUse, setSafeToUse] = useState(null);
  const [additionalNote, setAdditionalNote] = useState('');
  const [overallCondition, setOverallCondition] = useState(null);
  const [modalVisible, setModalVisible] = useState(false);
  const [faultyItems, setFaultyItems] = useState(0);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [savedPdfPath, setSavedPdfPath] = useState(null);
  const [reportId, setReportId] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isListening, setIsListening] = useState(false);
  const [voiceError, setVoiceError] = useState(null);
  const [micPermissionGranted, setMicPermissionGranted] = useState(false);
  const [isVoiceModuleAvailable, setIsVoiceModuleAvailable] = useState(false);
  const [isRecognitionAvailable, setIsRecognitionAvailable] = useState(true);
  const [voiceModalVisible, setVoiceModalVisible] = useState(false);
  const [partialText, setPartialText] = useState('');
  const [volume, setVolume] = useState(0);
  const [ignoreVoiceEvents, setIgnoreVoiceEvents] = useState(false);
  const [isStopping, setIsStopping] = useState(false);
  const [errors, setErrors] = useState({
    overallCondition: '',
    equipmentStatus: '',
    safeToUse: '',
    signature: '',
  });
  const additionalNoteRef = useRef(additionalNote);
  const lottieRef = useRef(null);
  const ignoreVoiceEventsRef = useRef(false);
  const isMountedRef = useRef(true);
  const listenersRef = useRef([]);
  const isCleaningUpRef = useRef(false);
  const isListeningRef = useRef(false);
  const cleanupTimeoutRef = useRef(null);
  const stopFallbackTimeoutRef = useRef(null);
  const pendingStopShouldAddRef = useRef(false);
  const hasStoppedRecordingRef = useRef(false);
  const hasReceivedFinalResultsRef = useRef(false);
  const recordingStartTimeRef = useRef(null);
  const isRestartingRef = useRef(false);
  const lastFinalTextRef = useRef('');

  useEffect(() => {
    additionalNoteRef.current = additionalNote;
  }, [additionalNote]);

  const saveState = useCallback(
    async (overrides = {}) => {
    try {
      const state = {
        equipmentStatus,
        priority,
        safeToUse,
        additionalNote,
        overallCondition,
        ...overrides,
      };
      await AsyncStorage.setItem('Summerystep3State', JSON.stringify(state));
      console.log('Persisted state saved:', state);
    } catch (error) {
      console.error('Error saving persisted state:', error);
      }
    },
    [equipmentStatus, priority, safeToUse, additionalNote, overallCondition]
  );

  const appendVoiceTextToAdditionalNote = useCallback((incomingText) => {
    const trimmedFinal = (incomingText || '').trim();
    if (!trimmedFinal) {
      return;
    }

    let nextValue = '';

    setAdditionalNote((prevValue) => {
      const currentValue = typeof prevValue === 'string' ? prevValue : '';
      const currentTrimmed = currentValue.trim();

      if (!currentTrimmed) {
        lastFinalTextRef.current = trimmedFinal;
        nextValue = trimmedFinal;
        return trimmedFinal;
      }

      if (lastFinalTextRef.current === trimmedFinal) {
        nextValue = currentValue;
        return currentValue;
      }

      if (currentTrimmed.endsWith(trimmedFinal)) {
        nextValue = currentValue;
        return currentValue;
      }

      const lastFewWords = currentTrimmed.split(' ').slice(-3).join(' ');
      const firstFewWords = trimmedFinal.split(' ').slice(0, 3).join(' ');
      if (lastFewWords && firstFewWords && lastFewWords === firstFewWords && lastFewWords.length > 5) {
        nextValue = currentValue;
        return currentValue;
      }

      if (trimmedFinal.length > 15 && currentTrimmed.includes(trimmedFinal)) {
        nextValue = currentValue;
        return currentValue;
      }

      const updated = currentTrimmed ? `${currentTrimmed} ${trimmedFinal}` : trimmedFinal;
      lastFinalTextRef.current = trimmedFinal;
      nextValue = updated;
      return updated;
    });

    if (nextValue) {
      setTimeout(() => {
        saveState({ additionalNote: nextValue });
      }, 0);
    }
  }, [saveState]);

  const onSpeechResults = useCallback(
    (event) => {
      if (!isMountedRef.current || isCleaningUpRef.current || ignoreVoiceEventsRef.current) {
        return;
      }

      hasReceivedFinalResultsRef.current = true;

      try {
        let payload = event?.value ?? event?.results ?? event?.text ?? event;
        let finalText = '';

        if (Array.isArray(payload)) {
          if (payload.length) {
            finalText = payload.join(' ');
          }
        } else if (typeof payload === 'string') {
          finalText = payload;
        } else if (payload && typeof payload === 'object') {
          const extracted = Object.values(payload)
            .flatMap((v) => (Array.isArray(v) ? v : [v]))
            .filter((v) => typeof v === 'string');
          if (extracted.length) {
            finalText = extracted.join(' ');
          } else {
            finalText = String(payload ?? '');
          }
        } else {
          finalText = String(payload ?? '');
        }

        if (finalText && finalText.trim()) {
          const trimmedFinal = finalText.trim();

          if (lastFinalTextRef.current === trimmedFinal) {
            setPartialText('');
            return;
          }

          appendVoiceTextToAdditionalNote(trimmedFinal);
          setPartialText('');
        }
      } catch (err) {
        console.warn('onSpeechResults parse error', err, event);
      }
    },
    [appendVoiceTextToAdditionalNote]
  );

  const onSpeechError = useCallback(
    (event) => {
      if (!isMountedRef.current || isCleaningUpRef.current) {
        return;
      }

      const message = event?.error?.message || event?.message || JSON.stringify(event);
      const lowerMessage = String(message).toLowerCase();
      const code = String(event?.error?.code ?? event?.code ?? '');
      const isSilenceError =
        lowerMessage.includes('no match') ||
        lowerMessage.includes('timeout') ||
        lowerMessage.includes('11') ||
        code.includes('11');

      if (isSilenceError) {
        setVoiceError(null);
        setIgnoreVoiceEvents(false);

        if (
          isMountedRef.current &&
          !isCleaningUpRef.current &&
          !hasStoppedRecordingRef.current &&
          isListeningRef.current &&
          !isRestartingRef.current
        ) {
          isRestartingRef.current = true;
          setTimeout(restartListening, 200);
        }
        return;
      }

      setVoiceError(`Speech recognition error: ${message}`);
      isListeningRef.current = false;
      setIsListening(false);
      setVoiceModalVisible(false);
      setIsStopping(false);
    },
    [restartListening]
  );

  const checkMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.RECORD_AUDIO);
      setMicPermissionGranted(granted);
      return granted;
    } catch (err) {
      console.warn('Permission check error:', err);
      return false;
    }
  };

  const requestMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.RECORD_AUDIO,
        {
          title: 'Microphone Permission',
          message: 'The app needs microphone access to capture your note by voice.',
          buttonPositive: 'OK',
        }
      );
      const isGranted = granted === PermissionsAndroid.RESULTS.GRANTED;
      setMicPermissionGranted(isGranted);
      return isGranted;
    } catch (err) {
      console.warn('Permission request error:', err);
      return false;
    }
  };

  const restartListening = useCallback(async () => {
    if (
      !VoiceToText ||
      !isMountedRef.current ||
      isCleaningUpRef.current ||
      hasStoppedRecordingRef.current
    ) {
      isRestartingRef.current = false;
        return;
      }

    try {
      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      }
    } catch (restartError) {
      console.warn('Failed to auto-restart recording:', restartError);
    } finally {
      isRestartingRef.current = false;
    }
  }, []);

  const finalizeStop = useCallback(
    (shouldAddPartialText = false) => {
      setIgnoreVoiceEvents(true);
      isListeningRef.current = false;
      hasStoppedRecordingRef.current = true;
      pendingStopShouldAddRef.current = false;

      if (!isMountedRef.current || isCleaningUpRef.current) {
        return;
      }

      if (shouldAddPartialText && !hasReceivedFinalResultsRef.current) {
        setPartialText((currentPartial) => {
          if (currentPartial && currentPartial.trim()) {
            appendVoiceTextToAdditionalNote(currentPartial);
          }
          return '';
        });
      } else {
        setPartialText('');
      }

      setIsListening(false);
      setVoiceModalVisible(false);
      setIsStopping(false);

      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }
    },
    [appendVoiceTextToAdditionalNote]
  );

  const safeStopRecording = useCallback(
    ({ shouldAddPartialText = false, showProcessing = false } = {}) => {
      pendingStopShouldAddRef.current = shouldAddPartialText;
      setIgnoreVoiceEvents(true);

      if (hasStoppedRecordingRef.current) {
        isListeningRef.current = false;
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          setIsStopping(true);
        }
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 200 : 0);
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          if (stopFallbackTimeoutRef.current) {
            clearTimeout(stopFallbackTimeoutRef.current);
          }
          stopFallbackTimeoutRef.current = setTimeout(() => {
            finalizeStop(pendingStopShouldAddRef.current);
          }, 3000);
        }
        return;
      }

      if (!isListeningRef.current) {
        isListeningRef.current = false;
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 200 : 0);
        if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
          if (stopFallbackTimeoutRef.current) {
            clearTimeout(stopFallbackTimeoutRef.current);
          }
          stopFallbackTimeoutRef.current = setTimeout(() => {
            finalizeStop(pendingStopShouldAddRef.current);
          }, 3000);
        }
        return;
      }

      if (showProcessing && isMountedRef.current && !isCleaningUpRef.current) {
        setIsStopping(true);
        if (stopFallbackTimeoutRef.current) {
          clearTimeout(stopFallbackTimeoutRef.current);
        }
        stopFallbackTimeoutRef.current = setTimeout(() => {
          finalizeStop(pendingStopShouldAddRef.current);
        }, 3000);
      }

      try {
        hasStoppedRecordingRef.current = true;
        isListeningRef.current = false;

        if (VoiceToText && isVoiceModuleAvailable) {
          let stopFunc = null;
          if (typeof VoiceToText.stopListening === 'function') {
            stopFunc = VoiceToText.stopListening;
          } else if (typeof VoiceToText.stop === 'function') {
            stopFunc = VoiceToText.stop;
          } else if (typeof VoiceToText.cancelListening === 'function') {
            stopFunc = VoiceToText.cancelListening;
          } else if (typeof VoiceToText.cancel === 'function') {
            stopFunc = VoiceToText.cancel;
          }

          if (stopFunc) {
            try {
              const result = stopFunc();
              if (result && typeof result.then === 'function') {
                result.catch((stopError) => console.warn('⚠️ Error stopping recording:', stopError));
              }
            } catch (stopError) {
              console.warn('⚠️ Error stopping recording:', stopError);
            }
          }
        }

        if (lottieRef.current) {
          lottieRef.current.pause();
          lottieRef.current.reset();
        }

        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }

        const delay = showProcessing ? 550 : 400;
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, delay);
      } catch (e) {
        console.warn('⚠️ Error in safeStopRecording:', e);
        if (cleanupTimeoutRef.current) {
          clearTimeout(cleanupTimeoutRef.current);
        }
        cleanupTimeoutRef.current = setTimeout(() => {
          finalizeStop(shouldAddPartialText);
        }, showProcessing ? 300 : 0);
      }
    },
    [finalizeStop, isVoiceModuleAvailable]
  );

  const startRecording = async () => {
    if (!isVoiceModuleAvailable) {
      setVoiceError('Speech engine unavailable. Please reinstall or relink the voice module.');
      return;
    }

    setVoiceError(null);

    if (Platform.OS === 'android' && !micPermissionGranted) {
      const granted = await requestMicrophonePermission();
      if (!granted) {
        setVoiceError('Microphone permission denied. Enable it in System Settings to use voice notes.');
        return;
      }
    }

    try {
      if (Platform.OS === 'ios' && !isRecognitionAvailable) {
        setVoiceError('Speech recognition not available on this iOS device.');
        return;
      }

      if (typeof VoiceToText.setRecognitionLanguage === 'function') {
        await VoiceToText.setRecognitionLanguage('en-US');
      }

      hasStoppedRecordingRef.current = false;
      hasReceivedFinalResultsRef.current = false;
      isRestartingRef.current = false;
      isListeningRef.current = true;
      setIgnoreVoiceEvents(false);
      setIsStopping(false);
      pendingStopShouldAddRef.current = false;
      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }
      setPartialText('');
      recordingStartTimeRef.current = Date.now();
      lastFinalTextRef.current = '';

    setVoiceModalVisible(true);
      setIsListening(true);

      if (lottieRef.current) {
        lottieRef.current.play();
      }

      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      } else {
        throw new Error('startListening not available on VoiceToText');
      }
    } catch (e) {
      const message = e?.message || JSON.stringify(e);
      setVoiceError(`Failed to start voice note: ${message}`);
      setIsListening(false);
      setVoiceModalVisible(false);
      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }
    }
  };

  const stopRecording = async () => {
    if (isStopping) {
      return;
    }
    safeStopRecording({ showProcessing: true });
  };

  const cancelRecording = async () => {
    if (isStopping) {
      return;
    }

    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }

    try {
      safeStopRecording({ shouldAddPartialText: false, showProcessing: true });
    } catch (e) {
      if (isMountedRef.current && !isCleaningUpRef.current) {
        setVoiceError(`Failed to cancel recording: ${e.message || JSON.stringify(e)}`);
      }
    }
  };

  const onDone = async () => {
    if (isStopping) {
      return;
    }

    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }

    try {
      safeStopRecording({ shouldAddPartialText: true, showProcessing: true });
    } catch (err) {
      console.warn('onDone stop error', err);
    }
  };

  useEffect(() => {
    ignoreVoiceEventsRef.current = ignoreVoiceEvents;
  }, [ignoreVoiceEvents]);

  useEffect(() => {
    if (!VoiceToText || typeof VoiceToText.addEventListener !== 'function') {
      const msg =
        'VoiceToText module is unavailable. Ensure @appcitor/react-native-voice-to-text is installed and linked correctly.';
      console.error(msg);
      setVoiceError(msg);
      return;
    }

    setIsVoiceModuleAvailable(true);
    isMountedRef.current = true;
    isCleaningUpRef.current = false;

    const resultsListener = VoiceToText.addEventListener(VoiceToTextEvents.RESULTS, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current) {
        onSpeechResults(event);
      }
    });
    const partialListener = VoiceToText.addEventListener(VoiceToTextEvents.PARTIAL_RESULTS, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current) {
        if (isListeningRef.current) {
          setPartialText(event?.value || '');
        }
      }
    });
    const volumeListener = VoiceToText.addEventListener(VoiceToTextEvents.VOLUME_CHANGED, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current) {
        setVolume(event?.value || 0);
      }
    });
    const errorListener = VoiceToText.addEventListener(VoiceToTextEvents.ERROR, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEventsRef.current) {
        onSpeechError(event);
      }
    });
    const startListener = VoiceToText.addEventListener(VoiceToTextEvents.START, () => {
      if (isMountedRef.current && !isCleaningUpRef.current) {
        isListeningRef.current = true;
        hasStoppedRecordingRef.current = false;
        hasReceivedFinalResultsRef.current = false;
        isRestartingRef.current = false;
        recordingStartTimeRef.current = Date.now();
        setIsListening(true);
        setVoiceModalVisible(true);
        setPartialText('');
        setIgnoreVoiceEvents(false);
        setIsStopping(false);
        if (stopFallbackTimeoutRef.current) {
          clearTimeout(stopFallbackTimeoutRef.current);
          stopFallbackTimeoutRef.current = null;
        }
        if (lottieRef.current) {
          lottieRef.current.play();
        }
      }
    });
    const endListener = VoiceToText.addEventListener(VoiceToTextEvents.END, () => {
      if (
        isMountedRef.current &&
        !isCleaningUpRef.current &&
        !hasStoppedRecordingRef.current &&
        isListeningRef.current &&
        !isRestartingRef.current
      ) {
        isRestartingRef.current = true;
        setTimeout(async () => {
          if (
            isMountedRef.current &&
            !isCleaningUpRef.current &&
            !hasStoppedRecordingRef.current &&
            isListeningRef.current
          ) {
            await restartListening();
          } else {
            isRestartingRef.current = false;
          }
        }, 200);
      }
    });

    listenersRef.current = [
      resultsListener,
      partialListener,
      volumeListener,
      errorListener,
      startListener,
      endListener,
    ];

    (async () => {
      try {
        if (typeof VoiceToText.isRecognitionAvailable === 'function') {
          const avail = await VoiceToText.isRecognitionAvailable();
          setIsRecognitionAvailable(avail === true);
          if (!avail && Platform.OS === 'ios') {
            setVoiceError(
              'Speech recognition not available on this device. Test on a real device and verify Info.plist permissions.'
            );
          }
        }
      } catch (err) {
        console.warn('isRecognitionAvailable check failed', err);
      }
    })();

    const checkPermissions = async () => {
      if (Platform.OS === 'android') {
        try {
          const granted = await checkMicrophonePermission();
          if (!granted) {
            const requested = await requestMicrophonePermission();
            if (!requested) {
              setVoiceError('Microphone permission denied. Please enable it in app settings.');
            }
          }
        } catch (err) {
          console.warn('Permission check error:', err);
          setVoiceError('Failed to check microphone permissions.');
        }
      }
    };

    checkPermissions();

    return () => {
      isMountedRef.current = false;
      isCleaningUpRef.current = true;

      if (cleanupTimeoutRef.current) {
        clearTimeout(cleanupTimeoutRef.current);
      }
      if (stopFallbackTimeoutRef.current) {
        clearTimeout(stopFallbackTimeoutRef.current);
        stopFallbackTimeoutRef.current = null;
      }

      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }

      const stopRecordingImmediately = async () => {
        try {
          if (isListeningRef.current && VoiceToText && !hasStoppedRecordingRef.current) {
            hasStoppedRecordingRef.current = true;

            const stopMethods = [
              () => VoiceToText.stopListening && VoiceToText.stopListening(),
              () => VoiceToText.stop && VoiceToText.stop(),
              () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
              () => VoiceToText.cancel && VoiceToText.cancel(),
              () => VoiceToText.destroy && VoiceToText.destroy(),
            ];

            for (const stopMethod of stopMethods) {
              try {
                if (typeof stopMethod === 'function') {
                  await stopMethod();
                  break;
                }
              } catch (e) {
                // continue
              }
            }
          }
        } catch (e) {
          console.warn('Error during cleanup (non-fatal):', e.message);
        }
      };

      stopRecordingImmediately();
      listenersRef.current = [];
    };
  }, [appendVoiceTextToAdditionalNote, restartListening, onSpeechResults, onSpeechError]);

  const shouldShowVoiceError =
    voiceError && !String(voiceError).toLowerCase().includes('unknown error: 11');

  useEffect(() => {
    const backHandler = BackHandler.addEventListener('hardwareBackPress', () => {
      if (isListeningRef.current) {
        safeStopRecording();
        return true;
      }
      return false;
    });

    return () => backHandler.remove();
  }, [safeStopRecording]);

  useEffect(() => {
    if (voiceModalVisible && isListening && lottieRef.current) {
      const timer = setTimeout(() => {
        if (lottieRef.current) {
          lottieRef.current.play();
        }
      }, 100);
      return () => clearTimeout(timer);
    } else if (!isListening && lottieRef.current) {
      lottieRef.current.pause();
      lottieRef.current.reset();
    }
  }, [voiceModalVisible, isListening]);

  const loadPersistedState = async () => {
    try {
      const persistedState = await AsyncStorage.getItem('Summerystep3State');
      if (persistedState) {
        const state = JSON.parse(persistedState);
        setEquipmentStatus(state.equipmentStatus || null);
        setPriority(state.priority || null);
        setSafeToUse(state.safeToUse || null);
        setAdditionalNote(state.additionalNote || '');
        setOverallCondition(state.overallCondition || null);
        console.log('Loaded persisted state:', state);
      }
    } catch (error) {
      console.error('Error loading persisted state:', error);
      Alert.alert('Error', 'Failed to load persisted state.');
    }
  };

  const loadFaultyItems = async () => {
    try {
      const allKeys = await AsyncStorage.getAllKeys();
      const step2Keys = allKeys.filter((k) => k.startsWith('step2-')); // Updated prefix to match Step 2
      if (step2Keys.length > 0) {
        const latestStep2Key = step2Keys.sort().reverse()[0];
        const step2Data = JSON.parse(await AsyncStorage.getItem(latestStep2Key));
        const itemStates = step2Data.itemStates || {};
        const faultyCount = Object.values(itemStates).filter(
          (state) => state.status === 'No' || state.status === 'Fail'
        ).length;
        setFaultyItems(faultyCount);
        console.log('Calculated faulty items:', faultyCount);
      } else {
        setFaultyItems(0);
        console.log('No Step 2 data found for faulty items');
      }
    } catch (error) {
      console.error('Error loading faulty items:', error);
      setFaultyItems(0);
      Alert.alert('Error', 'Failed to load faulty items.');
    }
  };

  useEffect(() => {
    loadFaultyItems();
    loadPersistedState();
  }, [signatureStatus, signatureData]);

  const statusOptions = ['Fit to travel', 'Unfit to travel'];
  const priorityOptions = [
    { label: 'Low', image: require('../../../Images/adminhome/inspreport/Priority1.png') },
    { label: 'Medium', image: require('../../../Images/adminhome/inspreport/Priority2.png') },
    { label: 'High', image: require('../../../Images/adminhome/inspreport/Priority3.png') },
    { label: 'Emergency', image: require('../../../Images/adminhome/inspreport/Priority4.png') },
  ];
  const safeOptions = ['Yes', 'No'];
  const conditionOptions = [
    'Proceed with journey',
    'Do not proceed with journey',
    'Report to Supervisor',
  ];

  const handleStatusSelect = (option) => {
    setEquipmentStatus(option);
    if (option !== 'Unfit to travel') {
      setPriority(null);
    }
    setErrors((prev) => ({ ...prev, equipmentStatus: '' }));
    saveState();
    console.log('Driver status selected:', option);
  };

  

  const handlePrioritySelect = (option) => {
    setPriority(option);
    saveState();
    console.log('Priority selected:', option);
  };

  const handleSafeSelect = (option) => {
    setSafeToUse(option);
    setErrors((prev) => ({ ...prev, safeToUse: '' }));
    saveState();
    console.log('Safe to Use selected:', option);
  };

  const handleConditionSelect = (option) => {
    setOverallCondition(option);
    setModalVisible(false);
    setErrors((prev) => ({ ...prev, overallCondition: '' }));
    saveState();
    console.log('Overall Condition selected:', option);
  };

  const handleAdditionalNoteChange = (text) => {
    setAdditionalNote(text);
    saveState({ additionalNote: text });
    console.log('Additional Note updated:', text);
  };

  const handleSignaturePress = () => {
    saveState();
    setErrors((prev) => ({ ...prev, signature: '' }));
    onSignaturePress();
    console.log('Signature pressed');
  };

  const showPriority = equipmentStatus === 'Unfit to travel';

  const validateFields = () => {
    const newErrors = {
      overallCondition: '',
      equipmentStatus: '',
      safeToUse: '',
      signature: '',
    };
    let hasError = false;

    if (!overallCondition) {
      newErrors.overallCondition = 'Overall Condition is required';
      hasError = true;
    }

    if (!equipmentStatus) {
      newErrors.equipmentStatus = 'Driver status is required';
      hasError = true;
    } 

    if (!safeToUse) {
      newErrors.safeToUse = 'Equipment Safe to Use is required';
      hasError = true;
    }

    if (!signatureStatus || !signatureData) {
      newErrors.signature = 'Inspector Signature is required';
      hasError = true;
    }

    setErrors(newErrors);
    return !hasError;
  };

  const requestStoragePermission = async () => {
    try {
      // Android 13+ uses Photo Picker which doesn't require READ_MEDIA_IMAGES
      // Only request legacy storage permissions for Android 12 and below
      const permissions = Platform.OS === 'android'
        ? Platform.Version < 33
          ? [PERMISSIONS.ANDROID.READ_EXTERNAL_STORAGE, PERMISSIONS.ANDROID.WRITE_EXTERNAL_STORAGE]
          : []
        : [];

      if (permissions.length === 0) {
        return true;
      }

      let allGranted = true;
      for (const permission of permissions) {
        const result = await check(permission);
        if (result === RESULTS.DENIED) {
          const requestResult = await request(permission);
          if (requestResult !== RESULTS.GRANTED) {
            allGranted = false;
            Alert.alert('Permission Denied', 'Storage permission is required to save the report.', [{ text: 'OK' }]);
          }
        } else if (result === RESULTS.BLOCKED) {
          allGranted = false;
          Alert.alert(
            'Permission Blocked',
            'Storage permission is blocked. Please enable it in Settings.',
            [{ text: 'OK', onPress: () => Linking.openSettings() }]
          );
        }
      }
      return allGranted;
    } catch (error) {
      console.error('Error requesting permission:', error);
      Alert.alert('Error', 'Failed to request permissions.');
      return false;
    }
  };

  // Firebase storage functions removed - using MongoDB only

  const saveReportToMongoDB = async (userUid, report, pdfDownloadUrl, signatureDownloadUrl, pdfLocalPath) => {
    try {
      console.log('📤 Saving report to MongoDB...');
      console.log('📄 Report ID:', report.id);
      console.log('👤 User UID:', userUid);
      console.log('📋 DVIR Number:', report.dvirNumber);
      console.log('📁 PDF Path:', pdfLocalPath);
      
      const reportData = {
        reportId: report.id,
        id: report.id,
        userUid: userUid,
        dvirNumber: report.dvirNumber,
        step1: report.step1,
        step2: report.step2,
        step3: {
          ...report.step3,
          // Don't set pdfDownloadUrl initially if it's null - will be set after upload
          ...(pdfDownloadUrl && { pdfDownloadUrl: pdfDownloadUrl }),
          pdfLocalPath: pdfLocalPath,
          signatureDownloadUrl: signatureDownloadUrl || null,
        },
        createdAt: report.createdAt,
        updatedAt: new Date().toISOString(),
      };
      
      console.log('📊 Report data prepared for MongoDB:', {
        reportId: reportData.reportId,
        userUid: reportData.userUid,
        dvirNumber: reportData.dvirNumber,
        hasStep1: !!reportData.step1,
        hasStep2: !!reportData.step2,
        hasStep3: !!reportData.step3,
        initialPdfUrl: reportData.step3?.pdfDownloadUrl
      });

      // First, upload the PDF file if it exists
      let uploadedPdfInfo = null;
      if (pdfLocalPath && await RNFS.exists(pdfLocalPath)) {
        try {
          console.log('📤 Uploading PDF to MongoDB server...');
          
          // Create FormData for PDF upload
          const formData = new FormData();
          formData.append('pdf', {
            uri: `file://${pdfLocalPath}`,
            type: 'application/pdf',
            name: `dvir-report-${report.id}.pdf`,
          });
          formData.append('userUid', userUid);
          formData.append('reportId', report.id);

          // Create AbortController for timeout handling
          const controller = new AbortController();
          const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout
          
          const uploadResponse = await fetch(`${MONGODB_DVIR_API_URL}/reports/upload-pdf`, {
            method: 'POST',
            body: formData,
            signal: controller.signal,
            // Don't set Content-Type header - let React Native set it automatically with boundary
          });
          
          clearTimeout(timeoutId);

          if (uploadResponse.ok) {
            uploadedPdfInfo = await uploadResponse.json();
            console.log('✅ PDF uploaded successfully:', uploadedPdfInfo);
            
            // Update the report data with the new PDF URL - Backend returns downloadUrl directly
            console.log('🔍 Upload response analysis:', {
              hasUploadedPdfInfo: !!uploadedPdfInfo,
              success: uploadedPdfInfo?.success,
              hasDownloadUrl: !!uploadedPdfInfo?.downloadUrl,
              downloadUrl: uploadedPdfInfo?.downloadUrl,
              fileName: uploadedPdfInfo?.fileName,
              filePath: uploadedPdfInfo?.filePath
            });
            
            if (uploadedPdfInfo && uploadedPdfInfo.success && uploadedPdfInfo.downloadUrl) {
              // Convert relative URL to full URL
              const fullPdfUrl = `https://api.titandrillingzm.com:6006${uploadedPdfInfo.downloadUrl}`;
              reportData.step3.pdfDownloadUrl = fullPdfUrl;
              reportData.step3.pdfLocalPath = uploadedPdfInfo.filePath;
              console.log('✅ Successfully set PDF download URL:', fullPdfUrl);
              console.log('✅ Successfully set PDF local path:', uploadedPdfInfo.filePath);
            } else if (uploadedPdfInfo && uploadedPdfInfo.fileName) {
              // Fallback: construct URL if response structure is different
              const fallbackUrl = `https://api.titandrillingzm.com:6006/reports/download-file/${userUid}/${uploadedPdfInfo.fileName}`;
              reportData.step3.pdfDownloadUrl = fallbackUrl;
              reportData.step3.pdfLocalPath = uploadedPdfInfo.filePath;
              console.log('⚠️ Using fallback URL construction:', fallbackUrl);
            } else {
              console.error('❌ Invalid upload response structure:', JSON.stringify(uploadedPdfInfo, null, 2));
            }
          } else {
            const errorText = await uploadResponse.text();
            console.warn('⚠️ PDF upload failed with status:', uploadResponse.status);
            console.warn('⚠️ PDF upload error response:', errorText);
          }
        } catch (uploadError) {
          if (uploadError.name === 'AbortError') {
            console.warn('⚠️ PDF upload timed out after 30 seconds');
          } else {
            console.warn('⚠️ PDF upload error:', uploadError.message);
          }
          // Continue without PDF upload - the report can still be saved
        }
      }

      // Final validation - ensure PDF URL is set if upload was successful
      if (uploadedPdfInfo && !reportData.step3.pdfDownloadUrl) {
        console.error('❌ PDF upload succeeded but URL not set properly!');
        console.error('❌ UploadedPdfInfo:', JSON.stringify(uploadedPdfInfo, null, 2));
        // Try to construct URL as fallback
        if (uploadedPdfInfo.fileName) {
          reportData.step3.pdfDownloadUrl = `https://api.titandrillingzm.com:6006/reports/download-file/${userUid}/${uploadedPdfInfo.fileName}`;
          console.log('🔧 Fallback URL constructed:', reportData.step3.pdfDownloadUrl);
        }
      }

      // Now save the report data to MongoDB
      console.log('🔄 Sending report data to MongoDB API...');
      console.log('📋 Final report data before saving:', {
        reportId: reportData.reportId,
        pdfDownloadUrl: reportData.step3?.pdfDownloadUrl,
        pdfLocalPath: reportData.step3?.pdfLocalPath,
        uploadedPdfInfo: uploadedPdfInfo ? 'Available' : 'None'
      });
      
      // Create AbortController for timeout handling
      const reportController = new AbortController();
      const reportTimeoutId = setTimeout(() => reportController.abort(), 30000); // 30 second timeout
      
      const response = await fetch(`${MONGODB_DVIR_API_URL}/reports`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(reportData),
        signal: reportController.signal,
      });
      
      clearTimeout(reportTimeoutId);

      console.log('📡 MongoDB API response status:', response.status);
      console.log('📡 MongoDB API response ok:', response.ok);

      if (!response.ok) {
        const errorText = await response.text();
        console.error('❌ MongoDB API error response:', errorText);
        throw new Error(`MongoDB API error: ${response.status} - ${errorText}`);
      }

      const result = await response.json();
      console.log('📊 MongoDB API result:', {
        success: result.success,
        message: result.message,
        isUpdate: result.isUpdate,
        hasData: !!result.data
      });
      
      if (result.success) {
        console.log('✅ Report saved to MongoDB successfully');
        console.log('📄 Saved report ID:', result.data?.id || result.data?._id);
        console.log('📁 PDF info:', uploadedPdfInfo ? 'Uploaded successfully' : 'No PDF uploaded');
        return { ...result.data, pdfInfo: uploadedPdfInfo };
      } else {
        console.error('❌ MongoDB save failed:', result.error);
        throw new Error(result.error || 'Failed to save to MongoDB');
      }
    } catch (error) {
      if (error.name === 'AbortError') {
        console.error('❌ MongoDB request timed out after 30 seconds');
        throw new Error('Request timed out. Please check your internet connection and try again.');
      } else {
        console.error('❌ Error saving report to MongoDB:', error);
        throw error;
      }
    }
  };

  // Send DVIR notifications (push, in-app, email)
  const sendDVIRNotifications = async (dvirData, createdByName, inspectorEmail, pdfUrl) => {
    try {
      console.log('📤 Sending DVIR notifications...');
      console.log('📋 DVIR Number:', dvirData.dvirNumber);
      console.log('👤 Inspector:', createdByName);
      console.log('📧 Inspector Email:', inspectorEmail);
      console.log('📎 PDF URL:', pdfUrl);
      
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout
      
      const response = await fetch(`${DVIR_NOTIFICATIONS_API_URL}/api/dvir-notifications/new-dvir`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          dvirData,
          createdByName,
          inspectorEmail,
          pdfUrl,
        }),
        signal: controller.signal,
      });
      
      clearTimeout(timeoutId);
      
      if (response.ok) {
        const result = await response.json();
        console.log('✅ DVIR notifications sent successfully:', result);
        console.log(`📊 Recipients: ${result.recipients?.total || 0} users notified`);
        console.log(`📱 Push: ${result.results?.push?.sent || 0} sent`);
        console.log(`📧 Emails: ${result.results?.emails?.sent || 0} sent`);
        return result;
      } else {
        const errorText = await response.text();
        console.error('❌ DVIR notifications failed:', errorText);
        return { success: false, error: errorText };
      }
    } catch (error) {
      if (error.name === 'AbortError') {
        console.error('❌ DVIR notifications request timed out');
      } else {
        console.error('❌ Error sending DVIR notifications:', error);
      }
      // Don't fail the report generation if notifications fail
      return { success: false, error: error.message };
    }
  };

  // Firebase save function removed - using MongoDB only

  const generatePDFWithHtml = async (report) => {
    try {
      console.log('Generating PDF for report:', report.id);
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};

      let checklistRows = '';
      // Use the new step2.items array
      const items = Array.isArray(step2.items) ? step2.items : [];
      let lastSection = null;
      items.forEach((item, idx) => {
        if (!item.status || item.status === '' || item.status === 'N/A') return; // Only show selected/checked
        if (item.section !== lastSection) {
          checklistRows += `
            <tr style="background-color: #e0e0e0;">
              <td colspan="7"><strong>${item.section}</strong></td>
            </tr>
          `;
          lastSection = item.section;
        }
        // Determine class and checkmarks
        const isYes = item.status.toLowerCase() === 'yes' || item.status.toLowerCase() === 'fixed' || item.status.toLowerCase() === 'pass';
        const isNo = item.status.toLowerCase() === 'no' || item.status.toLowerCase() === 'not fixed' || item.status.toLowerCase() === 'fail';
        const isNA = item.status.toLowerCase() === 'na' || item.status.toLowerCase() === 'no need';
        checklistRows += `
          <tr>
            <td>${item.label}</td>
            <td>A</td>
            <td>${isYes ? '✓' : ''}</td>
            <td>${isNo ? '✓' : ''}</td>
            <td>${isNA ? 'N/A' : (item.note || '')}</td>
          </tr>
        `;
      });
      if (!checklistRows) {
        checklistRows = `
          <tr>
            <td colspan="5">No inspection items selected</td>
          </tr>
        `;
      }

      // Create signature HTML - use base64 data directly if available, otherwise use download URL
      let signatureHtml = '';
      if (step3.signatureData && step3.signatureData.startsWith('data:image')) {
        // Use the base64 signature data directly
        signatureHtml = `
          <div class="signature-container">
            <h4>Inspector Signature:</h4>
            <img src="${step3.signatureData}" class="signature-image" alt="Signature" style="border: 1px solid #ccc; padding: 5px;" />
          </div>
        `;
      } else if (step3.signatureDownloadUrl) {
        // Use the Firebase Storage URL
        signatureHtml = `
          <div class="signature-container">
            <h4>Inspector Signature:</h4>
            <img src="${step3.signatureDownloadUrl}" class="signature-image" alt="Signature" style="border: 1px solid #ccc; padding: 5px;" />
          </div>
        `;
      } else {
        signatureHtml = `
          <div class="signature-container">
            <h4>Inspector Signature:</h4>
            <p><strong>Signature:</strong> Not Provided</p>
          </div>
        `;
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8" />
          <meta name="viewport" content="width=device-width, initial-scale=1.0" />
          <title>DRIVER’S PRE-TRIP RELEASE FORM</title>
          <style>
            body { font-family: Arial, sans-serif; padding: 20px; font-size: 12px; }
            .header, .section { border: 1px solid #000; padding: 10px; margin-bottom: 10px; }
            .header-table, .checklist-table { width: 100%; border-collapse: collapse; margin-bottom: 10px; }
            .header-table td, .checklist-table th, .checklist-table td { border: 1px solid #000; padding: 5px; text-align: left; }
            .title { text-align: center; font-weight: bold; font-size: 16px; margin: 20px 0; }
            .notes { font-size: 10px; margin-bottom: 10px; }
            .checklist-table th { background-color: #ddd; font-weight: bold; }
            .signature-container { margin-top: 30px; text-align: center; border-top: 2px solid #000; padding-top: 20px; }
            .signature-image { width: 200px; height: 100px; object-fit: contain; }
            .summary-section { margin-top: 20px; }
            .summary-section p { margin: 5px 0; }
            .signature-container h4 { margin-bottom: 10px; color: #015185; }
            .sub-label { display: block; font-size: 10px; font-weight: normal; margin-top: 2px; }
            .checklist-table th:nth-child(1), .checklist-table td:nth-child(1) { width: 42%; text-align: left; }
            .checklist-table th:nth-child(2), .checklist-table td:nth-child(2) { width: 12%; text-align: center; }
            .checklist-table th:nth-child(3), .checklist-table td:nth-child(3) { width: 18%; text-align: center; }
            .checklist-table th:nth-child(4), .checklist-table td:nth-child(4) { width: 18%; text-align: center; }
            .checklist-table th:nth-child(5), .checklist-table td:nth-child(5) { width: 20%; text-align: left; }
          </style>
        </head>
        <body>
          <div class="header">
            <table class="header-table">
              <tr>
                <td rowspan="5"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" width="100"></td>
                <td><strong>Document Name:</strong>HSE-FOR-8.1.1 Driver Pre-Trip Release</td>
              </tr>
              <tr><td><strong>Document ID:</strong>HSE-137</td></tr>
              <tr><td><strong>Version Date:</strong> 09/10/2024</td></tr>
              <tr><td><strong>Version:</strong> 1</td></tr>
              <tr><td><strong>Approved:</strong> CEO</td></tr>
            </table>
          </div>

          <div class="title">DRIVER’S PRE-TRIP RELEASE FORM</div>
          <div class="section">
            <strong>NO:</strong> ${step1.requestNumber || '0001'}<br />
            <strong>DATE:</strong> ${new Date().toLocaleDateString()}<br />
            <strong>RIG:</strong> ${step1.selectedEquipment?.equipmentName || 'N/A'}<br />
            <strong>ODOMETER READING:</strong> ${step1.meterReading || 'N/A'} ${step1.selectedEquipment?.meterUnit || 'Miles'}<br />
            <strong>FUEL LEVEL:</strong> N/A
          </div>

          <div class="section notes">
            1. ALL DEFECTS UNDER "A" CLASSIFICATION TO BE ATTENDED IMMEDIATELY<br />
            2. ALL DEFECTS UNDER "B" CLASSIFICATION TO BE ATTENDED WITHIN 48 HOURS<br />
            3. ALL DEFECTS UNDER "C" CLASSIFICATION TO BE ATTENDED AS SOON AS POSSIBLE
          </div>

          <table class="checklist-table">
            <tr>
              <th>Description</th>
              <th>Class</th>
              <th>In Order (✓)<span class="sub-label">Yes</span></th>
              <th>Defective (✓)<span class="sub-label">No</span></th>
              <th>Comment</th>
            </tr>
            ${checklistRows}
          </table>

          <div class="summary-section">
            <h3>Summary</h3>
            <p><strong>Country:</strong> ${step1.country || 'N/A'}</p>
            <p><strong>Project:</strong> ${step1.project || 'N/A'}</p>
            <p><strong>Inspector:</strong> ${step1.inspector || 'N/A'}</p>
            <p><strong>Faulty Items:</strong> ${faultyItems || '0'}</p>
            <p><strong>Overall Condition:</strong> ${step3.overallCondition || 'N/A'}</p>
            <p><strong>Driver status:</strong> ${step3.equipmentStatus || 'N/A'}</p>
            ${step3.priority ? `<p><strong>Priority:</strong> ${step3.priority}</p>` : ''}
            <p><strong>Safe to Use:</strong> ${step3.safeToUse || 'N/A'}</p>
            <p><strong>Additional Note:</strong> ${step3.additionalNote || 'N/A'}</p>
          </div>

          ${signatureHtml}
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `Daily_DD_Rig_Prestart_Checklist_${report.id}`,
        directory: Platform.OS === 'ios' ? 'Documents' : 'Download',
        width: 595,
        height: 842,
        padding: 24,
        base64: false,
        bgColor: '#FFFFFF',
      };

      const file = await RNHTMLtoPDF.convert(options);
      console.log('PDF generated at:', file.filePath);
      
      // Verify file exists on Android
      if (Platform.OS === 'android') {
        const fileExists = await RNFS.exists(file.filePath);
        if (!fileExists) {
          throw new Error('PDF file was not created successfully on Android');
        }
      }
      
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      if (Platform.OS === 'android') {
        // Try alternative directory for Android
        try {
          const androidOptions = {
            ...options,
            directory: 'Documents',
          };
          const file = await RNHTMLtoPDF.convert(androidOptions);
          console.log('PDF generated at (Android fallback):', file.filePath);
          return file.filePath;
        } catch (fallbackError) {
          console.error('Android fallback PDF generation failed:', fallbackError);
          throw new Error(`PDF generation failed on Android: ${fallbackError.message}`);
        }
      }
      throw error;
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('LoginScreens');
        return;
      }

      let pdfUrl = null;
      let reportData = null;

      // Try MongoDB first
      try {
        console.log('🔍 Fetching report from MongoDB...');
        const response = await fetch(`${MONGODB_DVIR_API_URL}/reports/${reportId}`);
        
        if (response.ok) {
          const result = await response.json();
          if (result.success && result.data) {
            reportData = result.data;
            pdfUrl = reportData.step3?.pdfDownloadUrl;
            console.log('✅ Report fetched from MongoDB');
            console.log('🔍 Retrieved PDF URL:', pdfUrl);
            console.log('🔍 Full step3 data:', reportData.step3);
          } else {
            console.warn('❌ MongoDB response not successful:', result);
          }
        } else {
          console.warn('❌ MongoDB fetch failed with status:', response.status);
        }
      } catch (mongoError) {
        console.error('❌ MongoDB fetch failed:', mongoError.message);
      }

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      if (!pdfUrl) {
        console.error('❌ No PDF URL found in report data');
        console.error('❌ Report step3 data:', reportData.step3);
        Alert.alert(
          'Error', 
          'PDF URL not available in the report. The PDF may not have been uploaded successfully. Please try regenerating the report.',
          [{ text: 'OK' }]
        );
        return;
      }

      console.log('🚀 Navigating to PdfviewPage with URL:', pdfUrl);
      console.log('🚀 URL validation - starts with http:', pdfUrl.startsWith('http'));
      console.log('🚀 URL validation - contains download:', pdfUrl.includes('download'));

      // Navigate to PdfviewPage
      navigation.replace('PdfviewPage', {
        pdfUrl,
        onGoBack: () => navigation.replace('DviranddotInHome'),
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const handleGenerateReport = async () => {
    if (!validateFields()) {
      return;
    }

    const hasPermission = await requestStoragePermission();
    if (!hasPermission) return;

    setSyncModalVisible(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('LoginScreens');
        setSyncModalVisible(false);
        return;
      }

      // Get latest Step 1 and Step 2 data
      const allKeys = await AsyncStorage.getAllKeys();
      const step1Keys = allKeys.filter((k) => k.startsWith('step1-'));
      const step2Keys = allKeys.filter((k) => k.startsWith('step2-'));
      if (!step1Keys.length || !step2Keys.length) {
        Alert.alert('Error', 'Data from previous steps is missing. Please complete all steps.');
        setSyncModalVisible(false);
        return;
      }
      const lastStep1Key = step1Keys.sort().reverse()[0];
      const lastStep2Key = step2Keys.sort().reverse()[0];
      const step1 = JSON.parse(await AsyncStorage.getItem(lastStep1Key)) || {};
      const step2Raw = JSON.parse(await AsyncStorage.getItem(lastStep2Key)) || {};

      console.log('Step 1 Data Retrieved:', step1);
      console.log('Step 2 Data Retrieved:', step2Raw);

      if (!step1.country || !step1.project || !step1.inspector || !step2Raw.itemStates) {
        Alert.alert('Error', 'Incomplete data from previous steps. Please complete all steps.');
        setSyncModalVisible(false);
        return;
      }

      // --- Build detailed Step 2 items array ---
      const itemStates = step2Raw.itemStates || {};
      const checklistId = step2Raw.checklistId || null;
      const timestamp = step2Raw.timestamp || null;
      const checklists = JSON.parse(await AsyncStorage.getItem('checklists') || '[]');
      const customChecklist = checklists.find((c) => c.id === checklistId);
      const step2Items = [];

      // Helper to push item to step2Items
      const pushItem = (section, itemId, label, status, note) => {
        step2Items.push({ section, itemId, label, status, note });
      };

      // Requirements
      DvirPChecklist.requirements.items.forEach((item) => {
        const key = `requirements_${item.id}_Status`;
        const state = itemStates[key] || {};
        pushItem(DvirPChecklist.requirements.label, item.id, item.label, state.status || '', state.note || '');
      });
      // Hazards
      DvirPChecklist.hazards.items.forEach((item) => {
        const key = `hazards_${item.id}_Status`;
        const state = itemStates[key] || {};
        pushItem(DvirPChecklist.hazards.label, item.id, item.label, state.status || '', state.note || '');
      });
      // Custom checklist sections (if any)
      if (customChecklist && Array.isArray(customChecklist.items)) {
        customChecklist.items.forEach((section) => {
          const sectionLabel = section.text || section.label || section.id || 'Custom Section';
          if (Array.isArray(section.subItems)) {
            section.subItems.forEach((item) => {
              const key = `${section.key || section.id}_${item.id}_${item.itemName}`;
              const state = itemStates[key] || {};
              pushItem(sectionLabel, item.id, item.itemName || 'Unknown', state.status || '', state.note || '');
            });
          }
        });
      }
      // --- End build detailed Step 2 items array ---

      // --- Generate DVIR number and report ID ---
      const reportId = Date.now().toString(); // Generate a unique report ID
      setReportId(reportId);
      
      // Generate DVIR number using MongoDB count API
      let dvirNumber = `DVIR-001`; // Default fallback
      try {
        console.log('🔢 Generating DVIR number from MongoDB count...');
        const dvirCountResponse = await fetch(`${MONGODB_DVIR_API_URL}/reports/count`);
        if (dvirCountResponse.ok) {
          const dvirCountResult = await dvirCountResponse.json();
          const newDvirNumber = (dvirCountResult.count || 0) + 1;
          dvirNumber = `DVIR-${String(newDvirNumber).padStart(3, '0')}`;
          console.log(`✅ Generated DVIR number: ${dvirNumber}`);
        } else {
          console.warn('⚠️ Could not get DVIR count, using fallback');
        }
      } catch (countError) {
        console.warn('⚠️ Error getting DVIR count:', countError.message, '- using fallback');
      }

      // Save Step 3 data
      const step3Data = {
        faultyItems,
        overallCondition,
        equipmentStatus,
        priority,
        safeToUse,
        additionalNote,
        signatureStatus,
        signatureData,
        createdAt: new Date().toISOString(),
        timestamp: new Date().getTime(),
        dvirNumber, // Store the generated DVIR number in step3
      };
      
      console.log('Step 3 data created:', step3Data);

      // Skip Firebase signature upload to avoid storage errors
      // Signature will be stored directly in MongoDB with the report
      let signatureDownloadUrl = null;

      // Combine all data
      const step2 = {
        items: step2Items,
        checklistId,
        timestamp,
      };
      const report = {
        id: reportId,
        dvirNumber, // Store the generated DVIR number at the top level
        step1,
        step2,
        step3: step3Data,
        createdAt: new Date().toISOString(),
      };
      
      console.log('Final report structure before saving:', {
        id: report.id,
        dvirNumber: report.dvirNumber,
        hasStep1: !!report.step1,
        hasStep2: !!report.step2,
        hasStep3: !!report.step3,
        step3Fields: report.step3 ? Object.keys(report.step3) : 'No step3',
        createdAt: report.createdAt
      });

      // Generate PDF
      const pdfPath = await generatePDFWithHtml(report);
      
      // Save to MongoDB only (removed Firebase fallback to avoid storage errors)
      let saveSuccess = false;
      let savedReportData = null;
      
      try {
        console.log('🔄 Attempting to save to MongoDB...');
        console.log('📄 PDF Path:', pdfPath);
        console.log('📄 PDF exists:', await RNFS.exists(pdfPath));
        
        // Verify PDF exists before attempting upload
        if (!await RNFS.exists(pdfPath)) {
          throw new Error('PDF file not found at path: ' + pdfPath);
        }
        
        // Save to MongoDB with PDF upload
        savedReportData = await saveReportToMongoDB(userUid, report, null, signatureDownloadUrl, pdfPath);
        console.log('✅ Report saved to MongoDB successfully');
        
        // Add a small delay to ensure file processing is complete
        if (savedReportData && savedReportData.pdfInfo) {
          console.log('⏳ Waiting for file processing to complete...');
          await new Promise(resolve => setTimeout(resolve, 2000)); // 2 second delay
        }
        
        saveSuccess = true;
        
        // Send notifications after successful save (fire-and-forget)
        if (savedReportData && savedReportData.step3?.pdfDownloadUrl) {
          setTimeout(async () => {
            try {
              // Get inspector name and email
              const inspectorName = step1.inspector || 'Inspector';
              
              // Get current user's email for confirmation
              let inspectorEmail = '';
              try {
                const userData = await AsyncStorage.getItem('userData');
                if (userData) {
                  const user = JSON.parse(userData);
                  inspectorEmail = user.email || '';
                }
              } catch (emailError) {
                console.warn('⚠️ Could not get inspector email:', emailError);
              }
              
              // Prepare DVIR data for notifications
              const dvirNotificationData = {
                _id: savedReportData._id || report.id,
                id: savedReportData._id || report.id,
                dvirNumber: report.dvirNumber,
                requestNumber: step1.requestNumber,
                equipmentName: step1.selectedEquipment?.equipmentName || 'N/A',
                country: step1.country,
                project: step1.project,
                faultyItems: faultyItems,
              };
              
              console.log('🔔 Sending DVIR notifications in background...');
              const notifResult = await sendDVIRNotifications(
                dvirNotificationData,
                inspectorName,
                inspectorEmail,
                savedReportData.step3.pdfDownloadUrl
              );
              
              if (notifResult.success) {
                console.log(`✅ DVIR notifications sent to ${notifResult.recipients?.total || 0} users`);
              } else {
                console.warn('⚠️ DVIR notifications failed:', notifResult.error);
              }
            } catch (notifError) {
              console.error('❌ Background notification error:', notifError);
            }
          }, 500); // 500ms delay for background execution
        }
      } catch (mongoError) {
        console.error('❌ MongoDB save failed:', mongoError.message);
        
        // Show specific error message instead of Firebase fallback
        Alert.alert(
          'Save Error', 
          `Failed to save report to database: ${mongoError.message}. The PDF has been generated and is available for sharing.`,
          [{ text: 'OK' }]
        );
        
        // Still mark as success for PDF sharing purposes
        saveSuccess = true; // Allow the user to still share the PDF
      }

      // Share PDF
      try {
        await Share.share({
          url: `file://${pdfPath}`,
          title: `Inspection Report ${dvirNumber}`,
          message: `Inspection Report ${dvirNumber} generated on ${new Date().toLocaleDateString()}`,
        });
        console.log('PDF shared successfully');
      } catch (shareError) {
        console.error('Error sharing PDF:', shareError);
        // PDF is still generated, just sharing failed
        Alert.alert(
          'Info', 
          'Report generated successfully. You can find it in your device storage.',
          [{ text: 'OK' }]
        );
      }

      // Clean up temporary file only if MongoDB upload was successful and PDF was uploaded
      if (savedReportData && savedReportData.pdfInfo) {
        console.log('🧹 MongoDB upload successful, cleaning up temporary PDF...');
        await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));
      } else {
        console.log('⚠️ Keeping PDF file - upload may have failed or PDF not uploaded');
        // Keep the PDF file since it's needed for sharing and the upload may have failed
      }

      // Clear temporary data
      await AsyncStorage.removeItem('Summerystep3State');
      await AsyncStorage.removeItem('selectedChecklistIds');
      // Remove all step1-* and step2-* keys
      const allKeysCleanup = await AsyncStorage.getAllKeys();
      const step1KeysCleanup = allKeysCleanup.filter((k) => k.startsWith('step1-'));
      const step2KeysCleanup = allKeysCleanup.filter((k) => k.startsWith('step2-'));
      await AsyncStorage.multiRemove([...step1KeysCleanup, ...step2KeysCleanup]);
      console.log('Cleared all step1 and step2 AsyncStorage data');

      setSavedPdfPath(pdfPath);
      setShowSuccessModal(true);
    } catch (error) {
      console.error('Error generating report:', error);
      Alert.alert('Error', `Failed to generate report: ${error.message}`);
    } finally {
      setSyncModalVisible(false);
    }
  };

  return (
    <KeyboardAvoidingView
      style={styles.container}
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <Modal
        animationType="fade"
        transparent={true}
        visible={modalVisible}
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>
              Overall Condition <Text style={styles.requiredAsterisk}>*</Text>
            </Text>
            {conditionOptions.map((option) => (
              <View key={option}>
                <TouchableOpacity
                  style={styles.modalRow}
                  onPress={() => handleConditionSelect(option)}
                >
                  <Text style={styles.modalOptionText}>{option}</Text>
                  <View
                    style={[
                      styles.radioCircle,
                      overallCondition === option && styles.radioCircleSelected,
                    ]}
                  >
                    {overallCondition === option && <View style={styles.radioDot} />}
                  </View>
                </TouchableOpacity>
                <View style={styles.modalDivider} />
              </View>
            ))}
          </View>
        </View>
      </Modal>

      <Modal
        animationType="fade"
        transparent={true}
        visible={showSuccessModal}
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.popupModalContent}>
            <Text style={styles.popupTitle}>Success</Text>
            <MaterialIcons name="check" size={44} color="#015185" style={{ alignSelf: 'center', marginVertical: 8 }} />
            <Text style={styles.popupSubtitle}>Inspection Report Generated and Saved</Text>
            <View style={{ flexDirection: 'row', justifyContent: 'center', marginTop: 18 }}>
              <TouchableOpacity
                style={styles.popupButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.reset({
                    index: 1,
                    routes: [
                      { name: 'MainApp' },
                      { name: 'DviranddotInHome' },
                    ],
                  });
                }}
              >
                <Text style={styles.popupButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.popupButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.popupButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <ScrollView
        style={styles.mainContent}
        contentContainerStyle={styles.scrollContent}
        keyboardShouldPersistTaps="handled"
      >
        <Modal
          visible={voiceModalVisible}
          transparent
          animationType="fade"
          onRequestClose={() => {
            if (isListening) {
              cancelRecording();
            }
          }}
        >
          <View style={styles.modalOverlay}>
            <View style={styles.modalContent}>
              {isStopping ? (
                <View style={styles.processingContainer}>
                  <ActivityIndicator size="large" color="#015185" />
                  <Text style={styles.processingText}>Finishing up...</Text>
                </View>
              ) : (
                <>
                  <View style={styles.lottieContainer}>
                    <LottieView
                      ref={lottieRef}
                      source={require('../../../Images/micanimation.json')}
                      autoPlay={false}
                      loop={true}
                      style={styles.lottieAnimation}
                      resizeMode="contain"
                    />
                  </View>

                  <Text style={styles.partial}>
                    {partialText
                      ? partialText
                      : isListening
                      ? 'Listening... Speak now'
                      : 'Preparing...'}
                  </Text>

                  {volume > 0 && (
                    <View style={styles.volumeIndicator}>
                      <View style={[styles.volumeBar, { width: `${Math.min(volume * 100, 100)}%` }]} />
                    </View>
                  )}

                  <View style={styles.modalButtons}>
                    <TouchableOpacity
                      style={[styles.modalButton, styles.cancelBtn]}
                      onPress={cancelRecording}
                      activeOpacity={0.7}
                      disabled={isStopping}
                    >
                      <Text style={[styles.modalButtonText, { color: '#015185' }]}>Cancel</Text>
                    </TouchableOpacity>
                    <TouchableOpacity
                      style={styles.modalButton}
                      onPress={onDone}
                      activeOpacity={0.7}
                      disabled={isStopping}
                    >
                      <Text style={styles.modalButtonText}>Done</Text>
                    </TouchableOpacity>
                  </View>
                </>
              )}
            </View>
          </View>
        </Modal>

        <View style={styles.rowContainer}>
          <Text style={styles.labelTextFault}>Faulty Items</Text>
          <Text style={styles.valueTextFault}>{faultyItems}</Text>
        </View>

        <View style={styles.rowContainer}>
          <Text style={styles.labelTextFault}>
            Overall Condition <Text style={styles.requiredAsterisk}>*</Text>
          </Text>
          <View style={styles.buttonContainer}>
            <TouchableOpacity style={styles.selectButton} onPress={() => setModalVisible(true)}>
              <Text style={styles.buttonText}>{overallCondition || 'Select'}</Text>
              <MaterialIcons name="chevron-right" size={20} color="#fff" style={styles.iconRight} />
            </TouchableOpacity>
          </View>
        </View>
        {errors.overallCondition ? (
          <Text style={styles.errorText}>{errors.overallCondition}</Text>
        ) : null}

        <Text style={styles.sectionLabelequip}>
          Driver status <Text style={styles.requiredAsterisk}>*</Text>
        </Text>
        <View style={styles.optionsRow}>
          {statusOptions.map((option) => {
            const isSelected = equipmentStatus === option;
            return (
              <TouchableOpacity
                key={option}
                style={[styles.statusOptionButton, isSelected && styles.statusOptionButtonSelected]}
                onPress={() => handleStatusSelect(option)}
              >
                <Text style={isSelected ? styles.optionTextSelected : styles.optionText}>{option}</Text>
              </TouchableOpacity>
            );
          })}
        </View>
        {errors.equipmentStatus ? (
          <Text style={styles.errorText}>{errors.equipmentStatus}</Text>
        ) : null}

        {showPriority && (
          <>
            <Text style={styles.sectionLabelequip}>Priority</Text>
            <View style={styles.priorityContainer}>
              {priorityOptions.map((option) => {
                const isSelected = priority === option.label;
                return (
                  <View key={option.label} style={styles.priorityItem}>
                    <TouchableOpacity
                      style={[
                        styles.priorityButton,
                        isSelected ? styles.priorityButtonSelected : styles.priorityButtonUnselected,
                      ]}
                      onPress={() => handlePrioritySelect(option.label)}
                    >
                      <Image source={option.image} style={styles.priorityImage} />
                      <Text style={isSelected ? styles.priorityTextSelected : styles.priorityText}>
                        {option.label}
                      </Text>
                    </TouchableOpacity>
                  </View>
                );
              })}
            </View>
          </>
        )}

        <Text style={styles.sectionLabelequip}>
          Equipment Safe to Use <Text style={styles.requiredAsterisk}>*</Text>
        </Text>
        <View style={styles.safeOptionsRow}>
          {safeOptions.map((option) => {
            const isSelected = safeToUse === option;
            return (
              <TouchableOpacity
                key={option}
                style={[styles.optionButton, isSelected && styles.optionButtonSelected, { marginRight: 20 }]}
                onPress={() => handleSafeSelect(option)}
              >
                <Text style={isSelected ? styles.optionTextSelected : styles.optionText}>{option}</Text>
              </TouchableOpacity>
            );
          })}
        </View>
        {errors.safeToUse ? (
          <Text style={styles.errorText}>{errors.safeToUse}</Text>
        ) : null}

        <Text style={styles.voiceStatus}>Status: {isListening ? 'Listening...' : 'Stopped'}</Text>
        {shouldShowVoiceError ? (
          <Text style={styles.voiceErrorText}>Error: {voiceError}</Text>
        ) : null}
        <View style={styles.inputContainer}>
          <TextInput
            style={styles.input}
            placeholder="Additional Note"
            value={additionalNote}
            onChangeText={handleAdditionalNoteChange}
            multiline={true}
            numberOfLines={4}
            textAlignVertical="top"
          />
          <TouchableOpacity
            style={[
              styles.inputIconContainer,
              isListening && styles.inputIconContainerActive,
              (!isListening && !isVoiceModuleAvailable) && styles.inputIconContainerDisabled,
            ]}
            onPress={isListening ? stopRecording : startRecording}
            disabled={isStopping || (!isListening && !isVoiceModuleAvailable)}
            activeOpacity={0.8}
          >
            <MaterialIcons name={isListening ? 'mic-off' : 'mic'} size={24} color="#fff" />
          </TouchableOpacity>
        </View>

        <Text style={styles.signatureLabel}>
          Inspector Signature <Text style={styles.requiredAsterisk}>*</Text>
        </Text>
        {signatureStatus && signatureData ? (
          <View style={styles.signatureContainer}>
            <Text style={styles.signatureStatus}>Signature Done</Text>
            <Image
              source={{ uri: signatureData }}
              style={styles.signatureImage}
              resizeMode="contain"
              onError={(e) => console.log('Image loading error:', e.nativeEvent.error)}
            />
            <TouchableOpacity
              style={styles.uploadAgainButton}
              onPress={handleSignaturePress}
            >
              <Text style={styles.uploadAgainText}>Upload again</Text>
            </TouchableOpacity>
          </View>
        ) : (
          <TouchableOpacity onPress={handleSignaturePress}>
            <View style={styles.signatureBox}>
              <MaterialIcons name="add" size={40} color="#0078D4" />
            </View>
          </TouchableOpacity>
        )}
        {errors.signature ? (
          <Text style={styles.errorText}>{errors.signature}</Text>
        ) : null}

        <View style={styles.acknowledgmentContainer}>
          <View style={styles.acknowledgmentHeader}>
            <MaterialIcons name="check" size={20} color="#015185" />
            <Text style={styles.acknowledgmentTitle}>Operator's Acknowledgment</Text>
          </View>
          <Text style={styles.acknowledgmentText}>
            I hereby certify that the inspection has been done accurately and completely according to compliance with all applicable state and Federal Motor Carrier Safety Regulations.
          </Text>
        </View>
      </ScrollView>

      <View style={styles.navButtonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(2)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.navButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleGenerateReport}>
          <Text style={styles.navButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingVertical: 10,
    paddingHorizontal: '5%',
    paddingBottom: 120,
  },
  rowContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    width: '100%',
    padding: 10,
    borderRadius: 5,
    marginBottom: 10,
  },
  labelTextFault: {
    fontSize: 16,
    fontWeight: '500',
    color: '#015185',
  },
  valueTextFault: {
    fontSize: 16,
    fontWeight: '600',
    color: '#FF0000',
  },
  buttonContainer: {
    alignItems: 'center',
  },
  selectButton: {
    backgroundColor: '#015185',
    paddingVertical: 4,
    paddingHorizontal: 10,
    borderRadius: 5,
    justifyContent: 'center',
    flexDirection: 'row',
    alignItems: 'center',
  },
  buttonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  iconRight: {
    marginLeft: 5,
  },
  sectionLabelequip: {
    fontSize: 16,
    fontWeight: '500',
    color: '#015185',
    marginVertical: 10,
    marginLeft: '0%',
  },
  optionsRow: {
    flexDirection: 'row',
    marginBottom: 8,
  },
  safeOptionsRow: {
    flexDirection: 'row',
    justifyContent: 'flex-start',
    marginBottom: 15,
  },
  statusOptionButton: {
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#ECF0F3',
    paddingHorizontal: 10,
    paddingVertical: 9,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  statusOptionButtonSelected: {
    backgroundColor: '#015185',
    paddingHorizontal: 10,
    paddingVertical: 9,
    borderRadius: 5,
    marginHorizontal: 5,
  },
  optionButton: {
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    alignItems: 'center',
    justifyContent: 'center',
    minWidth: 80,
    backgroundColor: '#ECF0F3',
  },
  optionButtonSelected: {
    paddingHorizontal: 15,
    paddingVertical: 8,
    borderRadius: 5,
    alignItems: 'center',
    justifyContent: 'center',
    minWidth: 80,
    backgroundColor: '#015185',
  },
  optionText: {
    fontSize: 13,
    fontWeight: '600',
  },
  optionTextSelected: {
    fontSize: 13,
    color: '#fff',
    fontWeight: '600',
  },
  priorityContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginBottom: 15,
    backgroundColor: '#fff',
    borderRadius: 5,
    width: '100%',
  },
  priorityItem: {
    alignItems: 'center',
  },
  priorityButton: {
    padding: 8,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    width: 85,
    height: 60,
    justifyContent: 'center',
    alignItems: 'center',
  },
  priorityButtonSelected: {
    backgroundColor: '#015185',
  },
  priorityButtonUnselected: {
    backgroundColor: '#FFFFFF',
  },
  priorityImage: {
    width: 24,
    height: 24,
  },
  priorityText: {
    fontSize: 10,
    color: '#333',
    marginTop: 3,
    textAlign: 'center',
  },
  priorityTextSelected: {
    fontSize: 10,
    color: '#fff',
    fontWeight: '600',
    marginTop: 3,
    textAlign: 'center',
  },
  inputContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#ECF0F3',
    borderRadius: 5,
    paddingHorizontal: 10,
    marginBottom: 20,
    width: '100%',
  },
  input: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    paddingVertical: 10,
  },
  inputIconContainer: {
    backgroundColor: '#015185',
    padding: 5,
    left: 8,
  },
  signatureLabel: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    marginVertical: 10,
  },
  signatureContainer: {
    alignItems: 'center',
  },
  signatureStatus: {
    fontSize: 16,
    color: '#015185',
    textAlign: 'center',
    marginVertical: 5,
  },
  signatureImage: {
    width: width * 0.6,
    height: height * 0.15,
    alignSelf: 'center',
    marginTop: 10,
    borderRadius: 10,
    backgroundColor: '#F5F5F5',
  },
  uploadAgainButton: {
    backgroundColor: '#d3d3d3',
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 10,
    alignSelf: 'center',
  },
  uploadAgainText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  signatureBox: {
    height: height / 3.5,
    backgroundColor: '#F5F5F5',
    borderRadius: 10,
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    marginTop: 10,
  },
  acknowledgmentContainer: {
    backgroundColor: '#fff',
    marginTop: 20,
    marginBottom: 20,
    padding: 16,
    borderRadius: 8,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.05,
    shadowRadius: 4,
    elevation: 2,
  },
  acknowledgmentHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  acknowledgmentTitle: {
    fontWeight: 'bold',
    color: '#015185',
    fontSize: 17,
    marginLeft: 6,
  },
  acknowledgmentText: {
    fontSize: 13,
    color: '#222',
    marginLeft: 2,
  },
  navButtonContainer: {
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 15,
    paddingVertical: 10,
    backgroundColor: '#f0f0f0',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    paddingHorizontal: 15,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 30,
    borderRadius: 5,
  },
  navButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 10,
  },
  modalOptionText: {
    fontSize: 16,
    color: '#333',
  },
  modalDivider: {
    height: 1,
    backgroundColor: '#d3d3d3',
  },
  requiredAsterisk: {
    color: '#FF0000',
  },
  errorText: {
    color: '#FF0000',
    fontSize: 12,
    marginBottom: 10,
    marginLeft: 5,
  },
  radioCircle: {
    width: 24,
    height: 24,
    borderRadius: 12,
    borderWidth: 2,
    borderColor: '#015185',
    justifyContent: 'center',
    alignItems: 'center',
  },
  radioCircleSelected: {
    borderColor: '#015185',
  },
  radioDot: {
    width: 12,
    height: 12,
    borderRadius: 6,
    backgroundColor: '#015185',
  },
  popupModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
  },
  popupTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  popupSubtitle: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  popupButton: {
    backgroundColor: '#0078D4',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  popupButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  voiceErrorText: {
    color: '#d32f2f',
    marginTop: -10,
    marginBottom: 12,
    fontSize: 12,
  },
  voiceStatus: {
    marginBottom: 6,
    marginLeft: 2,
    fontSize: 14,
    color: '#666',
  },
  inputIconContainerActive: {
    backgroundColor: '#FF4B2B',
  },
  inputIconContainerDisabled: {
    opacity: 0.5,
  },
  processingContainer: {
    width: '100%',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 24,
  },
  processingText: {
    marginTop: 20,
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    textAlign: 'center',
  },
  lottieContainer: {
    width: '100%',
    alignItems: 'center',
    justifyContent: 'center',
  },
  lottieAnimation: {
    width: 140,
    height: 140,
  },
  partial: {
    fontSize: 16,
    color: '#015185',
    textAlign: 'center',
    marginVertical: 12,
    paddingHorizontal: 10,
  },
  volumeIndicator: {
    width: '80%',
    height: 8,
    backgroundColor: '#e0e0e0',
    borderRadius: 4,
    overflow: 'hidden',
    alignSelf: 'center',
    marginBottom: 16,
  },
  volumeBar: {
    height: '100%',
    backgroundColor: '#015185',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: 10,
  },
  modalButton: {
    flex: 1,
    backgroundColor: '#015185',
    paddingVertical: 10,
    marginHorizontal: 5,
    borderRadius: 6,
    alignItems: 'center',
  },
  cancelBtn: {
    backgroundColor: '#F2F6FF',
    borderWidth: 1,
    borderColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  bottomicon: {
    marginTop: 2,
  },
  listeningModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  listeningModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  listeningModalText: {
    fontSize: 18,
    color: '#333',
    marginBottom: 15,
  },
  stopButton: {
    backgroundColor: '#FF0000',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  stopButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
});

export default DvirSummeryStep3;