import React, { useState, useEffect, useRef } from 'react';
import { View, Text, StyleSheet, StatusBar, TouchableOpacity, Dimensions, Alert, Animated, Modal         } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useIsFocused } from '@react-navigation/native';
import NetInfo from '@react-native-community/netinfo';
// Firebase import removed - using MongoDB only
const MONGODB_DVIR_API_URL = 'https://api.titandrillingzm.com:6006';
import DvirCorrectiveActionStep1 from './DvirCorrectiveActionStep1';
import DvirCorrectiveActionStep2 from './DvirCorrectiveActionStep2';


const { height } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = height * 0.2 - 36;

const DvirCorrectiveActionHeader = ({ route, navigation }) => {
  const { report: initialReport } = route.params;
  const isFocused = useIsFocused();
  const [activeStep, setActiveStep] = useState(1);
  const [subItemStatuses, setSubItemStatuses] = useState({});
  const [signatureScreen, setSignatureScreen] = useState(null);
  const [mechanicSignatureUri, setMechanicSignatureUri] = useState(null);
  const [operatorSignatureUri, setOperatorSignatureUri] = useState(null);
  const [hasUnsavedSignature, setHasUnsavedSignature] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [report, setReport] = useState(initialReport); // <-- store the latest report

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });   

  // Fetch latest report from database when focused
  useEffect(() => {
    if (isFocused && initialReport?.userUid && initialReport?.id) {
      const fetchLatestReport = async () => {
        setSyncModalVisible(true);
        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 1000,
            useNativeDriver: true,
          })
        ).start();
        try {
          const state = await NetInfo.fetch();
          if (!state.isConnected) {
            setSyncModalVisible(false);
            rotateAnim.setValue(0);
            return;
          }
          // Fetch latest report from MongoDB instead of Firebase
          try {
            const response = await fetch(`${MONGODB_DVIR_API_URL}/reports/${initialReport.id}`);
            if (response.ok) {
              const result = await response.json();
              if (result.success && result.data) {
                setReport({ ...result.data, id: initialReport.id, userUid: initialReport.userUid });
              }
            }
          } catch (fetchError) {
            console.log('Error fetching latest report from MongoDB:', fetchError);
          }
        } catch (error) {
          console.error('Error fetching latest report:', error);
        } finally {
          setTimeout(() => {
            setSyncModalVisible(false);
            rotateAnim.setValue(0);
          }, 1000);
        }
      };
      fetchLatestReport();
    }
  }, [isFocused, initialReport]);

  // Initialize subItemStatuses from report.correctiveAction or load from Firebase
  useEffect(() => {
    const loadSubItemStatuses = async () => {
      try {
        // First try to get from report.correctiveAction
        if (report?.correctiveAction?.subItemStatuses) {
          setSubItemStatuses(report.correctiveAction.subItemStatuses);
          return;
        }
        
        // If not in report, try to load from MongoDB corrective action
        if (report?.id && report?.userUid) {
          const userUid = await AsyncStorage.getItem('userUid');
          if (userUid) {
            try {
              const response = await fetch(`${MONGODB_DVIR_API_URL}/dvir-corrective-actions/report/${report.id}?userUid=${userUid}`);
              if (response.ok) {
                const result = await response.json();
                if (result.success && result.data?.subItemStatuses) {
                  console.log('Loading subItemStatuses from MongoDB:', result.data.subItemStatuses);
                  setSubItemStatuses(result.data.subItemStatuses);
                  return;
                }
              }
            } catch (fetchError) {
              console.log('Error fetching corrective action from MongoDB:', fetchError);
            }
          }
        }
        
        // If nothing found, keep existing state or set empty
        console.log('No subItemStatuses found, keeping existing state');
      } catch (error) {
        console.error('Error loading subItemStatuses:', error);
      }
    };
    
    loadSubItemStatuses();
  }, [report?.id, report?.userUid]);

  // Reset subItemStatuses when navigating back
  const resetSubItemStatuses = () => {
    setSubItemStatuses({});
  };

  const steps = [
    {
      id: 1,
      title: 'Step 1',
      subtitle: 'Corrective Action',
      component: (
        <DvirCorrectiveActionStep1
          report={report}
          setActiveStep={setActiveStep}
          navigation={navigation}
          subItemStatuses={subItemStatuses}
          setSubItemStatuses={setSubItemStatuses}
          resetSubItemStatuses={resetSubItemStatuses}
        />
      ),
    },
    {
      id: 2,
      title: 'Step 2',
      subtitle: 'Signature',
      component: (
        <DvirCorrectiveActionStep2
          report={report}
          setActiveStep={setActiveStep}
          navigation={navigation}
          subItemStatuses={subItemStatuses}
          setSubItemStatuses={setSubItemStatuses}
          setSignatureScreen={setSignatureScreen}
          mechanicSignatureUri={mechanicSignatureUri}
          setMechanicSignatureUri={setMechanicSignatureUri}
          operatorSignatureUri={operatorSignatureUri}
          setOperatorSignatureUri={setOperatorSignatureUri}
        />
      ),
    },
  ];

  const handleBackFromSignature = () => {
    if (hasUnsavedSignature) {
      Alert.alert(
        'Unsaved Changes',
        'You have unsaved signature changes. Are you sure you want to go back?',
        [
          { text: 'Cancel', style: 'cancel' },
          {
            text: 'Go Back',
            onPress: () => {
              setSignatureScreen(null);
              setHasUnsavedSignature(false);
            },
            style: 'destructive',
          },
        ]
      );
    } else {
      setSignatureScreen(null);
    }
  };

  const renderContent = () => {
    if (signatureScreen === 'Mechanic') {
      return (
        <MachanicSignaturedo
          report={report}
          onSave={(signatureUri) => {
            console.log('Mechanic signature saved:', signatureUri);
            setMechanicSignatureUri(signatureUri);
            setSignatureScreen(null);
            setHasUnsavedSignature(false);
          }}
          onBack={handleBackFromSignature}
          setHasUnsavedSignature={setHasUnsavedSignature}
        />
      );
    }
    if (signatureScreen === 'Operator') {
      return (
        <OperatorSignaturedo
          report={report}
          onSave={(signatureUri) => {
            console.log('Operator signature saved:', signatureUri);
            setOperatorSignatureUri(signatureUri);
            setSignatureScreen(null);
            setHasUnsavedSignature(false);
          }}
          onBack={handleBackFromSignature}
          setHasUnsavedSignature={setHasUnsavedSignature}
        />
      );
    }
    return steps.find((step) => step.id === activeStep).component;
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={styles.header}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <TouchableOpacity style={styles.homeButton} onPress={() => navigation.goBack()}>
          <MaterialIcons name="chevron-left" size={22} color="#fff" />
          <Text style={styles.headerHomeText}>Home</Text>
        </TouchableOpacity>
        <View style={styles.stepsRow}>
          <View style={styles.stepContainer}>
            {activeStep === 2 ? (
              <View style={styles.stepBadgeActive}>
                <MaterialIcons name="check" size={18} color="#fff" />
              </View>
            ) : (
              <View style={styles.stepBadgeActive}>
                <Text style={styles.stepBadgeTextActive}>1</Text>
              </View>
            )}
            <View style={styles.stepTextBlock}>
              <Text style={styles.stepLabelWhite}>Step 1</Text>
              <Text style={styles.stepSubtitle}>Corrective Action</Text>
            </View>
          </View>
          <View style={styles.stepContainer}>
            <View style={[styles.stepBadgeInactive, activeStep === 2 && { backgroundColor: '#4cd964' }]}>
              <Text style={[styles.stepBadgeTextInactive, activeStep === 2 && { color: '#fff' }]}>2</Text>
            </View>
            <View style={styles.stepTextBlock}>
              <Text style={styles.stepLabelWhite}>Step 2</Text>
              <Text style={styles.stepSubtitle}>Signature</Text>
            </View>
          </View>
        </View>
      </LinearGradient>
      <View style={styles.mainContent}>{renderContent()}</View>
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    height: 110,
    paddingTop: 32,
    paddingBottom: 0,
    paddingHorizontal: 0,
    justifyContent: 'flex-start',
    borderBottomLeftRadius: 0,
    borderBottomRightRadius: 0,
  },
  homeButton: {
    position: 'absolute',
    top: 24,
    left: 10,
    flexDirection: 'row',
    alignItems: 'center',
    zIndex: 2,
  },
  headerHomeText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 2,
  },
  stepsRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'flex-end',
    marginTop: 30,
    width: '100%',
  },
  stepContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 32,
  },
  stepBadgeActive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#4cd964',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeInactive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#e0e6ed',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeTextActive: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepBadgeTextInactive: {
    color: '#3481BC',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepTextBlock: {
    flexDirection: 'column',
    alignItems: 'flex-start',
  },
  stepLabelWhite: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    lineHeight: 18,
  },
  stepSubtitle: {
    color: '#fff',
    fontSize: 12,
    marginTop: -2,
  },
  mainContent: {
    flex: 1,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default DvirCorrectiveActionHeader;