import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Dimensions,
  Image,
  Animated,
  Modal,
  FlatList,
  Alert,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;
const API_BASE_URL = 'https://api.titandrillingzm.com:6008'; // Daily Reports API base URL

const DailyIns_report_home = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [activeTab, setActiveTab] = useState('running');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [reports, setReports] = useState([]);
  const [filteredReports, setFilteredReports] = useState([]);
  const [isOnline, setIsOnline] = useState(true);
  const [menuVisible, setMenuVisible] = useState(null);
  const [menuPosition, setMenuPosition] = useState({ x: 0, y: 0 });
  const [menuWidth, setMenuWidth] = useState(200); // Default dropdown width
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [userPermissions, setUserPermissions] = useState(null);
  const [isLoadingPermissions, setIsLoadingPermissions] = useState(false);
  const [uploadModalVisible, setUploadModalVisible] = useState(false);
  const [isUploading, setIsUploading] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Fetch reports from MongoDB - only user's own reports
  useEffect(() => {
    let unsubscribeNetInfo;

    const fetchReports = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          Toast.show({
            type: 'error',
            text1: 'Error',
            text2: 'User not authenticated.',
            position: 'bottom',
          });
          return;
        }

        // Fetch only current user's reports from MongoDB DailyReports endpoint
        const response = await fetch(`${API_BASE_URL}/daily-reports/${userUid}`);
        const result = await response.json();

        if (result.success) {
          const reportList = result.data.map(report => {
            // Use inspectionDate from step1 for accurate date, fallback to createdAt
            const inspectionDate = report.step1?.inspectionDate || report.createdAt;
            const status = report.pdfDownloadUrl || report.step3?.pdfDownloadUrl ? 'submitted' : 'running';
            
            return {
              id: report.id || report._id,
              requestNumber: report.step1?.requestNumber || report.requestNumber || 'N/A',
              equipmentName: report.step1?.selectedEquipment?.equipmentName || report.selectedEquipment?.equipmentName || 'N/A',
              createdAt: report.createdAt || '',
              inspectionDate: inspectionDate, // Actual inspection date
              status: status,
              pdfUrl: report.pdfDownloadUrl || report.step3?.pdfDownloadUrl || null,
              generatedAt: report.step3?.createdAt || report.updatedAt || null,
            };
          });

          // Sort by inspection date (most recent first)
          reportList.sort((a, b) => new Date(b.inspectionDate) - new Date(a.inspectionDate));
          setReports(reportList);
          filterReports(reportList, searchQuery, activeTab);
        } else {
          throw new Error(result.error || 'Failed to fetch reports');
        }

        // Check network status
        const netInfo = await NetInfo.fetch();
        setIsOnline(netInfo.isConnected);

        unsubscribeNetInfo = NetInfo.addEventListener((state) => {
          setIsOnline(state.isConnected);
        });
      } catch (error) {
        console.error('Error fetching reports:', error);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Failed to load reports.',
          position: 'bottom',
        });
      }
    };

    fetchReports();

    return () => {
      if (unsubscribeNetInfo) {
        unsubscribeNetInfo();
      }
    };
  }, []);

  // Filter reports based on search query and active tab - show only one report per day
  const filterReports = (reportList, query, tab) => {
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Reset to start of day
    const thirtyDaysAgo = new Date(today);
    thirtyDaysAgo.setDate(today.getDate() - 30);

    // First, filter by search query and tab
    let filtered = reportList.filter((report) => {
      // Search query matching
      const matchesQuery =
        report.requestNumber.toLowerCase().includes(query.toLowerCase()) ||
        report.equipmentName.toLowerCase().includes(query.toLowerCase());

      // Tab filtering
      let matchesTab = true;
      if (tab === 'submitted') {
        // Only show submitted reports older than 30 days (based on inspection date)
        const reportDate = new Date(report.inspectionDate || report.createdAt);
        reportDate.setHours(0, 0, 0, 0);
        matchesTab = report.status === 'submitted' && reportDate <= thirtyDaysAgo;
      } else if (tab === 'running') {
        // Show all reports in the Running tab
        matchesTab = true;
      }

      return matchesQuery && matchesTab;
    });

    // Group reports by date and keep only the latest report for each day
    const reportsByDate = {};
    filtered.forEach((report) => {
      const reportDate = new Date(report.inspectionDate || report.createdAt);
      const dateKey = reportDate.toISOString().slice(0, 10); // YYYY-MM-DD format
      
      // If no report for this date yet, or this report is more recent, use it
      if (!reportsByDate[dateKey] || 
          new Date(report.createdAt) > new Date(reportsByDate[dateKey].createdAt)) {
        reportsByDate[dateKey] = report;
      }
    });

    // Convert back to array and sort by date (most recent first)
    const uniqueReports = Object.values(reportsByDate);
    uniqueReports.sort((a, b) => new Date(b.inspectionDate || b.createdAt) - new Date(a.inspectionDate || a.createdAt));
    
    setFilteredReports(uniqueReports);
  };

  // Handle search query change
  useEffect(() => {
    filterReports(reports, searchQuery, activeTab);
  }, [searchQuery, activeTab, reports]);

  // Handle cloud sync press
  const handleCloudPress = () => {
    if (!isOnline) {
      Toast.show({
        type: 'error',
        text1: 'Offline',
        text2: 'Cannot sync data while offline.',
        position: 'bottom',
      });
      return;
    }

    setSyncModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    // Simulate sync (replace with actual sync logic if needed)
    setTimeout(() => {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      Toast.show({
        type: 'success',
        text1: 'Sync Complete',
        text2: 'Data synced successfully.',
        position: 'bottom',
      });
    }, 2000);
  };

  // Handle view PDF
  const handleViewPDF = (report) => {
    if (!isOnline) {
      Toast.show({
        type: 'error',
        text1: 'Offline',
        text2: 'Cannot view PDF while offline.',
        position: 'bottom',
      });
      return;
    }

    if (!report.pdfUrl) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'No PDF available.',
        position: 'bottom',
      });
      return;
    }

    setMenuVisible(null);
    navigation.navigate('PdfviewPage', {
      pdfUrl: report.pdfUrl,
      onGoBack: () => navigation.navigate('DailyIns_report_home'),
    });
  };

  // Handle delete report
  const handleDeleteReport = (reportId) => {
    setMenuVisible(null);
    Alert.alert(
      'Delete Report',
      'Are you sure you want to delete this report? This action cannot be undone.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Delete',
          style: 'destructive',
          onPress: async () => {
            try {
              const userUid = await AsyncStorage.getItem('userUid');
              if (!userUid) {
                Toast.show({
                  type: 'error',
                  text1: 'Error',
                  text2: 'User not authenticated.',
                  position: 'bottom',
                });
                return;
              }

              const response = await fetch(`${API_BASE_URL}/daily-reports/${userUid}/${reportId}`, {
                method: 'DELETE',
                headers: {
                  'Content-Type': 'application/json',
                },
              });

              const result = await response.json();

              if (result.success) {
                Toast.show({
                  type: 'success',
                  text1: 'Success',
                  text2: 'Report deleted successfully.',
                  position: 'bottom',
                });
                // Refresh the reports list
                const refreshResponse = await fetch(`${API_BASE_URL}/daily-reports/${userUid}`);
                const refreshResult = await refreshResponse.json();
                if (refreshResult.success) {
                  const reportList = refreshResult.data.map(report => {
                    const inspectionDate = report.step1?.inspectionDate || report.createdAt;
                    const status = report.pdfDownloadUrl || report.step3?.pdfDownloadUrl ? 'submitted' : 'running';
                    
                    return {
                      id: report.id || report._id,
                      requestNumber: report.step1?.requestNumber || report.requestNumber || 'N/A',
                      equipmentName: report.step1?.selectedEquipment?.equipmentName || report.selectedEquipment?.equipmentName || 'N/A',
                      createdAt: report.createdAt || '',
                      inspectionDate: inspectionDate,
                      status: status,
                      pdfUrl: report.pdfDownloadUrl || report.step3?.pdfDownloadUrl || null,
                      generatedAt: report.step3?.createdAt || report.updatedAt || null,
                    };
                  });
                  // Sort by inspection date (most recent first)
                  reportList.sort((a, b) => new Date(b.inspectionDate) - new Date(a.inspectionDate));
                  setReports(reportList);
                  filterReports(reportList, searchQuery, activeTab);
                }
              } else {
                throw new Error(result.error || 'Failed to delete report');
              }
            } catch (error) {
              console.error('Error deleting report:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Failed to delete report.',
                position: 'bottom',
              });
            }
          },
        },
      ]
    );
  };

  // Handle edit report
  const handleEditReport = (report) => {
    if (report.status !== 'running') {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Only running reports can be edited.',
        position: 'bottom',
      });
      return;
    }
    setMenuVisible(null);
    navigation.navigate('Daily_Ins_Header', { inspectionId: report.id });
  };

  // Fetch user permissions
  const fetchUserPermissions = async () => {
    try {
      setIsLoadingPermissions(true);
      const userUid = await AsyncStorage.getItem('userUid');

      if (!userUid) {
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'User not authenticated or UID not found.',
          position: 'bottom',
        });
        return;
      }

      // Fetch user profile and permissions from Admin Users API
      const response = await fetch(`https://api.titandrillingzm.com:6003/users/${userUid}`);
      const result = await response.json();

      if (result.success && result.data) {
        const userData = result.data;

        // Transform modules array into a flat object (same pattern as other components)
        const modules = {};
        (userData.modules || []).forEach((module) => {
          modules[module.module] = module.permissions && module.permissions.length > 0;
        });
        
        console.log('🔍 Debug - User modules array:', userData.modules);
        console.log('🔍 Debug - Transformed modules object:', modules);
        console.log('🔍 Debug - Daily Report enabled:', !!modules['dailyReport']);

        // Structure permissions data to include country, project, and all module permissions
        setUserPermissions({
          profile: {
            country: Array.isArray(userData.countries) && userData.countries.length > 0
              ? userData.countries.join(', ')
              : (userData.selectedCountry || userData.country || (userData.profile && userData.profile.country) || 'Not Set'),
            project: Array.isArray(userData.projects) && userData.projects.length > 0
              ? userData.projects.join(', ')
              : (userData.selectedProject || userData.project || (userData.profile && userData.profile.project) || 'Not Set'),
            email: userData.email || 'Not Available',
            uid: userUid,
          },
          dailyReportEnabled: !!modules['dailyReport'],
          allModules: modules, // Store all module permissions
          modulesList: userData.modules || [], // Store original modules array for detailed info
        });
        setPermissionModalVisible(true);
      } else {
        throw new Error(result.error || 'Failed to fetch user data');
      }
    } catch (error) {
      console.error('Error fetching permissions:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to load permissions.',
        position: 'bottom',
      });
    } finally {
      setIsLoadingPermissions(false);
    }
  };

  // Handle permission icon press
  const handlePermissionPress = () => {
    fetchUserPermissions();
  };

  // Handle upload DailyChecklistJSON
  const handleUploadChecklist = () => {
    if (!isOnline) {
      Toast.show({
        type: 'error',
        text1: 'Offline',
        text2: 'Cannot upload while offline.',
        position: 'bottom',
      });
      return;
    }
    setUploadModalVisible(true);
  };

  // Upload JSON file to MongoDB
  const uploadJSONFile = async () => {
    try {
      setIsUploading(true);
      console.log('🚀 Starting JSON upload process...');
      
      // Import the local JSON file
      const checklistsData = require('./DailyInsChecklistJson/DailyInsChecklistAll.json');
      console.log('📁 JSON file loaded successfully');

      // Validate JSON structure
      if (!checklistsData || typeof checklistsData !== 'object') {
        throw new Error('Invalid checklist data format');
      }

      // Get the AllDailyReportChecklistsJson data
      const allChecklistsData = checklistsData.AllDailyReportChecklistsJson;
      
      if (!allChecklistsData || typeof allChecklistsData !== 'object') {
        throw new Error('AllDailyReportChecklistsJson not found in file');
      }

      // Filter out non-checklist entries (like downloadUrl)
      const validChecklists = Object.keys(allChecklistsData).filter(key => 
        key !== 'downloadUrl' && 
        allChecklistsData[key] && 
        typeof allChecklistsData[key] === 'object' &&
        allChecklistsData[key].formTitle
      );

      console.log('📊 Valid checklists found:', validChecklists.length);
      console.log('📋 Checklist IDs:', validChecklists);

      if (validChecklists.length === 0) {
        throw new Error('No valid checklists found in the JSON file');
      }

      // Create a clean data object with only valid checklists
      const cleanChecklistsData = {};
      validChecklists.forEach(key => {
        cleanChecklistsData[key] = allChecklistsData[key];
      });

      console.log('🔄 Uploading to MongoDB...');

      // Upload to MongoDB
      const uploadResponse = await fetch(`${API_BASE_URL}/checklists/upload`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          checklistsData: cleanChecklistsData
        })
      });

      console.log('📡 Upload response status:', uploadResponse.status);

      if (!uploadResponse.ok) {
        let errorMessage = `HTTP error! status: ${uploadResponse.status}`;
        try {
          const errorData = await uploadResponse.json();
          errorMessage = errorData.error || errorMessage;
        } catch (parseError) {
          console.warn('Could not parse error response:', parseError);
        }
        throw new Error(errorMessage);
      }

      const uploadResult = await uploadResponse.json();
      console.log('📥 Upload result:', uploadResult);
      
      if (!uploadResult.success) {
        throw new Error(uploadResult.error || 'Upload failed');
      }

      console.log('✅ MongoDB upload completed successfully!');
      
      Toast.show({
        type: 'success',
        text1: 'Upload Successful',
        text2: `${validChecklists.length} checklists uploaded to MongoDB!`,
        position: 'bottom',
        visibilityTime: 4000,
      });

    } catch (error) {
      console.error('❌ Error uploading file:', error);
      
      let errorMessage = 'Failed to upload checklists.';
      if (error.message.includes('Invalid') || error.message.includes('not found')) {
        errorMessage = error.message;
      } else if (error.message.includes('Network request failed') || error.message.includes('fetch')) {
        errorMessage = 'Network error. Please check your connection and try again.';
      } else if (error.message.includes('HTTP error')) {
        errorMessage = `Server error: ${error.message}`;
      } else {
        errorMessage = error.message || errorMessage;
      }
      
      Toast.show({
        type: 'error',
        text1: 'Upload Failed',
        text2: errorMessage,
        position: 'bottom',
        visibilityTime: 5000,
      });
    } finally {
      setIsUploading(false);
      setUploadModalVisible(false);
    }
  };

  // Helper: check if a report exists for today (using inspection date)
  const today = new Date();
  today.setHours(0, 0, 0, 0);
  const todayStr = today.toISOString().slice(0, 10);
  
  // Check if there's a report for today's date using inspectionDate
  const hasTodayReport = reports.some(r => {
    if (!r.inspectionDate && !r.createdAt) return false;
    const reportDate = new Date(r.inspectionDate || r.createdAt);
    const reportDateStr = reportDate.toISOString().slice(0, 10);
    return reportDateStr === todayStr;
  });

  // Render report item
  const renderReportItem = ({ item }) => {
    // Use inspectionDate for accurate date display
    const reportDate = item.inspectionDate ? new Date(item.inspectionDate) : (item.createdAt ? new Date(item.createdAt) : null);
    const dayNames = ['M', 'T', 'W', 'T', 'F', 'S', 'S'];
    const dayIdx = reportDate ? (reportDate.getDay() === 0 ? 6 : reportDate.getDay() - 1) : null;
    const formattedDate = reportDate
      ? `${reportDate.getDate().toString().padStart(2, '0')}-${reportDate.toLocaleString('en-US', { month: 'short' })}-${reportDate.getFullYear()} ${reportDate.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit', hour12: true })}`
      : 'N/A';
    return (
      <View style={styles.reportCardCustom}>
        <View style={styles.reportRowCustom}>
          <Text style={styles.reportLabelCustom}>Report#:</Text>
          <Text style={styles.reportValueCustom}>{item.requestNumber}</Text>
        </View>
        <View style={styles.dividerCustom} />
        <View style={styles.reportRowCustom}>
          <Text style={styles.reportLabelCustom}>Vehicle</Text>
          <Text style={styles.reportValueCustom}>{item.equipmentName}</Text>
        </View>
        <View style={styles.dividerCustom} />
        <View style={[styles.reportRowCustom, { alignItems: 'center' }]}> 
          <Text style={styles.reportLabelCustom}>Date</Text>
          <Text style={styles.reportValueCustom}>{formattedDate}</Text>
          <TouchableOpacity
            style={styles.menuButtonCustom}
            onPress={event => {
              event.stopPropagation();
              event.target.measure((fx, fy, width, height, px, py) => {
                // Adjust for dropdown width and screen width
                let left = px;
                let top = py + height;
                if (left + menuWidth > Dimensions.get('window').width) {
                  left = Dimensions.get('window').width - menuWidth - 10;
                }
                if (top + 100 > Dimensions.get('window').height) {
                  top = Dimensions.get('window').height - 120;
                }
                setMenuPosition({ x: left, y: top });
                setMenuVisible(item.id);
              });
            }}
          >
            <MaterialIcons name="more-vert" size={22} color="#333" />
          </TouchableOpacity>
        </View>
        {/* Days of week bar */}
        <View style={styles.daysBarCustom}>
          {dayNames.map((d, idx) => (
            <View key={`${item.id}-${idx}`} style={[styles.dayBoxCustom, dayIdx === idx && styles.dayBoxActiveCustom]}>
              <Text style={[styles.dayTextCustom, dayIdx === idx && styles.dayTextActiveCustom]}>{d}</Text>
            </View>
          ))}
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.navigate('MainApp')}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Daily Inspection</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity style={styles.iconButton} onPress={() => navigation.navigate('TradckDailyreport')}>
              <MaterialIcons name="track-changes" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            {/* <TouchableOpacity style={styles.iconButton} onPress={handleUploadChecklist}>
              <MaterialIcons name="cloud-upload" size={width * 0.06} color="#fff" />
            </TouchableOpacity> */}
            <TouchableOpacity style={styles.iconButton} onPress={handleCloudPress}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.icons12}
              />
            </TouchableOpacity>
            <TouchableOpacity style={styles.iconButton} onPress={handlePermissionPress}>
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by equipment or request number"
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.searchIcon} />
          </View>
        </View>
        {/* Tabs */}
        <View style={styles.tabsContainer}>
          <TouchableOpacity
            style={[styles.tab]}
            onPress={() => setActiveTab('running')}
          >
            <Text style={[styles.tabText, activeTab === 'running' && styles.activeTabText]}>Running</Text>
            {activeTab === 'running' && <View style={styles.tabIndicator} />}
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.tab]}
            onPress={() => setActiveTab('submitted')}
          >
            <Text style={[styles.tabText, activeTab === 'submitted' && styles.activeTabText]}>Submitted</Text>
            {activeTab === 'submitted' && <View style={styles.tabIndicator} />}
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Reports List */}
      <FlatList
        data={filteredReports}
        renderItem={renderReportItem}
        keyExtractor={(item) => item.id}
        contentContainerStyle={styles.reportsContainer}
        ListHeaderComponent={
          !hasTodayReport ? (
            <View style={styles.reportCardCustom}>
              <View style={styles.reportRowCustom}>
                <Text style={styles.reportLabelCustom}>Report#:</Text>
                <Text style={styles.reportValueCustom}>-</Text>
              </View>
              <View style={styles.dividerCustom} />
              <View style={styles.reportRowCustom}>
                <Text style={styles.reportLabelCustom}>Vehicle</Text>
                <Text style={styles.reportValueCustom}>-</Text>
              </View>
              <View style={styles.dividerCustom} />
              <View style={[styles.reportRowCustom, { alignItems: 'center' }]}> 
                <Text style={styles.reportLabelCustom}>Date</Text>
                <Text style={styles.reportValueCustom}>{`${today.getDate().toString().padStart(2, '0')}-${today.toLocaleString('en-US', { month: 'short' })}-${today.getFullYear()} ${today.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit', hour12: true })}`}</Text>
                <View style={styles.menuButtonCustom} />
              </View>
              {/* Days of week bar with button */}
              <View style={styles.daysBarCustom}>
                {['M', 'T', 'W', 'T', 'F', 'S', 'S'].map((d, idx) => {
                  const isToday = (today.getDay() === 0 ? 6 : today.getDay() - 1) === idx;
                  return (
                    <View key={`today-${idx}`} style={[styles.dayBoxCustom, isToday && styles.dayBoxActiveCustom]}> 
                      <Text style={[styles.dayTextCustom, isToday && styles.dayTextActiveCustom]}>{d}</Text>
                    </View>
                  );
                })}
                <TouchableOpacity style={styles.todayBtnCustom} onPress={() => navigation.navigate('Daily_Ins_Header')}>
                  <Text style={styles.todayBtnTextCustom}>Today Start Inspection</Text>
                </TouchableOpacity>
              </View>
            </View>
          ) : null
        }
        ListEmptyComponent={
          <View style={styles.emptyContainer}>
            <Text style={styles.emptyText}>
              {activeTab === 'running'
                ? 'No inspections found.'
                : 'No submitted inspections found older than 30 days.'}
            </Text>
          </View>
        }
      />

      {/* Bottom Button */}
      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={styles.startButton}
            onPress={() => navigation.navigate('Daily_Ins_Header')}
          >
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}> Start New Inspection</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Permission Modal */}
      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{
            width: '85%',
            backgroundColor: '#fff',
            borderRadius: 18,
            padding: 28,
            alignItems: 'center',
            shadowColor: '#000',
            shadowOffset: { width: 0, height: 2 },
            shadowOpacity: 0.2,
            shadowRadius: 8,
            elevation: 8,
          }}>
            <Text style={{
              fontSize: 22,
              fontWeight: 'bold',
              color: '#025383',
              marginBottom: 18,
            }}>
              User Details
            </Text>
            {isLoadingPermissions ? (
              <ActivityIndicator size="large" color="#025383" />
            ) : !userPermissions ? (
              <Text style={{ color: '#FF6B6B', fontSize: 16, marginBottom: 12, textAlign: 'center' }}>Failed to load permissions</Text>
            ) : (
              <View style={{ width: '100%', marginBottom: 18 }}>
                <View style={{ marginBottom: 12 }}>
                  <Text style={{ color: '#888', fontSize: 14 }}>Country</Text>
                  <Text style={{ fontSize: 17, fontWeight: '600', color: '#025383', marginTop: 2 }}>{userPermissions.profile.country}</Text>
                </View>
                <View style={{ marginBottom: 12 }}>
                  <Text style={{ color: '#888', fontSize: 14 }}>Project</Text>
                  <Text style={{ fontSize: 17, fontWeight: '600', color: '#025383', marginTop: 2 }}>{userPermissions.profile.project}</Text>
                </View>
                
                {/* Daily Report Permissions Section */}
                <View style={{ marginBottom: 12 }}>
                  <Text style={{ color: '#888', fontSize: 14, marginBottom: 8 }}>Daily Report Permissions</Text>
                  {userPermissions.modulesList && userPermissions.modulesList.length > 0 ? (
                    (() => {
                      const dailyReportModule = userPermissions.modulesList.find(m => 
                        m.module === 'dailyReport' || m.module === 'DailyReport'
                      );
                      
                      if (!dailyReportModule) {
                        return (
                          <Text style={{ color: '#FF6B6B', fontSize: 14, fontStyle: 'italic' }}>
                            Daily Report module not assigned
                          </Text>
                        );
                      }
                      
                      const permissions = dailyReportModule.permissions || [];
                      const permissionTypes = ['create', 'edit', 'delete', 'view'];
                      
                      return (
                        <View style={{ 
                          backgroundColor: '#f8f9fa',
                          borderRadius: 8,
                          padding: 12,
                          borderWidth: 1,
                          borderColor: '#e0e0e0'
                        }}>
                          <View style={{ flexDirection: 'row', flexWrap: 'wrap', gap: 8 }}>
                            {permissionTypes.map((permission, index) => {
                              const hasPermission = permissions.includes(permission);
                              return (
                                <View key={index} style={{
                                  backgroundColor: hasPermission ? '#4CAF50' : '#FF6B6B',
                                  paddingHorizontal: 12,
                                  paddingVertical: 6,
                                  borderRadius: 16,
                                  marginRight: 8,
                                  marginBottom: 4
                                }}>
                                  <Text style={{
                                    color: '#fff',
                                    fontSize: 12,
                                    fontWeight: 'bold',
                                    textTransform: 'capitalize'
                                  }}>
                                    {permission}
                                  </Text>
                                </View>
                              );
                            })}
                          </View>
                          {permissions.length === 0 && (
                            <Text style={{ color: '#FF6B6B', fontSize: 12, marginTop: 8, fontStyle: 'italic' }}>
                              No permissions assigned
                            </Text>
                          )}
                        </View>
                      );
                    })()
                  ) : (
                    <Text style={{ color: '#888', fontSize: 14, fontStyle: 'italic' }}>No modules assigned</Text>
                  )}
                </View>
              </View>
            )}
            <TouchableOpacity
              style={{
                marginTop: 8,
                backgroundColor: '#025383',
                borderRadius: 8,
                paddingVertical: 10,
                paddingHorizontal: 32,
              }}
              onPress={() => setPermissionModalVisible(false)}
              accessibilityLabel="Close user details"
              accessibilityRole="button"
            >
              <Text style={{
                color: '#fff',
                fontWeight: 'bold',
                fontSize: 16,
              }}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Upload Modal */}
      <Modal
        visible={uploadModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setUploadModalVisible(false)}
      >
        <View style={{ flex: 1, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' }}>
          <View style={{
            width: '85%',
            backgroundColor: '#fff',
            borderRadius: 18,
            padding: 28,
            alignItems: 'center',
            shadowColor: '#000',
            shadowOffset: { width: 0, height: 2 },
            shadowOpacity: 0.2,
            shadowRadius: 8,
            elevation: 8,
          }}>
            <MaterialIcons name="cloud-upload" size={48} color="#025383" style={{ marginBottom: 16 }} />
            <Text style={{
              fontSize: 22,
              fontWeight: 'bold',
              color: '#025383',
              marginBottom: 8,
              textAlign: 'center',
            }}>
              Upload DailyChecklistJSON
            </Text>
            <Text style={{
              fontSize: 16,
              color: '#666',
              marginBottom: 24,
              textAlign: 'center',
              lineHeight: 22,
            }}>
              Upload DailyInsChecklistAll.json to MongoDB DailyChecklistJSON collection.
            </Text>
            
            {isUploading ? (
              <View style={{ alignItems: 'center' }}>
                <ActivityIndicator size="large" color="#025383" />
                <Text style={{ color: '#025383', fontSize: 16, marginTop: 12 }}>Uploading...</Text>
              </View>
            ) : (
              <View style={{ flexDirection: 'row', gap: 12 }}>
                <TouchableOpacity
                  style={{
                    backgroundColor: '#025383',
                    borderRadius: 8,
                    paddingVertical: 12,
                    paddingHorizontal: 24,
                    minWidth: 120,
                  }}
                  onPress={uploadJSONFile}
                >
                  <Text style={{
                    color: '#fff',
                    fontWeight: 'bold',
                    fontSize: 16,
                    textAlign: 'center',
                  }}>Upload Now</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={{
                    backgroundColor: '#ccc',
                    borderRadius: 8,
                    paddingVertical: 12,
                    paddingHorizontal: 24,
                    minWidth: 120,
                  }}
                  onPress={() => setUploadModalVisible(false)}
                >
                  <Text style={{
                    color: '#333',
                    fontWeight: 'bold',
                    fontSize: 16,
                    textAlign: 'center',
                  }}>Cancel</Text>
                </TouchableOpacity>
              </View>
            )}
          </View>
        </View>
      </Modal>

      {menuVisible && (
        <TouchableOpacity
          activeOpacity={1}
          style={{
            position: 'absolute',
            top: 0,
            left: 0,
            right: 0,
            bottom: 0,
            zIndex: 9999,
            elevation: 20,
          }}
          onPress={() => setMenuVisible(null)}
        >
          <View
            style={[
              styles.dropdownMenuCustom,
              {
                position: 'absolute',
                left: menuPosition.x,
                top: menuPosition.y,
                width: menuWidth,
                backgroundColor: '#fff',
                borderRadius: 8,
                shadowColor: '#000',
                shadowOffset: { width: 0, height: 2 },
                shadowOpacity: 0.2,
                shadowRadius: 4,
                elevation: 10,
              },
            ]}
            onLayout={e => setMenuWidth(e.nativeEvent.layout.width)}
          >
            {(() => {
              const item = filteredReports.find(r => r.id === menuVisible);
              if (!item) return null;
              return (
                <>
                  {item.pdfUrl && (
                    <TouchableOpacity style={styles.dropdownMenuItemCustom} onPress={() => { handleViewPDF(item); setMenuVisible(null); }}>
                      <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" />
                      <Text style={styles.dropdownMenuTextCustom}>View PDF</Text>
                    </TouchableOpacity>
                  )}
                  <TouchableOpacity style={styles.dropdownMenuItemCustom} onPress={() => { handleDeleteReport(item.id); setMenuVisible(null); }}>
                    <MaterialIcons name="delete" size={22} color="#e74c3c" />
                    <Text style={[styles.dropdownMenuTextCustom, { color: '#e74c3c' }]}>Delete</Text>
                  </TouchableOpacity>
                </>
              );
            })()}
          </View>
        </TouchableOpacity>
      )}

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    alignItems: 'center',
    width: '100%',
    paddingTop: width * 0.025,
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  searchIcon: {
    marginLeft: width * 0.012,
  },
  tabsContainer: {
    flexDirection: 'row',
    backgroundColor: 'transparent',
    marginTop: 0,
    paddingHorizontal: 20,
  },
  tab: {
    flex: 1,
    paddingVertical: 15,
    alignItems: 'center',
    position: 'relative',
  },
  tabText: {
    fontSize: 16,
    color: 'rgba(255, 255, 255, 0.7)',
    fontWeight: '500',
  },
  activeTabText: {
    color: '#fff',
  },
  tabIndicator: {
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    height: 3,
    backgroundColor: '#fff',
  },
  reportsContainer: {
    padding: 20,
    paddingBottom: 100, // Space for bottom button
  },
  reportCardCustom: {
    backgroundColor: '#fff',
    borderRadius: 1,
    borderWidth: 1,
    borderColor: '#e0e0e0',
    marginBottom: 16,
    padding: 0,
    elevation: 1,
  },
  reportRowCustom: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
    paddingHorizontal: 16,
    paddingVertical: 10,
    backgroundColor: '#fff',
  },
  reportLabelCustom: {
    fontWeight: 'bold',
    color: '#333',
    fontSize: 14,
    minWidth: 70,
  },
  reportValueCustom: {
    color: '#222',
    fontSize: 14,
    flex: 1,
    textAlign: 'right',
    marginLeft: 10,
  },
  dividerCustom: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginHorizontal: 16,
  },
  menuButtonCustom: {
    marginLeft: 8,
    padding: 4,
  },
  daysBarCustom: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#fff',
    paddingVertical: 8,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
  },
  dayBoxCustom: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: 2,
  },
  dayBoxActiveCustom: {
    backgroundColor: '#7be0b0',
    borderRadius: 1,
    paddingHorizontal: 8,
    paddingVertical: 2,
  },
  dayTextCustom: {
    color: '#888',
    fontSize: 14,
    fontWeight: '500',
  },
  dayTextActiveCustom: {
    color: '#fff',
    fontWeight: 'bold',
  },
  menuDropdown: {
    position: 'absolute',
    right: 0,
    top: 30,
    backgroundColor: '#fff',
    borderRadius: 1,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 3,
    zIndex: 1000,
  },
  dropdownMenuCustom: {
    position: 'absolute',
    right: 0,
    top: 30,
    backgroundColor: '#fff',
    borderRadius: 6,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    zIndex: 1000,
    minWidth: 180,
    paddingVertical: 4,
    paddingHorizontal: 0,
  },
  dropdownMenuItemCustom: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 16,
  },
  dropdownMenuTextCustom: {
    marginLeft: 12,
    fontSize: 16,
    color: '#015185',
  },
  emptyContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    padding: 20,
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    width: '100%',
    marginBottom: width * 0.025,
  },
  todayBtnCustom: {
    backgroundColor: '#3481BC',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    marginLeft: 8,
    alignSelf: 'center',
  },
  todayBtnTextCustom: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 13,
  },
  permissionModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  permissionModalContent: {
    width: '85%',
    maxHeight: '80%',
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
  },
  permissionModalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
  },
  closeIconButton: {
    padding: 5,
  },
  permissionModalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#333',
  },
  profileSection: {
    marginBottom: 15,
  },
  sectionTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 10,
  },
  profileItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  profileLabel: {
    fontSize: 15,
    fontWeight: '600',
    color: '#015185',
  },
  profileValue: {
    fontSize: 15,
    color: '#666',
    fontWeight: '500',
  },
  divider: {
    height: 1,
    backgroundColor: '#e0e0e0',
    marginVertical: 15,
  },
  modulesSection: {
    marginTop: 15,
  },
  moduleItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  moduleName: {
    fontSize: 15,
    fontWeight: '600',
    color: '#015185',
  },
  statusIndicator: {
    paddingVertical: 4,
    paddingHorizontal: 10,
    borderRadius: 5,
  },
  statusText: {
    color: '#fff',
    fontSize: 13,
    fontWeight: 'bold',
  },
  noModulesText: {
    fontSize: 14,
    color: '#888',
    textAlign: 'center',
    marginTop: 10,
  },
  errorText: {
    fontSize: 16,
    color: '#e74c3c',
    textAlign: 'center',
    marginTop: 10,
  },
  loadingContainer: {
    flexDirection: 'column',
    alignItems: 'center',
    marginTop: 20,
  },
  loadingText: {
    fontSize: 15,
    color: '#333',
    marginTop: 10,
  },
  closeButton: {
    backgroundColor: '#3481BC',
    borderRadius: 8,
    paddingVertical: 12,
    paddingHorizontal: 24,
    marginTop: 20,
    alignSelf: 'center',
  },
  closeButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export default DailyIns_report_home;