import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Dimensions,
  Alert,
  Modal,
  ScrollView,
  ActivityIndicator,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import Checklistfile from './Checklistfile.json';
import AllJsons from './AllJsons.json';

const ChecklistUploadNavigation = () => {
  const navigation = useNavigation();
  const [isUploading, setIsUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [uploadStatus, setUploadStatus] = useState('');
  const [showProgressModal, setShowProgressModal] = useState(false);
  const [uploadHistory, setUploadHistory] = useState([]);
  const [databaseStats, setDatabaseStats] = useState({
    totalDocuments: 0,
    totalUsers: 0,
    lastUpdated: null
  });
  const [uploadType, setUploadType] = useState('checklist'); // 'checklist', 'equipment', 'dailyreport', 'adminchecklist', 'dailyinspection'

  const scale = (size) => (Dimensions.get('window').width / 375) * size;

  useEffect(() => {
    loadUploadHistory();
    loadDatabaseStats();
    
    // Test AllJsons import
    console.log('=== ALLJSONS IMPORT TEST ===');
    console.log('AllJsons imported:', typeof AllJsons);
    console.log('AllJsons keys:', Object.keys(AllJsons || {}));
    console.log('AllJsons.checklist length:', AllJsons?.checklist?.length);
    console.log('First checklist:', AllJsons?.checklist?.[0]?.formTitle);
    console.log('=== END IMPORT TEST ===');
  }, []);

  const loadUploadHistory = async () => {
    try {
      const history = await AsyncStorage.getItem('checklistUploadHistory');
      if (history) {
        setUploadHistory(JSON.parse(history));
      }
    } catch (error) {
      console.error('Error loading upload history:', error);
    }
  };

  const loadDatabaseStats = async () => {
    try {
      // Check the correct Firebase path for checklist data
      const snapshot = await database()
        .ref('AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents')
        .once('value');
      
      const data = snapshot.val();
      if (data) {
        let totalDocs = 0;
        let totalUsers = 1; // Since this is a global path, count as 1 user
        let lastUpdate = null;

        // Count documents in the checklist data
        Object.keys(data).forEach(docKey => {
          const docData = data[docKey];
          if (docData && docData.document_id && docData.document_name) {
            totalDocs++;
            if (docData.uploadedAt) {
              const updateTime = new Date(docData.uploadedAt);
              if (!lastUpdate || updateTime > lastUpdate) {
                lastUpdate = updateTime;
              }
            }
          }
        });

        setDatabaseStats({
          totalDocuments: totalDocs,
          totalUsers: totalUsers,
          lastUpdated: lastUpdate
        });
      } else {
        // If no data found, set default values
        setDatabaseStats({
          totalDocuments: 0,
          totalUsers: 0,
          lastUpdated: null
        });
      }
    } catch (error) {
      console.error('Error loading database stats:', error);
      // Set default values on error
      setDatabaseStats({
        totalDocuments: 0,
        totalUsers: 0,
        lastUpdated: null
      });
    }
  };

  const saveUploadHistory = async (uploadData) => {
    try {
      const newHistory = [
        {
          timestamp: new Date().toISOString(),
          documentsCount: uploadData.documentsCount,
          status: uploadData.status,
          uploadType: uploadData.uploadType,
          uploadedBy: await AsyncStorage.getItem('userEmail') || 'Unknown',
        },
        ...uploadHistory.slice(0, 9), // Keep only last 10 entries
      ];
      await AsyncStorage.setItem('checklistUploadHistory', JSON.stringify(newHistory));
      setUploadHistory(newHistory);
    } catch (error) {
      console.error('Error saving upload history:', error);
    }
  };

  const uploadDataToDatabase = async (type) => {
    try {
      setIsUploading(true);
      setShowProgressModal(true);
      setUploadProgress(0);
      setUploadStatus('Initializing upload...');

      // Get user email
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        throw new Error('User not authenticated');
      }

      setUploadStatus('Preparing data...');
      setUploadProgress(10);

      // Prepare the data structure based on type
      const userKey = userEmail.replace(/[.#$[\]]/g, '_');
      let uploadData = {};
      let databasePath = '';

      switch (type) {
        case 'checklist':
          // Upload actual checklist data from Checklistfile.json
          const checklistDocuments = Checklistfile.Checklistfile.documents || [];

          // Convert array to object with document IDs as keys (preserve original ID case; only sanitize Firebase-forbidden chars)
          const checklistObject = {};
          checklistDocuments.forEach((document, index) => {
            const baseId = document.document_id ? String(document.document_id) : '';
            const sanitizedId = baseId.replace(/[.#$\[\]]/g, '_');
            const uniqueFallback = `doc_${Date.now()}_${index}`;
            const key = sanitizedId && sanitizedId.trim().length > 0 ? sanitizedId : uniqueFallback;

            checklistObject[key] = {
              ...document, // keep ALL fields as-is
              uploadedAt: new Date().toISOString(),
              uploadedBy: userEmail,
            };
          });

          uploadData = checklistObject;
          databasePath = `AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents`;
          break;

        case 'equipment':
          uploadData = {
            equipmentList: [], // Empty equipment data
            lastUpdated: new Date().toISOString(),
            uploadedBy: userEmail,
            version: '1.0',
          };
          databasePath = `EquipmentsDataJSON/GlobalEquipmentJsonData/${userKey}`;
          break;

        case 'dailyreport':
          uploadData = {
            dailyReports: [], // Empty daily report data
            lastUpdated: new Date().toISOString(),
            uploadedBy: userEmail,
            version: '1.0',
          };
          databasePath = `GlobalDailyreportdata/admins/Dailyreportnumber/${userKey}`;
          break;

        case 'adminchecklist':
          // Upload actual checklist data for admin checklist
          const adminChecklistDocuments = Checklistfile.Checklistfile.documents || [];

          // Convert array to object with document IDs as keys (preserve original case; sanitize only forbidden chars)
          const adminChecklistObject = {};
          adminChecklistDocuments.forEach((document, index) => {
            const baseId = document.document_id ? String(document.document_id) : '';
            const sanitizedId = baseId.replace(/[.#$\[\]]/g, '_');
            const uniqueFallback = `doc_${Date.now()}_${index}`;
            const key = sanitizedId && sanitizedId.trim().length > 0 ? sanitizedId : uniqueFallback;

            adminChecklistObject[key] = {
              ...document, // keep ALL fields as-is
              uploadedAt: new Date().toISOString(),
              uploadedBy: userEmail,
            };
          });

          uploadData = adminChecklistObject;
          databasePath = `AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents`;
          break;

        case 'dailyinspection':
          // Upload daily inspection data from AllJsons.json
          console.log('=== DAILY INSPECTION UPLOAD START ===');
          console.log('AllJsons data type:', typeof AllJsons);
          console.log('AllJsons structure:', Object.keys(AllJsons));
          
          // Get the checklist array from the correct structure
          const checklistArray = AllJsons.checklist || AllJsons;
          console.log('Checklist array length:', checklistArray?.length);
          console.log('Checklist array type:', typeof checklistArray);
          
          // Convert array to object with unique IDs as keys (matching your Firebase format)
          const dailyInspectionObject = {};
          
          // Check if checklistArray is an array and has data
          if (Array.isArray(checklistArray) && checklistArray.length > 0) {
            console.log(`Processing ${checklistArray.length} daily inspection checklists...`);
            
            checklistArray.forEach((checklist, index) => {
              // Validate checklist data
              if (!checklist.formTitle) {
                console.warn(`Skipping checklist at index ${index} - missing formTitle`);
                return;
              }
              
              // Create a unique key based on checklist name and timestamp
              const timestamp = Date.now();
              const safeKey = checklist.formTitle
                ? checklist.formTitle.toLowerCase().replace(/[^a-zA-Z0-9_-]/g, '_') + '_' + timestamp + '_' + index
                : `daily_inspection_${timestamp}_${index}`;
              
              // Add required fields to match your Firebase structure EXACTLY
              dailyInspectionObject[safeKey] = {
                // Core checklist data
                formTitle: checklist.formTitle,
                document: checklist.document,
                machineDetails: {
                  machineName: checklist.machineDetails?.machineName || "",
                  model: checklist.machineDetails?.model || "",
                  month: checklist.machineDetails?.month || "",
                  serialNumber: checklist.machineDetails?.serialNumber || ""
                },
                header_inputs: checklist.header_inputs || {},
                footer_inputs: checklist.footer_inputs || {},
                days: checklist.days || [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31],
                inspectionItems: checklist.inspectionItems || [],
                inspectionMatrix: checklist.inspectionMatrix || [],
                inspectedByInitials: {
                  type: "initials",
                  label: "Inspected by (Initials)"
                },
                // Firebase specific fields
                id: safeKey,
                uploadedAt: new Date().toISOString(),
                uploadedBy: userEmail
              };
              
              console.log(`Processed checklist ${index + 1}: ${checklist.formTitle} -> Key: ${safeKey}`);
            });
            
            console.log('Successfully processed daily inspection data. Object keys:', Object.keys(dailyInspectionObject));
            console.log('Total checklists to upload:', Object.keys(dailyInspectionObject).length);
            console.log('=== DAILY INSPECTION UPLOAD END ===');
          } else {
            throw new Error(`Invalid checklist data. Expected array, got: ${typeof checklistArray}. Length: ${checklistArray?.length}`);
          }

          uploadData = dailyInspectionObject;
          databasePath = `DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson`;
          
          // Verify data structure before upload
          console.log('=== DATA VERIFICATION ===');
          console.log('Upload data type:', typeof uploadData);
          console.log('Upload data keys count:', Object.keys(uploadData).length);
          console.log('Sample checklist structure:', Object.keys(uploadData)[0]);
          console.log('Sample checklist data:', uploadData[Object.keys(uploadData)[0]]);
          console.log('=== END VERIFICATION ===');
          break;

        default:
          throw new Error('Invalid upload type');
      }

      setUploadStatus(type === 'dailyinspection' ? 'Uploading to MongoDB...' : 'Uploading to Firebase...');
      setUploadProgress(30);

      if (type === 'dailyinspection') {
        // Upload to MongoDB for daily inspection checklists
        console.log('MongoDB upload data keys count:', Object.keys(uploadData).length);
        console.log('MongoDB upload data sample key:', Object.keys(uploadData)[0]);
        
        const uploadResponse = await fetch('https://api.titandrillingzm.com:6008/checklists/upload', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            checklistsData: uploadData
          })
        });

        if (!uploadResponse.ok) {
          const errorData = await uploadResponse.json();
          throw new Error(errorData.error || `HTTP error! status: ${uploadResponse.status}`);
        }

        const uploadResult = await uploadResponse.json();
        
        if (!uploadResult.success) {
          throw new Error(uploadResult.error || 'Upload failed');
        }
        
        console.log('MongoDB upload completed successfully!');
      } else {
        // Upload to Firebase for other types
        console.log('Firebase upload path:', databasePath);
        console.log('Upload data keys count:', Object.keys(uploadData).length);
        console.log('Upload data sample key:', Object.keys(uploadData)[0]);
        
        const dbRef = database().ref(databasePath);
        console.log('Starting Firebase upload...');
        await dbRef.set(uploadData);
        console.log('Firebase upload completed successfully!');
      }

      setUploadStatus('Finalizing upload...');
      setUploadProgress(80);

      // Update progress to completion
      setUploadProgress(100);
      setUploadStatus('Upload completed successfully!');

      // Save upload history
      let documentsCount = 0;
      if (type === 'checklist' || type === 'adminchecklist') {
        documentsCount = Checklistfile.Checklistfile.documents?.length || 0;
      } else if (type === 'dailyinspection') {
        const checklistArray = AllJsons.checklist || AllJsons;
        documentsCount = checklistArray.length || 0;
      }
      
      await saveUploadHistory({
        documentsCount: documentsCount,
        status: 'success',
        uploadType: type,
      });

      // Reload database stats after successful upload
      await loadDatabaseStats();

      // Show success message
      setTimeout(() => {
        setShowProgressModal(false);
        setIsUploading(false);
        Alert.alert(
          'Upload Successful',
          `Successfully uploaded ${type} data to ${type === 'dailyinspection' ? 'MongoDB' : 'Firebase'}!\n\n${type === 'dailyinspection' ? 'MongoDB Collection: DailyChecklistJSON' : `Path: ${databasePath}`}\nTotal Items: ${Object.keys(uploadData).length}\n\nAll ${type === 'dailyinspection' ? 'daily inspection' : type} checklists have been uploaded successfully.`,
          [{ text: 'OK' }]
        );
      }, 1500);

    } catch (error) {
      console.error('Upload error:', error);
      setUploadStatus(`Upload failed: ${error.message}`);
      
      // Save failed upload to history
      let documentsCount = 0;
      if (type === 'checklist' || type === 'adminchecklist') {
        documentsCount = Checklistfile.Checklistfile.documents?.length || 0;
      } else if (type === 'dailyinspection') {
        const checklistArray = AllJsons.checklist || AllJsons;
        documentsCount = checklistArray.length || 0;
      }
      
      await saveUploadHistory({
        documentsCount: documentsCount,
        status: 'failed',
        uploadType: type,
      });

      setTimeout(() => {
        setShowProgressModal(false);
        setIsUploading(false);
        Alert.alert('Upload Failed', error.message, [{ text: 'OK' }]);
      }, 2000);
    }
  };



  const downloadFromDatabase = async () => {
    try {
      setIsUploading(true);
      setShowProgressModal(true);
      setUploadProgress(0);
      setUploadStatus('Connecting to database...');

      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        throw new Error('User not authenticated');
      }

      setUploadStatus('Downloading checklist data...');
      setUploadProgress(30);

      // Download from Firebase using the correct path
      const snapshot = await database()
        .ref('AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents')
        .once('value');

      const data = snapshot.val();
      if (!data) {
        throw new Error('No checklist data found in database');
      }

      setUploadStatus('Processing downloaded data...');
      setUploadProgress(70);

      // Convert object back to array format for local storage
      const checklistArray = Object.values(data).filter(doc => 
        doc && doc.document_id && doc.document_name
      );

      // Save to local storage
      await AsyncStorage.setItem('checklists', JSON.stringify(checklistArray));

      setUploadProgress(100);
      setUploadStatus('Download completed successfully!');

      setTimeout(() => {
        setShowProgressModal(false);
        setIsUploading(false);
        Alert.alert(
          'Download Successful',
          `Downloaded ${checklistArray.length} checklist documents and saved locally.`,
          [{ text: 'OK' }]
        );
      }, 1500);

    } catch (error) {
      console.error('Download error:', error);
      setUploadStatus(`Download failed: ${error.message}`);
      
      setTimeout(() => {
        setShowProgressModal(false);
        setIsUploading(false);
        Alert.alert('Download Failed', error.message, [{ text: 'OK' }]);
      }, 2000);
    }
  };

  return (
    <View style={styles.container}>
      <StatusBar barStyle="light-content" backgroundColor="#015185" />

      {/* Header */}
      <LinearGradient colors={['#015185', '#3481BC']} style={styles.header}>
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.headerLeft}
            onPress={() => navigation.goBack()}
          >
            <MaterialIcons name="chevron-left" size={scale(28)} color="#fff" />
            <Text style={styles.headerText}>Checklist Database</Text>
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Main Content */}
      <ScrollView style={styles.content} showsVerticalScrollIndicator={false}>
        {/* Database Stats Section */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Database Statistics</Text>
          <View style={styles.statsContainer}>
            <View style={styles.statItem}>
              <MaterialIcons name="description" size={24} color="#015185" />
              <Text style={styles.statNumber}>{databaseStats.totalDocuments}</Text>
              <Text style={styles.statLabel}>Total Documents</Text>
            </View>
            <View style={styles.statItem}>
              <MaterialIcons name="people" size={24} color="#015185" />
              <Text style={styles.statNumber}>{databaseStats.totalUsers}</Text>
              <Text style={styles.statLabel}>Total Users</Text>
            </View>
            <View style={styles.statItem}>
              <MaterialIcons name="schedule" size={24} color="#015185" />
              <Text style={styles.statNumber}>
                {databaseStats.lastUpdated ? 
                  new Date(databaseStats.lastUpdated).toLocaleDateString() : 
                  'Never'
                }
              </Text>
              <Text style={styles.statLabel}>Last Updated</Text>
            </View>
          </View>
        </View>

        {/* Upload Section */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Upload to Database</Text>
          <Text style={styles.sectionDescription}>
            Upload different types of data to Firebase database
          </Text>
          
          {/* Test Data Button */}
          <TouchableOpacity
            style={[styles.uploadButton, { marginBottom: 10, backgroundColor: '#ffc107' }]}
            onPress={() => {
              console.log('=== TEST BUTTON CLICKED ===');
              console.log('AllJsons type:', typeof AllJsons);
              console.log('AllJsons structure:', Object.keys(AllJsons || {}));
              console.log('AllJsons.checklist length:', AllJsons?.checklist?.length);
              console.log('Sample checklist:', AllJsons?.checklist?.[0]);
              Alert.alert('Data Test', `AllJsons imported: ${typeof AllJsons}\nChecklist count: ${AllJsons?.checklist?.length || 0}`);
            }}
          >
            <Text style={[styles.buttonText, { color: '#000' }]}>Test AllJsons Data</Text>
          </TouchableOpacity>
          
          {/* Upload Type Selector */}
          <View style={styles.uploadTypeContainer}>
            <TouchableOpacity
              style={[
                styles.uploadTypeButton,
                uploadType === 'checklist' && styles.uploadTypeButtonActive
              ]}
              onPress={() => setUploadType('checklist')}
            >
              <MaterialIcons name="checklist" size={20} color={uploadType === 'checklist' ? '#fff' : '#015185'} />
              <Text style={[styles.uploadTypeText, uploadType === 'checklist' && styles.uploadTypeTextActive]}>
                Checklist
              </Text>
            </TouchableOpacity>
            
            <TouchableOpacity
              style={[
                styles.uploadTypeButton,
                uploadType === 'equipment' && styles.uploadTypeButtonActive
              ]}
              onPress={() => setUploadType('equipment')}
            >
              <MaterialIcons name="build" size={20} color={uploadType === 'equipment' ? '#fff' : '#015185'} />
              <Text style={[styles.uploadTypeText, uploadType === 'equipment' && styles.uploadTypeTextActive]}>
                Equipment
              </Text>
            </TouchableOpacity>
            
            <TouchableOpacity
              style={[
                styles.uploadTypeButton,
                uploadType === 'dailyreport' && styles.uploadTypeButtonActive
              ]}
              onPress={() => setUploadType('dailyreport')}
            >
              <MaterialIcons name="report" size={20} color={uploadType === 'dailyreport' ? '#fff' : '#015185'} />
              <Text style={[styles.uploadTypeText, uploadType === 'dailyreport' && styles.uploadTypeTextActive]}>
                Daily Report
              </Text>
            </TouchableOpacity>
            
            <TouchableOpacity
              style={[
                styles.uploadTypeButton,
                uploadType === 'adminchecklist' && styles.uploadTypeButtonActive
              ]}
              onPress={() => setUploadType('adminchecklist')}
            >
              <MaterialIcons name="admin-panel-settings" size={20} color={uploadType === 'adminchecklist' ? '#fff' : '#015185'} />
              <Text style={[styles.uploadTypeText, uploadType === 'adminchecklist' && styles.uploadTypeTextActive]}>
                Admin Checklist
              </Text>
            </TouchableOpacity>
            
            <TouchableOpacity
              style={[
                styles.uploadTypeButton,
                uploadType === 'dailyinspection' && styles.uploadTypeButtonActive
              ]}
              onPress={() => setUploadType('dailyinspection')}
            >
              <MaterialIcons name="assignment" size={20} color={uploadType === 'dailyinspection' ? '#fff' : '#015185'} />
              <Text style={[styles.uploadTypeText, uploadType === 'dailyinspection' && styles.uploadTypeTextActive]}>
                Daily Inspection
              </Text>
            </TouchableOpacity>
          </View>
          
          <TouchableOpacity
            style={styles.uploadButton}
            onPress={() => uploadDataToDatabase(uploadType)}
            disabled={isUploading}
          >
            <LinearGradient 
              colors={["#28a745", "#20c997"]} 
              style={styles.gradientButton}
            >
              <MaterialIcons name="cloud-upload" size={24} color="#fff" />
              <Text style={styles.buttonText}>
                {uploadType === 'dailyinspection' ? 'Upload Daily Inspection Data' : `Upload ${uploadType} Data`}
              </Text>
            </LinearGradient>
          </TouchableOpacity>
        </View>

        {/* Download Section */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Download from Database</Text>
          <Text style={styles.sectionDescription}>
            Download checklist data from Firebase database
          </Text>
          
          <TouchableOpacity
            style={styles.downloadButton}
            onPress={downloadFromDatabase}
            disabled={isUploading}
          >
            <LinearGradient 
              colors={["#007bff", "#0056b3"]} 
              style={styles.gradientButton}
            >
              <MaterialIcons name="cloud-download" size={24} color="#fff" />
              <Text style={styles.buttonText}>Download from Database</Text>
            </LinearGradient>
          </TouchableOpacity>
        </View>

        {/* Upload History */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Upload History</Text>
          <View style={styles.historyContainer}>
            {uploadHistory.length > 0 ? (
              uploadHistory.map((item, index) => (
                <View key={index} style={styles.historyItem}>
                  <View style={styles.historyHeader}>
                    <MaterialIcons 
                      name={item.status === 'success' ? 'check-circle' : 'error'} 
                      size={20} 
                      color={item.status === 'success' ? '#28a745' : '#dc3545'} 
                    />
                    <Text style={styles.historyDate}>
                      {new Date(item.timestamp).toLocaleDateString()}
                    </Text>
                  </View>
                  <Text style={styles.historyDetails}>
                    {item.documentsCount} documents • {item.uploadedBy}
                  </Text>
                </View>
              ))
            ) : (
              <Text style={styles.noHistoryText}>No upload history available</Text>
            )}
          </View>
        </View>
      </ScrollView>

      {/* Progress Modal */}
      <Modal
        animationType="fade"
        transparent={true}
        visible={showProgressModal}
        onRequestClose={() => !isUploading && setShowProgressModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <ActivityIndicator size="large" color="#015185" />
            <Text style={styles.modalTitle}>Processing...</Text>
            <Text style={styles.modalStatus}>{uploadStatus}</Text>
            
            {/* Progress Bar */}
            <View style={styles.progressContainer}>
              <View style={styles.progressBar}>
                <View 
                  style={[
                    styles.progressFill, 
                    { width: `${uploadProgress}%` }
                  ]} 
                />
              </View>
              <Text style={styles.progressText}>{uploadProgress}%</Text>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  header: {
    height: 100,
    justifyContent: 'center',
    alignItems: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    width: '100%',
    paddingHorizontal: 20,
    top: 20,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerText: {
    color: '#fff',
    fontSize: 20,
    fontWeight: '600',
    marginLeft: 10,
  },
  content: {
    flex: 1,
    padding: 20,
  },
  section: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 20,
    marginBottom: 20,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 8,
  },
  sectionDescription: {
    fontSize: 14,
    color: '#666',
    marginBottom: 16,
    lineHeight: 20,
  },
  uploadButton: {
    borderRadius: 12,
    overflow: 'hidden',
    marginBottom: 10,
  },
  downloadButton: {
    borderRadius: 12,
    overflow: 'hidden',
    marginBottom: 10,
  },
  gradientButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 16,
    paddingHorizontal: 24,
  },
  buttonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 8,
  },
  historyContainer: {
    marginTop: 10,
  },
  historyItem: {
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    padding: 12,
    marginBottom: 8,
    borderLeftWidth: 4,
    borderLeftColor: '#015185',
  },
  historyHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: 4,
  },
  historyDate: {
    fontSize: 12,
    color: '#666',
    fontWeight: '500',
  },
  historyDetails: {
    fontSize: 12,
    color: '#888',
  },
  noHistoryText: {
    textAlign: 'center',
    color: '#999',
    fontStyle: 'italic',
    padding: 20,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 30,
    alignItems: 'center',
    width: '80%',
    maxWidth: 300,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 16,
    marginBottom: 8,
  },
  modalStatus: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginBottom: 20,
  },
  progressContainer: {
    width: '100%',
    alignItems: 'center',
  },
  progressBar: {
    width: '100%',
    height: 8,
    backgroundColor: '#e9ecef',
    borderRadius: 4,
    overflow: 'hidden',
    marginBottom: 8,
  },
  progressFill: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 4,
  },
  progressText: {
    fontSize: 12,
    color: '#015185',
    fontWeight: '600',
  },
  statsContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginTop: 10,
  },
  statItem: {
    alignItems: 'center',
    flex: 1,
  },
  statNumber: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 4,
  },
  statLabel: {
    fontSize: 12,
    color: '#666',
    marginTop: 2,
    textAlign: 'center',
  },
  uploadTypeContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 16,
    flexWrap: 'wrap',
    gap: 8,
  },
  uploadTypeButton: {
    flex: 1,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    paddingVertical: 8,
    paddingHorizontal: 12,
    marginHorizontal: 4,
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  uploadTypeButtonActive: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  uploadTypeText: {
    fontSize: 12,
    color: '#015185',
    fontWeight: '500',
    marginTop: 4,
  },
  uploadTypeTextActive: {
    color: '#fff',
  },
});

export default ChecklistUploadNavigation;

// import React, { useState, useEffect } from 'react';
// import { View, Text, StyleSheet, TouchableOpacity, ActivityIndicator, Platform, ToastAndroid, Modal } from 'react-native';
// import LinearGradient from 'react-native-linear-gradient';
// import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
// import AsyncStorage from '@react-native-async-storage/async-storage';
// import database from '@react-native-firebase/database';
// import storage from '@react-native-firebase/storage';
// import auth from '@react-native-firebase/auth';
// import NetInfo from '@react-native-community/netinfo';
// import Toast from 'react-native-toast-message';
// import checklistData from './AllJsons.json'; // Path to your checklist JSON (ensure it contains the `checklist` array)
// import { encode as b64encode } from 'base-64';
// import { encode as utf8encode } from 'utf8';

// const ChecklistUploadNavigation = () => {
//   const [isUploading, setIsUploading] = useState(false);
//   const [uploadStatus, setUploadStatus] = useState('Ready to upload');
//   const [permissionsModalVisible, setPermissionsModalVisible] = useState(false);
//   const [userPermissions, setUserPermissions] = useState({
//     countries: [],
//     projects: [],
//     checklistPermissions: [],
//   });

//   // Validate checklist data structure
//   const validateChecklistData = (data) => {
//     try {
//       if (!data || !Array.isArray(data.checklist)) {
//         return { valid: false, error: 'Checklist data must contain an array under the "checklist" key' };
//       }
      
//       console.log(`Validating ${data.checklist.length} checklist forms...`);
      
//       for (let i = 0; i < data.checklist.length; i++) {
//         const form = data.checklist[i];
        
//         // Basic structure validation
//         if (!form.formTitle) {
//           return { valid: false, error: `Form at index ${i} is missing formTitle` };
//         }
        
//         if (!form.inspectionItems || !Array.isArray(form.inspectionItems)) {
//           return { valid: false, error: `Form "${form.formTitle}" is missing or has invalid inspectionItems` };
//         }
        
//         if (!form.inspectionMatrix || !Array.isArray(form.inspectionMatrix)) {
//           return { valid: false, error: `Form "${form.formTitle}" is missing or has invalid inspectionMatrix` };
//         }
        
//         // Validate inspection matrix structure
//         for (let j = 0; j < form.inspectionMatrix.length; j++) {
//           const item = form.inspectionMatrix[j];
//           if (!item.item) {
//             return { valid: false, error: `Form "${form.formTitle}" has invalid inspection matrix item at index ${j}` };
//           }
          
//           if (!item.checks || !Array.isArray(item.checks)) {
//             return { valid: false, error: `Form "${form.formTitle}" item "${item.item}" has invalid checks array` };
//           }
          
//           // Check if checks array has 31 elements (for daily inspection)
//           if (item.checks.length !== 31) {
//             console.warn(`Form "${form.formTitle}" item "${item.item}" has ${item.checks.length} checks instead of 31`);
//           }
//         }
        
//         console.log(`✓ Form "${form.formTitle}" validated successfully`);
//       }
      
//       console.log('All checklist forms validated successfully!');
//       return { valid: true };
//     } catch (error) {
//       return { valid: false, error: `Validation error: ${error.message}` };
//     }
//   };

//   // Check network status
//   const checkNetworkStatus = async () => {
//     try {
//       const netInfo = await NetInfo.fetch();
//       return netInfo.isConnected;
//     } catch (error) {
//       console.error('Error checking network status:', error);
//       return false;
//     }
//   };

//   // Show toast notifications
//   const showToast = (type, text1, text2) => {
//     if (Platform.OS === 'android') {
//       ToastAndroid.show(`${text1}: ${text2}`, ToastAndroid.LONG);
//     } else {
//       Toast.show({
//         type,
//         text1,
//         text2,
//         position: 'bottom',
//       });
//     }
//   };

//   // Fetch user permissions from Firebase
//   const fetchUserPermissions = async () => {
//     try {
//       const userUid = await AsyncStorage.getItem('userUid');
//       if (!userUid) {
//         throw new Error('User not authenticated. Please log in again.');
//       }

//       const userSnapshot = await database()
//         .ref(`/Globalusers/admins/${userUid}`)
//         .once('value');
//       const userData = userSnapshot.val();

//       if (!userData) {
//         throw new Error('User data not found.');
//       }

//       const checklistModule = userData.modules?.find((mod) => mod.module === 'checklist');
//       const permissions = checklistModule?.permissions || [];

//       setUserPermissions({
//         countries: userData.countries || [],
//         projects: userData.projects || [],
//         checklistPermissions: permissions,
//       });

//       if (!permissions.includes('upload')) {
//         throw new Error('User lacks permission to upload checklists.');
//       }
//     } catch (error) {
//       console.error('Error fetching user permissions:', error);
//       showToast('error', 'Error', error.message || 'Failed to fetch user permissions.');
//       setUserPermissions({
//         countries: [],
//         projects: [],
//         checklistPermissions: [],
//       });
//     }
//   };

//   // Upload checklist data to Firebase
//   const uploadToFirebase = async () => {
//     setIsUploading(true);
//     setUploadStatus('Uploading...');

//     try {
//       const user = auth().currentUser;
//       if (!user) {
//         throw new Error('Please log in to upload data');
//       }

//       const userUid = await AsyncStorage.getItem('userUid');
//       if (!userUid) {
//         throw new Error('User authentication data missing');
//       }

//       const isOnline = await checkNetworkStatus();
//       if (!isOnline) {
//         setUploadStatus('Failed: No internet connection');
//         showToast('info', 'Offline', 'Data will be uploaded when online');
//         await AsyncStorage.setItem('pendingChecklistUpload', JSON.stringify(checklistData));
//         setIsUploading(false);
//         return;
//       }

//       // Validate checklist data
//       console.log('=== STARTING UPLOAD PROCESS ===');
//       console.log('Raw checklist data:', typeof checklistData);
//       console.log('Checklist data keys:', Object.keys(checklistData || {}));
      
//       const validation = validateChecklistData(checklistData);
//       if (!validation.valid) {
//         throw new Error(validation.error);
//       }

//       console.log('✓ Data validation passed');
//       console.log('Uploading checklist data:', checklistData.checklist.length, 'forms');

//       // Firebase Realtime Database path
//       const dbRef = database().ref('/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson');
//       console.log('Firebase database path:', '/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson');

//       // Upload to Realtime Database
//       console.log('Starting database upload...');
//       await dbRef.set(checklistData.checklist);
//       console.log('✓ Data uploaded to Realtime Database successfully');

//       // Upload to Firebase Storage (simplified approach)
//       try {
//         const storageRef = storage().ref('/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson/AllJsons.json');
//         const jsonString = JSON.stringify(checklistData.checklist);
//         const blob = new Blob([jsonString], { type: 'application/json' });
//         await storageRef.put(blob);
//         const downloadUrl = await storageRef.getDownloadURL();
//         console.log('Data uploaded to Storage, download URL:', downloadUrl);

//         // Update Realtime Database with download URL
//         await dbRef.update({ downloadUrl });
//       } catch (storageError) {
//         console.warn('Storage upload failed, but database upload succeeded:', storageError.message);
//         // Continue even if storage fails
//       }

//       // Cache locally
//       await AsyncStorage.setItem('checklist_data', JSON.stringify(checklistData.checklist));

//       console.log('=== UPLOAD COMPLETED SUCCESSFULLY ===');
//       console.log('Total forms uploaded:', checklistData.checklist.length);
//       console.log('Database path:', '/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson');
      
//       setUploadStatus('Success: Data uploaded to Firebase');
//       showToast('success', 'Success', `Successfully uploaded ${checklistData.checklist.length} checklist forms to Firebase!`);
//       await AsyncStorage.removeItem('pendingChecklistUpload');
//     } catch (error) {
//       console.error('Error uploading to Firebase:', error);
//       setUploadStatus(`Failed: ${error.message}`);
//       showToast('error', 'Error', `Failed to upload data: ${error.message}`);
//       await AsyncStorage.setItem('pendingChecklistUpload', JSON.stringify(checklistData));
//     } finally {
//       setIsUploading(false);
//     }
//   };

//   // Check for pending uploads and initialize on mount
//   useEffect(() => {
//     const initialize = async () => {
//       await fetchUserPermissions();

//       const pendingData = await AsyncStorage.getItem('pendingChecklistUpload');
//       if (pendingData && (await checkNetworkStatus())) {
//         setIsUploading(true);
//         setUploadStatus('Syncing pending upload...');
//         try {
//           const data = JSON.parse(pendingData);
//           const validation = validateChecklistData({ checklist: data });
//           if (!validation.valid) {
//             throw new Error(validation.error);
//           }

//           console.log('Syncing pending data:', data.length, 'forms');

//           // Firebase Realtime Database path
//           const dbRef = database().ref('/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson');

//           // Upload to Realtime Database
//           await dbRef.set(data);
//           console.log('Pending data uploaded to Realtime Database');

//           // Upload to Firebase Storage (simplified approach)
//           try {
//             const storageRef = storage().ref('/DailyInspectionReportJsonChecklists/AllDailyReportChecklistsJson/AllJsons.json');
//             const jsonString = JSON.stringify(data);
//             const blob = new Blob([jsonString], { type: 'application/json' });
//             await storageRef.put(blob);
//             const downloadUrl = await storageRef.getDownloadURL();
//             console.log('Pending data uploaded to Storage, download URL:', downloadUrl);

//             // Update Realtime Database with download URL
//             await dbRef.update({ downloadUrl });
//           } catch (storageError) {
//             console.warn('Storage upload failed for pending data, but database upload succeeded:', storageError.message);
//             // Continue even if storage fails
//           }

//           // Cache locally
//           await AsyncStorage.setItem('checklist_data', JSON.stringify(data));

//           setUploadStatus('Success: Pending data synced');
//           showToast('success', 'Success', 'Pending data uploaded to Firebase');
//           await AsyncStorage.removeItem('pendingChecklistUpload');
//         } catch (error) {
//           console.error('Error syncing pending upload:', error);
//           setUploadStatus(`Failed to sync: ${error.message}`);
//           showToast('error', 'Error', `Failed to sync pending data: ${error.message}`);
//         } finally {
//           setIsUploading(false);
//         }
//       }
//     };

//     initialize();

//     const unsubscribe = auth().onAuthStateChanged((user) => {
//       if (!user) {
//         setUploadStatus('Please log in to upload data');
//       } else {
//         setUploadStatus('Ready to upload');
//       }
//     });

//     return () => unsubscribe();
//   }, []);

//   return (
//     <View style={styles.container}>
//       <LinearGradient colors={['#3481BC', '#025383']} style={styles.header}>
//         <View style={styles.headerContent}>
//           <TouchableOpacity onPress={() => {}}>
//             <MaterialIcons name="arrow-back" size={24} color="#fff" />
//           </TouchableOpacity>
//           <Text style={styles.headerTitle}>Checklist Upload</Text>
//           <View style={styles.headerRight}>
//             <TouchableOpacity
//               style={styles.iconButton}
//               onPress={() => setPermissionsModalVisible(true)}
//             >
//               <MaterialIcons name="lock" size={24} color="#fff" />
//             </TouchableOpacity>
//           </View>
//         </View>
//       </LinearGradient>

//       <View style={styles.mainContent}>
//         <Text style={styles.countText}>
//           Total Checklists: {checklistData?.checklist?.length || 0}
//         </Text>
//         <Text style={styles.statusText}>Status: {uploadStatus}</Text>
//         <TouchableOpacity
//           style={[styles.uploadButton, isUploading && styles.disabledButton]}
//           onPress={uploadToFirebase}
//           disabled={isUploading}
//         >
//           <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
//             <Text style={styles.buttonText}>
//               {isUploading ? 'Uploading...' : 'Upload Checklist Data'}
//             </Text>
//             {isUploading && <ActivityIndicator color="#fff" style={styles.loader} />}
//           </LinearGradient>
//         </TouchableOpacity>
//       </View>

//       <Modal
//         visible={permissionsModalVisible}
//         transparent={true}
//         animationType="fade"
//         onRequestClose={() => setPermissionsModalVisible(false)}
//       >
//         <View style={styles.modalOverlay}>
//           <View style={styles.permissionsModalContent}>
//             <Text style={styles.permissionsModalTitle}>User Permissions</Text>
//             <View style={styles.permissionsSection}>
//               <Text style={styles.permissionsLabel}>Countries</Text>
//               <Text style={styles.permissionsValue}>
//                 {userPermissions.countries.length > 0
//                   ? userPermissions.countries.join(', ')
//                   : 'None'}
//               </Text>
//             </View>
//             <View style={styles.permissionsSection}>
//               <Text style={styles.permissionsLabel}>Projects</Text>
//               <Text style={styles.permissionsValue}>
//                 {userPermissions.projects.length > 0
//                   ? userPermissions.projects.join(', ')
//                   : 'None'}
//               </Text>
//             </View>
//             <View style={styles.permissionsSection}>
//               <Text style={styles.permissionsLabel}>Checklist Permissions</Text>
//               <Text style={styles.permissionsValue}>
//                 {userPermissions.checklistPermissions.length > 0
//                   ? userPermissions.checklistPermissions.join(', ')
//                   : 'None'}
//               </Text>
//             </View>
//             <TouchableOpacity
//               style={[styles.modalButton, styles.cancelButton]}
//               onPress={() => setPermissionsModalVisible(false)}
//             >
//               <Text style={styles.modalButtonText}>Close</Text>
//             </TouchableOpacity>
//           </View>
//         </View>
//       </Modal>

//       <Toast />
//     </View>
//   );
// };

// const styles = StyleSheet.create({
//   container: {
//     flex: 1,
//     backgroundColor: '#fff',
//   },
//   header: {
//     paddingTop: Platform.OS === 'ios' ? 50 : 30,
//     paddingBottom: 20,
//     paddingHorizontal: 16,
//   },
//   headerContent: {
//     flexDirection: 'row',
//     alignItems: 'center',
//     justifyContent: 'space-between',
//   },
//   headerRight: {
//     flexDirection: 'row',
//     alignItems: 'center',
//   },
//   headerTitle: {
//     color: '#fff',
//     fontSize: 20,
//     fontWeight: 'bold',
//     marginLeft: 10,
//   },
//   iconButton: {
//     padding: 5,
//   },
//   mainContent: {
//     flex: 1,
//     padding: 16,
//     justifyContent: 'center',
//     alignItems: 'center',
//   },
//   statusText: {
//     fontSize: 16,
//     color: '#333',
//     marginBottom: 20,
//     textAlign: 'center',
//   },
//   uploadButton: {
//     borderRadius: 8,
//     overflow: 'hidden',
//   },
//   disabledButton: {
//     opacity: 0.6,
//   },
//   buttonGradient: {
//     flexDirection: 'row',
//     alignItems: 'center',
//     paddingVertical: 12,
//     paddingHorizontal: 24,
//   },
//   buttonText: {
//     color: '#fff',
//     fontSize: 16,
//     fontWeight: '600',
//   },
//   loader: {
//     marginLeft: 10,
//   },
//   modalOverlay: {
//     flex: 1,
//     justifyContent: 'center',
//     alignItems: 'center',
//     backgroundColor: 'rgba(0,0,0,0.5)',
//   },
//   permissionsModalContent: {
//     backgroundColor: '#fff',
//     borderRadius: 10,
//     padding: 20,
//     width: '80%',
//     alignItems: 'center',
//   },
//   permissionsModalTitle: {
//     fontSize: 18,
//     fontWeight: 'bold',
//     color: '#3481BC',
//     marginBottom: 15,
//   },
//   permissionsSection: {
//     width: '100%',
//     marginBottom: 15,
//   },
//   permissionsLabel: {
//     fontSize: 16,
//     fontWeight: '600',
//     color: '#333',
//     marginBottom: 5,
//   },
//   permissionsValue: {
//     fontSize: 16,
//     color: '#333',
//     textAlign: 'left',
//   },
//   modalButton: {
//     flex: 1,
//     paddingVertical: 10,
//     alignItems: 'center',
//     borderRadius: 5,
//     marginHorizontal: 5,
//   },
//   cancelButton: {
//     backgroundColor: '#f0f0f0',
//   },
//   modalButtonText: {
//     fontSize: 16,
//     color: '#333',
//     fontWeight: '600',
//   },
//   countText: {
//     fontSize: 18,
//     color: '#025383',
//     fontWeight: 'bold',
//     marginBottom: 10,
//     textAlign: 'center',
//   },
// });

// export default ChecklistUploadNavigation;
