import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, StatusBar, Platform, Dimensions, TouchableOpacity, Modal, ActivityIndicator } from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import LinearGradient from 'react-native-linear-gradient';
import { useNavigation } from '@react-navigation/native';
import { MongoAPI } from '../../ConfigMongo/MongoDBConfig';
import auth from '@react-native-firebase/auth';

const ViewProfilepage = () => {
  const navigation = useNavigation();
  const { width } = Dimensions.get('window');
  const TOTAL_HEADER_HEIGHT = width * 0.25;
  const [loading, setLoading] = useState(true);
  const [profileData, setProfileData] = useState({
    name: '',
    role: '',
    email: '',
    employeeNumber: '',
  });

  useEffect(() => {
    // Fetch profile data using the same pattern as MainNavigation
    const fetchProfileData = async () => {
      try {
        setLoading(true);
        
        // Try to use cached MongoDB data first
        const cachedStr = await AsyncStorage.getItem('userData');
        let cachedUser = cachedStr ? JSON.parse(cachedStr) : null;
        if (cachedUser) {
          setProfileData({
            name: (cachedUser.name || '').trim(),
            role: (cachedUser.role || '').trim(),
            email: (cachedUser.email || '').trim(),
            employeeNumber: (cachedUser.employeeNumber || '').trim(),
          });
        }

        // Attempt to fetch latest from MongoDB using stored uid
        const uid = cachedUser?._id || cachedUser?.uid || await AsyncStorage.getItem('userUid');
        if (uid) {
          try {
            const mongoRes = await MongoAPI.getUserById(uid);
            if (mongoRes?.success && mongoRes.data) {
              const fresh = mongoRes.data;
              const sanitized = {
                name: (fresh.name || '').trim(),
                role: (fresh.role || '').trim(),
                email: (fresh.email || '').trim(),
                employeeNumber: (fresh.employeeNumber || '').trim(),
              };
              setProfileData(sanitized);
              await AsyncStorage.setItem('userData', JSON.stringify({ ...fresh, ...sanitized }));
              cachedUser = fresh;
            }
          } catch (mongoError) {
            console.warn('⚠️ Mongo profile fetch failed:', mongoError.message);
          }
        }

        // Fall back to Firebase auth details if name/email still empty
        const user = auth().currentUser;
        if (user) {
          setProfileData(prev => ({
            name: prev.name || user.displayName || '',
            role: prev.role,
            email: prev.email || user.email || '',
            employeeNumber: prev.employeeNumber,
          }));
        }
        
      } catch (error) {
        console.error('Error fetching profile data:', error);
        // Set fallback data
        setProfileData({
          name: 'User',
          role: 'User',
          email: '',
          employeeNumber: '',
        });
      } finally {
        setLoading(false);
      }
    };

    fetchProfileData();
  }, []);

  return (
    <View style={styles.container}>
      <StatusBar
        translucent
        backgroundColor="transparent"
        barStyle="light-content"
      />
      {/* Header with gradient */}
      <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        <View style={styles.headerContent}>
          <TouchableOpacity style={styles.backButton} onPress={() => navigation.goBack()}>
            <MaterialIcons name="arrow-back" size={24} color="white" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>User Profile</Text>
        </View>
      </LinearGradient>
      
      {/* Loading Modal */}
      <Modal
        transparent={true}
        visible={loading}
        animationType="fade"
      >
        <View style={styles.loadingModal}>
          <View style={styles.loadingContent}>
            <ActivityIndicator size="large" color="#3481BC" />
            <Text style={styles.loadingText}>Loading Profile...</Text>
          </View>
        </View>
      </Modal>

      {/* Profile Fields */}
      <View style={styles.content}>
        {/* Location/Country */}
        {/* <View style={styles.fieldContainer}>
          <MaterialIcons name="location-on" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>LOCATION/COUNTRY</Text>
            <Text style={styles.value}>
              {profileData.countries && profileData.countries.length > 0 
                ? profileData.countries.join(', ') 
                : 'Not set'}
            </Text>
          </View>
        </View> */}
        {/* Project */}
        {/* <View style={styles.fieldContainer}>
          <MaterialIcons name="work" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>PROJECT</Text>
            <Text style={styles.value}>
              {profileData.projects && profileData.projects.length > 0 
                ? profileData.projects.join(', ') 
                : 'Not set'}
            </Text>
          </View>
        </View> */}
        {/* Name */}
        <View style={styles.fieldContainer}>
          <MaterialIcons name="person" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>NAME</Text>
            <Text style={styles.value}>{profileData.name || 'Not set'}</Text>
          </View>
        </View>
        {/* Role */}
        <View style={styles.fieldContainer}>
          <MaterialIcons name="badge" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>ROLE</Text>
            <Text style={styles.value}>{profileData.role || 'Not set'}</Text>
          </View>
        </View>
        {/* Email */}
        <View style={styles.fieldContainer}>
          <MaterialIcons name="email" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>EMAIL</Text>
            <Text style={styles.value}>{profileData.email || 'Not set'}</Text>
          </View>
        </View>
        {/* Employee ID */}
        <View style={styles.fieldContainer}>
          <MaterialIcons name="assignment-ind" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>EMPLOYEE ID</Text>
            <Text style={styles.value}>{profileData.employeeNumber || 'Not set'}</Text>
          </View>
        </View>
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f6f7f9',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 16,
    paddingTop: Platform.OS === 'ios' ? 32 : 16,
    height: '100%',
    top: 7,
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    color: 'white',
    fontSize: 20,
    fontWeight: 'bold',
    flex: 1,
    textAlign: 'center',
  },
  content: {
    flex: 1,
    padding: 20,
  },
  fieldContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 22,
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 12,
    shadowColor: '#000',
    shadowOpacity: 0.04,
    shadowRadius: 4,
    shadowOffset: { width: 0, height: 2 },
    elevation: 2,
  },
  icon: {
    marginRight: 15,
  },
  fieldContent: {
    flex: 1,
    borderBottomWidth: 0,
    paddingBottom: 0,
  },
  label: {
    fontSize: 12,
    color: '#888',
    marginBottom: 5,
    fontWeight: 'bold',
    letterSpacing: 0.5,
  },
  value: {
    fontSize: 16,
    color: '#29486A',
    fontWeight: '500',
  },
  loadingModal: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  loadingContent: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    alignItems: 'center',
  },
  loadingText: {
    marginTop: 10,
    fontSize: 16,
    color: '#333',
  },
});

export default ViewProfilepage;