import React, { useEffect, useState, useRef, useCallback } from 'react';
import { View, Text, TouchableOpacity, StyleSheet, Platform, PermissionsAndroid, Modal, TextInput, BackHandler } from 'react-native';
import { useFocusEffect, useNavigation } from '@react-navigation/native';
import VoiceToText, { VoiceToTextEvents } from '@appcitor/react-native-voice-to-text';
import LottieView from 'lottie-react-native';

/*
  SpeechToText component using @appcitor/react-native-voice-to-text

  Installation notes:
  - yarn add @appcitor/react-native-voice-to-text
  - or: npm install @appcitor/react-native-voice-to-text --save
  - iOS: cd ios && pod install
  - Android: Ensure <uses-permission android:name="android.permission.RECORD_AUDIO"/> in AndroidManifest.xml
  - Test on a real device to avoid simulator-related audio issues
*/

const SpeechToText = () => {
  const navigation = useNavigation();
  const [isListening, setIsListening] = useState(false);
  const [transcript, setTranscript] = useState('');
  const [error, setError] = useState(null);
  const [micPermissionGranted, setMicPermissionGranted] = useState(false);
  const [isVoiceModuleAvailable, setIsVoiceModuleAvailable] = useState(false);
  const [isRecognitionAvailable, setIsRecognitionAvailable] = useState(true);
  const [modalVisible, setModalVisible] = useState(false);
  const [partialText, setPartialText] = useState('');
  const [volume, setVolume] = useState(0);
  const [ignoreVoiceEvents, setIgnoreVoiceEvents] = useState(false);
  const lottieRef = useRef(null);
  const isMountedRef = useRef(true);
  const listenersRef = useRef([]);
  const isCleaningUpRef = useRef(false);
  const isListeningRef = useRef(false);
  const cleanupTimeoutRef = useRef(null);
  const hasStoppedRecordingRef = useRef(false);
  const hasReceivedFinalResultsRef = useRef(false); // Track if we got final results to prevent duplicates
  const recordingStartTimeRef = useRef(null);
  const isRestartingRef = useRef(false); // Track if we're currently restarting
  const lastFinalTextRef = useRef(''); // Track last final text to prevent duplicates

  useEffect(() => {
    // Check if chosen native module is available
    if (!VoiceToText || typeof VoiceToText.addEventListener !== 'function') {
      const msg = 'VoiceToText module is unavailable. Ensure @appcitor/react-native-voice-to-text is installed and linked correctly.';
      console.error(msg);
      setError(msg);
      return;
    }

    setIsVoiceModuleAvailable(true);

    // attach listeners with safe state updates
    const resultsListener = VoiceToText.addEventListener(VoiceToTextEvents.RESULTS, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEvents) {
        onSpeechResults(event);
      }
    });
    const partialListener = VoiceToText.addEventListener(VoiceToTextEvents.PARTIAL_RESULTS, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEvents) {
        // Only update partial text if we're still listening
        if (isListeningRef.current) {
          setPartialText(event?.value || '');
        }
      }
    });
    const volumeListener = VoiceToText.addEventListener(VoiceToTextEvents.VOLUME_CHANGED, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEvents) {
        setVolume(event?.value || 0);
      }
    });
    const errorListener = VoiceToText.addEventListener(VoiceToTextEvents.ERROR, (event) => {
      if (isMountedRef.current && !isCleaningUpRef.current && !ignoreVoiceEvents) {
        onSpeechError(event);
      }
    });
    const startListener = VoiceToText.addEventListener(VoiceToTextEvents.START, () => {
      if (isMountedRef.current && !isCleaningUpRef.current) {
        console.log('🎤 Recording started event received');
        isListeningRef.current = true;
        hasStoppedRecordingRef.current = false;
        hasReceivedFinalResultsRef.current = false;
        isRestartingRef.current = false; // Reset restart flag
        recordingStartTimeRef.current = Date.now();
        setIsListening(true);
        // Always ensure modal is visible when recording starts
        setModalVisible(true);
        setPartialText('');
        setIgnoreVoiceEvents(false);
        // Start Lottie animation immediately
        if (lottieRef.current) {
          lottieRef.current.play();
        }
        console.log('✅ Recording state updated, animation started');
      }
    });
    const endListener = VoiceToText.addEventListener(VoiceToTextEvents.END, () => {
      // Handle END event - some libraries auto-stop after timeout (3-5 seconds of silence)
      // For continuous recording, we need to restart automatically
      console.log('⚠️ END event received - recording session ended');
      
      if (isMountedRef.current && !isCleaningUpRef.current && !hasStoppedRecordingRef.current && isListeningRef.current && !isRestartingRef.current) {
        // Recording ended due to timeout/pause, but user wants continuous mode
        // Auto-restart after a brief delay to maintain continuous recording
        console.log('🔄 Auto-restarting recording for continuous mode...');
        isRestartingRef.current = true;
        
        // Small delay before restarting to allow native module to clean up
        setTimeout(async () => {
          if (isMountedRef.current && !isCleaningUpRef.current && !hasStoppedRecordingRef.current && isListeningRef.current) {
            try {
              console.log('🔄 Attempting to restart recording...');
              if (VoiceToText && typeof VoiceToText.startListening === 'function') {
                await VoiceToText.startListening();
                console.log('✅ Recording restarted automatically');
                isRestartingRef.current = false;
              } else if (VoiceToText && typeof VoiceToText.start === 'function') {
                await VoiceToText.start();
                console.log('✅ Recording restarted automatically');
                isRestartingRef.current = false;
              }
            } catch (restartError) {
              console.warn('⚠️ Failed to auto-restart recording:', restartError);
              isRestartingRef.current = false;
              // If restart fails, don't stop - just log it and let user control
            }
          } else {
            isRestartingRef.current = false;
          }
        }, 200);
      } else {
        console.log('ℹ️ END event ignored (user stopped or cleaning up)');
      }
    });

    // Store listeners for cleanup
    listenersRef.current = [
      resultsListener,
      partialListener,
      volumeListener,
      errorListener,
      startListener,
      endListener
    ];

    // Log for debugging
    console.log('VoiceToText module initialized:', VoiceToText);

    // On iOS, check speech recognition availability
    (async () => {
      try {
        if (typeof VoiceToText.isRecognitionAvailable === 'function') {
          const avail = await VoiceToText.isRecognitionAvailable();
          setIsRecognitionAvailable(avail === true);
          if (!avail && Platform.OS === 'ios') {
            setError('Speech recognition not available on this device. Test on a real device and ensure permissions are set in Info.plist.');
          }
        }
      } catch (err) {
        console.warn('isRecognitionAvailable check failed', err);
      }
    })();

    // Check/request microphone permission on Android
    const checkPermissions = async () => {
      if (Platform.OS === 'android') {
        try {
          const granted = await checkMicrophonePermission();
          if (!granted) {
            const requested = await requestMicrophonePermission();
            if (!requested) {
              setError('Microphone permission denied. Please enable it in app settings.');
            }
          }
        } catch (err) {
          console.warn('Permission check error:', err);
          setError('Failed to check microphone permissions.');
        }
      }
    };

    checkPermissions();

    // Cleanup on unmount - DO NOT remove listeners to avoid native crash
    // The native module's removeListeners has a bug causing ConcurrentModificationException
    // Instead, we just stop recording and let React Native handle cleanup
    return () => {
      console.log('🧹 Component unmounting, stopping recording...');
      isMountedRef.current = false;
      isCleaningUpRef.current = true;
      
      // Clear any pending timeouts
      if (cleanupTimeoutRef.current) {
        clearTimeout(cleanupTimeoutRef.current);
      }
      
      // Stop Lottie animation
      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }
      
      // Stop recording immediately without waiting
      const stopRecordingImmediately = async () => {
        try {
          if (isListeningRef.current && VoiceToText && !hasStoppedRecordingRef.current) {
            hasStoppedRecordingRef.current = true;
            console.log('🛑 Stopping recording on unmount...');
            
            // Try to stop recording using all available methods
            const stopMethods = [
              () => VoiceToText.stopListening && VoiceToText.stopListening(),
              () => VoiceToText.stop && VoiceToText.stop(),
              () => VoiceToText.cancelListening && VoiceToText.cancelListening(),
              () => VoiceToText.cancel && VoiceToText.cancel(),
              () => VoiceToText.destroy && VoiceToText.destroy(),
            ];
            
            for (const stopMethod of stopMethods) {
              try {
                if (typeof stopMethod === 'function') {
                  await stopMethod();
                  console.log('✅ Recording stopped successfully');
                  break;
                }
              } catch (e) {
                // Continue to next method
              }
            }
          }
        } catch (e) {
          // Ignore all errors during cleanup
          console.warn('⚠️ Error during cleanup (non-fatal):', e.message);
        }
      };
      
      // Stop recording but DON'T remove listeners
      // Removing listeners causes the native crash
      stopRecordingImmediately();
      
      // Clear listeners array reference but don't call remove()
      // This avoids triggering the buggy native removeListeners function
      listenersRef.current = [];
    };
  }, []);

  const onSpeechResults = (e) => {
    // Don't update state if component is unmounting or cleaning up
    if (!isMountedRef.current || isCleaningUpRef.current || ignoreVoiceEvents) {
      return;
    }

    // Mark that we received final results to prevent duplicate addition
    hasReceivedFinalResultsRef.current = true;

    // The event payload shape differs between libraries / platforms.
    // Handle arrays, strings and objects defensively to avoid "join is not a function".
    try {
      let payload = e?.value ?? e?.results ?? e?.text ?? e;
      let finalText = '';

      if (Array.isArray(payload)) {
        if (payload.length) {
          finalText = payload.join(' ');
        }
      } else if (typeof payload === 'string') {
        finalText = payload;
      } else if (payload && typeof payload === 'object') {
        // If payload is an object with numeric keys or nested results, try to extract strings
        const extracted = Object.values(payload)
          .flatMap((v) => (Array.isArray(v) ? v : [v]))
          .filter((v) => typeof v === 'string');
        if (extracted.length) {
          finalText = extracted.join(' ');
        } else {
          finalText = String(payload ?? '');
        }
      } else {
        finalText = String(payload ?? '');
      }

      // Append to existing transcript (don't replace, to support continuous recording)
      // But avoid duplicates by checking if the text is already in the transcript
      if (finalText && finalText.trim() && isMountedRef.current) {
        const trimmedFinal = finalText.trim();
        
        // Check if this is the same as the last final text we added
        if (lastFinalTextRef.current === trimmedFinal) {
          console.log('⚠️ Duplicate final text detected, skipping:', trimmedFinal);
          setPartialText('');
          return; // Skip duplicate
        }
        
        setTranscript((prev) => {
          // If no previous text, just set it
          if (!prev || !prev.trim()) {
            lastFinalTextRef.current = trimmedFinal;
            return trimmedFinal;
          }
          
          const prevTrimmed = prev.trim();
          
          // Check if this text is already at the end of the transcript (exact match)
          if (prevTrimmed.endsWith(trimmedFinal)) {
            console.log('⚠️ Text already at end, skipping duplicate');
            return prev; // Already added, don't duplicate
          }
          
          // Check if the last few words match (to catch partial duplicates)
          const lastFewWords = prevTrimmed.split(' ').slice(-3).join(' ');
          const firstFewWords = trimmedFinal.split(' ').slice(0, 3).join(' ');
          if (lastFewWords === firstFewWords && lastFewWords.length > 5) {
            console.log('⚠️ Overlapping text detected, skipping duplicate');
            return prev; // Overlapping, skip
          }
          
          // Check if it's a substring that's already included (for longer text)
          // Only skip if it's a significant portion (>15 chars) to avoid false positives
          if (trimmedFinal.length > 15 && prevTrimmed.includes(trimmedFinal)) {
            console.log('⚠️ Text already included, skipping duplicate');
            return prev; // Already included, skip
          }
          
          // Append new text with space and update last final text
          lastFinalTextRef.current = trimmedFinal;
          return prevTrimmed + ' ' + trimmedFinal;
        });
        
        // Clear partial text since we got final results
        setPartialText('');
      }
    } catch (err) {
      console.warn('onSpeechResults parse error', err, e);
    }
  };

  const onSpeechError = (e) => {
    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }
    const message = e?.error?.message || e?.message || JSON.stringify(e);
    setError(`Speech recognition error: ${message}`);
    isListeningRef.current = false;
    setIsListening(false);
    setModalVisible(false);
  };

  const checkMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.RECORD_AUDIO);
      setMicPermissionGranted(granted);
      return granted;
    } catch (err) {
      console.warn('Permission check error:', err);
      return false;
    }
  };

  const requestMicrophonePermission = async () => {
    if (Platform.OS !== 'android') return true;
    try {
      const granted = await PermissionsAndroid.request(
        PermissionsAndroid.PERMISSIONS.RECORD_AUDIO,
        {
          title: 'Microphone Permission',
          message: 'This app needs microphone access for speech recognition.',
          buttonPositive: 'OK',
        }
      );
      const isGranted = granted === PermissionsAndroid.RESULTS.GRANTED;
      setMicPermissionGranted(isGranted);
      return isGranted;
    } catch (err) {
      console.warn('Permission request error:', err);
      return false;
    }
  };

  const startRecording = async () => {
    if (!isVoiceModuleAvailable) {
      setError('Voice module is not available. Please check installation.');
      return;
    }

    setError(null);
    // Don't clear transcript - allow continuous recording
    // setTranscript(''); // Commented out to allow appending

    if (Platform.OS === 'android' && !micPermissionGranted) {
      const granted = await requestMicrophonePermission();
      if (!granted) {
        setError('Microphone permission denied.');
        return;
      }
    }

    try {
      // on iOS ensure recognition available
      if (Platform.OS === 'ios' && !isRecognitionAvailable) {
        setError('Speech recognition not available on this iOS device or permission denied. Check Info.plist and device settings.');
        return;
      }

      if (typeof VoiceToText.setRecognitionLanguage === 'function') {
        await VoiceToText.setRecognitionLanguage('en-US');
      }

      // Reset all flags when starting new recording
      hasStoppedRecordingRef.current = false;
      hasReceivedFinalResultsRef.current = false;
      isRestartingRef.current = false;
      isListeningRef.current = true;
      setIgnoreVoiceEvents(false);
      setPartialText('');
      lastFinalTextRef.current = ''; // Reset last final text
      recordingStartTimeRef.current = Date.now();
      
      // Show modal immediately and start animation
      setModalVisible(true);
      setIsListening(true);
      // Start Lottie animation immediately
      if (lottieRef.current) {
        lottieRef.current.play();
      }
      console.log('🎬 Animation started, modal visible');

      if (typeof VoiceToText.startListening === 'function') {
        await VoiceToText.startListening();
      } else if (typeof VoiceToText.start === 'function') {
        await VoiceToText.start();
      } else {
        throw new Error('startListening not available on VoiceToText');
      }
      
      console.log('✅ Recording started successfully');
    } catch (e) {
      const message = e?.message || JSON.stringify(e);
      if (Platform.OS === 'ios' && message.includes('IsFormatSampleRateAndChannelCountValid')) {
        setError('iOS audio format error. Test on a real device and ensure Info.plist includes NSMicrophoneUsageDescription and NSSpeechRecognitionUsageDescription.');
      } else {
        setError(`Failed to start recording: ${message}`);
      }
      setIsListening(false);
      setModalVisible(false);
      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }
    }
  };

  const stopRecording = async () => {
    await safeStopRecording();
  };

  const cancelRecording = async () => {
    if (!isVoiceModuleAvailable || !isMountedRef.current || isCleaningUpRef.current) {
      return;
    }
    try {
      // Cancel and don't add partial text
      await safeStopRecording(false);
    } catch (e) {
      if (isMountedRef.current && !isCleaningUpRef.current) {
        setError(`Failed to cancel recording: ${e.message || JSON.stringify(e)}`);
      }
    }
  };

  const onDone = async () => {
    if (!isMountedRef.current || isCleaningUpRef.current) {
      return;
    }
    try {
      // Stop recording and optionally add partial text if no final results received
      await safeStopRecording(true); // Pass true to add partial text if needed
    } catch (err) {
      console.warn('onDone stop error', err);
    }
  };


  // Safe stop recording function that can be called from anywhere
  const safeStopRecording = useCallback(async (shouldAddPartialText = false) => {
    if (hasStoppedRecordingRef.current) {
      console.log('⚠️ Already stopped, ignoring stop request');
      return;
    }
    
    if (!isListeningRef.current) {
      console.log('⚠️ Not listening, ignoring stop request');
      return;
    }
    
    try {
      hasStoppedRecordingRef.current = true;
      isListeningRef.current = false;
      setIgnoreVoiceEvents(true);
      
      console.log('🛑 Safe stop recording called...');
      
      if (VoiceToText && isVoiceModuleAvailable) {
        try {
          if (typeof VoiceToText.stopListening === 'function') {
            await VoiceToText.stopListening();
          } else if (typeof VoiceToText.stop === 'function') {
            await VoiceToText.stop();
          } else if (typeof VoiceToText.cancelListening === 'function') {
            await VoiceToText.cancelListening();
          } else if (typeof VoiceToText.cancel === 'function') {
            await VoiceToText.cancel();
          }
          console.log('✅ Recording stopped safely');
        } catch (stopError) {
          console.warn('⚠️ Error stopping recording:', stopError);
        }
      }
      
      // Stop Lottie animation immediately
      if (lottieRef.current) {
        lottieRef.current.pause();
        lottieRef.current.reset();
      }
      
      // Update state if still mounted
      if (isMountedRef.current && !isCleaningUpRef.current) {
        setIsListening(false);
        
        // Wait a moment to see if final results come in after stopping
        // This prevents adding partial text if final results are about to arrive
        setTimeout(() => {
          if (!isMountedRef.current || isCleaningUpRef.current) {
            return;
          }
          
          // Only add partial text if:
          // 1. User explicitly wants it (Done button)
          // 2. We haven't received final results yet (to avoid duplicates)
          // 3. Partial text is not empty
          if (shouldAddPartialText && !hasReceivedFinalResultsRef.current) {
            // Get current partial text and add it if not duplicate
            setPartialText((currentPartial) => {
              if (currentPartial && currentPartial.trim()) {
                setTranscript((prev) => {
                  // Only append if not already in transcript (avoid duplicates)
                  const trimmedPartial = currentPartial.trim();
                  // Check if the partial text is already in the transcript
                  if (prev && prev.trim().endsWith(trimmedPartial)) {
                    return prev; // Already added, don't duplicate
                  }
                  // Check if it's a substring that's already included
                  if (prev && prev.includes(trimmedPartial) && trimmedPartial.length > 5) {
                    return prev; // Likely already included, don't duplicate
                  }
                  return prev && prev.trim() ? prev + ' ' + trimmedPartial : trimmedPartial;
                });
              }
              return '';
            });
          } else {
            // Clear partial text without adding (final results already added or user cancelled)
            setPartialText('');
          }
          
          // Close modal after processing
          if (isMountedRef.current) {
            setModalVisible(false);
          }
        }, 500); // Wait 500ms for any pending final results
      }
    } catch (e) {
      console.warn('⚠️ Error in safeStopRecording:', e);
    }
  }, [isVoiceModuleAvailable]);

  // Handle screen focus/blur - stop recording when screen loses focus
  useFocusEffect(
    useCallback(() => {
      // Screen focused - reset stopped flag
      hasStoppedRecordingRef.current = false;
      isCleaningUpRef.current = false;
      
      return () => {
        // Screen blurred/unfocused - stop recording immediately
        console.log('📱 Screen losing focus, stopping recording...');
        safeStopRecording();
      };
    }, [safeStopRecording])
  );

  // Handle Android back button
  useEffect(() => {
    const backHandler = BackHandler.addEventListener('hardwareBackPress', () => {
      if (isListeningRef.current) {
        console.log('🔙 Back button pressed during recording, stopping...');
        safeStopRecording();
        return true; // Prevent default back action
      }
      return false; // Allow default back action
    });

    return () => backHandler.remove();
  }, [safeStopRecording]);

  // Handle navigation events to stop recording before leaving screen
  useEffect(() => {
    const unsubscribe = navigation.addListener('beforeRemove', (e) => {
      if (isListeningRef.current) {
        console.log('🚪 Navigation away detected, stopping recording...');
        safeStopRecording();
      }
    });

    return unsubscribe;
  }, [navigation, safeStopRecording]);

  // Handle Lottie animation playback based on modal visibility and recording state
  useEffect(() => {
    if (modalVisible && isListening && lottieRef.current) {
      // Small delay to ensure the modal and Lottie view are rendered
      const timer = setTimeout(() => {
        if (lottieRef.current) {
          lottieRef.current.play();
        }
      }, 100);
      return () => clearTimeout(timer);
    } else if (!isListening && lottieRef.current) {
      // Stop animation when not listening
      lottieRef.current.pause();
      lottieRef.current.reset();
    }
  }, [modalVisible, isListening]);

  return (
    <View style={styles.container}>
      <Text style={styles.title}>Speech to Text</Text>
      <Text style={styles.status}>Status: {isListening ? 'Listening...' : 'Stopped'}</Text>
      <Text style={styles.transcript}>{transcript || 'Press Start and speak'}</Text>
      {error && <Text style={styles.error}>Error: {error}</Text>}

      <View style={styles.inputRow}>
        <TextInput
          style={styles.input}
          value={transcript}
          onChangeText={(t) => setTranscript(t)}
          placeholder="Transcribed text will appear here"
          multiline
        />
        <TouchableOpacity
          style={[styles.button, isListening && styles.disabledButton, { width: 90, marginLeft: 8 }]}
          onPress={startRecording}
          disabled={isListening || !isVoiceModuleAvailable}
        >
          <Text style={styles.buttonText}>Start</Text>
        </TouchableOpacity>
      </View>
      <Modal 
        visible={modalVisible} 
        transparent 
        animationType="fade"
        onRequestClose={() => {
          // Handle Android back button - stop recording instead of closing modal
          if (isListening) {
            cancelRecording();
          }
        }}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            {/* Lottie microphone animation */}
            <View style={styles.lottieContainer}>
              <LottieView
                ref={lottieRef}
                source={require('../../../Images/micanimation.json')}
                autoPlay={false}
                loop={true}
                style={styles.lottieAnimation}
                resizeMode="contain"
              />
            </View>

            {/* Status text showing partial results or listening status */}
            <Text style={styles.partial}>
              {partialText ? (
                partialText
              ) : (
                isListening ? 'Listening... Speak now' : 'Preparing...'
              )}
            </Text>
            
            {/* Volume indicator (optional) */}
            {volume > 0 && (
              <View style={styles.volumeIndicator}>
                <View style={[styles.volumeBar, { width: `${Math.min(volume * 100, 100)}%` }]} />
              </View>
            )}

            {/* Action buttons */}
            <View style={styles.modalButtons}>
              <TouchableOpacity 
                style={[styles.modalButton, styles.cancelBtn]} 
                onPress={cancelRecording}
                activeOpacity={0.7}
              >
                <Text style={[styles.modalButtonText, { color: '#015185' }]}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalButton} 
                onPress={onDone}
                activeOpacity={0.7}
              >
                <Text style={styles.modalButtonText}>Done</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    padding: 16,
    backgroundColor: '#fff',
  },
  inputRow: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    marginTop: 12,
  },
  input: {
    flex: 1,
    minHeight: 100,
    padding: 12,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
    backgroundColor: '#fff',
    fontSize: 16,
    color: '#333',
    textAlignVertical: 'top',
  },
  title: {
    fontSize: 22,
    fontWeight: '700',
    marginBottom: 8,
    color: '#015185',
  },
  status: {
    marginBottom: 12,
    color: '#666',
    fontSize: 14,
  },
  transcript: {
    minHeight: 80,
    padding: 12,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginBottom: 12,
    fontSize: 15,
    color: '#333',
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  error: {
    color: '#d32f2f',
    marginBottom: 8,
    fontSize: 14,
    backgroundColor: '#ffebee',
    padding: 8,
    borderRadius: 4,
  },
  buttonRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  button: {
    paddingVertical: 12,
    paddingHorizontal: 24,
    backgroundColor: '#015185',
    borderRadius: 8,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  disabledButton: {
    backgroundColor: '#9e9e9e',
  },
  buttonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 16,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.7)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '85%',
    maxWidth: 400,
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
    elevation: 8,
  },
  lottieContainer: {
    width: 200,
    height: 200,
    justifyContent: 'center',
    alignItems: 'center',
    marginBottom: 20,
  },
  lottieAnimation: {
    width: '100%',
    height: '100%',
  },
  partial: {
    minHeight: 50,
    textAlign: 'center',
    marginBottom: 20,
    fontSize: 16,
    color: '#333',
    fontWeight: '500',
    paddingHorizontal: 16,
  },
  modalButtons: {
    flexDirection: 'row',
    width: '100%',
    justifyContent: 'space-between',
    gap: 12,
  },
  modalButton: {
    flex: 1,
    paddingVertical: 12,
    backgroundColor: '#015185',
    borderRadius: 8,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  cancelBtn: {
    backgroundColor: '#fff',
    borderWidth: 2,
    borderColor: '#015185',
  },
  modalButtonText: {
    color: '#fff',
    fontWeight: '600',
    fontSize: 16,
  },
  volumeIndicator: {
    width: '80%',
    height: 4,
    backgroundColor: '#e0e0e0',
    borderRadius: 2,
    marginBottom: 16,
    overflow: 'hidden',
  },
  volumeBar: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 2,
  },
});

export default SpeechToText;