import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Alert,
  Dimensions,
  Modal,
  TextInput,
  Platform,
  Share,
  Linking,
  Animated,
  KeyboardAvoidingView,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import { Buffer } from 'buffer';
import HSEInspectionAPI from '../../../ConfigMongo/HSEInspectionMongoConfig';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_071step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [step1Data, setStep1Data] = useState(null);
  const [step2Data, setStep2Data] = useState(null);
  const [formData, setFormData] = useState({});
  const [isLoading, setIsLoading] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [tempDateValue, setTempDateValue] = useState(new Date());
  const [uploadProgress, setUploadProgress] = useState(0);
  const [reportId, setReportId] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  
  const isMounted = useRef(true);

  useEffect(() => {
    isMounted.current = true;
    loadData();
    return () => {
      isMounted.current = false;
    };
  }, []);

  const loadData = async () => {
    try {
      setIsLoading(true);
      
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      if (savedStep1) {
        setStep1Data(JSON.parse(savedStep1));
      }

      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      if (savedStep2) {
        setStep2Data(JSON.parse(savedStep2));
        setFormData(JSON.parse(savedStep2).formData || {});
      }
    } catch (error) {
      console.error('Error loading data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load data.', position: 'top' });
    } finally {
      setIsLoading(false);
    }
  };

  const handleBackPress = () => {
    setActiveStep(2);
  };

  const formatDataForPreview = () => {
    // preview removed
  };

  const handleIOSDatePickerDone = async () => {
    try {
      if (tempDateValue && tempDateValue instanceof Date && !isNaN(tempDateValue.getTime())) {
        setSelectedDate(tempDateValue);
        const formattedDate = tempDateValue.toLocaleDateString('en-GB');
        handleInputChange('Date', formattedDate);
      }
    } catch (error) {
      console.error('Error in handleIOSDatePickerDone:', error);
    } finally {
      setShowDatePicker(false);
    }
  };

  const handleIOSDatePickerCancel = () => {
    setShowDatePicker(false);
  };

  const handleInputChange = async (field, value) => {
    const newFormData = { ...formData, [field]: value };
    setFormData(newFormData);
    
    try {
      const step2Data = await AsyncStorage.getItem('hseStep2');
      if (step2Data) {
        const updatedStep2 = JSON.parse(step2Data);
        updatedStep2.formData = newFormData;
        await AsyncStorage.setItem('hseStep2', JSON.stringify(updatedStep2));
      }
    } catch (error) {
      console.error('Error saving form data:', error);
    }
  };

  const handleCasingChange = async (casingType, value) => {
    const newFormData = {
      ...formData,
      Casing: {
        ...formData.Casing,
        [casingType]: value
      }
    };
    setFormData(newFormData);
    
    try {
      const step2Data = await AsyncStorage.getItem('hseStep2');
      if (step2Data) {
        const updatedStep2 = JSON.parse(step2Data);
        updatedStep2.formData = newFormData;
        await AsyncStorage.setItem('hseStep2', JSON.stringify(updatedStep2));
      }
    } catch (error) {
      console.error('Error saving casing data:', error);
    }
  };

  const showDatePickerModal = () => {
    setTempDateValue(selectedDate instanceof Date && !isNaN(selectedDate.getTime()) ? selectedDate : new Date());
    setShowDatePicker(true);
  };

  const handleDateChange = (event, selectedDate) => {
    if (Platform.OS === 'android') {
      setShowDatePicker(false);
      if (selectedDate) {
        setSelectedDate(selectedDate);
        const formattedDate = selectedDate.toLocaleDateString('en-GB');
        handleInputChange('Date', formattedDate);
      }
    } else {
      // iOS: update temp value while picker is open
      if (selectedDate && selectedDate instanceof Date && !isNaN(selectedDate.getTime())) {
        setTempDateValue(selectedDate);
      }
    }
  };

  const validateForm = () => {
    const requiredFields = [
      'Supervisor', 'Operator', 'Date', 'Shift', 'Rig', 'Vehicle ID', 
      'Ori Tool Serial ID', 'Controller Serial ID', 'SS Survey Dip', 'Bit ID',
      'Hole ID', 'Hole Depth', 'Shift Meters', 'Bit Meters', 'Azi',
      'Engine Hours', 'Month Safety Topic', 'Daily Safety Topic'
    ];
    
    const missingFields = requiredFields.filter(field => !formData[field]);
    
    if (missingFields.length > 0) {
      Toast.show({
        type: 'error',
        text1: 'Required Fields Missing',
        text2: `Please fill: ${missingFields.join(', ')}`,
        position: 'top'
      });
      return false;
    }
    return true;
  };

  const getAdminUserId = async () => {
    const userId = await AsyncStorage.getItem('userUid');
    if (!userId) throw new Error('User ID not found. Please log in again.');
    return userId;
  };

  // MongoDB helper functions for report numbering
  const getCurrentReportNumber = async () => {
    try {
      const response = await HSEInspectionAPI.getCurrentCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || response.data.nextNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error getting current report number from MongoDB:', error);
      return '001';
    }
  };

  const incrementReportCounter = async () => {
    try {
      const response = await HSEInspectionAPI.incrementCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error incrementing report counter in MongoDB:', error);
      return '001';
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};

      const checklistData = step1.selectedChecklist || {};
      const formData = step2.formData || {};
      const incomingShiftRequirements = formData['Incoming Shift Requirements'] || {};
      const equipmentChecklist = formData['Equipment Checklist'] || {};
      const combinedInputs = step2.combinedInputs || [];

      // Generate incoming shift requirements rows
      let incomingShiftRequirementsRows = '';
      const shiftRequirementItems = [
        'Water Swivel', 'Whipchecks', 'Hoist Plug', 'NQ Head Assembly', 
        'HQ Head Assembly', 'PQ Head Assembly', 'Rig & Site Presentation', 
        'Barricade Straight & Level', 'Gyro #', 'Gyro Controller #', 'Gyro Charger'
      ];
      
      shiftRequirementItems.forEach((item, index) => {
        const value = incomingShiftRequirements[item] || '';
        incomingShiftRequirementsRows += `
          <tr>
            <td>${index + 1}</td>
            <td>${item}</td>
            <td>${value}</td>
          </tr>
        `;
      });

      // Generate equipment checklist rows
      let equipmentChecklistRows = '';
      const equipmentItems = [
        'Fire Extinguisher x3 & Signage', 'Emergency Assembly Sign', 'Reverse Parking Sign',
        'Safety Meeting Register', 'Pre Start Check List', 'SWP File', 'Emergency Response Procedure',
        'MSDS File', 'Stretcher', 'Spill Kit', '1st Aid Box & Signage', 'Tape Measure',
        'Safety Harness x2', 'Admin Pen', 'Marker Pen', '50ltr Red Bin', '50ltr Yellow Bin',
        '50ltr Green Bin', 'Rig Tool Box Presentation', '36" Wrench x2', '24" Wrench x1',
        'Innertube Spanner x2', 'Outertube Spanner x1', 'Overshot Safety Chain', 'Grease Pump',
        'Spirit Level', 'Rod Stand', 'Rod Storage Stands', 'Core Barrel Stand', 'Rod Handlers',
        'Strap Rod Handler', 'Innertube Work Stand', 'Core Pump Out Stand', 'Ori Marker',
        'Core Breaking Chisel', 'Hammer', 'Sump Entrance Sign', 'Life Jacket', 'Chemical Storage',
        'Bypass Canon & Whipchecks', 'Mixing Jug x2', 'Marsh Funnel', 'Mud Mix Ratio',
        'Viscosity', 'Water Return', 'Blue Wash Buckets x4', 'Site Water Storage', 'Rake',
        'Shovel', 'Axe', 'Pick', 'Rig Diesel', 'Petrol Consumption', 'Core Lifter Consumption',
        'PQ Scrap Rod', 'HQ Scrap Rod', 'NQ Scrap Rod', 'PQ Good Rod', 'HQ Good Rod', 'NQ Good Rod'
      ];

      equipmentItems.forEach((item, index) => {
        const selection = equipmentChecklist[item] || '';
        equipmentChecklistRows += `
          <tr>
            <td>${index + 1}</td>
            <td>${item}</td>
            <td style="text-align: center;">${selection === 'Yes' ? '✓' : ''}</td>
            <td style="text-align: center;">${selection === 'No' ? '✓' : ''}</td>
          </tr>
        `;
      });

      // Generate combined inputs rows
      let combinedInputsRows = '';
      if (combinedInputs && combinedInputs.length > 0) {
        combinedInputs.forEach((input, index) => {
          combinedInputsRows += `
            <tr>
              <td>${index + 1}</td>
              <td>${input.incomingShiftRequirement || ''}</td>
              <td>${input.shiftConcern || ''}</td>
            </tr>
          `;
        });
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>DD Supervisor Shift Report</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              font-size: 13px;
              margin: 20px;
            }
            .header {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              border-bottom: 2px solid #000;
              padding-bottom: 10px;
            }
            .logo {
              display: flex;
              align-items: center;
              gap: 10px;
            }
            .logo img {
              height: 80px;
            }
            .doc-info {
              font-size: 12px;
              line-height: 1.4;
              border: 1px solid #000;
              padding: 5px 10px;
            }
            h2 {
              text-align: center;
              color: red;
              margin: 10px 0;
              text-decoration: underline;
            }
            table {
              border-collapse: collapse;
              width: 100%;
              margin-bottom: 20px;
            }
            table, th, td {
              border: 1px solid #000;
            }
            th, td {
              padding: 4px 6px;
              vertical-align: top;
            }
            th {
              background-color: #f2f2f2;
              text-align: left;
            }
            .no-border td {
              border: none;
            }
            .section-title {
              background: #e6e6e6;
              font-weight: bold;
            }
          </style>
        </head>
        <body>
          <div class="header">
            <div class="logo">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo">
              <div>
                <strong>TITAN DRILLING</strong>
              </div>
            </div>
            <div class="doc-info">
              <div><strong>Document Name:</strong> ${checklistData.document_name || 'HSE-FOR-8.1.1 DD Supervisor Shift Report'}</div>
              <div><strong>Document ID:</strong> ${checklistData.document_id || 'HSE-071'}</div>
              <div><strong>Version Date:</strong> ${checklistData.version_date || '01/07/2021'}</div>
              <div><strong>Version:</strong> ${checklistData.version || '1'}</div>
              <div><strong>Approved:</strong> ${checklistData.approved_by || 'CEO'}</div>
            </div>
          </div>

          <h2>DD SUPERVISOR SHIFT REPORT</h2>

          <table>
            <tr>
              <td><strong>Supervisor -</strong> ${formData.Supervisor || ''}</td>
              <td><strong>Hole ID -</strong> ${formData['Hole ID'] || ''}</td>
              <td><strong>Date -</strong> ${formData.Date || ''}</td>
              <td><strong>Shift -</strong> ${formData.Shift || 'Day / Night'}</td>
            </tr>
            <tr>
              <td><strong>Operator -</strong> ${formData.Operator || ''}</td>
              <td><strong>Hole Depth -</strong> ${formData['Hole Depth'] || ''}</td>
              <td><strong>Rig -</strong> ${formData.Rig || ''}</td>
              <td></td>
            </tr>
            <tr>
              <td><strong>Geologist -</strong> ${formData.Geologist || ''}</td>
              <td><strong>Shift Meters -</strong> ${formData['Shift Meters'] || ''}</td>
              <td><strong>Engine Hours -</strong> ${formData['Engine Hours'] || ''}</td>
              <td></td>
            </tr>
            <tr>
              <td><strong>Vehicle ID -</strong> ${formData['Vehicle ID'] || ''}</td>
              <td><strong>Bit Meters -</strong> ${formData['Bit Meters'] || ''}</td>
              <td><strong>Casing</strong> PW - ${formData.Casing?.PW || ''} PQ - ${formData.Casing?.PQ || ''} HQ - ${formData.Casing?.HQ || ''}</td>
              <td></td>
            </tr>
            <tr>
              <td><strong>Ori Tool Serial ID -</strong> ${formData['Ori Tool Serial ID'] || ''}</td>
              <td><strong>Bit ID -</strong> ${formData['Bit ID'] || ''}</td>
              <td></td>
              <td></td>
            </tr>
            <tr>
              <td><strong>Ori Tool Serial ID -</strong> ${formData['Ori Tool Serial ID'] || ''}</td>
              <td><strong>Month Safety Topic -</strong> ${formData['Month Safety Topic'] || ''}</td>
              <td></td>
              <td></td>
            </tr>
            <tr>
              <td><strong>Controller Serial ID -</strong> ${formData['Controller Serial ID'] || ''}</td>
              <td><strong>Daily Safety Topic -</strong> ${formData['Daily Safety Topic'] || ''}</td>
              <td></td>
              <td></td>
            </tr>
            <tr>
              <td><strong>SS Survey Dip -</strong> ${formData['SS Survey Dip'] || ''}</td>
              <td><strong>Azi -</strong> ${formData.Azi || ''}</td>
              <td><strong>P.T.O -</strong> ${formData['P.T.O'] || ''}</td>
              <td></td>
            </tr>
          </table>

          <table>
            <tr class="section-title">
              <td colspan="2"><strong>Incoming Shift Requirements</strong></td>
            </tr>
            <tr>
              <th style="width: 70%;">Equipment/Item</th>
              <th style="width: 30%; text-align: center;">Value</th>
            </tr>
            <tr>
              <td>Water Swivel</td>
              <td style="text-align: center;">${incomingShiftRequirements['Water Swivel'] || ''}</td>
            </tr>
            <tr>
              <td>Whipchecks</td>
              <td style="text-align: center;">${incomingShiftRequirements['Whipchecks'] || ''}</td>
            </tr>
            <tr>
              <td>Hoist Plug</td>
              <td style="text-align: center;">${incomingShiftRequirements['Hoist Plug'] || ''}</td>
            </tr>
            <tr>
              <td>NQ Head Assembly</td>
              <td style="text-align: center;">${incomingShiftRequirements['NQ Head Assembly'] || ''}</td>
            </tr>
            <tr>
              <td>HQ Head Assembly</td>
              <td style="text-align: center;">${incomingShiftRequirements['HQ Head Assembly'] || ''}</td>
            </tr>
            <tr>
              <td>PQ Head Assembly</td>
              <td style="text-align: center;">${incomingShiftRequirements['PQ Head Assembly'] || ''}</td>
            </tr>
            <tr>
              <td>Rig & Site Presentation</td>
              <td style="text-align: center;">${incomingShiftRequirements['Rig & Site Presentation'] || ''}</td>
            </tr>
            <tr>
              <td>Barricade Straight & Level</td>
              <td style="text-align: center;">${incomingShiftRequirements['Barricade Straight & Level'] || ''}</td>
            </tr>
            <tr>
              <td>Gyro #</td>
              <td style="text-align: center;">${incomingShiftRequirements['Gyro #'] || ''}</td>
            </tr>
            <tr>
              <td>Gyro Controller #</td>
              <td style="text-align: center;">${incomingShiftRequirements['Gyro Controller #'] || ''}</td>
            </tr>
            <tr>
              <td>Gyro Charger</td>
              <td style="text-align: center;">${incomingShiftRequirements['Gyro Charger'] || ''}</td>
            </tr>
          </table>

          <table>
            <tr class="section-title">
              <td colspan="2"><strong>Incoming Shift Requirements & Shift Concerns</strong></td>
            </tr>
            <tr>
              <th style="width: 50%;">Incoming Shift Requirement</th>
              <th style="width: 50%;">Shift Concern</th>
            </tr>
            ${(() => {
              const combinedInputs = step2.combinedInputs || [];
              let rows = '';
              combinedInputs.forEach((input, index) => {
                rows += `
                  <tr>
                    <td>${input.incomingShiftRequirement || ''}</td>
                    <td>${input.shiftConcern || ''}</td>
                  </tr>
                `;
              });
              // Add empty rows if no data to maintain table structure
              if (combinedInputs.length === 0) {
                rows = `
                  <tr>
                    <td></td>
                    <td></td>
                  </tr>
                  <tr>
                    <td></td>
                    <td></td>
                  </tr>
                  <tr>
                    <td></td>
                    <td></td>
                  </tr>
                `;
              }
              return rows;
            })()}
          </table>

          <table>
            <tr class="section-title">
              <td colspan="3"><strong>Equipment Checklist</strong></td>
            </tr>
            <tr>
              <th style="width: 60%;">Equipment/Item</th>
              <th style="width: 20%; text-align: center;">Yes</th>
              <th style="width: 20%; text-align: center;">No</th>
            </tr>
            <tr>
              <td>Fire Extinguisher x3 & Signage</td>
              <td style="text-align: center;">${equipmentChecklist['Fire Extinguisher x3 & Signage'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Fire Extinguisher x3 & Signage'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Emergency Assembly Sign</td>
              <td style="text-align: center;">${equipmentChecklist['Emergency Assembly Sign'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Emergency Assembly Sign'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Reverse Parking Sign</td>
              <td style="text-align: center;">${equipmentChecklist['Reverse Parking Sign'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Reverse Parking Sign'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Safety Meeting Register</td>
              <td style="text-align: center;">${equipmentChecklist['Safety Meeting Register'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Safety Meeting Register'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Pre Start Check List</td>
              <td style="text-align: center;">${equipmentChecklist['Pre Start Check List'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Pre Start Check List'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>SWP File</td>
              <td style="text-align: center;">${equipmentChecklist['SWP File'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['SWP File'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Emergency Response Procedure</td>
              <td style="text-align: center;">${equipmentChecklist['Emergency Response Procedure'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Emergency Response Procedure'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>MSDS File</td>
              <td style="text-align: center;">${equipmentChecklist['MSDS File'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['MSDS File'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Stretcher</td>
              <td style="text-align: center;">${equipmentChecklist['Stretcher'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Stretcher'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Spill Kit</td>
              <td style="text-align: center;">${equipmentChecklist['Spill Kit'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Spill Kit'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>1st Aid Box & Signage</td>
              <td style="text-align: center;">${equipmentChecklist['1st Aid Box & Signage'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['1st Aid Box & Signage'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Tape Measure</td>
              <td style="text-align: center;">${equipmentChecklist['Tape Measure'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Tape Measure'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Safety Harness x2</td>
              <td style="text-align: center;">${equipmentChecklist['Safety Harness x2'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Safety Harness x2'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Admin Pen</td>
              <td style="text-align: center;">${equipmentChecklist['Admin Pen'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Admin Pen'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Marker Pen</td>
              <td style="text-align: center;">${equipmentChecklist['Marker Pen'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Marker Pen'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>50ltr Red Bin</td>
              <td style="text-align: center;">${equipmentChecklist['50ltr Red Bin'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['50ltr Red Bin'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>50ltr Yellow Bin</td>
              <td style="text-align: center;">${equipmentChecklist['50ltr Yellow Bin'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['50ltr Yellow Bin'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>50ltr Green Bin</td>
              <td style="text-align: center;">${equipmentChecklist['50ltr Green Bin'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['50ltr Green Bin'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Rig Tool Box Presentation</td>
              <td style="text-align: center;">${equipmentChecklist['Rig Tool Box Presentation'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Rig Tool Box Presentation'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>36" Wrench x2</td>
              <td style="text-align: center;">${equipmentChecklist['36" Wrench x2'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['36" Wrench x2'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>24" Wrench x1</td>
              <td style="text-align: center;">${equipmentChecklist['24" Wrench x1'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['24" Wrench x1'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Innertube Spanner x2</td>
              <td style="text-align: center;">${equipmentChecklist['Innertube Spanner x2'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Innertube Spanner x2'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Outertube Spanner x1</td>
              <td style="text-align: center;">${equipmentChecklist['Outertube Spanner x1'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Outertube Spanner x1'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Overshot Safety Chain</td>
              <td style="text-align: center;">${equipmentChecklist['Overshot Safety Chain'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Overshot Safety Chain'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Grease Pump</td>
              <td style="text-align: center;">${equipmentChecklist['Grease Pump'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Grease Pump'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Spirit Level</td>
              <td style="text-align: center;">${equipmentChecklist['Spirit Level'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Spirit Level'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Rod Stand</td>
              <td style="text-align: center;">${equipmentChecklist['Rod Stand'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Rod Stand'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Rod Storage Stands</td>
              <td style="text-align: center;">${equipmentChecklist['Rod Storage Stands'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Rod Storage Stands'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Core Barrel Stand</td>
              <td style="text-align: center;">${equipmentChecklist['Core Barrel Stand'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Core Barrel Stand'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Rod Handlers</td>
              <td style="text-align: center;">${equipmentChecklist['Rod Handlers'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Rod Handlers'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Strap Rod Handler</td>
              <td style="text-align: center;">${equipmentChecklist['Strap Rod Handler'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Strap Rod Handler'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Innertube Work Stand</td>
              <td style="text-align: center;">${equipmentChecklist['Innertube Work Stand'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Innertube Work Stand'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Core Pump Out Stand</td>
              <td style="text-align: center;">${equipmentChecklist['Core Pump Out Stand'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Core Pump Out Stand'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Ori Marker</td>
              <td style="text-align: center;">${equipmentChecklist['Ori Marker'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Ori Marker'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Core Breaking Chisel</td>
              <td style="text-align: center;">${equipmentChecklist['Core Breaking Chisel'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Core Breaking Chisel'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Hammer</td>
              <td style="text-align: center;">${equipmentChecklist['Hammer'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Hammer'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Sump Entrance Sign</td>
              <td style="text-align: center;">${equipmentChecklist['Sump Entrance Sign'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Sump Entrance Sign'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Life Jacket</td>
              <td style="text-align: center;">${equipmentChecklist['Life Jacket'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Life Jacket'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Chemical Storage</td>
              <td style="text-align: center;">${equipmentChecklist['Chemical Storage'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Chemical Storage'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Bypass Canon & Whipchecks</td>
              <td style="text-align: center;">${equipmentChecklist['Bypass Canon & Whipchecks'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Bypass Canon & Whipchecks'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Mixing Jug x2</td>
              <td style="text-align: center;">${equipmentChecklist['Mixing Jug x2'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Mixing Jug x2'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Marsh Funnel</td>
              <td style="text-align: center;">${equipmentChecklist['Marsh Funnel'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Marsh Funnel'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Mud Mix Ratio</td>
              <td style="text-align: center;">${equipmentChecklist['Mud Mix Ratio'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Mud Mix Ratio'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Viscosity</td>
              <td style="text-align: center;">${equipmentChecklist['Viscosity'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Viscosity'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Water Return</td>
              <td style="text-align: center;">${equipmentChecklist['Water Return'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Water Return'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Blue Wash Buckets x4</td>
              <td style="text-align: center;">${equipmentChecklist['Blue Wash Buckets x4'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Blue Wash Buckets x4'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Site Water Storage</td>
              <td style="text-align: center;">${equipmentChecklist['Site Water Storage'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Site Water Storage'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Rake</td>
              <td style="text-align: center;">${equipmentChecklist['Rake'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Rake'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Shovel</td>
              <td style="text-align: center;">${equipmentChecklist['Shovel'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Shovel'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Axe</td>
              <td style="text-align: center;">${equipmentChecklist['Axe'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Axe'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Pick</td>
              <td style="text-align: center;">${equipmentChecklist['Pick'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Pick'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Rig Diesel</td>
              <td style="text-align: center;">${equipmentChecklist['Rig Diesel'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Rig Diesel'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Petrol Consumption</td>
              <td style="text-align: center;">${equipmentChecklist['Petrol Consumption'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Petrol Consumption'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>Core Lifter Consumption</td>
              <td style="text-align: center;">${equipmentChecklist['Core Lifter Consumption'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['Core Lifter Consumption'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>PQ Scrap Rod</td>
              <td style="text-align: center;">${equipmentChecklist['PQ Scrap Rod'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['PQ Scrap Rod'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>HQ Scrap Rod</td>
              <td style="text-align: center;">${equipmentChecklist['HQ Scrap Rod'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['HQ Scrap Rod'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>NQ Scrap Rod</td>
              <td style="text-align: center;">${equipmentChecklist['NQ Scrap Rod'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['NQ Scrap Rod'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>PQ Good Rod</td>
              <td style="text-align: center;">${equipmentChecklist['PQ Good Rod'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['PQ Good Rod'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>HQ Good Rod</td>
              <td style="text-align: center;">${equipmentChecklist['HQ Good Rod'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['HQ Good Rod'] === 'No' ? '✓' : ''}</td>
            </tr>
            <tr>
              <td>NQ Good Rod</td>
              <td style="text-align: center;">${equipmentChecklist['NQ Good Rod'] === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${equipmentChecklist['NQ Good Rod'] === 'No' ? '✓' : ''}</td>
            </tr>
          </table>

          <table>
            <tr class="section-title">
              <td colspan="4"><strong>Standard Procedures</strong></td>
            </tr>
            <tr>
              <td colspan="4">
                1. Grease head assembly each run<br>
                2. Grease chuck every 3rd run<br>
                3. Grease water swivel every 3rd run<br>
                4. Grease hoist plug before pulling rods<br>
                5. Note engine rpm during each run<br>
                6. Note downhole pressure each run<br>
                7. Note drill head rpm during each run<br>
                8. Note weight on bit during each run<br>
                9. Note pull back during each run<br>
                10. Note turnaround time at end of run<br>
                11. Note formation on each run<br>
                12. Note muds viscosity during each run<br>
                13. Note drill bit meters after each run<br>
                14. Note depth of core grinding<br>
                15. Compare core grinding to above notes
              </td>
            </tr>
          </table>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `HSE_071_Report_${report.id}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const uploadPDFToMongoDB = async (filePath, userId, onProgress) => {
    try {
      const pdfFile = {
        uri: `file://${filePath}`,
        type: 'application/pdf',
        name: `hse_071_${userId}_${Date.now()}.pdf`
      };
      
      const response = await HSEInspectionAPI.uploadPDF(userId, pdfFile);
      if (!response.success) throw new Error('Failed to upload PDF to MongoDB');
      
      if (onProgress) onProgress(100);
      return response.downloadUrl;
    } catch (error) {
      console.error('❌ Error uploading PDF to MongoDB:', error);
      throw error;
    }
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hseStep3');
      await AsyncStorage.removeItem(`hseNotes_${hseId}`);
      await AsyncStorage.removeItem(`hseResults_${hseId}`);
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const response = await HSEInspectionAPI.getHSEInspectionById(reportId);
      
      if (!response.success || !response.data) {
        Alert.alert('Error', 'Report not found in MongoDB.');
        return;
      }

      const pdfUrl = response.data.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      // Navigate to HSEViewPDF screen
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => {
          setShowSuccessModal(false);
          navigation.navigate('HSEInspectionHome');
        }
      });
    } catch (error) {
      console.error('❌ Error fetching PDF URL from MongoDB:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL from MongoDB. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const handleSubmit = async () => {
    if (!validateForm()) {
      return;
    }

    setSyncModalVisible(true);
    setUploadProgress(0);

    try {
      const userId = await getAdminUserId();
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      
      if (!savedStep1 || !savedStep2) {
        throw new Error('Previous step data (hseStep1 or hseStep2) is missing');
      }
      
      const step1 = JSON.parse(savedStep1);
      const step2 = JSON.parse(savedStep2);

      // Use the request number from step 1
      const requestId = step1.requestNumber;
      if (!requestId) {
        throw new Error('Request number not found in step 1 data');
      }
      setReportId(requestId);

      // Store only the essential filled data from PDF
      const pdfData = {
        // Basic report info
        report_id: requestId,
        request_number: requestId,
        
        // Only filled form data from step 3
        supervisor: formData.Supervisor || '',
        operator: formData.Operator || '',
        geologist: formData.Geologist || '',
        date: formData.Date || '',
        shift: formData.Shift || '',
        rig: formData.Rig || '',
        hole_id: formData['Hole ID'] || '',
        hole_depth: formData['Hole Depth'] || '',
        shift_meters: formData['Shift Meters'] || '',
        bit_meters: formData['Bit Meters'] || '',
        vehicle_id: formData['Vehicle ID'] || '',
        ori_tool_serial_id: formData['Ori Tool Serial ID'] || '',
        controller_serial_id: formData['Controller Serial ID'] || '',
        ss_survey_dip: formData['SS Survey Dip'] || '',
        bit_id: formData['Bit ID'] || '',
        engine_hours: formData['Engine Hours'] || '',
        azi: formData.Azi || '',
        pto: formData['P.T.O'] || '',
        month_safety_topic: formData['Month Safety Topic'] || '',
        daily_safety_topic: formData['Daily Safety Topic'] || '',
        casing_pw: formData.Casing?.PW || '',
        casing_pq: formData.Casing?.PQ || '',
        casing_hq: formData.Casing?.HQ || '',
      };

      // Generate PDF
      console.log('📄 Generating PDF for HSE-071...');
      const reportData = {
        id: requestId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        step1,
        step2,
        step3: formData,
        pdfData,
      };

      const pdfPath = await generatePDFWithHtml(reportData);
      console.log('✅ PDF generated at:', pdfPath);
      setUploadProgress(40);

      // Upload PDF to MongoDB
      console.log('📤 Uploading PDF to MongoDB...');
      const pdfDownloadUrl = await uploadPDFToMongoDB(pdfPath, userId, (progress) => {
        const pdfProgress = 40 + (progress / 3);
        if (isMounted.current) {
          setUploadProgress(pdfProgress);
        }
      });
      console.log('✅ PDF uploaded:', pdfDownloadUrl);
      setUploadProgress(70);

      // Save complete report to MongoDB
      console.log('💾 Saving report to MongoDB...');
      const completeReportData = {
        _id: requestId,
        userId,
        requestNumber: requestId,
        status: 'completed',
        step1,
        step2,
        step3: formData,
        pdfData,
        pdfDownloadUrl,
        createdAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId
      };

      const saveResponse = await HSEInspectionAPI.createHSEInspection(userId, completeReportData);
      if (!saveResponse.success) throw new Error('Failed to save report to MongoDB');
      console.log('✅ Report saved to MongoDB');
      setUploadProgress(90);

      // Send HSE inspection notifications
      console.log('📧 Sending HSE inspection notifications...');
      try {
        const userData = await AsyncStorage.getItem('userData');
        const user = userData ? JSON.parse(userData) : null;
        const createdByName = user?.name || 'Unknown Inspector';
        const inspectorEmail = user?.email || '';
        
        const notificationPayload = {
          hseInspectionData: {
            _id: requestId,
            id: requestId,
            requestNumber: requestId,
            checklistName: 'HSE-071 DD Supervisor Shift Report',
            country: step1.country || '',
            project: step1.project || '',
            status: 'completed',
          },
          createdByName,
          inspectorEmail,
          pdfUrl: pdfDownloadUrl
        };
        
        const notifResponse = await fetch('https://api.titandrillingzm.com:6028/api/hse-inspection-notifications/new-inspection', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(notificationPayload),
        });
        
        if (notifResponse.ok) {
          console.log('✅ HSE notifications sent successfully');
        }
      } catch (notifError) {
        console.error('❌ Error sending HSE notifications:', notifError);
      }

      // Delete temp PDF
      await RNFS.unlink(pdfPath).catch((err) => console.warn('⚠️ Error deleting temp PDF:', err));
      
      // Increment counter
      await incrementReportCounter();
      setUploadProgress(100);
      
      // Clear local data
      await flushStepData(requestId);

      if (isMounted.current) {
        setSyncModalVisible(false);
        setShowSuccessModal(true);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'DD Supervisor Shift Report submitted successfully.',
          position: 'top',
        });
      }
    } catch (error) {
      console.error('Error in handleSubmit:', error);
      if (isMounted.current) {
        setSyncModalVisible(false);
        setUploadProgress(0);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to submit report.',
          position: 'top',
        });
        Alert.alert('Submit Error', `Failed to submit report.\n\nError: ${error.message}`, [{ text: 'OK' }], { cancelable: false });
      }
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
      }
    }
  };

  const renderInputField = (label, value, onChangeText, placeholder = '', keyboardType = 'default', isRequired = false) => (
    <View style={styles.inputContainer}>
      <Text style={styles.inputLabel}>
        {label}
        {isRequired && <Text style={styles.requiredAsterisk}> *</Text>}
      </Text>
      <TextInput
        style={styles.textInput}
        value={value}
        onChangeText={onChangeText}
        placeholder={placeholder}
        keyboardType={keyboardType}
        placeholderTextColor="#999"
      />
    </View>
  );

  const renderCasingInputs = () => (
    <View style={styles.sectionContainer}>
      <Text style={styles.sectionTitle}>Casing</Text>
      <View style={styles.casingContainer}>
        <View style={styles.casingInputWrapper}>
          {renderInputField('PW', formData.Casing?.PW || '', (value) => handleCasingChange('PW', value))}
        </View>
        <View style={styles.casingInputWrapper}>
          {renderInputField('PQ', formData.Casing?.PQ || '', (value) => handleCasingChange('PQ', value))}
        </View>
        <View style={styles.casingInputWrapper}>
          {renderInputField('HQ', formData.Casing?.HQ || '', (value) => handleCasingChange('HQ', value))}
        </View>
      </View>
    </View>
  );

  if (isLoading) {
    return (
      <View style={styles.loadingContainer}>
        <Text style={styles.loadingText}>Loading form data...</Text>
      </View>
    );
  }

  return (
    <KeyboardAvoidingView style={styles.container} behavior={Platform.OS === 'ios' ? 'padding' : 'height'}>
      <ScrollView 
        style={styles.scrollView} 
        showsVerticalScrollIndicator={false}
        contentContainerStyle={styles.scrollContent}
      >
        <View style={styles.content}>
          <Text style={styles.headerTitle}>DD Supervisor Shift Report - Form Data</Text>
          

          
          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Personnel Information</Text>
            {renderInputField('Supervisor', formData.Supervisor, (value) => handleInputChange('Supervisor', value), 'Enter supervisor name', 'default', true)}
            {renderInputField('Operator', formData.Operator, (value) => handleInputChange('Operator', value), 'Enter operator name', 'default', true)}
            {renderInputField('Geologist', formData.Geologist, (value) => handleInputChange('Geologist', value), 'Enter geologist name', 'default', true)}
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Equipment Information</Text>
            {renderInputField('Vehicle ID', formData['Vehicle ID'], (value) => handleInputChange('Vehicle ID', value), 'Enter vehicle ID', 'default', true)}
            {renderInputField('Ori Tool Serial ID', formData['Ori Tool Serial ID'], (value) => handleInputChange('Ori Tool Serial ID', value), 'Enter Ori Tool Serial ID', 'default', true)}
            {renderInputField('Controller Serial ID', formData['Controller Serial ID'], (value) => handleInputChange('Controller Serial ID', value), 'Enter Controller Serial ID', 'default', true)}
            {renderInputField('SS Survey Dip', formData['SS Survey Dip'], (value) => handleInputChange('SS Survey Dip', value), 'Enter SS Survey Dip', 'default', true)}
            {renderInputField('Bit ID', formData['Bit ID'], (value) => handleInputChange('Bit ID', value), 'Enter Bit ID', 'default', true)}
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Hole Information</Text>
            {renderInputField('Hole ID', formData['Hole ID'], (value) => handleInputChange('Hole ID', value), 'Enter Hole ID', 'default', true)}
            {renderInputField('Hole Depth', formData['Hole Depth'], (value) => handleInputChange('Hole Depth', value), 'Enter hole depth', 'numeric', true)}
            {renderInputField('Shift Meters', formData['Shift Meters'], (value) => handleInputChange('Shift Meters', value), 'Enter shift meters', 'numeric', true)}
            {renderInputField('Bit Meters', formData['Bit Meters'], (value) => handleInputChange('Bit Meters', value), 'Enter bit meters', 'numeric', true)}
            {renderInputField('Azi', formData.Azi, (value) => handleInputChange('Azi', value), 'Enter Azi', 'numeric', true)}
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Safety Topics</Text>
            {renderInputField('Month Safety Topic', formData['Month Safety Topic'], (value) => handleInputChange('Month Safety Topic', value), 'Enter monthly safety topic', 'default', true)}
            {renderInputField('Daily Safety Topic', formData['Daily Safety Topic'], (value) => handleInputChange('Daily Safety Topic', value), 'Enter daily safety topic', 'default', true)}
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Date and Shift</Text>
            <View style={styles.dateContainer}>
              <Text style={styles.inputLabel}>Date</Text>
              <TouchableOpacity style={styles.datePickerButton} onPress={showDatePickerModal}>
                <Text style={styles.datePickerText}>{formData.Date || 'Select Date'}</Text>
                <MaterialIcons name="calendar-today" size={20} color="#0078D4" />
              </TouchableOpacity>
            </View>
            <View style={styles.shiftContainer}>
              <Text style={styles.inputLabel}>Shift</Text>
              <View style={styles.shiftButtons}>
                <TouchableOpacity
                  style={[styles.shiftButton, formData.Shift === 'Day' && styles.shiftButtonActive]}
                  onPress={() => handleInputChange('Shift', 'Day')}
                >
                  <Text style={[styles.shiftButtonText, formData.Shift === 'Day' && styles.shiftButtonTextActive]}>Day</Text>
                </TouchableOpacity>
                <TouchableOpacity
                  style={[styles.shiftButton, formData.Shift === 'Night' && styles.shiftButtonActive]}
                  onPress={() => handleInputChange('Shift', 'Night')}
                >
                  <Text style={[styles.shiftButtonText, formData.Shift === 'Night' && styles.shiftButtonTextActive]}>Night</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>

          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Rig and Additional Information</Text>
            {renderInputField('Rig', formData.Rig, (value) => handleInputChange('Rig', value), 'Enter rig information', 'default', true)}
            {renderInputField('Engine Hours', formData['Engine Hours'], (value) => handleInputChange('Engine Hours', value), 'Enter engine hours', 'numeric', true)}
            {renderInputField('P.T.O', formData['P.T.O'], (value) => handleInputChange('P.T.O', value), 'Enter P.T.O information', 'default', true)}
          </View>

          {renderCasingInputs()}
          
          {/* Request ID Display Section */}
          <View style={styles.sectionContainer}>
            <Text style={styles.sectionTitle}>Request Information</Text>
            <View style={styles.requestIdContainer}>
              <Text style={styles.requestIdLabel}>Request ID:</Text>
              <Text style={styles.requestIdValue}>
                {step1Data?.requestNumber || 'Not Generated Yet'}
              </Text>
            </View>
          </View>
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <View style={{ flex: 1 }} />
        <TouchableOpacity
          style={[styles.nextButton, syncModalVisible && styles.disabledButton]}
          onPress={handleSubmit}
          disabled={syncModalVisible}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {showDatePicker && Platform.OS === 'ios' ? (
        <Modal
          visible={showDatePicker}
          transparent
          animationType="slide"
          onRequestClose={handleIOSDatePickerCancel}
        >
          <View style={styles.iosPickerOverlay}>
            <View style={styles.iosPickerContainer}>
              <View style={styles.iosPickerToolbar}>
                <TouchableOpacity onPress={handleIOSDatePickerCancel} style={styles.iosPickerButton}>
                  <Text style={styles.iosPickerButtonText}>Cancel</Text>
                </TouchableOpacity>
                <TouchableOpacity onPress={handleIOSDatePickerDone} style={styles.iosPickerButton}>
                  <Text style={[styles.iosPickerButtonText, { fontWeight: '600' }]}>Done</Text>
                </TouchableOpacity>
              </View>
              <DateTimePicker
                value={tempDateValue || selectedDate}
                mode="date"
                display="spinner"
                onChange={handleDateChange}
                style={{ backgroundColor: '#fff' }}
              />
            </View>
          </View>
        </Modal>
      ) : (
        showDatePicker && (
          <DateTimePicker
            value={selectedDate}
            mode="date"
            display={Platform.OS === 'android' ? 'default' : 'calendar'}
            onChange={handleDateChange}
          />
        )
      )}

      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              DD Supervisor Shift Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Data preview removed */}
      
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message={`Generating Report`} />
      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 120,
  },
  content: {
    padding: 20,
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 20,
  },
  sectionContainer: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 16,
    marginBottom: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
    marginBottom: 16,
  },
  inputContainer: {
    marginBottom: 16,
  },
  inputLabel: {
    fontSize: 14,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  dateContainer: {
    marginBottom: 16,
  },
  datePickerButton: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    padding: 12,
    backgroundColor: '#fff',
  },
  datePickerText: {
    fontSize: 16,
    color: '#333',
  },
  shiftContainer: {
    marginBottom: 16,
  },
  shiftButtons: {
    flexDirection: 'row',
    marginTop: 8,
  },
  shiftButton: {
    flex: 1,
    padding: 12,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 8,
    marginHorizontal: 4,
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  shiftButtonActive: {
    backgroundColor: '#0078D4',
    borderColor: '#0078D4',
  },
  shiftButtonText: {
    fontSize: 16,
    color: '#666',
    fontWeight: '500',
  },
  shiftButtonTextActive: {
    color: '#fff',
  },
  casingContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    gap: 10,
  },
  casingInputWrapper: {
    flex: 1,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  previewButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  previewButtonText: {
    color: '#FF9800',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  dataPreviewModalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    width: '90%',
    maxWidth: 600,
    maxHeight: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  dataPreviewHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  dataPreviewTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#333',
  },
  closeButton: {
    padding: 4,
  },
  dataPreviewScrollView: {
    maxHeight: 400,
    padding: 16,
  },
  dataPreviewText: {
    fontSize: 12,
    color: '#333',
    fontFamily: Platform.OS === 'ios' ? 'Courier' : 'monospace',
    lineHeight: 16,
  },
  dataPreviewButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    padding: 16,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
  },
  copyButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#0078D4',
    paddingHorizontal: 16,
    paddingVertical: 8,
    borderRadius: 6,
  },
  copyButtonText: {
    marginLeft: 8,
    fontSize: 14,
    color: '#fff',
    fontWeight: '500',
  },
  closeDataButton: {
    paddingHorizontal: 16,
    paddingVertical: 8,
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#666',
  },
  closeDataButtonText: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
  safetySummarySection: {
    backgroundColor: '#f8f9fa',
    borderRadius: 12,
    padding: 16,
    marginBottom: 16,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    borderWidth: 2,
    borderColor: '#015185',
  },
  safetySummaryHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 16,
  },
  safetySummaryTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
  },
  refreshButton: {
    padding: 4,
  },
  safetySummaryGrid: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    flexWrap: 'wrap',
  },
  safetySummaryItem: {
    alignItems: 'center',
    marginVertical: 10,
    width: '45%',
    backgroundColor: '#fff',
    padding: 12,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  safetySummaryLabel: {
    fontSize: 14,
    color: '#666',
    marginBottom: 5,
    fontWeight: '500',
  },
  safetySummaryValue: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#2e7d32',
  },
  safetyPercentageValue: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#2e7d32',
  },
  totalObservationsContainer: {
    alignItems: 'center',
    marginTop: 10,
    paddingTop: 10,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
  },
  totalObservationsText: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
  requiredAsterisk: {
    color: 'red',
    fontSize: 16,
    fontWeight: 'bold',
  },
  requestIdContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#f0f8ff',
    padding: 16,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#015185',
  },
  requestIdLabel: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginRight: 10,
  },
  requestIdValue: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    flex: 1,
  },
  iosPickerOverlay: {
    flex: 1,
    justifyContent: 'flex-end',
    backgroundColor: 'rgba(0,0,0,0.4)',
  },
  iosPickerContainer: {
    backgroundColor: '#fff',
  },
  iosPickerToolbar: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderTopWidth: 1,
    borderTopColor: '#e6e6e6',
    backgroundColor: '#f7f7f7',
  },
  iosPickerButton: {
    paddingHorizontal: 8,
    paddingVertical: 6,
  },
  iosPickerButtonText: {
    fontSize: 16,
    color: '#007AFF',
  },
});

export default HSE_071step3;