import React, { useEffect, useRef, useState } from 'react';
import { View, Text, StyleSheet, ScrollView, TouchableOpacity, TextInput, Image, Modal, ActivityIndicator, Animated, Dimensions } from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import HSE_085_JSON from './HSE_085_json.json';
import { useNavigation } from '@react-navigation/native';
import SignatureCanvas from 'react-native-signature-canvas';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import NetInfo from '@react-native-community/netinfo';
import HSEInspectionAPI from '../../../ConfigMongo/HSEInspectionMongoConfig';

const { width } = Dimensions.get('window');

// SyncModal component for showing progress
const SyncModal = ({ visible, onClose, message = 'Loading', progress = 0 }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Generating PDF Report</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}    
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
            {progress > 0 && (
              <View style={styles.progressContainer}>
                <View style={styles.progressBar}>
                  <View style={[styles.progressFill, { width: `${progress}%` }]} />
                </View>
                <Text style={styles.progressText}>{progress}%</Text>
              </View>
            )}
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_085_step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [form, setForm] = useState({
    make: '', model: '', serialNumber: '', capacity: '', boomLength: '', jibLength: '', combinedLength: '',
    loadWeight: '', riggingWeight: '', totalWeight: '', heightOfLift: '', radiusOfLift: '',
    slingDiameter: '', slingLength: '', slingCapacity: '', inspected: '', totalLoad: '', numberOfLegs: '',
    loadOnEachLeg: '', angleFactor: '', tensionLoad: '', selectedAngle: '',
    crewRoles: [], operatorName: '', supervisorName: '', operatorSignatureUri: '', supervisorSignatureUri: ''
  });
  const [missingKeys, setMissingKeys] = useState({});

  const requiredCraneKeys = ['make','model','serialNumber','capacity','boomLength','jibLength','combinedLength'];
  const requiredLoadKeys = ['loadWeight','riggingWeight','heightOfLift','radiusOfLift'];
  const [crewModalVisible, setCrewModalVisible] = useState(false);
  const [signatureModalVisible, setSignatureModalVisible] = useState(false);
  const [signatureFor, setSignatureFor] = useState(null); // 'operator' | 'supervisor'
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [reportId, setReportId] = useState(null);
  const signatureCanvasStyle = `
    html, body { height: 100%; margin: 0; }
    .m-signature-pad { height: 100%; box-shadow: none; border: 1px solid #e0e0e0; }
    .m-signature-pad--body { height: calc(100% - 60px); border: 0; }
    .m-signature-pad--footer { position: absolute; bottom: 8px; left: 0; right: 0; }
    .m-signature-pad--footer .button { font-size: 14px; }
  `;

  const isMounted = useRef(true);

  useEffect(() => { 
    isMounted.current = true;
    loadSaved(); 
  }, []);

  useEffect(() => {
    return () => {
      isMounted.current = false;
    };
  }, []);

  const loadSaved = async () => {
    try {
      const saved = await AsyncStorage.getItem('hse085Step3');
      if (saved) setForm(JSON.parse(saved));
    } catch {}
    try {
      const step2 = await AsyncStorage.getItem('hse085Step2');
      if (step2) {
        const d = JSON.parse(step2);
        // precompute total weight and load on each leg if numbers present
        const totalWeight = (parseFloat(form.loadWeight) || 0) + (parseFloat(form.riggingWeight) || 0);
        const loadOnEachLeg = (parseFloat(form.totalLoad) || 0) / Math.max(parseFloat(form.numberOfLegs) || 1, 1);
        setForm((prev) => ({ ...prev, totalWeight: String(totalWeight || ''), loadOnEachLeg: String(loadOnEachLeg || '') }));
      }
    } catch {}
  };

  const save = async (next) => {
    const updated = { ...form, ...next };
    setForm(updated);
    await AsyncStorage.setItem('hse085Step3', JSON.stringify(updated));
  };

  const toNumber = (v) => {
    const n = parseFloat(String(v).replace(/,/g, ''));
    return isNaN(n) ? 0 : n;
  };

  const formatNum = (n) => {
    if (n === '' || n === null || n === undefined) return '';
    const num = Number(n);
    if (!isNaN(num)) return String(Number(num.toFixed(3))); // trim trailing zeros later
    return String(n);
  };

  const recompute = (partial) => {
    const next = { ...form, ...partial };
    const totalWeight = toNumber(next.loadWeight) + toNumber(next.riggingWeight);
    const legs = Math.max(toNumber(next.numberOfLegs) || 1, 1);
    const loadOnEachLeg = toNumber(next.totalLoad) / legs;
    const tensionLoad = loadOnEachLeg && toNumber(next.angleFactor)
      ? loadOnEachLeg * toNumber(next.angleFactor)
      : '';
    return {
      ...partial,
      totalWeight: totalWeight ? formatNum(totalWeight) : '',
      loadOnEachLeg: loadOnEachLeg ? formatNum(loadOnEachLeg) : '',
      tensionLoad: tensionLoad !== '' ? formatNum(tensionLoad) : ''
    };
  };

  const onChange = (key, value) => {
    // numeric fields: recompute totals
    const numericKeys = ['loadWeight','riggingWeight','totalLoad','numberOfLegs','angleFactor'];
    let next = { [key]: value };
    if (numericKeys.includes(key)) {
      next = recompute(next);
    }
    save(next);
  };

  const handleSignatureOK = async (signature) => {
    try {
      const signatureUri = signature?.startsWith('data:image') ? signature : `data:image/png;base64,${signature}`;
      if (signatureFor === 'operator') {
        await save({ operatorSignatureUri: signatureUri });
      } else if (signatureFor === 'supervisor') {
        await save({ supervisorSignatureUri: signatureUri });
      }
      setSignatureModalVisible(false);
      Toast.show({ type: 'success', text1: 'Signature saved' });
    } catch (error) {
      console.error('Error saving signature:', error);
      Toast.show({ type: 'error', text1: 'Failed to save signature' });
    }
  };

  const handleSignatureCancel = () => {
    setSignatureModalVisible(false);
  };

  const openSignature = (forWhom) => {
    setSignatureFor(forWhom);
    setSignatureModalVisible(true);
  };

  const handleSelectAngle = (angle) => {
    const factor = HSE_085_JSON.sections.find(s => s.key === 'rigging_data')?.angle_factors?.find(f => f.angle === angle)?.factor;
    save({ selectedAngle: angle, angleFactor: factor || '' });
  };

  // Helper functions for database operations
  const getAdminUserId = async () => {
    const userId = await AsyncStorage.getItem('userUid');
    if (!userId) throw new Error('User ID not found. Please log in again.');
    return userId;
  };

  const deepCleanObject = (obj) => {
    if (typeof obj !== 'object' || obj === null) return obj;
    if (Array.isArray(obj)) return obj.map(deepCleanObject);
    const cleaned = {};
    Object.keys(obj).forEach((key) => {
      const value = obj[key];
      cleaned[key] = typeof value === 'object' && value !== null ? deepCleanObject(value) : String(value || '');
    });
    return cleaned;
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hse085Step1');
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hse085Step2');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hse085Step3');
      const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
      const updatedQueue = syncQueue.filter((item) => item.type !== 'hse085Step1' || item.data.id !== hseId);
      await AsyncStorage.setItem('syncQueue', JSON.stringify(updatedQueue));
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const queueOfflineReport = async (reportData) => {
    try {
      const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
      syncQueue.push({ type: 'hse085Report', data: reportData });
      await AsyncStorage.setItem('syncQueue', JSON.stringify(syncQueue));
    } catch (error) {
      console.error('Error queuing offline report:', error);
      throw new Error('Failed to queue report for offline sync');
    }
  };

  const getCurrentReportNumber = async () => {
    try {
      const response = await HSEInspectionAPI.getCurrentCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || response.data.nextNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error getting current report number from MongoDB:', error);
      return '001';
    }
  };

  const incrementReportCounter = async () => {
    try {
      const response = await HSEInspectionAPI.incrementCounter();
      if (response.success && response.data) {
        return response.data.formattedNumber || '001';
      }
      return '001';
    } catch (error) {
      console.error('❌ Error incrementing report counter in MongoDB:', error);
      return '001';
    }
  };

  const uploadPDFToMongoDB = async (filePath, userId) => {
    try {
      console.log('📤 Uploading PDF to MongoDB backend...');
      const pdfFile = {
        uri: `file://${filePath}`,
        type: 'application/pdf',
        name: `hse_085_${userId}_${Date.now()}.pdf`
      };

      const uploadResponse = await HSEInspectionAPI.uploadPDF(userId, pdfFile);
      console.log('📥 Upload response:', uploadResponse);
      
      if (!uploadResponse || !uploadResponse.success) {
        const errorMsg = uploadResponse?.error || 'Unknown upload error';
        console.error('❌ Upload failed with response:', uploadResponse);
        throw new Error(`PDF upload failed: ${errorMsg}`);
      }

      const pdfDownloadUrl = uploadResponse.downloadUrl;
      console.log('✅ PDF uploaded, download URL:', pdfDownloadUrl);
      return pdfDownloadUrl;
    } catch (error) {
      console.error('❌ PDF upload failed:', error);
      throw new Error(`PDF upload failed: ${error.message}`);
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>HSE-085 Lift Plan Report</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 0;
              padding: 20px;
              box-sizing: border-box;
              font-size: 12px;
            }
            .container {
              max-width: 800px;
              margin: 0 auto;
              border: 2px solid #000;
              border-radius: 4px;
              overflow: hidden;
            }
            .header {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              padding: 15px;
              border-bottom: 2px solid #000;
              background-color: #f7f7f7;
            }
            .logo {
              width: 130px;
              height: auto;
            }
            .header-details {
              width: 55%;
            }
            .header-details table {
              border-collapse: collapse;
              width: 100%;
            }
            .header-details td {
              border: 1px solid #000;
              padding: 8px;
              font-size: 11px;
              font-weight: bold;
            }
            .main-title {
              text-align: center;
              font-size: 18px;
              font-weight: bold;
              padding: 15px;
              border-bottom: 2px solid #000;
              background-color: #f0f0f0;
            }
            .general-info {
              display: grid;
              grid-template-columns: 1fr 2fr;
              border-bottom: 1px solid #000;
            }
            .general-info div {
              padding: 8px;
              border-right: 1px solid #000;
              border-bottom: 1px solid #000;
              font-weight: bold;
            }
            .general-info div:last-child {
              border-right: none;
            }
            .section-header {
              background-color: #e9ecef;
              font-weight: bold;
              padding: 8px;
              border-bottom: 1px solid #000;
              font-size: 14px;
            }
            .data-table {
              width: 100%;
              border-collapse: collapse;
            }
            .data-table td {
              border: 1px solid #000;
              padding: 6px;
              font-size: 11px;
            }
            .data-table .label {
              font-weight: bold;
              background-color: #f5f5f5;
              width: 20%;
            }
            .angle-table {
              width: 100%;
              border-collapse: collapse;
              margin: 10px 0;
            }
            .angle-table th,
            .angle-table td {
              border: 1px solid #000;
              padding: 6px;
              text-align: center;
              font-size: 11px;
            }
            .angle-table th {
              background-color: #e9ecef;
              font-weight: bold;
            }
            .checklist-section {
              margin: 15px 0;
            }
            .checklist-item {
              display: flex;
              border-bottom: 1px solid #ddd;
              padding: 8px 0;
            }
            .checklist-question {
              flex: 1;
              font-size: 11px;
              padding-right: 10px;
            }
            .checklist-status {
              width: 80px;
              text-align: center;
              font-weight: bold;
              color: #015185;
              font-size: 11px;
            }
            .loading-sketch {
              border: 2px dashed #015185;
              padding: 40px;
              text-align: center;
              margin: 15px 0;
              background-color: #f8f9fa;
              min-height: 120px;
              display: flex;
              align-items: center;
              justify-content: center;
            }
            .crew-section {
              margin: 15px 0;
            }
            .crew-row {
              display: flex;
              border-bottom: 1px solid #ddd;
              padding: 8px 0;
            }
            .crew-label {
              width: 200px;
              font-weight: bold;
              font-size: 11px;
            }
            .crew-value {
              flex: 1;
              font-size: 11px;
            }
            .signature-box {
              border: 2px dashed #015185;
              padding: 20px;
              text-align: center;
              margin: 10px 0;
              min-height: 60px;
              display: flex;
              align-items: center;
              justify-content: center;
              background-color: #f8f9fa;
            }
            .footer {
              background-color: #f5f5f5;
              padding: 15px;
              text-align: center;
              color: #666;
              font-size: 10px;
              border-top: 1px solid #ddd;
            }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" class="logo">
              <div class="header-details">
                <table>
                  <tr><td>Document Name</td><td>HSEC-FOR-8.1.1 Lift Plan</td></tr>
                  <tr><td>Document ID</td><td>HSE-085</td></tr>
                  <tr><td>Version Date</td><td>24/08/2021</td></tr>
                  <tr><td>Version</td><td>1</td></tr>
                  <tr><td>Approved</td><td>HSM</td></tr>
                  <tr><td>Request #</td><td>${report.id || 'N/A'}</td></tr>
                </table>
              </div>
            </div>

            <div class="main-title">LIFTING PLAN USING CRANE / HIAB</div>

            <div class="general-info">
              <div>DATE:</div><div>${new Date().toLocaleDateString()}</div>
              <div>TIME:</div><div>${new Date().toLocaleTimeString()}</div>
              <div>WORK AREA / LOCATION:</div><div>${step1.workArea || step1.location || 'N/A'}</div>
            </div>

            <div class="section-header">CRANE INFORMATION</div>
            <table class="data-table">
              <tr>
                <td class="label">MAKE:</td><td>${step3.make || 'N/A'}</td>
                <td class="label">MODEL:</td><td>${step3.model || 'N/A'}</td>
                <td class="label">S/N:</td><td>${step3.serialNumber || 'N/A'}</td>
                <td class="label">CAPACITY:</td><td>${step3.capacity || 'N/A'}</td>
              </tr>
              <tr>
                <td class="label">BOOM LENGTH:</td><td>${step3.boomLength || 'N/A'}</td>
                <td class="label">JIB LENGTH:</td><td>${step3.jibLength || 'N/A'}</td>
                <td class="label">COMBINED LENGTH:</td><td colspan="3">${step3.combinedLength || 'N/A'}</td>
              </tr>
            </table>

            <div class="section-header">LOAD DATA</div>
            <table class="data-table">
              <tr>
                <td class="label">LOAD WEIGHT:</td><td>${step3.loadWeight || 'N/A'}</td>
                <td class="label">RIGGING WEIGHT:</td><td>${step3.riggingWeight || 'N/A'}</td>
                <td class="label">TOTAL WEIGHT:</td><td>${step3.totalWeight || 'N/A'}</td>
                <td class="label">HEIGHT OF LIFT:</td><td>${step3.heightOfLift || 'N/A'}</td>
                <td class="label">RADIUS OF LIFT:</td><td>${step3.radiusOfLift || 'N/A'}</td>
              </tr>
            </table>

            <div class="section-header">RIGGING DATA</div>
            <table class="data-table">
              <tr>
                <td class="label">SLING DIAMETER</td><td>${step3.slingDiameter || 'N/A'}</td>
                <td class="label">SLING LENGTH</td><td>${step3.slingLength || 'N/A'}</td>
                <td class="label">SLING CAPACITY</td><td>${step3.slingCapacity || 'N/A'}</td>
                <td class="label">INSPECTED: Y/N</td><td>${step3.inspected || 'N/A'}</td>
              </tr>
              <tr>
                <td class="label">LOAD ON EACH LEG:</td><td>${step3.loadOnEachLeg || 'N/A'}</td>
                <td class="label">TOTAL LOAD (kg):</td><td>${step3.totalLoad || 'N/A'}</td>
                <td class="label">NUMBER OF LEGS:</td><td colspan="3">${step3.numberOfLegs || 'N/A'}</td>
              </tr>
            </table>

            <div class="section-header">ANGLE FACTOR</div>
            <table class="angle-table">
              <tr>
                <th>Leg Angle</th>
                ${(HSE_085_JSON.sections.find(s => s.key === 'rigging_data')?.angle_factors || []).map(f => `<th>${f.angle}°</th>`).join('')}
              </tr>
              <tr>
                <td><strong>Load Factor</strong></td>
                ${(HSE_085_JSON.sections.find(s => s.key === 'rigging_data')?.angle_factors || []).map(f => `<td>${f.factor}</td>`).join('')}
              </tr>
            </table>

            <div class="section-header">TENSION LOAD CALCULATION: TOTAL LOAD PER LEG × ANGLE FACTOR</div>
            <table class="data-table">
              <tr>
                <td class="label" style="width: 40%;">TOTAL LOAD PER LEG</td><td>${step3.loadOnEachLeg || 'N/A'}</td>
                <td class="label" style="width: 30%;">× ANGLE FACTOR</td><td>${step3.angleFactor || 'N/A'}</td>
                <td class="label">=</td><td>${step3.tensionLoad || 'N/A'}</td>
              </tr>
            </table>

            <div class="section-header">SUMMARY OF KEY CONTROL IDENTIFIED AS PART OF THE PRE-TASK RISK ASSESSMENT</div>
            <div style="padding: 10px; border-bottom: 1px solid #000;">
              ${step2.summaryText || 'No summary provided'}
            </div>

            <div class="section-header">CHECKLIST</div>
            <div class="checklist-section">
              ${Object.entries(step2.selections || {}).map(([key, value]) => {
                const item = HSE_085_JSON.checklist.find(c => c.key === key);
                if (item) {
                  return `
                    <div class="checklist-item">
                      <div class="checklist-question">${item.question}</div>
                      <div class="checklist-status">${value}</div>
                    </div>
                  `;
                }
                return '';
              }).join('')}
            </div>

            <div class="section-header">CREW</div>
            <div class="crew-section">
              <div class="crew-row">
                <div class="crew-label">CREW:</div>
                <div class="crew-value">${(step3.crewRoles || []).map(r => r === 'operator' ? 'Operator' : 'Supervisor').join(', ') || 'N/A'}</div>
              </div>
              ${step3.crewRoles && step3.crewRoles.includes('operator') ? `
                <div class="crew-row">
                  <div class="crew-label">OPERATOR'S NAME:</div>
                  <div class="crew-value">${step3.operatorName || 'N/A'}</div>
                </div>
                <div class="crew-row">
                  <div class="crew-label">OPERATOR'S SIGNATURE:</div>
                  <div class="crew-value">
                    ${step3.operatorSignatureUri ? 
                      '<div class="signature-box">Signature captured</div>' : 
                      '<div class="signature-box">No signature</div>'
                    }
                  </div>
                </div>
              ` : ''}
              ${step3.crewRoles && step3.crewRoles.includes('supervisor') ? `
                <div class="crew-row">
                  <div class="crew-label">SUPERVISOR'S NAME:</div>
                  <div class="crew-value">${step3.supervisorName || 'N/A'}</div>
                </div>
                <div class="crew-row">
                  <div class="crew-label">SUPERVISOR'S SIGNATURE:</div>
                  <div class="crew-value">
                    ${step3.supervisorSignatureUri ? 
                      '<div class="signature-box">Signature captured</div>' : 
                      '<div class="signature-box">No signature</div>'
                    }
                  </div>
                </div>
              ` : ''}
            </div>

            <div class="footer">
              <p>Report generated on ${new Date().toLocaleDateString()} at ${new Date().toLocaleTimeString()}</p>
              <p>HSE-085 Lift Plan Report - Titan Drilling</p>
            </div>
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: `HSE-085-${report.id}`,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const handleGenerateReport = async () => {
    setSyncModalVisible(true);
    setUploadProgress(0);
    
    try {
      // Validation
      const missing = {};
      const conditionalCrew = [];
      if (Array.isArray(form.crewRoles)) {
        if (form.crewRoles.includes('operator')) conditionalCrew.push('operatorName');
        if (form.crewRoles.includes('supervisor')) conditionalCrew.push('supervisorName');
      }
      
      [...requiredCraneKeys, ...requiredLoadKeys, ...conditionalCrew].forEach((k) => {
        if (!String(form[k] ?? '').trim()) missing[k] = true;
      });
      
      if (Object.keys(missing).length > 0) {
        const missingLabels = Object.keys(missing).map((k) => {
          const labelMap = {
            make: 'MAKE', model: 'MODEL', serialNumber: 'S/N', capacity: 'CAPACITY', 
            boomLength: 'BOOM LENGTH', jibLength: 'JIB LENGTH', combinedLength: 'COMBINED LENGTH',
            loadWeight: 'LOAD WEIGHT', riggingWeight: 'RIGGING WEIGHT', 
            heightOfLift: 'HEIGHT OF LIFT', radiusOfLift: 'RADIUS OF LIFT'
          };
          return labelMap[k] || k;
        });
        
        Toast.show({ type: 'error', text1: 'Required fields missing', text2: missingLabels.join(', ') });
        setSyncModalVisible(false);
        return;
      }

      const userId = await getAdminUserId();
      
      // Try to get step data, but don't fail if it doesn't exist
      let step1Data = {};
      let step2Data = {};
      
      try {
        const savedStep1 = await AsyncStorage.getItem('hseStep1');
        if (savedStep1) {
          step1Data = JSON.parse(savedStep1);
        } else {
          // Create default step1 data if none exists
          step1Data = {
            workArea: 'Site Location',
            location: 'Project Site',
            createdAt: new Date().toISOString()
          };
        }
      } catch (e) {
        console.log('No step 1 data found, using defaults');
        step1Data = {
          workArea: 'Site Location',
          location: 'Project Site',
          createdAt: new Date().toISOString()
        };
      }
      
      try {
        let savedStep2 = await AsyncStorage.getItem('hseStep2');
        if (!savedStep2) savedStep2 = await AsyncStorage.getItem('hse085Step2');
        if (savedStep2) {
          step2Data = JSON.parse(savedStep2);
        } else {
          // Create default step2 data if none exists
          step2Data = {
            summaryText: 'Pre-task risk assessment completed',
            selections: {},
            createdAt: new Date().toISOString()
          };
        }
      } catch (e) {
        console.log('No step 2 data found, using defaults');
        step2Data = {
          summaryText: 'Pre-task risk assessment completed',
          selections: {},
          createdAt: new Date().toISOString()
        };
      }
      
      // Use request number from Step 1 if available (same as 43), otherwise fallback
      const timestamp = Date.now();
      const newReportId = step1Data?.requestNumber ? String(step1Data.requestNumber) : `HSE-085-${timestamp}`;
      setReportId(newReportId);
      
      setUploadProgress(25);

      // Prepare report data
      const reportData = {
        id: newReportId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        documentType: 'HSE-085',
        documentName: 'LIFT PLAN USING CRANE / HIAB',
        step1: deepCleanObject(step1Data),
        step2: deepCleanObject(step2Data),
        step3: deepCleanObject(form),
      };

      setUploadProgress(40);

      // Generate PDF
      console.log('📄 Generating PDF...');
      let pdfPath;
      try {
        pdfPath = await generatePDFWithHtml(reportData);
        console.log('✅ PDF generated at:', pdfPath);
      } catch (pdfError) {
        console.error('❌ PDF generation failed:', pdfError);
        throw new Error(`PDF generation failed: ${pdfError.message}`);
      }
      
      setUploadProgress(60);

      // Upload PDF to MongoDB backend
      let pdfDownloadUrl;
      try {
        pdfDownloadUrl = await uploadPDFToMongoDB(pdfPath, userId);
      } catch (uploadError) {
        console.error('❌ PDF upload failed:', uploadError);
        throw new Error(`PDF upload failed: ${uploadError.message}`);
      }
      
      setUploadProgress(80);

      // Save complete report to MongoDB
      console.log('💾 Saving report to MongoDB...');
      
      // Sanitize step3 data - convert form to clean object
      const sanitizedStep3 = {
        ...form,
        crewRoles: Array.isArray(form.crewRoles) ? form.crewRoles : [],
        make: form.make || '',
        model: form.model || '',
        serialNumber: form.serialNumber || '',
        capacity: form.capacity || '',
        boomLength: form.boomLength || '',
        jibLength: form.jibLength || '',
        combinedLength: form.combinedLength || '',
        loadWeight: form.loadWeight || '',
        riggingWeight: form.riggingWeight || '',
        totalWeight: form.totalWeight || '',
        heightOfLift: form.heightOfLift || '',
        radiusOfLift: form.radiusOfLift || '',
        slingDiameter: form.slingDiameter || '',
        slingLength: form.slingLength || '',
        slingCapacity: form.slingCapacity || '',
        inspected: form.inspected || '',
        totalLoad: form.totalLoad || '',
        numberOfLegs: form.numberOfLegs || '',
        loadOnEachLeg: form.loadOnEachLeg || '',
        angleFactor: form.angleFactor || '',
        tensionLoad: form.tensionLoad || '',
        selectedAngle: form.selectedAngle || '',
        operatorName: form.operatorName || '',
        supervisorName: form.supervisorName || '',
        operatorSignatureUri: form.operatorSignatureUri || '',
        supervisorSignatureUri: form.supervisorSignatureUri || ''
      };
      
      const completeReportData = {
        _id: newReportId,
        userId,
        requestNumber: newReportId,
        status: 'completed',
        documentType: 'HSE-085',
        documentName: 'LIFT PLAN USING CRANE / HIAB',
        step1: deepCleanObject(step1Data),
        step2: deepCleanObject(step2Data),
        step3: sanitizedStep3,
        pdfDownloadUrl,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        createdBy: userId,
        lastModifiedBy: userId
      };

      console.log('📋 Report data to save:', JSON.stringify(completeReportData, null, 2));
      
      let saveResponse;
      try {
        saveResponse = await HSEInspectionAPI.createHSEInspection(userId, completeReportData);
        console.log('📥 Save response:', saveResponse);
      } catch (saveError) {
        console.error('❌ MongoDB save failed:', saveError);
        throw new Error(`MongoDB save failed: ${saveError.message}`);
      }
      
      if (!saveResponse || !saveResponse.success) {
        const errorMsg = saveResponse?.error || 'Unknown save error';
        console.error('❌ Save failed with response:', saveResponse);
        throw new Error(`MongoDB save failed: ${errorMsg}`);
      }

      setUploadProgress(90);

      // Send HSE inspection notifications
      console.log('📧 Sending HSE inspection notifications...');
      try {
        const userData = await AsyncStorage.getItem('userData');
        const user = userData ? JSON.parse(userData) : null;
        const createdByName = user?.name || 'Unknown Inspector';
        const inspectorEmail = user?.email || '';
        
        const notificationPayload = {
          hseInspectionData: {
            _id: newReportId,
            id: newReportId,
            requestNumber: newReportId,
            checklistName: 'HSE-085 Lift Plan Using Crane/HIAB',
            country: step1Data.country || '',
            project: step1Data.project || '',
            status: 'completed',
          },
          createdByName,
          inspectorEmail,
          pdfUrl: pdfDownloadUrl
        };
        
        const notifResponse = await fetch('https://api.titandrillingzm.com:6028/api/hse-inspection-notifications/new-inspection', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(notificationPayload),
        });
        
        if (notifResponse.ok) {
          console.log('✅ HSE notifications sent successfully');
        }
      } catch (notifError) {
        console.error('❌ Error sending HSE notifications:', notifError);
      }

      // Delete temp PDF file
      await RNFS.unlink(pdfPath).catch((err) => console.warn('⚠️ Error deleting temp PDF:', err));

      // Increment the global report counter
      console.log('🔢 Incrementing counter...');
      await incrementReportCounter();
      setUploadProgress(100);

      // Clear local storage
      await flushStepData(newReportId);

      console.log('✅ HSE-085 Lift Plan Report saved to MongoDB:', newReportId);

      if (isMounted.current) {
        setSyncModalVisible(false);
        setShowSuccessModal(true);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Lift Plan report saved to MongoDB successfully.',
          position: 'bottom',
        });
      }
    } catch (error) {
      console.error('❌ Error in handleGenerateReport:', error);
      if (isMounted.current) {
        setSyncModalVisible(false);
        setUploadProgress(0);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to save report to MongoDB.',
          position: 'bottom',
        });
      }
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
      }
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Report ID not available.' });
      return;
    }

    try {
      const response = await HSEInspectionAPI.getHSEInspectionById(reportId);

      if (!response.success || !response.data || !response.data.pdfDownloadUrl) {
        Toast.show({ type: 'error', text1: 'Error', text2: 'PDF not available in MongoDB.' });
        return;
      }

      // Close modal first
      setShowSuccessModal(false);
      
      // Navigate to HSEViewPDF screen
      navigation.navigate('HSEViewPDF', {
        pdfUrl: response.data.pdfDownloadUrl,
        onGoBack: () => {
          navigation.navigate('HSEInspectionHome');
        }
      });
    } catch (error) {
      console.error('❌ Error fetching PDF URL from MongoDB:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Could not fetch PDF from MongoDB.' });
    }
  };

  const renderField = (label, key, keyboardType='default', editable=true) => (
    <View style={styles.inputBlock}>
      <Text style={styles.label}>
        {label}
        {(requiredCraneKeys.includes(key) || requiredLoadKeys.includes(key) ||
          (key==='operatorName' && form.crewRoles?.includes('operator')) ||
          (key==='supervisorName' && form.crewRoles?.includes('supervisor'))
        ) && editable ? (
          <Text style={styles.required}> *</Text>
        ) : null}
      </Text>
      <TextInput style={[styles.input, !editable && styles.readonly, missingKeys[key] && styles.inputError]}
        value={String(form[key] ?? '')}
        onChangeText={(v)=>onChange(key,v)}
        keyboardType={keyboardType}
        editable={editable}
        selectTextOnFocus={editable}
      />
    </View>
  );

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.title}>Crane Information</Text>
        {renderField('MAKE', 'make')}
        {renderField('MODEL', 'model')}
        {renderField('S/N', 'serialNumber')}
        {renderField('CAPACITY', 'capacity', 'numeric')}
        {renderField('BOOM LENGTH', 'boomLength', 'numeric')}
        {renderField('JIB LENGTH', 'jibLength', 'numeric')}
        {renderField('COMBINED LENGTH', 'combinedLength', 'numeric')}

        <Text style={styles.title}>Load Data</Text>
        {renderField('LOAD WEIGHT', 'loadWeight', 'numeric')}
        {renderField('RIGGING WEIGHT', 'riggingWeight', 'numeric')}
        {renderField('TOTAL WEIGHT', 'totalWeight', 'numeric', false)}
        {renderField('HEIGHT OF LIFT', 'heightOfLift', 'numeric')}
        {renderField('RADIUS OF LIFT', 'radiusOfLift', 'numeric')}

        <Text style={styles.title}>Rigging Data</Text>
        {renderField('SLING DIAMETER', 'slingDiameter')}
        {renderField('SLING LENGTH', 'slingLength')}
        {renderField('SLING CAPACITY', 'slingCapacity')}
        {renderField('INSPECTED (Y/N)', 'inspected')}
        {renderField('TOTAL LOAD (kg)', 'totalLoad', 'numeric')}
        {renderField('NUMBER OF LEGS', 'numberOfLegs', 'numeric')}
        {renderField('LOAD ON EACH LEG', 'loadOnEachLeg', 'numeric', false)}

        <Text style={styles.title}>Angle Factor</Text>
        <ScrollView horizontal showsHorizontalScrollIndicator={false} style={styles.angleTableScroll}>
          <View style={styles.angleTable}>
            <View style={[styles.angleRow, styles.angleHeaderRow]}>
              <View style={[styles.angleLabelCell]}>
                <Text style={styles.angleLabelText}>Leg Angle</Text>
              </View>
              {(HSE_085_JSON.sections.find(s=>s.key==='rigging_data')?.angle_factors || []).map((item) => {
                const isSelected = String(form.selectedAngle) === String(item.angle);
                return (
                  <TouchableOpacity key={`ang-${item.angle}`} style={[styles.angleCell, isSelected && styles.angleCellSelected]} onPress={() => handleSelectAngle(item.angle)}>
                    <Text style={[styles.angleCellText, isSelected && styles.angleCellTextSelected]}>{`${item.angle}°`}</Text>
                  </TouchableOpacity>
                );
              })}
            </View>
            <View style={styles.angleRow}>
              <View style={styles.angleLabelCell}>
                <Text style={styles.angleLabelText}>Load Factor</Text>
              </View>
              {(HSE_085_JSON.sections.find(s=>s.key==='rigging_data')?.angle_factors || []).map((item) => {
                const isSelected = String(form.selectedAngle) === String(item.angle);
                return (
                  <TouchableOpacity key={`fac-${item.angle}`} style={[styles.angleCell, isSelected && styles.angleCellSelected]} onPress={() => handleSelectAngle(item.angle)}>
                    <Text style={[styles.angleCellText, isSelected && styles.angleCellTextSelected]}>{String(item.factor)}</Text>
                  </TouchableOpacity>
                );
              })}
            </View>
          </View>
        </ScrollView>

        <Text style={styles.title}>Tension Load Calculation</Text>
        {renderField('ANGLE FACTOR', 'angleFactor', 'numeric', false)}
        {renderField('RESULT (TENSION LOAD)', 'tensionLoad', 'numeric', false)}

        <Text style={styles.title}>Crew</Text>
        {/* Crew roles multi-select */}
        <View style={styles.inputBlock}>
          <Text style={styles.label}>Crew member</Text>
          <TouchableOpacity style={styles.dropdownControl} onPress={() => setCrewModalVisible(!crewModalVisible)}>
            {Array.isArray(form.crewRoles) && form.crewRoles.length > 0 ? (
              <View style={styles.chipsRow}>
                {form.crewRoles.map((r) => (
                  <View key={r} style={styles.chip}><Text style={styles.chipText}>{r === 'operator' ? 'Operator' : 'Supervisor'}</Text></View>
                ))}
              </View>
            ) : (
              <Text style={{ color: '#666' }}>Select crew roles</Text>
            )}
            <MaterialIcons name="arrow-drop-down" size={22} color="#666" />
          </TouchableOpacity>
          {crewModalVisible && (
            <View style={styles.dropdownPanel}>
              {[
                { key: 'operator', label: 'Operator' },
                { key: 'supervisor', label: 'Supervisor' },
              ].map((opt) => {
                const selected = Array.isArray(form.crewRoles) && form.crewRoles.includes(opt.key);
                return (
                  <TouchableOpacity key={opt.key} style={styles.optionRow} onPress={() => {
                    let nextRoles = Array.isArray(form.crewRoles) ? [...form.crewRoles] : [];
                    if (selected) {
                      nextRoles = nextRoles.filter((r) => r !== opt.key);
                    } else {
                      nextRoles.push(opt.key);
                    }
                    save({ crewRoles: nextRoles });
                  }}>
                    <MaterialIcons name={selected ? 'check-box' : 'check-box-outline-blank'} size={20} color={selected ? '#015185' : '#666'} />
                    <Text style={styles.optionLabel}>{opt.label}</Text>
                  </TouchableOpacity>
                );
              })}
              <View style={{ alignItems: 'flex-end', marginTop: 6 }}>
                <TouchableOpacity style={styles.doneBtn} onPress={() => setCrewModalVisible(false)}>
                  <Text style={styles.doneBtnText}>Done</Text>
                </TouchableOpacity>
              </View>
            </View>
          )}
        </View>

        {form.crewRoles?.includes('operator') && renderField("OPERATOR'S NAME", 'operatorName')}
        {form.crewRoles?.includes('supervisor') && renderField("SUPERVISOR'S NAME", 'supervisorName')}

        {form.crewRoles?.includes('operator') && (
          <View style={styles.signatureBlockFull}> 
            <Text style={styles.label}>OPERATOR'S SIGNATURE</Text>
            {form.operatorSignatureUri ? (
              <View>
                <Image source={{ uri: form.operatorSignatureUri }} style={styles.signatureImage} resizeMode="contain" />
                <View style={styles.sigActions}>
                  <TouchableOpacity onPress={() => openSignature('operator')}><Text style={styles.link}>Re-sign</Text></TouchableOpacity>
                  <TouchableOpacity onPress={() => save({ operatorSignatureUri: '' })}><Text style={[styles.link, { color: '#d32f2f' }]}>Remove</Text></TouchableOpacity>
                </View>
              </View>
            ) : (
              <TouchableOpacity style={styles.signatureBox} onPress={() => openSignature('operator')}>
                <MaterialIcons name="add" size={36} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Add Signature</Text>
              </TouchableOpacity>
            )}
          </View>
        )}
        {form.crewRoles?.includes('supervisor') && (
          <View style={styles.signatureBlockFull}> 
            <Text style={styles.label}>SUPERVISOR'S SIGNATURE</Text>
            {form.supervisorSignatureUri ? (
    <View>
                <Image source={{ uri: form.supervisorSignatureUri }} style={styles.signatureImage} resizeMode="contain" />
                <View style={styles.sigActions}>
                  <TouchableOpacity onPress={() => openSignature('supervisor')}><Text style={styles.link}>Re-sign</Text></TouchableOpacity>
                  <TouchableOpacity onPress={() => save({ supervisorSignatureUri: '' })}><Text style={[styles.link, { color: '#d32f2f' }]}>Remove</Text></TouchableOpacity>
                </View>
              </View>
            ) : (
              <TouchableOpacity style={styles.signatureBox} onPress={() => openSignature('supervisor')}>
                <MaterialIcons name="add" size={36} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Add Signature</Text>
              </TouchableOpacity>
            )}
          </View>
        )}

        {/* Inline dropdown replaces modal */}
      </ScrollView>

      <View style={styles.footer}>
        <TouchableOpacity style={styles.backBtn} onPress={() => setActiveStep(2)}>
          <MaterialIcons name="chevron-left" size={16} color="#015185" />
          <Text style={styles.backText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextBtn}
          onPress={handleGenerateReport}
        >
          <Text style={styles.nextText}>Generate Report</Text>
          <MaterialIcons name="check" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      {/* Signature Modal */}
      <Modal visible={signatureModalVisible} transparent animationType="fade" onRequestClose={handleSignatureCancel}>
        <View style={styles.modalOverlayCentered}>
          <View style={styles.signatureModalBox}>
            <View style={{flexDirection:'row',justifyContent:'space-between',alignItems:'center',marginBottom:8}}>
              <Text style={{fontWeight:'700',color:'#015185'}}>{signatureFor === 'operator' ? "Operator's" : "Supervisor's"} Signature</Text>
              <TouchableOpacity onPress={handleSignatureCancel}><MaterialIcons name="close" size={20} color="#333" /></TouchableOpacity>
            </View>
            <View style={styles.signatureCanvasWrapper}>
              <SignatureCanvas
                onOK={handleSignatureOK}
                onEmpty={()=>Toast.show({type:'error',text1:'Please sign before confirming'})}
                descriptionText="Sign here"
                clearText="Clear"
                confirmText="Confirm"
                webStyle={signatureCanvasStyle}
                backgroundColor="#ffffff"
                penColor="#000"
              />
            </View>
          </View>
        </View>
      </Modal>

      {/* Generating Modal (like 43) */}
      {syncModalVisible && (
        <SyncModal visible={syncModalVisible} message={`Processing... ${uploadProgress}%`} />
      )}

      {/* Success Modal */}
      <Modal visible={showSuccessModal} transparent animationType="fade" onRequestClose={()=>setShowSuccessModal(false)}>
        <View style={styles.modalOverlayCentered}>
          <View style={styles.successBox}>
            <MaterialIcons name="check-circle" size={46} color="#4CAF50" />
            <Text style={{marginTop:8,fontWeight:'700',color:'#015185'}}>Success</Text>
            <Text style={{marginTop:6,color:'#333'}}>Lift Plan generated and saved</Text>
            <View style={{flexDirection:'row',marginTop:12}}>
              <TouchableOpacity style={styles.successBtn} onPress={()=>{setShowSuccessModal(false); navigation.navigate('HSEInspectionHome');}}>
                <Text style={styles.successBtnText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.successBtn,{marginLeft:10}]} onPress={handleViewPdf}>
                <Text style={styles.successBtnText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: '#fff' },
  mainContent: { flex: 1 },
  scrollContent: { paddingBottom: 100, paddingHorizontal: 16, paddingTop: 10 },
  title: { fontSize: 18, fontWeight: '700', color: '#015185', marginBottom: 10, marginTop: 6 },
  inputBlock: { backgroundColor: '#f8f9fa', borderWidth: 1, borderColor: '#e9ecef', borderRadius: 8, padding: 10, marginBottom: 10 },
  label: { fontSize: 14, color: '#333', marginBottom: 6 },
  input: { borderWidth: 1, borderColor: '#ddd', borderRadius: 6, paddingHorizontal: 12, paddingVertical: 10, backgroundColor: '#fff', color: '#333' },
  readonly: { backgroundColor: '#f5f5f5' },
  required: { color: '#d32f2f' },
  inputError: { borderColor: '#d32f2f' },
  footer: { position: 'absolute', left: 0, right: 0, bottom: 0, backgroundColor: '#F2F2F2', paddingHorizontal: 15, paddingTop: 12, paddingBottom: 20, borderTopWidth: 1, borderTopColor: '#ddd', flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center' },
  backBtn: { flexDirection: 'row', alignItems: 'center' },
  backText: { color: '#015185', fontSize: 16, fontWeight: '600', marginLeft: 4 },
  nextBtn: { flexDirection: 'row', alignItems: 'center' },
  nextText: { color: '#015185', fontSize: 16, fontWeight: '600', marginRight: 6 },
  angleTableScroll: { marginBottom: 10 },
  angleTable: { borderWidth: 1, borderColor: '#333', borderRadius: 6, overflow: 'hidden' },
  angleRow: { flexDirection: 'row' },
  angleHeaderRow: { borderBottomWidth: 1, borderBottomColor: '#333' },
  angleLabelCell: { width: 100, paddingVertical: 10, paddingHorizontal: 8, borderRightWidth: 1, borderRightColor: '#333', backgroundColor: '#f0f0f0', justifyContent: 'center' },
  angleLabelText: { fontWeight: '700', color: '#333' },
  angleCell: { minWidth: 70, paddingVertical: 10, paddingHorizontal: 12, borderRightWidth: 1, borderRightColor: '#333', justifyContent: 'center', alignItems: 'center' },
  angleCellSelected: { backgroundColor: '#e8f5e9' },
  angleCellText: { color: '#333', fontWeight: '600' },
  angleCellTextSelected: { color: '#015185' },
  signatureRow: { flexDirection: 'row', gap: 12, marginBottom: 14 },
  signatureBlock: { flex: 1, backgroundColor: '#f8f9fa', borderWidth: 1, borderColor: '#e9ecef', borderRadius: 8, padding: 10 },
  signatureBlockFull: { backgroundColor: '#f8f9fa', borderWidth: 1, borderColor: '#e9ecef', borderRadius: 8, padding: 10, marginBottom: 12 },
  sigPicker: { flexDirection: 'row', alignItems: 'center', borderWidth: 1, borderColor: '#015185', borderRadius: 6, paddingVertical: 8, paddingHorizontal: 10, alignSelf: 'flex-start' },
  sigPickerText: { marginLeft: 6, color: '#015185', fontWeight: '600' },
  signatureImage: { width: '100%', height: 80, borderWidth: 1, borderColor: '#ddd', borderRadius: 6 },
  sigActions: { flexDirection: 'row', gap: 16, marginTop: 6 },
  link: { color: '#015185', fontWeight: '600' },
  signatureBox: { height: 90, borderWidth: 2, borderColor: '#015185', borderStyle: 'dashed', borderRadius: 8, alignItems: 'center', justifyContent: 'center', backgroundColor: '#f8f9fa' },
  signaturePlaceholder: { color: '#666', marginTop: 6, fontWeight: '600' },
  dropdownControl: { flexDirection: 'row', alignItems: 'center', justifyContent: 'space-between', borderWidth: 1, borderColor: '#ddd', borderRadius: 6, paddingHorizontal: 12, paddingVertical: 10, backgroundColor: '#fff' },
  chipsRow: { flexDirection: 'row', flexWrap: 'wrap', gap: 6 },
  chip: { backgroundColor: '#e8f5e9', borderColor: '#4CAF50', borderWidth: 1, paddingHorizontal: 8, paddingVertical: 4, borderRadius: 16 },
  chipText: { color: '#1b5e20', fontWeight: '700' },
  modalOverlay: { position: 'absolute', top: 0, bottom: 0, left: 0, right: 0, backgroundColor: 'rgba(0,0,0,0.4)', justifyContent: 'center', alignItems: 'center' },
  dropdownModal: { width: '86%', backgroundColor: '#fff', borderRadius: 8, padding: 14 },
  modalTitle: { fontSize: 16, fontWeight: '700', color: '#015185', marginBottom: 8 },
  optionRow: { flexDirection: 'row', alignItems: 'center', paddingVertical: 8 },
  optionLabel: { marginLeft: 8, color: '#333', fontWeight: '600' },
  doneBtn: { backgroundColor: '#015185', paddingHorizontal: 14, paddingVertical: 8, borderRadius: 6 },
  doneBtnText: { color: '#fff', fontWeight: '700' },
  modalOverlayCentered: { flex:1, backgroundColor:'rgba(0,0,0,0.4)', justifyContent:'center', alignItems:'center' },
  signatureModalBox: { width:'92%', backgroundColor:'#fff', borderRadius:8, padding:12 },
  signatureCanvasWrapper: { height: 320, width: '100%' },
  progressBox: { backgroundColor:'#fff', padding:16, borderRadius:8, alignItems:'center' },
  successBox: { backgroundColor:'#fff', padding:18, borderRadius:8, alignItems:'center', width:'80%' },
  successBtn: { backgroundColor:'#015185', paddingVertical:8, paddingHorizontal:16, borderRadius:6 },
  successBtnText: { color:'#fff', fontWeight:'700' },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
  },
  progressContainer: {
    marginTop: 15,
    width: '100%',
    alignItems: 'center',
  },
  progressBar: {
    width: '100%',
    height: 8,
    backgroundColor: '#e0e0e0',
    borderRadius: 4,
    overflow: 'hidden',
    marginBottom: 8,
  },
  progressFill: {
    height: '100%',
    backgroundColor: '#015185',
    borderRadius: 4,
  },
  progressText: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
  },
});

export default HSE_085_step3;