import React, { useEffect, useMemo, useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  Alert,
  Dimensions,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import HSEcorrectiveactionStep1 from './HSEcorrectiveactionStep1';
import HSECAStep2 from './HSECAStep2';
import HSECorrectiveActionAPI from '../../../ConfigMongo/HSECorrectiveActionConfig';

const extractUnsafeItems = (report) => {
  if (!report?.step2?.sections) {
    return [];
  }

  const selections = report.step2.selections;
  const countInputs = report.step2.countInputs;
  const sections = Array.isArray(report.step2.sections) ? report.step2.sections : [];
  const UNSAFE_VALUES = new Set(['faulty', 'unsafe', 'reject', 'not ok', 'not okay']);

  const unsafeItems = [];
  sections.forEach((section) => {
    const subItems = Array.isArray(section?.subItems) ? section.subItems : [];
    subItems.forEach((subItem) => {
      const itemKey = subItem?.key;
      const selectionRaw = selections?.[itemKey];
      const selection = selectionRaw ? String(selectionRaw).trim().toLowerCase() : '';
      const counts = countInputs?.[itemKey] || {};
      const unsafeCount = Number(counts.unsafe) || 0;
      const safeCount = Number(counts.safe) || 0;
      const naValue = counts.notApplicable;
      const naCount =
        typeof naValue === 'number'
          ? naValue
          : naValue === true || String(naValue).trim().toLowerCase() === 'true'
          ? 1
          : 0;

      const isUnsafe =
        unsafeCount > 0 ||
        (selection && UNSAFE_VALUES.has(selection));

      if (isUnsafe) {
        unsafeItems.push({
          id: itemKey,
          title: subItem.itemName || 'Unknown Item',
          section: section.name || 'General',
          unsafeCount,
          safeCount,
          naCount,
          selection: selectionRaw || '',
        });
      }
    });
  });

  return unsafeItems;
};

const groupBySection = (unsafeItems = []) => {
  const map = new Map();
  unsafeItems.forEach((item) => {
    const key = item.section || 'General';
    if (!map.has(key)) {
      map.set(key, []);
    }
    map.get(key).push(item);
  });
  return Array.from(map.entries()).map(([section, items]) => ({ section, items }));
};

const { height } = Dimensions.get('window');
const STATUS_BAR_HEIGHT = Platform.OS === 'ios' ? 44 : StatusBar.currentHeight || 0;
const TOTAL_HEADER_HEIGHT = height * 0.15 - STATUS_BAR_HEIGHT;

const HSECSHeader = ({ route, navigation }) => {
  const report = route?.params?.report;
  const [activeStep, setActiveStep] = useState(1);
  const [subItemStatuses, setSubItemStatuses] = useState({});

  const unsafeItems = useMemo(() => extractUnsafeItems(report), [report]);
  const groupedSections = useMemo(() => groupBySection(unsafeItems), [unsafeItems]);
  const unsafeTotal = useMemo(
    () =>
      unsafeItems.reduce((sum, item) => {
        const count = Number(item?.unsafeCount);
        if (Number.isFinite(count) && count > 0) {
          return sum + count;
        }
        return sum + 1;
      }, 0),
    [unsafeItems]
  );
  const checklistTitle =
    report?.step1?.selectedChecklist?.document_name ||
    report?.step1?.checklistTitle ||
    report?.step1?.documentTitle ||
    report?.step1?.title ||
    '';
  const isUpdate = Boolean(route?.params?.isUpdate);
  const onRefresh = route?.params?.onRefresh;

  useEffect(() => {
    let isMounted = true;
    const loadExistingCorrectiveAction = async () => {
      try {
        const reportId = report?._id || report?.id;
        if (!reportId) return;
        const response = await HSECorrectiveActionAPI.getCorrectiveAction(reportId);
        if (isMounted && response?.success && response.data) {
          setSubItemStatuses(response.data.subItemStatuses || {});
        }
      } catch (error) {
        console.warn('[HSE Corrective Action] Failed to load existing data:', error);
      }
    };
    loadExistingCorrectiveAction();
    return () => {
      isMounted = false;
    };
  }, [report]);

  const resetState = () => {
    setActiveStep(1);
    setSubItemStatuses({});
  };

  const confirmExit = (onConfirm) => {
    Alert.alert(
      'Discard Changes',
      'Are you sure you want to exit? Unsaved changes will be lost.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Exit',
          style: 'destructive',
          onPress: onConfirm,
        },
      ]
    );
  };

  const handleClose = () => {
    confirmExit(() => {
      resetState();
      navigation.goBack();
    });
  };

  const handleStatusChange = (itemId, status) => {
    setSubItemStatuses((prev) => ({
      ...prev,
      [itemId]: status,
    }));
  };

  return (
    <View style={styles.container}>
      <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
      <View style={styles.headerWrapper}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.headerGradient}>
          <View style={styles.headerContent}>
            <TouchableOpacity style={styles.backButton} onPress={handleClose}>
              <MaterialIcons name="chevron-left" size={22} color="#fff" />
              <Text style={styles.backButtonText}>Back</Text>
            </TouchableOpacity>
            <View style={styles.stepsRow}>
              <View style={styles.stepContainer}>
                {activeStep === 2 ? (
                  <View style={styles.stepBadgeActive}>
                    <MaterialIcons name="check" size={18} color="#fff" />
                  </View>
                ) : (
                  <View style={styles.stepBadgeActive}>
                    <Text style={styles.stepBadgeTextActive}>1</Text>
                  </View>
                )}
                <View style={styles.stepTextBlock}>
                  <Text style={styles.stepLabelWhite}>Step 1</Text>
                  <Text style={styles.stepSubtitle}>Review</Text>
                </View>
              </View>
              <View style={styles.stepContainer}>
                <View style={[styles.stepBadgeInactive, activeStep === 2 && { backgroundColor: '#4cd964' }]}>
                  <Text style={[styles.stepBadgeTextInactive, activeStep === 2 && { color: '#fff' }]}>2</Text>
                </View>
                <View style={styles.stepTextBlock}>
                  <Text style={styles.stepLabelWhite}>Step 2</Text>
                  <Text style={styles.stepSubtitle}>Action</Text>
                </View>
              </View>
            </View>
          </View>
        </LinearGradient>
      </View>

      <View style={styles.mainContent}>
        {activeStep === 1 ? (
          <HSEcorrectiveactionStep1
            checklistTitle={checklistTitle}
            groupedSections={groupedSections}
            unsafeItems={unsafeItems}
            unsafeTotal={unsafeTotal}
            subItemStatuses={subItemStatuses}
            onChangeStatus={handleStatusChange}
            onNext={() => setActiveStep(2)}
            onCancel={handleClose}
          />
        ) : (
          <HSECAStep2
            report={report}
            navigation={navigation}
            subItemStatuses={subItemStatuses}
            setSubItemStatuses={setSubItemStatuses}
            setActiveStep={setActiveStep}
            isUpdate={isUpdate}
            onRefresh={onRefresh}
          />
        )}
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  headerWrapper: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    zIndex: 10,
  },
  headerGradient: {
    width: '100%',
    height: STATUS_BAR_HEIGHT + TOTAL_HEADER_HEIGHT,
  },
  headerContent: {
    flex: 1,
    paddingTop: STATUS_BAR_HEIGHT,
  },
  backButton: {
    position: 'absolute',
    top: STATUS_BAR_HEIGHT + 10,
    left: 10,
    flexDirection: 'row',
    alignItems: 'center',
    zIndex: 2,
  },
  backButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
    marginLeft: 2,
  },
  stepsRow: {
    flexDirection: 'row',
    justifyContent: 'center',
    alignItems: 'flex-end',
    marginTop: 30,
    width: '100%',
  },
  stepContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 32,
  },
  stepBadgeActive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#4cd964',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeInactive: {
    width: 28,
    height: 28,
    borderRadius: 14,
    backgroundColor: '#e0e6ed',
    alignItems: 'center',
    justifyContent: 'center',
    marginRight: 6,
  },
  stepBadgeTextActive: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepBadgeTextInactive: {
    color: '#3481BC',
    fontWeight: 'bold',
    fontSize: 16,
  },
  stepTextBlock: {
    flexDirection: 'column',
    alignItems: 'flex-start',
  },
  stepLabelWhite: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
    lineHeight: 18,
  },
  stepSubtitle: {
    color: '#fff',
    fontSize: 12,
    marginTop: -2,
  },
  mainContent: {
    flex: 1,
    marginTop: STATUS_BAR_HEIGHT + TOTAL_HEADER_HEIGHT,
  },
});

export default HSECSHeader;