// MongoDB Configuration for Inspection Reports
// Use external IP for both development and production since mobile can't reach localhost
const INSPECTION_API_BASE_URL = 'https://api.titandrillingzm.com:6004';
const RISK_ASSESSMENT_API_BASE_URL = 'https://api.titandrillingzm.com:6038';

// Helper function to build URLs with query parameters
const buildURL = (endpoint, params = {}) => {
  const url = new URL(`${INSPECTION_API_BASE_URL}${endpoint}`);
  Object.keys(params).forEach(key => {
    if (params[key] !== null && params[key] !== undefined) {
      url.searchParams.append(key, params[key]);
    }
  });
  return url.toString();
};

const buildRiskURL = (endpoint, params = {}) => {
  const url = new URL(`${RISK_ASSESSMENT_API_BASE_URL}${endpoint}`);
  Object.keys(params).forEach(key => {
    if (params[key] !== null && params[key] !== undefined) {
      url.searchParams.append(key, params[key]);
    }
  });
  return url.toString();
};

  // API call helper
const DEFAULT_TIMEOUT_MS = 150000;

const apiCall = async (url, options = {}) => {
  try {
    console.log('🌐 Making API call to:', url);
    console.log('🌐 Options:', { method: options.method, bodySize: options.body?.length });
    
    // Allow callers to override timeout when needed (e.g. large payload uploads)
    const timeoutMs = typeof options.timeout === 'number' ? options.timeout : DEFAULT_TIMEOUT_MS;
    const controller = new AbortSignalController(timeoutMs);
    const signal = controller.signal;

    const { timeout, ...fetchOptions } = options;

    const response = await fetch(url, {
      headers: {
        'Content-Type': 'application/json',
        ...fetchOptions.headers,
      },
      signal,
      ...fetchOptions,
    });
    
    controller.clear();

      console.log('📡 Response status:', response.status, response.statusText);
      
      let data;
      try {
        data = await response.json();
        console.log('📡 Response data:', { success: data.success, hasError: !!data.error });
      } catch (jsonError) {
        console.error('❌ Failed to parse JSON response:', jsonError);
        const textResponse = await response.text();
        console.error('❌ Raw response:', textResponse);
        throw new Error(`Invalid JSON response: ${textResponse.substring(0, 200)}`);
      }
      
      if (!response.ok) {
        console.error('❌ API call failed:', {
          status: response.status,
          statusText: response.statusText,
          error: data.error,
          data: data
        });
        throw new Error(data.error || `HTTP ${response.status}: ${response.statusText}`);
      }
      
      return data;
  } catch (error) {
    console.error('❌ API call failed:', error);
    console.error('❌ Error details:', {
      message: error.message,
      name: error.name,
      url: url,
      options: options
    });
    
    // Handle specific error types
    if (error.name === 'AbortError') {
      throw new Error('Request timeout: Server took too long to respond. Please try again.');
    } else if (error.message.includes('Network request failed')) {
      throw new Error('Network error: Cannot connect to server. Please check your internet connection.');
    } else if (error.message.includes('Failed to fetch')) {
      throw new Error('Connection failed: Unable to reach the server. Please check your network connection.');
    } else if (error.message.includes('Invalid JSON response')) {
      throw new Error('Server returned invalid response. Please try again.');
    }
    
    throw error;
  }
};

class AbortSignalController {
  constructor(timeoutMs) {
    this.controller = new AbortController();
    this.timeoutId = setTimeout(() => this.controller.abort(), timeoutMs);
  }

  get signal() {
    return this.controller.signal;
  }

  clear() {
    clearTimeout(this.timeoutId);
  }
}

// Inspection API functions
const InspectionAPI = {
  // Health check
  healthCheck: async () => {
    const url = buildURL('/health');
    return await apiCall(url);
  },

  // Get all inspection reports for a user
  getInspections: async (userId, options = {}) => {
    const { page = 1, limit = 50, status, country, project } = options;
    const url = buildURL(`/inspections/${userId}`, {
      page,
      limit,
      status,
      country,
      project
    });
    return await apiCall(url);
  },

  // Get total count of inspection reports (optimized)
  getTotalCount: async () => {
    const url = buildURL('/inspections/count');
    
    console.log('🔢 getTotalCount API call:', url);
    const result = await apiCall(url);
    
    console.log('🔢 getTotalCount result:', {
      success: result.success,
      total: result.total,
      message: result.message
    });
    
    return result;
  },

  // Get all inspection reports from all users (for global request number generation)
  getAllReports: async (options = {}) => {
    const { page = 1, limit = 10000 } = options;
    const url = buildURL('/inspections/all', {
      page,
      limit
    });
    
    console.log('🔍 getAllReports API call:', url);
    const result = await apiCall(url);
    
    console.log('🔍 getAllReports result:', {
      success: result.success,
      dataLength: result.data ? result.data.length : 'no data',
      totalReports: result.data ? result.data.length : 0,
      pagination: result.pagination
    });
    
    if (result.data && result.data.length > 0) {
      console.log('🔍 Sample report:', {
        _id: result.data[0]._id,
        userId: result.data[0].userId,
        requestNumber: result.data[0].requestNumber,
        status: result.data[0].status
      });
    }
    
    return result;
  },

  // Get a specific inspection report by reportId only (for cross-user viewing)
  getInspectionById: async (reportId) => {
    const url = buildURL(`/inspections/report/${reportId}`);
    return await apiCall(url);
  },

  // Get a specific inspection report (original endpoint for user's own reports)
  getInspection: async (userId, reportId) => {
    const url = buildURL(`/inspections/${userId}/${reportId}`);
    return await apiCall(url);
  },

  // Create a new inspection report
  createInspection: async (userId, reportData) => {
    try {
      console.log('📤 Creating inspection for userId:', userId);
      console.log('📤 Report data keys:', Object.keys(reportData || {}));
      console.log('📤 Report status:', reportData.status);
      console.log('📤 Request number:', reportData.requestNumber);
      
      const url = buildURL('/inspections');
      
      // Clean and validate the report data - PRESERVE ALL FIELDS for offline sync
      const cleanedReportData = {
        _id: reportData._id,
        userId: reportData.userId || userId,
        requestNumber: reportData.requestNumber,
        status: reportData.status || 'completed', // Default to completed for offline reports
        step1: reportData.step1 || {},
        step2: reportData.step2 || {},
        step3: reportData.step3 || {},
        createdAt: reportData.createdAt || new Date().toISOString(),
        updatedAt: reportData.updatedAt || new Date().toISOString(),
        createdBy: reportData.createdBy || userId,
        lastModifiedBy: reportData.lastModifiedBy || userId,
      };
      
      const payload = { userId, reportData: cleanedReportData };
      
      console.log('📤 Payload size:', JSON.stringify(payload).length);
      console.log('📤 API URL:', url);
      console.log('📤 Cleaned report data:', {
        _id: cleanedReportData._id,
        requestNumber: cleanedReportData.requestNumber,
        status: cleanedReportData.status,
        hasPDF: !!cleanedReportData.step3?.pdfDownloadUrl
      });
      
      const result = await apiCall(url, {
        method: 'POST',
        body: JSON.stringify(payload)
      });
      
      console.log('✅ Inspection created successfully:', result.success);
      return result;
    } catch (error) {
      console.error('❌ Error in createInspection:', error);
      console.error('❌ Error details:', {
        message: error.message,
        name: error.name,
        stack: error.stack
      });
      
      // If the error is due to large payload, try with minimal data
      if (error.message.includes('413') || error.message.includes('too large')) {
        console.log('🔄 Trying with minimal data due to size limit...');
        try {
          const minimalData = {
            _id: reportData._id,
            userId: reportData.userId,
            requestNumber: reportData.requestNumber,
            status: 'completed',
            step1: {
              country: reportData.step1?.country,
              project: reportData.step1?.project,
              inspector: reportData.step1?.inspector,
              requestNumber: reportData.step1?.requestNumber
            },
            step2: {
              sections: reportData.step2?.sections || [],
              headerInputValues: reportData.step2?.headerInputValues || {}
            },
            step3: {
              pdfDownloadUrl: reportData.step3?.pdfDownloadUrl,
              localPdfPath: reportData.step3?.localPdfPath,
              signatureFields: reportData.step3?.signatureFields || []
            },
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
          };
          
          const minimalPayload = { userId, reportData: minimalData };
          console.log('📤 Minimal payload size:', JSON.stringify(minimalPayload).length);
          
          const result = await apiCall(url, {
            method: 'POST',
            body: JSON.stringify(minimalPayload)
          });
          
          console.log('✅ Inspection created with minimal data:', result.success);
          return result;
        } catch (minimalError) {
          console.error('❌ Minimal data also failed:', minimalError);
          throw error; // Throw original error
        }
      }
      
      throw error;
    }
  },

  // Update an inspection report
  updateInspection: async (userId, reportId, updateData) => {
    const url = buildURL(`/inspections/${userId}/${reportId}`);
    return await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(updateData)
    });
  },

  // Update specific step of an inspection report
  updateInspectionStep: async (userId, reportId, step, stepData) => {
    const url = buildURL(`/inspections/${userId}/${reportId}/${step}`);
    return await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(stepData)
    });
  },

  // Complete an inspection report
  completeInspection: async (userId, reportId, pdfDownloadUrl = null, signatureDownloadUrl = null) => {
    const url = buildURL(`/inspections/${userId}/${reportId}/complete`);
    return await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify({ pdfDownloadUrl, signatureDownloadUrl })
    });
  },

  // Delete an inspection report (requires userId and reportId to match)
  deleteInspection: async (userId, reportId) => {
    const url = buildURL(`/inspections/${userId}/${reportId}`);
    return await apiCall(url, {
      method: 'DELETE'
    });
  },

  // Delete an inspection report by ID only (for cross-user deletion - admin/view permissions)
  deleteInspectionById: async (reportId) => {
    const url = buildURL(`/inspections/delete-by-id/${reportId}`);
    return await apiCall(url, {
      method: 'DELETE'
    });
  },

  // Get inspection statistics
  getInspectionStats: async (userId, options = {}) => {
    const { country, project, startDate, endDate } = options;
    const url = buildURL(`/inspections/${userId}/stats`, {
      country,
      project,
      startDate,
      endDate
    });
    return await apiCall(url);
  },

  // Upload file for inspection report
  uploadFile: async (userId, reportId, step, itemKey, file) => {
    const url = buildURL(`/inspections/${userId}/${reportId}/upload`);
    
    const formData = new FormData();
    formData.append('file', {
      uri: file.uri,
      type: file.type || 'application/octet-stream',
      name: file.name || 'file'
    });
    formData.append('step', step);
    formData.append('itemKey', itemKey);

    return await apiCall(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'multipart/form-data',
      },
      body: formData
    });
  },

  // Upload PDF for inspection report
  uploadPdf: async (userId, formData) => {
    try {
      console.log('📤 Uploading PDF for userId:', userId);
      const url = `${INSPECTION_API_BASE_URL}/inspections/upload-pdf/${userId}`;
      
      console.log('📤 Upload URL:', url);
      
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'multipart/form-data',
        },
        body: formData
      });

      console.log('📡 PDF Upload response status:', response.status);
      
      const data = await response.json();
      console.log('📡 PDF Upload response data:', data);
      
      if (!response.ok) {
        throw new Error(data.error || `HTTP ${response.status}: ${response.statusText}`);
      }
      
      return data;
    } catch (error) {
      console.error('❌ PDF upload failed:', error);
      throw error;
    }
  },

  // Upload signature for inspection report
  uploadSignature: async (userId, formData) => {
    try {
      console.log('📤 Uploading signature for userId:', userId);
      const url = `${INSPECTION_API_BASE_URL}/inspections/upload-signature/${userId}`;
      
      console.log('📤 Upload URL:', url);
      
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'multipart/form-data',
        },
        body: formData
      });

      console.log('📡 Signature Upload response status:', response.status);
      
      const data = await response.json();
      console.log('📡 Signature Upload response data:', data);
      
      if (!response.ok) {
        throw new Error(data.error || `HTTP ${response.status}: ${response.statusText}`);
      }
      
      return data;
    } catch (error) {
      console.error('❌ Signature upload failed:', error);
      throw error;
    }
  },

  // Risk Management
  getRiskManagementRecord: async (adminId, inspectionId) => {
    const url = buildRiskURL(`/risk-management/${adminId}/${inspectionId}`);
    try {
      return await apiCall(url);
    } catch (error) {
      if (error.message && error.message.includes('404')) {
        return { success: false, error: 'Risk management record not found', status: 404 };
      }
      throw error;
    }
  },

  saveRiskManagementStep: async (adminId, inspectionId, step, payload) => {
    const url = buildRiskURL(`/risk-management/${adminId}/${inspectionId}/${step}`);
    return await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(payload),
    });
  },

  uploadRiskManagementPdf: async (adminId, inspectionId, formData) => {
    try {
      console.log('📤 Uploading risk management PDF:', { adminId, inspectionId });
      const url = `${RISK_ASSESSMENT_API_BASE_URL}/risk-management/upload-pdf/${adminId}/${inspectionId}`;

      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'multipart/form-data',
        },
        body: formData,
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || `HTTP ${response.status}: ${response.statusText}`);
      }

      if (data.downloadUrl && !/^https?:\/\//i.test(data.downloadUrl)) {
        const startsWithSlash = data.downloadUrl.startsWith('/');
        data.downloadUrl = `${RISK_ASSESSMENT_API_BASE_URL}${startsWithSlash ? '' : '/'}${data.downloadUrl}`;
      }

      return data;
    } catch (error) {
      console.error('❌ Risk management PDF upload failed:', error);
      throw error;
    }
  },

  getRiskManagementPdfUrl: (adminId, fileName) => {
    return `${RISK_ASSESSMENT_API_BASE_URL}/risk-management/download-pdf/${adminId}/${fileName}`;
  },

  // Get download URL for file
  getDownloadUrl: (userId, fileName) => {
    return `${INSPECTION_API_BASE_URL}/inspections/download-file/${userId}/${fileName}`;
  },

  // Search inspection reports
  searchInspections: async (userId, query, options = {}) => {
    const { page = 1, limit = 50 } = options;
    const url = buildURL(`/inspections/${userId}/search`, {
      q: query,
      page,
      limit
    });
    return await apiCall(url);
  },

  // Get file URL
  getFileUrl: (filename) => {
    return `${INSPECTION_API_BASE_URL}/inspections/files/${filename}`;
  },

  // Upload all checklists
  uploadAllChecklists: async (checklistsData, uploadedBy = 'system') => {
    const url = buildURL('/checklists/upload-all');
    return await apiCall(url, {
      method: 'POST',
      body: JSON.stringify({
        checklistsData,
        uploadedBy
      })
    });
  },

  // Get all checklists
  getChecklists: async (options = {}) => {
    const { page = 1, limit = 50, search = '' } = options;
    const params = { page, limit };
    if (search) params.search = search;
    
    const url = buildURL('/checklists', params);
    return await apiCall(url);
  },

  // Get specific checklist
  getChecklist: async (documentId) => {
    const url = buildURL(`/checklists/${documentId}`);
    return await apiCall(url);
  },

  // Delete checklist
  deleteChecklist: async (documentId) => {
    const url = buildURL(`/checklists/${documentId}`);
    return await apiCall(url, {
      method: 'DELETE'
    });
  },

  // ========== RISK ASSESSMENT API (PORT 5031) ==========
  getRiskAssessmentRecord: async (inspectionId) => {
    const url = buildRiskURL(`/risk-assessments/${inspectionId}`);
    return await apiCall(url);
  },

  saveRiskAssessmentRecord: async (inspectionId, payload) => {
    const url = buildRiskURL(`/risk-assessments/${inspectionId}`);
    return await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(payload),
      timeout: 20000,
    });
  },

  deleteRiskAssessmentRecord: async (inspectionId) => {
    const url = buildRiskURL(`/risk-assessments/${inspectionId}`);
    return await apiCall(url, {
      method: 'DELETE',
    });
  },

  listRiskAssessments: async (options = {}) => {
    const { adminId, page = 1, limit = 50 } = options;
    const url = buildRiskURL('/risk-assessments', {
      adminId,
      page,
      limit,
    });
    return await apiCall(url);
  },

  // ========== CORRECTIVE ACTION API FUNCTIONS ==========

  // Get corrective action by report ID
  getCorrectiveAction: async (reportId) => {
    const url = buildURL(`/corrective-actions/${reportId}`);
    return await apiCall(url);
  },

  // Create or update corrective action
  saveCorrectiveAction: async (reportId, userId, correctiveActionData) => {
    const url = buildURL('/corrective-actions');
    return await apiCall(url, {
      method: 'POST',
      body: JSON.stringify({
        reportId,
        userId,
        correctiveActionData
      })
    });
  },

  // Update corrective action step
  updateCorrectiveActionStep: async (reportId, stepNumber, stepData) => {
    const url = buildURL(`/corrective-actions/${reportId}/step/${stepNumber}`);
    return await apiCall(url, {
      method: 'PUT',
      body: JSON.stringify(stepData)
    });
  },

  // Upload corrective action PDF
  uploadCorrectiveActionPdf: async (reportId, formData, userId) => {
    try {
      console.log('📤 Uploading corrective action PDF for reportId:', reportId, 'userId:', userId);
      const url = buildURL(`/corrective-actions/${reportId}/upload-pdf`);
      
      console.log('📤 Upload URL:', url);
      
      // Add userId to formData
      if (userId) {
        formData.append('userId', userId);
      }
      
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'multipart/form-data',
        },
        body: formData
      });

      console.log('📡 Corrective Action PDF Upload response status:', response.status);
      
      const data = await response.json();
      console.log('📡 Corrective Action PDF Upload response data:', data);
      
      if (!response.ok) {
        throw new Error(data.error || `HTTP ${response.status}: ${response.statusText}`);
      }
      
      return data;
    } catch (error) {
      console.error('❌ Corrective action PDF upload failed:', error);
      throw error;
    }
  },

  // Upload corrective action signature
  uploadCorrectiveActionSignature: async (reportId, signatureType, formData, userId) => {
    try {
      console.log('📤 Uploading corrective action signature for reportId:', reportId, 'type:', signatureType, 'userId:', userId);
      const url = buildURL(`/corrective-actions/${reportId}/upload-signature`);
      
      console.log('📤 Upload URL:', url);
      
      // Add signature type and userId to form data
      formData.append('signatureType', signatureType);
      if (userId) {
        formData.append('userId', userId);
      }
      
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'multipart/form-data',
        },
        body: formData
      });

      console.log('📡 Corrective Action Signature Upload response status:', response.status);
      
      const data = await response.json();
      console.log('📡 Corrective Action Signature Upload response data:', data);
      
      if (!response.ok) {
        throw new Error(data.error || `HTTP ${response.status}: ${response.statusText}`);
      }
      
      return data;
    } catch (error) {
      console.error('❌ Corrective action signature upload failed:', error);
      throw error;
    }
  },

  // Get corrective action PDF download URL
  getCorrectiveActionPdfDownloadUrl: (fileName) => {
    return buildURL(`/corrective-actions/download-pdf/${fileName}`);
  },

  // Get corrective action signature download URL
  getCorrectiveActionSignatureDownloadUrl: (fileName) => {
    return buildURL(`/corrective-actions/download-signature/${fileName}`);
  },

  // Delete corrective action
  deleteCorrectiveAction: async (reportId) => {
    const url = buildURL(`/corrective-actions/${reportId}`);
    return await apiCall(url, {
      method: 'DELETE'
    });
  }
};

export default InspectionAPI;
