// HSE Inspection MongoDB API Configuration
const BASE_URLS = [
  'https://api.titandrillingzm.com:5014',
  'https://api.titandrillingzm.com:6014', // fallback for legacy routing
];
const PRIMARY_BASE_URL = BASE_URLS[0];

const fetchWithFallback = async (path) => {
  let lastError;
  for (const baseUrl of BASE_URLS) {
    try {
      const response = await fetch(`${baseUrl}${path}`);
      const data = await response.json();
      return data;
    } catch (error) {
      lastError = error;
      console.warn(`[HSEInspectionAPI] Request to ${baseUrl}${path} failed:`, error.message);
    }
  }
  throw lastError || new Error('All HSE Inspection API endpoints unreachable');
};

const HSEInspectionAPI = {
  // Get all HSE inspections for a user
  getHSEInspections: async (userId, options = {}) => {
    try {
      const { page = 1, limit = 50, status, country, project } = options;
      
      let path = `/hse-inspections/${userId}?page=${page}&limit=${limit}`;
      if (status) path += `&status=${status}`;
      if (country) path += `&country=${country}`;
      if (project) path += `&project=${project}`;
      
      const data = await fetchWithFallback(path);
      return data;
    } catch (error) {
      console.error('Error fetching HSE inspections:', error);
      return { success: false, error: error.message, data: [], pagination: { total: 0 } };
    }
  },

  // Get all HSE inspections from all users
  getAllHSEInspections: async (options = {}) => {
    try {
      const { page = 1, limit = 100 } = options;
      
      const path = `/hse-inspections/all?page=${page}&limit=${limit}`;
      const data = await fetchWithFallback(path);
      
      return data;
    } catch (error) {
      console.error('Error fetching all HSE inspections:', error);
      return { success: false, error: error.message, data: [], pagination: { total: 0 } };
    }
  },

  // Get specific HSE inspection by ID
  getHSEInspectionById: async (reportId) => {
    try {
      const url = `${PRIMARY_BASE_URL}/hse-inspections/report/${reportId}`;
      
      const response = await fetch(url);
      const data = await response.json();
      
      return data;
    } catch (error) {
      console.error('Error fetching HSE inspection:', error);
      return { success: false, error: error.message };
    }
  },

  // Create new HSE inspection
  createHSEInspection: async (userId, reportData) => {
    try {
      const url = `${PRIMARY_BASE_URL}/hse-inspections`;
      
      console.log('📤 Creating HSE Inspection:', {
        url,
        userId,
        reportId: reportData._id || reportData.id
      });
      
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ userId, reportData }),
      });
      
      console.log('📥 Response status:', response.status);
      
      const data = await response.json();
      console.log('📥 Response data:', data);
      
      return data;
    } catch (error) {
      console.error('❌ Error creating HSE inspection:', error);
      console.error('❌ Error details:', {
        message: error.message,
        stack: error.stack
      });
      return { success: false, error: error.message };
    }
  },

  // Update HSE inspection
  updateHSEInspection: async (userId, reportId, updateData) => {
    try {
      const url = `${PRIMARY_BASE_URL}/hse-inspections/${userId}/${reportId}`;
      
      const response = await fetch(url, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(updateData),
      });
      
      const data = await response.json();
      return data;
    } catch (error) {
      console.error('Error updating HSE inspection:', error);
      return { success: false, error: error.message };
    }
  },

  // Complete HSE inspection
  completeHSEInspection: async (userId, reportId, pdfDownloadUrl) => {
    try {
      const url = `${PRIMARY_BASE_URL}/hse-inspections/${userId}/${reportId}/complete`;
      
      const response = await fetch(url, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ pdfDownloadUrl }),
      });
      
      const data = await response.json();
      return data;
    } catch (error) {
      console.error('Error completing HSE inspection:', error);
      return { success: false, error: error.message };
    }
  },

  // Delete HSE inspection
  deleteHSEInspection: async (userId, reportId) => {
    try {
      const url = `${PRIMARY_BASE_URL}/hse-inspections/${userId}/${reportId}`;
      
      const response = await fetch(url, {
        method: 'DELETE',
      });
      
      const data = await response.json();
      return data;
    } catch (error) {
      console.error('Error deleting HSE inspection:', error);
      return { success: false, error: error.message };
    }
  },

  // Upload PDF
  uploadPDF: async (userId, pdfFile) => {
    try {
      console.log('📤 Uploading PDF to MongoDB backend:', {
        userId,
        fileName: pdfFile.name,
        uri: pdfFile.uri
      });
      
      const formData = new FormData();
      formData.append('pdf', {
        uri: pdfFile.uri || pdfFile.path,
        type: 'application/pdf',
        name: pdfFile.name || `hse_inspection_${Date.now()}.pdf`,
      });

      const url = `${PRIMARY_BASE_URL}/hse-inspections/upload-pdf/${userId}`;
      console.log('📤 Upload URL:', url);
      
      const response = await fetch(url, {
        method: 'POST',
        body: formData,
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      });
      
      console.log('📥 Upload response status:', response.status);
      
      const data = await response.json();
      console.log('📥 Upload response data:', data);
      
      return data;
    } catch (error) {
      console.error('❌ Error uploading PDF:', error);
      console.error('❌ Error details:', {
        message: error.message,
        stack: error.stack
      });
      return { success: false, error: error.message };
    }
  },

  // Upload attachment
  uploadAttachment: async (userId, file) => {
    try {
      const formData = new FormData();
      formData.append('attachment', {
        uri: file.uri || file.path,
        type: file.type || 'image/jpeg',
        name: file.name || `attachment_${Date.now()}.jpg`,
      });

      const url = `${PRIMARY_BASE_URL}/hse-inspections/upload-attachment/${userId}`;
      
      const response = await fetch(url, {
        method: 'POST',
        body: formData,
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      });
      
      const data = await response.json();
      return data;
    } catch (error) {
      console.error('Error uploading attachment:', error);
      return { success: false, error: error.message };
    }
  },

  // Get current counter
  getCurrentCounter: async () => {
    try {
      const url = `${PRIMARY_BASE_URL}/hse-inspections/counter/current`;
      
      const response = await fetch(url);
      const data = await response.json();
      
      return data;
    } catch (error) {
      console.error('Error fetching counter:', error);
      return { success: false, error: error.message };
    }
  },

  // Increment counter
  incrementCounter: async () => {
    try {
      const url = `${PRIMARY_BASE_URL}/hse-inspections/counter/increment`;
      
      const response = await fetch(url, {
        method: 'POST',
      });
      
      const data = await response.json();
      return data;
    } catch (error) {
      console.error('Error incrementing counter:', error);
      return { success: false, error: error.message };
    }
  },

  // Get statistics
  getStatistics: async (userId, filters = {}) => {
    try {
      const { country, project, startDate, endDate } = filters;
      
      let url = `${PRIMARY_BASE_URL}/hse-inspections/${userId}/stats?`;
      if (country) url += `&country=${country}`;
      if (project) url += `&project=${project}`;
      if (startDate) url += `&startDate=${startDate}`;
      if (endDate) url += `&endDate=${endDate}`;
      
      const response = await fetch(url);
      const data = await response.json();
      
      return data;
    } catch (error) {
      console.error('Error fetching statistics:', error);
      return { success: false, error: error.message };
    }
  },

  // Search HSE inspections
  searchHSEInspections: async (userId, searchQuery, options = {}) => {
    try {
      const { page = 1, limit = 50 } = options;
      
      const url = `${PRIMARY_BASE_URL}/hse-inspections/${userId}/search?q=${encodeURIComponent(searchQuery)}&page=${page}&limit=${limit}`;
      
      const response = await fetch(url);
      const data = await response.json();
      
      return data;
    } catch (error) {
      console.error('Error searching HSE inspections:', error);
      return { success: false, error: error.message, data: [], pagination: { total: 0 } };
    }
  },
};

export default HSEInspectionAPI;

