const express = require('express');
const mongoose = require('mongoose');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const cors = require('cors');

const app = express();
const PORT = process.env.INSPECTION_PORT || 5004;

// Middleware
app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// MongoDB connection
const MONGODB_URI = process.env.MONGODB_URI || 'mongodb://titandrillingadminuser:mrdThsSnATwD231hbOPmnhj@api.titandrillingzm.com:27017/titan_drilling?authSource=admin';

mongoose.connect(MONGODB_URI, {
  useNewUrlParser: true,
  useUnifiedTopology: true,
})
.then(() => {
  console.log('✅ MongoDB connected successfully for Inspection Handler');
})
.catch((err) => {
  console.error('❌ MongoDB connection error:', err);
  process.exit(1);
});

// Inspection Report Schema
const inspectionReportSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  userId: { type: String, required: true },
  requestNumber: { type: String, required: true },
  status: { 
    type: String, 
    enum: ['draft', 'completed', 'archived'], 
    default: 'draft' 
  },
  
  // Step 1: Inspection Info
  step1: {
    id: String,
    requestNumber: String,
    country: String,
    project: String,
    inspector: String,
    selectedEquipment: {
      id: String,
      equipmentName: String,
      mainCategory: String,
      model: String,
      vehicleNumber: String,
      meterReading: Number,
      status: String
    },
    meterReading: Number,
    currentMeterReading: Number,
    selectedChecklistId: String,
    checklistTitle: String,
    coordinates: String,
    gpsAddress: String,
    createdAt: Date
  },
  
  // Step 2: Do Inspection
  step2: {
    selections: mongoose.Schema.Types.Mixed, // Object with item keys and their selected values
    selectedFiles: mongoose.Schema.Types.Mixed, // Object with file data
    numericInputs: mongoose.Schema.Types.Mixed, // Object with numeric input values
    notes: mongoose.Schema.Types.Mixed, // Object with notes
    sections: [{
      name: String,
      subItems: [{
        key: String,
        itemName: String,
        inputType: String,
        description: String,
        instruction: String,
        sortOrder: Number,
        isInspectionRequired: Boolean,
        category: String
      }]
    }],
    checklistTitle: String,
    headerInputValues: mongoose.Schema.Types.Mixed,
    footerInputValues: mongoose.Schema.Types.Mixed,
    faultyItems: Number,
    timestamp: Date
  },
  
  // Step 3: Summary
  step3: {
    faultyItems: Number,
    overallCondition: String,
    equipmentStatus: String,
    priority: String,
    safeToUse: String,
    additionalNote: String,
    signatureStatus: String,
    signatureData: mongoose.Schema.Types.Mixed,
    signatureDownloadUrl: String,
    signatureFields: [{
      key: String,
      label: String,
      roleName: String,
      required: Boolean,
      showInput: Boolean,
      showSignature: Boolean,
      nameKey: String,
      signatureKey: String
    }],
    signatureValues: mongoose.Schema.Types.Mixed,
    selectedSignatureType: String,
    selectedSignatureTypes: [String],
    individualSignatures: mongoose.Schema.Types.Mixed,
    individualSignatureStatus: mongoose.Schema.Types.Mixed,
    pdfDownloadUrl: String,
    createdAt: Date
  },
  
  // Corrective Action Data
  correctiveActionData: {
    subItemStatuses: mongoose.Schema.Types.Mixed, // Object with item keys and their status
    faultyCount: Number,
    isSatisfactory: Boolean,
    role: String, // 'Mechanic', 'Operator', 'Both'
    
    // Mechanic data
    mechanicName: String,
    mechanicSignatureUri: String,
    mechanicSignatureDownloadUrl: String,
    
    // Operator data
    operatorName: String,
    operatorSignatureUri: String,
    operatorSignatureDownloadUrl: String,
    
    // PDF data
    pdfDownloadUrl: String,
    
    // Metadata
    createdAt: { type: Date, default: Date.now },
    updatedAt: { type: Date, default: Date.now },
    createdBy: String,
    lastModifiedBy: String
  },
  
  // Metadata
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
  createdBy: String,
  lastModifiedBy: String
});

// Create indexes for better performances
inspectionReportSchema.index({ userId: 1, createdAt: -1 });
inspectionReportSchema.index({ requestNumber: 1 });
inspectionReportSchema.index({ 'step1.country': 1, 'step1.project': 1 });
inspectionReportSchema.index({ 'step1.inspector': 1 });
inspectionReportSchema.index({ status: 1 });

const InspectionReport = mongoose.model('InspectionReport', inspectionReportSchema, 'inspectionreports');

// Create a more flexible model for the /inspections/all endpoint
const FlexibleInspectionReport = mongoose.model('FlexibleInspectionReport', new mongoose.Schema({}, { strict: false }), 'inspectionreports');

// Inspection Checklist Schema
const inspectionChecklistSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  documentId: { type: String, required: true, unique: true },
  headerTitle: String,
  approvedBy: String,
  checklist: mongoose.Schema.Types.Mixed,
  documentName: String,
  documentPdfName: String,
  version: String,
  versionDate: String,
  header_inputs: mongoose.Schema.Types.Mixed,
  footer_inputs: mongoose.Schema.Types.Mixed,
  uploadedAt: { type: Date, default: Date.now },
  uploadedBy: String
}, { 
  strict: false  // Allow additional fields not defined in schema
});

// Create indexes for checklist
inspectionChecklistSchema.index({ documentId: 1 });
inspectionChecklistSchema.index({ headerTitle: 1 });

const InspectionChecklist = mongoose.model('InspectionChecklist', inspectionChecklistSchema);

// Corrective Action Schema
const correctiveActionSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  reportId: { type: String, required: true },
  userId: { type: String, required: true },
  
  // Step 1: Faulty Items Status
  subItemStatuses: mongoose.Schema.Types.Mixed, // Object with item keys and their status
  faultyCount: Number,
  
  // Step 2: Signature Data
  isSatisfactory: Boolean,
  role: String, // 'Mechanic', 'Operator', 'Both'
  
  // Mechanic data
  mechanicName: String,
  mechanicSignatureUri: String,
  mechanicSignatureDownloadUrl: String,
  
  // Operator data
  operatorName: String,
  operatorSignatureUri: String,
  operatorSignatureDownloadUrl: String,
  
  // PDF data
  pdfDownloadUrl: String,
  
  // Metadata
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
  createdBy: String,
  lastModifiedBy: String
});

// Create indexes for corrective actions
correctiveActionSchema.index({ reportId: 1 });
correctiveActionSchema.index({ userId: 1, createdAt: -1 });
correctiveActionSchema.index({ createdAt: -1 });

const CorrectiveAction = mongoose.model('CorrectiveAction', correctiveActionSchema, 'correctiveactions');

// Multer configuration for file uploads - Same as other backend services
const inspectionStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    // Store in temp directory first, then move to user-specific folder
    const tempDir = path.join(__dirname, 'uploads', 'temp');
    
    // Ensure temp directory exists
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
      console.log(`📁 Created temp directory: ${tempDir}`);
    }
    
    console.log(`📁 Storing file temporarily in: ${tempDir}`);
    cb(null, tempDir);
  },
  filename: function (req, file, cb) {
    // Generate unique filename with timestamp and original extension
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const fileExtension = path.extname(file.originalname);
    cb(null, file.fieldname + '-' + uniqueSuffix + fileExtension);
  }
});

const fileUpload = multer({ 
  storage: inspectionStorage,
  limits: { fileSize: 50 * 1024 * 1024 } // 50MB limit
});

// Routes

// Health check
app.get('/inspectionreports', (req, res) => {
  res.json({ 
    status: 'OK', 
    service: 'Inspection Handler',
    port: PORT,
    timestamp: new Date().toISOString()
  });
});

// Debug endpoint to check database connection and collections
app.get('/debug/db', async (req, res) => {
  try {
    const db = mongoose.connection.db;
    const collections = await db.listCollections().toArray();
    const inspectionReportsCount = await db.collection('inspectionreports').countDocuments({});
    const sampleDoc = await db.collection('inspectionreports').findOne({});
    
    res.json({
      success: true,
      database: mongoose.connection.name,
      connectionState: mongoose.connection.readyState,
      collections: collections.map(c => c.name),
      inspectionReportsCount,
      sampleDocument: sampleDoc ? {
        _id: sampleDoc._id,
        userId: sampleDoc.userId,
        requestNumber: sampleDoc.requestNumber
      } : null
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get total count of inspection reports (optimized endpoint)
app.get('/inspections/count', async (req, res) => {
  try {
    console.log('🔢 /inspections/count endpoint called');
    
    const db = mongoose.connection.db;
    const collection = db.collection('inspectionreports');
    
    console.log('📊 Database connection state:', mongoose.connection.readyState);
    console.log('📊 Database name:', mongoose.connection.name);
    console.log('📊 Collection name: inspectionreports');
    
    // Check if collection exists
    const collections = await db.listCollections({ name: 'inspectionreports' }).toArray();
    console.log('📊 Collections found:', collections.length);
    
    // Get total count only
    const total = await collection.countDocuments({});
    console.log('📊 Total documents in collection:', total);
    
    // Also try with mongoose model
    const modelCount = await InspectionReport.countDocuments({});
    console.log('📊 Model count:', modelCount);
    
    // Get a sample document to verify structure
    const sampleDoc = await collection.findOne({});
    console.log('📊 Sample document:', sampleDoc ? {
      _id: sampleDoc._id,
      userId: sampleDoc.userId,
      requestNumber: sampleDoc.requestNumber
    } : 'No documents found');
    
    res.json({
      success: true,
      total,
      modelCount,
      message: `Found ${total} inspection reports in database`,
      collectionExists: collections.length > 0,
      sampleDocument: sampleDoc ? {
        _id: sampleDoc._id,
        userId: sampleDoc.userId,
        requestNumber: sampleDoc.requestNumber
      } : null
    });
  } catch (error) {
    console.error('Error fetching inspection reports count:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection reports count: ' + error.message 
    });
  }
});

// Get all inspection reports for a user
app.get('/inspections/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    const { page = 1, limit = 50, status, country, project } = req.query;
    
    const query = { userId };
    if (status) query.status = status;
    if (country) query['step1.country'] = country;
    if (project) query['step1.project'] = project;
    
    const skip = (page - 1) * limit;
    
    const reports = await InspectionReport.find(query)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await InspectionReport.countDocuments(query);
    
    res.json({
      success: true,
      data: reports,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching inspection reports:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection reports' 
    });
  }
});

// Get all inspection reports from all users (for global request number generation)
app.get('/inspections/all', async (req, res) => {
  try {
    const { page = 1, limit = 10000 } = req.query;
    
    console.log('🔍 /inspections/all endpoint called with params:', { page, limit });
    
    // Get all documents without any pagination or filtering first
    const db = mongoose.connection.db;
    const collection = db.collection('inspectionreports');
    
    console.log('📊 Database connection state:', mongoose.connection.readyState);
    console.log('📊 Database name:', mongoose.connection.name);
    
    // Get total count
    const total = await collection.countDocuments({});
    console.log('📊 Total documents in collection:', total);
    
    // Get ALL documents first to see what we have
    const allReports = await collection.find({}).toArray();
    console.log('📊 All documents found:', allReports.length);
    
    if (allReports.length > 0) {
      console.log('📊 Sample document keys:', Object.keys(allReports[0]));
      console.log('📊 Sample document _id:', allReports[0]._id);
      console.log('📊 Sample document userId:', allReports[0].userId);
    }
    
    // Apply pagination manually
    const skip = (page - 1) * limit;
    const paginatedReports = allReports.slice(skip, skip + parseInt(limit));
    
    console.log('📊 Reports after pagination:', paginatedReports.length);
    
    // Get corrective actions for all reports to check status
    const reportIds = paginatedReports.map(report => report._id);
    const correctiveActions = {};
    
    try {
      const correctiveActionsData = await CorrectiveAction.find({ 
        reportId: { $in: reportIds } 
      }).select('reportId pdfDownloadUrl createdAt');
      
      correctiveActionsData.forEach(ca => {
        correctiveActions[ca.reportId] = {
          hasCorrectiveAction: true,
          pdfDownloadUrl: ca.pdfDownloadUrl,
          correctiveActionNumber: `CA-${ca.reportId}`,
          correctiveActionCreatedAt: ca.createdAt
        };
      });
    } catch (error) {
      console.error('Error fetching corrective actions:', error);
    }
    
    // Enhance reports with corrective action status
    const enhancedReports = paginatedReports.map(report => ({
      ...report,
      // Add corrective action status if exists
      ...(correctiveActions[report._id] || {})
    }));
    
    res.json({
      success: true,
      data: enhancedReports,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching all inspection reports:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch all inspection reports: ' + error.message 
    });
  }
});

// Get a specific inspection report by reportId only (for cross-user viewing)
app.get('/inspections/report/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    
    const report = await InspectionReport.findOne({ 
      _id: reportId 
    }).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report
    });
  } catch (error) {
    console.error('Error fetching inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection report' 
    });
  }
});

// Get a specific inspection report (original endpoint for user's own reports)
app.get('/inspections/:userId/:reportId', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    
    const report = await InspectionReport.findOne({ 
      _id: reportId, 
      userId 
    }).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report
    });
  } catch (error) {
    console.error('Error fetching inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection report' 
    });
  }
});

// Create a new inspection report
app.post('/inspections', async (req, res) => {
  try {
    const { userId, reportData } = req.body;
    
    // Debug logging (can be removed in production)
    console.log('Creating inspection report for userId:', userId);
    
    if (!userId || !reportData) {
      return res.status(400).json({ 
        success: false, 
        error: 'userId and reportData are required' 
      });
    }
    
    // Generate unique report ID if not provided
    const reportId = reportData._id || reportData.id || Date.now().toString();
    
    console.log('Creating inspection report with data:', {
      _id: reportId,
      userId,
      requestNumber: reportData.step1?.requestNumber || reportId,
      step1Keys: Object.keys(reportData.step1 || {}),
      step2Keys: Object.keys(reportData.step2 || {}),
      step3Keys: Object.keys(reportData.step3 || {})
    });

    const inspectionReport = new InspectionReport({
      _id: reportId,
      userId,
      requestNumber: reportData.step1?.requestNumber || reportId,
      status: 'draft',
      step1: reportData.step1 || {},
      step2: reportData.step2 || {},
      step3: reportData.step3 || {},
      createdBy: userId,
      lastModifiedBy: userId
    });
    
    console.log('Attempting to save inspection report...');
    await inspectionReport.save();
    console.log('Inspection report saved successfully');
    
    res.status(201).json({
      success: true,
      data: inspectionReport,
      message: 'Inspection report created successfully'
    });
  } catch (error) {
    console.error('Error creating inspection report:', error);
    console.error('Error details:', {
      message: error.message,
      name: error.name,
      code: error.code,
      stack: error.stack
    });
    res.status(500).json({ 
      success: false, 
      error: 'Failed to create inspection report: ' + error.message 
    });
  }
});

// Update an inspection report
app.put('/inspections/:userId/:reportId', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    const updateData = req.body;
    
    // Remove _id and userId from update data to prevent conflicts
    delete updateData._id;
    delete updateData.userId;
    
    updateData.updatedAt = new Date();
    updateData.lastModifiedBy = userId;
    
    const report = await InspectionReport.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report,
      message: 'Inspection report updated successfully'
    });
  } catch (error) {
    console.error('Error updating inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to update inspection report' 
    });
  }
});

// Update specific step of an inspection report
app.put('/inspections/:userId/:reportId/:step', async (req, res) => {
  try {
    const { userId, reportId, step } = req.params;
    const stepData = req.body;
    
    if (!['step1', 'step2', 'step3'].includes(step)) {
      return res.status(400).json({ 
        success: false, 
        error: 'Invalid step. Must be step1, step2, or step3' 
      });
    }
    
    const updateData = {
      [`${step}`]: stepData,
      updatedAt: new Date(),
      lastModifiedBy: userId
    };
    
    const report = await InspectionReport.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report,
      message: `${step} updated successfully`
    });
  } catch (error) {
    console.error(`Error updating ${req.params.step}:`, error);
    res.status(500).json({ 
      success: false, 
      error: `Failed to update ${req.params.step}` 
    });
  }
});

// Complete an inspection report
app.put('/inspections/:userId/:reportId/complete', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    const { pdfDownloadUrl, signatureDownloadUrl } = req.body;
    
    const updateData = {
      status: 'completed',
      updatedAt: new Date(),
      lastModifiedBy: userId
    };
    
    if (pdfDownloadUrl) {
      updateData['step3.pdfDownloadUrl'] = pdfDownloadUrl;
    }
    
    if (signatureDownloadUrl) {
      updateData['step3.signatureDownloadUrl'] = signatureDownloadUrl;
    }
    
    const report = await InspectionReport.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData },
      { new: true, runValidators: true }
    ).select('-__v');
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      data: report,
      message: 'Inspection report completed successfully'
    });
  } catch (error) {
    console.error('Error completing inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to complete inspection report' 
    });
  }
});

// Delete an inspection report
app.delete('/inspections/:userId/:reportId', async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    
    const report = await InspectionReport.findOneAndDelete({ 
      _id: reportId, 
      userId 
    });
    
    if (!report) {
      return res.status(404).json({ 
        success: false, 
        error: 'Inspection report not found' 
      });
    }
    
    res.json({
      success: true,
      message: 'Inspection report deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting inspection report:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete inspection report' 
    });
  }
});

// Get inspection statistics
app.get('/inspections/:userId/stats', async (req, res) => {
  try {
    const { userId } = req.params;
    const { country, project, startDate, endDate } = req.query;
    
    const matchQuery = { userId };
    if (country) matchQuery['step1.country'] = country;
    if (project) matchQuery['step1.project'] = project;
    if (startDate || endDate) {
      matchQuery.createdAt = {};
      if (startDate) matchQuery.createdAt.$gte = new Date(startDate);
      if (endDate) matchQuery.createdAt.$lte = new Date(endDate);
    }
    
    const stats = await InspectionReport.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: null,
          totalReports: { $sum: 1 },
          completedReports: {
            $sum: { $cond: [{ $eq: ['$status', 'completed'] }, 1, 0] }
          },
          draftReports: {
            $sum: { $cond: [{ $eq: ['$status', 'draft'] }, 1, 0] }
          },
          totalFaultyItems: { $sum: '$step3.faultyItems' },
          avgFaultyItems: { $avg: '$step3.faultyItems' }
        }
      }
    ]);
    
    const countryStats = await InspectionReport.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: '$step1.country',
          count: { $sum: 1 }
        }
      },
      { $sort: { count: -1 } }
    ]);
    
    const projectStats = await InspectionReport.aggregate([
      { $match: matchQuery },
      {
        $group: {
          _id: '$step1.project',
          count: { $sum: 1 }
        }
      },
      { $sort: { count: -1 } }
    ]);
    
    res.json({
      success: true,
      data: {
        overview: stats[0] || {
          totalReports: 0,
          completedReports: 0,
          draftReports: 0,
          totalFaultyItems: 0,
          avgFaultyItems: 0
        },
        byCountry: countryStats,
        byProject: projectStats
      }
    });
  } catch (error) {
    console.error('Error fetching inspection statistics:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch inspection statistics' 
    });
  }
});

// Upload file for inspection report
app.post('/inspections/:userId/:reportId/upload', fileUpload.single('file'), async (req, res) => {
  try {
    const { userId, reportId } = req.params;
    const { step, itemKey } = req.body;
    
    if (!req.file) {
      return res.status(400).json({ 
        success: false, 
        error: 'No file uploaded' 
      });
    }
    
    const fileUrl = `/inspections/files/${req.file.filename}`;
    
    // Update the report with file information
    const updateData = {
      [`${step}.selectedFiles.${itemKey}`]: {
        filename: req.file.filename,
        originalName: req.file.originalname,
        url: fileUrl,
        uploadedAt: new Date()
      },
      updatedAt: new Date(),
      lastModifiedBy: userId
    };
    
    await InspectionReport.findOneAndUpdate(
      { _id: reportId, userId },
      { $set: updateData }
    );
    
    res.json({
      success: true,
      data: {
        filename: req.file.filename,
        originalName: req.file.originalname,
        url: fileUrl,
        uploadedAt: new Date()
      },
      message: 'File uploaded successfully'
    });
  } catch (error) {
    console.error('Error uploading file:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to upload file' 
    });
  }
});

// Serve uploaded files
app.use('/inspections/files', express.static(path.join(__dirname, 'uploads', 'inspections')));

// Search inspection reports
app.get('/inspections/:userId/search', async (req, res) => {
  try {
    const { userId } = req.params;
    const { q, page = 1, limit = 50 } = req.query;
    
    if (!q) {
      return res.status(400).json({ 
        success: false, 
        error: 'Search query is required' 
      });
    }
    
    const searchQuery = {
      userId,
      $or: [
        { requestNumber: { $regex: q, $options: 'i' } },
        { 'step1.inspector': { $regex: q, $options: 'i' } },
        { 'step1.country': { $regex: q, $options: 'i' } },
        { 'step1.project': { $regex: q, $options: 'i' } },
        { 'step1.selectedEquipment.equipmentName': { $regex: q, $options: 'i' } }
      ]
    };
    
    const skip = (page - 1) * limit;
    
    const reports = await InspectionReport.find(searchQuery)
      .sort({ createdAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await InspectionReport.countDocuments(searchQuery);
    
    res.json({
      success: true,
      data: reports,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error searching inspection reports:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to search inspection reports' 
    });
  }
});

// ========== FILE UPLOAD ENDPOINTS ==========

// Upload PDF to local storage
app.post('/inspections/upload-pdf/:userId', fileUpload.single('pdf'), async (req, res) => {
  try {
    console.log('📤 PDF Upload Request received:', {
      userId: req.params.userId,
      hasFile: !!req.file,
      fileInfo: req.file ? {
        originalname: req.file.originalname,
        mimetype: req.file.mimetype,
        size: req.file.size,
        path: req.file.path
      } : null,
      headers: req.headers
    });
    
    if (!req.file) {
      console.log('❌ No PDF file in request');
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    const userId = req.params.userId;
    const fileName = `inspection_${userId}_${Date.now()}.pdf`;
    
    // Create user-specific directory
    const userDir = path.join(__dirname, 'uploads', 'documents', 'inspectionreports', userId);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to user directory:', moveError);
      // If move fails, use the temp path
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/inspections/download-file/${userId}/${fileName}`,
      message: 'PDF uploaded successfully'
    });

  } catch (error) {
    console.error('PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload PDF: ' + error.message
    });
  }
});

// Upload signature image to local storage
app.post('/inspections/upload-signature/:userId', fileUpload.single('signature'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No signature image uploaded'
      });
    }

    const userId = req.params.userId;
    const fileName = `signature_${userId}_${Date.now()}.png`;
    
    // Create user-specific directory
    const userDir = path.join(__dirname, 'uploads', 'documents', 'inspectionreports', userId);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to user directory:', moveError);
      // If move fails, use the temp path
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/inspections/download-file/${userId}/${fileName}`,
      message: 'Signature uploaded successfully'
    });

  } catch (error) {
    console.error('Signature upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload signature: ' + error.message
    });
  }
});

// Download file from local storage
app.get('/inspections/download-file/:userId/:fileName', (req, res) => {
  try {
    const { userId, fileName } = req.params;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'inspectionreports', userId, fileName);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'File not found'
      });
    }

    // Set appropriate headers
    const ext = path.extname(fileName).toLowerCase();
    let contentType = 'application/octet-stream';
    
    if (ext === '.pdf') {
      contentType = 'application/pdf';
    } else if (ext === '.png') {
      contentType = 'image/png';
    } else if (ext === '.jpg' || ext === '.jpeg') {
      contentType = 'image/jpeg';
    }
    
    res.set({
      'Content-Type': contentType,
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('File download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download file'
      });
    });

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download file: ' + error.message
    });
  }
});

// ========== CHECKLIST ENDPOINTS ==========

// Test endpoint
app.post('/checklists/test', (req, res) => {
  console.log('🧪 TEST ENDPOINT CALLED');
  res.json({ success: true, message: 'Test endpoint works' });
});

// Upload all checklists from JSON data
app.post('/checklists/upload-all', async (req, res) => {
  console.log('🚀 UPLOAD ENDPOINT CALLED');
  try {
    const { checklistsData, uploadedBy = 'system' } = req.body;
    console.log('📦 Received data keys:', Object.keys(checklistsData || {}));
    
    if (!checklistsData || typeof checklistsData !== 'object') {
      console.log('❌ Invalid checklists data');
      return res.status(400).json({ 
        success: false, 
        error: 'Checklists data is required' 
      });
    }
    
    // Debug: Check first checklist structure
    const firstKey = Object.keys(checklistsData)[0];
    const firstChecklist = checklistsData[firstKey];
    console.log('🔍 First checklist key:', firstKey);
    console.log('🔍 First checklist header_inputs:', JSON.stringify(firstChecklist.header_inputs));
    console.log('🔍 First checklist footer_inputs:', JSON.stringify(firstChecklist.footer_inputs));
    
    const results = [];
    const errors = [];
    
    // Get MongoDB collection
    const collection = mongoose.connection.db.collection('inspectionchecklists');
    
    for (const [documentId, checklistData] of Object.entries(checklistsData)) {
      try {
        // Delete existing if it exists
        await collection.deleteOne({ documentId });
        
        // Create document with ALL fields including header_inputs and footer_inputs
        const document = {
          _id: documentId,
          documentId: documentId,
          headerTitle: checklistData.Header_title || checklistData.headerTitle,
          approvedBy: checklistData.approved_by || checklistData.approvedBy,
          checklist: checklistData.checklist,
          documentName: checklistData.document_name || checklistData.documentName,
          documentPdfName: checklistData.document_pdf_name || checklistData.documentPdfName,
          version: checklistData.version,
          versionDate: checklistData.version_date || checklistData.versionDate,
          header_inputs: checklistData.header_inputs || {},
          footer_inputs: checklistData.footer_inputs || {},
          uploadedBy: uploadedBy,
          uploadedAt: new Date()
        };
        
        // Insert the document
        const result = await collection.insertOne(document);
        
        // Verify the inserted document
        const insertedDoc = await collection.findOne({ _id: documentId });
        
        results.push({
          documentId,
          action: 'created',
          data: insertedDoc
        });
      } catch (error) {
        errors.push({
          documentId,
          error: error.message
        });
      }
    }
    
    console.log('✅ Upload completed successfully');
    res.json({
      success: true,
      message: `Processed ${results.length} checklists successfully`,
      data: {
        processed: results.length,
        errors: errors.length,
        results,
        errors
      }
    });
  } catch (error) {
    console.error('❌ Error uploading checklists:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to upload checklists: ' + error.message 
    });
  }
});

// Get all checklists
app.get('/checklists', async (req, res) => {
  try {
    const { page = 1, limit = 50, search } = req.query;
    
    const query = {};
    if (search) {
      query.$or = [
        { documentId: { $regex: search, $options: 'i' } },
        { headerTitle: { $regex: search, $options: 'i' } },
        { documentName: { $regex: search, $options: 'i' } }
      ];
    }
    
    const skip = (page - 1) * limit;
    
    const checklists = await InspectionChecklist.find(query)
      .sort({ uploadedAt: -1 })
      .skip(skip)
      .limit(parseInt(limit))
      .select('-__v');
    
    const total = await InspectionChecklist.countDocuments(query);
    
    res.json({
      success: true,
      data: checklists,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    console.error('Error fetching checklists:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch checklists' 
    });
  }
});

// Get a specific checklist
app.get('/checklists/:documentId', async (req, res) => {
  try {
    const { documentId } = req.params;
    
    const checklist = await InspectionChecklist.findOne({ documentId }).select('-__v');
    
    if (!checklist) {
      return res.status(404).json({ 
        success: false, 
        error: 'Checklist not found' 
      });
    }
    
    res.json({
      success: true,
      data: checklist
    });
  } catch (error) {
    console.error('Error fetching checklist:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch checklist' 
    });
  }
});

// Delete a checklist
app.delete('/checklists/:documentId', async (req, res) => {
  try {
    const { documentId } = req.params;
    
    const checklist = await InspectionChecklist.findOneAndDelete({ documentId });
    
    if (!checklist) {
      return res.status(404).json({ 
        success: false, 
        error: 'Checklist not found' 
      });
    }
    
    res.json({
      success: true,
      message: 'Checklist deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting checklist:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete checklist' 
    });
  }
});

// ========== CORRECTIVE ACTION ENDPOINTS ==========

// Get corrective action by report ID
app.get('/corrective-actions/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    
    const correctiveAction = await CorrectiveAction.findOne({ reportId }).select('-__v');
    
    if (!correctiveAction) {
      return res.status(404).json({ 
        success: false, 
        error: 'Corrective action not found' 
      });
    }
    
    res.json({
      success: true,
      data: correctiveAction
    });
  } catch (error) {
    console.error('Error fetching corrective action:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch corrective action' 
    });
  }
});

// Create or update corrective action
app.post('/corrective-actions', async (req, res) => {
  console.log('🔧 CORRECTIVE ACTIONS POST ENDPOINT CALLED!');
  console.log('🔧 Request body:', req.body);
  try {
    const { reportId, userId, correctiveActionData } = req.body;
    
    if (!reportId || !userId || !correctiveActionData) {
      return res.status(400).json({ 
        success: false, 
        error: 'reportId, userId, and correctiveActionData are required' 
      });
    }
    
    // Generate unique corrective action ID if not provided
    const correctiveActionId = correctiveActionData._id || `ca_${reportId}_${Date.now()}`;
    
    console.log('Creating/updating corrective action:', {
      _id: correctiveActionId,
      reportId,
      userId,
      hasSubItemStatuses: !!correctiveActionData.subItemStatuses,
      hasPdfUrl: !!correctiveActionData.pdfDownloadUrl
    });

    // Check if corrective action already exists
    const existingAction = await CorrectiveAction.findOne({ reportId });
    
    let correctiveAction;
    if (existingAction) {
      // Update existing
      const updateData = {
        ...correctiveActionData,
        updatedAt: new Date(),
        lastModifiedBy: userId
      };
      
      correctiveAction = await CorrectiveAction.findOneAndUpdate(
        { reportId },
        { $set: updateData },
        { new: true, runValidators: true }
      ).select('-__v');
      
      console.log('Updated existing corrective action');
    } else {
      // Create new
      correctiveAction = new CorrectiveAction({
        _id: correctiveActionId,
        reportId,
        userId,
        ...correctiveActionData,
        createdBy: userId,
        lastModifiedBy: userId
      });
      
      await correctiveAction.save();
      console.log('Created new corrective action');
    }
    
    // Also update the inspection report with correctiveActionData
    try {
      const inspectionReport = await InspectionReport.findOne({ _id: reportId });
      if (inspectionReport) {
        const correctiveDataForReport = {
          subItemStatuses: correctiveActionData.subItemStatuses,
          faultyCount: correctiveActionData.faultyCount,
          isSatisfactory: correctiveActionData.isSatisfactory,
          role: correctiveActionData.role,
          mechanicName: correctiveActionData.mechanicName,
          mechanicSignatureUri: correctiveActionData.mechanicSignatureUri,
          mechanicSignatureDownloadUrl: correctiveActionData.mechanicSignatureDownloadUrl,
          operatorName: correctiveActionData.operatorName,
          operatorSignatureUri: correctiveActionData.operatorSignatureUri,
          operatorSignatureDownloadUrl: correctiveActionData.operatorSignatureDownloadUrl,
          pdfDownloadUrl: correctiveActionData.pdfDownloadUrl,
          updatedAt: new Date(),
          lastModifiedBy: userId
        };
        
        // Add createdAt if this is a new corrective action
        if (!existingAction) {
          correctiveDataForReport.createdAt = new Date();
          correctiveDataForReport.createdBy = userId;
        }
        
        await InspectionReport.findOneAndUpdate(
          { _id: reportId },
          { 
            $set: { 
              correctiveActionData: correctiveDataForReport,
              updatedAt: new Date(),
              lastModifiedBy: userId
            } 
          }
        );
        
        console.log('✅ Updated inspection report with correctiveActionData');
      } else {
        console.warn('⚠️ Inspection report not found for corrective action data update');
      }
    } catch (reportUpdateError) {
      console.error('❌ Error updating inspection report with corrective action data:', reportUpdateError);
      // Don't fail the entire operation if report update fails
    }
    
    res.status(201).json({
      success: true,
      data: correctiveAction,
      message: 'Corrective action saved successfully'
    });
  } catch (error) {
    console.error('Error saving corrective action:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to save corrective action: ' + error.message 
    });
  }
});

// Update corrective action step (for step-by-step updates)
app.put('/corrective-actions/:reportId/step/:stepNumber', async (req, res) => {
  try {
    const { reportId, stepNumber } = req.params;
    const stepData = req.body;
    
    if (!['1', '2'].includes(stepNumber)) {
      return res.status(400).json({ 
        success: false, 
        error: 'Invalid step number. Must be 1 or 2' 
      });
    }
    
    const updateData = {
      ...stepData,
      updatedAt: new Date()
    };
    
    const correctiveAction = await CorrectiveAction.findOneAndUpdate(
      { reportId },
      { $set: updateData },
      { new: true, runValidators: true, upsert: true }
    ).select('-__v');
    
    res.json({
      success: true,
      data: correctiveAction,
      message: `Step ${stepNumber} updated successfully`
    });
  } catch (error) {
    console.error(`Error updating corrective action step ${req.params.stepNumber}:`, error);
    res.status(500).json({ 
      success: false, 
      error: `Failed to update step ${req.params.stepNumber}` 
    });
  }
});

// Upload corrective action PDF
app.post('/corrective-actions/:reportId/upload-pdf', fileUpload.single('pdf'), async (req, res) => {
  try {
    const { reportId } = req.params;
    const { userId } = req.body;
    
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    // Create user-specific directory - correctiveactionreports/uid structure
    const userDir = path.join(__dirname, 'uploads', 'documents', 'correctiveactionreports', userId || 'unknown');
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created corrective action directory: ${userDir}`);
    }
    
    // Use the filename from the uploaded file if available, otherwise generate one
    let fileName = req.file.originalname;
    if (!fileName || !fileName.endsWith('.pdf')) {
      const timestamp = Date.now();
      fileName = `corrective_action_${reportId}_${timestamp}.pdf`;
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 Corrective action PDF moved to: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving corrective action PDF:', moveError);
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/corrective-actions/download-pdf/${fileName}?userId=${userId}`,
      message: 'Corrective action PDF uploaded successfully'
    });

  } catch (error) {
    console.error('Corrective action PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload corrective action PDF: ' + error.message
    });
  }
});

// Upload corrective action signature
app.post('/corrective-actions/:reportId/upload-signature', fileUpload.single('signature'), async (req, res) => {
  try {
    const { reportId } = req.params;
    const { signatureType, userId } = req.body; // 'mechanic' or 'operator'
    
    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No signature image uploaded'
      });
    }

    if (!signatureType || !['mechanic', 'operator'].includes(signatureType)) {
      return res.status(400).json({
        success: false,
        error: 'signatureType must be "mechanic" or "operator"'
      });
    }

    // Use the filename from the uploaded file if available, otherwise generate one
    let fileName = req.file.originalname;
    if (!fileName || !fileName.endsWith('.png')) {
      fileName = `signature_${reportId}_${signatureType}_${Date.now()}.png`;
    }
    
    // Create user-specific directory - correctiveactionreports/uid structure
    const userDir = path.join(__dirname, 'uploads', 'documents', 'correctiveactionreports', userId || 'unknown');
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created corrective action directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalPath = path.join(userDir, fileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 Corrective action signature moved to: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving corrective action signature:', moveError);
      const tempPath = req.file.path;
      finalPath = tempPath;
    }
    
    res.json({
      success: true,
      fileName: fileName,
      filePath: finalPath,
      downloadUrl: `/corrective-actions/download-signature/${fileName}?userId=${userId}`,
      signatureType: signatureType,
      message: 'Corrective action signature uploaded successfully'
    });

  } catch (error) {
    console.error('Corrective action signature upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path); // Clean up temp file
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload corrective action signature: ' + error.message
    });
  }
});

// Download corrective action PDF
app.get('/corrective-actions/download-pdf/:fileName', (req, res) => {
  try {
    const { fileName } = req.params;
    const { userId } = req.query;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'correctiveactionreports', userId || 'unknown', fileName);
    
    console.log('🔍 Looking for PDF file at:', filePath);
    console.log('🔍 User ID:', userId);
    console.log('🔍 File name:', fileName);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      console.log('❌ File not found at:', filePath);
      return res.status(404).json({
        success: false,
        error: 'PDF file not found'
      });
    }
    
    console.log('✅ File found at:', filePath);

    // Set appropriate headers
    res.set({
      'Content-Type': 'application/pdf',
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('Corrective action PDF download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download PDF'
      });
    });

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download PDF: ' + error.message
    });
  }
});

// Download corrective action signature
app.get('/corrective-actions/download-signature/:fileName', (req, res) => {
  try {
    const { fileName } = req.params;
    const { userId } = req.query;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'correctiveactionreports', userId || 'unknown', fileName);
    
    // Check if file exists
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'Signature file not found'
      });
    }

    // Set appropriate headers
    res.set({
      'Content-Type': 'image/png',
      'Content-Disposition': `attachment; filename="${fileName}"`
    });

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

    fileStream.on('error', (error) => {
      console.error('Corrective action signature download error:', error);
      res.status(500).json({
        success: false,
        error: 'Failed to download signature'
      });
    });

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to download signature: ' + error.message
    });
  }
});

// Delete corrective action
app.delete('/corrective-actions/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    
    const correctiveAction = await CorrectiveAction.findOneAndDelete({ reportId });
    
    if (!correctiveAction) {
      return res.status(404).json({ 
        success: false, 
        error: 'Corrective action not found' 
      });
    }
    
    res.json({
      success: true,
      message: 'Corrective action deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting corrective action:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to delete corrective action' 
    });
  }
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('Unhandled error:', error);
  res.status(500).json({ 
    success: false, 
    error: 'Internal server error' 
  });
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({ 
    success: false, 
    error: 'Endpoint not found' 
  });
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`🔍 Inspection Handler server running on port ${PORT}`);
  console.log(`📊 Health check: http://localhost:${PORT}/inspectionreports`);
  console.log(`📊 External access: https://api.titandrillingzm.com:${PORT}/inspectionreports`);
  console.log(`🔢 Total count: https://api.titandrillingzm.com:${PORT}/inspections/count`);
  console.log(`📁 PDF upload: https://api.titandrillingzm.com:${PORT}/inspections/upload-pdf/:userId`);
  console.log(`📁 Signature upload: https://api.titandrillingzm.com:${PORT}/inspections/upload-signature/:userId`);
  console.log(`📁 File download: https://api.titandrillingzm.com:${PORT}/inspections/download-file/:userId/:fileName`);
  console.log(`📋 Checklist upload: https://api.titandrillingzm.com:${PORT}/checklists/upload-all`);
  console.log(`🔍 Get report by ID: https://api.titandrillingzm.com:${PORT}/inspections/report/:reportId`);
  console.log(`🔧 Corrective Actions:`);
  console.log(`   📋 Get corrective action: https://api.titandrillingzm.com:${PORT}/corrective-actions/:reportId`);
  console.log(`   💾 Save corrective action: https://api.titandrillingzm.com:${PORT}/corrective-actions`);
  console.log(`   📁 Upload corrective PDF: https://api.titandrillingzm.com:${PORT}/corrective-actions/:reportId/upload-pdf`);
  console.log(`   ✍️ Upload signature: https://api.titandrillingzm.com:${PORT}/corrective-actions/:reportId/upload-signature`);
  console.log(`   📄 Download PDF: https://api.titandrillingzm.com:${PORT}/corrective-actions/download-pdf/:fileName`);
  console.log(`   🖋️ Download signature: https://api.titandrillingzm.com:${PORT}/corrective-actions/download-signature/:fileName`);
});

module.exports = app;
