const express = require("express");
const mongoose = require('mongoose');
const cors = require("cors");
const http = require("http");
const { Server } = require("socket.io");
require('dotenv').config();

// MongoDB connection string from environment variables
const MONGO_HOST = process.env.MONGO_HOST || '168.231.113.219';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'mrdThsSnATwD231hbOPmnhj';
const MONGO_DATABASE = process.env.MONGO_DATABASE || 'titan_drilling';
const MONGO_AUTH_SOURCE = process.env.MONGO_AUTH_SOURCE || 'admin';

const MONGO_URI = `mongodb://${MONGO_USER}:${MONGO_PASSWORD}@${MONGO_HOST}:${MONGO_PORT}/${MONGO_DATABASE}?authSource=${MONGO_AUTH_SOURCE}`;

// Server configuration
const PORT = process.env.EQUIPMENT_PORT || 5001;
const SERVER_IP = process.env.SERVER_IP || '168.231.113.219';
const CORS_ORIGIN = process.env.CORS_ORIGIN || '*';
const BODY_PARSER_LIMIT = process.env.BODY_PARSER_LIMIT || '100mb';

// Create Express app and HTTP server
const app = express();
const server = http.createServer(app);

// Create Socket.IO server
const io = new Server(server, {
  cors: {
    origin: CORS_ORIGIN,
    methods: ['GET', 'POST', 'PUT', 'DELETE'],
    credentials: true,
  },
  transports: ['websocket', 'polling'],
});

// Middleware
app.use(express.json({ limit: BODY_PARSER_LIMIT }));
app.use(express.urlencoded({ limit: BODY_PARSER_LIMIT, extended: true }));
app.use(cors({
  origin: CORS_ORIGIN,
  methods: ['GET', 'POST', 'PUT', 'DELETE'],
  allowedHeaders: ['Content-Type'],
}));

// Socket.IO Connection Handler
io.on('connection', (socket) => {
  console.log('✅ Client connected:', socket.id);
  
  socket.on('disconnect', () => {
    console.log('❌ Client disconnected:', socket.id);
  });
});

// Connect to MongoDB
mongoose.connect(MONGO_URI, { useNewUrlParser: true, useUnifiedTopology: true })
  .then(() => console.log("✅ MongoDB connected successfully from GlobalEquipmentJsonData!"))
  .catch((err) => console.error("❌ MongoDB connection error:", err));

// Equipment Schema
const EquipmentSchema = new mongoose.Schema({
  id: String,
  equipmentName: String,
  mainCategory: String,
  category: String,
  equipmentType: String,
  status: String,
  country: String,
  firebaseKey: String,
  make: String,
  model: String,
  year: String,
  serialNumber: String,
  location: String,
  description: String,
  specifications: mongoose.Schema.Types.Mixed,
  customName: String,
  databaseName: String,
  uploadedAt: { type: Date, default: Date.now },
  uploadedFrom: { type: String, default: "React Native App" },
  mongoUploadVersion: { type: String, default: "1.0.0" },
  firebaseData: mongoose.Schema.Types.Mixed,
}, { 
  strict: false,
  timestamps: true 
});

const Equipment = mongoose.model("Equipment", EquipmentSchema);

// Sample equipment data (you can replace this with your actual data)
const sampleEquipmentData = [
  {
    id: "1",
    equipmentName: "Sample Equipment 1",
    mainCategory: "Heavy Equipment",
    category: "Excavator",
    equipmentType: "CAT 320",
    status: "Active",
    country: "DRC",
    make: "Caterpillar",
    model: "320",
    year: "2020",
    serialNumber: "CAT320001",
    location: "Site A",
    description: "Heavy duty excavator",
    databaseName: "titan_drilling",
    uploadedAt: new Date(),
    uploadedFrom: "Auto-upload on server start",
    mongoUploadVersion: "1.0.0"
  },
  {
    id: "2", 
    equipmentName: "Sample Equipment 2",
    mainCategory: "Light Vehicle",
    category: "Truck",
    equipmentType: "Toyota Hilux",
    status: "Active",
    country: "DRC",
    make: "Toyota",
    model: "Hilux",
    year: "2021",
    serialNumber: "HILUX001",
    location: "Site B",
    description: "Light utility vehicle",
    databaseName: "titan_drilling",
    uploadedAt: new Date(),
    uploadedFrom: "Auto-upload on server start",
    mongoUploadVersion: "1.0.0"
  }
];

// Function to auto-populate equipment data
const autoPopulateEquipment = async () => {
  try {
    console.log("🔄 Starting auto-population of equipment data...");
    
    // Check if data already exists
    const existingCount = await Equipment.countDocuments();
    if (existingCount > 0) {
      console.log(`✅ Equipment data already exists (${existingCount} items). Skipping auto-population.`);
      return;
    }

    // Insert sample data
    const result = await Equipment.insertMany(sampleEquipmentData);
    console.log(`✅ Auto-populated ${result.length} equipment items successfully!`);
    
  } catch (error) {
    console.error("❌ Error auto-populating equipment data:", error);
  }
};

// Function to get equipment data
const getEquipmentData = async (req, res) => {
  try {
    const equipment = await Equipment.find().sort({ uploadedAt: -1 });
    res.json({
      success: true,
      message: "Equipment data retrieved successfully",
      data: equipment,
      count: equipment.length
    });
  } catch (error) {
    console.error("❌ Error fetching equipment data:", error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
};

// Function to add equipment data
const addEquipmentData = async (req, res) => {
  try {
    const { equipmentData } = req.body;
    
    if (!equipmentData || !Array.isArray(equipmentData)) {
      return res.status(400).json({
        success: false,
        error: "equipmentData must be an array"
      });
    }

    const equipmentWithMetadata = equipmentData.map(item => ({
      ...item,
      databaseName: "titan_drilling",
      uploadedAt: new Date(),
      uploadedFrom: "React Native App",
      mongoUploadVersion: "1.0.0"
    }));

    const result = await Equipment.insertMany(equipmentWithMetadata, { ordered: false });
    
    res.json({
      success: true,
      message: `Successfully added ${result.length} equipment items`,
      count: result.length,
      data: result
    });
  } catch (error) {
    console.error("❌ Error adding equipment data:", error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
};

// Function to clear all equipment data
const clearEquipmentData = async (req, res) => {
  try {
    const result = await Equipment.deleteMany({});
    res.json({
      success: true,
      message: `Cleared ${result.deletedCount} equipment items`,
      deletedCount: result.deletedCount
    });
  } catch (error) {
    console.error("❌ Error clearing equipment data:", error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
};

// Get equipment count
const getEquipmentCount = async (req, res) => {
  try {
    const count = await Equipment.countDocuments();
    res.json({ count });
  } catch (err) {
    console.error("❌ Error fetching count:", err);
    res.status(500).json({ error: err.message });
  }
};

// Get all equipment (old path)
const getAllEquipment = async (req, res) => {
  try {
    const equipment = await Equipment.find().sort({ uploadedAt: -1 });
    res.json(equipment);
  } catch (err) {
    console.error("❌ Error fetching equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Get equipment by database name
const getEquipmentByDatabase = async (req, res) => {
  try {
    const { databaseName } = req.params;
    const equipment = await Equipment.find({ databaseName }).sort({ uploadedAt: -1 });
    res.json(equipment);
  } catch (err) {
    console.error("❌ Error fetching equipment by database:", err);
    res.status(500).json({ error: err.message });
  }
};

// Get equipment by ID
const getEquipmentById = async (req, res) => {
  try {
    const equipment = await Equipment.findOne({ id: req.params.id });
    if (!equipment) {
      return res.status(404).json({ error: "Equipment not found" });
    }
    res.json(equipment);
  } catch (err) {
    console.error("❌ Error fetching equipment by ID:", err);
    res.status(500).json({ error: err.message });
  }
};

// Add single equipment
const addSingleEquipment = async (req, res) => {
  try {
    const equipmentData = {
      ...req.body,
      uploadedAt: new Date(),
      uploadedFrom: req.body.uploadedFrom || "React Native App",
      mongoUploadVersion: "1.0.0",
      databaseName: req.body.databaseName || "titan_drilling",
    };
    
    const newEquipment = new Equipment(equipmentData);
    await newEquipment.save();
    
    console.log(`✅ Equipment added: ${equipmentData.equipmentName || equipmentData.id}`);
    
    // Emit real-time update via Socket.IO
    io.emit('equipment:added', newEquipment);
    
    res.json({ 
      message: "✅ Equipment added successfully!", 
      data: newEquipment 
    });
  } catch (err) {
    console.error("❌ Error adding equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Update equipment
const updateEquipment = async (req, res) => {
  try {
    const equipment = await Equipment.findOneAndUpdate(
      { id: req.params.id },
      { ...req.body, updatedAt: new Date() },
      { new: true }
    );
    if (!equipment) {
      return res.status(404).json({ error: "Equipment not found" });
    }
    
    // Emit real-time update via Socket.IO
    io.emit('equipment:updated', equipment);
    
    res.json({ message: "✅ Equipment updated!", data: equipment });
  } catch (err) {
    console.error("❌ Error updating equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Delete equipment by ID
const deleteEquipmentById = async (req, res) => {
  try {
    const equipment = await Equipment.findOneAndDelete({ id: req.params.id });
    if (!equipment) {
      return res.status(404).json({ error: "Equipment not found" });
    }
    
    // Emit real-time delete via Socket.IO
    io.emit('equipment:deleted', { id: req.params.id });
    
    res.json({ message: "✅ Equipment deleted!" });
  } catch (err) {
    console.error("❌ Error deleting equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Clear all equipment (old path)
const clearAllEquipment = async (req, res) => {
  try {
    const result = await Equipment.deleteMany({});
    res.json({ 
      message: `✅ Cleared ${result.deletedCount} equipment items`,
      deletedCount: result.deletedCount 
    });
  } catch (err) {
    console.error("❌ Error clearing equipment:", err);
    res.status(500).json({ error: err.message });
  }
};

// Bulk upload equipment data (old path)
const bulkUploadEquipment = async (req, res) => {
  try {
    console.log("📥 Received bulk upload request with body size:", Buffer.byteLength(JSON.stringify(req.body)));
    const { equipmentData, databaseName } = req.body;
    
    if (!equipmentData || !Array.isArray(equipmentData)) {
      return res.status(400).json({ error: "equipmentData must be an array" });
    }

    const equipmentWithMetadata = equipmentData.map(item => ({
      ...item,
      databaseName: databaseName || "titan_drilling",
      uploadedAt: new Date(),
      uploadedFrom: "React Native App",
      mongoUploadVersion: "1.0.0",
    }));

    console.log(`📦 Processing ${equipmentWithMetadata.length} items for upload`);
    const result = await Equipment.insertMany(equipmentWithMetadata, { ordered: false });
    
    console.log(`✅ Bulk uploaded ${result.length} equipment items`);
    res.json({ 
      message: `✅ Successfully uploaded ${result.length} equipment items`,
      count: result.length,
      data: result 
    });
  } catch (err) {
    console.error("❌ Error bulk uploading equipment:", err);
    
    if (err.name === 'PayloadTooLargeError') {
      res.status(413).json({ error: "Request entity too large. Please reduce the data size or increase the server limit." });
    } else if (err.writeErrors) {
      const successCount = err.insertedDocs?.length || 0;
      console.log(`⚠️ Partial success: ${successCount} items uploaded, ${err.writeErrors.length} duplicates skipped`);
      res.json({
        message: `⚠️ Uploaded ${successCount} items, ${err.writeErrors.length} duplicates skipped`,
        successCount,
        errorCount: err.writeErrors.length,
      });
    } else {
      res.status(500).json({ error: err.message });
    }
  }
};

// Get list of databases
const getDatabases = async (req, res) => {
  try {
    if (mongoose.connection.readyState !== 1) {
      throw new Error("MongoDB connection not ready");
    }
    const admin = mongoose.connection.db.admin();
    try {
      const { databases } = await admin.listDatabases();
      const dbNames = databases.map(db => db.name);
      console.log("✅ Databases retrieved:", dbNames);
      res.json({ databases: dbNames });
    } catch (dbErr) {
      console.error("❌ Error in listDatabases:", dbErr);
      throw new Error("Insufficient permissions to list databases");
    }
  } catch (err) {
    console.error("❌ Error fetching databases:", err);
    res.status(500).json({ error: `Failed to fetch databases: ${err.message}` });
  }
};

// Delete equipment from specific database
const clearDatabaseEquipment = async (req, res) => {
  try {
    const { databaseName } = req.params;
    const result = await Equipment.deleteMany({ databaseName });
    res.json({ 
      message: `✅ Cleared ${result.deletedCount} equipment items from ${databaseName}`,
      deletedCount: result.deletedCount 
    });
  } catch (err) {
    console.error("❌ Error clearing database:", err);
    res.status(500).json({ error: err.message });
  }
};

// ===== ROUTES SETUP =====

// Health check
app.get("/", (req, res) => {
  res.json({ message: "✅ Server is running and MongoDB connected!" });
});

// NEW PATHS (GlobalEquipmentJsonData)
app.get("/EquipmentsDataJSON/GlobalEquipmentJsonData", getEquipmentData);
app.post("/EquipmentsDataJSON/GlobalEquipmentJsonData", addEquipmentData);
app.delete("/EquipmentsDataJSON/GlobalEquipmentJsonData", clearEquipmentData);

// OLD PATHS (for compatibility)
app.get("/data", getAllEquipment);
app.get("/data/:databaseName", getEquipmentByDatabase);
app.get("/count", getEquipmentCount);
app.get("/equipment/:id", getEquipmentById);
app.post("/add", addSingleEquipment);
app.put("/equipment/:id", updateEquipment);
app.delete("/equipment/:id", deleteEquipmentById);
app.delete("/clear-all", clearAllEquipment);
app.post("/bulk-upload", bulkUploadEquipment);
app.get("/databases", getDatabases);
app.delete("/clear-database/:databaseName", clearDatabaseEquipment);

// Handle undefined routes
app.use((req, res) => {
  res.status(404).json({ error: `Route ${req.method} ${req.originalUrl} not found` });
});

// ===== SERVER STARTUP =====
server.listen(PORT, "0.0.0.0", () => {
  const serverUrl = `http://${SERVER_IP}:${PORT}`;
  
  console.log(`🚀 Server running on port ${PORT}`);
  console.log(`📡 Server URL: ${serverUrl}`);
  console.log(`🔗 Health check: ${serverUrl}/`);
  console.log(`⚡ Socket.IO: Enabled for real-time updates`);
  console.log(`\n🆕 NEW PATHS:`);
  console.log(`📊 Get equipment: GET ${serverUrl}/EquipmentsDataJSON/GlobalEquipmentJsonData`);
  console.log(`📤 Add equipment: POST ${serverUrl}/EquipmentsDataJSON/GlobalEquipmentJsonData`);
  console.log(`🗑️ Clear equipment: DELETE ${serverUrl}/EquipmentsDataJSON/GlobalEquipmentJsonData`);
  console.log(`\n📋 OLD PATHS (compatibility):`);
  console.log(`📊 Get equipment: GET ${serverUrl}/data`);
  console.log(`📊 Count: GET ${serverUrl}/count`);
  console.log(`📤 Bulk upload: POST ${serverUrl}/bulk-upload`);
  console.log(`🗄️ List databases: GET ${serverUrl}/databases`);
  console.log(`\n🔒 Environment: ${process.env.NODE_ENV || 'development'}`);
});

// Auto-populate equipment data when server starts
setTimeout(async () => {
  await autoPopulateEquipment();
}, 2000); // Wait 2 seconds for MongoDB connection

module.exports = {
  app,
  io,
  server,
  autoPopulateEquipment,
  getEquipmentData,
  addEquipmentData,
  clearEquipmentData,
  getEquipmentCount,
  getAllEquipment,
  getEquipmentByDatabase,
  getEquipmentById,
  addSingleEquipment,
  updateEquipment,
  deleteEquipmentById,
  clearAllEquipment,
  bulkUploadEquipment,
  getDatabases,
  clearDatabaseEquipment,
  Equipment
};
