const express = require('express');
const bodyParser = require('body-parser');
const cors = require('cors');
const bcrypt = require('bcryptjs');
const { MongoClient } = require('mongodb');

const app = express();
const PORT = process.env.WEBPROADMIN_PORT || 5050;

// MongoDB connection
const MONGO_URI = 'mongodb://titandrillingadminuser:mrdThsSnATwD231hbOPmnhj@api.titandrillingzm.com:27017/titan_drilling?authSource=admin';
const DB_NAME = 'titan_drilling';
const COLLECTION_NAME = 'WebProadmin';
const GLOBAL_USERS_COLLECTION = 'GlobalUsers';

let db;
let webProAdminCollection;
let globalUsersCollection;

// Middleware
app.use(cors());
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));

// Connect to MongoDB
async function connectDB() {
    try {
        const client = await MongoClient.connect(MONGO_URI, {
            useNewUrlParser: true,
            useUnifiedTopology: true
        });
        db = client.db(DB_NAME);
        webProAdminCollection = db.collection(COLLECTION_NAME);
        globalUsersCollection = db.collection(GLOBAL_USERS_COLLECTION);
        console.log(`✅ Connected to MongoDB: ${DB_NAME}.${COLLECTION_NAME}`);
        console.log(`✅ Connected to MongoDB: ${DB_NAME}.${GLOBAL_USERS_COLLECTION}`);
        
        // Create initial admin user if collection is empty
        await createInitialUser();
    } catch (error) {
        console.error('❌ MongoDB connection error:', error);
        process.exit(1);
    }
}

// Create initial admin user from ProWebAdmindot.json
async function createInitialUser() {
    try {
        const count = await webProAdminCollection.countDocuments();
        
        if (count === 0) {
            console.log('📝 Creating initial Web Pro Admin user...');
            
            const initialUser = {
                "_id": "user_1761833101603_9zsp1be58",
                "uid": "user_1761833101603_9zsp1be58",
                "name": "Amit.coder",
                "email": "admin@titandrillingzm.com",
                "role": "WebProAdmin",
                "employeeNumber": "TYU7777989",
                "department": "UIN87979",
                "countries": ["DRC", "Zambia"],
                "modules": [
                    { "module": "Equipments", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "Documents", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "teamEmploy", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "inspectionReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "dvirReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "checklist", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "incidentReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "dailyReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "requestMaintenance", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "jobcard", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "serviceSchedule", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "fuelLog", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "logbook", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "hrForm", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "hrApplications", "permissions": ["create", "edit", "delete", "view", "update"] },
                    { "module": "teamKpi", "permissions": ["create", "edit", "delete", "view", "update"] }
                ],
                "loginStatus": false,
                "lastLogin": new Date(),
                "password": "admin@titandrillingzm.comA1",
                "hashedPassword": "$2b$10$CUWpUf3by/2GiP5Lx0GMB.qKOZT4AgHj44evqypFceEU/sE3KN2ge",
                "customUID": "user_1761833101603_9zsp1be58",
                "nrcIdNumber": "YIOO888089",
                "mobile": "86787678678",
                "projects": ["Lubumbashi", "Musompo", "IME", "Kamoa", "Kansanshi", "Kalumbila", "Kimteto", "Kobold", "FQM Exploration", "Mimosa"],
                "createdAt": new Date(),
                "updatedAt": new Date()
            };
            
            await webProAdminCollection.insertOne(initialUser);
            console.log('✅ Initial Web Pro Admin user created successfully!');
            console.log('📧 Email: admin@titandrillingzm.com');
            console.log('🔑 Password: admin@titandrillingzm.comA1');
        } else {
            console.log('✅ Web Pro Admin users already exist');
        }
    } catch (error) {
        console.error('❌ Error creating initial user:', error);
    }
}

// API: Login - Supports email/employeeNumber and password, checks both WebProadmin and GlobalUsers
app.post('/api/webproadmin/login', async (req, res) => {
    try {
        const { email, password, employeeNumber } = req.body;
        
        // Accept email or employeeNumber (ID)
        const loginIdentifier = email || employeeNumber;
        const loginType = email ? 'email' : (employeeNumber ? 'employeeNumber' : null);
        
        if (!loginIdentifier || !password) {
            return res.status(400).json({
                success: false,
                message: 'Email/Employee Number and password are required'
            });
        }
        
        // Normalize the login identifier
        const normalizedEmail = email ? email.toLowerCase().trim() : null;
        const normalizedEmployeeNumber = employeeNumber ? employeeNumber.trim() : null;
        
        console.log(`🔍 Login attempt - Type: ${loginType}, Identifier: ${loginIdentifier}`);
        
        let user = null;
        let userSource = null;
        let isPasswordValid = false;
        
        // First, try to find user in WebProadmin collection
        // Support login by email OR employeeNumber
        const webProAdminQuery = {};
        if (normalizedEmail && normalizedEmployeeNumber) {
            // Both provided - search by either
            webProAdminQuery.$or = [
                { email: normalizedEmail },
                { employeeNumber: normalizedEmployeeNumber }
            ];
        } else if (normalizedEmail) {
            // Only email provided
            webProAdminQuery.email = normalizedEmail;
        } else if (normalizedEmployeeNumber) {
            // Only employeeNumber provided
            webProAdminQuery.employeeNumber = normalizedEmployeeNumber;
        }
        
        user = await webProAdminCollection.findOne(webProAdminQuery);
        
        if (user) {
            userSource = 'WebProadmin';
            console.log(`✅ User found in WebProadmin collection: ${user.email}`);
            console.log(`🔐 Checking password for WebProadmin user...`);
            
            // Verify password using bcrypt for WebProadmin users
            if (user.hashedPassword) {
                try {
                    isPasswordValid = await bcrypt.compare(password, user.hashedPassword);
                    console.log(`🔐 Bcrypt comparison result: ${isPasswordValid}`);
                } catch (bcryptError) {
                    console.error('❌ Bcrypt comparison error:', bcryptError);
                    isPasswordValid = false;
                }
            }
            
            // Fallback to plain text comparison if no hashed password or bcrypt failed
            if (!isPasswordValid && user.password) {
                isPasswordValid = password === user.password;
                console.log(`🔐 Plain text password comparison result: ${isPasswordValid}`);
            }
        } else {
            // If not found in WebProadmin, check GlobalUsers
            console.log(`🔍 User not found in WebProadmin, checking GlobalUsers for ProAdmin...`);
            
            // Support login by email OR employeeNumber for ProAdmin
            const globalUsersQuery = {};
            if (normalizedEmail && normalizedEmployeeNumber) {
                // Both provided - search by either
                globalUsersQuery.$or = [
                    { email: normalizedEmail },
                    { employeeNumber: normalizedEmployeeNumber }
                ];
            } else if (normalizedEmail) {
                // Only email provided
                globalUsersQuery.email = normalizedEmail;
            } else if (normalizedEmployeeNumber) {
                // Only employeeNumber provided
                globalUsersQuery.employeeNumber = normalizedEmployeeNumber;
            }
            
            // Only allow ProAdmin role from GlobalUsers
            globalUsersQuery.role = 'ProAdmin';
            user = await globalUsersCollection.findOne(globalUsersQuery);
            
            if (user) {
                // Verify user has ProAdmin role
                if (user.role !== 'ProAdmin') {
                    console.log(`❌ User found but role is not ProAdmin: ${user.role}`);
                    user = null; // Reset user to null so it fails authentication
                } else {
                    userSource = 'GlobalUsers';
                    console.log(`✅ ProAdmin found in GlobalUsers collection: ${user.email}`);
                    console.log(`👤 User role: ${user.role}`);
                    console.log(`🔐 Checking password for GlobalUsers ProAdmin...`);
                    console.log(`🔐 User has password field: ${!!user.password}`);
                    console.log(`🔐 User has hashedPassword field: ${!!user.hashedPassword}`);
                    
                    // For GlobalUsers ProAdmin, verify password - check hashedPassword first, then plain text password
                    if (user.hashedPassword) {
                        try {
                            isPasswordValid = await bcrypt.compare(password, user.hashedPassword);
                            console.log(`🔐 Bcrypt comparison result: ${isPasswordValid}`);
                        } catch (bcryptError) {
                            console.error('❌ Bcrypt comparison error:', bcryptError);
                            isPasswordValid = false;
                        }
                    }
                    
                    // If bcrypt failed or no hashedPassword, check plain text password
                    if (!isPasswordValid && user.password) {
                        isPasswordValid = password === user.password;
                        console.log(`🔐 Plain text password comparison: ${isPasswordValid}`);
                        console.log(`🔐 Expected: "${user.password}", Received: "${password}"`);
                    }
                    
                    // Additional fallback: check employeeNumber or email (for backward compatibility)
                    if (!isPasswordValid) {
                        isPasswordValid = password === user.employeeNumber || password === user.email;
                        if (isPasswordValid) {
                            console.log(`🔐 Password matched employeeNumber or email`);
                        }
                    }
                }
            } else {
                console.log(`❌ No ProAdmin found in GlobalUsers with email/employeeNumber: ${loginIdentifier}`);
            }
        }
        
        // If user not found in either collection
        if (!user) {
            console.log(`❌ User not found with ${loginType}: ${loginIdentifier}`);
            return res.status(401).json({
                success: false,
                message: `Invalid ${loginType === 'email' ? 'email' : 'employee number'} or password`
            });
        }
        
        // If password is invalid
        if (!isPasswordValid) {
            console.log(`❌ Invalid password for user: ${user.email || user.employeeNumber}`);
            console.log(`🔐 Password check failed - User has password: ${!!user.password}, hashedPassword: ${!!user.hashedPassword}`);
            return res.status(401).json({
                success: false,
                message: `Invalid ${loginType === 'email' ? 'email' : 'employee number'} or password`
            });
        }
        
        // Update login status and last login
        const updateData = {
            $set: { 
                loginStatus: true, 
                lastLogin: new Date() 
            }
        };
        
        if (userSource === 'WebProadmin') {
            await webProAdminCollection.updateOne(
                { _id: user._id },
                updateData
            );
        } else {
            await globalUsersCollection.updateOne(
                { _id: user._id },
                updateData
            );
        }
        
        // Remove sensitive data before sending response
        delete user.password;
        delete user.hashedPassword;
        
        // Add source information to response
        user.userSource = userSource;
        
        // If ProAdmin from GlobalUsers, assign all modules with all permissions
        if (userSource === 'GlobalUsers' && user.role === 'ProAdmin') {
            console.log(`🔧 Assigning all modules with all permissions to ProAdmin...`);
            user.modules = [
                { "module": "Equipments", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "Documents", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "teamEmploy", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "inspectionReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "dvirReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "checklist", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "incidentReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "dailyReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "requestMaintenance", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "jobcard", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "serviceSchedule", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "fuelLog", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "logbook", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "hrForm", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "hrApplications", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "teamKpi", "permissions": ["create", "edit", "delete", "view", "update"] }
            ];
            console.log(`✅ All modules assigned to ProAdmin`);
        }
        
        const userIdentifier = user.email || user.employeeNumber || user.name;
        console.log(`✅ Login successful for: ${userIdentifier} (Source: ${userSource}, Role: ${user.role || 'N/A'}, Login Type: ${loginType})`);
        
        res.json({
            success: true,
            message: 'Login successful',
            user: user
        });
        
    } catch (error) {
        console.error('❌ Login error:', error);
        console.error('❌ Error stack:', error.stack);
        res.status(500).json({
            success: false,
            message: 'Internal server error',
            error: error.message
        });
    }
});

// API: Logout - Supports both WebProadmin and GlobalUsers
app.post('/api/webproadmin/logout', async (req, res) => {
    try {
        const { uid } = req.body;
        
        if (!uid) {
            return res.status(400).json({
                success: false,
                message: 'User ID is required'
            });
        }
        
        // Try to logout from WebProadmin collection
        const webProAdminResult = await webProAdminCollection.updateOne(
            { uid: uid },
            { $set: { loginStatus: false } }
        );
        
        // If not found in WebProadmin, try GlobalUsers
        if (webProAdminResult.matchedCount === 0) {
            await globalUsersCollection.updateOne(
                { uid: uid },
                { $set: { loginStatus: false } }
            );
        }
        
        res.json({
            success: true,
            message: 'Logout successful'
        });
        
    } catch (error) {
        console.error('❌ Logout error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// API: Get user profile - Supports both WebProadmin and GlobalUsers
app.get('/api/webproadmin/profile/:uid', async (req, res) => {
    try {
        const { uid } = req.params;
        
        // First try WebProadmin collection
        let user = await webProAdminCollection.findOne({ uid: uid });
        let userSource = 'WebProadmin';
        
        // If not found, try GlobalUsers (only ProAdmin)
        if (!user) {
            user = await globalUsersCollection.findOne({ 
                uid: uid,
                role: 'ProAdmin'
            });
            userSource = 'GlobalUsers';
        }
        
        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }
        
        // Remove sensitive data
        delete user.password;
        delete user.hashedPassword;
        
        user.userSource = userSource;
        
        // If ProAdmin from GlobalUsers, assign all modules with all permissions
        if (userSource === 'GlobalUsers' && user.role === 'ProAdmin') {
            user.modules = [
                { "module": "Equipments", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "Documents", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "teamEmploy", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "inspectionReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "dvirReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "checklist", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "incidentReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "dailyReport", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "requestMaintenance", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "jobcard", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "serviceSchedule", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "fuelLog", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "logbook", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "hrForm", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "hrApplications", "permissions": ["create", "edit", "delete", "view", "update"] },
                { "module": "teamKpi", "permissions": ["create", "edit", "delete", "view", "update"] }
            ];
        }
        
        res.json({
            success: true,
            user: user
        });
        
    } catch (error) {
        console.error('❌ Get profile error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// API: Get all Web Pro Admins
app.get('/api/webproadmin/all', async (req, res) => {
    try {
        const users = await webProAdminCollection.find({}).toArray();
        
        // Remove sensitive data
        users.forEach(user => {
            delete user.password;
            delete user.hashedPassword;
        });
        
        res.json({
            success: true,
            count: users.length,
            users: users
        });
        
    } catch (error) {
        console.error('❌ Get all users error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// API: Create new Web Pro Admin
app.post('/api/webproadmin/create', async (req, res) => {
    try {
        const userData = req.body;
        
        // Check if email already exists
        const existingUser = await webProAdminCollection.findOne({ email: userData.email });
        
        if (existingUser) {
            return res.status(400).json({
                success: false,
                message: 'Email already exists'
            });
        }
        
        // Hash password
        const hashedPassword = await bcrypt.hash(userData.password, 10);
        
        const newUser = {
            ...userData,
            hashedPassword: hashedPassword,
            loginStatus: false,
            createdAt: new Date(),
            updatedAt: new Date()
        };
        
        await webProAdminCollection.insertOne(newUser);
        
        // Remove sensitive data
        delete newUser.password;
        delete newUser.hashedPassword;
        
        res.json({
            success: true,
            message: 'Web Pro Admin created successfully',
            user: newUser
        });
        
    } catch (error) {
        console.error('❌ Create user error:', error);
        res.status(500).json({
            success: false,
            message: 'Internal server error'
        });
    }
});

// Health check
app.get('/health', (req, res) => {
    res.json({
        status: 'OK',
        service: 'Web Pro Admin Login',
        port: PORT,
        timestamp: new Date()
    });
});

// Start server
connectDB().then(() => {
    app.listen(PORT, () => {
        console.log(`🔐 Web Pro Admin Login API running on port ${PORT}`);
        console.log(`📡 https://api.titandrillingzm.com:${PORT}`);
    });
});


