const express = require('express');
const cors = require('cors');
const path = require('path');
const fs = require('fs');
const multer = require('multer');
const { mongoose } = require('./dbConnection');

const app = express();
const PORT = process.env.HSE_CORRECTIVE_ACTION_PORT || 5032;

app.use(cors({ origin: '*', methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'], allowedHeaders: ['Content-Type', 'Authorization'] }));
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Basic request logger
app.use((req, res, next) => {
  console.log(`📡 [HSE Corrective Action] ${req.method} ${req.originalUrl}`);
  next();
});

console.log('✅ [HSE Corrective Action] Using shared MongoDB connection');

const correctiveActionSchema = new mongoose.Schema(
  {
    _id: { type: String, default: () => `hsecorr_${Date.now()}` },
    reportId: { type: String, required: true, unique: true, index: true },
    userUid: { type: String, required: true, index: true },
    status: { type: String, enum: ['draft', 'completed'], default: 'draft' },
    checklistTitle: String,
    unsafeSummary: {
      totalUnsafeItems: Number,
      unsafeItems: [mongoose.Schema.Types.Mixed],
    },
    subItemStatuses: { type: mongoose.Schema.Types.Mixed, default: {} },
    faultyCount: { type: Number, default: 0 },
    isSatisfactory: { type: Boolean, default: null },
    role: { type: String, default: null },
    mechanicName: String,
    operatorName: String,
    mechanicSignatureUri: String,
    operatorSignatureUri: String,
    mechanicSignatureDownloadUrl: String,
    operatorSignatureDownloadUrl: String,
    pdfDownloadUrl: String,
    pdfLocalPath: String,
    notes: mongoose.Schema.Types.Mixed,
    metadata: mongoose.Schema.Types.Mixed,
    createdBy: String,
    lastModifiedBy: String,
    createdAt: { type: Date, default: Date.now },
    updatedAt: { type: Date, default: Date.now },
  },
  {
    strict: false,
    collection: 'hsecorrectiveactions',
  }
);

correctiveActionSchema.pre('save', function (next) {
  this.updatedAt = new Date();
  if (!this.status) {
    this.status = 'draft';
  }
  next();
});

const HSECorrectiveAction =
  mongoose.models.HSECorrectiveAction ||
  mongoose.model('HSECorrectiveAction', correctiveActionSchema);

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const tempDir = path.join(__dirname, 'uploads', 'temp');
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
    }
    cb(null, tempDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
    cb(null, `${file.fieldname}-${uniqueSuffix}${path.extname(file.originalname)}`);
  },
});

const fileUpload = multer({
  storage,
  limits: { fileSize: 50 * 1024 * 1024 },
});

const ensureUserDirectory = (userId = 'unknown') => {
  const userDir = path.join(__dirname, 'uploads', 'documents', 'hsecorrectiveactions', userId);
  if (!fs.existsSync(userDir)) {
    fs.mkdirSync(userDir, { recursive: true });
    console.log(`📁 Created directory: ${userDir}`);
  }
  return userDir;
};

const getExternalBaseUrl = () => {
  const externalPort = Number(PORT) + 1000;
  return `https://api.titandrillingzm.com:${externalPort}`;
};

const determineStatus = (data = {}) => {
  const statuses = Object.values(data.subItemStatuses || {});
  if (!statuses.length) {
    return data.status || 'draft';
  }
  const isComplete = statuses.every((status) => {
    const normalized = String(status || '').toLowerCase();
    return normalized === 'fixed'.toLowerCase() || normalized === 'no need'.toLowerCase();
  });
  return isComplete ? 'completed' : 'draft';
};

app.get('/hse-corrective-actions/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'HSE Corrective Action Handler',
    port: PORT,
    timestamp: new Date().toISOString(),
    mongoose: {
      connected: mongoose.connection.readyState === 1,
      state: mongoose.connection.readyState,
    },
  });
});

app.get('/hse-corrective-actions', async (req, res) => {
  try {
    const {
      limit = 300,
      status,
      reportId,
      requestNumber,
      search,
    } = req.query;

    const parsedLimit = Math.min(Math.max(parseInt(limit, 10) || 300, 1), 1000);
    const query = {};

    if (status && status !== 'all') {
      query.status = String(status).toLowerCase();
    }

    if (reportId) {
      query.reportId = reportId;
    }

    if (requestNumber) {
      query.requestNumber = requestNumber;
    }

    if (search) {
      const regex = new RegExp(String(search).trim(), 'i');
      query.$or = [
        { reportId: regex },
        { requestNumber: regex },
        { correctiveActionNumber: regex },
        { 'metadata.correctiveActionNumber': regex },
      ];
    }

    const correctiveActions = await HSECorrectiveAction.find(query)
      .sort({ updatedAt: -1 })
      .limit(parsedLimit)
      .select('-__v');

    res.json({
      success: true,
      count: correctiveActions.length,
      data: correctiveActions,
    });
  } catch (error) {
    console.error('[HSE Corrective Action] List fetch error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch corrective actions',
    });
  }
});

app.get('/hse-corrective-actions/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    const correctiveAction = await HSECorrectiveAction.findOne({ reportId }).select('-__v');
    if (!correctiveAction) {
      return res.status(404).json({
        success: false,
        error: 'Corrective action not found',
      });
    }
    res.json({ success: true, data: correctiveAction });
  } catch (error) {
    console.error('[HSE Corrective Action] Fetch error:', error);
    res.status(500).json({ success: false, error: 'Failed to fetch corrective action' });
  }
});

app.post('/hse-corrective-actions', async (req, res) => {
  try {
    const { reportId, userId, correctiveActionData } = req.body;
    if (!reportId || !userId || !correctiveActionData) {
      return res.status(400).json({
        success: false,
        error: 'reportId, userId, and correctiveActionData are required',
      });
    }

    const payload = {
      ...correctiveActionData,
      reportId,
      userUid: userId,
      lastModifiedBy: userId,
      updatedAt: new Date(),
    };

    if (!payload.createdBy) {
      payload.createdBy = userId;
    }

    if (payload.unsafeItems && payload.unsafeItems.length) {
      payload.unsafeSummary = {
        totalUnsafeItems: payload.unsafeItems.length,
        unsafeItems: payload.unsafeItems,
      };
    }

     payload.status = determineStatus(payload);

    const correctiveAction = await HSECorrectiveAction.findOneAndUpdate(
      { reportId },
      { $set: payload },
      { upsert: true, new: true, setDefaultsOnInsert: true }
    ).select('-__v');

    res.status(201).json({
      success: true,
      data: correctiveAction,
      message: 'HSE corrective action saved successfully',
    });
  } catch (error) {
    console.error('[HSE Corrective Action] Save error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to save corrective action: ' + error.message,
    });
  }
});

app.post('/hse-corrective-actions/:reportId/upload-pdf', fileUpload.single('pdf'), async (req, res) => {
  try {
    const { reportId } = req.params;
    const { userId = 'unknown', fileName: clientFileName } = req.body;

    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded',
      });
    }

    const userDir = ensureUserDirectory(userId);
    const sanitizedClientName =
      typeof clientFileName === 'string' && clientFileName.trim().length > 0
        ? clientFileName.trim()
        : null;
    const targetFileName =
      sanitizedClientName && sanitizedClientName.toLowerCase().endsWith('.pdf')
        ? sanitizedClientName
        : `hse_corrective_${reportId}_${Date.now()}.pdf`;

    const finalPath = path.join(userDir, targetFileName);
    try {
      fs.renameSync(req.file.path, finalPath);
    } catch (moveError) {
      console.error('[HSE Corrective Action] File move failed, attempting copy:', moveError);
      fs.copyFileSync(req.file.path, finalPath);
      fs.unlinkSync(req.file.path);
    }

    const downloadPath = `/hse-corrective-actions/download-pdf/${userId}/${targetFileName}`;
    const downloadUrl = `${getExternalBaseUrl()}${downloadPath}`;

    await HSECorrectiveAction.findOneAndUpdate(
      { reportId },
      {
        $set: {
          pdfDownloadUrl: downloadUrl,
          pdfLocalPath: finalPath,
          updatedAt: new Date(),
        },
      },
      { upsert: true }
    );

    const refreshed = await HSECorrectiveAction.findOne({ reportId });
    if (refreshed) {
      const nextStatus = determineStatus(refreshed);
      if (nextStatus !== refreshed.status) {
        refreshed.status = nextStatus;
        await refreshed.save();
      }
    }

    res.json({
      success: true,
      fileName: targetFileName,
      filePath: finalPath,
      downloadUrl,
      downloadPath,
      message: 'Corrective action PDF uploaded successfully',
    });
  } catch (error) {
    console.error('[HSE Corrective Action] PDF upload error:', error);
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    res.status(500).json({
      success: false,
      error: 'Failed to upload corrective action PDF: ' + error.message,
    });
  }
});

app.get('/hse-corrective-actions/download-pdf/:userId/:fileName', (req, res) => {
  try {
    const { userId, fileName } = req.params;
    const filePath = path.join(__dirname, 'uploads', 'documents', 'hsecorrectiveactions', userId || 'unknown', fileName);

    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        error: 'PDF file not found',
      });
    }

    res.set({
      'Content-Type': 'application/pdf',
      'Content-Disposition': `attachment; filename="${fileName}"`,
    });

    const stream = fs.createReadStream(filePath);
    stream.pipe(res);
    stream.on('error', (err) => {
      console.error('[HSE Corrective Action] PDF stream error:', err);
      res.status(500).json({ success: false, error: 'Failed to stream PDF' });
    });
  } catch (error) {
    console.error('[HSE Corrective Action] PDF download error:', error);
    res.status(500).json({ success: false, error: 'Failed to download PDF' });
  }
});

app.delete('/hse-corrective-actions/:reportId', async (req, res) => {
  try {
    const { reportId } = req.params;
    const deleted = await HSECorrectiveAction.findOneAndDelete({ reportId });
    if (!deleted) {
      return res.status(404).json({ success: false, error: 'Corrective action not found' });
    }
    res.json({ success: true, message: 'Corrective action deleted successfully' });
  } catch (error) {
    console.error('[HSE Corrective Action] Delete error:', error);
    res.status(500).json({ success: false, error: 'Failed to delete corrective action' });
  }
});

app.use((req, res) => {
  res.status(404).json({ success: false, error: 'Endpoint not found' });
});

app.use((error, req, res, next) => {
  console.error('[HSE Corrective Action] Unhandled error:', error);
  res.status(500).json({ success: false, error: 'Internal server error' });
});

app.listen(PORT, '0.0.0.0', () => {
  const externalPort = Number(PORT) + 1000;
  console.log(`🛠️ HSE Corrective Action server running on port ${PORT}`);
  console.log(`   📋 Save corrective action: https://api.titandrillingzm.com:${externalPort}/hse-corrective-actions`);
  console.log(`   📄 Download PDF: https://api.titandrillingzm.com:${externalPort}/hse-corrective-actions/download-pdf/:userId/:fileName`);
});

module.exports = app;

