const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const cors = require('cors');
const multer = require('multer');
const path = require('path');
const fs = require('fs');

const app = express();
const PORT = 5009;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// MongoDB Connection - Use the same connection as GlobalAdminUsersHandle
// Connect to MongoDB with better error handling
const connectDB = async () => {
  try {
    // Using shared MongoDB connection from dbConnection.js
    console.log('✅ MongoDB connected successfully for Request Maintenance Handler!');
  } catch (err) {
    console.error('❌ MongoDB connection error:', err);
    console.error('⚠️ Will retry connection in 10 seconds...');
    setTimeout(() => {
      connectDB();
    }, 10000);
  }
};

// MongoDB connection event handlers
mongoose.connection.on('disconnected', () => {
  console.log('⚠️ MongoDB disconnected');
  setTimeout(() => {
    connectDB();
  }, 5000);
});

mongoose.connection.on('error', (err) => {
  console.error('❌ MongoDB connection error:', err);
});

// Request Maintenance Schema
const RequestMaintenanceSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  id: { type: String, required: true },
  adminUid: { type: String, required: true },
  requestNumber: { type: String, required: true },
  workOrderTitle: { type: String, required: true },
  requestType: { type: String, required: true },
  priority: { type: String, required: true },
  description: { type: String, required: true },
  equipment: {
    _id: String,
    id: String,
    equipmentName: String,
    equipmentNumber: String,
    model: String,
    serialNumber: String,
    mainCategory: String,
    status: String,
    country: String,
    project: String
  },
  assignTo: [{
    _id: String,
    fullName: String,
    name: String,
    email: String,
    phoneNumber: String,
    role: String
  }],
  country: { type: String, required: true },
  project: { type: String, required: true },
  tasks: [{
    id: String,
    title: String,
    description: String,
    note: String,
    status: String,
    createdAt: String,
    updatedAt: String,
    assignees: [{
      id: String,
      fullName: String,
      employeeNumber: String,
      role: String,
      profilePic: {
        url: String,
        path: String
      }
    }]
  }],
  status: { type: String, default: 'Requested' },
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now },
  userId: { type: String, required: true },
  isArchived: { type: Boolean, default: false }
}, { timestamps: true });

// Admin Users Schema for permissions - Use the same schema as GlobalAdminUsersHandle
const AdminUsersSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  uid: { type: String, required: true },
  name: { type: String, required: true },
  email: { type: String, required: true },
  employeeNumber: { type: String },
  department: { type: String },
  countries: [String],
  projects: [String],
  modules: [Object], // Use Object to match the actual MongoDB structure
  notifications: Object,
  loginStatus: { type: Boolean, default: false },
  lastLogin: { type: Date },
  authCreated: { type: Boolean, default: false },
  uploadTimestamp: { type: Date, default: Date.now },
  originalUserId: { type: String },
  password: { type: String },
  hashedPassword: { type: String },
  role: { type: String, default: 'TeamMember' },
  // Store all other fields dynamically
}, {
  strict: false,
  collection: 'GlobalUsers' // Use the correct collection name from MongoDB Compass
});

// Check if models already exist before creating them
const RequestMaintenance = mongoose.models.GlobalrequestMaintenance || mongoose.model('GlobalrequestMaintenance', RequestMaintenanceSchema);
const AdminUsers = mongoose.models.GlobalUser || mongoose.model('GlobalUser', AdminUsersSchema);

// Multer configuration for file uploads
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const userId = req.params.userId || 'default';
    const uploadPath = `/root/node-mongo-api/uploads/documents/RequestMaintenance/${userId}`;
    // Create directory if it doesn't exist
    if (!fs.existsSync(uploadPath)) {
      fs.mkdirSync(uploadPath, { recursive: true });
    }
    cb(null, uploadPath);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({ storage: storage });

// Routes

// Get all request maintenance records
app.get('/request-maintenance', async (req, res) => {
  try {
    console.log('📋 Fetching all request maintenance records...');
    
    // MongoDB connection will be handled by mongoose automatically
    
    const requests = await RequestMaintenance.find({ isArchived: false }).sort({ createdAt: -1 });
    
    res.json({
      success: true,
      count: requests.length,
      data: requests
    });
  } catch (error) {
    console.error('❌ Error fetching request maintenance records:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get request maintenance records by user ID
app.get('/request-maintenance/user/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    console.log(`📋 Fetching request maintenance records for user: ${userId}`);
    
    // MongoDB connection will be handled by mongoose automatically
    
    const requests = await RequestMaintenance.find({ 
      userId: userId,
      isArchived: false 
    }).sort({ createdAt: -1 });
    
    res.json({
      success: true,
      count: requests.length,
      data: requests
    });
  } catch (error) {
    console.error('❌ Error fetching user request maintenance records:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get single request maintenance record by ID
app.get('/request-maintenance/:id', async (req, res) => {
  try {
    const { id } = req.params;
    console.log(`📋 Fetching request maintenance record with ID: ${id}`);
    
    // MongoDB connection will be handled by mongoose automatically
    
    const request = await RequestMaintenance.findOne({ 
      $or: [
        { id: id },
        { _id: id }
      ],
      isArchived: false 
    });
    
    if (!request) {
      console.log(`❌ Request maintenance record not found for ID: ${id}`);
      return res.status(404).json({
        success: false,
        error: 'Request maintenance record not found'
      });
    }
    
    console.log(`✅ Request maintenance record found: ${request.requestNumber}`);
    
    res.json({
      success: true,
      data: request
    });
  } catch (error) {
    console.error('❌ Error fetching request maintenance record:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get user permissions and data by email
app.get('/admin-users/email/:email', async (req, res) => {
  try {
    const { email } = req.params;
    console.log(`👤 Fetching admin user data for email: ${email}`);
    
    // MongoDB connection will be handled by mongoose automatically
    
    const user = await AdminUsers.findOne({ email: email.toLowerCase().trim() });
    
    if (!user) {
      console.log(`❌ User not found for email: ${email}`);
      return res.status(404).json({
        success: false,
        error: 'User not found'
      });
    }
    
    console.log(`✅ User found: ${user.name} (${user.email})`);
    console.log(`📋 User modules:`, user.modules);
    
    res.json({
      success: true,
      data: user
    });
  } catch (error) {
    console.error('❌ Error fetching admin user data:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get user permissions and data by UID
app.get('/admin-users/uid/:uid', async (req, res) => {
  try {
    const { uid } = req.params;
    console.log(`👤 Fetching admin user data for UID: ${uid}`);
    
    // MongoDB connection will be handled by mongoose automatically
    
    const user = await AdminUsers.findOne({ uid: uid });
    
    if (!user) {
      console.log(`❌ User not found for UID: ${uid}`);
      return res.status(404).json({
        success: false,
        error: 'User not found'
      });
    }
    
    console.log(`✅ User found: ${user.name} (${user.email})`);
    console.log(`📋 User modules:`, user.modules);
    
    res.json({
      success: true,
      data: user
    });
  } catch (error) {
    console.error('❌ Error fetching admin user data by UID:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Helper function to sanitize request data (remove offline-specific fields)
const sanitizeRequestData = (data) => {
  const cleanedData = { ...data };

  // Remove offline-specific fields that shouldn't be stored in MongoDB
  delete cleanedData.offlineCreated;
  delete cleanedData.offlineSyncPending;
  delete cleanedData.offlineCreatedAt;
  delete cleanedData.userPermissions;
  delete cleanedData.userCountries;
  delete cleanedData.userProjects;
  delete cleanedData.syncedAt;

  return cleanedData;
};

// Create new request maintenance
app.post('/request-maintenance', async (req, res) => {
  try {
    console.log('📝 Creating new request maintenance record...');
    console.log('📊 Request data includes:', {
      requestNumber: req.body.requestNumber,
      workOrderTitle: req.body.workOrderTitle,
      tasksCount: req.body.tasks?.length || 0,
      tasks: req.body.tasks?.map(task => ({
        id: task.id,
        title: task.title,
        description: task.description,
        note: task.note || 'No note',
        status: task.status || 'No status',
        assigneesCount: task.assignees?.length || 0
      })) || [],
      assignToCount: req.body.assignTo?.length || 0,
      equipment: req.body.equipment ? 'Selected' : 'None'
    });
    
    // Sanitize request data to remove offline-specific fields
    const sanitizedBody = sanitizeRequestData(req.body);
    
    const requestData = {
      _id: sanitizedBody.id || Date.now().toString(),
      id: sanitizedBody.id || Date.now().toString(),
      ...sanitizedBody,
      createdAt: new Date(),
      updatedAt: new Date()
    };
    
    const newRequest = new RequestMaintenance(requestData);
    await newRequest.save();
    
    console.log('✅ Request maintenance record created:', newRequest.id);
    console.log('📊 Saved record includes:', {
      requestNumber: newRequest.requestNumber,
      tasksCount: newRequest.tasks?.length || 0,
      tasks: newRequest.tasks?.map(task => ({
        id: task.id,
        title: task.title,
        description: task.description,
        note: task.note || 'No note',
        status: task.status || 'No status',
        assigneesCount: task.assignees?.length || 0
      })) || [],
      assignToCount: newRequest.assignTo?.length || 0
    });
    
    res.json({
      success: true,
      data: newRequest
    });
  } catch (error) {
    console.error('❌ Error creating request maintenance record:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Update request maintenance
app.put('/request-maintenance/:id', async (req, res) => {
  try {
    const { id } = req.params;
    console.log(`📝 Updating request maintenance record: ${id}`);
    
    // Sanitize request data to remove offline-specific fields
    const sanitizedBody = sanitizeRequestData(req.body);
    
    const updateData = {
      ...sanitizedBody,
      updatedAt: new Date()
    };
    
    const updatedRequest = await RequestMaintenance.findOneAndUpdate(
      { id: id },
      updateData,
      { new: true, upsert: true }
    );
    
    console.log('✅ Request maintenance record updated:', updatedRequest.id);
    
    res.json({
      success: true,
      data: updatedRequest
    });
  } catch (error) {
    console.error('❌ Error updating request maintenance record:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Delete request maintenance
app.delete('/request-maintenance/:id', async (req, res) => {
  try {
    const { id } = req.params;
    console.log(`🗑️ Deleting request maintenance record: ${id}`);
    
    const deletedRequest = await RequestMaintenance.findOneAndUpdate(
      { id: id },
      { isArchived: true, updatedAt: new Date() },
      { new: true }
    );
    
    if (!deletedRequest) {
      return res.status(404).json({
        success: false,
        error: 'Request maintenance record not found'
      });
    }
    
    console.log('✅ Request maintenance record archived:', deletedRequest.id);
    
    res.json({
      success: true,
      message: 'Request maintenance record archived successfully',
      data: deletedRequest
    });
  } catch (error) {
    console.error('❌ Error deleting request maintenance record:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// File upload endpoint for request maintenance documents
app.post('/request-maintenance/upload/:userId', upload.single('file'), async (req, res) => {
  try {
    const { userId } = req.params;
    console.log(`📤 File upload request for user: ${userId}`);
    console.log('📤 Request body:', req.body);
    console.log('📤 Request file:', req.file);
    
    if (!req.file) {
      console.log('❌ No file received in upload request');
      return res.status(400).json({
        success: false,
        error: 'No file uploaded'
      });
    }

    const fileUrl = `https://api.titandrillingzm.com:6009/uploads/documents/RequestMaintenance/${userId}/${req.file.filename}`;
    
    console.log(`✅ File uploaded successfully for user ${userId}`);
    console.log(`📁 File saved to: ${req.file.path}`);
    console.log(`🔗 File URL: ${fileUrl}`);
    
    res.json({
      success: true,
      message: 'File uploaded successfully',
      fileUrl: fileUrl,
      fileName: req.file.filename,
      filePath: req.file.path
    });
  } catch (err) {
    console.error('❌ Error uploading file:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Test endpoint
app.get('/', (req, res) => {
  res.json({
    status: 'Request Maintenance Server is running',
    port: PORT,
    collections: ['GlobalrequestMaintenance', 'GlobalUsers'],
    endpoints: {
      requestMaintenance: '/request-maintenance',
      userRequests: '/request-maintenance/user/:userId',
      adminUsersByEmail: '/admin-users/email/:email',
      adminUsersByUid: '/admin-users/uid/:uid',
      fileUpload: '/request-maintenance/upload/:userId'
    }
  });
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('❌ Unhandled error:', error);
  res.status(500).json({
    success: false,
    error: 'Internal server error'
  });
});

// Start server
const startServer = async () => {
  try {
    // Wait for MongoDB connection first
    await connectDB();
    
    // Wait a bit for connection to stabilize
    await new Promise(resolve => setTimeout(resolve, 2000));
    
    app.listen(PORT, () => {
      console.log(`🚀 Request Maintenance Handler server running on port ${PORT}`);
    });
  } catch (error) {
    console.error('❌ Failed to start server:', error);
    process.exit(1);
  }
};

startServer();

module.exports = app;
